#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
TEX constructors
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Basic constructors
=============================================
|#

(deff tex-paragraph (&rest body)
 (line (line body)))

(deff tex-brace (&rest x)
 (list "{" x "}"))

(deff tex-bracket (&rest x)
 (list "[" x "]"))

(deff tex-quote (&rest x)
 (list "``" x "''"))

(deff tex-cmd (x &rest y)
 (list "\\" x " " y))

(deff tex-begin (&rest x)
 (tex-cmd "begin" (tex-brace x)))

(deff tex-end (&rest x)
 (tex-cmd "end" (tex-brace x)))

(deff tex-block (x &rest y)
 (list (tex-begin x) y (tex-end x)))

(deff tex-framebox (&rest x)
 (tex-cmd "framebox" (tex-brace x)))

#|
=============================================
Paragraph with margins
=============================================
|#

(deff tex-setlength (length &rest ct)
 (tex-cmd "setlength"
  (tex-brace (tex-cmd length))
  (tex-brace ct)))

(deff tex-one-item-list
 (&key
  (leftmargin  "0mm")
  (rightmargin "0mm")
  (topsep      "0mm")
  (partopsep   "0mm")
  (itemsep     "0mm")
  (parsep      "0mm")
  body)
 (tex-block "list"
  (tex-brace "")
  (tex-brace
   (line (tex-setlength "leftmargin"  leftmargin ))
   (line (tex-setlength "rightmargin" rightmargin))
   (line (tex-setlength "topsep"      topsep	))
   (line (tex-setlength "partopsep"   partopsep  ))
   (line (tex-setlength "itemsep"     itemsep	))
   (line (tex-setlength "parsep"      parsep	)))
  (tex-cmd "item" body)))

#|
=============================================
Style commands
=============================================
|#

(deff tex-bf (&rest x)
 (tex-brace (tex-cmd "bf") x))

(deff tex-em (&rest x)
 (tex-brace (tex-cmd "em") x))

(deff tex-Large (&rest x)
 (tex-brace (tex-cmd "Large") x))

(deff tex-right (&rest x)
 (tex-block "flushright" x))

(deff tex-left (&rest x)
 (tex-block "flushleft" x))

(deff tex-em-right (&rest x)
 (tex-right (tex-em x)))

(deff tex-vspace (&rest x)
 (tex-paragraph (tex-cmd "vspace" (tex-brace x))))

#|
=============================================
Section commands
=============================================
|#

(deff tex-section (&rest x)
 (tex-paragraph (tex-cmd "section" (tex-brace x))))

(deff tex-section* (&rest x)
 (tex-paragraph (tex-cmd "section*" (tex-brace x))))

(deff tex-subsection (&rest x)
 (tex-paragraph (tex-cmd "subsection" (tex-brace x))))

(deff tex-subsection* (&rest x)
 (tex-paragraph (tex-cmd "subsection*" (tex-brace x))))

#|
=============================================
TeX links
=============================================
*hyperbaseurl* is set by the "backend" function in backend.lisp. It is nil for non-submitted pages. For submitted pages it equals an url such as
  http://logiweb.eu/logiweb/server/relay/go/99999999999999/2
where "http://logiweb.eu/logiweb/server/relay" is the relay of the server to which the page is submitted and 99999999999999 is the reference of the current page.

(symbol-anchor ref id) is a string that identifies a symbol uniquely.

(tex-ref url &rest ct) forms a link to the given url.

(tex-link url &rest ct) does the same but appends a newline.

(tex-anchor url &rest ct) forms an anchor.

(tex-localref url &rest ct) forms an internal link to the given anchor.

|#

(defvar *hyperbaseurl*)

(deff symbol-anchor (ref id)
 (list "url:" (html-dec ref) ":" (html-dec id)))

(deff tex-ref (url &rest ct)
 (tex-cmd "href" (tex-brace url) (tex-brace ct)))

(deff tex-link (url &rest ct)
 (line (tex-ref url ct)))

(deff tex-anchor (url &rest ct)
 (tex-cmd "hypertarget" (tex-brace url) (tex-brace ct)))

(deff tex-localref (url &rest ct)
 (tex-cmd "hyperlink" (tex-brace url) (tex-brace ct)))

(deff tex-signature ()
 (tex-ref
  "http://www.diku.dk/~grue/index.html"
  "Klaus Grue"))

#|
=============================================
Document constructors
=============================================
|#

(deff tex-date (lgt)
 (list
  (line (tex-ref (help-abs-link "misc/time") (lgt-2-grd-utc-string lgt)) " = ")
  (line (tex-ref (help-abs-link "misc/time") (lgt-2-mjd-tai-string lgt)) " = ")
  (line (tex-ref (help-abs-link "misc/time") (lgt-2-string lgt)))))

(deff tex-footer (lgt)
 (tex-paragraph
  (tex-em
   (line (tex-ref (help-abs-link "compiler/index") "The Logiweb compiler") ",")
   (line "version " *logiweb-version*)
   (line "by")
   (tex-signature)
   (line "\\\\")
   (tex-date lgt))))

(deff tex-makebox (width position &rest ct)
 (tex-cmd "makebox"
  (tex-bracket width)
  (tex-bracket position)
  (tex-brace ct)))

(deff tex-eps (x1 y1 x2 y2 &rest ct)
 (tex-cmd "includegraphics"
  (tex-bracket x1 "," y1)
  (tex-bracket x2 "," y2)
  (tex-brace ct)))

(deff tex-right-margin-box (&rest ct)
 (tex-makebox "0mm" "l"
  (tex-makebox (tex-cmd "lgwlinewidth") "r" ct)))

#|
The icon is included with
lower left x  =  0.0mm
lower left y  = 18.5mm
upper right x = 16.5mm
upper right y = 19.0mm

The left and right x (0mm and 16.5mm, respectively) are real.
The icon is 16.5 mm wide.

The upper y (19mm) is real. The icon is 19mm high.

The lower y (18.5mm) defines the footline.

The icon extends 0.5mm above and 18.5mm below the footline
|#
(deff tex-icon (&rest body)
 (:let old (tex-brace (tex-cmd "linewidth")))
 (:let new (tex-brace (tex-cmd "lgwlinewidth")))
 (list
  (line (tex-cmd "newlength") new)
  (line (tex-cmd "setlength") new old)
  (tex-one-item-list :rightmargin "20mm" :body
   (list
    (tex-right-margin-box
     (tex-eps "0mm" "18.5mm" "16.5mm" "19mm"
      (option "varhome") "/doc/logiweb.eps"))
    (line "%")
    body))))

(deff tex-title (&rest x)
 (list
  (tex-icon (tex-paragraph (tex-bf (tex-Large x))))
  (tex-vspace "12.5mm")))

(deff set-hyperbaseurl (path)
 (:when (null *hyperbaseurl*) nil)
 (tex-paragraph "\\hyperbaseurl{" *hyperbaseurl* path "}"))

(deff tex-page (path lgt title &rest body)
 (list
  (tex-paragraph "\\documentclass [fleqn]{article}")
  (tex-paragraph "\\setlength {\\overfullrule }{1mm}")
  (tex-paragraph "\\usepackage{latexsym}")
  (tex-paragraph "\\setlength{\\parindent}{0em}")
  (tex-paragraph "\\setlength{\\parskip}{1ex}")
  (tex-paragraph "\\usepackage{graphicx}")
  ;(tex-paragraph "\\usepackage[dvipdfm=true,margin=1in]{geometry}")
  (tex-paragraph "\\usepackage[dvipdfm=true]{hyperref}")
  (tex-paragraph "\\hypersetup{pdfpagemode=UseNone}")
  (tex-paragraph "\\hypersetup{pdfstartpage=1}")
  (tex-paragraph "\\hypersetup{pdfstartview=FitBH}")
  (tex-paragraph "\\hypersetup{pdfpagescrop={120 80 490 730}}")
  (tex-paragraph "\\hypersetup{pdftitle=" title "}")
  (tex-paragraph "\\hypersetup{colorlinks=true}")
  (tex-paragraph "\\everymath{\\rm}")
  (set-hyperbaseurl path)
  (tex-paragraph "\\begin {document}\\raggedright")
  body
  (tex-paragraph "\\vspace{2ex}")
  (tex-footer lgt)
  (tex-paragraph "\\end{document}")))

