#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Logiweb machine
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Event identifiers
=============================================
quit  : return status code <Quit,status>
boot  : initial event      <Boot,argv,env,cache>
write : write to stdout    <Out,ct>
read  : read from stdin    <In,char>
exec  : execute            <Exec,process,handler>
int   : interrupt          <Int,id,process>
ext   : extend             <Ext,ct>
extd  : extended           <Ext,status,ct>
|#

(defc id-event-quit  (list 0 (string2card "quit" )))
(defc id-event-boot  (list 0 (string2card "boot" )))
(defc id-event-write (list 0 (string2card "write")))
(defc id-event-read  (list 0 (string2card "read" )))
(defc id-event-exec  (list 0 (string2card "exec" )))
(defc id-event-int   (list 0 (string2card "int"  )))
(defc id-event-ext   (list 0 (string2card "ext"  )))
(defc id-event-extd  (list 0 (string2card "extd" )))

#|
=============================================
Interrupt identifiers
=============================================
exit : <Int,<Exit>,process> ('process' is an rnf)
time : <Int,<Time>,process>
mem  : <Int,<Mem>,process>
|#

(defc id-int-exit (list 0 (string2card "exit")))
(defc id-int-time (list 0 (string2card "time")))
(defc id-int-mem  (list 0 (string2card "mem" )))

#|
=============================================
Keyboard input
=============================================
|#

(deff read-char-no-echo-no-buffer-no-hang ()
 (:when (null (socket:socket-status *keyboard-input* 0)) nil)
 (:let char (read-char ext:*keyboard-input*))
 (:when (char-key char) nil)
 (:let code (char-code (character char)))
 (:when (= code 13) 10)
 (:when (< 31 code 127) code)
 nil)

#|
(deff read-char-no-echo-no-buffer-no-hang ()
 (:when (null (socket:socket-status *keyboard-input* 0)) nil)
 (:let char (read-char ext:*keyboard-input*))
 (:let key (char-key char))
 (:when (null key) (char-code (character char)))
 (:unless (characterp key) nil)
 (:let code (char-code key))
 (:when (evenp (char-bits char)) code)
 (logand code (lognot 64)))

(deff read-char-no-echo ()
 (with-keyboard (format t "~s~%" (read-char ext:*keyboard-input*))))
|#

#|
=============================================
Construct input events
=============================================
(m-boot cache) constructs a one element list whose element is a boot event.
(m-read event*) reads one character and prepends an input event to event*
(m-exit process event*) prepends an exit interupt event to event*
(m-extd success message event*) prepends an extended event to event*
|#

(deff m-boot (cache)
 (list (list id-event-boot (ct2ct *args*) (ct2ct (getenv)) cache)))

(deff code2vector (code)
 (make-array '(1) :element-type '(unsigned-byte 8) :initial-element code))

(deff m-read (event*)
 (:let code (read-char-no-echo-no-buffer-no-hang))
 (:when (null code) event*)
 (:when (> code 127)
  (format t "~%Unicode characters above code 127 not supported, sorry.~%")
  event*)
 (:let code (code2vector code))
 (cons (list id-event-read code) event*))

(deff m-exit (process event*)
 (cons (list id-event-int (list id-int-exit) process) event*))

(deff m-extd (value event*)
 (:let event (list id-event-extd value))
 (cons event event*))

#|
=============================================
Execute output events
=============================================
*id2exec* is a hash table that translates event id's to functions.

(exec-event* output* input*) executes output*. Input events generated by output events are accumulated in input*. exec-event* returns an integer if the Logiweb machine should halt and a cons of form (cons handler input*) if the Logiweb machine should be reinvoked.

(m-quit arg* output* input*) returns an integer which indicates that the Logiweb machine should halt.

(m-exec arg* output* input*) adds an exit and a read event to input* and returns (cons handler input*). The cons return value indicates that the Logiweb machine should be reinvoked.

(m-write arg* output* input*) executes a write event and then executes output*.

(m-ext arg* output* input*) executes an extend event and then executes output*.
|#

(defc *id2exec* (make-hash-table :test 'equalp))

(deff m-hash (id fct)
 (setf (gethash id *id2exec*) fct)
 (:let (ref idx) id)
 (:let idx (card2vector idx))
 (:let id (list ref idx))
 (setf (gethash id *id2exec*) fct))

(m-hash id-event-quit  'm-quit )
(m-hash id-event-exec  'm-exec )
(m-hash id-event-write 'm-write)
(m-hash id-event-ext   'm-ext  )

(deff exec-event* (output* input*)
 (:when (null output*) 0)
 (:let ((id . arg*) . output*) output*)
 (:let exec (gethash id *id2exec*))
 (:when (null exec) (exec-event* output* input*))
 (funcall exec arg* output* input*))

(deff m-quit (arg* :output* :input*)
 (:let (arg) arg*)
 (:when (integerp arg) arg)
 0)

(deff m-exec (arg* :output* input*)
 (:let (process handler) arg*)
 (:let input* (reverse input*))
 (:let input* (m-read input*))
 (:let input* (m-exit process input*)) ; this should be interruptible
 (cons handler input*))

(deff m-write (arg* output* input*)
 (:let (arg) arg*)
 (m-write1 arg)
 (exec-event* output* input*))

(deff m-write1 (val)
 (:when (arrayp val) (m-write2 val))
 (:when (integerp val) (m-write2 (card2vector val)))
 (:when (consp val) (m-write1 (car val)) (m-write1 (cdr val)))
 nil)

(deff m-write2 (vector)
 (dotimes (n (length vector)) (m-write3 (aref vector n))))

(deff m-write3 (card)
 (:when (= card 10) (terpri))
 (:when (< card 32) nil)
 (princ (card2char card)))

(deff m-ext (:arg* output* input*)
 (format t "~%This primitive implementation ignores Extend events, sorry.~%")
 (:let input* (m-extd nil input*))
 (exec-event* output* input*))

#|
=============================================
Execute handler
=============================================
(m-mainloop handler input*) applies the given handler to the given input*, executes the return value, and possibly reinvokes m-mainloop.
|#

(deff m-mainloop (handler input*)
 (:let output* (tm2tv (map-apply (map-tail (tv2tm handler)) (tv2tm input*))))
 (:let result (exec-event* output* nil))
 (:when (integerp result) result)
 (:let (handler . input*) result)
 (m-mainloop handler input*))

#|
=============================================
Machine invocation
=============================================
|#

(deff machine (ref state)
 (:let cache (aget state (id-cache) ref))
;(:let def (c-get-page-aspect ref card-execute cache))
 (:let def (aget cache ref card-codex ref 0 0 card-execute))
 (:when (null def) (format t "No executable found~%") 0)
 (:let (:define :claim :lhs tree) def)
 (:let term (tree2term cache nil tree))
 (:let handler (tm2tv (term2rnf term nil)))
 (with-keyboard (m-mainloop handler (m-boot cache))))

(deff named-machine (handler cache)
 (with-keyboard (m-mainloop handler (m-boot cache))))

#|
=============================================
Dump machine
=============================================
|#

(deff dump-add-newlines (string*)
 (:when (atom string*) nil)
 (:let (string . string*) string*)
 (list* string f-newline (dump-add-newlines string*)))

(deff dump-machine (ref :state name)
 (:when (equalp name "") nil)
 (ct2file name
  (list
   (line "#!" (option "interpreter") " script")
   (line "execute")
   (line (b-ref-2-base16 ref))))
 (run-chmod "+x" name))

(deff dump-machines (ref state dir)
 (:when (equalp dir "/") nil)
 (progress "Dumping executables")
 (:let cache (aget state (id-cache) ref))
 (:let codex (aget cache ref card-codex))
 (dump-newmachines1 cache dir (aget codex 0))
 (:let def (aget codex ref 0 0 card-executables))
 (:when (null def) 0)
 (:let (:define :claim :lhs tree) def)
 (:let term (tree2term cache nil tree))
 (:let assoc (tm2tv (term2rnf term nil)))
 (dump-machines1 cache dir assoc))

(deff dump-machines1 (cache dir assoc)
 (:when (atom assoc) nil)
 (:let ((name handler . rest) . assoc) assoc)
 (dump-machines2 cache dir name handler rest)
 (dump-machines1 cache dir assoc))

(deff dump-machines2 (cache dir name :handler rest)
 (:unless (intp name) (format t "Names of machines must be strings~%"))
 (:let name (card2vector name))
 (:let name (vector2string name))
 (:when (invalid-name name)
  (format t "Machine not dumped due to invalid file name: ~s~%" name))
 (when (verbose '> 0) (format t "Dumping ~a~%" name))
 (:let ref (aget cache 0))
 (ct2file (cat dir name)
  (list
   (line "#!" (option "interpreter") " script")
   (line "executables")
   (line (b-ref-2-base16 ref))
   (line name)
   (dump-add-newlines rest)))
 (run-chmod "+x" (cat dir name)))

(deff dump-newmachines1 (cache dir codex)
 (:when (atom codex) nil)
 (:let (head . tail) codex)
 (:when (intp head) (dump-newmachines2 cache dir head tail))
 (dump-newmachines1 cache dir head)
 (dump-newmachines1 cache dir tail))

(deff dump-newmachines2 (cache dir name codex)
 (:let def (aget codex 0 card-execute))
 (:when (null def) nil)
 (:let (:define :aspect :lhs tree) def)
 (:let term (tree2term cache nil tree))
 (:let (:handler . rest) (tm2tv (term2rnf term nil)))
 (:let name (card2vector name))
 (:let name (vector2string name))
 (:when (invalid-name name)
  (format t "Machine not dumped due to invalid file name: ~s~%" name))
 (when (verbose '> 0) (format t "Dumping ~a~%" name))
 (:let ref (aget cache 0))
 (ct2file (cat dir name)
  (list
   (line "#!" (option "interpreter") " script")
   (line "string")
   (line (b-ref-2-base16 ref))
   (line name)
   (line "execute")
   (dump-add-newlines rest)))
 (run-chmod "+x" (cat dir name)))














