# Copyright (C) Internet Systems Consortium, Inc. ("ISC")
#
# SPDX-License-Identifier: MPL-2.0
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0.  If a copy of the MPL was not distributed with this
# file, you can obtain one at https://mozilla.org/MPL/2.0/.
#
# See the COPYRIGHT file distributed with this work for additional
# information regarding copyright ownership.

import pytest

from rollover.common import (
    CDSS,
    DEFAULT_CONFIG,
    DURATION,
    ROLLOVER_MARK,
    UNSIGNING_CONFIG,
)
from rollover.setup import configure_going_insecure, configure_root, configure_tld

import isctest

pytestmark = ROLLOVER_MARK


def bootstrap():
    data = {
        "tlds": [],
        "trust_anchors": [],
    }

    tlds = []
    tld_name = "kasp"
    delegations = configure_going_insecure(tld_name, reconfig=True)
    tld = configure_tld(tld_name, delegations)
    tlds.append(tld)
    data["tlds"].append(tld_name)
    ta = configure_root(tlds)
    data["trust_anchors"].append(ta)
    return data


@pytest.fixture(scope="module", autouse=True)
def after_servers_start(ns3, templates):
    templates.render("ns3/named.conf", {"policy": "insecure"})
    ns3.reconfigure()  # move from "unsigning" to "insecure"


@pytest.mark.parametrize(
    "zone",
    [
        "going-insecure.kasp",
        "going-insecure-dynamic.kasp",
    ],
)
def test_going_insecure_reconfig_step1(zone, ns3, default_algorithm):
    config = DEFAULT_CONFIG
    policy = "insecure"
    szone = f"step1.{zone}"

    isctest.kasp.wait_keymgr_done(ns3, szone, reconfig=True)

    # Key goal states should be HIDDEN.
    # The DS may be removed if we are going insecure.
    step = {
        "zone": szone,
        "cdss": CDSS,
        "keyprops": [
            f"ksk 0 {default_algorithm.number} {default_algorithm.bits} goal:hidden dnskey:omnipresent krrsig:omnipresent ds:unretentive offset:{-DURATION['P10D']}",
            f"zsk {DURATION['P60D']} {default_algorithm.number} {default_algorithm.bits} goal:hidden dnskey:omnipresent zrrsig:omnipresent offset:{-DURATION['P10D']}",
        ],
        # Next key event is when the DS becomes HIDDEN. This
        # happens after the# parent propagation delay plus DS TTL.
        "nextev": DEFAULT_CONFIG["ds-ttl"] + DEFAULT_CONFIG["parent-propagation-delay"],
        # Going insecure, check for CDS/CDNSKEY DELETE, and skip key timing checks.
        "cds-delete": True,
        "check-keytimes": False,
    }
    isctest.kasp.check_rollover_step(ns3, config, policy, step)

    with ns3.watch_log_from_start() as watcher:
        if "dynamic" in zone:
            watcher.wait_for_line(
                f"zone {szone}/IN: dsyncfetch: send NOTIFY(CDS) query to scanner.kasp"
            )
        else:
            watcher.wait_for_line(
                f"zone {szone}/IN (signed): dsyncfetch: send NOTIFY(CDS) query to scanner.kasp"
            )


@pytest.mark.parametrize(
    "zone",
    [
        "going-insecure.kasp",
        "going-insecure-dynamic.kasp",
    ],
)
def test_going_insecure_reconfig_step2(zone, ns3, default_algorithm):
    config = DEFAULT_CONFIG
    policy = "insecure"
    zone = f"step2.{zone}"

    isctest.kasp.wait_keymgr_done(ns3, zone, reconfig=True)

    # The DS is long enough removed from the zone to be considered
    # HIDDEN.  This means the DNSKEY and the KSK signatures can be
    # removed.
    step = {
        "zone": zone,
        "cdss": CDSS,
        "keyprops": [
            f"ksk 0 {default_algorithm.number} {default_algorithm.bits} goal:hidden dnskey:unretentive krrsig:unretentive ds:hidden offset:{-DURATION['P10D']}",
            f"zsk {DURATION['P60D']} {default_algorithm.number} {default_algorithm.bits} goal:hidden dnskey:unretentive zrrsig:unretentive offset:{-DURATION['P10D']}",
        ],
        # Next key event is when the DNSKEY becomes HIDDEN.
        # This happens after the propagation delay, plus DNSKEY TTL.
        "nextev": UNSIGNING_CONFIG["dnskey-ttl"]
        + DEFAULT_CONFIG["zone-propagation-delay"],
        # Zone is no longer signed.
        "zone-signed": False,
        "check-keytimes": False,
    }
    isctest.kasp.check_rollover_step(ns3, config, policy, step)

    assert f"zone {zone}/IN (signed): dsyncfetch" not in ns3.log
