/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.util.Preconditions;

public class TimeUtils {
    private static final Map<String, ChronoUnit> LABEL_TO_UNIT_MAP = Collections.unmodifiableMap(TimeUtils.initMap());

    public static Duration parseDuration(String text) {
        long value;
        char current;
        int pos;
        Preconditions.checkNotNull(text);
        String trimmed = text.trim();
        Preconditions.checkArgument(!trimmed.isEmpty(), "argument is an empty- or whitespace-only string");
        int len = trimmed.length();
        for (pos = 0; pos < len && (current = trimmed.charAt(pos)) >= '0' && current <= '9'; ++pos) {
        }
        String number = trimmed.substring(0, pos);
        String unitLabel = trimmed.substring(pos).trim().toLowerCase(Locale.US);
        if (number.isEmpty()) {
            throw new NumberFormatException("text does not start with a number");
        }
        try {
            value = Long.parseLong(number);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("The value '" + number + "' cannot be re represented as 64bit number (numeric overflow).");
        }
        if (unitLabel.isEmpty()) {
            return Duration.of(value, ChronoUnit.MILLIS);
        }
        ChronoUnit unit = LABEL_TO_UNIT_MAP.get(unitLabel);
        if (unit != null) {
            return Duration.of(value, unit);
        }
        throw new IllegalArgumentException("Time interval unit label '" + unitLabel + "' does not match any of the recognized units: " + TimeUnit.getAllUnits());
    }

    private static Map<String, ChronoUnit> initMap() {
        HashMap<String, ChronoUnit> labelToUnit = new HashMap<String, ChronoUnit>();
        for (TimeUnit timeUnit : TimeUnit.values()) {
            for (String label : timeUnit.getLabels()) {
                labelToUnit.put(label, timeUnit.getUnit());
            }
        }
        return labelToUnit;
    }

    public static String getStringInMillis(Duration duration) {
        return duration.toMillis() + (String)TimeUnit.MILLISECONDS.labels.get(0);
    }

    public static String formatWithHighestUnit(Duration duration) {
        long nanos = duration.toNanos();
        List<TimeUnit> orderedUnits = Arrays.asList(TimeUnit.NANOSECONDS, TimeUnit.MICROSECONDS, TimeUnit.MILLISECONDS, TimeUnit.SECONDS, TimeUnit.MINUTES, TimeUnit.HOURS, TimeUnit.DAYS);
        TimeUnit highestIntegerUnit = IntStream.range(0, orderedUnits.size()).sequential().filter(idx -> nanos % ((TimeUnit)((Object)((Object)orderedUnits.get(idx)))).unit.getDuration().toNanos() != 0L).boxed().findFirst().map(idx -> {
            if (idx == 0) {
                return (TimeUnit)((Object)((Object)orderedUnits.get(0)));
            }
            return (TimeUnit)((Object)((Object)orderedUnits.get(idx - 1)));
        }).orElse(TimeUnit.MILLISECONDS);
        return String.format("%d %s", nanos / highestIntegerUnit.unit.getDuration().toNanos(), highestIntegerUnit.getLabels().get(0));
    }

    public static Duration toDuration(Time time) {
        return Duration.of(time.getSize(), TimeUtils.toChronoUnit(time.getUnit()));
    }

    private static ChronoUnit toChronoUnit(java.util.concurrent.TimeUnit timeUnit) {
        switch (timeUnit) {
            case NANOSECONDS: {
                return ChronoUnit.NANOS;
            }
            case MICROSECONDS: {
                return ChronoUnit.MICROS;
            }
            case MILLISECONDS: {
                return ChronoUnit.MILLIS;
            }
            case SECONDS: {
                return ChronoUnit.SECONDS;
            }
            case MINUTES: {
                return ChronoUnit.MINUTES;
            }
            case HOURS: {
                return ChronoUnit.HOURS;
            }
            case DAYS: {
                return ChronoUnit.DAYS;
            }
        }
        throw new IllegalArgumentException(String.format("Unsupported time unit %s.", new Object[]{timeUnit}));
    }

    private static enum TimeUnit {
        DAYS(ChronoUnit.DAYS, TimeUnit.singular("d"), TimeUnit.plural("day")),
        HOURS(ChronoUnit.HOURS, TimeUnit.singular("h"), TimeUnit.plural("hour")),
        MINUTES(ChronoUnit.MINUTES, TimeUnit.singular("min"), TimeUnit.plural("minute")),
        SECONDS(ChronoUnit.SECONDS, TimeUnit.singular("s"), TimeUnit.plural("sec"), TimeUnit.plural("second")),
        MILLISECONDS(ChronoUnit.MILLIS, TimeUnit.singular("ms"), TimeUnit.plural("milli"), TimeUnit.plural("millisecond")),
        MICROSECONDS(ChronoUnit.MICROS, TimeUnit.singular("\u00b5s"), TimeUnit.plural("micro"), TimeUnit.plural("microsecond")),
        NANOSECONDS(ChronoUnit.NANOS, TimeUnit.singular("ns"), TimeUnit.plural("nano"), TimeUnit.plural("nanosecond"));

        private static final String PLURAL_SUFFIX = "s";
        private final List<String> labels;
        private final ChronoUnit unit;

        private TimeUnit(ChronoUnit unit, String[] ... labels) {
            this.unit = unit;
            this.labels = Arrays.stream(labels).flatMap(ls -> Arrays.stream(ls)).collect(Collectors.toList());
        }

        private static String[] singular(String label) {
            return new String[]{label};
        }

        private static String[] plural(String label) {
            return new String[]{label, label + PLURAL_SUFFIX};
        }

        public List<String> getLabels() {
            return this.labels;
        }

        public ChronoUnit getUnit() {
            return this.unit;
        }

        public static String getAllUnits() {
            return Arrays.stream(TimeUnit.values()).map(TimeUnit::createTimeUnitString).collect(Collectors.joining(", "));
        }

        private static String createTimeUnitString(TimeUnit timeUnit) {
            return timeUnit.name() + ": (" + String.join((CharSequence)" | ", timeUnit.getLabels()) + ")";
        }
    }
}

