/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dubbo.common.utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;

public class LFUCache<K, V> {
    private Map<K, CacheNode<K, V>> map;
    private Map<Long, CacheDeque<K, V>> freqTable;
    private final int capacity;
    private int evictionCount;
    private int curSize = 0;
    private long removeFreqEntryTimeout;
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private static final int DEFAULT_INITIAL_CAPACITY = 1000;
    private static final float DEFAULT_EVICTION_FACTOR = 0.75f;
    private static final long DEFAULT_REMOVE_FREQ_TABLE_TIME_OUT = 1800000L;

    public LFUCache() {
        this(1000, 0.75f, 1800000L);
    }

    public LFUCache(int maxCapacity, float evictionFactor) {
        boolean factorInRange;
        if (maxCapacity <= 0) {
            throw new IllegalArgumentException("Illegal initial capacity: " + maxCapacity);
        }
        boolean bl = factorInRange = evictionFactor <= 1.0f && evictionFactor > 0.0f;
        if (!factorInRange || Float.isNaN(evictionFactor)) {
            throw new IllegalArgumentException("Illegal eviction factor value:" + evictionFactor);
        }
        this.capacity = maxCapacity;
        this.evictionCount = (int)((float)this.capacity * evictionFactor);
        this.map = new HashMap<K, CacheNode<K, V>>();
        this.freqTable = new TreeMap<Long, CacheDeque<K, V>>(Long::compareTo);
        this.freqTable.put(1L, new CacheDeque());
    }

    public LFUCache(int maxCapacity, float evictionFactor, long removeFreqEntryTimeout) {
        boolean factorInRange;
        if (maxCapacity <= 0) {
            throw new IllegalArgumentException("Illegal initial capacity: " + maxCapacity);
        }
        boolean bl = factorInRange = evictionFactor <= 1.0f && evictionFactor > 0.0f;
        if (!factorInRange || Float.isNaN(evictionFactor)) {
            throw new IllegalArgumentException("Illegal eviction factor value:" + evictionFactor);
        }
        this.capacity = maxCapacity;
        this.evictionCount = (int)((float)this.capacity * evictionFactor);
        this.removeFreqEntryTimeout = removeFreqEntryTimeout;
        this.map = new HashMap<K, CacheNode<K, V>>();
        this.freqTable = new TreeMap<Long, CacheDeque<K, V>>(Long::compareTo);
        this.freqTable.put(1L, new CacheDeque());
    }

    public int getCapacity() {
        return this.capacity;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public V put(K key, V value) {
        CacheNode<K, V> node;
        this.lock.writeLock().lock();
        try {
            node = this.map.get(key);
            if (node != null) {
                CacheNode.withdrawNode(node);
                node.value = value;
                this.moveToNextFreqQueue(node.incrFreq(), node);
                this.map.put(key, node);
            } else {
                if (this.curSize + 1 > this.capacity) {
                    this.proceedEviction();
                }
                node = this.freqTable.get(1L).addLast(key, value);
                this.map.put(key, node);
                ++this.curSize;
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        return node.value;
    }

    public V remove(K key) {
        CacheNode<K, V> node = null;
        this.lock.writeLock().lock();
        try {
            if (this.map.containsKey(key)) {
                node = this.map.remove(key);
                if (node != null) {
                    CacheNode.withdrawNode(node);
                }
                --this.curSize;
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        return node != null ? (V)node.value : null;
    }

    public V get(K key) {
        CacheNode<K, V> node = null;
        this.lock.writeLock().lock();
        try {
            if (this.map.containsKey(key)) {
                node = this.map.get(key);
                CacheNode.withdrawNode(node);
                this.moveToNextFreqQueue(node.incrFreq(), node);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        return node != null ? (V)node.value : null;
    }

    public int getFreqTableSize() {
        return this.freqTable.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Long getFreq(K key) {
        CacheNode<K, V> node = null;
        this.lock.readLock().lock();
        try {
            if (this.map.containsKey(key)) {
                node = this.map.get(key);
                Long l = node.getFreq();
                return l;
            }
        }
        finally {
            this.lock.readLock().unlock();
        }
        return null;
    }

    private List<CacheNode<K, V>> getFreqList(Long freq) {
        if (freq == null) {
            return null;
        }
        this.lock.writeLock().lock();
        try {
            if (this.freqTable.containsKey(freq) && this.freqTable.get((Object)freq).nodeMap.size() > 0) {
                ArrayList arrayList = new ArrayList(this.freqTable.get((Object)freq).nodeMap.values());
                return arrayList;
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        return null;
    }

    public int getFreqListSize(Long freq) {
        if (freq == null) {
            return 0;
        }
        this.lock.writeLock().lock();
        try {
            if (this.freqTable.containsKey(freq)) {
                int n = this.freqTable.get((Object)freq).size.get();
                return n;
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        return 0;
    }

    private int proceedEviction() {
        int targetSize = this.capacity - this.evictionCount - 1;
        int evictedElements = 0;
        Set<Long> freqKeys = this.freqTable.keySet();
        boolean evictionEnd = false;
        for (Long freq : freqKeys) {
            CacheDeque<K, V> q = this.freqTable.get(freq);
            if (!evictionEnd) {
                while (!q.isEmpty()) {
                    CacheNode<K, V> node = q.pollFirst();
                    this.remove(node.key);
                    ++evictedElements;
                    if (targetSize < this.curSize) continue;
                    evictionEnd = true;
                    break;
                }
            }
            if (this.removeFreqEntryTimeout <= 0L || freq <= 1L || !q.isEmpty() || System.currentTimeMillis() - q.getLastReqTime() < this.removeFreqEntryTimeout) continue;
            this.freqTable.remove(freq);
        }
        return evictedElements;
    }

    private void moveToNextFreqQueue(long newFreq, CacheNode<K, V> node) {
        this.freqTable.putIfAbsent(newFreq, new CacheDeque());
        this.freqTable.get(newFreq).addLastNode(node);
    }

    public int getSize() {
        return this.curSize;
    }

    static class CacheDeque<K, V> {
        CacheNode<K, V> last;
        CacheNode<K, V> first;
        Map<K, CacheNode<K, V>> nodeMap;
        long lastReqTime;
        volatile AtomicInteger size = new AtomicInteger(0);

        CacheDeque() {
            this.last = new CacheNode();
            this.first = new CacheNode();
            this.last.next = this.first;
            this.first.prev = this.last;
            this.nodeMap = new HashMap<K, CacheNode<K, V>>();
        }

        CacheNode<K, V> addLast(K key, V value) {
            CacheNode<K, V> node = new CacheNode<K, V>(key, value);
            node.owner = this;
            node.next = this.last.next;
            node.prev = this.last;
            node.next.prev = node;
            this.last.next = node;
            this.setLastReqTime(System.currentTimeMillis());
            this.size.incrementAndGet();
            return node;
        }

        CacheNode<K, V> addLastNode(CacheNode<K, V> node) {
            node.owner = this;
            node.next = this.last.next;
            node.prev = this.last;
            node.next.prev = node;
            this.last.next = node;
            this.setLastReqTime(System.currentTimeMillis());
            this.nodeMap.put(node.key, node);
            this.size.incrementAndGet();
            return node;
        }

        CacheNode<K, V> pollFirst() {
            CacheNode node = null;
            if (this.first.prev != this.last) {
                node = this.first.prev;
                this.first.prev = node.prev;
                this.first.prev.next = this.first;
                node.prev = null;
                node.next = null;
                this.nodeMap.remove(node.key);
                this.size.decrementAndGet();
            }
            return node;
        }

        boolean isEmpty() {
            return this.last.next == this.first;
        }

        public CacheDeque<K, V> setLastReqTime(long lastReqTime) {
            this.lastReqTime = lastReqTime;
            return this;
        }

        public long getLastReqTime() {
            return this.lastReqTime;
        }
    }

    static class CacheNode<K, V> {
        CacheNode<K, V> prev;
        CacheNode<K, V> next;
        K key;
        V value;
        volatile AtomicLong freq = new AtomicLong(1L);
        CacheDeque<K, V> owner;

        CacheNode() {
        }

        CacheNode(K key, V value) {
            this.key = key;
            this.value = value;
        }

        long incrFreq() {
            return this.freq.incrementAndGet();
        }

        long getFreq() {
            return this.freq.get();
        }

        static <K, V> CacheNode<K, V> withdrawNode(CacheNode<K, V> node) {
            if (node != null && node.prev != null) {
                node.prev.next = node.next;
                if (node.next != null) {
                    node.next.prev = node.prev;
                    node.owner.nodeMap.remove(node.key);
                    node.owner.size.decrementAndGet();
                }
            }
            return node;
        }
    }
}

