/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */

package org.grails.events.gorm

import java.beans.Introspector

import groovy.transform.AutoFinal
import groovy.transform.CompileStatic

import org.springframework.context.ApplicationEvent

import grails.events.bus.EventBus
import org.grails.datastore.mapping.core.Datastore
import org.grails.datastore.mapping.engine.event.AbstractPersistenceEvent
import org.grails.datastore.mapping.engine.event.AbstractPersistenceEventListener

/**
 * Dispatches GORM events to the {@link EventBus}
 *
 * @author Graeme Rocher
 * @since 3.3
 */
@AutoFinal
@CompileStatic
class GormEventDispatcher extends AbstractPersistenceEventListener {

    private static final String GORM_NAMESPACE = 'gorm:'
    protected final EventBus eventBus
    protected final Map<Class<? extends AbstractPersistenceEvent>, String> subscribedEvents
    protected final Set<Class<? extends AbstractPersistenceEvent>> listenedForEvents = []
    protected final List<GormAnnotatedListener> listeners
    protected final boolean hasEventSubscribers
    protected final boolean hasListeners

    GormEventDispatcher(EventBus eventBus, Datastore datastore, Set<Class<? extends AbstractPersistenceEvent>> subscribedEvents, List<GormAnnotatedListener> listeners) {
        super(datastore)
        this.eventBus = eventBus
        Map<Class<? extends AbstractPersistenceEvent>, String> subscribedEventMap = [:]
        for (event in subscribedEvents) {
            subscribedEventMap.put(event, GORM_NAMESPACE + (Introspector.decapitalize(event.simpleName) - 'Event'))
        }
        this.subscribedEvents = Collections.unmodifiableMap(subscribedEventMap)
        this.listeners = Collections.unmodifiableList(listeners)
        this.hasListeners = !listeners.isEmpty()
        this.hasEventSubscribers = !subscribedEvents.isEmpty() || hasListeners
        if (hasListeners) {
            for (listener in listeners) {
                listenedForEvents.addAll(listener.subscribedEvents)
            }
        }
    }

    @Override
    protected void onPersistenceEvent(AbstractPersistenceEvent event) {
        if (hasListeners && listenedForEvents.contains(event.getClass())) {
            for (listener in listeners) {
                if (listener.supports(event)) {
                    listener.dispatch(event)
                }
            }
        }

        String eventName = subscribedEvents.get(event.getClass())
        if (eventName != null) {
            eventBus.notify(eventName, event)
        }
    }

    @Override
    boolean supportsSourceType(Class<?> sourceType) {
        return hasEventSubscribers && super.supportsSourceType(sourceType)
    }

    @Override
    boolean supportsEventType(Class<? extends ApplicationEvent> aClass) {
        return hasEventSubscribers &&
                AbstractPersistenceEvent.isAssignableFrom(aClass) &&
                (subscribedEvents.containsKey(aClass) || listenedForEvents.contains(aClass))
    }
}
