/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.utils;

import io.netty.buffer.ByteBuf;
import io.netty.util.internal.PlatformDependent;
import java.lang.invoke.MethodHandles;
import java.lang.ref.SoftReference;
import org.apache.activemq.artemis.api.core.ActiveMQBuffer;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.logs.ActiveMQUtilBundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class UTF8Util {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final ThreadLocal<SoftReference<StringUtilBuffer>> currentBuffer = new ThreadLocal();

    private UTF8Util() {
    }

    public static void writeNullableString(ByteBuf buffer, String val) {
        if (val == null) {
            buffer.writeByte(0);
        } else {
            buffer.writeByte(1);
            UTF8Util.writeString(buffer, val);
        }
    }

    private static void writeAsShorts(ByteBuf buffer, String val) {
        for (int i = 0; i < val.length(); ++i) {
            buffer.writeShort((int)((short)val.charAt(i)));
        }
    }

    public static void writeString(ByteBuf buffer, String val) {
        int length = val.length();
        buffer.writeInt(length);
        if (length < 9) {
            UTF8Util.writeAsShorts(buffer, val);
        } else if (length < 4095) {
            UTF8Util.saveUTF(buffer, val);
        } else {
            SimpleString.writeSimpleString(buffer, SimpleString.of(val));
        }
    }

    public static void saveUTF(ByteBuf out, String str) {
        if (str.length() > 65535) {
            throw ActiveMQUtilBundle.BUNDLE.stringTooLong(str.length());
        }
        int len = UTF8Util.calculateUTFSize(str);
        if (len > 65535) {
            throw ActiveMQUtilBundle.BUNDLE.stringTooLong(len);
        }
        out.writeShort((int)((short)len));
        int stringLength = str.length();
        if (logger.isTraceEnabled()) {
            logger.trace("Saving string with utfSize={} stringSize={}", (Object)len, (Object)stringLength);
        }
        if (out.hasArray()) {
            out.ensureWritable(len);
            byte[] bytes = out.array();
            int writerIndex = out.writerIndex();
            int index = out.arrayOffset() + writerIndex;
            if (PlatformDependent.hasUnsafe()) {
                UTF8Util.unsafeOnHeapWriteUTF(str, bytes, index, stringLength);
            } else {
                UTF8Util.writeUTF(str, bytes, index, stringLength);
            }
            out.writerIndex(writerIndex + len);
        } else if (PlatformDependent.hasUnsafe() && out.hasMemoryAddress()) {
            out.ensureWritable(len);
            long addressBytes = out.memoryAddress();
            int writerIndex = out.writerIndex();
            UTF8Util.unsafeOffHeapWriteUTF(str, addressBytes, writerIndex, stringLength);
            out.writerIndex(writerIndex + len);
        } else {
            StringUtilBuffer buffer = UTF8Util.getThreadLocalBuffer();
            byte[] bytes = buffer.borrowByteBuffer(len);
            UTF8Util.writeUTF(str, bytes, 0, stringLength);
            out.writeBytes(bytes, 0, len);
        }
    }

    private static int writeUTF(CharSequence str, byte[] bytes, int index, int length) {
        int charCount = index;
        for (int i = 0; i < length; ++i) {
            char charAtPos = str.charAt(i);
            if (charAtPos <= '\u007f') {
                bytes[charCount++] = (byte)charAtPos;
                continue;
            }
            if (charAtPos >= '\u0800') {
                bytes[charCount++] = (byte)(0xE0 | charAtPos >> 12 & 0xF);
                bytes[charCount++] = (byte)(0x80 | charAtPos >> 6 & 0x3F);
                bytes[charCount++] = (byte)(0x80 | charAtPos >> 0 & 0x3F);
                continue;
            }
            bytes[charCount++] = (byte)(0xC0 | charAtPos >> 6 & 0x1F);
            bytes[charCount++] = (byte)(0x80 | charAtPos >> 0 & 0x3F);
        }
        int writtenBytes = charCount - index;
        return writtenBytes;
    }

    private static int unsafeOnHeapWriteUTF(CharSequence str, byte[] bytes, int index, int length) {
        int charCount = index;
        for (int i = 0; i < length; ++i) {
            char charAtPos = str.charAt(i);
            if (charAtPos <= '\u007f') {
                PlatformDependent.putByte((byte[])bytes, (int)charCount++, (byte)((byte)charAtPos));
                continue;
            }
            if (charAtPos >= '\u0800') {
                PlatformDependent.putByte((byte[])bytes, (int)charCount++, (byte)((byte)(0xE0 | charAtPos >> 12 & 0xF)));
                PlatformDependent.putByte((byte[])bytes, (int)charCount++, (byte)((byte)(0x80 | charAtPos >> 6 & 0x3F)));
                PlatformDependent.putByte((byte[])bytes, (int)charCount++, (byte)((byte)(0x80 | charAtPos >> 0 & 0x3F)));
                continue;
            }
            PlatformDependent.putByte((byte[])bytes, (int)charCount++, (byte)((byte)(0xC0 | charAtPos >> 6 & 0x1F)));
            PlatformDependent.putByte((byte[])bytes, (int)charCount++, (byte)((byte)(0x80 | charAtPos >> 0 & 0x3F)));
        }
        int writtenBytes = charCount - index;
        return writtenBytes;
    }

    private static int unsafeOffHeapWriteUTF(CharSequence str, long addressBytes, int index, int length) {
        int charCount = index;
        for (int i = 0; i < length; ++i) {
            char charAtPos = str.charAt(i);
            if (charAtPos <= '\u007f') {
                PlatformDependent.putByte((long)(addressBytes + (long)charCount++), (byte)((byte)charAtPos));
                continue;
            }
            if (charAtPos >= '\u0800') {
                PlatformDependent.putByte((long)(addressBytes + (long)charCount++), (byte)((byte)(0xE0 | charAtPos >> 12 & 0xF)));
                PlatformDependent.putByte((long)(addressBytes + (long)charCount++), (byte)((byte)(0x80 | charAtPos >> 6 & 0x3F)));
                PlatformDependent.putByte((long)(addressBytes + (long)charCount++), (byte)((byte)(0x80 | charAtPos >> 0 & 0x3F)));
                continue;
            }
            PlatformDependent.putByte((long)(addressBytes + (long)charCount++), (byte)((byte)(0xC0 | charAtPos >> 6 & 0x1F)));
            PlatformDependent.putByte((long)(addressBytes + (long)charCount++), (byte)((byte)(0x80 | charAtPos >> 0 & 0x3F)));
        }
        int writtenBytes = charCount - index;
        return writtenBytes;
    }

    public static String readUTF(ActiveMQBuffer input) {
        int index;
        byte[] bytes;
        StringUtilBuffer buffer = UTF8Util.getThreadLocalBuffer();
        int size = input.readUnsignedShort();
        if (logger.isTraceEnabled()) {
            logger.trace("Reading string with utfSize={}", (Object)size);
        }
        if (PlatformDependent.hasUnsafe() && input.byteBuf() != null && input.byteBuf().hasMemoryAddress()) {
            ByteBuf byteBuf = input.byteBuf();
            long addressBytes = byteBuf.memoryAddress();
            int index2 = byteBuf.readerIndex();
            byteBuf.skipBytes(size);
            char[] chars = buffer.borrowCharBuffer(size);
            return UTF8Util.unsafeOffHeapReadUTF(addressBytes, index2, chars, size);
        }
        if (input.byteBuf() != null && input.byteBuf().hasArray()) {
            ByteBuf byteBuf = input.byteBuf();
            bytes = byteBuf.array();
            index = byteBuf.arrayOffset() + byteBuf.readerIndex();
            byteBuf.skipBytes(size);
        } else {
            bytes = buffer.borrowByteBuffer(size);
            index = 0;
            input.readBytes(bytes, 0, size);
        }
        char[] chars = buffer.borrowCharBuffer(size);
        if (PlatformDependent.hasUnsafe()) {
            return UTF8Util.unsafeOnHeapReadUTF(bytes, index, chars, size);
        }
        return UTF8Util.readUTF(bytes, index, chars, size);
    }

    private static String readUTF(byte[] bytes, int index, char[] chars, int size) {
        int count = index;
        int limit = index + size;
        int charCount = 0;
        block4: while (count < limit) {
            byte byte1;
            if ((byte1 = bytes[count++]) >= 0 && byte1 <= 127) {
                chars[charCount++] = (char)byte1;
                continue;
            }
            int c = byte1 & 0xFF;
            switch (c >> 4) {
                case 12: 
                case 13: {
                    byte byte2 = bytes[count++];
                    chars[charCount++] = (char)((c & 0x1F) << 6 | byte2 & 0x3F);
                    continue block4;
                }
                case 14: {
                    byte byte2 = bytes[count++];
                    byte byte3 = bytes[count++];
                    chars[charCount++] = (char)((c & 0xF) << 12 | (byte2 & 0x3F) << 6 | (byte3 & 0x3F) << 0);
                    continue block4;
                }
            }
            throw new InternalError("unhandled utf8 byte " + c);
        }
        return new String(chars, 0, charCount);
    }

    private static String unsafeOnHeapReadUTF(byte[] bytes, int index, char[] chars, int size) {
        int count = index;
        int limit = index + size;
        int charCount = 0;
        block4: while (count < limit) {
            byte byte1;
            if ((byte1 = PlatformDependent.getByte((byte[])bytes, (int)count++)) >= 0 && byte1 <= 127) {
                chars[charCount++] = (char)byte1;
                continue;
            }
            int c = byte1 & 0xFF;
            switch (c >> 4) {
                case 12: 
                case 13: {
                    byte byte2 = PlatformDependent.getByte((byte[])bytes, (int)count++);
                    chars[charCount++] = (char)((c & 0x1F) << 6 | byte2 & 0x3F);
                    continue block4;
                }
                case 14: {
                    byte byte2 = PlatformDependent.getByte((byte[])bytes, (int)count++);
                    byte byte3 = PlatformDependent.getByte((byte[])bytes, (int)count++);
                    chars[charCount++] = (char)((c & 0xF) << 12 | (byte2 & 0x3F) << 6 | (byte3 & 0x3F) << 0);
                    continue block4;
                }
            }
            throw new InternalError("unhandled utf8 byte " + c);
        }
        return new String(chars, 0, charCount);
    }

    private static String unsafeOffHeapReadUTF(long addressBytes, int index, char[] chars, int size) {
        int count = index;
        int limit = index + size;
        int charCount = 0;
        block4: while (count < limit) {
            byte byte1;
            if ((byte1 = PlatformDependent.getByte((long)(addressBytes + (long)count++))) >= 0 && byte1 <= 127) {
                chars[charCount++] = (char)byte1;
                continue;
            }
            int c = byte1 & 0xFF;
            switch (c >> 4) {
                case 12: 
                case 13: {
                    byte byte2 = PlatformDependent.getByte((long)(addressBytes + (long)count++));
                    chars[charCount++] = (char)((c & 0x1F) << 6 | byte2 & 0x3F);
                    continue block4;
                }
                case 14: {
                    byte byte2 = PlatformDependent.getByte((long)(addressBytes + (long)count++));
                    byte byte3 = PlatformDependent.getByte((long)(addressBytes + (long)count++));
                    chars[charCount++] = (char)((c & 0xF) << 12 | (byte2 & 0x3F) << 6 | (byte3 & 0x3F) << 0);
                    continue block4;
                }
            }
            throw new InternalError("unhandled utf8 byte " + c);
        }
        return new String(chars, 0, charCount);
    }

    private static StringUtilBuffer getThreadLocalBuffer() {
        StringUtilBuffer value;
        SoftReference<StringUtilBuffer> softReference = currentBuffer.get();
        if (softReference == null) {
            value = new StringUtilBuffer();
            softReference = new SoftReference<StringUtilBuffer>(value);
            currentBuffer.set(softReference);
        } else {
            value = softReference.get();
        }
        if (value == null) {
            value = new StringUtilBuffer();
            softReference = new SoftReference<StringUtilBuffer>(value);
            currentBuffer.set(softReference);
        }
        return value;
    }

    public static void clearBuffer() {
        SoftReference<StringUtilBuffer> ref = currentBuffer.get();
        if (ref != null && ref.get() != null) {
            ref.clear();
        }
    }

    public static int calculateUTFSize(String str) {
        int calculatedLen = 0;
        int stringLength = str.length();
        for (int i = 0; i < stringLength; ++i) {
            char c = str.charAt(i);
            if (c <= '\u007f') {
                ++calculatedLen;
                continue;
            }
            if (c >= '\u0800') {
                calculatedLen += 3;
                continue;
            }
            calculatedLen += 2;
        }
        return calculatedLen;
    }

    private static final class StringUtilBuffer {
        private char[] charBuffer = null;
        private byte[] byteBuffer = null;

        private StringUtilBuffer() {
        }

        public char[] borrowCharBuffer(int newSize) {
            if (this.charBuffer == null || newSize > this.charBuffer.length) {
                this.charBuffer = new char[newSize];
            }
            return this.charBuffer;
        }

        public byte[] borrowByteBuffer(int newSize) {
            if (this.byteBuffer == null || newSize > this.byteBuffer.length) {
                this.byteBuffer = new byte[newSize];
            }
            return this.byteBuffer;
        }
    }
}

