
/* mkbin.c - generate binary version from radio text file and template.

   Copyright (c) 1997 Riku Kalinen, OH2LWO, oh2lwo@sral.fi

   RCS $Header: /home/mole/riku/CVS/yaesu/ft50mkbin.c,v 1.2 1997/12/11 15:01:50 riku Exp $ */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "yaesu.h"

/* Generate debug output? */

int debug = 0;

/* Data buffers. */

unsigned char b1[10], b2[16], b3[112], b4[16], b5[16], b6[1776], b7[1776];
unsigned char b8[1];

/* Set channel name. */

static int nameset (unsigned char *base, char *name)
{
  int i, j;
  
  if (strlen (name) != 4)
    return -1;

  for (j = 0; j < 4; j++) {
    i = char_bin (name[j]);
    if (i < 0)
      return -1;
    base[j] = (unsigned char) (i & 0xff);
  }

  return 0;
}

static int power_step_set (char *power, char *step)
{
  int ret = 0, i;

  i = txpo_bin (power);
  if (i < 0) return -1;
  ret |= i << 4;

  i = step_bin (step);
  if (i < 0) return -2;
  ret |= i;

  return ret;
}

/* Set binary dtmf, paging and shift. */

static int dtmf_pg_shift_set (char *dtmf, char *page, char *shift)
{
  int ret = 0, i;

  i = code_bin (dtmf);
  if (i < 0) return -1;
  ret |= i << 5;

  i = paging_bin (page);
  if (i < 0) return -2;
  ret |= i << 2;

  i = shift_bin (shift);
  if (i < 0) return -3;
  ret |= i;

  return ret;
}

static int sel_ctcss_set (char *sel, char *ctcss)
{
  int ret = 0, i;

  i = selcal_bin (sel);
  if (i < 0) return -1;
  ret |= i << 6;

  i = ctcss_bin (ctcss);
  if (i < 0) return -2;
  ret |= i;

  return ret;
}

/* Error codes for hook functions:
   -1: General error
   -2: Argument count error
   -10: Tag error
   -11: Parameter 1 error
   ...
   -23: Parameter 13 error */

/* Set one channel slot. */

static int chanset (unsigned char *base, char *power_s, char *step_s,
		    char *dtmf_s, char *page_s, char *shift_s, char *sel_s,
		    char *ctcss_s, char *dcs_s, char *mode_s, char *rx_s,
		    char *txoff_s, char *name_s)
{
  int ret, i;
  unsigned char freq[8];
  
  if (name_s) {
    base[0] = 0x80;
    ret = nameset (base + 12, name_s);
    if (ret < 0) return -12;
  } else {
    base[0] = 0x00;
    memset (base + 12, 0x24, 4);
  }

  i = power_step_set (power_s, step_s);
  if (i < 0) return i;
  base[1] = (unsigned char) (i & 0xff);
  
  i = dtmf_pg_shift_set (dtmf_s, page_s, shift_s);
  if (i < 0) {
    if (debug) printf ("chanset: dtmf_pg_shift_set %s %s %s returned %d\n",
		       dtmf_s, page_s, shift_s, i);
    return -2 + i;
  }
  base[2] = (unsigned char) (i & 0xff);
  
  i = sel_ctcss_set (sel_s, ctcss_s);
  if (i < 0) return -5 + i;
  base[3] = (unsigned char) (i & 0xff);
  
  i = dcs_bin (dcs_s);
  if (i < 0) return -8;
  base[4] = (unsigned char) (i & 0xff);
  
  i = mode_bin (mode_s);
  if (i < 0) return -9;
  base[5] = (unsigned char) (i & 0xff);

  if (freq_bin (freq, rx_s) != 0) return -10;
  (void) memcpy (base + 6, freq, 3);
  
  if (freq_bin (freq, txoff_s) != 0) return -11;
  (void) memcpy (base + 9, freq, 3);
  
  return 0;
}

#define DEFUN(NAME) int NAME (char *tag, char *d1, char *d2, char *d3, \
			      char *d4, char *d5, char *d6, char *d7, \
			      char *d8, char *d9, char *d10, char *d11, \
			      char *d12, char *d13)

#define DEBUG(NAME) if (debug) \
printf ("%s: %s %s %s %s %s %s %s %s %s %s %s %s %s %s\n", NAME, tag, \
	d1 ? d1 : "NULL", d2 ? d2 : "NULL", d3 ? d3 : "NULL", \
	d4 ? d4 : "NULL", d5 ? d5 : "NULL", d6 ? d6 : "NULL", \
	d7 ? d7 : "NULL", d8 ? d8 : "NULL", d9 ? d9 : "NULL", \
	d10 ? d10 : "NULL", d11 ? d11 : "NULL", d12 ? d12 : "NULL", \
	d13 ? d13 : "NULL")

#if 0
/* Set current VFO (A or B). */

static DEFUN(set_curr_vfo)
{
  unsigned char uc;

  DEBUG("set_curr_vfo");

  if (d1 == NULL) return -2;

  switch (*d1) {
  case 'A': uc = 0x06; break;
  case 'B': uc = 0x00; break;
  default: return -11; break;
  }

  b7[1662] = uc;

  return 0;
}
#endif

/* Set current VFO numbers. */

static DEFUN(set_curr_vfox)
{
  int i, no;
  char c = tag[8];

  DEBUG("set_curr_vfox");

  switch (c) {
  case 'A': no = 0; break;
  case 'B': no = 5; break;
  default: return -10; break;
  }

  if (d1 == NULL) return -2;

  i = vfo_bin (d1);
  if ((i < no) || (i > no+4)) return -11;

  if (no == 0)
    b7[1660] = i;
  else
    b7[1661] = i - no;

  return 0;
}

/* Set current data. */

static DEFUN(set_curr)
{
  int i;

  DEBUG("set_curr");

  if (d12 == NULL) return -2;
  
  i = txpo_bin (d2);
  if (i < 0) return -12;
  b7[1663] = (unsigned char) (i & 0xff);

  i = step_bin (d3);
  if (i < 0) return -13;
  b7[1664] = (unsigned char) (i & 0xff);

  i = code_bin (d4);
  if (i < 0) return -14;
  b7[1665] = (unsigned char) (i & 0xff);

  i = paging_bin (d5);
  if (i < 0) return -15;
  b7[1666] = (unsigned char) (i & 0xff);

  i = shift_bin (d6);
  if (i < 0) return -16;
  b7[1667] = (unsigned char) (i & 0xff);

  i = selcal_bin (d7);
  if (i < 0) return -17;
  b7[1668] = (unsigned char) (i & 0xff);

  i = ctcss_bin (d8);
  if (i < 0) return -18;
  b7[1669] = (unsigned char) (i & 0xff);

  i = dcs_bin (d9);
  if (i < 0) return -19;
  b7[1670] = (unsigned char) (i & 0xff);

  i = mode_bin (d10);
  if (i < 0) return -20;
  b7[1671] = (unsigned char) (i & 0xff);

  if (freq_bin (b7 + 1672, d11) != 0) return -21;
  if (freq_bin (b7 + 1676, d12) != 0) return -22;

  return 0;
  
}

/* Set home channel. */

static DEFUN(set_xhfh)
{
  char chan = tag[0];
  unsigned char *base;
  int i;

  DEBUG("set_xhfh");

  if (d12 == NULL) return -2;
  
  switch (chan) {
  case 'V': base = b4; break;
  case 'U': base = b5; break;
  default: return -10;
  }
  
  i = chanset (base, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11, d12, d13);
  if (i < 0)
    return i - 11;
  else
    return 0;
  
}

/* Set memory channel. */
 
static DEFUN(set_m)
{
  int inx, ret = 0, i;
  unsigned char flags, *base;
  char name[8], *cp;

  DEBUG("set_m");

  if (d12 == NULL) return -2;
  
  (void) strcpy (name, tag);
  for (cp = name; *cp; cp ++)
    if (*cp == ':') *cp = 0;
  
  inx = chname_bin (name);
  if (inx < 0) {
    if (debug) printf ("set_m: chname_bin %s returned %d\n", name, inx);
    return -10;
  }

  i = flag_bin (d1);
  if (i < 0)
    return -11;
  flags = (unsigned char) (i & 0xff);
  
  base = b6 + inx * 16;

  /* If txpo == "XX", we just fill data slot with 0xff. */
  if (strcasecmp ("XX", b2) == 0) {
    memset (base, 0xff, 16);
  } else {
    /* Do not touch it unless marked used. */
    if (flags & 0x01)
      ret = chanset (base, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11, d12, d13);
  }

  /* For some mysterious reason, channel flags are duplicated. */
  b3[inx] = flags;
  b7[inx + 2] = flags;

  if (ret < 0)
    return ret - 11;
  else
    return 0;
}
 
/* Set VFO. */
 
static DEFUN(set_v)
{
  unsigned char *base = b7 + 116;
  int i;
  
  DEBUG("set_v");

  if (d12 == NULL) return -2;
  
  switch (tag[0]) {
  case 'A': break;
  case 'B': base += 5 * 16; break;
  default: return -10;
  }
  
  switch (tag[1]) {
  case '1': break;
  case '2': base += 16; break;
  case '3': base += 16 * 2; break;
  case '4': base += 16 * 3; break;
  case '8': base += 16 * 4; break;
  default: return -10;
  }

  i = chanset (base, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11, d12, d13);
  if (i < 0)
    return i - 11;
  else
    return 0;
  
}
 
/* Set SUB. */
 
static DEFUN(set_sub)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_sub");

  if (d1 == NULL) return -2;
  
  i = sub_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[301] = uc;
  return 0;
}
 
/* Set APO. */
 
static DEFUN(set_apo)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_apo");

  if (d1 == NULL) return -2;
  
  i = apo_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[303] = uc;
  return 0;
}
 
/* Set TOT. */
 
static DEFUN(set_tot)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_tot");

  if (d1 == NULL) return -2;
  
  i = tot_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[304] = uc;
  return 0;
}
 
/* Set LOCK. */
 
static DEFUN(set_lock)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_lock");

  if (d1 == NULL) return -2;
  
  i = lock_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[305] = uc;
  return 0;
}
 
/* Set RSAV. */
 
static DEFUN(set_rsav)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_rsav");

  if (d1 == NULL) return -2;
  
  i = rsav_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[306] = uc;
  return 0;
}
 
/* Set LAMP. */
 
static DEFUN(set_lamp)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_lamp");

  if (d1 == NULL) return -2;
  
  i = lamp_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[307] = uc;
  return 0;
}
 
/* Set BELL. */
 
static DEFUN(set_bell)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_bell");

  if (d1 == NULL) return -2;
  
  i = bell_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[308] = uc;
  return 0;
}
 
/* Set ID. */
 
static DEFUN(set_id)
{
  int i, j, ret = 0;
  unsigned char *base = b7 + 309;
  char newid[32];
  
  DEBUG("set_id");

  if (d1 == NULL)
    strcpy (newid, "");
  else
    strcpy (newid, d1);
  
  for (j = 0; j < 16; j ++) {
    if (newid[j] == 0)
      break;
    i = char_bin (newid[j]);
    if (i < 0) ret = -11;
    base[j] = (unsigned char) (i & 0xff);
  }
  for (; j < 16; j ++)
    base[j] = 0xff;

  return ret;
}
 
/* Set ARTS. */
 
static DEFUN(set_arts)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_arts");

  if (d1 == NULL) return -2;
  
  i = arts_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[326] = uc;
  return 0;
}
 
/* Set ARTS/BEEP. */
 
static DEFUN(set_arts_beep)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_arts_beep");

  if (d1 == NULL) return -2;
  
  i = arts_beep_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[327] = uc;
  return 0;
}
 
/* Set autodial memories. */
 
static DEFUN(set_ad)
{
  unsigned char *base;
  int no, i, j, ret = 0, limit = 16;
  char newad[64];
    
  DEBUG("set_ad");

  if (d1 == NULL)
    strcpy (newad, "");
  else
    strcpy (newad, d1);
  
  no = atoi (tag + 2);
  if (no == 9) limit = 32;
  base = b7 + 330 + 4 + ((no - 1) * 20);

  for (j = 0; j < limit; j ++) {
    if (newad[j] == 0)
      break;
    i = dtmf_bin (newad[j]);
    if (i < 0) ret = -11;
    base[j] = (unsigned char) (i & 0xff);
  }
  for (; j < limit; j ++)
    base[j] = 0xff;

  return ret;
}
 
/* Set DTMF (code) memories. */
 
static DEFUN(set_co)
{
  int inx, i, j, decode = 0;
  unsigned char f0 = 0, f1 = 0, uc;
  
  DEBUG("set_co");

  if (d2 == NULL) return -2;

  switch (tag[1]) {
  case 'C': inx = 0; break;
  case 'P': inx = 1; break;
  case '1': inx = 2; break;
  case '2': inx = 3; break;
  case '3': inx = 4; break;
  case '4': inx = 5; break;
  case '5': inx = 6; break;
  case '6': inx = 7; break;
  default: return -10;
  }

  if (*d1 == 'D')
    decode = 1;

  if (strlen (d2) != 3) return -12;

  for (j = 0; j < 3; j ++) {
    if (! isdigit (d2[j])) return -12;
    i = dtmf_bin (d2[j]);
    if (i < 0) return -12;
    uc = (unsigned char) (i & 0xff);
    switch (j) {
    case 0: f0 = uc & 0xf; break;
    case 1: f1 |= (uc & 0xf) << 4; break;
    case 2: f1 |= uc & 0xf; break;
    default: return -12; break;
    }
  }

  b7[526 + inx * 2] = f0;
  b7[526 + inx * 2 + 1] = f1;

  setbit (b7 + 542, inx, decode);
  
  return 0;
}
 
/* Set PAGE/SPED. */
 
static DEFUN(set_page_sped)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_page_sped");

  if (d1 == NULL) return -2;
  
  i = page_sped_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[543] = uc;
  return 0;
}
 
/* Set PAGE/DLAY. */
 
static DEFUN(set_page_dlay)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_page_dlay");

  if (d1 == NULL) return -2;
  
  i = page_dlay_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[544] = uc;
  return 0;
}
 
/* Set PAGE/BELL. */
 
static DEFUN(set_page_bell)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_page_bell");

  if (d1 == NULL) return -2;
  
  i = bell_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[545] = uc;
  return 0;
}
 
/* Set PAGE/ASBK. */
 
static DEFUN(set_page_asbk)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_page_asbk");

  if (d1 == NULL) return -2;
  
  i = page_asbk_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[546] = uc;
  return 0;
}
 
/* Set SQL. */
 
static DEFUN(set_sql)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_sql");

  if (d1 == NULL) return -2;
  
  i = sql_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[1686] = uc;
  return 0;
}
 
/* Set WSQL. */
 
static DEFUN(set_wsql)
{
  int i;
  unsigned char uc;
  
  DEBUG("set_wsql");

  if (d1 == NULL) return -2;
  
  i = wsql_bin (d1);
  if (i < 0) return -11;

  uc = (unsigned char) (i & 0xff);

  b7[1687] = uc;
  return 0;
}
 
/* Set RPTL. */
 
static DEFUN(set_rptl)
{
  int i;
  
  DEBUG("set_rptl");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1688, 7, i);

  return 0;
}
 
/* Set AMOD. */
 
static DEFUN(set_amod)
{
  int i;
  
  DEBUG("set_amod");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1688, 6, i);

  return 0;
}
 
/* Set SCNL. */
 
static DEFUN(set_scnl)
{
  int i;
  
  DEBUG("set_scnl");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1688, 5, i);

  return 0;
}
 
/* Set RESM. */
 
static DEFUN(set_resm)
{
  int i;
  
  DEBUG("set_resm");

  if (d1 == NULL) return -2;
  
  i = carr5sec_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1688, 4, i);

  return 0;
}
 
/* Set ARS. */
 
static DEFUN(set_ars)
{
  int i;
  
  DEBUG("set_ars");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1688, 3, i);

  return 0;
}
 
/* Set BEEP. */
 
static DEFUN(set_beep)
{
  int i;
  
  DEBUG("set_beep");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1688, 2, i);

  return 0;
}
 
/* Set LCK. */
 
static DEFUN(set_lck)
{
  int i;
  
  DEBUG("set_lck");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1688, 1, i);

  return 0;
}
 
/* Set LGT. */
 
static DEFUN(set_lgt)
{
  int i;
  
  DEBUG("set_lgt");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1689, 7, i);

  return 0;
}
 
/* Set PAGE/AMSG. */
 
static DEFUN(set_page_amsg)
{
  int i;
  
  DEBUG("set_page_amsg");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1689, 6, i);

  return 0;
}
 
/* Set BCLO. */
 
static DEFUN(set_bclo)
{
  int i;
  
  DEBUG("set_bclo");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1689, 4, i);

  return 0;
}
 
/* Set CWID. */
 
static DEFUN(set_cwid)
{
  int i;
  
  DEBUG("set_cwid");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1689, 1, i);

  return 0;
}
 
/* Set CWID. */
 
static DEFUN(set_tsav)
{
  int i;
  
  DEBUG("set_tsav");

  if (d1 == NULL) return -2;
  
  i = onoff_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1689, 0, i);

  return 0;
}
 
/* Set ARTS/SPED. */
 
static DEFUN(set_arts_sped)
{
  int j, i;
  
  DEBUG("set_arts_sped");

  if (d1 == NULL) return -2;
  
  j = atoi (d1);
  switch (j) {
  case 15: i = 0; break;
  case 25: i = 1; break;
  default: return -11; break;
  }

  setbit (b7 + 1690, 3, i);

  return 0;
}
 
/* Set RVHM. */
 
static DEFUN(set_rvhm)
{
  int i;
  
  DEBUG("set_rvhm");

  if (d1 == NULL) return -2;
  
  i = revhome_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1690, 1, i);

  return 0;
}
 
/* Set MON. */
 
static DEFUN(set_mon)
{
  int i;
  
  DEBUG("set_mon");

  if (d1 == NULL) return -2;
  
  i = montcal_bin (d1);
  if (i < 0) return -11;

  setbit (b7 + 1690, 0, i);

  return 0;
}
 
/* Tag table. This binds line tags and corresponding function together. */

typedef struct tagfunc {
  char *tag;
  int (*func)(char *, char *, char *, char *, char *, char *, char *, char *,
	      char *, char *, char *, char *, char *, char *);
} Tag;

Tag tags[] = {
  /* Home channels */
  { "VHFH:", set_xhfh }, { "UHFH:", set_xhfh },
  /* Memories */
  { "M1:", set_m }, { "M2:", set_m }, { "M3:", set_m }, { "M4:", set_m },
  { "M5:", set_m }, { "M6:", set_m }, { "M7:", set_m }, { "M8:", set_m },
  { "M9:", set_m }, { "M10:", set_m }, { "M11:", set_m }, { "M12:", set_m },
  { "M13:", set_m }, { "M14:", set_m }, { "M15:", set_m }, { "M16:", set_m },
  { "M17:", set_m }, { "M18:", set_m }, { "M19:", set_m }, { "M20:", set_m },
  { "M21:", set_m }, { "M22:", set_m }, { "M23:", set_m }, { "M24:", set_m },
  { "M25:", set_m }, { "M26:", set_m }, { "M27:", set_m }, { "M28:", set_m },
  { "M29:", set_m }, { "M30:", set_m }, { "M31:", set_m }, { "M32:", set_m },
  { "M33:", set_m }, { "M34:", set_m }, { "M35:", set_m }, { "M36:", set_m },
  { "M37:", set_m }, { "M38:", set_m }, { "M39:", set_m }, { "M40:", set_m },
  { "M41:", set_m }, { "M42:", set_m }, { "M43:", set_m }, { "M44:", set_m },
  { "M45:", set_m }, { "M46:", set_m }, { "M47:", set_m }, { "M48:", set_m },
  { "M49:", set_m }, { "M50:", set_m }, { "M51:", set_m }, { "M52:", set_m },
  { "M53:", set_m }, { "M54:", set_m }, { "M55:", set_m }, { "M56:", set_m },
  { "M57:", set_m }, { "M58:", set_m }, { "M59:", set_m }, { "M60:", set_m },
  { "M61:", set_m }, { "M62:", set_m }, { "M63:", set_m }, { "M64:", set_m },
  { "M65:", set_m }, { "M66:", set_m }, { "M67:", set_m }, { "M68:", set_m },
  { "M69:", set_m }, { "M70:", set_m }, { "M71:", set_m }, { "M72:", set_m },
  { "M73:", set_m }, { "M74:", set_m }, { "M75:", set_m }, { "M76:", set_m },
  { "M77:", set_m }, { "M78:", set_m }, { "M79:", set_m }, { "M80:", set_m },
  { "M81:", set_m }, { "M82:", set_m }, { "M83:", set_m }, { "M84:", set_m },
  { "M85:", set_m }, { "M86:", set_m }, { "M87:", set_m }, { "M88:", set_m },
  { "M89:", set_m }, { "M90:", set_m }, { "M91:", set_m }, { "M92:", set_m },
  { "M93:", set_m }, { "M94:", set_m }, { "M95:", set_m }, { "M96:", set_m },
  { "M97:", set_m }, { "M98:", set_m }, { "M99:", set_m }, { "M100:", set_m },
  { "ML1:", set_m }, { "MU1:", set_m }, { "ML2:", set_m }, { "MU2:", set_m },
  { "ML3:", set_m }, { "MU3:", set_m }, { "ML4:", set_m }, { "MU4:", set_m },
  { "ML5:", set_m }, { "MU5:", set_m },
  /* VFOs */
  { "A145:", set_v }, { "A220:", set_v }, { "A380:", set_v },
  { "A430:", set_v }, { "A800:", set_v }, { "B145:", set_v },
  { "B220:", set_v }, { "B380:", set_v }, { "B430:", set_v },
  { "B800:", set_v },
  /* Misc. data */
  { "SUB:", set_sub }, { "APO:", set_apo }, { "TOT:", set_tot },
  { "LOCK:", set_lock }, { "RSAV:", set_rsav }, { "LAMP:", set_lamp },
  { "BELL:", set_bell }, { "ID:", set_id }, { "ARTS:", set_arts },
  { "ARTS/BEEP:", set_arts_beep },
  /* Autodial memories */
  { "AD1:", set_ad }, { "AD2:", set_ad }, { "AD3:", set_ad },
  { "AD4:", set_ad }, { "AD5:", set_ad }, { "AD6:", set_ad },
  { "AD7:", set_ad }, { "AD8:", set_ad }, { "AD9:", set_ad },
  /* DTMF (code) memories */
  { "DC:", set_co }, { "DP:", set_co }, { "D1:", set_co }, { "D2:", set_co },
  { "D3:", set_co }, { "D4:", set_co }, { "D5:", set_co }, { "D6:", set_co },
  /* Misc. data continues. */
  { "PAGE/SPED:", set_page_sped }, { "PAGE/DLAY:", set_page_dlay }, 
  { "PAGE/BELL:", set_page_bell }, { "PAGE/ASBK:", set_page_asbk },
  { "SQL:", set_sql }, { "WSQL:", set_wsql }, { "RPTL:", set_rptl },
  { "AMOD:", set_amod }, { "SCNL:", set_scnl }, { "RESM:", set_resm },
  { "ARS:", set_ars }, { "BEEP:", set_beep }, { "LCK:", set_lck },
  { "LGT:", set_lgt }, { "PAGE/AMSG:", set_page_amsg },
  { "BCLO:", set_bclo }, { "CWID:", set_cwid }, { "TSAV:", set_tsav },
  { "ARTS/SPED:", set_arts_sped }, { "RVHM:", set_rvhm }, { "MON:", set_mon },
  /* 'Current' information. */
  { "CURR/VFOA:", set_curr_vfox }, { "CURR/VFOB:", set_curr_vfox },
#if 0 /* Uncertain where this actually is? */
  { "CURR/VFO:", set_curr_vfo },
#endif
  { "CURR:", set_curr },
  /* Has to end with pair of nulls. Scanning loop depends on this. */
  { 0, 0 }
};

int main (int argc, char *argv[])
{
  int ret, fd_temp, fd_out, i, matches, lineno = 0;
  FILE *fp_in;
  unsigned char cksum;
  char inbuf[512], saved[512];
  char *cp, *new_inbuf, *tag;
  char *d1, *d2, *d3, *d4, *d5, *d6, *d7, *d8, *d9, *d10, *d11, *d12, *d13;
 
  if (argc != 4) {
    printf ("Usage: mkbin <infile> <template> <outfile>\n");
    exit (1);
  }

  fp_in = fopen (argv[1], "r");
  if (! fp_in) {
    perror (argv[1]);
    exit (1);
  }

  fd_temp = open (argv[2], O_RDONLY);
  if (fd_temp < 0) {
    perror (argv[2]);
    exit (1);
  }

  /* Read template.  Basically every line from user modifies this data. */

  ret = read_save_file (fd_temp, b1, b2, b3, b4, b5, b6, b7, b8);
  if (ret != 0) {
    perror ("read_save_file");
    exit (1);
  }

  cksum = calculate_cksum (b1, b2, b3, b4, b5, b6, b7);

  if (cksum != b8[0]) {
    printf ("Checksum error!\n");
    printf ("Calculated checksum = %02x, file cheksum = %02x\n",
	    cksum, b8[0]);
    printf ("Aborting.\n");
    exit (1);
  }

  /* Read data from user and act accordingly. */

  lineno = 0;
  while (fgets (inbuf, sizeof (inbuf), fp_in)) {

    /* Save buffer for later reference. */
    (void) strcpy (saved, inbuf);
    lineno ++;

    /* Trim leading .. */
    new_inbuf = inbuf;
    for (cp = inbuf; *cp; cp ++)
      if (isspace (*cp))
	new_inbuf = cp + 1;
      else
	break;
    
    /* .. and trailing white space. */
    for (cp = new_inbuf; *cp; cp ++) ;
    for (cp --; cp > new_inbuf; cp --)
      if (isspace (*cp))
	*cp = 0;
      else
	break;
    
    /* Strip comments. */
    for (cp = new_inbuf; *cp; cp ++)
      if (*cp == '#') {
	*cp = 0;
	break;
      }
    
    /* Skip empty lines. */
    if (*new_inbuf == 0)
      continue;
    
    /* Just skip hex dump lines. */
    if ((new_inbuf[0] == 'b') && isdigit (new_inbuf[1])
	&& (new_inbuf[2] == ' ') && isxdigit (new_inbuf[3])
	&& isxdigit (new_inbuf[4]) && isxdigit (new_inbuf[5]))
      continue;
    
    /* At this point, we should have valid input.  Modify template data
       accordingly. */
    if (debug)
      printf ("%5d %s\n", lineno, new_inbuf);

    /* Separate tag and data. */
    tag = strtok (new_inbuf, " \t\n");
    d1 = strtok (NULL, " \t\n"); d2 = strtok (NULL, " \t\n");
    d3 = strtok (NULL, " \t\n"); d4 = strtok (NULL, " \t\n");
    d5 = strtok (NULL, " \t\n"); d6 = strtok (NULL, " \t\n");
    d7 = strtok (NULL, " \t\n"); d8 = strtok (NULL, " \t\n");
    d9 = strtok (NULL, " \t\n"); d10 = strtok (NULL, " \t\n");
    d11 = strtok (NULL, " \t\n"); d12 = strtok (NULL, " \t\n");
    d13 = strtok (NULL, " \t\n");

    if (! tag) {
      printf ("Ignored: No valid tag on line %d >%s", lineno, saved);
      continue;
    }

    /* Go thru tag table and try to find a match. */
    for (i = 0, matches = 0; tags[i].tag; i ++) {
      if (strcasecmp (tags[i].tag, tag) == 0) {
	/* match. */
	matches ++;
	ret = (*tags[i].func)(tag, d1, d2, d3, d4, d5, d6, d7, d8, d9, d10,
			      d11, d12, d13);
	if (ret != 0) {
	  printf ("Problem with line %d >%s",
		  lineno, saved);
	  switch (ret) {
	  case -1:
	    printf ("Unspecified error.\n");
	    break;
	  case -2:
	    printf ("Wrong number of arguments in line.\n");
	    break;
	  case -10:
	    printf ("Problem with tag (%s).\n", tag);
	    break;
	  case -11:
	  case -12:
	  case -13:
	  case -14:
	  case -15:
	  case -16:
	  case -17:
	  case -18:
	  case -19:
	  case -20:
	  case -21:
	  case -22:
	  case -23:
	    printf ("Error in argument %d.\n", -10 - ret);
	    break;
	  default:
	    printf ("Unknown error.\n");
	    break;
	  }
	  printf ("Aborting..\n");
	  exit (1);
	}
      }
    }
    switch (matches) {
    case 0:
      printf ("Ignored: Tag invalid, line %d >%s", lineno, saved);
      break;
    case 1:
      if (debug)
	printf ("Line %d processed ok\n", lineno);
      break;
    default:
      printf ("Warning: Processed more than once (%d times), line %d >%s",
	      lineno, matches, saved);
      break;
    }
    
  }

  fclose (fp_in);

  /* Save template as modified per user data. */
  
  fd_out = open (argv[3], O_WRONLY | O_TRUNC | O_CREAT, 0666);
  if (fd_out < 0) {
    perror (argv[1]);
    printf ("Failed to open savefile %s. Aborting.\n", argv[3]);
    exit (1);
  }

  b8[0] = calculate_cksum (b1, b2, b3, b4, b5, b6, b7);

  ret = write_save_file (fd_out, b1, b2, b3, b4, b5, b6, b7, b8);
  if (ret != 0) {
    perror ("write_save_file");
    exit (1);
  }

  close (fd_out);

  exit (0);
  
}
