/*
 * s2mps11.c
 *
 * Copyright (c) 2012-2014 Samsung Electronics Co., Ltd
 *              http://www.samsung.com
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/bug.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/regmap.h>
#include <linux/platform_device.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#include <linux/regulator/of_regulator.h>
#include <linux/of_gpio.h>
#include <linux/mfd/samsung/core.h>
#include <linux/mfd/samsung/s2mps11.h>
#include <linux/mfd/samsung/s2mps14.h>
#include <linux/mfd/samsung/s2mpu02.h>

struct s2mps11_info {
	unsigned int rdev_num;
	int ramp_delay2;
	int ramp_delay34;
	int ramp_delay5;
	int ramp_delay16;
	int ramp_delay7810;
	int ramp_delay9;

	enum sec_device_type dev_type;

	/*
	 * One bit for each S2MPS14/S2MPU02 regulator whether the suspend mode
	 * was enabled.
	 */
	unsigned long long s2mps14_suspend_state:35;

	/* Array of size rdev_num with GPIO-s for external sleep control */
	int *ext_control_gpio;
};

static int get_ramp_delay(int ramp_delay)
{
	unsigned char cnt = 0;

	ramp_delay /= 6250;

	while (true) {
		ramp_delay = ramp_delay >> 1;
		if (ramp_delay == 0)
			break;
		cnt++;
	}

	if (cnt > 3)
		cnt = 3;

	return cnt;
}

static int s2mps11_regulator_set_voltage_time_sel(struct regulator_dev *rdev,
				   unsigned int old_selector,
				   unsigned int new_selector)
{
	struct s2mps11_info *s2mps11 = rdev_get_drvdata(rdev);
	unsigned int ramp_delay = 0;
	int old_volt, new_volt;

	switch (rdev_get_id(rdev)) {
	case S2MPS11_BUCK2:
		ramp_delay = s2mps11->ramp_delay2;
		break;
	case S2MPS11_BUCK3:
	case S2MPS11_BUCK4:
		ramp_delay = s2mps11->ramp_delay34;
		break;
	case S2MPS11_BUCK5:
		ramp_delay = s2mps11->ramp_delay5;
		break;
	case S2MPS11_BUCK6:
	case S2MPS11_BUCK1:
		ramp_delay = s2mps11->ramp_delay16;
		break;
	case S2MPS11_BUCK7:
	case S2MPS11_BUCK8:
	case S2MPS11_BUCK10:
		ramp_delay = s2mps11->ramp_delay7810;
		break;
	case S2MPS11_BUCK9:
		ramp_delay = s2mps11->ramp_delay9;
	}

	if (ramp_delay == 0)
		ramp_delay = rdev->desc->ramp_delay;

	old_volt = rdev->desc->min_uV + (rdev->desc->uV_step * old_selector);
	new_volt = rdev->desc->min_uV + (rdev->desc->uV_step * new_selector);

	return DIV_ROUND_UP(abs(new_volt - old_volt), ramp_delay);
}

static int s2mps11_set_ramp_delay(struct regulator_dev *rdev, int ramp_delay)
{
	struct s2mps11_info *s2mps11 = rdev_get_drvdata(rdev);
	unsigned int ramp_val, ramp_shift, ramp_reg = S2MPS11_REG_RAMP_BUCK;
	unsigned int ramp_enable = 1, enable_shift = 0;
	int ret;

	switch (rdev_get_id(rdev)) {
	case S2MPS11_BUCK1:
		if (ramp_delay > s2mps11->ramp_delay16)
			s2mps11->ramp_delay16 = ramp_delay;
		else
			ramp_delay = s2mps11->ramp_delay16;

		ramp_shift = S2MPS11_BUCK16_RAMP_SHIFT;
		break;
	case S2MPS11_BUCK2:
		enable_shift = S2MPS11_BUCK2_RAMP_EN_SHIFT;
		if (!ramp_delay) {
			ramp_enable = 0;
			break;
		}

		s2mps11->ramp_delay2 = ramp_delay;
		ramp_shift = S2MPS11_BUCK2_RAMP_SHIFT;
		ramp_reg = S2MPS11_REG_RAMP;
		break;
	case S2MPS11_BUCK3:
		enable_shift = S2MPS11_BUCK3_RAMP_EN_SHIFT;
		if (!ramp_delay) {
			ramp_enable = 0;
			break;
		}

		if (ramp_delay > s2mps11->ramp_delay34)
			s2mps11->ramp_delay34 = ramp_delay;
		else
			ramp_delay = s2mps11->ramp_delay34;

		ramp_shift = S2MPS11_BUCK34_RAMP_SHIFT;
		ramp_reg = S2MPS11_REG_RAMP;
		break;
	case S2MPS11_BUCK4:
		enable_shift = S2MPS11_BUCK4_RAMP_EN_SHIFT;
		if (!ramp_delay) {
			ramp_enable = 0;
			break;
		}

		if (ramp_delay > s2mps11->ramp_delay34)
			s2mps11->ramp_delay34 = ramp_delay;
		else
			ramp_delay = s2mps11->ramp_delay34;

		ramp_shift = S2MPS11_BUCK34_RAMP_SHIFT;
		ramp_reg = S2MPS11_REG_RAMP;
		break;
	case S2MPS11_BUCK5:
		s2mps11->ramp_delay5 = ramp_delay;
		ramp_shift = S2MPS11_BUCK5_RAMP_SHIFT;
		break;
	case S2MPS11_BUCK6:
		enable_shift = S2MPS11_BUCK6_RAMP_EN_SHIFT;
		if (!ramp_delay) {
			ramp_enable = 0;
			break;
		}

		if (ramp_delay > s2mps11->ramp_delay16)
			s2mps11->ramp_delay16 = ramp_delay;
		else
			ramp_delay = s2mps11->ramp_delay16;

		ramp_shift = S2MPS11_BUCK16_RAMP_SHIFT;
		break;
	case S2MPS11_BUCK7:
	case S2MPS11_BUCK8:
	case S2MPS11_BUCK10:
		if (ramp_delay > s2mps11->ramp_delay7810)
			s2mps11->ramp_delay7810 = ramp_delay;
		else
			ramp_delay = s2mps11->ramp_delay7810;

		ramp_shift = S2MPS11_BUCK7810_RAMP_SHIFT;
		break;
	case S2MPS11_BUCK9:
		s2mps11->ramp_delay9 = ramp_delay;
		ramp_shift = S2MPS11_BUCK9_RAMP_SHIFT;
		break;
	default:
		return 0;
	}

	if (!ramp_enable)
		goto ramp_disable;

	/* Ramp delay can be enabled/disabled only for buck[2346] */
	if ((rdev_get_id(rdev) >= S2MPS11_BUCK2 &&
			rdev_get_id(rdev) <= S2MPS11_BUCK4) ||
			rdev_get_id(rdev) == S2MPS11_BUCK6)  {
		ret = regmap_update_bits(rdev->regmap, S2MPS11_REG_RAMP,
					 1 << enable_shift, 1 << enable_shift);
		if (ret) {
			dev_err(&rdev->dev, "failed to enable ramp rate\n");
			return ret;
		}
	}

	ramp_val = get_ramp_delay(ramp_delay);

	return regmap_update_bits(rdev->regmap, ramp_reg, 0x3 << ramp_shift,
				  ramp_val << ramp_shift);

ramp_disable:
	return regmap_update_bits(rdev->regmap, S2MPS11_REG_RAMP,
				  1 << enable_shift, 0);
}

static struct regulator_ops s2mps11_ldo_ops = {
	.list_voltage		= regulator_list_voltage_linear,
	.map_voltage		= regulator_map_voltage_linear,
	.is_enabled		= regulator_is_enabled_regmap,
	.enable			= regulator_enable_regmap,
	.disable		= regulator_disable_regmap,
	.get_voltage_sel	= regulator_get_voltage_sel_regmap,
	.set_voltage_sel	= regulator_set_voltage_sel_regmap,
	.set_voltage_time_sel	= regulator_set_voltage_time_sel,
};

static struct regulator_ops s2mps11_buck_ops = {
	.list_voltage		= regulator_list_voltage_linear,
	.map_voltage		= regulator_map_voltage_linear,
	.is_enabled		= regulator_is_enabled_regmap,
	.enable			= regulator_enable_regmap,
	.disable		= regulator_disable_regmap,
	.get_voltage_sel	= regulator_get_voltage_sel_regmap,
	.set_voltage_sel	= regulator_set_voltage_sel_regmap,
	.set_voltage_time_sel	= s2mps11_regulator_set_voltage_time_sel,
	.set_ramp_delay		= s2mps11_set_ramp_delay,
};

#define regulator_desc_s2mps11_ldo1(num)	{		\
	.name		= "LDO"#num,			\
	.id		= S2MPS11_LDO##num,		\
	.ops		= &s2mps11_ldo_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPS11_LDO_MIN,		\
	.uV_step	= S2MPS11_LDO_STEP1,		\
	.n_voltages	= S2MPS11_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPS11_REG_L1CTRL + num - 1,	\
	.vsel_mask	= S2MPS11_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPS11_REG_L1CTRL + num - 1,	\
	.enable_mask	= S2MPS11_ENABLE_MASK		\
}
#define regulator_desc_s2mps11_ldo2(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPS11_LDO##num,		\
	.ops		= &s2mps11_ldo_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPS11_LDO_MIN,		\
	.uV_step	= S2MPS11_LDO_STEP2,		\
	.n_voltages	= S2MPS11_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPS11_REG_L1CTRL + num - 1,	\
	.vsel_mask	= S2MPS11_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPS11_REG_L1CTRL + num - 1,	\
	.enable_mask	= S2MPS11_ENABLE_MASK		\
}

#define regulator_desc_s2mps11_buck1_4(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPS11_BUCK##num,			\
	.ops		= &s2mps11_buck_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPS11_BUCK_MIN1,			\
	.uV_step	= S2MPS11_BUCK_STEP1,			\
	.n_voltages	= S2MPS11_BUCK_N_VOLTAGES,		\
	.ramp_delay	= S2MPS11_RAMP_DELAY,			\
	.vsel_reg	= S2MPS11_REG_B1CTRL2 + (num - 1) * 2,	\
	.vsel_mask	= S2MPS11_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPS11_REG_B1CTRL1 + (num - 1) * 2,	\
	.enable_mask	= S2MPS11_ENABLE_MASK			\
}

#define regulator_desc_s2mps11_buck5 {				\
	.name		= "BUCK5",				\
	.id		= S2MPS11_BUCK5,			\
	.ops		= &s2mps11_buck_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPS11_BUCK_MIN1,			\
	.uV_step	= S2MPS11_BUCK_STEP1,			\
	.n_voltages	= S2MPS11_BUCK_N_VOLTAGES,		\
	.ramp_delay	= S2MPS11_RAMP_DELAY,			\
	.vsel_reg	= S2MPS11_REG_B5CTRL2,			\
	.vsel_mask	= S2MPS11_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPS11_REG_B5CTRL1,			\
	.enable_mask	= S2MPS11_ENABLE_MASK			\
}

#define regulator_desc_s2mps11_buck6_8(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPS11_BUCK##num,			\
	.ops		= &s2mps11_buck_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPS11_BUCK_MIN1,			\
	.uV_step	= S2MPS11_BUCK_STEP1,			\
	.n_voltages	= S2MPS11_BUCK_N_VOLTAGES,		\
	.ramp_delay	= S2MPS11_RAMP_DELAY,			\
	.vsel_reg	= S2MPS11_REG_B6CTRL2 + (num - 6) * 2,	\
	.vsel_mask	= S2MPS11_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPS11_REG_B6CTRL1 + (num - 6) * 2,	\
	.enable_mask	= S2MPS11_ENABLE_MASK			\
}

#define regulator_desc_s2mps11_buck9 {				\
	.name		= "BUCK9",				\
	.id		= S2MPS11_BUCK9,			\
	.ops		= &s2mps11_buck_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPS11_BUCK_MIN3,			\
	.uV_step	= S2MPS11_BUCK_STEP3,			\
	.n_voltages	= S2MPS11_BUCK_N_VOLTAGES,		\
	.ramp_delay	= S2MPS11_RAMP_DELAY,			\
	.vsel_reg	= S2MPS11_REG_B9CTRL2,			\
	.vsel_mask	= S2MPS11_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPS11_REG_B9CTRL1,			\
	.enable_mask	= S2MPS11_ENABLE_MASK			\
}

#define regulator_desc_s2mps11_buck10 {				\
	.name		= "BUCK10",				\
	.id		= S2MPS11_BUCK10,			\
	.ops		= &s2mps11_buck_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPS11_BUCK_MIN2,			\
	.uV_step	= S2MPS11_BUCK_STEP2,			\
	.n_voltages	= S2MPS11_BUCK_N_VOLTAGES,		\
	.ramp_delay	= S2MPS11_RAMP_DELAY,			\
	.vsel_reg	= S2MPS11_REG_B10CTRL2,			\
	.vsel_mask	= S2MPS11_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPS11_REG_B10CTRL1,			\
	.enable_mask	= S2MPS11_ENABLE_MASK			\
}

static const struct regulator_desc s2mps11_regulators[] = {
	regulator_desc_s2mps11_ldo2(1),
	regulator_desc_s2mps11_ldo1(2),
	regulator_desc_s2mps11_ldo1(3),
	regulator_desc_s2mps11_ldo1(4),
	regulator_desc_s2mps11_ldo1(5),
	regulator_desc_s2mps11_ldo2(6),
	regulator_desc_s2mps11_ldo1(7),
	regulator_desc_s2mps11_ldo1(8),
	regulator_desc_s2mps11_ldo1(9),
	regulator_desc_s2mps11_ldo1(10),
	regulator_desc_s2mps11_ldo2(11),
	regulator_desc_s2mps11_ldo1(12),
	regulator_desc_s2mps11_ldo1(13),
	regulator_desc_s2mps11_ldo1(14),
	regulator_desc_s2mps11_ldo1(15),
	regulator_desc_s2mps11_ldo1(16),
	regulator_desc_s2mps11_ldo1(17),
	regulator_desc_s2mps11_ldo1(18),
	regulator_desc_s2mps11_ldo1(19),
	regulator_desc_s2mps11_ldo1(20),
	regulator_desc_s2mps11_ldo1(21),
	regulator_desc_s2mps11_ldo2(22),
	regulator_desc_s2mps11_ldo2(23),
	regulator_desc_s2mps11_ldo1(24),
	regulator_desc_s2mps11_ldo1(25),
	regulator_desc_s2mps11_ldo1(26),
	regulator_desc_s2mps11_ldo2(27),
	regulator_desc_s2mps11_ldo1(28),
	regulator_desc_s2mps11_ldo1(29),
	regulator_desc_s2mps11_ldo1(30),
	regulator_desc_s2mps11_ldo1(31),
	regulator_desc_s2mps11_ldo1(32),
	regulator_desc_s2mps11_ldo1(33),
	regulator_desc_s2mps11_ldo1(34),
	regulator_desc_s2mps11_ldo1(35),
	regulator_desc_s2mps11_ldo1(36),
	regulator_desc_s2mps11_ldo1(37),
	regulator_desc_s2mps11_ldo1(38),
	regulator_desc_s2mps11_buck1_4(1),
	regulator_desc_s2mps11_buck1_4(2),
	regulator_desc_s2mps11_buck1_4(3),
	regulator_desc_s2mps11_buck1_4(4),
	regulator_desc_s2mps11_buck5,
	regulator_desc_s2mps11_buck6_8(6),
	regulator_desc_s2mps11_buck6_8(7),
	regulator_desc_s2mps11_buck6_8(8),
	regulator_desc_s2mps11_buck9,
	regulator_desc_s2mps11_buck10,
};

static int s2mps14_regulator_enable(struct regulator_dev *rdev)
{
	struct s2mps11_info *s2mps11 = rdev_get_drvdata(rdev);
	unsigned int val;

	switch (s2mps11->dev_type) {
	case S2MPS14X:
		if (s2mps11->s2mps14_suspend_state & (1 << rdev_get_id(rdev)))
			val = S2MPS14_ENABLE_SUSPEND;
		else if (gpio_is_valid(s2mps11->ext_control_gpio[rdev_get_id(rdev)]))
			val = S2MPS14_ENABLE_EXT_CONTROL;
		else
			val = rdev->desc->enable_mask;
		break;
	case S2MPU02:
		if (s2mps11->s2mps14_suspend_state & (1 << rdev_get_id(rdev)))
			val = S2MPU02_ENABLE_SUSPEND;
		else
			val = rdev->desc->enable_mask;
		break;
	default:
		return -EINVAL;
	};

	return regmap_update_bits(rdev->regmap, rdev->desc->enable_reg,
			rdev->desc->enable_mask, val);
}

static int s2mps14_regulator_set_suspend_disable(struct regulator_dev *rdev)
{
	int ret;
	unsigned int val, state;
	struct s2mps11_info *s2mps11 = rdev_get_drvdata(rdev);
	int rdev_id = rdev_get_id(rdev);

	/* Below LDO should be always on or does not support suspend mode. */
	switch (s2mps11->dev_type) {
	case S2MPS14X:
		switch (rdev_id) {
		case S2MPS14_LDO3:
			return 0;
		default:
			state = S2MPS14_ENABLE_SUSPEND;
			break;
		};
		break;
	case S2MPU02:
		switch (rdev_id) {
		case S2MPU02_LDO13:
		case S2MPU02_LDO14:
		case S2MPU02_LDO15:
		case S2MPU02_LDO17:
		case S2MPU02_BUCK7:
			state = S2MPU02_DISABLE_SUSPEND;
			break;
		default:
			state = S2MPU02_ENABLE_SUSPEND;
			break;
		};
		break;
	default:
		return -EINVAL;
	};

	ret = regmap_read(rdev->regmap, rdev->desc->enable_reg, &val);
	if (ret < 0)
		return ret;

	s2mps11->s2mps14_suspend_state |= (1 << rdev_get_id(rdev));
	/*
	 * Don't enable suspend mode if regulator is already disabled because
	 * this would effectively for a short time turn on the regulator after
	 * resuming.
	 * However we still want to toggle the suspend_state bit for regulator
	 * in case if it got enabled before suspending the system.
	 */
	if (!(val & rdev->desc->enable_mask))
		return 0;

	return regmap_update_bits(rdev->regmap, rdev->desc->enable_reg,
			rdev->desc->enable_mask, state);
}

static struct regulator_ops s2mps14_reg_ops = {
	.list_voltage		= regulator_list_voltage_linear,
	.map_voltage		= regulator_map_voltage_linear,
	.is_enabled		= regulator_is_enabled_regmap,
	.enable			= s2mps14_regulator_enable,
	.disable		= regulator_disable_regmap,
	.get_voltage_sel	= regulator_get_voltage_sel_regmap,
	.set_voltage_sel	= regulator_set_voltage_sel_regmap,
	.set_voltage_time_sel	= regulator_set_voltage_time_sel,
	.set_suspend_disable	= s2mps14_regulator_set_suspend_disable,
};

#define regulator_desc_s2mps14_ldo1(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPS14_LDO##num,		\
	.ops		= &s2mps14_reg_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPS14_LDO_MIN_800MV,	\
	.uV_step	= S2MPS14_LDO_STEP_25MV,	\
	.n_voltages	= S2MPS14_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPS14_REG_L1CTRL + num - 1,	\
	.vsel_mask	= S2MPS14_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPS14_REG_L1CTRL + num - 1,	\
	.enable_mask	= S2MPS14_ENABLE_MASK		\
}
#define regulator_desc_s2mps14_ldo2(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPS14_LDO##num,		\
	.ops		= &s2mps14_reg_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPS14_LDO_MIN_1800MV,	\
	.uV_step	= S2MPS14_LDO_STEP_25MV,	\
	.n_voltages	= S2MPS14_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPS14_REG_L1CTRL + num - 1,	\
	.vsel_mask	= S2MPS14_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPS14_REG_L1CTRL + num - 1,	\
	.enable_mask	= S2MPS14_ENABLE_MASK		\
}
#define regulator_desc_s2mps14_ldo3(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPS14_LDO##num,		\
	.ops		= &s2mps14_reg_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPS14_LDO_MIN_800MV,	\
	.uV_step	= S2MPS14_LDO_STEP_12_5MV,	\
	.n_voltages	= S2MPS14_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPS14_REG_L1CTRL + num - 1,	\
	.vsel_mask	= S2MPS14_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPS14_REG_L1CTRL + num - 1,	\
	.enable_mask	= S2MPS14_ENABLE_MASK		\
}
#define regulator_desc_s2mps14_buck1235(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPS14_BUCK##num,			\
	.ops		= &s2mps14_reg_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPS14_BUCK1235_MIN_600MV,		\
	.uV_step	= S2MPS14_BUCK1235_STEP_6_25MV,		\
	.n_voltages	= S2MPS14_BUCK_N_VOLTAGES,		\
	.linear_min_sel = S2MPS14_BUCK1235_START_SEL,		\
	.ramp_delay	= S2MPS14_BUCK_RAMP_DELAY,		\
	.vsel_reg	= S2MPS14_REG_B1CTRL2 + (num - 1) * 2,	\
	.vsel_mask	= S2MPS14_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPS14_REG_B1CTRL1 + (num - 1) * 2,	\
	.enable_mask	= S2MPS14_ENABLE_MASK			\
}
#define regulator_desc_s2mps14_buck4(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPS14_BUCK##num,			\
	.ops		= &s2mps14_reg_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPS14_BUCK4_MIN_1400MV,		\
	.uV_step	= S2MPS14_BUCK4_STEP_12_5MV,		\
	.n_voltages	= S2MPS14_BUCK_N_VOLTAGES,		\
	.linear_min_sel = S2MPS14_BUCK4_START_SEL,		\
	.ramp_delay	= S2MPS14_BUCK_RAMP_DELAY,		\
	.vsel_reg	= S2MPS14_REG_B1CTRL2 + (num - 1) * 2,	\
	.vsel_mask	= S2MPS14_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPS14_REG_B1CTRL1 + (num - 1) * 2,	\
	.enable_mask	= S2MPS14_ENABLE_MASK			\
}

static const struct regulator_desc s2mps14_regulators[] = {
	regulator_desc_s2mps14_ldo3(1),
	regulator_desc_s2mps14_ldo3(2),
	regulator_desc_s2mps14_ldo1(3),
	regulator_desc_s2mps14_ldo1(4),
	regulator_desc_s2mps14_ldo3(5),
	regulator_desc_s2mps14_ldo3(6),
	regulator_desc_s2mps14_ldo1(7),
	regulator_desc_s2mps14_ldo2(8),
	regulator_desc_s2mps14_ldo3(9),
	regulator_desc_s2mps14_ldo3(10),
	regulator_desc_s2mps14_ldo1(11),
	regulator_desc_s2mps14_ldo2(12),
	regulator_desc_s2mps14_ldo2(13),
	regulator_desc_s2mps14_ldo2(14),
	regulator_desc_s2mps14_ldo2(15),
	regulator_desc_s2mps14_ldo2(16),
	regulator_desc_s2mps14_ldo2(17),
	regulator_desc_s2mps14_ldo2(18),
	regulator_desc_s2mps14_ldo1(19),
	regulator_desc_s2mps14_ldo1(20),
	regulator_desc_s2mps14_ldo1(21),
	regulator_desc_s2mps14_ldo3(22),
	regulator_desc_s2mps14_ldo1(23),
	regulator_desc_s2mps14_ldo2(24),
	regulator_desc_s2mps14_ldo2(25),
	regulator_desc_s2mps14_buck1235(1),
	regulator_desc_s2mps14_buck1235(2),
	regulator_desc_s2mps14_buck1235(3),
	regulator_desc_s2mps14_buck4(4),
	regulator_desc_s2mps14_buck1235(5),
};

static int s2mps14_pmic_enable_ext_control(struct s2mps11_info *s2mps11,
		struct regulator_dev *rdev)
{
	return regmap_update_bits(rdev->regmap, rdev->desc->enable_reg,
			rdev->desc->enable_mask, S2MPS14_ENABLE_EXT_CONTROL);
}

static void s2mps14_pmic_dt_parse_ext_control_gpio(struct platform_device *pdev,
		struct of_regulator_match *rdata, struct s2mps11_info *s2mps11)
{
	int *gpio = s2mps11->ext_control_gpio;
	unsigned int i;
	unsigned int valid_regulators[3] = { S2MPS14_LDO10, S2MPS14_LDO11,
		S2MPS14_LDO12 };

	for (i = 0; i < ARRAY_SIZE(valid_regulators); i++) {
		unsigned int reg = valid_regulators[i];

		if (!rdata[reg].init_data || !rdata[reg].of_node)
			continue;

		gpio[reg] = of_get_named_gpio(rdata[reg].of_node,
				"samsung,ext-control-gpios", 0);
		if (gpio_is_valid(gpio[reg]))
			dev_dbg(&pdev->dev, "Using GPIO %d for ext-control over %d/%s\n",
					gpio[reg], reg, rdata[reg].name);
	}
}

static int s2mps11_pmic_dt_parse(struct platform_device *pdev,
		struct of_regulator_match *rdata, struct s2mps11_info *s2mps11)
{
	struct device_node *reg_np;

	reg_np = of_get_child_by_name(pdev->dev.parent->of_node, "regulators");
	if (!reg_np) {
		dev_err(&pdev->dev, "could not find regulators sub-node\n");
		return -EINVAL;
	}

	of_regulator_match(&pdev->dev, reg_np, rdata, s2mps11->rdev_num);
	if (s2mps11->dev_type == S2MPS14X)
		s2mps14_pmic_dt_parse_ext_control_gpio(pdev, rdata, s2mps11);

	of_node_put(reg_np);

	return 0;
}

static int s2mpu02_set_ramp_delay(struct regulator_dev *rdev, int ramp_delay)
{
	unsigned int ramp_val, ramp_shift, ramp_reg;

	switch (rdev_get_id(rdev)) {
	case S2MPU02_BUCK1:
		ramp_shift = S2MPU02_BUCK1_RAMP_SHIFT;
		break;
	case S2MPU02_BUCK2:
		ramp_shift = S2MPU02_BUCK2_RAMP_SHIFT;
		break;
	case S2MPU02_BUCK3:
		ramp_shift = S2MPU02_BUCK3_RAMP_SHIFT;
		break;
	case S2MPU02_BUCK4:
		ramp_shift = S2MPU02_BUCK4_RAMP_SHIFT;
		break;
	default:
		return 0;
	}
	ramp_reg = S2MPU02_REG_RAMP1;
	ramp_val = get_ramp_delay(ramp_delay);

	return regmap_update_bits(rdev->regmap, ramp_reg,
				  S2MPU02_BUCK1234_RAMP_MASK << ramp_shift,
				  ramp_val << ramp_shift);
}

static struct regulator_ops s2mpu02_ldo_ops = {
	.list_voltage		= regulator_list_voltage_linear,
	.map_voltage		= regulator_map_voltage_linear,
	.is_enabled		= regulator_is_enabled_regmap,
	.enable			= s2mps14_regulator_enable,
	.disable		= regulator_disable_regmap,
	.get_voltage_sel	= regulator_get_voltage_sel_regmap,
	.set_voltage_sel	= regulator_set_voltage_sel_regmap,
	.set_voltage_time_sel	= regulator_set_voltage_time_sel,
	.set_suspend_disable	= s2mps14_regulator_set_suspend_disable,
};

static struct regulator_ops s2mpu02_buck_ops = {
	.list_voltage		= regulator_list_voltage_linear,
	.map_voltage		= regulator_map_voltage_linear,
	.is_enabled		= regulator_is_enabled_regmap,
	.enable			= s2mps14_regulator_enable,
	.disable		= regulator_disable_regmap,
	.get_voltage_sel	= regulator_get_voltage_sel_regmap,
	.set_voltage_sel	= regulator_set_voltage_sel_regmap,
	.set_voltage_time_sel	= regulator_set_voltage_time_sel,
	.set_suspend_disable	= s2mps14_regulator_set_suspend_disable,
	.set_ramp_delay		= s2mpu02_set_ramp_delay,
};

#define regulator_desc_s2mpu02_ldo1(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPU02_LDO##num,		\
	.ops		= &s2mpu02_ldo_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPU02_LDO_MIN_900MV,	\
	.uV_step	= S2MPU02_LDO_STEP_12_5MV,	\
	.linear_min_sel	= S2MPU02_LDO_GROUP1_START_SEL,	\
	.n_voltages	= S2MPU02_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPU02_REG_L1CTRL,		\
	.vsel_mask	= S2MPU02_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPU02_REG_L1CTRL,		\
	.enable_mask	= S2MPU02_ENABLE_MASK		\
}
#define regulator_desc_s2mpu02_ldo2(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPU02_LDO##num,		\
	.ops		= &s2mpu02_ldo_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPU02_LDO_MIN_1050MV,	\
	.uV_step	= S2MPU02_LDO_STEP_25MV,	\
	.linear_min_sel	= S2MPU02_LDO_GROUP2_START_SEL,	\
	.n_voltages	= S2MPU02_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPU02_REG_L2CTRL1,		\
	.vsel_mask	= S2MPU02_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPU02_REG_L2CTRL1,		\
	.enable_mask	= S2MPU02_ENABLE_MASK		\
}
#define regulator_desc_s2mpu02_ldo3(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPU02_LDO##num,		\
	.ops		= &s2mpu02_ldo_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPU02_LDO_MIN_900MV,	\
	.uV_step	= S2MPU02_LDO_STEP_12_5MV,	\
	.linear_min_sel	= S2MPU02_LDO_GROUP1_START_SEL,	\
	.n_voltages	= S2MPU02_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPU02_REG_L3CTRL + num - 3,	\
	.vsel_mask	= S2MPU02_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPU02_REG_L3CTRL + num - 3,	\
	.enable_mask	= S2MPU02_ENABLE_MASK		\
}
#define regulator_desc_s2mpu02_ldo4(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPU02_LDO##num,		\
	.ops		= &s2mpu02_ldo_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPU02_LDO_MIN_1050MV,	\
	.uV_step	= S2MPU02_LDO_STEP_25MV,	\
	.linear_min_sel	= S2MPU02_LDO_GROUP2_START_SEL,	\
	.n_voltages	= S2MPU02_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPU02_REG_L3CTRL + num - 3,	\
	.vsel_mask	= S2MPU02_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPU02_REG_L3CTRL + num - 3,	\
	.enable_mask	= S2MPU02_ENABLE_MASK		\
}
#define regulator_desc_s2mpu02_ldo5(num) {		\
	.name		= "LDO"#num,			\
	.id		= S2MPU02_LDO##num,		\
	.ops		= &s2mpu02_ldo_ops,		\
	.type		= REGULATOR_VOLTAGE,		\
	.owner		= THIS_MODULE,			\
	.min_uV		= S2MPU02_LDO_MIN_1600MV,	\
	.uV_step	= S2MPU02_LDO_STEP_50MV,	\
	.linear_min_sel	= S2MPU02_LDO_GROUP3_START_SEL,	\
	.n_voltages	= S2MPU02_LDO_N_VOLTAGES,	\
	.vsel_reg	= S2MPU02_REG_L3CTRL + num - 3,	\
	.vsel_mask	= S2MPU02_LDO_VSEL_MASK,	\
	.enable_reg	= S2MPU02_REG_L3CTRL + num - 3,	\
	.enable_mask	= S2MPU02_ENABLE_MASK		\
}

#define regulator_desc_s2mpu02_buck1234(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPU02_BUCK##num,			\
	.ops		= &s2mpu02_buck_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPU02_BUCK1234_MIN_600MV,		\
	.uV_step	= S2MPU02_BUCK1234_STEP_6_25MV,		\
	.n_voltages	= S2MPU02_BUCK_N_VOLTAGES,		\
	.linear_min_sel = S2MPU02_BUCK1234_START_SEL,		\
	.ramp_delay	= S2MPU02_BUCK_RAMP_DELAY,		\
	.vsel_reg	= S2MPU02_REG_B1CTRL2 + (num - 1) * 2,	\
	.vsel_mask	= S2MPU02_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPU02_REG_B1CTRL1 + (num - 1) * 2,	\
	.enable_mask	= S2MPU02_ENABLE_MASK			\
}
#define regulator_desc_s2mpu02_buck5(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPU02_BUCK##num,			\
	.ops		= &s2mpu02_ldo_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPU02_BUCK5_MIN_1081_25MV,		\
	.uV_step	= S2MPU02_BUCK5_STEP_6_25MV,		\
	.n_voltages	= S2MPU02_BUCK_N_VOLTAGES,		\
	.linear_min_sel = S2MPU02_BUCK5_START_SEL,		\
	.ramp_delay	= S2MPU02_BUCK_RAMP_DELAY,		\
	.vsel_reg	= S2MPU02_REG_B5CTRL2,			\
	.vsel_mask	= S2MPU02_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPU02_REG_B5CTRL1,			\
	.enable_mask	= S2MPU02_ENABLE_MASK			\
}
#define regulator_desc_s2mpu02_buck6(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPU02_BUCK##num,			\
	.ops		= &s2mpu02_ldo_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPU02_BUCK6_MIN_1700MV,		\
	.uV_step	= S2MPU02_BUCK6_STEP_2_50MV,		\
	.n_voltages	= S2MPU02_BUCK_N_VOLTAGES,		\
	.linear_min_sel = S2MPU02_BUCK6_START_SEL,		\
	.ramp_delay	= S2MPU02_BUCK_RAMP_DELAY,		\
	.vsel_reg	= S2MPU02_REG_B6CTRL2,			\
	.vsel_mask	= S2MPU02_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPU02_REG_B6CTRL1,			\
	.enable_mask	= S2MPU02_ENABLE_MASK			\
}
#define regulator_desc_s2mpu02_buck7(num) {			\
	.name		= "BUCK"#num,				\
	.id		= S2MPU02_BUCK##num,			\
	.ops		= &s2mpu02_ldo_ops,			\
	.type		= REGULATOR_VOLTAGE,			\
	.owner		= THIS_MODULE,				\
	.min_uV		= S2MPU02_BUCK7_MIN_900MV,		\
	.uV_step	= S2MPU02_BUCK7_STEP_6_25MV,		\
	.n_voltages	= S2MPU02_BUCK_N_VOLTAGES,		\
	.linear_min_sel = S2MPU02_BUCK7_START_SEL,		\
	.ramp_delay	= S2MPU02_BUCK_RAMP_DELAY,		\
	.vsel_reg	= S2MPU02_REG_B7CTRL2,			\
	.vsel_mask	= S2MPU02_BUCK_VSEL_MASK,		\
	.enable_reg	= S2MPU02_REG_B7CTRL1,			\
	.enable_mask	= S2MPU02_ENABLE_MASK			\
}

static const struct regulator_desc s2mpu02_regulators[] = {
	regulator_desc_s2mpu02_ldo1(1),
	regulator_desc_s2mpu02_ldo2(2),
	regulator_desc_s2mpu02_ldo4(3),
	regulator_desc_s2mpu02_ldo5(4),
	regulator_desc_s2mpu02_ldo4(5),
	regulator_desc_s2mpu02_ldo3(6),
	regulator_desc_s2mpu02_ldo3(7),
	regulator_desc_s2mpu02_ldo4(8),
	regulator_desc_s2mpu02_ldo5(9),
	regulator_desc_s2mpu02_ldo3(10),
	regulator_desc_s2mpu02_ldo4(11),
	regulator_desc_s2mpu02_ldo5(12),
	regulator_desc_s2mpu02_ldo5(13),
	regulator_desc_s2mpu02_ldo5(14),
	regulator_desc_s2mpu02_ldo5(15),
	regulator_desc_s2mpu02_ldo5(16),
	regulator_desc_s2mpu02_ldo4(17),
	regulator_desc_s2mpu02_ldo5(18),
	regulator_desc_s2mpu02_ldo3(19),
	regulator_desc_s2mpu02_ldo4(20),
	regulator_desc_s2mpu02_ldo5(21),
	regulator_desc_s2mpu02_ldo5(22),
	regulator_desc_s2mpu02_ldo5(23),
	regulator_desc_s2mpu02_ldo4(24),
	regulator_desc_s2mpu02_ldo5(25),
	regulator_desc_s2mpu02_ldo4(26),
	regulator_desc_s2mpu02_ldo5(27),
	regulator_desc_s2mpu02_ldo5(28),
	regulator_desc_s2mpu02_buck1234(1),
	regulator_desc_s2mpu02_buck1234(2),
	regulator_desc_s2mpu02_buck1234(3),
	regulator_desc_s2mpu02_buck1234(4),
	regulator_desc_s2mpu02_buck5(5),
	regulator_desc_s2mpu02_buck6(6),
	regulator_desc_s2mpu02_buck7(7),
};

static int s2mps11_pmic_probe(struct platform_device *pdev)
{
	struct sec_pmic_dev *iodev = dev_get_drvdata(pdev->dev.parent);
	struct sec_platform_data *pdata = NULL;
	struct of_regulator_match *rdata = NULL;
	struct regulator_config config = { };
	struct s2mps11_info *s2mps11;
	int i, ret = 0;
	const struct regulator_desc *regulators;

	s2mps11 = devm_kzalloc(&pdev->dev, sizeof(struct s2mps11_info),
				GFP_KERNEL);
	if (!s2mps11)
		return -ENOMEM;

	s2mps11->dev_type = platform_get_device_id(pdev)->driver_data;
	switch (s2mps11->dev_type) {
	case S2MPS11X:
		s2mps11->rdev_num = ARRAY_SIZE(s2mps11_regulators);
		regulators = s2mps11_regulators;
		break;
	case S2MPS14X:
		s2mps11->rdev_num = ARRAY_SIZE(s2mps14_regulators);
		regulators = s2mps14_regulators;
		break;
	case S2MPU02:
		s2mps11->rdev_num = ARRAY_SIZE(s2mpu02_regulators);
		regulators = s2mpu02_regulators;
		break;
	default:
		dev_err(&pdev->dev, "Invalid device type: %u\n",
				    s2mps11->dev_type);
		return -EINVAL;
	};

	s2mps11->ext_control_gpio = devm_kzalloc(&pdev->dev,
			sizeof(*s2mps11->ext_control_gpio) * s2mps11->rdev_num,
			GFP_KERNEL);
	if (!s2mps11->ext_control_gpio)
		return -ENOMEM;
	/*
	 * 0 is a valid GPIO so initialize all GPIO-s to negative value
	 * to indicate that external control won't be used for this regulator.
	 */
	for (i = 0; i < s2mps11->rdev_num; i++)
		s2mps11->ext_control_gpio[i] = -EINVAL;

	if (!iodev->dev->of_node) {
		if (iodev->pdata) {
			pdata = iodev->pdata;
			goto common_reg;
		} else {
			dev_err(pdev->dev.parent,
				"Platform data or DT node not supplied\n");
			return -ENODEV;
		}
	}

	rdata = kzalloc(sizeof(*rdata) * s2mps11->rdev_num, GFP_KERNEL);
	if (!rdata)
		return -ENOMEM;

	for (i = 0; i < s2mps11->rdev_num; i++)
		rdata[i].name = regulators[i].name;

	ret = s2mps11_pmic_dt_parse(pdev, rdata, s2mps11);
	if (ret)
		goto out;

common_reg:
	platform_set_drvdata(pdev, s2mps11);

	config.dev = &pdev->dev;
	config.regmap = iodev->regmap_pmic;
	config.driver_data = s2mps11;
	config.ena_gpio_flags = GPIOF_OUT_INIT_HIGH;
	for (i = 0; i < s2mps11->rdev_num; i++) {
		struct regulator_dev *regulator;

		if (pdata) {
			config.init_data = pdata->regulators[i].initdata;
			config.of_node = pdata->regulators[i].reg_node;
		} else {
			config.init_data = rdata[i].init_data;
			config.of_node = rdata[i].of_node;
		}
		config.ena_gpio = s2mps11->ext_control_gpio[i];

		regulator = devm_regulator_register(&pdev->dev,
						&regulators[i], &config);
		if (IS_ERR(regulator)) {
			ret = PTR_ERR(regulator);
			dev_err(&pdev->dev, "regulator init failed for %d\n",
				i);
			goto out;
		}

		if (gpio_is_valid(s2mps11->ext_control_gpio[i])) {
			ret = s2mps14_pmic_enable_ext_control(s2mps11,
					regulator);
			if (ret < 0) {
				dev_err(&pdev->dev,
						"failed to enable GPIO control over %s: %d\n",
						regulator->desc->name, ret);
				goto out;
			}
		}
	}

out:
	kfree(rdata);

	return ret;
}

static const struct platform_device_id s2mps11_pmic_id[] = {
	{ "s2mps11-pmic", S2MPS11X},
	{ "s2mps14-pmic", S2MPS14X},
	{ "s2mpu02-pmic", S2MPU02},
	{ },
};
MODULE_DEVICE_TABLE(platform, s2mps11_pmic_id);

static struct platform_driver s2mps11_pmic_driver = {
	.driver = {
		.name = "s2mps11-pmic",
		.owner = THIS_MODULE,
	},
	.probe = s2mps11_pmic_probe,
	.id_table = s2mps11_pmic_id,
};

static int __init s2mps11_pmic_init(void)
{
	return platform_driver_register(&s2mps11_pmic_driver);
}
subsys_initcall(s2mps11_pmic_init);

static void __exit s2mps11_pmic_exit(void)
{
	platform_driver_unregister(&s2mps11_pmic_driver);
}
module_exit(s2mps11_pmic_exit);

/* Module information */
MODULE_AUTHOR("Sangbeom Kim <sbkim73@samsung.com>");
MODULE_DESCRIPTION("SAMSUNG S2MPS11/S2MPS14 Regulator Driver");
MODULE_LICENSE("GPL");
