/*

 spatialite.c -- SQLite3 spatial extension

 version 5.1.0, 2023 August 4

 Author: Sandro Furieri a.furieri@lqt.it

 ------------------------------------------------------------------------------
 
 Version: MPL 1.1/GPL 2.0/LGPL 2.1
 
 The contents of this file are subject to the Mozilla Public License Version
 1.1 (the "License"); you may not use this file except in compliance with
 the License. You may obtain a copy of the License at
 http://www.mozilla.org/MPL/
 
Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
for the specific language governing rights and limitations under the
License.

The Original Code is the SpatiaLite library

The Initial Developer of the Original Code is Alessandro Furieri
 
Portions created by the Initial Developer are Copyright (C) 2008-2023
the Initial Developer. All Rights Reserved.

Contributor(s):
Pepijn Van Eeckhoudt <pepijnvaneeckhoudt@luciad.com>
(implementing Android support)

Mark Johnson <mj10777@googlemail.com>

Alternatively, the contents of this file may be used under the terms of
either the GNU General Public License Version 2 or later (the "GPL"), or
the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
in which case the provisions of the GPL or the LGPL are applicable instead
of those above. If you wish to allow use of your version of this file only
under the terms of either the GPL or the LGPL, and not to allow others to
use your version of this file under the terms of the MPL, indicate your
decision by deleting the provisions above and replace them with the notice
and other provisions required by the GPL or the LGPL. If you do not delete
the provisions above, a recipient may use your version of this file under
the terms of any one of the MPL, the GPL or the LGPL.
 
*/

/*
 
CREDITS:

this module has been partly funded by:
Regione Toscana - Settore Sistema Informativo Territoriale ed Ambientale
(exposing liblwgeom APIs as SpatiaLite own SQL functions) 

*/

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <locale.h>

#if defined(_WIN32) && !defined(__MINGW32__)
#include <io.h>
#include <direct.h>
#else
#include <dirent.h>
#endif

#if defined(_WIN32) && !defined(__MINGW32__)
#include "config-msvc.h"
#else
#include "config.h"
#endif

#if defined(_WIN32) || defined(WIN32)
#include <io.h>
#define isatty	_isatty
#else
#include <unistd.h>
#endif


#include <spatialite/sqlite.h>
#include <spatialite/debug.h>

#include <spatialite/gaiaaux.h>
#include <spatialite/gaiageo.h>
#include <spatialite/gaiaexif.h>
#include <spatialite/geopackage.h>
#include <spatialite/spatialite_ext.h>
#include <spatialite/gg_advanced.h>
#include <spatialite/gg_dxf.h>
#include <spatialite/gaiamatrix.h>
#include <spatialite/geopackage.h>
#include <spatialite/control_points.h>
#include <spatialite/stored_procedures.h>
#include <spatialite.h>
#include <spatialite_private.h>

#ifdef ENABLE_LIBXML2		/* LIBXML2 (and thus WFS) enabled */
#include <spatialite/gg_wfs.h>
#endif

#ifndef OMIT_FREEXL		/* including FreeXL */
#include <freexl.h>
#endif

#ifndef OMIT_GEOS		/* including GEOS */
#ifdef GEOS_REENTRANT
#ifdef GEOS_ONLY_REENTRANT
#define GEOS_USE_ONLY_R_API	/* only fully thread-safe GEOS API */
#endif
#endif
#include <geos_c.h>
#endif

#ifndef OMIT_PROJ		/* including PROJ.4 */
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
#include <proj.h>
#else /* supporting old PROJ.4 */
#include <proj_api.h>
#endif
#endif

#ifdef _WIN32
#define strcasecmp	_stricmp
#endif /* not WIN32 */

/* 64 bit integer: portable format for printf() */
#if defined(_WIN32) && !defined(__MINGW32__)
#define FRMT64 "%I64d"
#else
#define FRMT64 "%lld"
#endif

#define GAIA_UNUSED() if (argc || argv) argc = argc;

#define LINESTRING_MIN_SEGMENT_LENGTH	1
#define LINESTRING_MAX_SEGMENT_LENGTH	2
#define LINESTRING_AVG_SEGMENT_LENGTH	3

struct gaia_geom_chain_item
{
/* a struct used to store a chain item */
    gaiaGeomCollPtr geom;
    struct gaia_geom_chain_item *next;
};

struct gaia_geom_chain
{
/* a struct used to store a dynamic chain of GeometryCollections */
    int all_polygs;
    struct gaia_geom_chain_item *first;
    struct gaia_geom_chain_item *last;
};

struct stddev_str
{
/* a struct to implement StandardVariation and Variance aggregate functions */
    int cleaned;
    double mean;
    double quot;
    double count;
};

struct string_list_str
{
/* a struct supporting MakeStringList aggregate function */
    char *string;
    char separator;
};

struct fdo_table
{
/* a struct to implement a linked-list for FDO-ORG table names */
    char *table;
    struct fdo_table *next;
};

struct gpkg_table
{
/* a struct to implement a linked-list for OGC GeoPackage table names */
    char *table;
    struct gpkg_table *next;
};


/*
************************************************************************
**
** the following code has been integrally copied from SQLite's own sources:
** -/ext/misc/eval.c
*/

/*
** 2014-11-10
**
** The author disclaims copyright to this source code.  In place of
** a legal notice, here is a blessing:
**
**    May you do good and not evil.
**    May you find forgiveness for yourself and forgive others.
**    May you share freely, never taking more than you give.
**
******************************************************************************
**
** This SQLite extension implements SQL function eval() which runs
** SQL statements recursively.
*/

struct EvalResult
{
/*
** Structure used to accumulate the output
*/
    char *z;			/* Accumulated output */
    const char *zSep;		/* Separator */
    int szSep;			/* Size of the separator string */
    unsigned int nAlloc;	/* Number of bytes allocated for z[] */
    int nUsed;			/* Number of bytes of z[] actually used */
};

static int
eval_callback (void *pCtx, int argc, char **argv, char **colnames)
{
/*
** Callback from sqlite_exec() for the eval() function.
*/
    struct EvalResult *p = (struct EvalResult *) pCtx;
    int i;

    if (colnames == NULL)
	colnames = NULL;	/* silencing stupid compiler warnings */

    for (i = 0; i < argc; i++)
      {
	  const char *z = argv[i] ? argv[i] : "";
	  size_t sz = strlen (z);
	  if (sz + p->nUsed + p->szSep + 1 > p->nAlloc)
	    {
		char *zNew;
		p->nAlloc = p->nAlloc * 2 + sz + p->szSep + 1;
		zNew = sqlite3_realloc (p->z, p->nAlloc);
		if (zNew == 0)
		  {
		      sqlite3_free (p->z);
		      memset (p, 0, sizeof (*p));
		      return 1;
		  }
		p->z = zNew;
	    }
	  if (p->nUsed > 0)
	    {
		memcpy (&p->z[p->nUsed], p->zSep, p->szSep);
		p->nUsed += p->szSep;
	    }
	  memcpy (&p->z[p->nUsed], z, sz);
	  p->nUsed += sz;
      }
    return 0;
}

static void
fnct_EvalFunc (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/*
** Implementation of the eval(X) and eval(X,Y) SQL functions.
**
** Evaluate the SQL text in X.  Return the results, using string
** Y as the separator.  If Y is omitted, use a single space character.
*/
    const char *zSql;
    sqlite3 *db;
    char *zErr = 0;
    int rc;
    struct EvalResult x;

    memset (&x, 0, sizeof (x));
    x.zSep = " ";
    zSql = (const char *) sqlite3_value_text (argv[0]);
    if (zSql == 0)
	return;
    if (argc > 1)
      {
	  x.zSep = (const char *) sqlite3_value_text (argv[1]);
	  if (x.zSep == 0)
	      return;
      }
    x.szSep = (int) strlen (x.zSep);
    db = sqlite3_context_db_handle (context);
    rc = sqlite3_exec (db, zSql, eval_callback, &x, &zErr);
    if (rc != SQLITE_OK)
      {
	  sqlite3_result_error (context, zErr, -1);
	  sqlite3_free (zErr);
      }
    else if (x.zSep == 0)
      {
	  sqlite3_result_error_nomem (context);
	  sqlite3_free (x.z);
      }
    else
      {
	  sqlite3_result_text (context, x.z, x.nUsed, sqlite3_free);
      }
}

static void
fnct_createMissingSystemTables (sqlite3_context * context, int argc,
				sqlite3_value ** argv)
{
/* SQL function:
/ CreateMissingSystemTables()
/  or
/ CreateMissingSystemTables(bool relaxed)
/  or
/ CreateMissingSystemTables(bool relaxed, bool transaction)
/
/ creates all missing system tables required by version 5
/ returns 1 on success
/ RAISES AN EXCEPTION on failure or on invalid arguments
*/
    int relaxed = 0;
    int transaction = 0;
    int ret;
    char *err_msg = NULL;
    char *msg = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc >= 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
	      goto invalid_arg1;
	  relaxed = sqlite3_value_int (argv[0]);
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	      goto invalid_arg2;
	  transaction = sqlite3_value_int (argv[1]);
      }

    ret =
	createMissingSystemTables (sqlite, cache, relaxed, transaction,
				   &err_msg);
    if (ret <= 0)
	goto error;
    msg =
	sqlite3_mprintf ("successfully executed (%d Table%s been created)", ret,
			 (ret == 1) ? " has" : "s have");
    updateSpatiaLiteHistory (sqlite, "*** CreateMissingSystemTables ***", NULL,
			     msg);
    sqlite3_free (msg);
    sqlite3_result_int (context, ret);
    return;

  invalid_arg1:
    msg =
	"CreateMissingSystemTables exception - first argument (relaxed) expected to be an INTEGER.";
    sqlite3_result_error (context, msg, -1);
    return;

  invalid_arg2:
    msg =
	"CreateMissingSystemTables exception - second argument (transaction) expected to be an INTEGER.";
    sqlite3_result_error (context, msg, -1);
    return;

  error:
    if (err_msg == NULL)
	msg =
	    sqlite3_mprintf
	    ("CreateMissingSystemTables exception - Unknown failure reason.");
    else
      {
	  msg =
	      sqlite3_mprintf ("CreateMissingSystemTables exception - %s.",
			       err_msg);
	  sqlite3_free (err_msg);
      }
    sqlite3_result_error (context, msg, -1);
    sqlite3_free (msg);
    return;
}

static void
fnct_createMissingRasterlite2Columns (sqlite3_context * context, int argc,
				      sqlite3_value ** argv)
{
/* SQL function:
/ CreateMissingRasterlite2Columns()
/
/ creates all missing columns on system tables required 
/ by RasterLite2 "stable"
/ returns 1 on success, 0 on failure
*/
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    ret = createMissingRasterlite2Columns (sqlite);
    if (ret == 0)
	sqlite3_result_int (context, 0);
    else
	sqlite3_result_int (context, 1);
}

static void
fnct_spatialite_version (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ spatialite_version()
/
/ return a text string representing the current SpatiaLite version
*/
    int len;
    const char *p_result = spatialite_version ();
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    len = strlen (p_result);
    sqlite3_result_text (context, p_result, len, SQLITE_TRANSIENT);
}

static int
do_check_dqs (sqlite3 * sqlite)
{
/* checking if SQLite supports the DQS misfeature */
    char *sql;
    int ret;
    int ok = 1;
    unsigned char rnd[16];
    char random[40];
    char *p = random;
    int i;
    char *table;

    sqlite3_randomness (16, rnd);
    for (i = 0; i < 16; i++)
      {
	  sprintf (p, "%02x", rnd[i]);
	  p += 2;
      }
    *p = '\0';
    table = sqlite3_mprintf ("tmp_%s", random);

/* NOTE: the following SQL statements are INTENTIONALLY badly quoted */
    sql = sqlite3_mprintf ("CREATE TEMPORARY TABLE %Q ('column' TEXT)", table);
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, NULL);
    sqlite3_free (sql);
    if (ret != SQLITE_OK)
      {
	  ok = 0;
	  goto stop;
      }

    sql = sqlite3_mprintf ("INSERT INTO %Q ('column') VALUES (\"one\")", table);
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, NULL);
    sqlite3_free (sql);
    if (ret != SQLITE_OK)
      {
	  ok = 0;
	  goto stop;
      }

  stop:
    sql = sqlite3_mprintf ("DROP TABLE IF EXISTS %Q", table);
    sqlite3_exec (sqlite, sql, NULL, NULL, NULL);
    sqlite3_free (table);
    return ok;
}

static void
fnct_check_strict_sql_quoting (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ check_strict_sql_quoting()
/
/ return TRUE of FALSE depending on SQLite3 supporting the DQS misfeature or not
*/
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    ret = do_check_dqs (sqlite);
    if (ret == 0)
	sqlite3_result_int (context, 1);
    else
	sqlite3_result_int (context, 0);
}

static void
fnct_spatialite_target_cpu (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ spatialite_target_cpu()
/
/ return a text string representing the current SpatiaLite Target CPU
*/
    int len;
    const char *p_result = spatialite_target_cpu ();
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    len = strlen (p_result);
    sqlite3_result_text (context, p_result, len, SQLITE_TRANSIENT);
}

static void
fnct_freexl_version (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ freexl_version()
/
/ return a text string representing the current FreeXL version
/ or NULL if FreeXL is currently unsupported
*/

#ifndef OMIT_FREEXL		/* FreeXL version */
    int len;
    const char *p_result = freexl_version ();
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    len = strlen (p_result);
    sqlite3_result_text (context, p_result, len, SQLITE_TRANSIENT);
#else
    sqlite3_result_null (context);
#endif
}

static void
fnct_geos_version (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ geos_version()
/
/ return a text string representing the current GEOS version
/ or NULL if GEOS is currently unsupported
*/

#ifndef OMIT_GEOS		/* GEOS version */
    int len;
    const char *p_result = GEOSversion ();
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    len = strlen (p_result);
    sqlite3_result_text (context, p_result, len, SQLITE_TRANSIENT);
#else
    sqlite3_result_null (context);
#endif
}

static void
fnct_proj4_version (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ proj4_version()
/ proj_version()
/
/ return a text string representing the current PROJ.4 version
/ or NULL if PROJ.4 is currently unsupported
*/

#ifndef OMIT_PROJ		/* PROJ.4 version */
    int len;
    const char *p_result;
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
    PJ_INFO info = proj_info ();
    p_result = info.release;
#else /* supporting old PROJ.4 */
    p_result = pj_get_release ();
#endif
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    len = strlen (p_result);
    sqlite3_result_text (context, p_result, len, SQLITE_TRANSIENT);
#else
    sqlite3_result_null (context);
#endif
}

static void
fnct_has_proj (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasProj()
/
/ return 1 if built including Proj.4; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_PROJ
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
    sqlite3_result_int (context, 1);
#else /* supporting old PROJ.4 */
#if defined(PJ_VERSION) && PJ_VERSION >= 490
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
#endif
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_proj6 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasProj6()
/
/ return 1 if built including Proj.6; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_PROJ
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
    sqlite3_result_int (context, 1);
#endif
#endif
    sqlite3_result_int (context, 0);
}

static void
fnct_has_proj_geodesic (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ HasProjGeodesic()
/
/ return 1 if built supporting Proj.4 Geodesic; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_PROJ		/* PROJ.4 is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geos (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasGeos()
/
/ return 1 if built including GEOS; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_GEOS		/* GEOS is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geos_advanced (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ HasGeosAdvanced()
/
/ return 1 if built including GEOS-ADVANCED; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef GEOS_ADVANCED		/* GEOS-ADVANCED is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geos_3100 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasGeos3100()
/
/ return 1 if built including GEOS-3100; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef GEOS_3100		/* GEOS-3100 is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geos_3110 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasGeos3110()
/
/ return 1 if built including GEOS-3110; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef GEOS_3110		/* GEOS-3100 is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geos_trunk (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasGeosTrunk()
/
/ return 1 if built including GEOS-TRUNK; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_int (context, 0);
}

static void
fnct_has_geos_reentrant (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ HasGeosReentrant()
/
/ return 1 if built including GEOS-REENTRANT; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef GEOS_REENTRANT		/* GEOS-REENTRANT is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geos_only_reentrant (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ HasGeosOnlyReentrant()
/
/ return 1 if built including GEOS-ONLY_REENTRANT; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef GEOS_REENTRANT		/* GEOS-REENTRANT is supported */
#ifdef GEOS_ONLY_REENTRANT	/* GEOS-ONLY-REENTRANT is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_minizip (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasMiniZip()
/
/ return 1 if built including MINIZIP; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef ENABLE_MINIZIP		/* MINIZIP is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_rttopo_version (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ rttopo_version()
/
/ return a text string representing the current RTTOPO version
/ or NULL if RTTOPO is currently unsupported
*/

#ifdef ENABLE_RTTOPO		/* RTTOPO version */
    int len;
    const char *p_result = splite_rttopo_version ();
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    len = strlen (p_result);
    sqlite3_result_text (context, p_result, len, SQLITE_TRANSIENT);
#else
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_null (context);
#endif
}

static void
fnct_libxml2_version (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ libxml2_version()
/
/ return a text string representing the current LIBXML2 version
/ or NULL if LIBXML2 is currently unsupported
*/

#ifdef ENABLE_LIBXML2		/* LIBXML2 version */
    int len;
    const char *p_result = gaia_libxml2_version ();
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    len = strlen (p_result);
    sqlite3_result_text (context, p_result, len, free);
#else
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_null (context);
#endif
}

static void
fnct_has_rttopo (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasRtTopo()
/
/ return 1 if built including RTTOPO; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef ENABLE_RTTOPO		/* RTTOPO is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_iconv (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasIconv()
/
/ return 1 if built including ICONV; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_ICONV		/* ICONV is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_math_sql (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasMathSql()
/
/ return 1 if built including MATHSQL; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_MATHSQL		/* MATHSQL is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geo_callbacks (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ HasGeoCallbacks()
/
/ always return 0 - since 5.1 GeoCallbacks have been
/ definitely removed from the code base
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_int (context, 0);
}

static void
fnct_has_freeXL (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasFreeXL()
/
/ return 1 if built including FreeXL; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_FREEXL		/* FreeXL is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_epsg (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasEpsg()
/
/ return 1 if built including EPSG; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_EPSG		/* EPSG is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_libxml2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasLibXML2()
/
/ return 1 if built including LIBXML2; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef ENABLE_LIBXML2		/* LIBXML2 is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_geopackage (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasGeoPackage()
/
/ return 1 if built including GeoPackage support (GPKG); otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_gcp (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasGCP()
/ HasGroundControlPoints()
/
/ return 1 if built including GroundControlPoints support (GGP); otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef ENABLE_GCP		/* GCP are supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_topology (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasTopology()
/
/ return 1 if built including GroundControlPoints support (GGP); otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifdef ENABLE_RTTOPO		/* RTTOPO is supported */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_knn (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasKNN()
/
/ since 5.1 (fake) KNN is always supported
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_int (context, 1);
}

static void
fnct_has_knn2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasKNN2()
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_GEOS		/* only if GEOS is enabled */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_has_routing (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HasRouting()
/
/ return 1 if built including VirtualRouting support; otherwise 0
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
#ifndef OMIT_GEOS		/* only if GEOS is enabled */
    sqlite3_result_int (context, 1);
#else
    sqlite3_result_int (context, 0);
#endif
}

static void
fnct_GeometryConstraints (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ GeometryConstraints(BLOBencoded geometry, geometry-type, srid)
/ GeometryConstraints(BLOBencoded geometry, geometry-type, srid, dimensions)
/
/ checks geometry constraints, returning:
/
/ -1 - if some error occurred
/ 1 - if geometry constraints validation passes
/ 0 - if geometry constraints validation fails
/
*/
    int little_endian;
    int endian_arch = gaiaEndianArch ();
    unsigned char *p_blob = NULL;
    int n_bytes = 0;
    int srid;
    int geom_srid = -1;
    const char *type;
    int xtype;
    int geom_type = -1;
    int geom_normalized_type;
    const unsigned char *dimensions;
    int dims = GAIA_XY;
    int ret;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB
	|| sqlite3_value_type (argv[0]) == SQLITE_NULL)
	;
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	type = (const char *) sqlite3_value_text (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  /* current metadata style >= v.4.0.0 */
	  type = "UNKNOWN";
	  switch (sqlite3_value_int (argv[1]))
	    {
	    case 0:
		type = "GEOMETRY";
		dims = GAIA_XY;
		break;
	    case 1:
		type = "POINT";
		dims = GAIA_XY;
		break;
	    case 2:
		type = "LINESTRING";
		dims = GAIA_XY;
		break;
	    case 3:
		type = "POLYGON";
		dims = GAIA_XY;
		break;
	    case 4:
		type = "MULTIPOINT";
		dims = GAIA_XY;
		break;
	    case 5:
		type = "MULTILINESTRING";
		dims = GAIA_XY;
		break;
	    case 6:
		type = "MULTIPOLYGON";
		dims = GAIA_XY;
		break;
	    case 7:
		type = "GEOMETRYCOLLECTION";
		dims = GAIA_XY;
		break;
	    case 1000:
		type = "GEOMETRY";
		dims = GAIA_XY_Z;
		break;
	    case 1001:
		type = "POINT";
		dims = GAIA_XY_Z;
		break;
	    case 1002:
		type = "LINESTRING";
		dims = GAIA_XY_Z;
		break;
	    case 1003:
		type = "POLYGON";
		dims = GAIA_XY_Z;
		break;
	    case 1004:
		type = "MULTIPOINT";
		dims = GAIA_XY_Z;
		break;
	    case 1005:
		type = "MULTILINESTRING";
		dims = GAIA_XY_Z;
		break;
	    case 1006:
		type = "MULTIPOLYGON";
		dims = GAIA_XY_Z;
		break;
	    case 1007:
		type = "GEOMETRYCOLLECTION";
		dims = GAIA_XY_Z;
		break;
	    case 2000:
		type = "GEOMETRY";
		dims = GAIA_XY_M;
		break;
	    case 2001:
		type = "POINT";
		dims = GAIA_XY_M;
		break;
	    case 2002:
		type = "LINESTRING";
		dims = GAIA_XY_M;
		break;
	    case 2003:
		type = "POLYGON";
		dims = GAIA_XY_M;
		break;
	    case 2004:
		type = "MULTIPOINT";
		dims = GAIA_XY_M;
		break;
	    case 2005:
		type = "MULTILINESTRING";
		dims = GAIA_XY_M;
		break;
	    case 2006:
		type = "MULTIPOLYGON";
		dims = GAIA_XY_M;
		break;
	    case 2007:
		type = "GEOMETRYCOLLECTION";
		dims = GAIA_XY_M;
		break;
	    case 3000:
		type = "GEOMETRY";
		dims = GAIA_XY_Z_M;
		break;
	    case 3001:
		type = "POINT";
		dims = GAIA_XY_Z_M;
		break;
	    case 3002:
		type = "LINESTRING";
		dims = GAIA_XY_Z_M;
		break;
	    case 3003:
		type = "POLYGON";
		dims = GAIA_XY_Z_M;
		break;
	    case 3004:
		type = "MULTIPOINT";
		dims = GAIA_XY_Z_M;
		break;
	    case 3005:
		type = "MULTILINESTRING";
		dims = GAIA_XY_Z_M;
		break;
	    case 3006:
		type = "MULTIPOLYGON";
		dims = GAIA_XY_Z_M;
		break;
	    case 3007:
		type = "GEOMETRYCOLLECTION";
		dims = GAIA_XY_Z_M;
		break;
	    };
      }
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[2]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (argc == 4)
      {
	  /* explicit dimensions - supporting XYZM */
	  dimensions = sqlite3_value_text (argv[3]);
	  if (strcasecmp ((char *) dimensions, "XYZ") == 0)
	      dims = GAIA_XY_Z;
	  else if (strcasecmp ((char *) dimensions, "XYM") == 0)
	      dims = GAIA_XY_M;
	  else if (strcasecmp ((char *) dimensions, "XYZM") == 0)
	      dims = GAIA_XY_Z_M;
	  else
	      dims = GAIA_XY;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
      }
    if (p_blob)
      {
	  if (n_bytes == 24 || n_bytes == 32 || n_bytes == 40)
	    {
		/* testing for a possible TinyPoint BLOB */
		if (*(p_blob + 0) == GAIA_MARK_START &&
		    (*(p_blob + 1) == GAIA_TINYPOINT_LITTLE_ENDIAN
		     || *(p_blob + 1) == GAIA_TINYPOINT_BIG_ENDIAN)
		    && *(p_blob + (n_bytes - 1)) == GAIA_MARK_END)
		  {
		      /* quick TinyPoint validation */
		      int pointType;
		      if (*(p_blob + 1) == GAIA_TINYPOINT_LITTLE_ENDIAN)
			  little_endian = 1;
		      else if (*(p_blob + 1) == GAIA_TINYPOINT_BIG_ENDIAN)
			  little_endian = 0;
		      else
			  goto illegal_geometry;	/* unknown encoding; neither little-endian nor big-endian */
		      geom_srid =
			  gaiaImport32 (p_blob + 2, little_endian, endian_arch);
		      pointType = *(p_blob + 6);
		      switch (pointType)
			{
			case GAIA_TINYPOINT_XY:
			    geom_type = GAIA_POINT;
			    break;
			case GAIA_TINYPOINT_XYZ:
			    geom_type = GAIA_POINTZ;
			    break;
			case GAIA_TINYPOINT_XYM:
			    geom_type = GAIA_POINTM;
			    break;
			case GAIA_TINYPOINT_XYZM:
			    geom_type = GAIA_POINTZM;
			    break;
			default:
			    goto illegal_geometry;
			};
		      goto valid_geometry;
		  }
	    }

	  /* quick Geometry validation */
	  if (n_bytes < 45)
	      goto illegal_geometry;	/* cannot be an internal BLOB WKB geometry */
	  if (*(p_blob + 0) != GAIA_MARK_START)
	      goto illegal_geometry;	/* failed to recognize START signature */
	  if (*(p_blob + (n_bytes - 1)) != GAIA_MARK_END)
	      goto illegal_geometry;	/* failed to recognize END signature */
	  if (*(p_blob + 38) != GAIA_MARK_MBR)
	      goto illegal_geometry;	/* failed to recognize MBR signature */
	  if (*(p_blob + 1) == GAIA_LITTLE_ENDIAN)
	      little_endian = 1;
	  else if (*(p_blob + 1) == GAIA_BIG_ENDIAN)
	      little_endian = 0;
	  else
	      goto illegal_geometry;	/* unknown encoding; neither little-endian nor big-endian */
	  geom_type = gaiaImport32 (p_blob + 39, little_endian, endian_arch);
	  geom_srid = gaiaImport32 (p_blob + 2, little_endian, endian_arch);
	  goto valid_geometry;
	illegal_geometry:
	  sqlite3_result_int (context, -1);
	  return;
      }
  valid_geometry:
    xtype = GAIA_UNKNOWN;
    if (strcasecmp ((char *) type, "POINT") == 0)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_POINTZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_POINTM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_POINTZM;
		break;
	    default:
		xtype = GAIA_POINT;
		break;
	    };
      }
    if (strcasecmp ((char *) type, "LINESTRING") == 0)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_LINESTRINGZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_LINESTRINGM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_LINESTRINGZM;
		break;
	    default:
		xtype = GAIA_LINESTRING;
		break;
	    };
      }
    if (strcasecmp ((char *) type, "POLYGON") == 0)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_POLYGONZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_POLYGONM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_POLYGONZM;
		break;
	    default:
		xtype = GAIA_POLYGON;
		break;
	    };
      }
    if (strcasecmp ((char *) type, "MULTIPOINT") == 0)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_MULTIPOINTZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_MULTIPOINTM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_MULTIPOINTZM;
		break;
	    default:
		xtype = GAIA_MULTIPOINT;
		break;
	    };
      }
    if (strcasecmp ((char *) type, "MULTILINESTRING") == 0)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_MULTILINESTRINGZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_MULTILINESTRINGM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_MULTILINESTRINGZM;
		break;
	    default:
		xtype = GAIA_MULTILINESTRING;
		break;
	    };
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGON") == 0)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_MULTIPOLYGONZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_MULTIPOLYGONM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_MULTIPOLYGONZM;
		break;
	    default:
		xtype = GAIA_MULTIPOLYGON;
		break;
	    };
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTION") == 0)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_GEOMETRYCOLLECTIONZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_GEOMETRYCOLLECTIONM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_GEOMETRYCOLLECTIONZM;
		break;
	    default:
		xtype = GAIA_GEOMETRYCOLLECTION;
		break;
	    };
      }
    switch (geom_type)
      {
	  /* adjusting COMPRESSED Geometries */
      case GAIA_COMPRESSED_LINESTRING:
	  geom_normalized_type = GAIA_LINESTRING;
	  break;
      case GAIA_COMPRESSED_LINESTRINGZ:
	  geom_normalized_type = GAIA_LINESTRINGZ;
	  break;
      case GAIA_COMPRESSED_LINESTRINGM:
	  geom_normalized_type = GAIA_LINESTRINGM;
	  break;
      case GAIA_COMPRESSED_LINESTRINGZM:
	  geom_normalized_type = GAIA_LINESTRINGZM;
	  break;
      case GAIA_COMPRESSED_POLYGON:
	  geom_normalized_type = GAIA_POLYGON;
	  break;
      case GAIA_COMPRESSED_POLYGONZ:
	  geom_normalized_type = GAIA_POLYGONZ;
	  break;
      case GAIA_COMPRESSED_POLYGONM:
	  geom_normalized_type = GAIA_POLYGONM;
	  break;
      case GAIA_COMPRESSED_POLYGONZM:
	  geom_normalized_type = GAIA_POLYGONZM;
	  break;
      default:
	  geom_normalized_type = geom_type;
	  break;
      };
    if (strcasecmp ((char *) type, "GEOMETRY") == 0)
	xtype = -1;
    if (xtype == GAIA_UNKNOWN)
	sqlite3_result_int (context, -1);
    else
      {
	  ret = 1;
	  if (p_blob)
	    {
		/* skipping NULL Geometry; this is assumed to be always good */
		if (geom_srid != srid)
		    ret = 0;
		if (xtype == -1)
		    ;
		else if (xtype != geom_normalized_type)
		    ret = 0;
	    }
	  sqlite3_result_int (context, ret);
      }
}

static void
fnct_RTreeAlign (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ RTreeAlign(RTree-table-name, PKID-value, BLOBencoded geometry)
/
/ attempts to update the associated R*Tree, returning:
/
/ -1 - if some invalid arg was passed
/ 1 - successful update
/ 0 - update failure
/
*/
    unsigned char *p_blob = NULL;
    int n_bytes = 0;
    sqlite3_int64 pkid;
    const char *rtree_table;
    char *table_name;
    int len;
    char pkv[64];
    gaiaGeomCollPtr geom = NULL;
    int ret;
    char *sql_statement;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	rtree_table = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	pkid = sqlite3_value_int64 (argv[1]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_BLOB
	|| sqlite3_value_type (argv[2]) == SQLITE_NULL)
	;
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_BLOB)
      {
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[2]);
	  n_bytes = sqlite3_value_bytes (argv[2]);
	  geom = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
      }

    if (geom == NULL)
      {
	  /* NULL geometry: nothing to do */
	  sqlite3_result_int (context, 1);
      }
    else
      {
	  /* INSERTing into the R*Tree */
	  if (*(rtree_table + 0) == '"'
	      && *(rtree_table + strlen (rtree_table) - 1) == '"')
	    {
		/* earlier versions may pass an already quoted name */
		char *dequoted_table_name;
		len = strlen (rtree_table);
		table_name = malloc (len + 1);
		strcpy (table_name, rtree_table);
		dequoted_table_name = gaiaDequotedSql (table_name);
		free (table_name);
		if (dequoted_table_name == NULL)
		  {
		      sqlite3_result_int (context, -1);
		      return;
		  }
		table_name = gaiaDoubleQuotedSql (dequoted_table_name);
		free (dequoted_table_name);
	    }
	  else
	      table_name = gaiaDoubleQuotedSql (rtree_table);
	  sprintf (pkv, FRMT64, pkid);
	  sql_statement =
	      sqlite3_mprintf
	      ("INSERT INTO \"%s\" (pkid, xmin, ymin, xmax, ymax) "
	       "VALUES (%s, %1.12f, %1.12f, %1.12f, %1.12f)", table_name,
	       pkv, geom->MinX, geom->MinY, geom->MaxX, geom->MaxY);
	  gaiaFreeGeomColl (geom);
	  ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
	  sqlite3_free (sql_statement);
	  free (table_name);
	  if (ret != SQLITE_OK)
	      sqlite3_result_int (context, 0);
	  else
	      sqlite3_result_int (context, 1);
      }
}

static void
fnct_TemporaryRTreeAlign (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ TemporaryRTreeAlign(db-prefix, RTree-table-name, PKID-value, BLOBencoded geometry)
/
/ attempts to update the associated R*Tree, returning:
/
/ -1 - if some invalid arg was passed
/ 1 - successful update
/ 0 - update failure
/
*/
    unsigned char *p_blob = NULL;
    int n_bytes = 0;
    sqlite3_int64 pkid;
    const char *db_prefix;
    const char *rtree_table;
    char *prefix;
    char *table_name;
    int len;
    char pkv[64];
    gaiaGeomCollPtr geom = NULL;
    int ret;
    char *sql_statement;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	db_prefix = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	rtree_table = (const char *) sqlite3_value_text (argv[1]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	pkid = sqlite3_value_int64 (argv[2]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_BLOB
	|| sqlite3_value_type (argv[3]) == SQLITE_NULL)
	;
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_BLOB)
      {
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[3]);
	  n_bytes = sqlite3_value_bytes (argv[3]);
	  geom = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
      }

    if (geom == NULL)
      {
	  /* NULL geometry: nothing to do */
	  sqlite3_result_int (context, 1);
      }
    else
      {
	  /* INSERTing into the R*Tree */
	  if (*(rtree_table + 0) == '"'
	      && *(rtree_table + strlen (rtree_table) - 1) == '"')
	    {
		/* earlier versions may pass an already quoted name */
		char *dequoted_table_name;
		len = strlen (rtree_table);
		table_name = malloc (len + 1);
		strcpy (table_name, rtree_table);
		dequoted_table_name = gaiaDequotedSql (table_name);
		free (table_name);
		if (dequoted_table_name == NULL)
		  {
		      sqlite3_result_int (context, -1);
		      return;
		  }
		table_name = gaiaDoubleQuotedSql (dequoted_table_name);
		free (dequoted_table_name);
	    }
	  else
	      table_name = gaiaDoubleQuotedSql (rtree_table);
	  prefix = gaiaDoubleQuotedSql (db_prefix);
	  sprintf (pkv, FRMT64, pkid);
	  sql_statement =
	      sqlite3_mprintf
	      ("INSERT INTO \"%s\".\"%s\" (pkid, xmin, ymin, xmax, ymax) "
	       "VALUES (%s, %1.12f, %1.12f, %1.12f, %1.12f)", prefix,
	       table_name, pkv, geom->MinX, geom->MinY, geom->MaxX, geom->MaxY);
	  free (prefix);
	  gaiaFreeGeomColl (geom);
	  ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
	  sqlite3_free (sql_statement);
	  free (table_name);
	  if (ret != SQLITE_OK)
	      sqlite3_result_int (context, 0);
	  else
	      sqlite3_result_int (context, 1);
      }
}

static void
fnct_IsValidFont (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsValidFont(BLOBencoded font)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return 0 (FALSE)
/ or -1 (INVALID ARGS)
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    sqlite3_result_int (context, 0);
}

static void
fnct_CheckFontFacename (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ CheckFontfaceName(TEXT facename, BLOBencoded font)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return NULL
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    sqlite3_result_int (context, 0);
}

static void
fnct_GetFontFamily (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GetFontFamily(BLOBencoded font)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return NULL
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_null (context);
}

static void
fnct_IsFontBold (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsFontBold(BLOBencoded font)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return -1
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_int (context, -1);
}

static void
fnct_IsFontItalic (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsFontItalic(BLOBencoded font)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return -1
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_int (context, -1);
}

static void
fnct_IsValidPixel (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsValidPixel(BLOBencoded pixel, text sample_type, int num_bands)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return 0 (FALSE)
/ or -1 (INVALID ARGS)
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    sqlite3_result_int (context, 0);
}

static void
fnct_IsValidRasterPalette (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ IsValidRasterPalette(BLOBencoded palette, text sample_type)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return 0 (FALSE)
/ or -1 (INVALID ARGS)
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    sqlite3_result_int (context, 0);
}

static void
fnct_IsValidRasterStatistics (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ IsValidRasterStatistics(text db_prefix, text coverage, 
/                         BLOBencoded statistics)
/   or
/ IsValidRasterStatistics(BLOBencoded statistics, text sample_type, 
/                         int num_bands)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return 0 (FALSE)
/ or -1 (INVALID ARGS)
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if ((sqlite3_value_type (argv[0]) == SQLITE_TEXT
	 || sqlite3_value_type (argv[0]) == SQLITE_NULL)
	&& sqlite3_value_type (argv[1]) == SQLITE_TEXT
	&& sqlite3_value_type (argv[2]) == SQLITE_BLOB)
	;
    else if (sqlite3_value_type (argv[0]) == SQLITE_BLOB
	     && sqlite3_value_type (argv[1]) == SQLITE_TEXT
	     && sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	;
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    sqlite3_result_int (context, 0);
}

static void
fnct_IsValidRasterTile (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ IsValidRasterTile(text db_prefix, text coverage, integer level, 
/                   BLOBencoded tile_odd, BLOBencoded tile_even)
/
/ basic version intended to be overloaded by RasterLite-2
/ always return 0 (FALSE)
/ or -1 (INVALID ARGS)
/
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT
	|| sqlite3_value_type (argv[0]) == SQLITE_NULL)
	;
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[3]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[4]) != SQLITE_BLOB
	&& sqlite3_value_type (argv[4]) != SQLITE_NULL)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    sqlite3_result_int (context, 0);
}

static void
fnct_IsPopulatedCoverage (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ IsPopulatedCoverage(text db_prefix, text coverage)
/
/ check if a RasterCoverage is already populated 
/ returns 1 if TRUE, 0 if FALSE
/ -1 on invalid arguments
*/
    const char *db_prefix = NULL;
    const char *coverage;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT
	|| sqlite3_value_type (argv[0]) == SQLITE_NULL)
	;
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    coverage = (const char *) sqlite3_value_text (argv[0]);
    ret = checkPopulatedCoverage (sqlite, db_prefix, coverage);
    sqlite3_result_int (context, ret);
    return;
}

static int
is_without_rowid_table (sqlite3 * sqlite, const char *table)
{
/* internal utility functions; checks for WITHOUT ROWID tables */
    char *sql;
    char *xtable;
    int ret;
    int i;
    char **results;
    int rows;
    int columns;
    int j;
    char **results2;
    int rows2;
    int columns2;
    char *errMsg = NULL;
    int without_rowid = 0;

    xtable = gaiaDoubleQuotedSql (table);
    sql = sqlite3_mprintf ("PRAGMA index_list(\"%s\")", xtable);
    free (xtable);
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, &errMsg);
    sqlite3_free (sql);
    if (ret != SQLITE_OK)
      {
	  sqlite3_free (errMsg);
	  return 1;
      }
    for (i = 1; i <= rows; i++)
      {
	  const char *index = results[(i * columns) + 1];
	  sql = sqlite3_mprintf ("SELECT count(*) FROM sqlite_master WHERE "
				 "type = 'index' AND Lower(tbl_name) = Lower(%Q) "
				 "AND Lower(name) = Lower(%Q)", table, index);
	  ret =
	      sqlite3_get_table (sqlite, sql, &results2, &rows2, &columns2,
				 &errMsg);
	  sqlite3_free (sql);
	  if (ret != SQLITE_OK)
	    {
		sqlite3_free (errMsg);
		return 1;
	    }
	  for (j = 1; j <= rows2; j++)
	    {
		if (atoi (results2[(j * columns2) + 0]) == 0)
		    without_rowid = 1;
	    }
	  sqlite3_free_table (results2);
      }
    sqlite3_free_table (results);
    return without_rowid;
}

static int
is_without_rowid_table_attached (sqlite3 * sqlite, const char *db_prefix,
				 const char *table)
{
/* internal utility functions; checks for WITHOUT ROWID tables */
    char *sql;
    char *xprefix;
    char *xtable;
    int ret;
    int i;
    char **results;
    int rows;
    int columns;
    int j;
    char **results2;
    int rows2;
    int columns2;
    char *errMsg = NULL;
    int without_rowid = 0;

    if (db_prefix == NULL)
	return 1;

    xprefix = gaiaDoubleQuotedSql (db_prefix);
    xtable = gaiaDoubleQuotedSql (table);
    sql = sqlite3_mprintf ("PRAGMA \"%s\".index_list(\"%s\")", xprefix, xtable);
    free (xprefix);
    free (xtable);
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, &errMsg);
    sqlite3_free (sql);
    if (ret != SQLITE_OK)
      {
	  sqlite3_free (errMsg);
	  return 1;
      }
    xprefix = gaiaDoubleQuotedSql (db_prefix);
    for (i = 1; i <= rows; i++)
      {
	  const char *index = results[(i * columns) + 1];
	  sql =
	      sqlite3_mprintf
	      ("SELECT count(*) FROM \"%s\".sqlite_master WHERE "
	       "type = 'index' AND Lower(tbl_name) = Lower(%Q) "
	       "AND Lower(name) = Lower(%Q)", table, index);
	  ret =
	      sqlite3_get_table (sqlite, sql, &results2, &rows2, &columns2,
				 &errMsg);
	  sqlite3_free (sql);
	  if (ret != SQLITE_OK)
	    {
		sqlite3_free (errMsg);
		return 1;
	    }
	  for (j = 1; j <= rows2; j++)
	    {
		if (atoi (results2[(j * columns2) + 0]) == 0)
		    without_rowid = 1;
	    }
	  sqlite3_free_table (results2);
      }
    free (xprefix);
    sqlite3_free_table (results);
    return without_rowid;
}

static int
is_attached_memory (sqlite3 * sqlite, const char *db_prefix)
{
/* internal utility functions; checks if an Attached Database is based on :memory: */
    const char *sql;
    int ret;
    int i;
    char **results;
    int rows;
    int columns;
    char *errMsg = NULL;
    int is_memory = 0;

    if (db_prefix == NULL)
	return 0;

    sql = "PRAGMA database_list";
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, &errMsg);
    if (ret != SQLITE_OK)
      {
	  sqlite3_free (errMsg);
	  return 0;
      }
    for (i = 1; i <= rows; i++)
      {
	  const char *name = results[(i * columns) + 1];
	  const char *file = results[(i * columns) + 2];
	  if (strcasecmp (name, db_prefix) == 0)
	    {
		if (file == NULL || strlen (file) == 0)
		    is_memory = 1;
	    }
      }
    sqlite3_free_table (results);
    return is_memory;
}

static int
checkDatabase (const sqlite3 * handle, const char *db_prefix)
{
/* testing if some ATTACHED-DB do really exist */
    sqlite3 *sqlite = (sqlite3 *) handle;
    char *xdb_prefix;
    char sql[1024];
    int ret;
    int i;
    char **results;
    int rows;
    int columns;
    int exists = 0;

    if (db_prefix == NULL)
	db_prefix = "main";
    xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
    sprintf (sql, "PRAGMA \"%s\".database_list", xdb_prefix);
    free (xdb_prefix);
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, NULL);
    if (ret != SQLITE_OK)
	return 0;
    if (rows < 1)
	;
    else
      {
	  for (i = 1; i <= rows; i++)
	    {
		const char *name = results[(i * columns) + 1];
		if (strcasecmp (name, db_prefix) == 0)
		    exists = 1;
	    }
      }
    sqlite3_free_table (results);
    return exists;
}

static int
checkGeoPackage (sqlite3 * handle, const char *db_prefix)
{
/* testing for GeoPackage meta-tables */
    sqlite3 *sqlite = (sqlite3 *) handle;
    char *xdb_prefix;
    char sql[1024];
    int ret;
    const char *name;
    int table_name = 0;
    int column_name = 0;
    int geometry_type_name = 0;
    int srs_id_gc = 0;
    int has_z = 0;
    int has_m = 0;
    int gpkg_gc = 0;
    int srs_id_srs = 0;
    int srs_name = 0;
    int gpkg_srs = 0;
    int i;
    char **results;
    int rows;
    int columns;

    if (!checkDatabase (handle, db_prefix))
	return -1;
/* checking the GPKG_GEOMETRY_COLUMNS table */
    if (db_prefix == NULL)
	db_prefix = "main";
    xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
    sprintf (sql, "PRAGMA \"%s\".table_info(gpkg_geometry_columns)",
	     xdb_prefix);
    free (xdb_prefix);
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, NULL);
    if (ret != SQLITE_OK)
	goto unknown;
    if (rows < 1)
	;
    else
      {
	  for (i = 1; i <= rows; i++)
	    {
		name = results[(i * columns) + 1];
		if (strcasecmp (name, "table_name") == 0)
		    table_name = 1;
		if (strcasecmp (name, "column_name") == 0)
		    column_name = 1;
		if (strcasecmp (name, "geometry_type_name") == 0)
		    geometry_type_name = 1;
		if (strcasecmp (name, "srs_id") == 0)
		    srs_id_gc = 1;
		if (strcasecmp (name, "z") == 0)
		    has_z = 1;
		if (strcasecmp (name, "m") == 0)
		    has_m = 1;
	    }
      }
    sqlite3_free_table (results);
    if (table_name && column_name && geometry_type_name && srs_id_gc && has_z
	&& has_m)
	gpkg_gc = 1;
/* checking the GPKG_SPATIAL_REF_SYS table */
    strcpy (sql, "PRAGMA table_info(gpkg_spatial_ref_sys)");
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, NULL);
    if (ret != SQLITE_OK)
	goto unknown;
    if (rows < 1)
	;
    else
      {
	  for (i = 1; i <= rows; i++)
	    {
		name = results[(i * columns) + 1];
		if (strcasecmp (name, "srs_id") == 0)
		    srs_id_srs = 1;
		if (strcasecmp (name, "srs_name") == 0)
		    srs_name = 1;
	    }
      }
    sqlite3_free_table (results);
    if (srs_id_srs && srs_name)
	gpkg_srs = 1;
    if (gpkg_gc && gpkg_srs)
	return 1;
  unknown:
    return 0;
}

SPATIALITE_PRIVATE int
checkSpatialMetaData (const void *handle)
{
/* just calls checkSpatialMetaData_ex */
    return checkSpatialMetaData_ex (handle, NULL);
}

SPATIALITE_PRIVATE int
checkSpatialMetaData_ex (const void *handle, const char *db_prefix)
{
/* internal utility function:
/
/ for FDO-OGR interoperability and cross-version seamless compatibility:
/ tests the SpatialMetadata type, returning:
/
/ -1 -  if no ATTACHED-DB identified by db_prefix exists
/ 0 - if no valid SpatialMetaData were found
/ 1 - if SpatiaLite-like (legacy) SpatialMetadata were found
/ 2 - if FDO-OGR-like SpatialMetadata were found
/ 3 - if SpatiaLite-like (current) SpatialMetadata were 
/ 4 - if GeoPackage SpatialMetadata were found
/
*/
    sqlite3 *sqlite = (sqlite3 *) handle;
    char *xdb_prefix;
    int spatialite_legacy_rs = 0;
    int spatialite_rs = 0;
    int fdo_rs = 0;
    int spatialite_legacy_gc = 0;
    int spatialite_gc = 0;
    int fdo_gc = 0;
    int rs_srid = 0;
    int auth_name = 0;
    int auth_srid = 0;
    int srtext = 0;
    int ref_sys_name = 0;
    int proj4text = 0;
    int f_table_name = 0;
    int f_geometry_column = 0;
    int geometry_type = 0;
    int coord_dimension = 0;
    int gc_srid = 0;
    int geometry_format = 0;
    int type = 0;
    int spatial_index_enabled = 0;
    char sql[1024];
    int ret;
    const char *name;
    int i;
    char **results;
    int rows;
    int columns;

    if (!checkDatabase (handle, db_prefix))
	return -1;
/* checking the GEOMETRY_COLUMNS table */
    if (db_prefix == NULL)
	db_prefix = "main";
    xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
    sprintf (sql, "PRAGMA \"%s\".table_info(geometry_columns)", xdb_prefix);
    free (xdb_prefix);
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, NULL);
    if (ret != SQLITE_OK)
	goto unknown;
    if (rows < 1)
	;
    else
      {
	  for (i = 1; i <= rows; i++)
	    {
		name = results[(i * columns) + 1];
		if (strcasecmp (name, "f_table_name") == 0)
		    f_table_name = 1;
		if (strcasecmp (name, "f_geometry_column") == 0)
		    f_geometry_column = 1;
		if (strcasecmp (name, "geometry_type") == 0)
		    geometry_type = 1;
		if (strcasecmp (name, "coord_dimension") == 0)
		    coord_dimension = 1;
		if (strcasecmp (name, "srid") == 0)
		    gc_srid = 1;
		if (strcasecmp (name, "geometry_format") == 0)
		    geometry_format = 1;
		if (strcasecmp (name, "type") == 0)
		    type = 1;
		if (strcasecmp (name, "spatial_index_enabled") == 0)
		    spatial_index_enabled = 1;
	    }
      }
    sqlite3_free_table (results);
    if (f_table_name && f_geometry_column && type && coord_dimension
	&& gc_srid && spatial_index_enabled)
	spatialite_legacy_gc = 1;
    if (f_table_name && f_geometry_column && geometry_type && coord_dimension
	&& gc_srid && spatial_index_enabled)
	spatialite_gc = 1;
    if (f_table_name && f_geometry_column && geometry_type && coord_dimension
	&& gc_srid && geometry_format)
	fdo_gc = 1;
/* checking the SPATIAL_REF_SYS table */
    strcpy (sql, "PRAGMA table_info(spatial_ref_sys)");
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, NULL);
    if (ret != SQLITE_OK)
	goto unknown;
    if (rows < 1)
	;
    else
      {
	  for (i = 1; i <= rows; i++)
	    {
		name = results[(i * columns) + 1];
		if (strcasecmp (name, "srid") == 0)
		    rs_srid = 1;
		if (strcasecmp (name, "auth_name") == 0)
		    auth_name = 1;
		if (strcasecmp (name, "auth_srid") == 0)
		    auth_srid = 1;
		if (strcasecmp (name, "srtext") == 0)
		    srtext = 1;
		if (strcasecmp (name, "ref_sys_name") == 0)
		    ref_sys_name = 1;
		if (strcasecmp (name, "proj4text") == 0)
		    proj4text = 1;
	    }
      }
    sqlite3_free_table (results);
    if (rs_srid && auth_name && auth_srid && ref_sys_name && proj4text
	&& srtext)
	spatialite_rs = 1;
    if (rs_srid && auth_name && auth_srid && ref_sys_name && proj4text)
	spatialite_legacy_rs = 1;
    if (rs_srid && auth_name && auth_srid && srtext)
	fdo_rs = 1;
/* verifying the MetaData format */
    if (spatialite_legacy_gc && spatialite_legacy_rs)
	return 1;
    if (fdo_gc && fdo_rs)
	return 2;
    if (spatialite_gc && spatialite_rs)
	return 3;
  unknown:
    if (checkGeoPackage (sqlite, db_prefix))
	return 4;
    return 0;
}

static void
add_fdo_table (struct fdo_table **first, struct fdo_table **last,
	       const char *table, int len)
{
/* adds an FDO-OGR styled Geometry Table to corresponding linked list */
    struct fdo_table *p = malloc (sizeof (struct fdo_table));
    p->table = malloc (len + 1);
    strcpy (p->table, table);
    p->next = NULL;
    if (!(*first))
	(*first) = p;
    if ((*last))
	(*last)->next = p;
    (*last) = p;
}

static void
free_fdo_tables (struct fdo_table *first)
{
/* memory cleanup; destroying the FDO-OGR tables linked list */
    struct fdo_table *p;
    struct fdo_table *pn;
    p = first;
    while (p)
      {
	  pn = p->next;
	  if (p->table)
	      free (p->table);
	  free (p);
	  p = pn;
      }
}

static void
fnct_AutoFDOStart (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AutoFDOStart(void)
/     or
/ AutoFDOStart(db_prefix TEXT)
/
/ for FDO-OGR interoperability:
/ tests the SpatialMetadata type, then automatically
/ creating a VirtualFDO table for each FDO-OGR main table 
/ declared within FDO-styled SpatialMetadata
/
*/
    int ret;
    const char *db_prefix = "main";
    const char *name;
    int i;
    char **results;
    int rows;
    int columns;
    char *sql_statement;
    int count = 0;
    struct fdo_table *first = NULL;
    struct fdo_table *last = NULL;
    struct fdo_table *p;
    int len;
    char *xname;
    char *xxname;
    char *xtable;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	      goto null_prefix;
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  db_prefix = (const char *) sqlite3_value_text (argv[0]);
      }
  null_prefix:
    if (checkSpatialMetaData_ex (sqlite, db_prefix) == 2)
      {
	  /* ok, creating VirtualFDO tables */
	  char *xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
	  sql_statement =
	      sqlite3_mprintf
	      ("SELECT DISTINCT f_table_name FROM \"%s\".geometry_columns",
	       xdb_prefix);
	  free (xdb_prefix);
	  ret = sqlite3_get_table (sqlite, sql_statement, &results, &rows,
				   &columns, NULL);
	  sqlite3_free (sql_statement);
	  if (ret != SQLITE_OK)
	      goto error;
	  if (rows < 1)
	      ;
	  else
	    {
		for (i = 1; i <= rows; i++)
		  {
		      name = results[(i * columns) + 0];
		      if (name)
			{
			    len = strlen (name);
			    add_fdo_table (&first, &last, name, len);
			}
		  }
	    }
	  sqlite3_free_table (results);
	  p = first;
	  while (p)
	    {
		/* destroying the VirtualFDO table [if existing] */
		xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
		xxname = sqlite3_mprintf ("fdo_%s", p->table);
		xname = gaiaDoubleQuotedSql (xxname);
		sqlite3_free (xxname);
		sql_statement =
		    sqlite3_mprintf ("DROP TABLE IF EXISTS \"%s\".\"%s\"",
				     xdb_prefix, xname);
		free (xname);
		free (xdb_prefix);
		ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
		sqlite3_free (sql_statement);
		if (ret != SQLITE_OK)
		    goto error;
		/* creating the VirtualFDO table */
		xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
		xxname = sqlite3_mprintf ("fdo_%s", p->table);
		xname = gaiaDoubleQuotedSql (xxname);
		sqlite3_free (xxname);
		xtable = gaiaDoubleQuotedSql (p->table);
		sql_statement =
		    sqlite3_mprintf
		    ("CREATE VIRTUAL TABLE \"%s\".\"%s\" USING VirtualFDO(\"%s\", \"%s\")",
		     xdb_prefix, xname, xdb_prefix, xtable);
		free (xname);
		free (xtable);
		free (xdb_prefix);
		ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
		sqlite3_free (sql_statement);
		if (ret != SQLITE_OK)
		    goto error;
		count++;
		p = p->next;
	    }
	error:
	  free_fdo_tables (first);
	  sqlite3_result_int (context, count);
	  return;
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_AutoFDOStop (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AutoFDOStop(void)
/     or
/ AutoFDOStop(db_prefix TEXT)
/
/ for FDO-OGR interoperability:
/ tests the SpatialMetadata type, then automatically
/ removes any VirtualFDO table 
/
*/
    int ret;
    const char *db_prefix = "main";
    const char *name;
    int i;
    char **results;
    int rows;
    int columns;
    char *sql_statement;
    int count = 0;
    struct fdo_table *first = NULL;
    struct fdo_table *last = NULL;
    struct fdo_table *p;
    int len;
    char *xname;
    char *xxname;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	      goto null_prefix;
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  db_prefix = (const char *) sqlite3_value_text (argv[0]);
      }
  null_prefix:
    if (checkSpatialMetaData_ex (sqlite, db_prefix) == 2)
      {
	  /* ok, removing VirtualFDO tables */
	  char *xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
	  sql_statement =
	      sqlite3_mprintf
	      ("SELECT DISTINCT f_table_name FROM \"%s\".geometry_columns",
	       xdb_prefix);
	  free (xdb_prefix);
	  ret = sqlite3_get_table (sqlite, sql_statement, &results, &rows,
				   &columns, NULL);
	  if (ret != SQLITE_OK)
	      goto error;
	  if (rows < 1)
	      ;
	  else
	    {
		for (i = 1; i <= rows; i++)
		  {
		      name = results[(i * columns) + 0];
		      if (name)
			{
			    len = strlen (name);
			    add_fdo_table (&first, &last, name, len);
			}
		  }
	    }
	  sqlite3_free_table (results);
	  p = first;
	  while (p)
	    {
		/* destroying the VirtualFDO table [if existing] */
		xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
		xxname = sqlite3_mprintf ("fdo_%s", p->table);
		xname = gaiaDoubleQuotedSql (xxname);
		sqlite3_free (xxname);
		sql_statement =
		    sqlite3_mprintf ("DROP TABLE IF EXISTS \"%s\".\"%s\"",
				     xdb_prefix, xname);
		free (xname);
		free (xdb_prefix);
		ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
		sqlite3_free (sql_statement);
		if (ret != SQLITE_OK)
		    goto error;
		count++;
		p = p->next;
	    }
	error:
	  free_fdo_tables (first);
	  sqlite3_result_int (context, count);
	  return;
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CheckSpatialMetaData (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ CheckSpatialMetaData(void)
/     or
/ CheckSpatialMetaData(db_prefix TEXT)
/
/ for FDO-OGR interoperability:
/ tests the SpatialMetadata type, returning:
/
/ -1 - on invalid args or if no ATTACHED-DB idenfied by db_prefix exists
/ 0 - if no valid SpatialMetaData were found
/ 1 - if SpatiaLite-legacy SpatialMetadata were found
/ 2 - if FDO-OGR-like SpatialMetadata were found
/ 3 - if SpatiaLite-current SpatialMetadata were found
/ 4 - if GeoPackage SpatialMetadata were found
/
*/
    const char *db_prefix = NULL;
    sqlite3 *sqlite;
    int ret;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  db_prefix = (const char *) sqlite3_value_text (argv[0]);
      }
    sqlite = sqlite3_context_db_handle (context);
    ret = checkSpatialMetaData_ex (sqlite, db_prefix);
    sqlite3_result_int (context, ret);
    return;
}

static void
fnct_InitSpatialMetaData (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ InitSpatialMetaData()
/     or
/ InitSpatialMetaData(text mode)
/     or
/ InitSpatialMetaData(integer transaction)
/     or
/ InitSpatialMetaData(integer transaction, text mode)
/
/ creates the SPATIAL_REF_SYS and GEOMETRY_COLUMNS tables
/ returns 1 on success
/ 0 on failure
*/
    char sql[8192];
    char *errMsg = NULL;
    int ret;
    int transaction = 0;
    const char *xmode;
    int mode = GAIA_EPSG_ANY;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	    {
		xmode = (const char *) sqlite3_value_text (argv[0]);
		if (strcasecmp (xmode, "NONE") == 0
		    || strcasecmp (xmode, "EMPTY") == 0)
		    mode = GAIA_EPSG_NONE;
		if (strcasecmp (xmode, "WGS84") == 0
		    || strcasecmp (xmode, "WGS84_ONLY") == 0)
		    mode = GAIA_EPSG_WGS84_ONLY;
	    }
	  else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	      transaction = sqlite3_value_int (argv[0]);
	  else
	    {
		spatialite_e
		    ("InitSpatialMetaData() error: argument 1 is not of the String or Integer type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
	    {
		spatialite_e
		    ("InitSpatialMetaData() error: argument 1 is not of the Integer type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("InitSpatialMetaData() error: argument 2 is not of the String type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  transaction = sqlite3_value_int (argv[0]);
	  xmode = (const char *) sqlite3_value_text (argv[1]);
	  if (strcasecmp (xmode, "NONE") == 0
	      || strcasecmp (xmode, "EMPTY") == 0)
	      mode = GAIA_EPSG_NONE;
	  if (strcasecmp (xmode, "WGS84") == 0
	      || strcasecmp (xmode, "WGS84_ONLY") == 0)
	      mode = GAIA_EPSG_WGS84_ONLY;
      }

    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }

/* creating the SPATIAL_REF_SYS table */
    strcpy (sql, "CREATE TABLE spatial_ref_sys (\n");
    strcat (sql, "srid INTEGER NOT NULL PRIMARY KEY,\n");
    strcat (sql, "auth_name TEXT NOT NULL,\n");
    strcat (sql, "auth_srid INTEGER NOT NULL,\n");
    strcat (sql, "ref_sys_name TEXT NOT NULL DEFAULT 'Unknown',\n");
    strcat (sql, "proj4text TEXT NOT NULL,\n");
    strcat (sql, "srtext TEXT NOT NULL DEFAULT 'Undefined')");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;
    strcpy (sql, "CREATE UNIQUE INDEX idx_spatial_ref_sys \n");
    strcat (sql, "ON spatial_ref_sys (auth_srid, auth_name)");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;
    updateSpatiaLiteHistory (sqlite, "spatial_ref_sys", NULL,
			     "table successfully created");

/* creating the GEOMETRY_COLUMNS table */
    if (!createGeometryColumns (sqlite))
	goto error;

/* creating the GEOM_COLS_REF_SYS view */
    strcpy (sql, "CREATE VIEW geom_cols_ref_sys AS\n");
    strcat (sql, "SELECT f_table_name, f_geometry_column, geometry_type,\n");
    strcat (sql, "coord_dimension, spatial_ref_sys.srid AS srid,\n");
    strcat (sql, "auth_name, auth_srid, ref_sys_name, proj4text, srtext\n");
    strcat (sql, "FROM geometry_columns, spatial_ref_sys\n");
    strcat (sql, "WHERE geometry_columns.srid = spatial_ref_sys.srid");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    updateSpatiaLiteHistory (sqlite, "geom_cols_ref_sys", NULL,
			     "view 'geom_cols_ref_sys' successfully created");
    if (ret != SQLITE_OK)
	goto error;
    if (spatial_ref_sys_init2 (sqlite, mode, 0))
      {
	  if (mode == GAIA_EPSG_NONE)
	      updateSpatiaLiteHistory (sqlite, "spatial_ref_sys", NULL,
				       "table successfully created [empty]");
	  else
	      updateSpatiaLiteHistory (sqlite, "spatial_ref_sys", NULL,
				       "table successfully populated");
      }
    if (!createAdvancedMetaData (sqlite))
	goto error;
/* creating the SpatialIndex VIRTUAL TABLE */
    strcpy (sql, "CREATE VIRTUAL TABLE SpatialIndex ");
    strcat (sql, "USING VirtualSpatialIndex()");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;
/* creating the ElementaryGeometries VIRTUAL TABLE */
    strcpy (sql, "CREATE VIRTUAL TABLE ElementaryGeometries ");
    strcat (sql, "USING VirtualElementary()");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;

/* creating the KNN2 VIRTUAL TABLE */
    strcpy (sql, "CREATE VIRTUAL TABLE KNN2 ");
    strcat (sql, "USING VirtualKNN2()");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;

    if (transaction)
      {
	  /* confirming the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }

    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("InitSpatiaMetaData() error:\"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    if (transaction)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e (" InitSpatiaMetaData() error:\"%s\"\n", errMsg);
		sqlite3_free (errMsg);
	    }
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_InitAdvancedMetaData (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ InitAdvancedlMetaData()
/     or
/ InitAdvancedMetaData(integer transaction)
/
/ safely creates several ancillary MetaData tables
/ returns 1 on success
/ 0 on failure
*/
    char sql[8192];
    char *errMsg = NULL;
    int ret;
    int transaction = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
	    {
		spatialite_e
		    ("InitAdvancedMetaData() error: argument 1 is not of the Integer type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  transaction = sqlite3_value_int (argv[0]);
      }

    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
/* creating the GEOM_COLS_REF_SYS view */
    strcpy (sql, "CREATE VIEW IF NOT EXISTS geom_cols_ref_sys AS\n");
    strcat (sql, "SELECT f_table_name, f_geometry_column, geometry_type,\n");
    strcat (sql, "coord_dimension, spatial_ref_sys.srid AS srid,\n");
    strcat (sql, "auth_name, auth_srid, ref_sys_name, proj4text, srtext\n");
    strcat (sql, "FROM geometry_columns, spatial_ref_sys\n");
    strcat (sql, "WHERE geometry_columns.srid = spatial_ref_sys.srid");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    updateSpatiaLiteHistory (sqlite, "geom_cols_ref_sys", NULL,
			     "view 'geom_cols_ref_sys' successfully created");
    if (ret != SQLITE_OK)
	goto error;
    if (!createAdvancedMetaData (sqlite))
	goto error;
/* creating the SpatialIndex VIRTUAL TABLE */
    strcpy (sql, "CREATE VIRTUAL TABLE IF NOT EXISTS SpatialIndex ");
    strcat (sql, "USING VirtualSpatialIndex()");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;
/* creating the ElementaryGeometries VIRTUAL TABLE */
    strcpy (sql, "CREATE VIRTUAL TABLE IF NOT EXISTS ElementaryGeometries ");
    strcat (sql, "USING VirtualElementary()");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;

/* creating the KNN VIRTUAL TABLE */
    strcpy (sql, "CREATE VIRTUAL TABLE IF NOT EXISTS KNN2 ");
    strcat (sql, "USING VirtualKNN2()");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;

    if (transaction)
      {
	  /* confirming the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }

    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("InitSpatiaMetaData() error:\"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    if (transaction)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e (" InitSpatiaMetaData() error:\"%s\"\n", errMsg);
		sqlite3_free (errMsg);
	    }
      }
    sqlite3_result_int (context, 0);
    return;
}

static int
do_execute_sql_with_retval (sqlite3 * sqlite, const char *sql, char **errMsg)
{
/* helper function for InitSpatialMetaDataFull */
    int retval = 0;
    int ret;
    int i;
    char **results;
    int rows;
    int columns;
    char *msg = NULL;

    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, &msg);
    if (ret != SQLITE_OK)
	goto end;
    if (rows < 1)
	;
    else
      {
	  for (i = 1; i <= rows; i++)
	    {
		if (atoi (results[(i * columns) + 0]) == 1)
		    retval = 1;
	    }
      }
    sqlite3_free_table (results);

  end:
    *errMsg = msg;
    return retval;
}

static void
fnct_InitSpatialMetaDataFull (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ InitSpatialMetaDataFull()
/     or
/ InitSpatialMetaDataFull(text mode)
/     or
/ InitSpatialMetaDataFull(integer transaction)
/     or
/ InitSpatialMetaDataFull(integer transaction, text mode)
/
/ conveniency "super" function internally calling in a single shot:
/     - InitSpatialMetaData()
/     - CreateIsoMetadataTables()
/     - CreateRasterCoveragesTable()
/     - CreateVectorCoveragesTables()
/     - CreateTopoTables()
/     - CreateStylingTables()
/     - WMS_CreateTables()
/     - StoredProc_CreateTables()
/ returns 1 on success
/ 0 on failure
*/
    char *errMsg = NULL;
    int ret;
    int transaction = 0;
    const char *xmode = NULL;
    int retval;
    char *sql;
    int ok_isometa = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	    {
		xmode = (const char *) sqlite3_value_text (argv[0]);
	    }
	  else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	      transaction = sqlite3_value_int (argv[0]);
	  else
	    {
		spatialite_e
		    ("InitSpatialMetaDataFull() error: argument 1 is not of the String or Integer type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
	    {
		spatialite_e
		    ("InitSpatialMetaDataFull() error: argument 1 is not of the Integer type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("InitSpatialMetaDataFull() error: argument 2 is not of the String type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  transaction = sqlite3_value_int (argv[0]);
	  xmode = (const char *) sqlite3_value_text (argv[1]);
      }

    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }

/* executing InitSpatialMetaData() */
    if (xmode != NULL)
	sql = sqlite3_mprintf ("SELECT InitSpatialMetaData(0, %Q)", xmode);
    else
	sql = sqlite3_mprintf ("SELECT InitSpatialMetaData(0)");
    retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
    sqlite3_free (sql);
    if (retval != 1)
	goto error;
    if (!createAdvancedMetaData (sqlite))
	goto error;

#ifdef ENABLE_LIBXML2		/* only if LibXML2 support is available */
    ok_isometa = 1;
    if (xmode != NULL)
      {
	  if (strcasecmp (xmode, "NONE") == 0
	      || strcasecmp (xmode, "EMPTY") == 0)
	      ok_isometa = 0;
      }
    if (ok_isometa)
      {
	  /* executing CreateIsoMetadataTables() */
	  sql = sqlite3_mprintf ("SELECT CreateIsoMetadataTables()");
	  retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
	  sqlite3_free (sql);
	  if (retval != 1)
	      goto error;
      }
#endif

/* executing CreateRasterCoveragesTable() */
    sql = sqlite3_mprintf ("SELECT CreateRasterCoveragesTable()");
    retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
    sqlite3_free (sql);
    if (retval != 1)
	goto error;

/* executing CreateVectorCoveragesTables() */
    sql = sqlite3_mprintf ("SELECT CreateVectorCoveragesTables()");
    retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
    sqlite3_free (sql);
    if (retval != 1)
	goto error;

#ifdef ENABLE_RTTOPO		/* only if RtTopo support is available */
/* executing CreateTopoTables() */
    sql = sqlite3_mprintf ("SELECT CreateTopoTables()");
    retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
    sqlite3_free (sql);
    if (retval != 1)
	goto error;
#endif

#ifdef ENABLE_LIBXML2		/* only if LibXML2 support is available */
/* executing CreateStylingTables() */
    sql = sqlite3_mprintf ("SELECT CreateStylingTables()");
    retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
    sqlite3_free (sql);
    if (retval != 1)
	goto error;
#endif

/* executing WMS_CreateTables() */
    sql = sqlite3_mprintf ("SELECT WMS_CreateTables()");
    retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
    sqlite3_free (sql);
    if (retval != 1)
	goto error;

/* executing StoredProc_CreateTables() */
    sql = sqlite3_mprintf ("SELECT StoredProc_CreateTables()");
    retval = do_execute_sql_with_retval (sqlite, sql, &errMsg);
    sqlite3_free (sql);
    if (retval != 1)
	goto error;

    if (transaction)
      {
	  /* confirming the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }

    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("InitSpatiaMetaDataFull() error:\"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    if (transaction)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e (" InitSpatiaMetaDataFull() error:\"%s\"\n",
			      errMsg);
		sqlite3_free (errMsg);
	    }
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CloneTable (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CloneTable(text db_prefix, text in_table, text out_table, integer transaction)
/ CloneTable(text db_prefix, text in_table, text out_table, integer transaction,
/            ... text option1 ..., ... text option2 ..., text option10)
/
/ cloning a whole table [CREATE and then COPY]
/ returns 1 on success
/ 0 on failure (NULL on invalid arguments)
*/
    int ret;
    char *errMsg = NULL;
    const char *db_prefix;
    const char *in_table;
    const char *out_table;
    int transaction = 0;
    int active = 0;
    const void *cloner = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	db_prefix = "MAIN";
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	db_prefix = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  spatialite_e
	      ("CloneTable() error: argument 1 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	in_table = (const char *) sqlite3_value_text (argv[1]);
    else
      {
	  spatialite_e
	      ("CloneTable() error: argument 2 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	out_table = (const char *) sqlite3_value_text (argv[2]);
    else
      {
	  spatialite_e
	      ("CloneTable() error: argument 3 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	transaction = sqlite3_value_int (argv[3]);
    else
      {
	  spatialite_e
	      ("CloneTable() error: argument 4 is not of the Integer type\n");
	  sqlite3_result_null (context);
	  return;
      }

/* additional options */
    if (argc > 4 && sqlite3_value_type (argv[4]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 5 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 5 && sqlite3_value_type (argv[5]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 6 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 6 && sqlite3_value_type (argv[6]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 7 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 7 && sqlite3_value_type (argv[7]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 8 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 8 && sqlite3_value_type (argv[8]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 9 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 9 && sqlite3_value_type (argv[9]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 10 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 10 && sqlite3_value_type (argv[10]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 11 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 11 && sqlite3_value_type (argv[11]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 12 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 12 && sqlite3_value_type (argv[12]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 13 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 13 && sqlite3_value_type (argv[13]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CloneTable() error: argument 14 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }

    cloner = gaiaAuxClonerCreate (sqlite, db_prefix, in_table, out_table);
    if (cloner == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }

/* additional options */
    if (argc > 4)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[4]));
    if (argc > 5)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[5]));
    if (argc > 6)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[6]));
    if (argc > 7)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[7]));
    if (argc > 8)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[8]));
    if (argc > 9)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[9]));
    if (argc > 10)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[10]));
    if (argc > 11)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[11]));
    if (argc > 12)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[12]));
    if (argc > 13)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[13]));

    if (!gaiaAuxClonerCheckValidTarget (cloner))
	goto error;

    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    active = 1;

    if (!gaiaAuxClonerExecute (cloner))
	goto error;
    gaiaAuxClonerDestroy (cloner);
    cloner = NULL;
    updateSpatiaLiteHistory (sqlite, out_table, NULL,
			     "table successfully cloned");

    if (transaction)
      {
	  /* confirming the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }

    sqlite3_result_int (context, 1);
    return;
  error:
    if (cloner != NULL)
	gaiaAuxClonerDestroy (cloner);
    spatialite_e ("CloneTable() error:\"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    if (transaction && active)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("CloneTable() error:\"%s\"\n", errMsg);
		sqlite3_free (errMsg);
	    }
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CreateClonedTable (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ CreateClonedTable(text db_prefix, text in_table, text out_table, integer transaction)
/ CreateClonedTable(text db_prefix, text in_table, text out_table, integer transaction,
/            ... text option1 ..., ... text option2 ..., text option10)
/
/ creating a cloned table [CREATE only without COPYing]
/ returns 1 on success
/ 0 on failure (NULL on invalid arguments)
*/
    int ret;
    char *errMsg = NULL;
    const char *db_prefix;
    const char *in_table;
    const char *out_table;
    int transaction = 0;
    int active = 0;
    const void *cloner = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	db_prefix = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 1 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	in_table = (const char *) sqlite3_value_text (argv[1]);
    else
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 2 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	out_table = (const char *) sqlite3_value_text (argv[2]);
    else
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 3 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	transaction = sqlite3_value_int (argv[3]);
    else
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 4 is not of the Integer type\n");
	  sqlite3_result_null (context);
	  return;
      }


/* additional options */
    if (argc > 4 && sqlite3_value_type (argv[4]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 5 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 5 && sqlite3_value_type (argv[5]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 6 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 6 && sqlite3_value_type (argv[6]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 7 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 7 && sqlite3_value_type (argv[7]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 8 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 8 && sqlite3_value_type (argv[8]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 9 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 9 && sqlite3_value_type (argv[9]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 10 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 10 && sqlite3_value_type (argv[10]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 11 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 11 && sqlite3_value_type (argv[11]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 12 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 12 && sqlite3_value_type (argv[12]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 13 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 13 && sqlite3_value_type (argv[13]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateClonedTable() error: argument 14 is not of the String or TEXT type\n");
	  sqlite3_result_null (context);
	  return;
      }

    cloner = gaiaAuxClonerCreateEx (sqlite, db_prefix, in_table, out_table, 1);
    if (cloner == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }

/* additional options */
    if (argc > 4)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[4]));
    if (argc > 5)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[5]));
    if (argc > 6)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[6]));
    if (argc > 7)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[7]));
    if (argc > 8)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[8]));
    if (argc > 9)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[9]));
    if (argc > 10)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[10]));
    if (argc > 11)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[11]));
    if (argc > 12)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[12]));
    if (argc > 13)
	gaiaAuxClonerAddOption (cloner,
				(const char *) sqlite3_value_text (argv[13]));

    if (!gaiaAuxClonerCheckValidTarget (cloner))
	goto error;

    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    active = 1;

    if (!gaiaAuxClonerExecute (cloner))
	goto error;
    gaiaAuxClonerDestroy (cloner);
    updateSpatiaLiteHistory (sqlite, out_table, NULL,
			     "table successfully cloned");

    if (transaction)
      {
	  /* confirming the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }

    sqlite3_result_int (context, 1);
    return;
  error:
    if (cloner != NULL)
	gaiaAuxClonerDestroy (cloner);
    spatialite_e ("CreateClonedTable() error:\"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    if (transaction && active)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("CreateClonedTable() error:\"%s\"\n", errMsg);
		sqlite3_free (errMsg);
	    }
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CheckGeoPackageMetaData (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ CheckGeoPackageMetaData(void)
/     or
/ CheckGeoPackageMetaData(db_prefix TEXT)
/
/ for OGC GeoPackage interoperability:
/ tests if GeoPackage metadata tables are found
/
*/
    const char *db_prefix = NULL;
    sqlite3 *sqlite;
    int ret;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  db_prefix = (const char *) sqlite3_value_text (argv[0]);
      }
    sqlite = sqlite3_context_db_handle (context);
    ret = checkGeoPackage (sqlite, db_prefix);
    sqlite3_result_int (context, ret);
    return;
}

#ifdef ENABLE_GEOPACKAGE	/* enabling GeoPackage extensions */

static void
add_gpkg_table (struct gpkg_table **first, struct gpkg_table **last,
		const char *table, int len)
{
/* adds a GPKG Geometry Table to corresponding linked list */
    struct gpkg_table *p = malloc (sizeof (struct gpkg_table));
    p->table = malloc (len + 1);
    strcpy (p->table, table);
    p->next = NULL;
    if (!(*first))
	(*first) = p;
    if ((*last))
	(*last)->next = p;
    (*last) = p;
}

static void
free_gpkg_tables (struct gpkg_table *first)
{
/* memory cleanup; destroying the GPKG tables linked list */
    struct gpkg_table *p;
    struct gpkg_table *pn;
    p = first;
    while (p)
      {
	  pn = p->next;
	  if (p->table)
	      free (p->table);
	  free (p);
	  p = pn;
      }
}

static void
fnct_AutoGPKGStart (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AutoGPKGStart(void)
/     or
/ AutoGPKGStart(db_prefix TEXT)
/
/ for OCG GeoPackage interoperability:
/ tests the DB layout, then automatically
/ creating a VirtualGPGK table for each GPKG main table 
/ declared within gpkg_geometry_colums
/
*/
    int ret;
    const char *db_prefix = "main";
    const char *name;
    int i;
    char **results;
    int rows;
    int columns;
    char *sql_statement;
    int count = 0;
    struct gpkg_table *first = NULL;
    struct gpkg_table *last = NULL;
    struct gpkg_table *p;
    int len;
    char *xname;
    char *xxname;
    char *xtable;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	      goto null_prefix;
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  db_prefix = (const char *) sqlite3_value_text (argv[0]);
      }
  null_prefix:
    if (checkGeoPackage (sqlite, db_prefix))
      {
	  /* ok, creating VirtualGPKG tables */
	  char *xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
	  sql_statement =
	      sqlite3_mprintf
	      ("SELECT DISTINCT table_name FROM \"%s\".gpkg_geometry_columns",
	       xdb_prefix);
	  free (xdb_prefix);
	  ret =
	      sqlite3_get_table (sqlite, sql_statement, &results, &rows,
				 &columns, NULL);
	  sqlite3_free (sql_statement);
	  if (ret != SQLITE_OK)
	      goto error;
	  if (rows < 1)
	      ;
	  else
	    {
		for (i = 1; i <= rows; i++)
		  {
		      name = results[(i * columns) + 0];
		      if (name)
			{
			    len = strlen (name);
			    add_gpkg_table (&first, &last, name, len);
			}
		  }
	    }
	  sqlite3_free_table (results);
	  p = first;
	  while (p)
	    {
		/* destroying the VirtualGPKG table [if existing] */
		xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
		xxname = sqlite3_mprintf ("vgpkg_%s", p->table);
		xname = gaiaDoubleQuotedSql (xxname);
		sqlite3_free (xxname);
		sql_statement =
		    sqlite3_mprintf ("DROP TABLE IF EXISTS \"%s\".\"%s\"",
				     xdb_prefix, xname);
		free (xname);
		free (xdb_prefix);
		ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
		sqlite3_free (sql_statement);
		if (ret != SQLITE_OK)
		    goto error;
		/* creating the VirtualGPKG table */
		xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
		xxname = sqlite3_mprintf ("vgpkg_%s", p->table);
		xname = gaiaDoubleQuotedSql (xxname);
		sqlite3_free (xxname);
		xtable = gaiaDoubleQuotedSql (p->table);
		sql_statement =
		    sqlite3_mprintf
		    ("CREATE VIRTUAL TABLE \"%s\".\"%s\" USING VirtualGPKG(\"%s\", \"%s\")",
		     xdb_prefix, xname, xdb_prefix, xtable);
		free (xname);
		free (xtable);
		free (xdb_prefix);
		ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
		sqlite3_free (sql_statement);
		if (ret != SQLITE_OK)
		    goto error;
		count++;
		p = p->next;
	    }
	error:
	  free_gpkg_tables (first);
	  sqlite3_result_int (context, count);
	  return;
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_AutoGPKGStop (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AutoGPKGStop(void)
/     or
/ AutoGPKGStop(db_prefix TEXT)
/
/ for OGC GeoPackage interoperability:
/ tests the DB layout, then automatically removes any VirtualGPKG table 
/
*/
    int ret;
    const char *db_prefix = "main";
    const char *name;
    int i;
    char **results;
    int rows;
    int columns;
    char *sql_statement;
    int count = 0;
    struct gpkg_table *first = NULL;
    struct gpkg_table *last = NULL;
    struct gpkg_table *p;
    int len;
    char *xname;
    char *xxname;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	      goto null_prefix;
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  db_prefix = (const char *) sqlite3_value_text (argv[0]);
      }
  null_prefix:
    if (checkGeoPackage (sqlite, db_prefix))
      {
	  /* ok, removing VirtualGPKG tables */
	  char *xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
	  sql_statement =
	      sqlite3_mprintf
	      ("SELECT DISTINCT table_name FROM \"%s\".gpkg_geometry_columns",
	       xdb_prefix);
	  free (xdb_prefix);
	  ret =
	      sqlite3_get_table (sqlite, sql_statement, &results, &rows,
				 &columns, NULL);
	  sqlite3_free (sql_statement);
	  if (ret != SQLITE_OK)
	      goto error;
	  if (rows < 1)
	      ;
	  else
	    {
		for (i = 1; i <= rows; i++)
		  {
		      name = results[(i * columns) + 0];
		      if (name)
			{
			    len = strlen (name);
			    add_gpkg_table (&first, &last, name, len);
			}
		  }
	    }
	  sqlite3_free_table (results);
	  p = first;
	  while (p)
	    {
		/* destroying the VirtualGPKG table [if existing] */
		xdb_prefix = gaiaDoubleQuotedSql (db_prefix);
		xxname = sqlite3_mprintf ("vgpkg_%s", p->table);
		xname = gaiaDoubleQuotedSql (xxname);
		sqlite3_free (xxname);
		sql_statement =
		    sqlite3_mprintf ("DROP TABLE IF EXISTS \"%s\".\"%s\"",
				     xdb_prefix, xname);
		free (xname);
		free (xdb_prefix);
		ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
		sqlite3_free (sql_statement);
		if (ret != SQLITE_OK)
		    goto error;
		count++;
		p = p->next;
	    }
	error:
	  free_gpkg_tables (first);
	  sqlite3_result_int (context, count);
	  return;
      }
    sqlite3_result_int (context, 0);
    return;
}

#endif /* end enabling GeoPackage extensions */

static void
fnct_InsertEpsgSrid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ InsertEpsgSrid(int srid)
/
/ returns 1 on success: 0 on failure
*/
    int srid;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_int (context, 0);
	  return;
      }
    ret = insert_epsg_srid (sqlite, srid);
    if (!ret)
	sqlite3_result_int (context, 0);
    else
	sqlite3_result_int (context, 1);
}

static void
fnct_SridIsGeographic (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ SridIsGeographic(int srid)
/
/ returns 1 on success: 0 on failure
/ NULL on invalid argument
*/
    int srid;
    int ret;
    int geographic;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    ret = srid_is_geographic (sqlite, srid, &geographic);
    if (!ret)
	sqlite3_result_null (context);
    else
      {
	  if (geographic)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
}

static void
fnct_SridIsProjected (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ SridIsProjected(int srid)
/
/ returns 1 on success: 0 on failure
/ NULL on invalid argument
*/
    int srid;
    int ret;
    int projected;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    ret = srid_is_projected (sqlite, srid, &projected);
    if (!ret)
	sqlite3_result_null (context);
    else
      {
	  if (projected)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
}

static void
fnct_SridHasFlippedAxes (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ SridHasFlippedAxes(int srid)
/
/ returns 1 on success: 0 on failure
/ NULL on invalid argument
*/
    int srid;
    int ret;
    int flipped;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    ret = srid_has_flipped_axes (sqlite, srid, &flipped);
    if (!ret)
	sqlite3_result_null (context);
    else
      {
	  if (flipped)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
}

static void
fnct_SridGetSpheroid (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ SridGetSpheroid(int srid)
/ or
/ SridGetEllipsoid(int srid)
/
/ returns the name of the Spheroid on success
/ NULL on failure or on invalid argument
*/
    int srid;
    char *spheroid = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    spheroid = srid_get_spheroid (sqlite, srid);
    if (spheroid == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, spheroid, strlen (spheroid), free);
}

static void
fnct_SridGetPrimeMeridian (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ SridGetPrimeMeridian(int srid)
/
/ returns the name of the Prime Meridian on success
/ NULL on failure or on invalid argument
*/
    int srid;
    char *prime_meridian = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    prime_meridian = srid_get_prime_meridian (sqlite, srid);
    if (prime_meridian == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, prime_meridian, strlen (prime_meridian),
			     free);
}

static void
fnct_SridGetProjection (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ SridGetProjection(int srid)
/
/ returns the name of the Projection on success
/ NULL on failure or on invalid argument
*/
    int srid;
    char *projection = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    projection = srid_get_projection (sqlite, srid);
    if (projection == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, projection, strlen (projection), free);
}

static void
fnct_SridGetDatum (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SridGetDatum(int srid)
/
/ returns the name of the Datum on success
/ NULL on failure or on invalid argument
*/
    int srid;
    char *datum = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    datum = srid_get_datum (sqlite, srid);
    if (datum == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, datum, strlen (datum), free);
}

static void
fnct_SridGetUnit (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SridGetUnit(int srid)
/
/ returns the name of the Spheroid on success
/ NULL on failure or on invalid argument
*/
    int srid;
    char *unit = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    unit = srid_get_unit (sqlite, srid);
    if (unit == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, unit, strlen (unit), free);
}

static void
common_srid_axis (sqlite3_context * context, int argc,
		  sqlite3_value ** argv, char axis, char mode)
{
/* commonn implentation - SRID Get Axis */
    int srid;
    char *result = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    result = srid_get_axis (sqlite, srid, axis, mode);
    if (result == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, result, strlen (result), free);
}

static void
fnct_SridGetAxis1Name (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ SridGetAxis_1_Name(int srid)
/
/ returns the name of the first Axis on success
/ NULL on failure or on invalid argument
*/
    common_srid_axis (context, argc, argv, SPLITE_AXIS_1, SPLITE_AXIS_NAME);
}

static void
fnct_SridGetAxis1Orientation (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ SridGetAxis_1_Orientation(int srid)
/
/ returns the orientation of the first Axis on success
/ NULL on failure or on invalid argument
*/
    common_srid_axis (context, argc, argv, SPLITE_AXIS_1,
		      SPLITE_AXIS_ORIENTATION);
}

static void
fnct_SridGetAxis2Name (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ SridGetAxis_2_Name(int srid)
/
/ returns the name of the second Axis on success
/ NULL on failure or on invalid argument
*/
    common_srid_axis (context, argc, argv, SPLITE_AXIS_2, SPLITE_AXIS_NAME);
}

static void
fnct_SridGetAxis2Orientation (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ SridGetAxis_2_Orientation(int srid)
/
/ returns the orientation of the second Axis on success
/ NULL on failure or on invalid argument
*/
    common_srid_axis (context, argc, argv, SPLITE_AXIS_2,
		      SPLITE_AXIS_ORIENTATION);
}

static int
recoverGeomColumn (sqlite3 * sqlite, const char *table,
		   const char *column, int xtype, int dims, int srid)
{
/* checks if TABLE.COLUMN exists and has the required features */
    int ok = 1;
    int type;
    sqlite3_stmt *stmt;
    gaiaGeomCollPtr geom;
    const void *blob_value;
    int len;
    int ret;
    int i_col;
    int is_nullable = 1;
    char *p_table;
    char *p_column;
    char *sql_statement;

/* testing if NOT NULL */
    p_table = gaiaDoubleQuotedSql (table);
    sql_statement = sqlite3_mprintf ("PRAGMA table_info(\"%s\")", p_table);
    free (p_table);
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("recoverGeomColumn: \"%s\"\n", sqlite3_errmsg (sqlite));
	  return 0;
      }
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	    {
		if (strcasecmp
		    (column, (const char *) sqlite3_column_text (stmt, 1)) == 0)
		  {
		      if (sqlite3_column_int (stmt, 2) != 0)
			  is_nullable = 0;
		  }
	    }
      }
    sqlite3_finalize (stmt);

    p_table = gaiaDoubleQuotedSql (table);
    p_column = gaiaDoubleQuotedSql (column);
    sql_statement =
	sqlite3_mprintf ("SELECT \"%s\" FROM \"%s\"", p_column, p_table);
    free (p_table);
    free (p_column);
/* compiling SQL prepared statement */
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("recoverGeomColumn: error %d \"%s\"\n",
			sqlite3_errcode (sqlite), sqlite3_errmsg (sqlite));
	  return 0;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
    sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	    {
		/* checking Geometry features */
		geom = NULL;
		for (i_col = 0; i_col < sqlite3_column_count (stmt); i_col++)
		  {
		      if (sqlite3_column_type (stmt, i_col) == SQLITE_NULL)
			{
			    /* found a NULL geometry */
			    if (!is_nullable)
				ok = 0;
			}
		      else if (sqlite3_column_type (stmt, i_col) != SQLITE_BLOB)
			  ok = 0;
		      else
			{
			    blob_value = sqlite3_column_blob (stmt, i_col);
			    len = sqlite3_column_bytes (stmt, i_col);
			    geom = gaiaFromSpatiaLiteBlobWkb (blob_value, len);
			    if (!geom)
				ok = 0;
			    else
			      {
				  if (geom->DimensionModel != dims)
				      ok = 0;
				  if (geom->Srid != srid)
				      ok = 0;
				  type = gaiaGeometryType (geom);
				  if (xtype == -1)
				      ;	/* GEOMETRY */
				  else
				    {
					if (xtype == type)
					    ;
					else
					    ok = 0;
				    }
				  gaiaFreeGeomColl (geom);
			      }
			}
		  }
	    }
	  if (!ok)
	      break;
      }
    ret = sqlite3_finalize (stmt);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("recoverGeomColumn: error %d \"%s\"\n",
			sqlite3_errcode (sqlite), sqlite3_errmsg (sqlite));
	  return 0;
      }
    return ok;
}

static void
fnct_AddGeometryColumn (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ AddGeometryColumn(table, column, srid, type [ , dimension  [  , not-null ] ] )
/
/ creates a new COLUMN of given TYPE into TABLE
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    const unsigned char *type;
    const unsigned char *txt_dims;
    int xtype;
    int srid = -1;
    int srid_exists = -1;
    int dimension = 2;
    int dims = -1;
    int auto_dims = -1;
    char sql[1024];
    int ret;
    int notNull = 0;
    int metadata_version;
    sqlite3_stmt *stmt;
    char *p_table = NULL;
    char *quoted_table;
    char *quoted_column;
    const char *p_type = NULL;
    const char *p_dims = NULL;
    int n_type = 0;
    int n_dims = 0;
    char *sql_statement;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddGeometryColumn() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddGeometryColumn() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("AddGeometryColumn() error: argument 3 [SRID] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    srid = sqlite3_value_int (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddGeometryColumn() error: argument 4 [geometry_type] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    type = sqlite3_value_text (argv[3]);
    if (argc > 4)
      {
	  if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	    {
		dimension = sqlite3_value_int (argv[4]);
		if (dimension == 2)
		    dims = GAIA_XY;
		if (dimension == 3)
		    dims = GAIA_XY_Z;
		if (dimension == 4)
		    dims = GAIA_XY_Z_M;
	    }
	  else if (sqlite3_value_type (argv[4]) == SQLITE_TEXT)
	    {
		txt_dims = sqlite3_value_text (argv[4]);
		if (strcasecmp ((char *) txt_dims, "XY") == 0)
		    dims = GAIA_XY;
		if (strcasecmp ((char *) txt_dims, "XYZ") == 0)
		    dims = GAIA_XY_Z;
		if (strcasecmp ((char *) txt_dims, "XYM") == 0)
		    dims = GAIA_XY_M;
		if (strcasecmp ((char *) txt_dims, "XYZM") == 0)
		    dims = GAIA_XY_Z_M;
	    }
	  else
	    {
		spatialite_e
		    ("AddGeometryColumn() error: argument 5 [dimension] is not of the Integer or Text type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
      }
    if (argc == 6)
      {
	  /* optional NOT NULL arg */
	  if (sqlite3_value_type (argv[5]) != SQLITE_INTEGER)
	    {
		spatialite_e
		    ("AddGeometryColumn() error: argument 6 [not null] is not of the Integer type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  notNull = sqlite3_value_int (argv[5]);
      }
    xtype = GAIA_UNKNOWN;
    if (strcasecmp ((char *) type, "POINT") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRING") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGON") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINT") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRING") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGON") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTION") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRY") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = -1;
      }
    if (xtype == GAIA_UNKNOWN)
      {
	  spatialite_e
	      ("AddGeometryColumn() error: argument 4 [geometry_type] has an illegal value\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (dims < 0)
	dims = auto_dims;
    if (dims == GAIA_XY || dims == GAIA_XY_Z || dims == GAIA_XY_M
	|| dims == GAIA_XY_Z_M)
	;
    else
      {
	  spatialite_e
	      ("AddGeometryColumn() error: argument 5 [dimension] ILLEGAL VALUE\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (auto_dims != GAIA_XY && dims != auto_dims)
      {
	  spatialite_e
	      ("AddGeometryColumn() error: argument 5 [dimension] ILLEGAL VALUE\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
/* checking if the table exists */
    strcpy (sql,
	    "SELECT name FROM sqlite_master WHERE type = 'table' AND Lower(name) = Lower(?)");
    ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("AddGeometryColumn: \"%s\"\n", sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	    {
		if (p_table != NULL)
		    sqlite3_free (p_table);
		p_table =
		    sqlite3_mprintf ("%s",
				     (const char *) sqlite3_column_text (stmt,
									 0));
	    }
      }
    sqlite3_finalize (stmt);
    if (!p_table)
      {
	  spatialite_e
	      ("AddGeometryColumn() error: table '%s' does not exist\n", table);
	  sqlite3_result_int (context, 0);
	  return;
      }
/* checking for WITHOUT ROWID */
    if (is_without_rowid_table (sqlite, table))
      {
	  spatialite_e
	      ("AddGeometryColumn() error: table '%s' is WITHOUT ROWID\n",
	       table);
	  sqlite3_result_int (context, 0);
	  return;
      }
    metadata_version = checkSpatialMetaData (sqlite);
    if (metadata_version == 1 || metadata_version == 3)
	;
    else
      {
	  spatialite_e
	      ("AddGeometryColumn() error: unexpected metadata layout\n");
	  sqlite3_result_int (context, 0);
	  return;
      }

/*
 * the following code has been contributed by Mark Johnson <mj10777@googlemail.com>
 * on 2019-01-26
*/
    sql_statement = NULL;
    sql_statement =
	sqlite3_mprintf
	("SELECT CASE WHEN (Exists(SELECT srid FROM spatial_ref_sys WHERE (auth_srid = %d)) = 0) THEN 0 ELSE 1 END",
	 srid);
    ret =
	sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			    &stmt, NULL);
    sqlite3_free (sql_statement);
    sql_statement = NULL;
    while (sqlite3_step (stmt) == SQLITE_ROW)
      {
	  if (sqlite3_column_type (stmt, 0) != SQLITE_NULL)
	    {
		srid_exists = sqlite3_column_int (stmt, 0);
	    }
      }
    sqlite3_finalize (stmt);
    if (srid_exists == 0)
	ret = insert_epsg_srid (sqlite, srid);
/* end Mark Johnson 2019-01-26 */

/* trying to add the column */
    switch (xtype)
      {
      case GAIA_POINT:
	  p_type = "POINT";
	  break;
      case GAIA_LINESTRING:
	  p_type = "LINESTRING";
	  break;
      case GAIA_POLYGON:
	  p_type = "POLYGON";
	  break;
      case GAIA_MULTIPOINT:
	  p_type = "MULTIPOINT";
	  break;
      case GAIA_MULTILINESTRING:
	  p_type = "MULTILINESTRING";
	  break;
      case GAIA_MULTIPOLYGON:
	  p_type = "MULTIPOLYGON";
	  break;
      case GAIA_GEOMETRYCOLLECTION:
	  p_type = "GEOMETRYCOLLECTION";
	  break;
      case -1:
	  p_type = "GEOMETRY";
	  break;
      };
    quoted_table = gaiaDoubleQuotedSql (p_table);
    quoted_column = gaiaDoubleQuotedSql (column);
    if (notNull)
      {
	  /* adding a NOT NULL clause */
	  sql_statement =
	      sqlite3_mprintf ("ALTER TABLE \"%s\" ADD COLUMN \"%s\" "
			       "%s NOT NULL DEFAULT ''", quoted_table,
			       quoted_column, p_type);
      }
    else
	sql_statement =
	    sqlite3_mprintf ("ALTER TABLE \"%s\" ADD COLUMN \"%s\" %s ",
			     quoted_table, quoted_column, p_type);
    free (quoted_table);
    free (quoted_column);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("AddGeometryColumn: \"%s\"\n", sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  sqlite3_free (p_table);
	  return;
      }
/* ok, inserting into geometry_columns [Spatial Metadata] */
    if (metadata_version == 1)
      {
	  /* legacy metadata style <= v.3.1.0 */
	  switch (xtype)
	    {
	    case GAIA_POINT:
		p_type = "POINT";
		break;
	    case GAIA_LINESTRING:
		p_type = "LINESTRING";
		break;
	    case GAIA_POLYGON:
		p_type = "POLYGON";
		break;
	    case GAIA_MULTIPOINT:
		p_type = "MULTIPOINT";
		break;
	    case GAIA_MULTILINESTRING:
		p_type = "MULTILINESTRING";
		break;
	    case GAIA_MULTIPOLYGON:
		p_type = "MULTIPOLYGON";
		break;
	    case GAIA_GEOMETRYCOLLECTION:
		p_type = "GEOMETRYCOLLECTION";
		break;
	    case -1:
		p_type = "GEOMETRY";
		break;
	    };
	  switch (dims)
	    {
	    case GAIA_XY:
		p_dims = "XY";
		break;
	    case GAIA_XY_Z:
		p_dims = "XYZ";
		break;
	    case GAIA_XY_M:
		p_dims = "XYM";
		break;
	    case GAIA_XY_Z_M:
		p_dims = "XYZM";
		break;
	    };
	  sql_statement = sqlite3_mprintf ("INSERT INTO geometry_columns "
					   "(f_table_name, f_geometry_column, type, coord_dimension, srid, "
					   "spatial_index_enabled) VALUES (?, ?, %Q, %Q, ?, 0)",
					   p_type, p_dims);
      }
    else
      {
	  /* current metadata style >= v.4.0.0 */
	  switch (xtype)
	    {
	    case GAIA_POINT:
		if (dims == GAIA_XY_Z)
		    n_type = 1001;
		else if (dims == GAIA_XY_M)
		    n_type = 2001;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3001;
		else
		    n_type = 1;
		break;
	    case GAIA_LINESTRING:
		if (dims == GAIA_XY_Z)
		    n_type = 1002;
		else if (dims == GAIA_XY_M)
		    n_type = 2002;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3002;
		else
		    n_type = 2;
		break;
	    case GAIA_POLYGON:
		if (dims == GAIA_XY_Z)
		    n_type = 1003;
		else if (dims == GAIA_XY_M)
		    n_type = 2003;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3003;
		else
		    n_type = 3;
		break;
	    case GAIA_MULTIPOINT:
		if (dims == GAIA_XY_Z)
		    n_type = 1004;
		else if (dims == GAIA_XY_M)
		    n_type = 2004;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3004;
		else
		    n_type = 4;
		break;
	    case GAIA_MULTILINESTRING:
		if (dims == GAIA_XY_Z)
		    n_type = 1005;
		else if (dims == GAIA_XY_M)
		    n_type = 2005;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3005;
		else
		    n_type = 5;
		break;
	    case GAIA_MULTIPOLYGON:
		if (dims == GAIA_XY_Z)
		    n_type = 1006;
		else if (dims == GAIA_XY_M)
		    n_type = 2006;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3006;
		else
		    n_type = 6;
		break;
	    case GAIA_GEOMETRYCOLLECTION:
		if (dims == GAIA_XY_Z)
		    n_type = 1007;
		else if (dims == GAIA_XY_M)
		    n_type = 2007;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3007;
		else
		    n_type = 7;
		break;
	    case -1:
		if (dims == GAIA_XY_Z)
		    n_type = 1000;
		else if (dims == GAIA_XY_M)
		    n_type = 2000;
		else if (dims == GAIA_XY_Z_M)
		    n_type = 3000;
		else
		    n_type = 0;
		break;
	    };
	  switch (dims)
	    {
	    case GAIA_XY:
		n_dims = 2;
		break;
	    case GAIA_XY_Z:
	    case GAIA_XY_M:
		n_dims = 3;
		break;
	    case GAIA_XY_Z_M:
		n_dims = 4;
		break;
	    };
	  sql_statement = sqlite3_mprintf ("INSERT INTO geometry_columns "
					   "(f_table_name, f_geometry_column, geometry_type, coord_dimension, "
					   "srid, spatial_index_enabled) VALUES (Lower(?), Lower(?), %d, %d, ?, 0)",
					   n_type, n_dims);
      }
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("AddGeometryColumn: \"%s\"\n", sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  sqlite3_free (p_table);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, p_table, strlen (p_table), SQLITE_STATIC);
    sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
    if (srid < 0)
	sqlite3_bind_int (stmt, 3, -1);
    else
	sqlite3_bind_int (stmt, 3, srid);
    ret = sqlite3_step (stmt);
    if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	;
    else
      {
	  spatialite_e ("AddGeometryColumn() error: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_finalize (stmt);
	  goto error;
      }
    sqlite3_finalize (stmt);
    if (metadata_version == 3)
      {
	  /* current metadata style >= v.4.0.0 */

	  /* inserting a row into GEOMETRY_COLUMNS_AUTH */
	  strcpy (sql,
		  "INSERT OR REPLACE INTO geometry_columns_auth (f_table_name, f_geometry_column, ");
	  strcat (sql, "read_only, hidden) VALUES (Lower(?), Lower(?), 0, 0)");
	  ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("AddGeometryColumn: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_result_int (context, 0);
		sqlite3_free (p_table);
		return;
	    }
	  sqlite3_reset (stmt);
	  sqlite3_clear_bindings (stmt);
	  sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
	  sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	      ;
	  else
	    {
		spatialite_e ("AddGeometryColumn() error: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto error;
	    }
	  sqlite3_finalize (stmt);
	  /* inserting a row into GEOMETRY_COLUMNS_STATISTICS */
	  strcpy (sql,
		  "INSERT OR REPLACE INTO geometry_columns_statistics (f_table_name, f_geometry_column) ");
	  strcat (sql, "VALUES (Lower(?), Lower(?))");
	  ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("AddGeometryColumn: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_result_int (context, 0);
		sqlite3_free (p_table);
		return;
	    }
	  sqlite3_reset (stmt);
	  sqlite3_clear_bindings (stmt);
	  sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
	  sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	      ;
	  else
	    {
		spatialite_e ("AddGeometryColumn() error: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto error;
	    }
	  sqlite3_finalize (stmt);
	  /* inserting a row into GEOMETRY_COLUMNS_TIME */
	  strcpy (sql,
		  "INSERT OR REPLACE INTO geometry_columns_time (f_table_name, f_geometry_column) ");
	  strcat (sql, "VALUES (Lower(?), Lower(?))");
	  ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("AddGeometryColumn: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_result_int (context, 0);
		sqlite3_free (p_table);
		return;
	    }
	  sqlite3_reset (stmt);
	  sqlite3_clear_bindings (stmt);
	  sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
	  sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	      ;
	  else
	    {
		spatialite_e ("AddGeometryColumn() error: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto error;
	    }
	  sqlite3_finalize (stmt);
      }
    updateGeometryTriggers (sqlite, table, column);
    sqlite3_result_int (context, 1);
    switch (xtype)
      {
      case GAIA_POINT:
	  p_type = "POINT";
	  break;
      case GAIA_LINESTRING:
	  p_type = "LINESTRING";
	  break;
      case GAIA_POLYGON:
	  p_type = "POLYGON";
	  break;
      case GAIA_MULTIPOINT:
	  p_type = "MULTIPOINT";
	  break;
      case GAIA_MULTILINESTRING:
	  p_type = "MULTILINESTRING";
	  break;
      case GAIA_MULTIPOLYGON:
	  p_type = "MULTIPOLYGON";
	  break;
      case GAIA_GEOMETRYCOLLECTION:
	  p_type = "GEOMETRYCOLLECTION";
	  break;
      case -1:
	  p_type = "GEOMETRY";
	  break;
      };
    switch (dims)
      {
      case GAIA_XY:
	  p_dims = "XY";
	  break;
      case GAIA_XY_Z:
	  p_dims = "XYZ";
	  break;
      case GAIA_XY_M:
	  p_dims = "XYM";
	  break;
      case GAIA_XY_Z_M:
	  p_dims = "XYZM";
	  break;
      };
    sql_statement =
	sqlite3_mprintf ("Geometry [%s,%s,SRID=%d] successfully created",
			 p_type, p_dims, (srid <= 0) ? -1 : srid);
    updateSpatiaLiteHistory (sqlite, table, column, sql_statement);
    sqlite3_free (sql_statement);
    sqlite3_free (p_table);
    return;
  error:
    sqlite3_result_int (context, 0);
    sqlite3_free (p_table);
    return;
}

static void
fnct_AddTemporaryGeometryColumn (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ AddTemporaryGeometryColumn(db-prefix, table, column, srid, type [ , dimension  [  , not-null ] ] )
/
/ creates a new COLUMN of given TYPE into TABLE
/ returns 1 on success
/ 0 on failure
/
/ INTENDED ONLY FOR TEMPORARY ATTACHED DATABASES
/
*/
    const char *db_prefix;
    const char *table;
    const char *column;
    const unsigned char *type;
    const unsigned char *txt_dims;
    int xtype;
    int srid = -1;
    int dimension = 2;
    int dims = -1;
    int auto_dims = -1;
    char *sql2;
    int ret;
    int notNull = 0;
    sqlite3_stmt *stmt;
    char *p_table = NULL;
    char *quoted_prefix;
    char *quoted_table;
    char *quoted_column;
    const char *p_type = NULL;
    int n_type = 0;
    int n_dims = 0;
    char *sql_statement;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 1 [DB-prefix] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    db_prefix = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 2 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 3 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 4 [SRID] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    srid = sqlite3_value_int (argv[3]);
    if (sqlite3_value_type (argv[4]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 5 [geometry_type] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    type = sqlite3_value_text (argv[4]);
    if (argc > 5)
      {
	  if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
	    {
		dimension = sqlite3_value_int (argv[5]);
		if (dimension == 2)
		    dims = GAIA_XY;
		if (dimension == 3)
		    dims = GAIA_XY_Z;
		if (dimension == 4)
		    dims = GAIA_XY_Z_M;
	    }
	  else if (sqlite3_value_type (argv[5]) == SQLITE_TEXT)
	    {
		txt_dims = sqlite3_value_text (argv[5]);
		if (strcasecmp ((char *) txt_dims, "XY") == 0)
		    dims = GAIA_XY;
		if (strcasecmp ((char *) txt_dims, "XYZ") == 0)
		    dims = GAIA_XY_Z;
		if (strcasecmp ((char *) txt_dims, "XYM") == 0)
		    dims = GAIA_XY_M;
		if (strcasecmp ((char *) txt_dims, "XYZM") == 0)
		    dims = GAIA_XY_Z_M;
	    }
	  else
	    {
		spatialite_e
		    ("AddTemporaryGeometryColumn() error: argument 6 [dimension] is not of the Integer or Text type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
      }
    if (argc == 7)
      {
	  /* optional NOT NULL arg */
	  if (sqlite3_value_type (argv[6]) != SQLITE_INTEGER)
	    {
		spatialite_e
		    ("AddTemporaryGeometryColumn() error: argument 7 [not null] is not of the Integer type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  notNull = sqlite3_value_int (argv[5]);
      }
/* checking if the Attached Database is actually based on :memory: */
    if (!is_attached_memory (sqlite, db_prefix))
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: Database '%s' does not exists or is not a Temporary one\n",
	       db_prefix);
	  sqlite3_result_int (context, 0);
	  return;
      }
    xtype = GAIA_UNKNOWN;
    if (strcasecmp ((char *) type, "POINT") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRING") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGON") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINT") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRING") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGON") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTION") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRY") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = -1;
      }
    if (xtype == GAIA_UNKNOWN)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 5 [geometry_type] has an illegal value\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (dims < 0)
	dims = auto_dims;
    if (dims == GAIA_XY || dims == GAIA_XY_Z || dims == GAIA_XY_M
	|| dims == GAIA_XY_Z_M)
	;
    else
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 6 [dimension] ILLEGAL VALUE\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (auto_dims != GAIA_XY && dims != auto_dims)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: argument 6 [dimension] ILLEGAL VALUE\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
/* checking if the table exists */
    quoted_prefix = gaiaDoubleQuotedSql (db_prefix);
    sql2 =
	sqlite3_mprintf
	("SELECT name FROM \"%s\".sqlite_master WHERE type = 'table' AND Lower(name) = Lower(?)",
	 quoted_prefix);
    free (quoted_prefix);
    ret = sqlite3_prepare_v2 (sqlite, sql2, strlen (sql2), &stmt, NULL);
    sqlite3_free (sql2);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("AddTemporaryGeometryColumn: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	    {
		if (p_table != NULL)
		    sqlite3_free (p_table);
		p_table =
		    sqlite3_mprintf ("%s",
				     (const char *) sqlite3_column_text (stmt,
									 0));
	    }
      }
    sqlite3_finalize (stmt);
    if (!p_table)
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: table '%s' does not exist\n",
	       table);
	  sqlite3_result_int (context, 0);
	  return;
      }
/* checking for WITHOUT ROWID */
    if (is_without_rowid_table_attached (sqlite, db_prefix, table))
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: table '%s' is WITHOUT ROWID\n",
	       table);
	  sqlite3_result_int (context, 0);
	  return;
      }

/* trying to add the column */
    switch (xtype)
      {
      case GAIA_POINT:
	  p_type = "POINT";
	  break;
      case GAIA_LINESTRING:
	  p_type = "LINESTRING";
	  break;
      case GAIA_POLYGON:
	  p_type = "POLYGON";
	  break;
      case GAIA_MULTIPOINT:
	  p_type = "MULTIPOINT";
	  break;
      case GAIA_MULTILINESTRING:
	  p_type = "MULTILINESTRING";
	  break;
      case GAIA_MULTIPOLYGON:
	  p_type = "MULTIPOLYGON";
	  break;
      case GAIA_GEOMETRYCOLLECTION:
	  p_type = "GEOMETRYCOLLECTION";
	  break;
      case -1:
	  p_type = "GEOMETRY";
	  break;
      };
    quoted_prefix = gaiaDoubleQuotedSql (db_prefix);
    quoted_table = gaiaDoubleQuotedSql (p_table);
    quoted_column = gaiaDoubleQuotedSql (column);
    if (notNull)
      {
	  /* adding a NOT NULL clause */
	  sql_statement =
	      sqlite3_mprintf ("ALTER TABLE \"%s\".\"%s\" ADD COLUMN \"%s\" "
			       "%s NOT NULL DEFAULT ''", quoted_prefix,
			       quoted_table, quoted_column, p_type);
      }
    else
	sql_statement =
	    sqlite3_mprintf ("ALTER TABLE \"%s\".\"%s\" ADD COLUMN \"%s\" %s ",
			     quoted_prefix, quoted_table, quoted_column,
			     p_type);
    free (quoted_prefix);
    free (quoted_table);
    free (quoted_column);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("AddTemporaryGeometryColumn: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  sqlite3_free (p_table);
	  return;
      }
/* ok, inserting into geometry_columns [Spatial Metadata] */
    switch (xtype)
      {
      case GAIA_POINT:
	  if (dims == GAIA_XY_Z)
	      n_type = 1001;
	  else if (dims == GAIA_XY_M)
	      n_type = 2001;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3001;
	  else
	      n_type = 1;
	  break;
      case GAIA_LINESTRING:
	  if (dims == GAIA_XY_Z)
	      n_type = 1002;
	  else if (dims == GAIA_XY_M)
	      n_type = 2002;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3002;
	  else
	      n_type = 2;
	  break;
      case GAIA_POLYGON:
	  if (dims == GAIA_XY_Z)
	      n_type = 1003;
	  else if (dims == GAIA_XY_M)
	      n_type = 2003;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3003;
	  else
	      n_type = 3;
	  break;
      case GAIA_MULTIPOINT:
	  if (dims == GAIA_XY_Z)
	      n_type = 1004;
	  else if (dims == GAIA_XY_M)
	      n_type = 2004;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3004;
	  else
	      n_type = 4;
	  break;
      case GAIA_MULTILINESTRING:
	  if (dims == GAIA_XY_Z)
	      n_type = 1005;
	  else if (dims == GAIA_XY_M)
	      n_type = 2005;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3005;
	  else
	      n_type = 5;
	  break;
      case GAIA_MULTIPOLYGON:
	  if (dims == GAIA_XY_Z)
	      n_type = 1006;
	  else if (dims == GAIA_XY_M)
	      n_type = 2006;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3006;
	  else
	      n_type = 6;
	  break;
      case GAIA_GEOMETRYCOLLECTION:
	  if (dims == GAIA_XY_Z)
	      n_type = 1007;
	  else if (dims == GAIA_XY_M)
	      n_type = 2007;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3007;
	  else
	      n_type = 7;
	  break;
      case -1:
	  if (dims == GAIA_XY_Z)
	      n_type = 1000;
	  else if (dims == GAIA_XY_M)
	      n_type = 2000;
	  else if (dims == GAIA_XY_Z_M)
	      n_type = 3000;
	  else
	      n_type = 0;
	  break;
      };
    switch (dims)
      {
      case GAIA_XY:
	  n_dims = 2;
	  break;
      case GAIA_XY_Z:
      case GAIA_XY_M:
	  n_dims = 3;
	  break;
      case GAIA_XY_Z_M:
	  n_dims = 4;
	  break;
      };

/* attempting to create spatial_ref_sys, just in case */
    if (!createTemporarySpatialRefSys (sqlite, db_prefix))
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: unable to create \"spatial_ref_sys\" on Database '%s'\n",
	       db_prefix);
	  sqlite3_result_int (context, 0);
	  return;
      }
/* attempting to create geometry_columns, just in case */
    if (!createTemporaryGeometryColumns (sqlite, db_prefix))
      {
	  spatialite_e
	      ("AddTemporaryGeometryColumn() error: unable to create \"geometry_columns\" on Database '%s'\n",
	       db_prefix);
	  sqlite3_result_int (context, 0);
	  return;
      }

    quoted_prefix = gaiaDoubleQuotedSql (db_prefix);
    sql_statement = sqlite3_mprintf ("INSERT INTO \"%s\".geometry_columns "
				     "(f_table_name, f_geometry_column, geometry_type, coord_dimension, "
				     "srid, spatial_index_enabled) VALUES (Lower(?), Lower(?), %d, %d, ?, 0)",
				     quoted_prefix, n_type, n_dims);
    free (quoted_prefix);
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("AddTemporaryGeometryColumn: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  sqlite3_free (p_table);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, p_table, strlen (p_table), SQLITE_STATIC);
    sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
    if (srid < 0)
	sqlite3_bind_int (stmt, 3, -1);
    else
	sqlite3_bind_int (stmt, 3, srid);
    ret = sqlite3_step (stmt);
    if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	;
    else
      {
	  spatialite_e ("AddTemporaryGeometryColumn() error: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_finalize (stmt);
	  goto error;
      }
    sqlite3_finalize (stmt);
    updateTemporaryGeometryTriggers (sqlite, db_prefix, table, column);
    sqlite3_result_int (context, 1);
    sqlite3_free (p_table);
    return;
  error:
    sqlite3_result_int (context, 0);
    sqlite3_free (p_table);
    return;
}

static void
fnct_RecoverGeometryColumn (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ RecoverGeometryColumn(table, column, srid, type , dimension )
/
/ checks if an existing TABLE.COLUMN satisfies the required geometric features
/ if yes adds it to SpatialMetaData and enabling triggers
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    const unsigned char *type;
    int xtype;
    int xxtype;
    int srid = -1;
    const unsigned char *txt_dims;
    int dimension = 2;
    int dims = -1;
    int auto_dims = -1;
    char sql[1024];
    int ret;
    int metadata_version;
    sqlite3_stmt *stmt;
    int exists = 0;
    const char *p_type = NULL;
    const char *p_dims = NULL;
    int n_type = GAIA_UNKNOWN;
    int n_dims = -1;
    char *sql_statement;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: argument 3 [SRID] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    srid = sqlite3_value_int (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: argument 4 [geometry_type] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    type = sqlite3_value_text (argv[3]);
    if (argc == 5)
      {
	  if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	    {
		dimension = sqlite3_value_int (argv[4]);
		if (dimension == 2)
		    dims = GAIA_XY;
		if (dimension == 3)
		    dims = GAIA_XY_Z;
		if (dimension == 4)
		    dims = GAIA_XY_Z_M;
	    }
	  else if (sqlite3_value_type (argv[4]) == SQLITE_TEXT)
	    {
		txt_dims = sqlite3_value_text (argv[4]);
		if (strcasecmp ((char *) txt_dims, "XY") == 0)
		    dims = GAIA_XY;
		if (strcasecmp ((char *) txt_dims, "XYZ") == 0)
		    dims = GAIA_XY_Z;
		if (strcasecmp ((char *) txt_dims, "XYM") == 0)
		    dims = GAIA_XY_M;
		if (strcasecmp ((char *) txt_dims, "XYZM") == 0)
		    dims = GAIA_XY_Z_M;
	    }
	  else
	    {
		spatialite_e
		    ("RecoverGeometryColumn() error: argument 5 [dimension] is not of the Integer or Text type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
      }
    xtype = GAIA_UNKNOWN;
    if (strcasecmp ((char *) type, "POINT") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRING") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGON") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINT") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRING") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGON") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTION") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRY") == 0)
      {
	  auto_dims = GAIA_XY;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYZ") == 0)
      {
	  auto_dims = GAIA_XY_Z;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYM") == 0)
      {
	  auto_dims = GAIA_XY_M;
	  xtype = -1;
      }
    if (strcasecmp ((char *) type, "POINTZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_POINT;
      }
    if (strcasecmp ((char *) type, "LINESTRINGZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_LINESTRING;
      }
    if (strcasecmp ((char *) type, "POLYGONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_POLYGON;
      }
    if (strcasecmp ((char *) type, "MULTIPOINTZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTIPOINT;
      }
    if (strcasecmp ((char *) type, "MULTILINESTRINGZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTILINESTRING;
      }
    if (strcasecmp ((char *) type, "MULTIPOLYGONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_MULTIPOLYGON;
      }
    if (strcasecmp ((char *) type, "GEOMETRYCOLLECTIONZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = GAIA_GEOMETRYCOLLECTION;
      }
    if (strcasecmp ((char *) type, "GEOMETRYZM") == 0)
      {
	  auto_dims = GAIA_XY_Z_M;
	  xtype = -1;
      }
    if (dims < 0)
	dims = auto_dims;
    if (xtype == GAIA_UNKNOWN)
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: argument 4 [geometry_type] has an illegal value\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (dims == GAIA_XY || dims == GAIA_XY_Z || dims == GAIA_XY_M
	|| dims == GAIA_XY_Z_M)
	;
    else
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: argument 5 [dimension] ILLEGAL VALUE\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (auto_dims != GAIA_XY && dims != auto_dims)
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: argument 5 [dimension] ILLEGAL VALUE\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    metadata_version = checkSpatialMetaData (sqlite);
    if (metadata_version == 1 || metadata_version == 3)
	;
    else
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: unexpected metadata layout\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
/* checking if the table exists */
    strcpy (sql,
	    "SELECT name FROM sqlite_master WHERE type = 'table' AND Lower(name) = Lower(?)");
    ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RecoverGeometryColumn: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	    {
		if (sqlite3_column_type (stmt, 0) == SQLITE_TEXT)
		    exists = 1;
	    }
      }
    sqlite3_finalize (stmt);
    if (!exists)
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: table '%s' does not exist\n",
	       table);
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (is_without_rowid_table (sqlite, table))
      {
	  spatialite_e
	      ("RecoverGeometryColumn() error: table '%s' is WITHOUT ROWID\n",
	       table);
	  sqlite3_result_int (context, 0);
	  return;
      }
/* adjusting the actual GeometryType */
    xxtype = xtype;
    xtype = GAIA_UNKNOWN;
    if (xxtype == GAIA_POINT)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_POINTZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_POINTM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_POINTZM;
		break;
	    default:
		xtype = GAIA_POINT;
		break;
	    };
      }
    if (xxtype == GAIA_LINESTRING)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_LINESTRINGZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_LINESTRINGM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_LINESTRINGZM;
		break;
	    default:
		xtype = GAIA_LINESTRING;
		break;
	    };
      }
    if (xxtype == GAIA_POLYGON)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_POLYGONZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_POLYGONM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_POLYGONZM;
		break;
	    default:
		xtype = GAIA_POLYGON;
		break;
	    };
      }
    if (xxtype == GAIA_MULTIPOINT)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_MULTIPOINTZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_MULTIPOINTM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_MULTIPOINTZM;
		break;
	    default:
		xtype = GAIA_MULTIPOINT;
		break;
	    };
      }
    if (xxtype == GAIA_MULTILINESTRING)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_MULTILINESTRINGZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_MULTILINESTRINGM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_MULTILINESTRINGZM;
		break;
	    default:
		xtype = GAIA_MULTILINESTRING;
		break;
	    };
      }
    if (xxtype == GAIA_MULTIPOLYGON)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_MULTIPOLYGONZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_MULTIPOLYGONM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_MULTIPOLYGONZM;
		break;
	    default:
		xtype = GAIA_MULTIPOLYGON;
		break;
	    };
      }
    if (xxtype == GAIA_GEOMETRYCOLLECTION)
      {
	  switch (dims)
	    {
	    case GAIA_XY_Z:
		xtype = GAIA_GEOMETRYCOLLECTIONZ;
		break;
	    case GAIA_XY_M:
		xtype = GAIA_GEOMETRYCOLLECTIONM;
		break;
	    case GAIA_XY_Z_M:
		xtype = GAIA_GEOMETRYCOLLECTIONZM;
		break;
	    default:
		xtype = GAIA_GEOMETRYCOLLECTION;
		break;
	    };
      }
    if (xxtype == -1)
	xtype = -1;		/* GEOMETRY */
    if (!recoverGeomColumn (sqlite, table, column, xtype, dims, srid))
      {
	  spatialite_e ("RecoverGeometryColumn(): validation failed\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
/* deleting anyway any previous definition */
    sql_statement = sqlite3_mprintf ("DELETE FROM geometry_columns "
				     "WHERE Lower(f_table_name) = Lower(?) AND "
				     "Lower(f_geometry_column) = Lower(?)");
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RecoverGeometryColumn: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
    sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
    ret = sqlite3_step (stmt);
    if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	;
    else
      {
	  spatialite_e ("RecoverGeometryColumn() error: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_finalize (stmt);
	  goto error;
      }
    sqlite3_finalize (stmt);

    if (metadata_version == 1)
      {
	  /* legacy metadata style <= v.3.1.0 */
	  switch (xtype)
	    {
	    case GAIA_POINT:
	    case GAIA_POINTZ:
	    case GAIA_POINTM:
	    case GAIA_POINTZM:
		p_type = "POINT";
		break;
	    case GAIA_LINESTRING:
	    case GAIA_LINESTRINGZ:
	    case GAIA_LINESTRINGM:
	    case GAIA_LINESTRINGZM:
		p_type = "LINESTRING";
		break;
	    case GAIA_POLYGON:
	    case GAIA_POLYGONZ:
	    case GAIA_POLYGONM:
	    case GAIA_POLYGONZM:
		p_type = "POLYGON";
		break;
	    case GAIA_MULTIPOINT:
	    case GAIA_MULTIPOINTZ:
	    case GAIA_MULTIPOINTM:
	    case GAIA_MULTIPOINTZM:
		p_type = "MULTIPOINT";
		break;
	    case GAIA_MULTILINESTRING:
	    case GAIA_MULTILINESTRINGZ:
	    case GAIA_MULTILINESTRINGM:
	    case GAIA_MULTILINESTRINGZM:
		p_type = "MULTILINESTRING";
		break;
	    case GAIA_MULTIPOLYGON:
	    case GAIA_MULTIPOLYGONZ:
	    case GAIA_MULTIPOLYGONM:
	    case GAIA_MULTIPOLYGONZM:
		p_type = "MULTIPOLYGON";
		break;
	    case GAIA_GEOMETRYCOLLECTION:
	    case GAIA_GEOMETRYCOLLECTIONZ:
	    case GAIA_GEOMETRYCOLLECTIONM:
	    case GAIA_GEOMETRYCOLLECTIONZM:
		p_type = "GEOMETRYCOLLECTION";
		break;
	    case -1:
		p_type = "GEOMETRY";
		break;
	    };
	  strcat (sql, "', '");
	  switch (dims)
	    {
	    case GAIA_XY:
		p_dims = "XY";
		break;
	    case GAIA_XY_Z:
		p_dims = "XYZ";
		break;
	    case GAIA_XY_M:
		p_dims = "XYM";
		break;
	    case GAIA_XY_Z_M:
		p_dims = "XYZM";
		break;
	    };
/* Sandro 2013-01-07
/ fixing an issue reported by Peter Aronson [ESRI] <paronson@esri.com>
	  sql_statement = sqlite3_mprintf ("INSERT INTO geometry_columns "
					   "(f_table_name, f_geometry_column, type, coord_dimension, srid, "
					   "spatial_index_enabled) VALUES (Lower(?), Lower(?), %Q, %Q, ?, 0)",
					   p_type, p_dims);
*/
	  sql_statement = sqlite3_mprintf ("INSERT INTO geometry_columns "
					   "(f_table_name, f_geometry_column, type, coord_dimension, srid, "
					   "spatial_index_enabled) VALUES (?, ?, %Q, %Q, ?, 0)",
					   p_type, p_dims);
      }
    else
      {
	  /* current metadata style >= v.4.0.0 */
	  switch (xtype)
	    {
	    case GAIA_POINT:
		n_type = 1;
		n_dims = 2;
		break;
	    case GAIA_POINTZ:
		n_type = 1001;
		n_dims = 3;
		break;
	    case GAIA_POINTM:
		n_type = 2001;
		n_dims = 3;
		break;
	    case GAIA_POINTZM:
		n_type = 3001;
		n_dims = 4;
		break;
	    case GAIA_LINESTRING:
		n_type = 2;
		n_dims = 2;
		break;
	    case GAIA_LINESTRINGZ:
		n_type = 1002;
		n_dims = 3;
		break;
	    case GAIA_LINESTRINGM:
		n_type = 2002;
		n_dims = 3;
		break;
	    case GAIA_LINESTRINGZM:
		n_type = 3002;
		n_dims = 4;
		break;
	    case GAIA_POLYGON:
		n_type = 3;
		n_dims = 2;
		break;
	    case GAIA_POLYGONZ:
		n_type = 1003;
		n_dims = 3;
		break;
	    case GAIA_POLYGONM:
		n_type = 2003;
		n_dims = 3;
		break;
	    case GAIA_POLYGONZM:
		n_type = 3003;
		n_dims = 4;
		break;
	    case GAIA_MULTIPOINT:
		n_type = 4;
		n_dims = 2;
		break;
	    case GAIA_MULTIPOINTZ:
		n_type = 1004;
		n_dims = 3;
		break;
	    case GAIA_MULTIPOINTM:
		n_type = 2004;
		n_dims = 3;
		break;
	    case GAIA_MULTIPOINTZM:
		n_type = 3004;
		n_dims = 4;
		break;
	    case GAIA_MULTILINESTRING:
		n_type = 5;
		n_dims = 2;
		break;
	    case GAIA_MULTILINESTRINGZ:
		n_type = 1005;
		n_dims = 3;
		break;
	    case GAIA_MULTILINESTRINGM:
		n_type = 2005;
		n_dims = 3;
		break;
	    case GAIA_MULTILINESTRINGZM:
		n_type = 3005;
		n_dims = 4;
		break;
	    case GAIA_MULTIPOLYGON:
		n_type = 6;
		n_dims = 2;
		break;
	    case GAIA_MULTIPOLYGONZ:
		n_type = 1006;
		n_dims = 3;
		break;
	    case GAIA_MULTIPOLYGONM:
		n_type = 2006;
		n_dims = 3;
		break;
	    case GAIA_MULTIPOLYGONZM:
		n_type = 3006;
		n_dims = 4;
		break;
	    case GAIA_GEOMETRYCOLLECTION:
		n_type = 7;
		n_dims = 2;
		break;
	    case GAIA_GEOMETRYCOLLECTIONZ:
		n_type = 1007;
		n_dims = 3;
		break;
	    case GAIA_GEOMETRYCOLLECTIONM:
		n_type = 2007;
		n_dims = 3;
		break;
	    case GAIA_GEOMETRYCOLLECTIONZM:
		n_type = 3007;
		n_dims = 4;
		break;
	    case -1:
		switch (dims)
		  {
		  case GAIA_XY:
		      n_type = 0;
		      n_dims = 2;
		      break;
		  case GAIA_XY_Z:
		      n_type = 1000;
		      n_dims = 3;
		      break;
		  case GAIA_XY_M:
		      n_type = 2000;
		      n_dims = 3;
		      break;
		  case GAIA_XY_Z_M:
		      n_type = 3000;
		      n_dims = 4;
		      break;
		  };
		break;
	    };
	  sql_statement = sqlite3_mprintf ("INSERT INTO geometry_columns "
					   "(f_table_name, f_geometry_column, geometry_type, coord_dimension, "
					   "srid, spatial_index_enabled) VALUES (Lower(?), Lower(?), %d, %d, ?, 0)",
					   n_type, n_dims);
      }
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RecoverGeometryColumn: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
    sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
    if (srid < 0)
	sqlite3_bind_int (stmt, 3, -1);
    else
	sqlite3_bind_int (stmt, 3, srid);
    ret = sqlite3_step (stmt);
    if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	;
    else
      {
	  spatialite_e ("RecoverGeometryColumn() error: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_finalize (stmt);
	  goto error;
      }
    sqlite3_finalize (stmt);
    if (metadata_version == 3)
      {
	  /* current metadata style >= v.4.0.0 */

	  /* inserting a row into GEOMETRY_COLUMNS_AUTH */
	  strcpy (sql,
		  "INSERT OR REPLACE INTO geometry_columns_auth (f_table_name, f_geometry_column, ");
	  strcat (sql, "read_only, hidden) VALUES (Lower(?), Lower(?), 0, 0)");
	  ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("RecoverGeometryColumn: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_result_int (context, 0);
		return;
	    }
	  sqlite3_reset (stmt);
	  sqlite3_clear_bindings (stmt);
	  sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
	  sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	      ;
	  else
	    {
		spatialite_e ("RecoverGeometryColumn() error: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto error;
	    }
	  sqlite3_finalize (stmt);
	  /* inserting a row into GEOMETRY_COLUMNS_STATISTICS */
	  strcpy (sql,
		  "INSERT OR REPLACE INTO geometry_columns_statistics (f_table_name, f_geometry_column) ");
	  strcat (sql, "VALUES (Lower(?), Lower(?))");
	  ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("RecoverGeometryColumn: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_result_int (context, 0);
		return;
	    }
	  sqlite3_reset (stmt);
	  sqlite3_clear_bindings (stmt);
	  sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
	  sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	      ;
	  else
	    {
		spatialite_e ("RecoverGeometryColumn() error: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto error;
	    }
	  sqlite3_finalize (stmt);
	  /* inserting a row into GEOMETRY_COLUMNS_TIME */
	  strcpy (sql,
		  "INSERT OR REPLACE INTO geometry_columns_time (f_table_name, f_geometry_column) ");
	  strcat (sql, "VALUES (Lower(?), Lower(?))");
	  ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("RecoverGeometryColumn: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_result_int (context, 0);
		return;
	    }
	  sqlite3_reset (stmt);
	  sqlite3_clear_bindings (stmt);
	  sqlite3_bind_text (stmt, 1, table, strlen (table), SQLITE_STATIC);
	  sqlite3_bind_text (stmt, 2, column, strlen (column), SQLITE_STATIC);
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	      ;
	  else
	    {
		spatialite_e ("RecoverGeometryColumn() error: \"%s\"\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto error;
	    }
	  sqlite3_finalize (stmt);
      }
    updateGeometryTriggers (sqlite, table, column);
    sqlite3_result_int (context, 1);
    switch (xtype)
      {
      case GAIA_POINT:
      case GAIA_POINTZ:
      case GAIA_POINTM:
      case GAIA_POINTZM:
	  p_type = "POINT";
	  break;
      case GAIA_LINESTRING:
      case GAIA_LINESTRINGZ:
      case GAIA_LINESTRINGM:
      case GAIA_LINESTRINGZM:
	  p_type = "LINESTRING";
	  break;
      case GAIA_POLYGON:
      case GAIA_POLYGONZ:
      case GAIA_POLYGONM:
      case GAIA_POLYGONZM:
	  p_type = "POLYGON";
	  break;
      case GAIA_MULTIPOINT:
      case GAIA_MULTIPOINTZ:
      case GAIA_MULTIPOINTM:
      case GAIA_MULTIPOINTZM:
	  p_type = "MULTIPOINT";
	  break;
      case GAIA_MULTILINESTRING:
      case GAIA_MULTILINESTRINGZ:
      case GAIA_MULTILINESTRINGM:
      case GAIA_MULTILINESTRINGZM:
	  p_type = "MULTILINESTRING";
	  break;
      case GAIA_MULTIPOLYGON:
      case GAIA_MULTIPOLYGONZ:
      case GAIA_MULTIPOLYGONM:
      case GAIA_MULTIPOLYGONZM:
	  p_type = "MULTIPOLYGON";
	  break;
      case GAIA_GEOMETRYCOLLECTION:
      case GAIA_GEOMETRYCOLLECTIONZ:
      case GAIA_GEOMETRYCOLLECTIONM:
      case GAIA_GEOMETRYCOLLECTIONZM:
	  p_type = "GEOMETRYCOLLECTION";
	  break;
      case -1:
	  p_type = "GEOMETRY";
	  break;
      };
    switch (dims)
      {
      case GAIA_XY:
	  p_dims = "XY";
	  break;
      case GAIA_XY_Z:
	  p_dims = "XYZ";
	  break;
      case GAIA_XY_M:
	  p_dims = "XYM";
	  break;
      case GAIA_XY_Z_M:
	  p_dims = "XYZM";
	  break;
      };
    sql_statement =
	sqlite3_mprintf ("Geometry [%s,%s,SRID=%d] successfully recovered",
			 p_type, p_dims, (srid <= 0) ? -1 : srid);
    updateSpatiaLiteHistory (sqlite, table, column, sql_statement);
    sqlite3_free (sql_statement);
    return;
  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_DiscardGeometryColumn (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ DiscardGeometryColumn(table, column)
/
/ removes TABLE.COLUMN from the Spatial MetaData [thus disabling triggers too]
/ returns 1 on success
/ 0 on failure
*/
    const unsigned char *table;
    const unsigned char *column;
    char *p_table = NULL;
    char *p_column = NULL;
    sqlite3_stmt *stmt;
    char *sql_statement;
    char *raw;
    char *quoted;
    char *errMsg = NULL;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("DiscardGeometryColumn() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("DiscardGeometryColumn() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = sqlite3_value_text (argv[1]);

    sql_statement = sqlite3_mprintf ("DELETE FROM geometry_columns "
				     "WHERE Lower(f_table_name) = Lower(?) "
				     "AND Lower(f_geometry_column) = Lower(?)");
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("DiscardGeometryColumn: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_int (context, 0);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, (const char *) table,
		       strlen ((const char *) table), SQLITE_STATIC);
    sqlite3_bind_text (stmt, 2, (const char *) column,
		       strlen ((const char *) column), SQLITE_STATIC);
    ret = sqlite3_step (stmt);
    if (ret == SQLITE_DONE || ret == SQLITE_ROW)
	;
    else
      {
	  spatialite_e ("DiscardGeometryColumn() error: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_finalize (stmt);
	  goto error;
      }
    sqlite3_finalize (stmt);
/* removing triggers too */
    if (!getRealSQLnames
	(sqlite, (const char *) table, (const char *) column, &p_table,
	 &p_column))
      {
	  spatialite_e
	      ("DiscardGeometryColumn() error: not existing Table or Column\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    raw = sqlite3_mprintf ("ggi_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("ggu_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gii_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("giu_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gid_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gci_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gcu_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gcd_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("tmi_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("tmu_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("tmd_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;

    /* trying to delete old versions [v2.0, v2.2] triggers[if any] */
    raw = sqlite3_mprintf ("gti_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gtu_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gsi_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    raw = sqlite3_mprintf ("gsu_%s_%s", p_table, p_column);
    quoted = gaiaDoubleQuotedSql (raw);
    sqlite3_free (raw);
    sql_statement =
	sqlite3_mprintf ("DROP TRIGGER IF EXISTS main.\"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    /* end deletion old versions [v2.0, v2.2] triggers[if any] */

    sqlite3_result_int (context, 1);
    updateSpatiaLiteHistory (sqlite, p_table,
			     p_column, "Geometry successfully discarded");
    free (p_table);
    free (p_column);
    return;
  error:
    if (p_table)
	free (p_table);
    if (p_column)
	free (p_column);
    spatialite_e ("DiscardGeometryColumn() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static int
registerVirtual (sqlite3 * sqlite, const char *table)
{
/* attempting to register a VirtualGeometry */
    char gtype[64];
    int xtype = -1;
    int srid = -1;
    char **results;
    int ret;
    int rows;
    int columns;
    int i;
    char *errMsg = NULL;
    int ok_virt_name = 0;
    int ok_virt_geometry = 0;
    int ok_srid = 0;
    int ok_geometry_type = 0;
    int ok_type = 0;
    int ok_coord_dimension = 0;
    int xdims;
    char *quoted;
    char *sql_statement;

/* testing the layout of virts_geometry_columns table */
    ret = sqlite3_get_table (sqlite,
			     "PRAGMA table_info(virts_geometry_columns)",
			     &results, &rows, &columns, &errMsg);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RegisterVirtualGeometry() error: \"%s\"\n", errMsg);
	  sqlite3_free (errMsg);
	  return 0;
      }
    for (i = 1; i <= rows; i++)
      {
	  if (strcasecmp ("virt_name", results[(i * columns) + 1]) == 0)
	      ok_virt_name = 1;
	  if (strcasecmp ("virt_geometry", results[(i * columns) + 1]) == 0)
	      ok_virt_geometry = 1;
	  if (strcasecmp ("srid", results[(i * columns) + 1]) == 0)
	      ok_srid = 1;
	  if (strcasecmp ("geometry_type", results[(i * columns) + 1]) == 0)
	      ok_geometry_type = 1;
	  if (strcasecmp ("type", results[(i * columns) + 1]) == 0)
	      ok_type = 1;
	  if (strcasecmp ("coord_dimension", results[(i * columns) + 1]) == 0)
	      ok_coord_dimension = 1;
      }
    sqlite3_free_table (results);

    if (ok_virt_name && ok_virt_geometry && ok_srid && ok_geometry_type
	&& ok_coord_dimension)
	;
    else if (ok_virt_name && ok_virt_geometry && ok_srid && ok_type)
	;
    else
	return 0;

/* determining Geometry Type and dims */
    quoted = gaiaDoubleQuotedSql (table);
    sql_statement =
	sqlite3_mprintf ("SELECT DISTINCT "
			 "ST_GeometryType(Geometry), ST_Srid(Geometry) FROM \"%s\"",
			 quoted);
    free (quoted);
    ret = sqlite3_get_table (sqlite, sql_statement, &results, &rows, &columns,
			     &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RegisterVirtualGeometry() error: \"%s\"\n", errMsg);
	  sqlite3_free (errMsg);
	  return 0;
      }
    for (i = 1; i <= rows; i++)
      {
	  if (results[(i * columns)] == NULL)
	      *gtype = '\0';
	  else
	      strcpy (gtype, results[(i * columns)]);
	  if (results[(i * columns) + 1] == NULL)
	      srid = 0;
	  else
	      srid = atoi (results[(i * columns) + 1]);
      }
    sqlite3_free_table (results);

/* normalized Geometry type */
    if (strcmp (gtype, "POINT") == 0)
	xtype = 1;
    if (strcmp (gtype, "POINT Z") == 0)
	xtype = 1001;
    if (strcmp (gtype, "POINT M") == 0)
	xtype = 2001;
    if (strcmp (gtype, "POINT ZM") == 0)
	xtype = 3001;
    if (strcmp (gtype, "LINESTRING") == 0)
	xtype = 2;
    if (strcmp (gtype, "LINESTRING Z") == 0)
	xtype = 1002;
    if (strcmp (gtype, "LINESTRING M") == 0)
	xtype = 2002;
    if (strcmp (gtype, "LINESTRING ZM") == 0)
	xtype = 3002;
    if (strcmp (gtype, "POLYGON") == 0)
	xtype = 3;
    if (strcmp (gtype, "POLYGON Z") == 0)
	xtype = 1003;
    if (strcmp (gtype, "POLYGON M") == 0)
	xtype = 2003;
    if (strcmp (gtype, "POLYGON ZM") == 0)
	xtype = 3003;
    if (strcmp (gtype, "MULTIPOINT") == 0)
	xtype = 4;
    if (strcmp (gtype, "MULTIPOINT Z") == 0)
	xtype = 1004;
    if (strcmp (gtype, "MULTIPOINT M") == 0)
	xtype = 2004;
    if (strcmp (gtype, "MULTIPOINT ZM") == 0)
	xtype = 3004;
    if (strcmp (gtype, "MULTILINESTRING") == 0)
	xtype = 5;
    if (strcmp (gtype, "MULTILINESTRING Z") == 0)
	xtype = 1005;
    if (strcmp (gtype, "MULTILINESTRING M") == 0)
	xtype = 2005;
    if (strcmp (gtype, "MULTILINESTRING ZM") == 0)
	xtype = 3005;
    if (strcmp (gtype, "MULTIPOLYGON") == 0)
	xtype = 6;
    if (strcmp (gtype, "MULTIPOLYGON Z") == 0)
	xtype = 1006;
    if (strcmp (gtype, "MULTIPOLYGON M") == 0)
	xtype = 2006;
    if (strcmp (gtype, "MULTIPOLYGON ZM") == 0)
	xtype = 3006;

/* updating metadata tables */
    xdims = -1;
    switch (xtype)
      {
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
      case 6:
	  xdims = 2;
	  break;
      case 1001:
      case 1002:
      case 1003:
      case 1004:
      case 1005:
      case 1006:
      case 2001:
      case 2002:
      case 2003:
      case 2004:
      case 2005:
      case 2006:
	  xdims = 3;
	  break;
      case 3001:
      case 3002:
      case 3003:
      case 3004:
      case 3005:
      case 3006:
	  xdims = 4;
	  break;
      };
    if (ok_geometry_type)
      {
	  /* has the "geometry_type" column */
	  sql_statement =
	      sqlite3_mprintf
	      ("INSERT OR REPLACE INTO virts_geometry_columns "
	       "(virt_name, virt_geometry, geometry_type, coord_dimension, srid) "
	       "VALUES (Lower(%Q), 'geometry', %d, %d, %d)", table, xtype,
	       xdims, srid);
      }
    else
      {
	  /* has the "type" column */
	  const char *xgtype = "UNKNOWN";
	  switch (xtype)
	    {
	    case 1:
	    case 1001:
	    case 2001:
	    case 3001:
		xgtype = "POINT";
		break;
	    case 2:
	    case 1002:
	    case 2002:
	    case 3002:
		xgtype = "LINESTRING";
		break;
	    case 3:
	    case 1003:
	    case 2003:
	    case 3003:
		xgtype = "POLYGON";
		break;
	    case 4:
	    case 1004:
	    case 2004:
	    case 3004:
		xgtype = "MULTIPOINT";
		break;
	    case 5:
	    case 1005:
	    case 2005:
	    case 3005:
		xgtype = "MULTILINESTRING";
		break;
	    case 6:
	    case 1006:
	    case 2006:
	    case 3006:
		xgtype = "MULTIPOLYGON";
		break;
	    };
	  sql_statement =
	      sqlite3_mprintf
	      ("INSERT OR REPLACE INTO virts_geometry_columns "
	       "(virt_name, virt_geometry, type, srid) "
	       "VALUES (Lower(%Q), 'geometry', %Q, %d)", table, xgtype, srid);
      }
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RegisterVirtualGeometry() error: \"%s\"\n", errMsg);
	  sqlite3_free (errMsg);
	  return 0;
      }
    if (checkSpatialMetaData (sqlite) == 3)
      {
	  /* current metadata style >= v.4.0.0 */

	  /* inserting a row into VIRTS_GEOMETRY_COLUMNS_AUTH */
	  sql_statement = sqlite3_mprintf ("INSERT OR REPLACE INTO "
					   "virts_geometry_columns_auth (virt_name, virt_geometry, hidden) "
					   "VALUES (Lower(%Q), 'geometry', 0)",
					   table);
	  ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
	  sqlite3_free (sql_statement);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("RegisterVirtualGeometry() error: \"%s\"\n",
			      errMsg);
		sqlite3_free (errMsg);
		return 0;
	    }
	  /* inserting a row into GEOMETRY_COLUMNS_STATISTICS */
	  sql_statement = sqlite3_mprintf ("INSERT OR REPLACE INTO "
					   "virts_geometry_columns_statistics (virt_name, virt_geometry) "
					   "VALUES (Lower(%Q), 'geometry')",
					   table);
	  ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
	  sqlite3_free (sql_statement);
	  if (ret != SQLITE_OK)
	    {
		spatialite_e ("RegisterVirtualGeometry() error: \"%s\"\n",
			      errMsg);
		sqlite3_free (errMsg);
		return 0;
	    }
      }
    return 1;
}

static void
fnct_RegisterVirtualGeometry (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ RegisterVirtualGeometry(table)
/
/ insert/updates TABLE.COLUMN into the Spatial MetaData [Virtual Table]
/ returns 1 on success
/ 0 on failure
*/
    const unsigned char *table;
    char sql[1024];
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RegisterVirtualGeometry() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = sqlite3_value_text (argv[0]);
    if (!registerVirtual (sqlite, (char *) table))
	goto error;
    sqlite3_result_int (context, 1);
    strcpy (sql, "Virtual Geometry successfully registered");
    updateSpatiaLiteHistory (sqlite, (const char *) table, "Geometry", sql);
    return;
  error:
    spatialite_e ("RegisterVirtualGeometry() error\n");
    sqlite3_result_int (context, 0);
    return;
}


static void
fnct_DropVirtualGeometry (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ DropVirtualGeometry(table)
/
/ removes TABLE.COLUMN from the Spatial MetaData and DROPs the Virtual Table
/ returns 1 on success
/ 0 on failure
*/
    const unsigned char *table;
    char *sql_statement;
    char *errMsg = NULL;
    int ret;
    char *quoted;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("DropVirtualGeometry() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = sqlite3_value_text (argv[0]);
    sql_statement = sqlite3_mprintf ("DELETE FROM virts_geometry_columns "
				     "WHERE Lower(virt_name) = Lower(%Q)",
				     table);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    quoted = gaiaDoubleQuotedSql ((const char *) table);
    sql_statement = sqlite3_mprintf ("DROP TABLE IF EXISTS \"%s\"", quoted);
    free (quoted);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    sqlite3_result_int (context, 1);
    updateSpatiaLiteHistory (sqlite, (const char *) table, "Geometry",
			     "Virtual Geometry successfully dropped");
    return;
  error:
    spatialite_e ("DropVirtualGeometry() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_InitFDOSpatialMetaData (sqlite3_context * context, int argc,
			     sqlite3_value ** argv)
{
/* SQL function:
/ InitFDOSpatialMetaData(void)
/
/ creates the FDO-styled SPATIAL_REF_SYS and GEOMETRY_COLUMNS tables
/ returns 1 on success
/ 0 on failure
*/
    char sql[1024];
    char *errMsg = NULL;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
/* creating the SPATIAL_REF_SYS tables */
    strcpy (sql, "CREATE TABLE spatial_ref_sys (\n");
    strcat (sql, "srid INTEGER PRIMARY KEY,\n");
    strcat (sql, "auth_name TEXT,\n");
    strcat (sql, "auth_srid INTEGER,\n");
    strcat (sql, "srtext TEXT)");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;
/* creating the GEOMETRY_COLUMN tables */
    strcpy (sql, "CREATE TABLE geometry_columns (\n");
    strcat (sql, "f_table_name TEXT,\n");
    strcat (sql, "f_geometry_column TEXT,\n");
    strcat (sql, "geometry_type INTEGER,\n");
    strcat (sql, "coord_dimension INTEGER,\n");
    strcat (sql, "srid INTEGER,\n");
    strcat (sql, "geometry_format TEXT)");
    ret = sqlite3_exec (sqlite, sql, NULL, NULL, &errMsg);
    if (ret != SQLITE_OK)
	goto error;
    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("InitFDOSpatiaMetaData() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static int
recoverFDOGeomColumn (sqlite3 * sqlite, const unsigned char *table,
		      const unsigned char *column, int xtype, int srid)
{
/* checks if TABLE.COLUMN exists and has the required features */
    int ok = 1;
    char *sql_statement;
    int type;
    sqlite3_stmt *stmt;
    gaiaGeomCollPtr geom;
    const void *blob_value;
    int len;
    int ret;
    int i_col;
    char *xcolumn;
    char *xtable;
    xcolumn = gaiaDoubleQuotedSql ((char *) column);
    xtable = gaiaDoubleQuotedSql ((char *) table);
    sql_statement =
	sqlite3_mprintf ("SELECT \"%s\" FROM \"%s\"", xcolumn, xtable);
    free (xcolumn);
    free (xtable);
/* compiling SQL prepared statement */
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("recoverFDOGeomColumn: error %d \"%s\"\n",
			sqlite3_errcode (sqlite), sqlite3_errmsg (sqlite));
	  return 0;
      }
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	    {
		/* checking Geometry features */
		geom = NULL;
		for (i_col = 0; i_col < sqlite3_column_count (stmt); i_col++)
		  {
		      if (sqlite3_column_type (stmt, i_col) != SQLITE_BLOB)
			  ok = 0;
		      else
			{
			    blob_value = sqlite3_column_blob (stmt, i_col);
			    len = sqlite3_column_bytes (stmt, i_col);
			    geom = gaiaFromSpatiaLiteBlobWkb (blob_value, len);
			    if (!geom)
				ok = 0;
			    else
			      {
				  if (geom->Srid != srid)
				      ok = 0;
				  /* normalizing Geometry Type */
				  switch (gaiaGeometryType (geom))
				    {
				    case GAIA_POINT:
				    case GAIA_POINTZ:
				    case GAIA_POINTM:
				    case GAIA_POINTZM:
					type = GAIA_POINT;
					break;
				    case GAIA_LINESTRING:
				    case GAIA_LINESTRINGZ:
				    case GAIA_LINESTRINGM:
				    case GAIA_LINESTRINGZM:
					type = GAIA_LINESTRING;
					break;
				    case GAIA_POLYGON:
				    case GAIA_POLYGONZ:
				    case GAIA_POLYGONM:
				    case GAIA_POLYGONZM:
					type = GAIA_POLYGON;
					break;
				    case GAIA_MULTIPOINT:
				    case GAIA_MULTIPOINTZ:
				    case GAIA_MULTIPOINTM:
				    case GAIA_MULTIPOINTZM:
					type = GAIA_MULTIPOINT;
					break;
				    case GAIA_MULTILINESTRING:
				    case GAIA_MULTILINESTRINGZ:
				    case GAIA_MULTILINESTRINGM:
				    case GAIA_MULTILINESTRINGZM:
					type = GAIA_MULTILINESTRING;
					break;
				    case GAIA_MULTIPOLYGON:
				    case GAIA_MULTIPOLYGONZ:
				    case GAIA_MULTIPOLYGONM:
				    case GAIA_MULTIPOLYGONZM:
					type = GAIA_MULTIPOLYGON;
					break;
				    case GAIA_GEOMETRYCOLLECTION:
				    case GAIA_GEOMETRYCOLLECTIONZ:
				    case GAIA_GEOMETRYCOLLECTIONM:
				    case GAIA_GEOMETRYCOLLECTIONZM:
					type = GAIA_GEOMETRYCOLLECTION;
					break;
				    default:
					type = -1;
					break;
				    };
				  if (xtype == type)
				      ;
				  else
				      ok = 0;
				  gaiaFreeGeomColl (geom);
			      }
			}
		  }
	    }
	  if (!ok)
	      break;
      }
    ret = sqlite3_finalize (stmt);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("recoverFDOGeomColumn: error %d \"%s\"\n",
			sqlite3_errcode (sqlite), sqlite3_errmsg (sqlite));
	  return 0;
      }
    return ok;
}

static void
fnct_AddFDOGeometryColumn (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ AddFDOGeometryColumn(table, column, srid, geometry_type , dimension, geometry_format )
/
/ creates a new COLUMN of given TYPE into TABLE
/ Adds a matching entry into spatial_ref_sys when needed [Mark Johnson patch]
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    const char *format;
    char xformat[64];
    int type;
    int srid = -1;
    int srid_exists = -1;
    struct epsg_defs *first = NULL;
    struct epsg_defs *last = NULL;
    sqlite3_stmt *stmt_sql;
    int dimension = 2;
    char *sql_statement;
    char *errMsg = NULL;
    int ret;
    char **results;
    int rows;
    int columns;
    int i;
    int oktbl;
    char *xtable;
    char *xcolumn;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 3 [SRID] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    srid = sqlite3_value_int (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 4 [geometry_type] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    type = sqlite3_value_int (argv[3]);
    if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 5 [dimension] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    dimension = sqlite3_value_int (argv[4]);
    if (sqlite3_value_type (argv[5]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 6 [geometry_format] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    format = (const char *) sqlite3_value_text (argv[5]);
    if (type ==
	GAIA_POINT
	|| type ==
	GAIA_LINESTRING
	|| type ==
	GAIA_POLYGON
	|| type ==
	GAIA_MULTIPOINT
	|| type ==
	GAIA_MULTILINESTRING
	|| type == GAIA_MULTIPOLYGON || type == GAIA_GEOMETRYCOLLECTION)
	;
    else
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 4 [geometry_type] has an illegal value\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (dimension < 2 || dimension > 4)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 5 [dimension] current version only accepts dimension=2,3,4\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (strcasecmp (format, "WKT") == 0)
	strcpy (xformat, "WKT");
    else if (strcasecmp (format, "WKB") == 0)
	strcpy (xformat, "WKB");
    else if (strcasecmp (format, "FGF") == 0)
	strcpy (xformat, "FGF");
    else if (strcasecmp (format, "SPATIALITE") == 0)
	strcpy (xformat, "SPATIALITE");
    else
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: argument 6 [geometry_format] has to be one of: WKT,WKB,FGF,SPATIALITE\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
/* checking if the table exists */
    xtable = gaiaDoubleQuotedSql (table);
    xcolumn = gaiaDoubleQuotedSql (column);
    sql_statement = sqlite3_mprintf ("SELECT name FROM sqlite_master "
				     "WHERE type = 'table' AND Upper(name) = Upper(%Q)",
				     table);
    free (xtable);
    free (xcolumn);
    ret = sqlite3_get_table (sqlite, sql_statement, &results, &rows, &columns,
			     &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("AddFDOGeometryColumn: \"%s\"\n", errMsg);
	  sqlite3_free (errMsg);
	  return;
      }
    oktbl = 0;
    for (i = 1; i <= rows; i++)
	oktbl = 1;
    sqlite3_free_table (results);
    if (!oktbl)
      {
	  spatialite_e
	      ("AddFDOGeometryColumn() error: table '%s' does not exist\n",
	       table);
	  sqlite3_result_int (context, 0);
	  return;
      }

/*
 * the following code has been contributed by Mark Johnson <mj10777@googlemail.com>
 * on 2019-01-26
*/
    sql_statement =
	sqlite3_mprintf
	("SELECT CASE WHEN (Exists(SELECT srid FROM spatial_ref_sys WHERE (auth_srid = %d)) = 0) THEN 0 ELSE 1 END",
	 srid);
    ret =
	sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			    &stmt_sql, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  sqlite3_result_error (context, sqlite3_errmsg (sqlite), -1);
	  sqlite3_free (errMsg);
	  return;
      }
    while (sqlite3_step (stmt_sql) == SQLITE_ROW)
      {
	  if (sqlite3_column_type (stmt_sql, 0) != SQLITE_NULL)
	    {
		srid_exists = sqlite3_column_int (stmt_sql, 0);
	    }
      }
    sqlite3_finalize (stmt_sql);
    if (srid_exists == 0)
      {
	  /* get the EPSG definition for this SRID from our master list */
	  initialize_epsg (srid, &first, &last);
	  if (first == NULL)
	    {
		sql_statement =
		    sqlite3_mprintf
		    ("AddFDOGeometryColumn() error: srid[%d] is not defined in the EPSG inlined dataset",
		     srid);
		sqlite3_result_error (context, sql_statement, -1);
		sqlite3_free (sql_statement);
		return;
	    }
	  /* add the definition for the SRID */
	  sql_statement =
	      sqlite3_mprintf
	      ("INSERT INTO spatial_ref_sys (srid, auth_name, auth_srid,srtext) VALUES (?, ?, ?, ?)");
	  ret =
	      sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
				  &stmt_sql, NULL);
	  if (ret != SQLITE_OK)
	    {
		sqlite3_free (sql_statement);
		sqlite3_result_error (context, sqlite3_errmsg (sqlite), -1);
		/* freeing the EPSG defs list */
		free_epsg (first);
		return;
	    }
	  sqlite3_bind_int (stmt_sql, 1, first->srid);
	  sqlite3_bind_text (stmt_sql, 2, first->auth_name,
			     strlen (first->auth_name), SQLITE_STATIC);
	  sqlite3_bind_int (stmt_sql, 3, first->auth_srid);
	  if (strlen (first->srs_wkt) == 0)
	    {
		sqlite3_bind_text (stmt_sql, 4, "Undefined", 9, SQLITE_STATIC);
	    }
	  else
	    {
		sqlite3_bind_text (stmt_sql, 4, first->srs_wkt,
				   strlen (first->srs_wkt), SQLITE_STATIC);
	    }
	  ret = sqlite3_step (stmt_sql);
	  /* freeing the EPSG defs list */
	  free_epsg (first);
	  if (stmt_sql != NULL)
	    {
		sqlite3_finalize (stmt_sql);
	    }
	  sqlite3_free (sql_statement);
	  if (ret != SQLITE_DONE && ret != SQLITE_ROW)
	    {
		sqlite3_result_error (context, sqlite3_errmsg (sqlite), -1);
		return;
	    }
      }
    /* The EPSG definition for this SRID exists in spatial_ref_sys */
/* end Mark Johnson 2019-01-26 */

/* trying to add the column */
    xtable = gaiaDoubleQuotedSql (table);
    xcolumn = gaiaDoubleQuotedSql (column);
    sql_statement = sqlite3_mprintf ("ALTER TABLE \"%s\" "
				     "ADD COLUMN \"%s\" BLOB", xtable, xcolumn);
    free (xtable);
    free (xcolumn);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
/*ok, inserting into geometry_columns [FDO Spatial Metadata] */
    sql_statement = sqlite3_mprintf ("INSERT INTO geometry_columns "
				     "(f_table_name, f_geometry_column, geometry_type, "
				     "coord_dimension, srid, geometry_format) VALUES (%Q, %Q, %d, %d, %d, %Q)",
				     table, column, type, dimension,
				     (srid <= 0) ? -1 : srid, xformat);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("AddFDOGeometryColumn() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_RecoverFDOGeometryColumn (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ RecoverFDOGeometryColumn(table, column, srid, geometry_type , dimension, geometry_format )
/
/ checks if an existing TABLE.COLUMN satisfies the required geometric features
/ if yes adds it to FDO-styled SpatialMetaData 
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    const char *format;
    char xformat[64];
    int type;
    int srid = -1;
    int dimension = 2;
    char *sql_statement;
    char *errMsg = NULL;
    int ret;
    char **results;
    int rows;
    int columns;
    int i;
    int ok_tbl;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 3 [SRID] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    srid = sqlite3_value_int (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 4 [geometry_type] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    type = sqlite3_value_int (argv[3]);
    if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 5 [dimension] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    dimension = sqlite3_value_int (argv[4]);
    if (sqlite3_value_type (argv[5]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 6 [geometry_format] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    format = (const char *) sqlite3_value_text (argv[5]);
    if (type ==
	GAIA_POINT
	|| type ==
	GAIA_LINESTRING
	|| type ==
	GAIA_POLYGON
	|| type ==
	GAIA_MULTIPOINT
	|| type ==
	GAIA_MULTILINESTRING
	|| type == GAIA_MULTIPOLYGON || type == GAIA_GEOMETRYCOLLECTION)
	;
    else
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 4 [geometry_type] has an illegal value\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (dimension < 2 || dimension > 4)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 5 [dimension] current version only accepts dimension=2,3,4\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (strcasecmp (format, "WKT") == 0)
	strcpy (xformat, "WKT");
    else if (strcasecmp (format, "WKB") == 0)
	strcpy (xformat, "WKB");
    else if (strcasecmp (format, "FGF") == 0)
	strcpy (xformat, "FGF");
    else if (strcasecmp (format, "SPATIALITE") == 0)
	strcpy (xformat, "SPATIALITE");
    else
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: argument 6 [geometry_format] has to be one of: WKT,WKB,FGF\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
/* checking if the table exists */
    sql_statement = sqlite3_mprintf ("SELECT name FROM sqlite_master "
				     "WHERE type = 'table' AND Upper(name) = Upper(%Q)",
				     table);
    ret = sqlite3_get_table (sqlite, sql_statement, &results, &rows, &columns,
			     &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RecoverFDOGeometryColumn: \"%s\"\n", errMsg);
	  sqlite3_free (errMsg);
	  return;
      }
    ok_tbl = 0;
    for (i = 1; i <= rows; i++)
	ok_tbl = 1;
    sqlite3_free_table (results);
    if (!ok_tbl)
      {
	  spatialite_e
	      ("RecoverFDOGeometryColumn() error: table '%s' does not exist\n",
	       table);
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (!recoverFDOGeomColumn
	(sqlite, (const unsigned char *) table,
	 (const unsigned char *) column, type, srid))
      {
	  spatialite_e ("RecoverFDOGeometryColumn(): validation failed\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    sql_statement = sqlite3_mprintf ("INSERT INTO geometry_columns "
				     "(f_table_name, f_geometry_column, geometry_type, "
				     "coord_dimension, srid, geometry_format) VALUES (%Q, %Q, %d, %d, %d, %Q)",
				     table, column, type, dimension,
				     (srid <= 0) ? -1 : srid, xformat);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("RecoverFDOGeometryColumn() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_DiscardFDOGeometryColumn (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ DiscardFDOGeometryColumn(table, column)
/
/ removes TABLE.COLUMN from the Spatial MetaData
/ returns 1 on success
/ 0 on failure
*/
    const unsigned char *table;
    const unsigned char *column;
    char *sql_statement;
    char *errMsg = NULL;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("DiscardFDOGeometryColumn() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("DiscardFDOGeometryColumn() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = sqlite3_value_text (argv[1]);
    sql_statement =
	sqlite3_mprintf
	("DELETE FROM geometry_columns WHERE Upper(f_table_name) = "
	 "Upper(%Q) AND Upper(f_geometry_column) = Upper(%Q)", table, column);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("DiscardFDOGeometryColumn() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_GetDbObjectScope (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ GetDbObjectScope(da-prefix, name)
/
/ returns the intended scope of some DB Object
/ returns a text string
/ NULL on invalid arguments
*/
    const char *db_prefix = NULL;
    const char *name;
    char *scope;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
	name = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    name = (const char *) sqlite3_value_text (argv[1]);
    scope = gaiaGetDbObjectScope (sqlite, db_prefix, name);
    if (scope == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_result_text (context, scope, strlen (scope), sqlite3_free);
}

static int
eval_rtree_entry (int ok_geom, double geom_value, int ok_rtree,
		  double rtree_value)
{
/* evaluating geom-coord and rtree-coord */
    if (!ok_geom && !ok_rtree)
	return 1;
    if (ok_geom && ok_rtree)
      {
	  float g = (float) geom_value;
	  float r = (float) rtree_value;
	  double tic = fabs (geom_value - r) * 2.0;
	  float diff = g - r;
	  if (diff >= 1.5)
	      return 0;
	  if (diff > tic)
	      return 0;
	  return 1;
      }
    return 0;
}

static int
check_spatial_index (sqlite3 * sqlite, const unsigned char *table,
		     const unsigned char *geom)
{
/* attempting to check an R*Tree for consistency */
    char *xtable = NULL;
    char *xgeom = NULL;
    char *idx_name;
    char *xidx_name = NULL;
    char sql[1024];
    char *sql_statement;
    int ret;
    int is_defined = 0;
    sqlite3_stmt *stmt;
    sqlite3_int64 count_geom = 0;
    sqlite3_int64 count_rtree = 0;
    sqlite3_int64 count_rev = 0;
    double g_xmin = DBL_MAX;
    double g_ymin = DBL_MAX;
    double g_xmax = 0.0 - DBL_MAX;
    double g_ymax = 0.0 - DBL_MAX;
    int ok_g_xmin;
    int ok_g_ymin;
    int ok_g_xmax;
    int ok_g_ymax;
    double i_xmin = DBL_MAX;
    double i_ymin = DBL_MAX;
    double i_xmax = 0.0 - DBL_MAX;
    double i_ymax = 0.0 - DBL_MAX;
    int ok_i_xmin;
    int ok_i_ymin;
    int ok_i_xmax;
    int ok_i_ymax;
    int rowid_column = 0;
    int without_rowid = 0;

    if (is_without_rowid_table (sqlite, (char *) table))
      {
	  spatialite_e
	      ("check_spatial_index: table \"%s\" is WITHOUT ROWID\n", table);
	  without_rowid = 1;
	  goto err_label;
      }

/* checking if the R*Tree Spatial Index is defined */
    sql_statement = sqlite3_mprintf ("SELECT Count(*) FROM geometry_columns "
				     "WHERE Upper(f_table_name) = Upper(%Q) "
				     "AND Upper(f_geometry_column) = Upper(%Q) AND spatial_index_enabled = 1",
				     table, geom);
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("CheckSpatialIndex SQL error: %s\n",
			sqlite3_errmsg (sqlite));
	  goto err_label;
      }
    while (1)
      {
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;
	  if (ret == SQLITE_ROW)
	      is_defined = sqlite3_column_int (stmt, 0);
	  else
	    {
		spatialite_e ("sqlite3_step() error: %s\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto err_label;
	    }
      }
    sqlite3_finalize (stmt);
    if (!is_defined)
	goto err_label;

    xgeom = gaiaDoubleQuotedSql ((char *) geom);
    xtable = gaiaDoubleQuotedSql ((char *) table);
    idx_name = sqlite3_mprintf ("idx_%s_%s", table, geom);
    xidx_name = gaiaDoubleQuotedSql (idx_name);
    sqlite3_free (idx_name);

    if (!validateRowid (sqlite, (const char *) table))
      {
	  /* a physical column named "rowid" shadows the real ROWID */
	  rowid_column = 1;
	  goto err_label;
      }

/* counting how many Geometries are set into the main-table */
    sql_statement = sqlite3_mprintf ("SELECT Count(*) FROM \"%s\" "
				     "WHERE ST_GeometryType(\"%s\") IS NOT NULL",
				     xtable, xgeom);
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("CheckSpatialIndex SQL error: %s\n",
			sqlite3_errmsg (sqlite));
	  goto err_label;
      }
    while (1)
      {
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;
	  if (ret == SQLITE_ROW)
	      count_geom = sqlite3_column_int64 (stmt, 0);
	  else
	    {
		spatialite_e ("sqlite3_step() error: %s\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto err_label;
	    }
      }
    sqlite3_finalize (stmt);

/* counting how many R*Tree entries are defined */
    sql_statement = sqlite3_mprintf ("SELECT Count(*) FROM \"%s\"", xidx_name);
    ret =
	sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			    &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("CheckSpatialIndex SQL error: %s\n",
			sqlite3_errmsg (sqlite));
	  goto err_label;
      }
    while (1)
      {
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;
	  if (ret == SQLITE_ROW)
	      count_rtree = sqlite3_column_int64 (stmt, 0);
	  else
	    {
		spatialite_e ("sqlite3_step() error: %s\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto err_label;
	    }
      }
    sqlite3_finalize (stmt);
    if (count_geom != count_rtree)
      {
	  /* unexpected count difference */
	  goto mismatching_zero;
      }

/* now we'll check the R*Tree against the corresponding geometry-table */
    sql_statement =
	sqlite3_mprintf ("SELECT MbrMinX(g.\"%s\"), MbrMinY(g.\"%s\"), "
			 "MbrMaxX(g.\"%s\"), MbrMaxY(g.\"%s\"), i.xmin, i.ymin, i.xmax, i.ymax\n"
			 "FROM \"%s\" AS i\nLEFT JOIN \"%s\" AS g ON (g.ROWID = i.pkid)",
			 xgeom, xgeom, xgeom, xgeom, xidx_name, xtable);
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("CheckSpatialIndex SQL error: %s\n",
			sqlite3_errmsg (sqlite));
	  goto err_label;
      }
    while (1)
      {
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;
	  if (ret == SQLITE_ROW)
	    {
		/* checking a row */
		count_rev++;
		ok_g_xmin = 1;
		ok_g_ymin = 1;
		ok_g_xmax = 1;
		ok_g_ymax = 1;
		ok_i_xmin = 1;
		ok_i_ymin = 1;
		ok_i_xmax = 1;
		ok_i_ymax = 1;
		if (sqlite3_column_type (stmt, 0) == SQLITE_NULL)
		    ok_g_xmin = 0;
		else
		    g_xmin = sqlite3_column_double (stmt, 0);
		if (sqlite3_column_type (stmt, 1) == SQLITE_NULL)
		    ok_g_ymin = 0;
		else
		    g_ymin = sqlite3_column_double (stmt, 1);
		if (sqlite3_column_type (stmt, 2) == SQLITE_NULL)
		    ok_g_xmax = 0;
		else
		    g_xmax = sqlite3_column_double (stmt, 2);
		if (sqlite3_column_type (stmt, 3) == SQLITE_NULL)
		    ok_g_ymax = 0;
		else
		    g_ymax = sqlite3_column_double (stmt, 3);
		if (sqlite3_column_type (stmt, 4) == SQLITE_NULL)
		    ok_i_xmin = 0;
		else
		    i_xmin = sqlite3_column_double (stmt, 4);
		if (sqlite3_column_type (stmt, 5) == SQLITE_NULL)
		    ok_i_ymin = 0;
		else
		    i_ymin = sqlite3_column_double (stmt, 5);
		if (sqlite3_column_type (stmt, 6) == SQLITE_NULL)
		    ok_i_xmax = 0;
		else
		    i_xmax = sqlite3_column_double (stmt, 6);
		if (sqlite3_column_type (stmt, 7) == SQLITE_NULL)
		    ok_i_ymax = 0;
		else
		    i_ymax = sqlite3_column_double (stmt, 7);
		if (eval_rtree_entry (ok_g_xmin, g_xmin, ok_i_xmin, i_xmin)
		    == 0)
		    goto mismatching;
		if (eval_rtree_entry (ok_g_ymin, g_ymin, ok_i_ymin, i_ymin)
		    == 0)
		    goto mismatching;
		if (eval_rtree_entry (ok_g_xmax, g_xmax, ok_i_xmax, i_xmax)
		    == 0)
		    goto mismatching;
		if (eval_rtree_entry (ok_g_ymax, g_ymax, ok_i_ymax, i_ymax)
		    == 0)
		    goto mismatching;
	    }
	  else
	    {
		spatialite_e ("sqlite3_step() error: %s\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		goto err_label;
	    }
      }
    sqlite3_finalize (stmt);
    if (count_geom != count_rev)
	goto mismatching;
    strcpy (sql, "Check SpatialIndex: is valid");
    updateSpatiaLiteHistory (sqlite, (const char *) table,
			     (const char *) geom, sql);
    free (xgeom);
    free (xtable);
    free (xidx_name);
    return 1;
  mismatching:
    sqlite3_finalize (stmt);
    strcpy (sql, "Check SpatialIndex: INCONSISTENCIES detected");
    updateSpatiaLiteHistory (sqlite, (const char *) table,
			     (const char *) geom, sql);
  mismatching_zero:
    if (xgeom)
	free (xgeom);
    if (xtable)
	free (xtable);
    if (xidx_name)
	free (xidx_name);
    return 0;
  err_label:
    if (xgeom)
	free (xgeom);
    if (xtable)
	free (xtable);
    if (xidx_name)
	free (xidx_name);
    if (rowid_column)
	return -2;
    if (without_rowid)
	return -3;
    return -1;
}

static int
check_any_spatial_index (sqlite3 * sqlite)
{
/* attempting to check any defined R*Tree for consistency */
    const unsigned char *table;
    const unsigned char *column;
    int status;
    char sql[1024];
    int ret;
    int invalid_rtree = 0;
    sqlite3_stmt *stmt;

/* retrieving any defined R*Tree */
    strcpy (sql,
	    "SELECT f_table_name, f_geometry_column FROM geometry_columns ");
    strcat (sql, "WHERE spatial_index_enabled = 1");
    ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("CheckSpatialIndex SQL error: %s\n",
			sqlite3_errmsg (sqlite));
	  return -1;
      }
    while (1)
      {
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;
	  if (ret == SQLITE_ROW)
	    {
		/* checking a single R*Tree */
		table = sqlite3_column_text (stmt, 0);
		column = sqlite3_column_text (stmt, 1);
		status = check_spatial_index (sqlite, table, column);
		if (status < 0)
		  {
		      sqlite3_finalize (stmt);
		      return status;
		  }
		if (status == 0)
		    invalid_rtree = 1;
	    }
	  else
	    {
		spatialite_e ("sqlite3_step() error: %s\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		return -1;
	    }
      }
    sqlite3_finalize (stmt);
    if (invalid_rtree)
	return 0;
    return 1;
}

static void
fnct_CheckSpatialIndex (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ CheckSpatialIndex()
/ CheckSpatialIndex(table, column)
/
/ checks a SpatialIndex for consistency, returning:
/ 1 - the R*Tree is fully consistent
/ 0 - the R*Tree is inconsistent
/ -1 if any physical "ROWID" column exist shadowing the real ROWID
/ NULL on failure
*/
    const unsigned char *table;
    const unsigned char *column;
    int status;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 0)
      {
	  /* no arguments: we must check any defined R*Tree */
	  status = check_any_spatial_index (sqlite);
	  if (status < 0)
	    {
		if (status == -2)
		    sqlite3_result_int (context, -1);
		else
		    sqlite3_result_null (context);
	    }
	  else if (status > 0)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }

    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CheckSpatialIndex() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_null (context);
	  return;
      }
    table = sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CheckSpatialIndex() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_null (context);
	  return;
      }
    column = sqlite3_value_text (argv[1]);
    status = check_spatial_index (sqlite, table, column);
    if (status == -2)
	sqlite3_result_int (context, -1);
    else if (status == -3)
	sqlite3_result_int (context, -1);
    else if (status < 0)
	sqlite3_result_null (context);
    else if (status > 0)
	sqlite3_result_int (context, 1);
    else
	sqlite3_result_int (context, 0);
}

static int
recover_spatial_index (sqlite3 * sqlite, const unsigned char *table,
		       const unsigned char *geom)
{
/* attempting to rebuild an R*Tree */
    char *sql_statement;
    char *errMsg = NULL;
    int ret;
    char *idx_name;
    char *xidx_name;
    char sql[1024];
    int is_defined = 0;
    sqlite3_stmt *stmt;
    int status;

/* checking if the R*Tree Spatial Index is defined */
    sql_statement = sqlite3_mprintf ("SELECT Count(*) FROM geometry_columns "
				     "WHERE Upper(f_table_name) = Upper(%Q) "
				     "AND Upper(f_geometry_column) = Upper(%Q) AND spatial_index_enabled = 1",
				     table, geom);
    ret = sqlite3_prepare_v2 (sqlite, sql_statement, strlen (sql_statement),
			      &stmt, NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RecoverSpatialIndex SQL error: %s\n",
			sqlite3_errmsg (sqlite));
	  return -1;
      }
    while (1)
      {
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;
	  if (ret == SQLITE_ROW)
	      is_defined = sqlite3_column_int (stmt, 0);
	  else
	    {
		spatialite_e ("sqlite3_step() error: %s\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		return -1;
	    }
      }
    sqlite3_finalize (stmt);
    if (!is_defined)
	return -1;

/* erasing the R*Tree table */
    idx_name = sqlite3_mprintf ("idx_%s_%s", table, geom);
    xidx_name = gaiaDoubleQuotedSql (idx_name);
    sqlite3_free (idx_name);
    sql_statement = sqlite3_mprintf ("DELETE FROM \"%s\"", xidx_name);
    free (xidx_name);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
/* populating the R*Tree table from scratch */
    status = buildSpatialIndexEx (sqlite, table, (const char *) geom);
    if (status == 0)
	;
    else
      {
	  if (status == -2)
	    {
		strcpy (sql,
			"SpatialIndex: a physical column named ROWID shadows the real ROWID");
		updateSpatiaLiteHistory (sqlite, (const char *) table,
					 (const char *) geom, sql);
	    }
	  else
	    {
		strcpy (sql, "SpatialIndex: unable to rebuild the R*Tree");
		updateSpatiaLiteHistory (sqlite, (const char *) table,
					 (const char *) geom, sql);
	    }
	  return status;
      }
    strcpy (sql, "SpatialIndex: successfully recovered");
    updateSpatiaLiteHistory (sqlite, (const char *) table,
			     (const char *) geom, sql);
    return 1;
  error:
    spatialite_e ("RecoverSpatialIndex() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    return 0;
}

static int
recover_any_spatial_index (sqlite3 * sqlite, int no_check)
{
/* attempting to rebuild any defined R*Tree */
    const unsigned char *table;
    const unsigned char *column;
    int status;
    char sql[1024];
    int ret;
    int to_be_fixed;
    int rowid_column = 0;
    int without_rowid = 0;
    sqlite3_stmt *stmt;

/* retrieving any defined R*Tree */
    strcpy (sql,
	    "SELECT f_table_name, f_geometry_column FROM geometry_columns ");
    strcat (sql, "WHERE spatial_index_enabled = 1");
    ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("RecoverSpatialIndex SQL error: %s\n",
			sqlite3_errmsg (sqlite));
	  return -1;
      }
    while (1)
      {
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;
	  if (ret == SQLITE_ROW)
	    {
		/* checking a single R*Tree */
		table = sqlite3_column_text (stmt, 0);
		column = sqlite3_column_text (stmt, 1);
		to_be_fixed = 1;
		if (!no_check)
		  {
		      status = check_spatial_index (sqlite, table, column);
		      if (status < 0)
			{
			    /* some unexpected error occurred */
			    if (status == -2)
				rowid_column = 1;
			    if (status == -3)
				without_rowid = 1;
			    goto fatal_error;
			}
		      else if (status > 0)
			{
			    /* the Spatial Index is already valid */
			    to_be_fixed = 0;
			}
		  }
		if (to_be_fixed)
		  {
		      /* rebuilding the Spatial Index */
		      status = recover_spatial_index (sqlite, table, column);
		      if (status < 0)
			{
			    /* some unexpected error occurred */
			    if (status == -2)
				rowid_column = 1;
			    if (status == -3)
				without_rowid = 1;
			    goto fatal_error;
			}
		      else if (status == 0)
			  goto error;
		  }
	    }
	  else
	    {
		spatialite_e ("sqlite3_step() error: %s\n",
			      sqlite3_errmsg (sqlite));
		sqlite3_finalize (stmt);
		return -1;
	    }
      }
    sqlite3_finalize (stmt);
    return 1;
  error:
    sqlite3_finalize (stmt);
    return 0;
  fatal_error:
    sqlite3_finalize (stmt);
    if (rowid_column)
	return -2;
    if (without_rowid)
	return -3;
    return -1;
}

static void
fnct_RecoverSpatialIndex (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ RecoverSpatialIndex()
/ RecoverSpatialIndex(no_check)
/ RecoverSpatialIndex(table, column)
/ RecoverSpatialIndex(table, column, no_check)
/
/ attempts to rebuild a SpatialIndex, returning:
/ 1 - on success
/ 0 - on failure
/ -1 if any physical "ROWID" column exist shadowing the real ROWID
/    or if the table was created WITHOUT ROWID
/ NULL if any syntax error is detected
*/
    const unsigned char *table;
    const unsigned char *column;
    int no_check = 0;
    int status;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc <= 1)
      {
	  /* no arguments: we must rebuild any defined R*Tree */
	  if (argc == 1)
	    {
		if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
		    no_check = sqlite3_value_int (argv[0]);
		else
		  {
		      spatialite_e
			  ("RecoverSpatialIndex() error: argument 1 [no_check] is not of the Integer type\n");
		      sqlite3_result_null (context);
		      return;
		  }
	    }
	  status = recover_any_spatial_index (sqlite, no_check);
	  if (status < 0)
	    {
		if (status == -2)
		    sqlite3_result_int (context, -1);
		else if (status == -3)
		    sqlite3_result_int (context, -1);
		else
		    sqlite3_result_null (context);
	    }
	  else if (status > 0)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }

    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverSpatialIndex() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_null (context);
	  return;
      }
    table = sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RecoverSpatialIndex() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_null (context);
	  return;
      }
    column = sqlite3_value_text (argv[1]);
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      no_check = sqlite3_value_int (argv[2]);
	  else
	    {
		spatialite_e
		    ("RecoverSpatialIndex() error: argument 2 [no_check] is not of the Integer type\n");
		sqlite3_result_null (context);
		return;
	    }
      }
    if (!no_check)
      {
	  /* checking the current SpatialIndex validity */
	  status = check_spatial_index (sqlite, table, column);
	  if (status < 0)
	    {
		/* some unexpected error occurred */
		if (status == -2 || status == -3)
		    sqlite3_result_int (context, -1);
		else
		    sqlite3_result_null (context);
		return;
	    }
	  else if (status > 0)
	    {
		/* the Spatial Index is already valid */
		sqlite3_result_int (context, 1);
		return;
	    }
      }
/* rebuilding the Spatial Index */
    status = recover_spatial_index (sqlite, table, column);
    if (status == -2)
	sqlite3_result_int (context, -1);
    else if (status < 0)
	sqlite3_result_null (context);
    else if (status > 0)
	sqlite3_result_int (context, 1);
    else
	sqlite3_result_int (context, 0);
}

static void
fnct_CheckShadowedRowid (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ CheckShadowedRowid(table)
/
/ checks if some table has a "rowid" physical column shadowing the real ROWID
/ 1 - yes, the ROWID is shadowed
/ 0 - no, the ROWID isn't shadowed
/ NULL on failure (e.g. not existing table)
*/
    const unsigned char *table;
    int ret;
    char sql[128];
    sqlite3_stmt *stmt;
    int exists = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CheckShadowedRowid() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_null (context);
	  return;
      }
    table = sqlite3_value_text (argv[0]);

/* checking if the table exists */
    strcpy (sql,
	    "SELECT name FROM sqlite_master WHERE type = 'table' AND Lower(name) = Lower(?)");
    ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("CheckShadowedRowid: \"%s\"\n",
			sqlite3_errmsg (sqlite));
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, (const char *) table,
		       strlen ((const char *) table), SQLITE_STATIC);
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	      exists = 1;
      }
    sqlite3_finalize (stmt);
    if (!exists)
	sqlite3_result_null (context);
    else
      {
	  if (!validateRowid (sqlite, (const char *) table))
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
}

static void
fnct_CheckWithoutRowid (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ CheckWithoutRowid(table)
/
/ checks if some table has been created WITHOUT ROWID
/ 1 - yes, the table is WITHOUT ROWID
/ 0 - no, the ROWID should be supported
/ NULL on failure (e.g. not existing table)
*/
    const unsigned char *table;
    int ret;
    char sql[128];
    sqlite3_stmt *stmt;
    int exists = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CheckWithoutRowid() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_null (context);
	  return;
      }
    table = sqlite3_value_text (argv[0]);

/* checking if the table exists */
    strcpy (sql,
	    "SELECT name FROM sqlite_master WHERE type = 'table' AND Lower(name) = Lower(?)");
    ret = sqlite3_prepare_v2 (sqlite, sql, strlen (sql), &stmt, NULL);
    if (ret != SQLITE_OK)
      {
	  spatialite_e ("CheckWithoutRowid: \"%s\"\n", sqlite3_errmsg (sqlite));
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_reset (stmt);
    sqlite3_clear_bindings (stmt);
    sqlite3_bind_text (stmt, 1, (const char *) table,
		       strlen ((const char *) table), SQLITE_STATIC);
    while (1)
      {
	  /* scrolling the result set rows */
	  ret = sqlite3_step (stmt);
	  if (ret == SQLITE_DONE)
	      break;		/* end of result set */
	  if (ret == SQLITE_ROW)
	      exists = 1;
      }
    sqlite3_finalize (stmt);
    if (!exists)
	sqlite3_result_null (context);
    else
      {
	  if (is_without_rowid_table (sqlite, (const char *) table))
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
}

static void
fnct_CreateTemporarySpatialIndex (sqlite3_context * context, int argc,
				  sqlite3_value ** argv)
{
/* SQL function:
/ CreateTemporarySpatialIndex(db_prefix, table, column )
/
/ creates a SpatialIndex based on Column and Table
/ returns 1 on success
/ 0 on failure
*/
    const char *db_prefix;
    const char *table;
    const char *column;
    char *sql_statement;
    char *prefix;
    char *errMsg = NULL;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateTemporarySpatialIndex() error: argument 1 [db-prefix] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    db_prefix = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateTemporarySpatialIndex() error: argument 2 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateTemporarySpatialIndex() error: argument 3 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[2]);
    if (is_without_rowid_table_attached (sqlite, db_prefix, table))
      {
	  spatialite_e
	      ("CreateTemporarySpatialIndex() error: table '%s' is WITHOUT ROWID\n",
	       table);
	  sqlite3_result_int (context, -1);
	  return;
      }
/* checking if the Attached Database is actually based on :memory: */
    if (!is_attached_memory (sqlite, db_prefix))
      {
	  spatialite_e
	      ("CreateTemporarySpatialIndex\n() error: Database '%s' does not exists or is not a Temporary one\n",
	       db_prefix);
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (!validateTemporaryRowid (sqlite, db_prefix, table))
      {
	  spatialite_e
	      ("CreateTemporarySpatialIndex() error: a physical column named ROWID shadows the real ROWID\n");
	  sqlite3_result_int (context, -1);
	  return;
      }
    prefix = gaiaDoubleQuotedSql (db_prefix);
    sql_statement =
	sqlite3_mprintf
	("UPDATE \"%s\".geometry_columns SET spatial_index_enabled = 1 "
	 "WHERE Upper(f_table_name) = Upper(%Q) AND "
	 "Upper(f_geometry_column) = Upper(%Q) AND spatial_index_enabled = 0",
	 prefix, table, column);
    free (prefix);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    if (sqlite3_changes (sqlite) == 0)
      {
	  spatialite_e
	      ("CreateTemporarySpatialIndex() error: either \"%s\".\"%s\" isn't a Geometry column or a SpatialIndex is already defined\n",
	       table, column);
	  sqlite3_result_int (context, 0);
	  return;
      }
    updateTemporaryGeometryTriggers (sqlite, db_prefix, table, column);
    sqlite3_result_int (context, 1);
    return;
  error:
    spatialite_e ("CreateTemporarySpatialIndex() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CreateSpatialIndex (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ CreateSpatialIndex(table, column )
/
/ creates a SpatialIndex based on Column and Table
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    char *sql_statement;
    char sql[1024];
    char *errMsg = NULL;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateSpatialIndex() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateSpatialIndex() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);
    if (is_without_rowid_table (sqlite, table))
      {
	  spatialite_e
	      ("CreateSpatialIndex() error: table '%s' is WITHOUT ROWID\n",
	       table);
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (!validateRowid (sqlite, table))
      {
	  spatialite_e
	      ("CreateSpatialIndex() error: a physical column named ROWID shadows the real ROWID\n");
	  sqlite3_result_int (context, -1);
	  return;
      }
    sql_statement =
	sqlite3_mprintf
	("UPDATE geometry_columns SET spatial_index_enabled = 1 "
	 "WHERE Upper(f_table_name) = Upper(%Q) AND "
	 "Upper(f_geometry_column) = Upper(%Q) AND spatial_index_enabled = 0",
	 table, column);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    if (sqlite3_changes (sqlite) == 0)
      {
	  spatialite_e
	      ("CreateSpatialIndex() error: either \"%s\".\"%s\" isn't a Geometry column or a SpatialIndex is already defined\n",
	       table, column);
	  sqlite3_result_int (context, 0);
	  return;
      }
    updateGeometryTriggers (sqlite, table, column);
    sqlite3_result_int (context, 1);
    strcpy (sql, "R*Tree Spatial Index successfully created");
    updateSpatiaLiteHistory (sqlite, table, column, sql);
    return;
  error:
    spatialite_e ("CreateSpatialIndex() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CreateMbrCache (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CreateMbrCache(table, column )
/
/ creates an MBR Cache based on Column and Table
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    char *sql_statement;
    char sql[1024];
    char *errMsg = NULL;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateMbrCache() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("CreateMbrCache() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);
    sql_statement =
	sqlite3_mprintf
	("UPDATE geometry_columns SET spatial_index_enabled = 2 "
	 "WHERE Upper(f_table_name) = Upper(%Q) "
	 "AND Upper(f_geometry_column) = Upper(%Q) AND spatial_index_enabled = 0",
	 table, column);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    if (sqlite3_changes (sqlite) == 0)
      {
	  spatialite_e
	      ("CreateMbrCache() error: either \"%s\".\"%s\" isn't a Geometry column or a SpatialIndex is already defined\n",
	       table, column);
	  sqlite3_result_int (context, 0);
	  return;
      }
    updateGeometryTriggers (sqlite, table, column);
    sqlite3_result_int (context, 1);
    strcpy (sql, "MbrCache successfully created");
    updateSpatiaLiteHistory (sqlite, table, column, sql);
    return;
  error:
    spatialite_e ("CreateMbrCache() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_DisableSpatialIndex (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ DisableSpatialIndex(table, column )
/
/ disables a SpatialIndex based on Column and Table
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    char sql[1024];
    char *sql_statement;
    char *errMsg = NULL;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("DisableSpatialIndex() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("DisableSpatialIndex() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);

    sql_statement =
	sqlite3_mprintf
	("UPDATE geometry_columns SET spatial_index_enabled = 0 "
	 "WHERE Upper(f_table_name) = Upper(%Q) AND "
	 "Upper(f_geometry_column) = Upper(%Q) AND spatial_index_enabled <> 0",
	 table, column);
    ret = sqlite3_exec (sqlite, sql_statement, NULL, NULL, &errMsg);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    if (sqlite3_changes (sqlite) == 0)
      {
	  spatialite_e
	      ("DisableSpatialIndex() error: either \"%s\".\"%s\" isn't a Geometry column or no SpatialIndex is defined\n",
	       table, column);
	  sqlite3_result_int (context, 0);
	  return;
      }
    updateGeometryTriggers (sqlite, table, column);
    sqlite3_result_int (context, 1);
    strcpy (sql, "SpatialIndex successfully disabled");
    updateSpatiaLiteHistory (sqlite, table, column, sql);
    return;
  error:
    spatialite_e ("DisableSpatialIndex() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_RebuildGeometryTriggers (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ RebuildGeometryTriggers(table, column )
/
/ rebuilds Geometry Triggers (constraints)  based on Column and Table
/ returns 1 on success
/ 0 on failure
*/
    const char *table;
    const char *column;
    char *sql_statement;
    char *errMsg = NULL;
    int ret;
    char **results;
    int rows;
    int columns;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RebuildGeometryTriggers() error: argument 1 [table_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    table = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  spatialite_e
	      ("RebuildGeometryTriggers() error: argument 2 [column_name] is not of the String type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    column = (const char *) sqlite3_value_text (argv[1]);
    sql_statement =
	sqlite3_mprintf ("SELECT f_table_name FROM geometry_columns "
			 "WHERE Upper(f_table_name) = Upper(%Q) AND Upper(f_geometry_column) = Upper (%Q)",
			 table, column);
    ret = sqlite3_get_table (sqlite, sql_statement, &results, &rows, &columns,
			     NULL);
    sqlite3_free (sql_statement);
    if (ret != SQLITE_OK)
	goto error;
    sqlite3_free_table (results);
    if (rows <= 0)
      {
	  spatialite_e
	      ("RebuildGeometryTriggers() error: \"%s\".\"%s\" isn't a Geometry column\n",
	       table, column);
	  sqlite3_result_int (context, 0);
	  return;
      }
    updateGeometryTriggers (sqlite, table, column);
    sqlite3_result_int (context, 1);
    updateSpatiaLiteHistory (sqlite, table, column,
			     "Geometry Triggers successfully rebuilt");
    return;
  error:
    spatialite_e ("RebuildGeometryTriggers() error: \"%s\"\n", errMsg);
    sqlite3_free (errMsg);
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_UpdateLayerStatistics (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ UpdateLayerStatistics(table, column )
/
/ Updates LAYER_STATISTICS [based on Column and Table]
/ returns 1 on success
/ 0 on failure
*/
    const char *sql;
    const char *table = NULL;
    const char *column = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc >= 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("UpdateLayerStatistics() error: argument 1 [table_name] is not of the String type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  table = (const char *) sqlite3_value_text (argv[0]);
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("UpdateLayerStatistics() error: argument 2 [column_name] is not of the String type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  column = (const char *) sqlite3_value_text (argv[1]);
      }
    if (!update_layer_statistics (sqlite, table, column))
	goto error;
    sqlite3_result_int (context, 1);
    sql = "UpdateLayerStatistics";
    if (table == NULL)
	table = "ALL-TABLES";
    if (column == NULL)
	column = "ALL-GEOMETRY-COLUMNS";
    updateSpatiaLiteHistory (sqlite, (const char *) table,
			     (const char *) column, sql);
    return;
  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_UpgradeGeometryTriggers (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ UpgradeGeometryTriggers(transaction TRUE|FALSE)
/
/ Upgrades (reinstalls) all Geometry Triggers - requires a DB > 4.0.0
/ returns 1 on success
/ 0 on failure (NULL on invalid args)
*/
    char *errMsg = NULL;
    int ret;
    int transaction = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("UpgradeGeometryTriggers() error: argument 1 [TRANSACTION] is not of the Integer type\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (checkSpatialMetaData (sqlite) < 3)
      {
	  spatialite_e
	      ("UpgradeGeometryTriggers() error: invalid DB Layout (< v.4.0.0)\n");
	  sqlite3_result_int (context, 0);
	  return;
      }
    transaction = sqlite3_value_int (argv[0]);
    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    if (!upgradeGeometryTriggers (sqlite))
	goto error;
    if (transaction)
      {
	  /* committing the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    updateSpatiaLiteHistory (sqlite, "ALL-TABLES", NULL,
			     "Upgraded Geometry Triggers");
    sqlite3_result_int (context, 1);
    return;

  error:
    if (transaction)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      sqlite3_free (errMsg);
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_GetLayerExtent (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GetLayerExtent(table)
/ GetLayerExtent(table, column )
/ GetLayerExtent(table, column, pessimistic )
/
/ Return a Geometry (Envelope) corresponding to the full layer
/ extent [eventually updating the supporting statistics
/ NULL on failure
*/
    const char *table = NULL;
    const char *column = NULL;
    int pessimistic = 0;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geom;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (argc >= 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("GetLayerExtent() error: argument 1 [table_name] is not of the String type\n");
		sqlite3_result_null (context);
		return;
	    }
	  table = (const char *) sqlite3_value_text (argv[0]);
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("GetLayerExtent() error: argument 2 [column_name] is not of the String type\n");
		sqlite3_result_null (context);
		return;
	    }
	  column = (const char *) sqlite3_value_text (argv[1]);
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
	    {
		spatialite_e
		    ("GetLayerExtent() error: argument 3 [OPTIMISTIC/PESSIMISTIC] is not of the Integer type\n");
		sqlite3_result_null (context);
		return;
	    }
	  pessimistic = sqlite3_value_int (argv[2]);
      }
    geom = gaiaGetLayerExtent (sqlite, table, column, pessimistic);
    if (!geom)
	goto error;
/* builds the BLOB geometry to be returned */
    gaiaToSpatiaLiteBlobWkbEx2 (geom, &p_result, &len, gpkg_mode, tiny_point);
    sqlite3_result_blob (context, p_result, len, free);
    gaiaFreeGeomColl (geom);
    return;
  error:
    sqlite3_result_null (context);
    return;
}

static void
fnct_InvalidateLayerStatistics (sqlite3_context * context, int argc,
				sqlite3_value ** argv)
{
/* SQL function:
/ InvalidateLayerStatistics(void)
/ InvalidateLayerStatistics(table)
/ InvalidateLayerStatistics(table, column )
/
/ Immediately and unconditionally invalidates Layer Statistics
/ returns 1 on success
/ 0 on failure
*/
    const char *sql;
    const char *table = NULL;
    const char *column = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc >= 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("InvalidateLayerStatistics() error: argument 1 [table_name] is not of the String type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  table = (const char *) sqlite3_value_text (argv[0]);
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
	    {
		spatialite_e
		    ("InvalidateLayerStatistics() error: argument 2 [column_name] is not of the String type\n");
		sqlite3_result_int (context, 0);
		return;
	    }
	  column = (const char *) sqlite3_value_text (argv[1]);
      }
    if (!gaiaStatisticsInvalidate (sqlite, table, column))
	goto error;
    sqlite3_result_int (context, 1);
    sql = "InvalidateLayerStatistics";
    if (table == NULL)
	table = "ALL-TABLES";
    if (column == NULL)
	column = "ALL-GEOMETRY-COLUMNS";
    updateSpatiaLiteHistory (sqlite, (const char *) table,
			     (const char *) column, sql);
    return;
  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CreateRasterCoveragesTable (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ CreateRasterCoveragesTable()
/
/ creates the main RasterCoverages table 
/ returns 1 on success
/ 0 on failure
*/
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (!createRasterCoveragesTable (sqlite))
	goto error;
    updateSpatiaLiteHistory (sqlite, "*** Raster Coverages ***", NULL,
			     "Main table successfully created");
    sqlite3_result_int (context, 1);
    return;

  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_ReCreateRasterCoveragesTriggers (sqlite3_context * context, int argc,
				      sqlite3_value ** argv)
{
/* SQL function:
/ ReCreateRasterCoveragesTriggers()
/
/ (re)creates the RasterCoverages triggers 
/ returns 1 on success
/ 0 on failure
*/
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (!reCreateRasterCoveragesTriggers (sqlite))
	goto error;
    updateSpatiaLiteHistory (sqlite, "*** Raster Coverages ***", NULL,
			     "Triggers successfully (re)created");
    sqlite3_result_int (context, 1);
    return;

  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CreateVectorCoveragesTables (sqlite3_context * context, int argc,
				  sqlite3_value ** argv)
{
/* SQL function:
/ CreateVectorCoveragesTables()
/
/ creates the main VectorCoverages table 
/ returns 1 on success
/ 0 on failure
*/
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (!createVectorCoveragesTable (sqlite))
	goto error;
    updateSpatiaLiteHistory (sqlite, "*** Vector Coverages ***", NULL,
			     "Main table successfully created");
    sqlite3_result_int (context, 1);
    return;

  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_ReCreateVectorCoveragesTriggers (sqlite3_context * context, int argc,
				      sqlite3_value ** argv)
{
/* SQL function:
/ ReCreateVectorCoveragesTriggers()
/
/ (re)creates the VectorCoverages triggers 
/ returns 1 on success
/ 0 on failure
*/
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (!reCreateVectorCoveragesTriggers (sqlite))
	goto error;
    updateSpatiaLiteHistory (sqlite, "*** Vector Coverages ***", NULL,
			     "Triggers successfully (re)created");
    sqlite3_result_int (context, 1);
    return;

  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_CreateWMSTables (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ WMS_CreateTables()
/
/ creates the WMS support tables
/ returns 1 on success
/ 0 on failure
*/
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (!createWMSTables (sqlite))
	goto error;
    updateSpatiaLiteHistory (sqlite, "*** WMS ***", NULL,
			     "Support tables successfully created");
    sqlite3_result_int (context, 1);
    return;

  error:
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_RegisterWMSGetCapabilities (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ WMS_RegisterGetCapabilities(Text url)
/   or
/ WMS_RegisterGetCapabilities(Text url, Text title,
/                        Text abstract)
/
/ inserts a WMS GetCapabilities
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *title = NULL;
    const char *abstract = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[2]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  title = (const char *) sqlite3_value_text (argv[1]);
	  abstract = (const char *) sqlite3_value_text (argv[2]);
      }
    ret = register_wms_getcapabilities (sqlite, url, title, abstract);
    sqlite3_result_int (context, ret);
}

static void
fnct_UnregisterWMSGetCapabilities (sqlite3_context * context, int argc,
				   sqlite3_value ** argv)
{
/* SQL function:
/ WMS_UnRegisterGetCapabilities(Text url)
/
/ deletes a WMS GetCapabilities (and all related children)
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    ret = unregister_wms_getcapabilities (sqlite, url);
    sqlite3_result_int (context, ret);
}

static void
fnct_SetWMSGetCapabilitiesInfos (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ WMS_SetGetCapabilitiesInfos(Text url, Text title,
/                        Text abstract)
/
/ updates the descriptive infos supporting a WMS GetCapabilities
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *title = NULL;
    const char *abstract = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[1]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    title = (const char *) sqlite3_value_text (argv[1]);
    abstract = (const char *) sqlite3_value_text (argv[2]);
    ret = set_wms_getcapabilities_infos (sqlite, url, title, abstract);
    sqlite3_result_int (context, ret);
}

static int
validate_wms_bgcolor (const char *bgcolor)
{
/* testing for a valid HexRGB color value */
    const char *p = bgcolor;
    int len = strlen (bgcolor);
    if (len != 6)
	return 0;
    while (*p != '\0')
      {
	  int ok = 0;
	  if (*p >= 'a' && *p <= 'f')
	      ok = 1;
	  if (*p >= 'A' && *p <= 'F')
	      ok = 1;
	  if (*p >= '0' && *p <= '9')
	      ok = 1;
	  if (!ok)
	      return 0;
	  p++;
      }
    return 1;
}

static void
fnct_RegisterWMSGetMap (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ WMS_RegisterGetMap(Text getcapabilitites_url, Text getmap_url,
/                    Text layer_name, Text version, Text ref_sys,
/                    Text image_format, Text style, Int transparent,
/                    Int flip_axes)
/   or
/ WMS_RegisterGetMap(Text getcapabilitites_url, Text getmap_url,
/                    Text layer_name, Text version, Text ref_sys,
/                    Text image_format, Text style, Int transparent,
/                    Int flip_axes,  Int tiled, Int cached, 
/                    Int tile_width, Int tile_height)
/   or
/ WMS_RegisterGetMap(Text getcapabilitites_url, Text getmap_url,
/                    Text layer_name, Text title, Text abstract,
/                    Text version, Text ref_sys, Text image_format,
/                    Text style, Int transparent, Int flip_axes,
/                    Int tiled, Int cached, Int tile_width, 
/                    Int tile_height, Text bgcolor, Int is_queryable,
/                    Text getfeatureinfo_url)
/   or
/ WMS_RegisterGetMap(Text getcapabilitites_url, Text getmap_url,
/                    Text layer_name, Text title, Text abstract,
/                    Text version, Text ref_sys, Text image_format,
/                    Text style, Int transparent, Int flip_axes,
/                    Int tiled, Int cached, Int tile_width, 
/                    Int tile_height, Text bgcolor, Int is_queryable,
/                    Text getfeatureinfo_url, int cascaded,
/                    Double min_scale_denominator, 
/                    Double max_scale_denominator)
/
/ inserts a WMS GetMap
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *getcapabilities_url;
    const char *getmap_url;
    const char *layer_name;
    const char *title = NULL;
    const char *abstract = NULL;
    const char *version;
    const char *ref_sys;
    const char *image_format;
    const char *style;
    int transparent;
    int flip_axes;
    int tiled = 0;
    int cached = 0;
    int tile_width = 512;
    int tile_height = 512;
    const char *bgcolor = NULL;
    int is_queryable = 0;
    const char *getfeatureinfo_url = NULL;
    int cascaded = -1;
    double min_scale = -1.0;
    double max_scale = -1.0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[1]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    getcapabilities_url = (const char *) sqlite3_value_text (argv[0]);
    getmap_url = (const char *) sqlite3_value_text (argv[1]);
    layer_name = (const char *) sqlite3_value_text (argv[2]);
    if (argc == 9 || argc == 13)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[4]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[5]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[6]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  version = (const char *) sqlite3_value_text (argv[3]);
	  ref_sys = (const char *) sqlite3_value_text (argv[4]);
	  image_format = (const char *) sqlite3_value_text (argv[5]);
	  style = (const char *) sqlite3_value_text (argv[6]);
	  if (sqlite3_value_type (argv[7]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[8]) != SQLITE_INTEGER)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  transparent = sqlite3_value_int (argv[7]);
	  flip_axes = sqlite3_value_int (argv[8]);
      }
    if (argc == 13)
      {
	  if (sqlite3_value_type (argv[9]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[10]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[11]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[12]) != SQLITE_INTEGER)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  tiled = sqlite3_value_int (argv[9]);
	  cached = sqlite3_value_int (argv[10]);
	  tile_width = sqlite3_value_int (argv[11]);
	  tile_height = sqlite3_value_int (argv[12]);
      }
    if (argc == 18)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[1]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[2]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[3]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[4]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[5]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[6]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[7]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[8]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  getcapabilities_url = (const char *) sqlite3_value_text (argv[0]);
	  getmap_url = (const char *) sqlite3_value_text (argv[1]);
	  layer_name = (const char *) sqlite3_value_text (argv[2]);
	  title = (const char *) sqlite3_value_text (argv[3]);
	  abstract = (const char *) sqlite3_value_text (argv[4]);
	  version = (const char *) sqlite3_value_text (argv[5]);
	  ref_sys = (const char *) sqlite3_value_text (argv[6]);
	  image_format = (const char *) sqlite3_value_text (argv[7]);
	  style = (const char *) sqlite3_value_text (argv[8]);
	  if (sqlite3_value_type (argv[9]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[10]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[11]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[12]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[13]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[14]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[16]) != SQLITE_INTEGER)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  transparent = sqlite3_value_int (argv[9]);
	  flip_axes = sqlite3_value_int (argv[10]);
	  tiled = sqlite3_value_int (argv[11]);
	  cached = sqlite3_value_int (argv[12]);
	  tile_width = sqlite3_value_int (argv[13]);
	  tile_height = sqlite3_value_int (argv[14]);
	  is_queryable = sqlite3_value_int (argv[16]);
	  if (sqlite3_value_type (argv[15]) == SQLITE_NULL)
	      bgcolor = NULL;
	  else if (sqlite3_value_type (argv[15]) == SQLITE_TEXT)
	    {
		bgcolor = (const char *) sqlite3_value_text (argv[15]);
		if (!validate_wms_bgcolor (bgcolor))
		  {
		      sqlite3_result_int (context, -1);
		      return;
		  }
	    }
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  if (sqlite3_value_type (argv[17]) == SQLITE_NULL)
	      getfeatureinfo_url = NULL;
	  else if (sqlite3_value_type (argv[17]) == SQLITE_TEXT)
	      getfeatureinfo_url = (const char *) sqlite3_value_text (argv[17]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    if (argc == 21)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[1]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[2]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[3]) != SQLITE_TEXT
	      || sqlite3_value_type (argv[4]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[5]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[6]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[7]) != SQLITE_TEXT ||
	      sqlite3_value_type (argv[8]) != SQLITE_TEXT)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  getcapabilities_url = (const char *) sqlite3_value_text (argv[0]);
	  getmap_url = (const char *) sqlite3_value_text (argv[1]);
	  layer_name = (const char *) sqlite3_value_text (argv[2]);
	  title = (const char *) sqlite3_value_text (argv[3]);
	  abstract = (const char *) sqlite3_value_text (argv[4]);
	  version = (const char *) sqlite3_value_text (argv[5]);
	  ref_sys = (const char *) sqlite3_value_text (argv[6]);
	  image_format = (const char *) sqlite3_value_text (argv[7]);
	  style = (const char *) sqlite3_value_text (argv[8]);
	  if (sqlite3_value_type (argv[9]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[10]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[11]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[12]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[13]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[14]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[16]) != SQLITE_INTEGER ||
	      (sqlite3_value_type (argv[18]) != SQLITE_INTEGER
	       && sqlite3_value_type (argv[18]) != SQLITE_NULL))
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  transparent = sqlite3_value_int (argv[9]);
	  flip_axes = sqlite3_value_int (argv[10]);
	  tiled = sqlite3_value_int (argv[11]);
	  cached = sqlite3_value_int (argv[12]);
	  tile_width = sqlite3_value_int (argv[13]);
	  tile_height = sqlite3_value_int (argv[14]);
	  is_queryable = sqlite3_value_int (argv[16]);
	  if (sqlite3_value_type (argv[15]) == SQLITE_NULL)
	      bgcolor = NULL;
	  else if (sqlite3_value_type (argv[15]) == SQLITE_TEXT)
	    {
		bgcolor = (const char *) sqlite3_value_text (argv[15]);
		if (!validate_wms_bgcolor (bgcolor))
		  {
		      sqlite3_result_int (context, -1);
		      return;
		  }
	    }
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  if (sqlite3_value_type (argv[17]) == SQLITE_NULL)
	      getfeatureinfo_url = NULL;
	  else if (sqlite3_value_type (argv[17]) == SQLITE_TEXT)
	      getfeatureinfo_url = (const char *) sqlite3_value_text (argv[17]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  if (sqlite3_value_type (argv[18]) == SQLITE_INTEGER)
	      cascaded = sqlite3_value_int (argv[18]);
	  if (sqlite3_value_type (argv[19]) == SQLITE_NULL)
	      ;
	  else if (sqlite3_value_type (argv[19]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[19]);
		min_scale = val;
	    }
	  else if (sqlite3_value_type (argv[19]) == SQLITE_FLOAT)
	      min_scale = sqlite3_value_double (argv[19]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  if (sqlite3_value_type (argv[20]) == SQLITE_NULL)
	      ;
	  else if (sqlite3_value_type (argv[20]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[20]);
		max_scale = val;
	    }
	  else if (sqlite3_value_type (argv[20]) == SQLITE_FLOAT)
	      max_scale = sqlite3_value_double (argv[20]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    ret =
	register_wms_getmap (sqlite, getcapabilities_url, getmap_url,
			     layer_name, title, abstract, version, ref_sys,
			     image_format, style, transparent, flip_axes,
			     tiled, cached, tile_width, tile_height, bgcolor,
			     is_queryable, getfeatureinfo_url, cascaded,
			     min_scale, max_scale);
    sqlite3_result_int (context, ret);
}

static void
fnct_UnregisterWMSGetMap (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ WMS_UnRegisterGetMap(Text url, Text layer_name)
/
/ deletes a WMS GetMap (and all related children)
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    ret = unregister_wms_getmap (sqlite, url, layer_name);
    sqlite3_result_int (context, ret);
}

static void
fnct_SetWMSGetMapInfos (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ WMS_SetGetMapInfos(Text url, Text layer_name, Text title, Text abstract)
/
/ updates the descriptive infos supporting a WMS GetMap
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *title;
    const char *abstract;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[1]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[2]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[3]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    title = (const char *) sqlite3_value_text (argv[2]);
    abstract = (const char *) sqlite3_value_text (argv[3]);
    ret = set_wms_getmap_infos (sqlite, url, layer_name, title, abstract);
    sqlite3_result_int (context, ret);
}

static void
fnct_SetWMSGetMapCopyright (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ WMS_SetGetMapCopyright(Text url, Text layer_name, Text copyright)
/    or
/ WMS_SetGetMapCopyright(Text url, Text layer_name, Text copyright,
/                        Text license)
/
/ updates copyright infos supporting a WMS GetMap
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *copyright = NULL;
    const char *license = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	copyright = (const char *) sqlite3_value_text (argv[2]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (argc >= 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_TEXT)
	      license = (const char *) sqlite3_value_text (argv[3]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    ret =
	set_wms_getmap_copyright (sqlite, url, layer_name, copyright, license);
    sqlite3_result_int (context, ret);
}

static void
fnct_SetWMSGetMapOptions (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ WMS_SetGetMapOptions(Text url, Text layer_name, Int transparent,
/                      Int flip_axes)
/   or
/ WMS_SetGetMapOptions(Text url, Text layer_name, Int tiled, Int cached,
/                      Int tile_width, Int tile_height)
/   or
/ WMS_SetGetMapOptions(Text url, Text layer_name, Int is_queryable,
/                      Text getfeatureinfo_url)
/   or
/ WMS_SetGetMapOptions(Text url, Text layer_name, Text bgcolor)
/
/ updates the options supporting a WMS GetMap
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    int transparent;
    int flip_axes;
    int is_queryable;
    int tiled;
    int cached;
    int tile_width = 512;
    int tile_height = 512;
    const char *getfeatureinfo_url = NULL;
    const char *bgcolor = NULL;
    char mode = '\0';
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT ||
	sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	    {
		mode = 'B';
		bgcolor = (const char *) sqlite3_value_text (argv[2]);
		if (!validate_wms_bgcolor (bgcolor))
		  {
		      sqlite3_result_int (context, -1);
		      return;
		  }
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_NULL)
	    {
		mode = 'B';
		bgcolor = NULL;
	    }
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER
	      && sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	    {
		mode = 'F';
		transparent = sqlite3_value_int (argv[2]);
		flip_axes = sqlite3_value_int (argv[3]);
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER
		   && sqlite3_value_type (argv[3]) == SQLITE_TEXT)
	    {
		mode = 'Q';
		is_queryable = sqlite3_value_int (argv[2]);
		getfeatureinfo_url =
		    (const char *) sqlite3_value_text (argv[3]);
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER
		   && sqlite3_value_type (argv[3]) == SQLITE_NULL)
	    {
		mode = 'Q';
		is_queryable = sqlite3_value_int (argv[2]);
		getfeatureinfo_url = NULL;
	    }
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    if (argc == 6)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[3]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[4]) != SQLITE_INTEGER ||
	      sqlite3_value_type (argv[5]) != SQLITE_INTEGER)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  mode = 'T';
	  tiled = sqlite3_value_int (argv[2]);
	  cached = sqlite3_value_int (argv[3]);
	  tile_width = sqlite3_value_int (argv[4]);
	  tile_height = sqlite3_value_int (argv[5]);
      }
    switch (mode)
      {
      case 'B':
	  ret = set_wms_getmap_bgcolor (sqlite, url, layer_name, bgcolor);
	  break;
      case 'F':
	  ret =
	      set_wms_getmap_options (sqlite, url, layer_name, transparent,
				      flip_axes);
	  break;
      case 'Q':
	  ret =
	      set_wms_getmap_queryable (sqlite, url, layer_name, is_queryable,
					getfeatureinfo_url);
	  break;
      case 'T':
	  ret =
	      set_wms_getmap_tiled (sqlite, url, layer_name, tiled, cached,
				    tile_width, tile_height);
	  break;
      default:
	  ret = -1;
      };
    sqlite3_result_int (context, ret);
}

static void
fnct_RegisterWMSSetting (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ WMS_RegisterSetting(Text url, Text layer_name, Text key, Text value)
/   or
/ WMS_RegisterSetting(Text url, Text layer_name, Text key, Text value,
/                     Int default)
/
/ inserts a WMS GetMap Setting
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *key;
    const char *value;
    int is_default = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    key = (const char *) sqlite3_value_text (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    value = (const char *) sqlite3_value_text (argv[3]);
    if (argc >= 5)
      {
	  if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  is_default = sqlite3_value_int (argv[4]);
      }
    ret =
	register_wms_setting (sqlite, url, layer_name, key, value, is_default);
    sqlite3_result_int (context, ret);
}

static void
fnct_UnregisterWMSSetting (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ WMS_UnRegisterSetting(Text url, Text layer_name, Text key, Text value)
/
/ deletes a WMS GetMap Setting
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *key;
    const char *value;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    key = (const char *) sqlite3_value_text (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    value = (const char *) sqlite3_value_text (argv[3]);
    ret = unregister_wms_setting (sqlite, url, layer_name, key, value);
    sqlite3_result_int (context, ret);
}

static void
fnct_DefaultWMSSetting (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ WMS_DefaultSetting(Text url, Text layer_name, Text key, Text value)
/
/ updates some GetMap default setting
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *key;
    const char *value;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[1]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[2]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[3]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    key = (const char *) sqlite3_value_text (argv[2]);
    value = (const char *) sqlite3_value_text (argv[3]);
    ret = set_wms_default_setting (sqlite, url, layer_name, key, value);
    sqlite3_result_int (context, ret);
}

static void
fnct_RegisterWMSStyle (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ WMS_RegisterStyle(Text url, Text layer_name, Text style_name,
/                   Text style_title, Text style_abstract)
/   or
/ WMS_RegisterStyle(Text url, Text layer_name, Text style_name,
/                   Text style_title, Text style_abstract,
/                   Int default)
/
/ inserts a WMS GetMap Style
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *style_name;
    const char *style_title;
    const char *style_abstract = NULL;
    int is_default = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }

    style_name = (const char *) sqlite3_value_text (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    style_title = (const char *) sqlite3_value_text (argv[3]);
    if (sqlite3_value_type (argv[4]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[4]) == SQLITE_TEXT)
	style_abstract = (const char *) sqlite3_value_text (argv[4]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (argc >= 6)
      {
	  if (sqlite3_value_type (argv[5]) != SQLITE_INTEGER)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  is_default = sqlite3_value_int (argv[5]);
      }
    ret =
	register_wms_style (sqlite, url, layer_name, style_name, style_title,
			    style_abstract, is_default);
    sqlite3_result_int (context, ret);
}

static void
fnct_RegisterWMSRefSys (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ WMS_RegisterRefSys(Text url, Text layer_name, Text ref_sys, Double minx,
/                    Double miny, Double maxx, Double maxy)
/   or
/ WMS_RegisterRefSys(Text url, Text layer_name, Text ref_sys, Double minx,
/                    Double miny, Double maxx, Double maxy, Int default)
/
/ inserts a WMS GetMap SRS
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *ref_sys;
    double minx;
    double miny;
    double maxx;
    double maxy;
    int is_default = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    ref_sys = (const char *) sqlite3_value_text (argv[2]);
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[3]);
	  minx = val;
      }
    else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	minx = sqlite3_value_double (argv[3]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[4]);
	  miny = val;
      }
    else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	miny = sqlite3_value_double (argv[4]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[5]);
	  maxx = val;
      }
    else if (sqlite3_value_type (argv[5]) == SQLITE_FLOAT)
	maxx = sqlite3_value_double (argv[5]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[6]);
	  maxy = val;
      }
    else if (sqlite3_value_type (argv[6]) == SQLITE_FLOAT)
	maxy = sqlite3_value_double (argv[6]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (argc >= 8)
      {
	  if (sqlite3_value_type (argv[7]) != SQLITE_INTEGER)
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
	  is_default = sqlite3_value_int (argv[7]);
      }
    ret =
	register_wms_srs (sqlite, url, layer_name, ref_sys, minx, miny, maxx,
			  maxy, is_default);
    sqlite3_result_int (context, ret);
}

static void
fnct_UnregisterWMSRefSys (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ WMS_UnRegisterRefSys(Text url, Text layer_name, Text ref_sys)
/
/ deletes a WMS GetMap SRS
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *ref_sys;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    ref_sys = (const char *) sqlite3_value_text (argv[2]);
    ret = unregister_wms_srs (sqlite, url, layer_name, ref_sys);
    sqlite3_result_int (context, ret);
}

static void
fnct_DefaultWMSRefSys (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ WMS_DefaultRefSys(Text url, Text layer_name, Text ref_sys)
/
/ updates some GetMap default SRS
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *url;
    const char *layer_name;
    const char *ref_sys;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[1]) != SQLITE_TEXT
	|| sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    url = (const char *) sqlite3_value_text (argv[0]);
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    ref_sys = (const char *) sqlite3_value_text (argv[2]);
    ret = set_wms_default_srs (sqlite, url, layer_name, ref_sys);
    sqlite3_result_int (context, ret);
}

static void
fnct_WMSGetMapRequestURL (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ WMS_GetMapRequestURL(Text getmap_url, Text layer_name, Int width,
/                      Int height, Double minx, Double miny,
/                      Double maxx, Double maxy)
/
/ returns a WMS GetMap request URL on success
/ NULL on invalid arguments
*/
    char *url;
    const char *getmap_url;
    const char *layer_name;
    int width;
    int height;
    double minx;
    double miny;
    double maxx;
    double maxy;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    getmap_url = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    width = sqlite3_value_int (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    height = sqlite3_value_int (argv[3]);
    if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	minx = sqlite3_value_double (argv[4]);
    else if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[4]);
	  minx = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[5]) == SQLITE_FLOAT)
	miny = sqlite3_value_double (argv[5]);
    else if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[5]);
	  miny = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[6]) == SQLITE_FLOAT)
	maxx = sqlite3_value_double (argv[6]);
    else if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[6]);
	  maxx = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[7]) == SQLITE_FLOAT)
	maxy = sqlite3_value_double (argv[7]);
    else if (sqlite3_value_type (argv[7]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[7]);
	  maxy = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    url =
	wms_getmap_request_url (sqlite, getmap_url, layer_name, width, height,
				minx, miny, maxx, maxy);
    if (url == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, url, strlen (url), sqlite3_free);
}

static void
fnct_WMSGetFeatureInfoRequestURL (sqlite3_context * context, int argc,
				  sqlite3_value ** argv)
{
/* SQL function:
/ WMS_GetFeatureInfoRequestURL(Text getmap_url, Text layer_name, Int width,
/                              Int height, int x, int y, Double minx,
/                              Double miny, Double maxx, Double maxy)
/   or
/ WMS_GetFeatureInfoRequestURL(Text getmap_url, Text layer_name, Int width,
/                              Int height, int x, int y, Double minx,
/                              Double miny, Double maxx, Double maxy,
/                              Int feature_count )
/
/ returns a WMS GetFeatureInfo request URL on success
/ NULL on invalid arguments
*/
    char *url;
    const char *getmap_url;
    const char *layer_name;
    int width;
    int height;
    int x;
    int y;
    double minx;
    double miny;
    double maxx;
    double maxy;
    int feature_count = 1;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    getmap_url = (const char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    layer_name = (const char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    width = sqlite3_value_int (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    height = sqlite3_value_int (argv[3]);
    if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = sqlite3_value_int (argv[4]);
    if (sqlite3_value_type (argv[5]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    y = sqlite3_value_int (argv[5]);
    if (sqlite3_value_type (argv[6]) == SQLITE_FLOAT)
	minx = sqlite3_value_double (argv[6]);
    else if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[6]);
	  minx = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[7]) == SQLITE_FLOAT)
	miny = sqlite3_value_double (argv[7]);
    else if (sqlite3_value_type (argv[7]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[7]);
	  miny = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[8]) == SQLITE_FLOAT)
	maxx = sqlite3_value_double (argv[8]);
    else if (sqlite3_value_type (argv[8]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[8]);
	  maxx = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[9]) == SQLITE_FLOAT)
	maxy = sqlite3_value_double (argv[9]);
    else if (sqlite3_value_type (argv[9]) == SQLITE_INTEGER)
      {
	  int val = sqlite3_value_int (argv[9]);
	  maxy = val;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 11)
      {
	  if (sqlite3_value_type (argv[10]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  feature_count = sqlite3_value_int (argv[10]);
      }
    url =
	wms_getfeatureinfo_request_url (sqlite, getmap_url, layer_name, width,
					height, x, y, minx, miny, maxx, maxy,
					feature_count);
    if (url == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, url, strlen (url), sqlite3_free);
}

static void
fnct_RegisterDataLicense (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ RegisterDataLicense(Text license_name)
/    or
/ RegisterDataLicense(Text license_name, Text license_url)
/
/ inserts a Data License
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *license_name;
    const char *url = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    license_name = (const char *) sqlite3_value_text (argv[0]);
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	      url = (const char *) sqlite3_value_text (argv[1]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    ret = register_data_license (sqlite, license_name, url);
    sqlite3_result_int (context, ret);
}

static void
fnct_UnRegisterDataLicense (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ UnRegisterDataLicense(Text license_name)
/
/ deletes a Data License
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *license_name;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    license_name = (const char *) sqlite3_value_text (argv[0]);
    ret = unregister_data_license (sqlite, license_name);
    sqlite3_result_int (context, ret);
}

static void
fnct_RenameDataLicense (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ RenameDataLicense(Text old_name, Text new_name)
/
/ renames an existing Data License
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *old_name;
    const char *new_name;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT ||
	sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    old_name = (const char *) sqlite3_value_text (argv[0]);
    new_name = (const char *) sqlite3_value_text (argv[1]);
    ret = rename_data_license (sqlite, old_name, new_name);
    sqlite3_result_int (context, ret);
}

static void
fnct_SetDataLicenseUrl (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ SetDataLicenseUrl(Text license_name, Text license_url)
/
/ updates a Data License URL
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    int ret;
    const char *license_name;
    const char *url;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT ||
	sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    license_name = (const char *) sqlite3_value_text (argv[0]);
    url = (const char *) sqlite3_value_text (argv[1]);
    ret = set_data_license_url (sqlite, license_name, url);
    sqlite3_result_int (context, ret);
}

static void
fnct_CreateMetaCatalogTables (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ CreateMetaCatalogTables(transaction TRUE|FALSE)
/
/ creates (or re-creates) both "splite_metacatalog"
/ and "splite_metacatalog_statistics" tables
/ returns 1 on success
/ 0 on failure
*/
    char *errMsg = NULL;
    int ret;
    int transaction = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("CreateMetaCatalogTables() error: argument 1 [TRANSACTION] is not of the Integer type\n");
	  sqlite3_result_null (context);
	  return;
      }
    transaction = sqlite3_value_int (argv[0]);
    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    if (!gaiaCreateMetaCatalogTables (sqlite))
	goto error;
    if (transaction)
      {
	  /* committing the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    updateSpatiaLiteHistory (sqlite, "*** MetaCatalog ***", NULL,
			     "Tables successfully created and initialized");
    sqlite3_result_int (context, 1);
    return;

  error:
    if (transaction)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      sqlite3_free (errMsg);
      }
    sqlite3_result_int (context, 0);
    return;
}

static void
fnct_UpdateMetaCatalogStatistics (sqlite3_context * context, int argc,
				  sqlite3_value ** argv)
{
/* SQL function:
/ UpdateMetaCatalogStatistics(transaction TRUE|FALSE, table, column)
/ UpdateMetaCatalogStatistics(transaction TRUE|FALSE, master_table, table_name, column_name)
/
/ updates the MetaCatalog statistics
/ returns 1 on success
/ 0 on failure
*/
    char *errMsg = NULL;
    int ret;
    int transaction = 0;
    const char *master_table = NULL;
    const char *table = NULL;
    const char *column = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("UpdateMetaCatalogStatistics() error: argument 1 [TRANSACTION] is not of the Integer type\n");
	  sqlite3_result_null (context);
	  return;
      }
    transaction = sqlite3_value_int (argv[0]);
    if (argc == 3)
      {
	  /* table & column mode */
	  if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	      table = (const char *) sqlite3_value_text (argv[1]);
	  else
	    {
		spatialite_e
		    ("UpdateMetaCatalogStatistics() error: argument 2 [TABLE_NAME] is not of the Text type\n");
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	      column = (const char *) sqlite3_value_text (argv[2]);
	  else
	    {
		spatialite_e
		    ("UpdateMetaCatalogStatistics() error: argument 2 [COLUMN_NAME] is not of the Text type\n");
		sqlite3_result_null (context);
		return;
	    }
      }
    else
      {
	  /* master-table & table_name & column_name mode */
	  if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	      master_table = (const char *) sqlite3_value_text (argv[1]);
	  else
	    {
		spatialite_e
		    ("UpdateMetaCatalogStatistics() error: argument 2 [MASTER_TABLE] is not of the Text type\n");
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	      table = (const char *) sqlite3_value_text (argv[2]);
	  else
	    {
		spatialite_e
		    ("UpdateMetaCatalogStatistics() error: argument 3 [TABLE_NAME] is not of the Text type\n");
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[3]) == SQLITE_TEXT)
	      column = (const char *) sqlite3_value_text (argv[3]);
	  else
	    {
		spatialite_e
		    ("UpdateMetaCatalogStatistics() error: argument 3 [COLUMN_NAME] is not of the Text type\n");
		sqlite3_result_null (context);
		return;
	    }
      }
    if (transaction)
      {
	  /* starting a Transaction */
	  ret = sqlite3_exec (sqlite, "BEGIN", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    if (master_table != NULL)
      {
	  if (!gaiaUpdateMetaCatalogStatisticsFromMaster
	      (sqlite, master_table, table, column))
	      goto error;
      }
    else
      {
	  if (!gaiaUpdateMetaCatalogStatistics (sqlite, table, column))
	      goto error;
      }
    if (transaction)
      {
	  /* committing the still pending Transaction */
	  ret = sqlite3_exec (sqlite, "COMMIT", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      goto error;
      }
    updateSpatiaLiteHistory (sqlite, "*** MetaCatalog ***", NULL,
			     "Statistics successfully updated");
    sqlite3_result_int (context, 1);
    return;

  error:
    if (transaction)
      {
	  /* performing a Rollback */
	  ret = sqlite3_exec (sqlite, "ROLLBACK", NULL, NULL, &errMsg);
	  if (ret != SQLITE_OK)
	      sqlite3_free (errMsg);
      }
    sqlite3_result_int (context, 0);
    return;
}

static gaiaPointPtr
simplePoint (gaiaGeomCollPtr geo)
{
/* helper function
/ if this GEOMETRY contains only one POINT, and no other elementary geometry
/ the POINT address will be returned
/ otherwise NULL will be returned
*/
    int cnt = 0;
    gaiaPointPtr point;
    gaiaPointPtr this_point = NULL;
    if (!geo)
	return NULL;
    if (geo->FirstLinestring || geo->FirstPolygon)
	return NULL;
    point = geo->FirstPoint;
    while (point)
      {
	  /* counting how many POINTs are there */
	  cnt++;
	  this_point = point;
	  point = point->Next;
      }
    if (cnt == 1 && this_point)
	return this_point;
    return NULL;
}

static gaiaLinestringPtr
simpleLinestring (gaiaGeomCollPtr geo)
{
/* helper function
/ if this GEOMETRY contains only one LINESTRING, and no other elementary geometry
/ the LINESTRING address will be returned
/ otherwise NULL will be returned
*/
    int cnt = 0;
    gaiaLinestringPtr line;
    gaiaLinestringPtr this_line = NULL;
    if (!geo)
	return NULL;
    if (geo->FirstPoint || geo->FirstPolygon)
	return NULL;
    line = geo->FirstLinestring;
    while (line)
      {
	  /* counting how many LINESTRINGs are there */
	  cnt++;
	  this_line = line;
	  line = line->Next;
      }
    if (cnt == 1 && this_line)
	return this_line;
    return NULL;
}

static gaiaPolygonPtr
simplePolygon (gaiaGeomCollPtr geo)
{
/* helper function
/ if this GEOMETRY contains only one POLYGON, and no other elementary geometry
/ the POLYGON address will be returned
/ otherwise NULL will be returned
*/
    int cnt = 0;
    gaiaPolygonPtr polyg;
    gaiaPolygonPtr this_polyg = NULL;
    if (!geo)
	return NULL;
    if (geo->FirstPoint || geo->FirstLinestring)
	return NULL;
    polyg = geo->FirstPolygon;
    while (polyg)
      {
	  /* counting how many POLYGONs are there */
	  cnt++;
	  this_polyg = polyg;
	  polyg = polyg->Next;
      }
    if (cnt == 1 && this_polyg)
	return this_polyg;
    return NULL;
}

static void
fnct_AsText (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsText(BLOB encoded geometry)
/
/ returns the corresponding WKT encoded value
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    int decimal_precision = -1;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  decimal_precision = cache->decimal_precision;
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    gaiaOutBufferInitialize (&out_buf);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (decimal_precision >= 0)
	      gaiaOutWktEx (&out_buf, geo, decimal_precision);
	  else
	      gaiaOutWkt (&out_buf, geo);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_AsWkt (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsWkt(BLOB encoded geometry [, Integer precision])
/
/ returns the corresponding WKT encoded value
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    int precision = 15;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      precision = sqlite3_value_int (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    gaiaOutBufferInitialize (&out_buf);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaOutWktStrict (&out_buf, geo, precision);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

/*
/
/ AsSvg(geometry,[relative], [precision]) implementation
/
////////////////////////////////////////////////////////////
/
/ Author: Klaus Foerster klaus.foerster@svg.cc
/ version 0.9. 2008 September 21
 /
 */

static void
fnct_AsSvg (sqlite3_context * context, int argc, sqlite3_value ** argv,
	    int relative, int precision)
{
/* SQL function:
   AsSvg(BLOB encoded geometry, [int relative], [int precision])
   returns the corresponding SVG encoded value or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  /* make sure relative is 0 or 1 */
	  if (relative > 0)
	      relative = 1;
	  else
	      relative = 0;
	  /* make sure precision is between 0 and 15 - default to 6 if absent */
	  if (precision > GAIA_SVG_DEFAULT_MAX_PRECISION)
	      precision = GAIA_SVG_DEFAULT_MAX_PRECISION;
	  if (precision < 0)
	      precision = 0;
	  /* produce SVG-notation - actual work is done in gaiageo/gg_wkt.c */
	  gaiaOutBufferInitialize (&out_buf);
	  gaiaOutSvg (&out_buf, geo, relative, precision);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_AsSvg1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* called without additional arguments */
    fnct_AsSvg (context, argc, argv, GAIA_SVG_DEFAULT_RELATIVE,
		GAIA_SVG_DEFAULT_PRECISION);
}

static void
fnct_AsSvg2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* called with relative-switch */
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	fnct_AsSvg (context, argc, argv, sqlite3_value_int (argv[1]),
		    GAIA_SVG_DEFAULT_PRECISION);
    else
	sqlite3_result_null (context);
}

static void
fnct_AsSvg3 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* called with relative-switch and precision-argument */
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER
	&& sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	fnct_AsSvg (context, argc, argv, sqlite3_value_int (argv[1]),
		    sqlite3_value_int (argv[2]));
    else
	sqlite3_result_null (context);
}

/* END of Klaus Foerster AsSvg() implementation */


#ifndef OMIT_PROJ		/* PROJ.4 is strictly required to support KML */
static void
fnct_AsKml1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsKml(BLOB encoded geometry [, Integer precision])
/
/ returns the corresponding 'bare geom' KML representation 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geo_wgs84;
    char *proj_from = NULL;
    char *proj_to = NULL;
    int precision = 15;
    void *data = sqlite3_user_data (context);
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      precision = sqlite3_value_int (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    gaiaOutBufferInitialize (&out_buf);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (geo->Srid == 4326)
	      ;			/* already WGS84 */
	  else if (geo->Srid <= 0)
	    {
		/* unknown SRID: giving up */
		sqlite3_result_null (context);
		goto stop;
	    }
	  else
	    {
		/* attempting to reproject into WGS84 */
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
		getProjAuthNameSrid (sqlite, geo->Srid, &proj_from);
		getProjAuthNameSrid (sqlite, 4326, &proj_to);
#else /* supporting old PROJ.4 */
		getProjParams (sqlite, geo->Srid, &proj_from);
		getProjParams (sqlite, 4326, &proj_to);
#endif
		if (proj_to == NULL || proj_from == NULL)
		  {
		      if (proj_from)
			  free (proj_from);
		      if (proj_to)
			  free (proj_to);
		      sqlite3_result_null (context);
		      goto stop;
		  }
		if (data != NULL)
		    geo_wgs84 = gaiaTransform_r (data, geo, proj_from, proj_to);
		else
		    geo_wgs84 = gaiaTransform (geo, proj_from, proj_to);
		free (proj_from);
		free (proj_to);
		if (!geo_wgs84)
		  {
		      sqlite3_result_null (context);
		      goto stop;
		  }
		/* ok, reprojection was successful */
		gaiaFreeGeomColl (geo);
		geo = geo_wgs84;
	    }
	  /* produce KML-notation - actual work is done in gaiageo/gg_wkt.c */
	  gaiaOutBareKml (&out_buf, geo, precision);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
  stop:
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_AsKml3 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsKml(Anything name, Anything description, BLOB encoded geometry [, Integer precision])
/
/ returns the corresponding 'full' KML representation 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geo_wgs84;
    sqlite3_int64 int_value;
    double dbl_value;
    const char *name;
    const char *desc;
    char *name_malloc = NULL;
    char *desc_malloc = NULL;
    char dummy[128];
    char *xdummy;
    char *proj_from = NULL;
    char *proj_to = NULL;
    int precision = 15;
    void *data = sqlite3_user_data (context);
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    switch (sqlite3_value_type (argv[0]))
      {
      case SQLITE_TEXT:
	  name = (const char *) sqlite3_value_text (argv[0]);
	  len = strlen (name);
	  name_malloc = malloc (len + 1);
	  strcpy (name_malloc, name);
	  name = name_malloc;
	  break;
      case SQLITE_INTEGER:
	  int_value = sqlite3_value_int64 (argv[0]);
	  sprintf (dummy, FRMT64, int_value);
	  len = strlen (dummy);
	  name_malloc = malloc (len + 1);
	  strcpy (name_malloc, dummy);
	  name = name_malloc;
	  break;
      case SQLITE_FLOAT:
	  dbl_value = sqlite3_value_double (argv[0]);
	  xdummy = sqlite3_mprintf ("%1.6f", dbl_value);
	  len = strlen (xdummy);
	  name_malloc = malloc (len + 1);
	  strcpy (name_malloc, xdummy);
	  sqlite3_free (xdummy);
	  name = name_malloc;
	  break;
      case SQLITE_BLOB:
	  name = "BLOB";
	  break;
      default:
	  name = "NULL";
	  break;
      };
    switch (sqlite3_value_type (argv[1]))
      {
      case SQLITE_TEXT:
	  desc = (const char *) sqlite3_value_text (argv[1]);
	  len = strlen (desc);
	  desc_malloc = malloc (len + 1);
	  strcpy (desc_malloc, desc);
	  desc = desc_malloc;
	  break;
      case SQLITE_INTEGER:
	  int_value = sqlite3_value_int64 (argv[1]);
	  sprintf (dummy, FRMT64, int_value);
	  len = strlen (dummy);
	  desc_malloc = malloc (len + 1);
	  strcpy (desc_malloc, dummy);
	  desc = desc_malloc;
	  break;
      case SQLITE_FLOAT:
	  dbl_value = sqlite3_value_double (argv[1]);
	  xdummy = sqlite3_mprintf ("%1.6f", dbl_value);
	  len = strlen (xdummy);
	  desc_malloc = malloc (len + 1);
	  strcpy (desc_malloc, xdummy);
	  sqlite3_free (xdummy);
	  desc = desc_malloc;
	  break;
      case SQLITE_BLOB:
	  desc = "BLOB";
	  break;
      default:
	  desc = "NULL";
	  break;
      };
    gaiaOutBufferInitialize (&out_buf);
    if (sqlite3_value_type (argv[2]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[2]);
    n_bytes = sqlite3_value_bytes (argv[2]);
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	      precision = sqlite3_value_int (argv[3]);
	  else
	    {
		sqlite3_result_null (context);
		goto stop;
	    }
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (geo->Srid == 4326)
	      ;			/* already WGS84 */
	  else if (geo->Srid == 0)
	    {
		/* unknown SRID: giving up */
		sqlite3_result_null (context);
		goto stop;
	    }
	  else
	    {
		/* attempting to reproject into WGS84 */
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
		getProjAuthNameSrid (sqlite, geo->Srid, &proj_from);
		getProjAuthNameSrid (sqlite, 4326, &proj_to);
#else /* supporting old PROJ.4 */
		getProjParams (sqlite, geo->Srid, &proj_from);
		getProjParams (sqlite, 4326, &proj_to);
#endif
		if (proj_to == NULL || proj_from == NULL)
		  {
		      if (proj_from != NULL)
			  free (proj_from);
		      if (proj_to != NULL)
			  free (proj_to);
		      sqlite3_result_null (context);
		      goto stop;
		  }
		if (data != NULL)
		    geo_wgs84 = gaiaTransform_r (data, geo, proj_from, proj_to);
		else
		    geo_wgs84 = gaiaTransform (geo, proj_from, proj_to);
		free (proj_from);
		free (proj_to);
		if (!geo_wgs84)
		  {
		      sqlite3_result_null (context);
		      goto stop;
		  }
		/* ok, reprojection was successful */
		gaiaFreeGeomColl (geo);
		geo = geo_wgs84;
	    }
	  /* produce KML-notation - actual work is done in gaiageo/gg_wkt.c */
	  gaiaOutFullKml (&out_buf, name, desc, geo, precision);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
  stop:
    gaiaFreeGeomColl (geo);
    if (name_malloc)
	free (name_malloc);
    if (desc_malloc)
	free (desc_malloc);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_AsKml (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsKml(Anything name, Anything description, BLOB encoded geometry)
/     or
/ AsKml(BLOB encoded geometry)
/
/ returns the corresponding KML representation 
/ or NULL if any error is encountered
*/
    if (argc == 3 || argc == 4)
	fnct_AsKml3 (context, argc, argv);
    else
	fnct_AsKml1 (context, argc, argv);
}
#endif /* end including PROJ.4 */

static void
fnct_AsGml (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsGml(BLOB encoded geometry)
/    or
/ AsGml(integer version, BLOB encoded geometry)
/    or
/ AsGml(integer version, BLOB encoded geometry, integer precision)
/
/ *version* may be 2 (GML 2.1.2) or 3 (GML 3.1.1)
/ default *version*: 2
/
/ *precision* is the number of output decimal digits
/ default *precision*: 15
/
/ returns the corresponding GML representation 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    int version = 2;
    int precision = 15;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	      version = sqlite3_value_int (argv[0]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
	  n_bytes = sqlite3_value_bytes (argv[1]);
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      precision = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    else if (argc == 2)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER
	      && sqlite3_value_type (argv[1]) == SQLITE_BLOB)
	    {
		version = sqlite3_value_int (argv[0]);
		p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
		n_bytes = sqlite3_value_bytes (argv[1]);
	    }
	  else if (sqlite3_value_type (argv[0]) == SQLITE_BLOB
		   && sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
		n_bytes = sqlite3_value_bytes (argv[0]);
		precision = sqlite3_value_int (argv[1]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    else
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
      }
    gaiaOutBufferInitialize (&out_buf);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  /* produce GML-notation - actual work is done in gaiageo/gg_wkt.c */
	  int flipped = 0;
	  if (version == 3)
	    {
		/* only if GML3 tests fow flipped axes */
		if (!srid_has_flipped_axes (sqlite, geo->Srid, &flipped))
		    flipped = 0;
	    }
	  gaiaOutGml_ex (&out_buf, version, flipped, precision, geo);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_AsGeoJSON (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsGeoJSON(BLOB encoded geometry)
/    or
/ AsGeoJSON(BLOB encoded geometry, integer precision)
/    or
/ AsGeoJSON(BLOB encoded geometry, integer precision, integer options)
/
/ *precision* is the number of output decimal digits
/ default *precision*: 15
/
/ *options* may be one of the followings:
/   0 = no options [default]
/   1 = GeoJSON MBR
/   2 = GeoJSON Short CRS (e.g EPSG:4326) 
/   3 = 1 + 2 (Mbr + shortCrs)
/   4 = GeoJSON Long CRS (e.g urn:ogc:def:crs:EPSG::4326)
/   5 = 1 + 4 (Mbr + longCrs)
/
/ returns the corresponding GML representation 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    int precision = 15;
    int options = 0;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	gpkg_amphibious = cache->gpkg_mode = cache->gpkg_mode;;
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_BLOB
	      && sqlite3_value_type (argv[1]) == SQLITE_INTEGER
	      && sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
		n_bytes = sqlite3_value_bytes (argv[0]);
		precision = sqlite3_value_int (argv[1]);
		options = sqlite3_value_int (argv[2]);
		if (options >= 1 && options <= 5)
		    ;
		else
		    options = 0;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    else if (argc == 2)
      {
	  if (sqlite3_value_type (argv[0]) == SQLITE_BLOB
	      && sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
		n_bytes = sqlite3_value_bytes (argv[0]);
		precision = sqlite3_value_int (argv[1]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    else
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
      }
    gaiaOutBufferInitialize (&out_buf);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  /* produce GeoJSON-notation - actual work is done in gaiageo/gg_wkt.c */
	  gaiaOutGeoJSON (&out_buf, geo, precision, options);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_AsBinary (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsBinary(BLOB encoded geometry)
/
/ returns the corresponding WKB encoded value
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaToWkb (geo, &p_result, &len);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_AsFGF (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsFGF(BLOB encoded geometry, int dims)
/
/ returns the corresponding FGF encoded value
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int coord_dims;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  spatialite_e
	      ("AsFGF() error: argument 2 [geom_coords] is not of the Integer type\n");
	  sqlite3_result_null (context);
	  return;
      }
    coord_dims = sqlite3_value_int (argv[1]);
    if (coord_dims
	== 0 || coord_dims == 1 || coord_dims == 2 || coord_dims == 3)
	;
    else
      {
	  spatialite_e
	      ("AsFGF() error: argument 2 [geom_coords] out of range [0,1,2,3]\n");
	  sqlite3_result_null (context);
	  return;
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaToFgf (geo, &p_result, &len, coord_dims);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_tiny_point_encode (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ TinyPointEncode(variable-type)
/
/ returns a BLOB TinyPoint if the received argument is a BLOB-GEOMETRY POINT
/ in any other case the received argument will be returned "as is"
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  int geom_point = 1;
	  const unsigned char *blob =
	      (const unsigned char *) sqlite3_value_blob (argv[0]);
	  int size = sqlite3_value_bytes (argv[0]);
	  if (size < 45)
	      geom_point = 0;
	  else
	    {
		int endian_arch = gaiaEndianArch ();
		int type;
		int little_endian = 0;
		if (*(blob + 0) != GAIA_MARK_START)
		    geom_point = 0;
		if (*(blob + (size - 1)) != GAIA_MARK_END)
		    geom_point = 0;
		if (*(blob + 38) != GAIA_MARK_MBR)
		    geom_point = 0;
		if (*(blob + 1) == GAIA_LITTLE_ENDIAN)
		    little_endian = 1;
		else if (*(blob + 1) == GAIA_BIG_ENDIAN)
		    ;
		else
		    geom_point = 0;
		type = gaiaImport32 (blob + 39, little_endian, endian_arch);
		if (type == GAIA_POINT || type == GAIA_POINTZ
		    || type == GAIA_POINTM || type == GAIA_POINTZM)
		    ;
		else
		    geom_point = 0;
	    }
	  if (geom_point)
	    {
		int endian_arch = gaiaEndianArch ();
		int type;
		int little_endian = 0;
		int srid;
		double x;
		double y;
		double z;
		double m;
		unsigned char *out;
		int out_sz;
		if (*(blob + 1) == GAIA_LITTLE_ENDIAN)
		    little_endian = 1;
		srid = gaiaImport32 (blob + 2, little_endian, endian_arch);
		type = gaiaImport32 (blob + 39, little_endian, endian_arch);
		x = gaiaImport64 (blob + 43, little_endian, endian_arch);
		y = gaiaImport64 (blob + 51, little_endian, endian_arch);
		switch (type)
		  {
		  case GAIA_POINT:
		      gaiaMakePointEx (1, x, y, srid, &out, &out_sz);
		      break;
		  case GAIA_POINTZ:
		      z = gaiaImport64 (blob + 59, little_endian, endian_arch);
		      gaiaMakePointZEx (1, x, y, z, srid, &out, &out_sz);
		      break;
		  case GAIA_POINTM:
		      m = gaiaImport64 (blob + 59, little_endian, endian_arch);
		      gaiaMakePointMEx (1, x, y, m, srid, &out, &out_sz);
		      break;
		  case GAIA_POINTZM:
		      z = gaiaImport64 (blob + 59, little_endian, endian_arch);
		      m = gaiaImport64 (blob + 67, little_endian, endian_arch);
		      gaiaMakePointZMEx (1, x, y, z, m, srid, &out, &out_sz);
		      break;
		  };
		sqlite3_result_blob (context, out, out_sz, free);
	    }
	  else
	      sqlite3_result_blob (context, blob, size, SQLITE_TRANSIENT);

      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	sqlite3_result_int (context, sqlite3_value_int (argv[0]));
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	sqlite3_result_double (context, sqlite3_value_double (argv[0]));
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	sqlite3_result_text (context,
			     (const char *) sqlite3_value_text (argv[0]),
			     sqlite3_value_bytes (argv[0]), SQLITE_TRANSIENT);
    else
	sqlite3_result_null (context);
}

static void
fnct_geometry_point_encode (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ GeometryPointEncode(variable-type)
/
/ returns a BLOB GEOMETRY if the received argument is a BLOB-TinyPoint POINT
/ in any other case the received argument will be returned "as is"
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  int tiny_point = 1;
	  const unsigned char *blob =
	      (const unsigned char *) sqlite3_value_blob (argv[0]);
	  int size = sqlite3_value_bytes (argv[0]);
	  if (size < 24)
	      tiny_point = 0;
	  else
	    {
		if (size == 24 || size == 32 || size == 40)
		    ;
		else
		    tiny_point = 0;
		if (*(blob + 0) != GAIA_MARK_START)
		    tiny_point = 0;
		if (*(blob + 1) == GAIA_TINYPOINT_LITTLE_ENDIAN
		    || *(blob + 1) == GAIA_TINYPOINT_BIG_ENDIAN)
		    ;
		else
		    tiny_point = 0;
		if (*(blob + 6) == GAIA_TINYPOINT_XY
		    || *(blob + 6) == GAIA_TINYPOINT_XYZ
		    || *(blob + 6) == GAIA_TINYPOINT_XYM
		    || *(blob + 6) == GAIA_TINYPOINT_XYZM)
		    ;
		else
		    tiny_point = 0;
		if (*(blob + (size - 1)) != GAIA_MARK_END)
		    tiny_point = 0;
	    }
	  if (tiny_point)
	    {
		int endian_arch = gaiaEndianArch ();
		int type = *(blob + 6);
		int little_endian = 0;
		int srid;
		double x;
		double y;
		double z;
		double m;
		unsigned char *out;
		int out_sz;
		if (*(blob + 1) == GAIA_TINYPOINT_LITTLE_ENDIAN)
		    little_endian = 1;
		srid = gaiaImport32 (blob + 2, little_endian, endian_arch);
		x = gaiaImport64 (blob + 7, little_endian, endian_arch);
		y = gaiaImport64 (blob + 15, little_endian, endian_arch);
		switch (type)
		  {
		  case GAIA_TINYPOINT_XY:
		      gaiaMakePointEx (0, x, y, srid, &out, &out_sz);
		      break;
		  case GAIA_TINYPOINT_XYZ:
		      z = gaiaImport64 (blob + 23, little_endian, endian_arch);
		      gaiaMakePointZEx (0, x, y, z, srid, &out, &out_sz);
		      break;
		  case GAIA_TINYPOINT_XYM:
		      m = gaiaImport64 (blob + 23, little_endian, endian_arch);
		      gaiaMakePointMEx (0, x, y, m, srid, &out, &out_sz);
		      break;
		  case GAIA_TINYPOINT_XYZM:
		      z = gaiaImport64 (blob + 23, little_endian, endian_arch);
		      m = gaiaImport64 (blob + 31, little_endian, endian_arch);
		      gaiaMakePointZMEx (0, x, y, z, m, srid, &out, &out_sz);
		      break;
		  };
		sqlite3_result_blob (context, out, out_sz, free);
	    }
	  else
	      sqlite3_result_blob (context, blob, size, SQLITE_TRANSIENT);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
	sqlite3_result_int (context, sqlite3_value_int (argv[0]));
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	sqlite3_result_double (context, sqlite3_value_double (argv[0]));
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	sqlite3_result_text (context,
			     (const char *) sqlite3_value_text (argv[0]),
			     sqlite3_value_bytes (argv[0]), SQLITE_TRANSIENT);
    else
	sqlite3_result_null (context);
}

static void
fnct_MakePoint1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePoint(double X, double Y)
/    alias
/ ST_Point(double X, double Y)
/
/ builds a POINT 
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointEx (tiny_point, x, y, 0, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_MakePoint2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePoint(double X, double Y, int SRID)
/
/ builds a POINT 
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    int srid;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointEx (tiny_point, x, y, srid, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_MakePointZ1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePointZ(double X, double Y, double Z)
/
/ builds a POINT Z 
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double z;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	z = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  z = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointZEx (tiny_point, x, y, z, 0, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_MakePointZ2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePointZ(double X, double Y, double Z, int SRID)
/
/ builds a POINT Z
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double z;
    int srid;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	z = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  z = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointZEx (tiny_point, x, y, z, srid, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_MakePointM1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePointM(double X, double Y, double M)
/
/ builds a POINT M
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double m;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	m = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  m = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointMEx (tiny_point, x, y, m, 0, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_MakePointM2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePointM(double X, double Y, double M, int SRID)
/
/ builds a POINT M
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double m;
    int srid;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	m = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  m = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointMEx (tiny_point, x, y, m, srid, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_MakePointZM1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePointZM(double X, double Y, double Z, double M)
/
/ builds a POINT ZM 
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double z;
    double m;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	z = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  z = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	m = sqlite3_value_double (argv[3]);
    else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[3]);
	  m = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointZMEx (tiny_point, x, y, z, m, 0, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_MakePointZM2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakePointZM(double X, double Y, double Z, double M, int SRID)
/
/ builds a POINT 
/ or NULL if any error is encountered
*/
    int len;
    int int_value;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double z;
    double m;
    int srid;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	tiny_point = cache->tinyPointEnabled;
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	z = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  z = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	m = sqlite3_value_double (argv[3]);
    else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[3]);
	  m = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[4]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaMakePointZMEx (tiny_point, x, y, z, m, srid, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
addGeomPointToDynamicLine (gaiaDynamicLinePtr dyn, gaiaGeomCollPtr geom)
{
/* appending a simple-Point Geometry to a Dynamic Line */
    int pts;
    int lns;
    int pgs;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;

    if (dyn == NULL)
	return;
    if (dyn->Error)
	return;
/* checking if GEOM simply is a POINT */
    if (geom == NULL)
      {
	  dyn->Error = 1;
	  return;
      }
    pts = 0;
    lns = 0;
    pgs = 0;
    pt = geom->FirstPoint;
    while (pt)
      {
	  pts++;
	  pt = pt->Next;
      }
    ln = geom->FirstLinestring;
    while (ln)
      {
	  lns++;
	  ln = ln->Next;
      }
    pg = geom->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pts == 1 && lns == 0 && pgs == 0)
	;
    else
      {
	  /* failure: not a simple POINT */
	  dyn->Error = 1;
	  return;
      }

    if (dyn->Srid != geom->Srid)
      {
	  /* failure: SRID mismatch */
	  dyn->Error = 1;
	  return;
      }

    switch (geom->FirstPoint->DimensionModel)
      {
      case GAIA_XY_Z_M:
	  gaiaAppendPointZMToDynamicLine (dyn, geom->FirstPoint->X,
					  geom->FirstPoint->Y,
					  geom->FirstPoint->Z,
					  geom->FirstPoint->M);
	  break;
      case GAIA_XY_Z:
	  gaiaAppendPointZToDynamicLine (dyn, geom->FirstPoint->X,
					 geom->FirstPoint->Y,
					 geom->FirstPoint->Z);
	  break;
      case GAIA_XY_M:
	  gaiaAppendPointMToDynamicLine (dyn, geom->FirstPoint->X,
					 geom->FirstPoint->Y,
					 geom->FirstPoint->M);
	  break;
      default:
	  gaiaAppendPointToDynamicLine (dyn, geom->FirstPoint->X,
					geom->FirstPoint->Y);
	  break;
      }
}

static void
fnct_MakeLine_step (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakeLine(BLOBencoded geom)
/
/ aggregate function - STEP
/
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    gaiaDynamicLinePtr *p;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	return;
    p = sqlite3_aggregate_context (context, sizeof (gaiaDynamicLinePtr));
    if (!(*p))
      {
	  /* this is the first row */
	  *p = gaiaAllocDynamicLine ();
	  (*p)->Srid = geom->Srid;
	  addGeomPointToDynamicLine (*p, geom);
	  gaiaFreeGeomColl (geom);
      }
    else
      {
	  /* subsequent rows */
	  addGeomPointToDynamicLine (*p, geom);
	  gaiaFreeGeomColl (geom);
      }
}

static gaiaGeomCollPtr
geomFromDynamicLine (gaiaDynamicLinePtr dyn)
{
/* attempting to build a Geometry from a Dynamic Line */
    gaiaGeomCollPtr geom = NULL;
    gaiaLinestringPtr ln = NULL;
    gaiaPointPtr pt;
    int iv;
    int count = 0;
    int dims = GAIA_XY;

    if (dyn == NULL)
	return NULL;
    if (dyn->Error)
	return NULL;

    pt = dyn->First;
    while (pt)
      {
	  /* counting points and checking dims */
	  count++;
	  if (dims == GAIA_XY && pt->DimensionModel != GAIA_XY)
	      dims = pt->DimensionModel;
	  if (dims == GAIA_XY_Z
	      && (pt->DimensionModel == GAIA_XY_M
		  || pt->DimensionModel == GAIA_XY_Z_M))
	      dims = GAIA_XY_Z_M;
	  if (dims == GAIA_XY_M
	      && (pt->DimensionModel == GAIA_XY_Z
		  || pt->DimensionModel == GAIA_XY_Z_M))
	      dims = GAIA_XY_Z_M;
	  pt = pt->Next;
      }
    if (count < 2)
	return NULL;

    switch (dims)
      {
      case GAIA_XY_Z_M:
	  geom = gaiaAllocGeomCollXYZM ();
	  ln = gaiaAllocLinestringXYZM (count);
	  break;
      case GAIA_XY_Z:
	  geom = gaiaAllocGeomCollXYZ ();
	  ln = gaiaAllocLinestringXYZ (count);
	  break;
      case GAIA_XY_M:
	  geom = gaiaAllocGeomCollXYM ();
	  ln = gaiaAllocLinestringXYM (count);
	  break;
      default:
	  geom = gaiaAllocGeomColl ();
	  ln = gaiaAllocLinestring (count);
	  break;
      };

    if (geom != NULL && ln != NULL)
      {
	  gaiaInsertLinestringInGeomColl (geom, ln);
	  geom->Srid = dyn->Srid;
      }
    else
      {
	  if (geom)
	      gaiaFreeGeomColl (geom);
	  if (ln)
	      gaiaFreeLinestring (ln);
	  return NULL;
      }

    iv = 0;
    pt = dyn->First;
    while (pt)
      {
	  /* setting linestring points */
	  if (dims == GAIA_XY_Z_M)
	    {
		gaiaSetPointXYZM (ln->Coords, iv, pt->X, pt->Y, pt->Z, pt->M);
	    }
	  else if (dims == GAIA_XY_Z)
	    {
		gaiaSetPointXYZ (ln->Coords, iv, pt->X, pt->Y, pt->Z);
	    }
	  else if (dims == GAIA_XY_M)
	    {
		gaiaSetPointXYM (ln->Coords, iv, pt->X, pt->Y, pt->M);
	    }
	  else
	    {
		gaiaSetPoint (ln->Coords, iv, pt->X, pt->Y);
	    }
	  iv++;
	  pt = pt->Next;
      }
    return geom;
}

static void
fnct_MakeLine_final (sqlite3_context * context)
{
/* SQL function:
/ MakeLine(BLOBencoded geom)
/
/ aggregate function - FINAL
/
*/
    gaiaGeomCollPtr result;
    gaiaDynamicLinePtr *p = sqlite3_aggregate_context (context, 0);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    result = geomFromDynamicLine (*p);
    gaiaFreeDynamicLine (*p);
    if (!result)
	sqlite3_result_null (context);
    else
      {
	  /* builds the BLOB geometry to be returned */
	  int len;
	  unsigned char *p_result = NULL;
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (result);
      }
}

static void
buildLineFromMultiPoint (sqlite3_context * context, gaiaGeomCollPtr geom,
			 int direction)
{
/* internal: building a Linestring from a MultiPolygon */
    gaiaGeomCollPtr result;
    gaiaDynamicLinePtr dyn;
    int n_pts = 0;
    int n_lns = 0;
    int n_pgs = 0;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (geom)
      {
	  pt = geom->FirstPoint;
	  while (pt)
	    {
		n_pts++;
		pt = pt->Next;
	    }
	  ln = geom->FirstLinestring;
	  while (ln)
	    {
		n_lns++;
		ln = ln->Next;
	    }
	  pg = geom->FirstPolygon;
	  while (pg)
	    {
		n_pgs++;
		pg = pg->Next;
	    }
      }
    /* checking if really is a MultiPoint */
    if (n_pts >= 2 && n_lns == 0 && n_pgs == 0)
	;
    else
      {
	  sqlite3_result_null (context);
	  goto end;
      }
    dyn = gaiaAllocDynamicLine ();
    dyn->Srid = geom->Srid;
    pt = geom->FirstPoint;
    while (pt)
      {
	  /* inserting all Points accordingly to required direction */
	  if (direction)
	    {
		/* conformant direction */
		switch (pt->DimensionModel)
		  {
		  case GAIA_XY_Z_M:
		      gaiaAppendPointZMToDynamicLine (dyn, pt->X, pt->Y,
						      pt->Z, pt->M);
		      break;
		  case GAIA_XY_Z:
		      gaiaAppendPointZToDynamicLine (dyn, pt->X, pt->Y, pt->Z);
		      break;
		  case GAIA_XY_M:
		      gaiaAppendPointMToDynamicLine (dyn, pt->X, pt->Y, pt->M);
		      break;
		  default:
		      gaiaAppendPointToDynamicLine (dyn, pt->X, pt->Y);
		      break;
		  }
	    }
	  else
	    {
		/* reverse direction */
		switch (pt->DimensionModel)
		  {
		  case GAIA_XY_Z_M:
		      gaiaPrependPointZMToDynamicLine (dyn, pt->X, pt->Y,
						       pt->Z, pt->M);
		      break;
		  case GAIA_XY_Z:
		      gaiaPrependPointZToDynamicLine (dyn, pt->X, pt->Y, pt->Z);
		      break;
		  case GAIA_XY_M:
		      gaiaPrependPointMToDynamicLine (dyn, pt->X, pt->Y, pt->M);
		      break;
		  default:
		      gaiaPrependPointToDynamicLine (dyn, pt->X, pt->Y);
		      break;
		  }
	    }
	  pt = pt->Next;
      }
    result = geomFromDynamicLine (dyn);
    gaiaFreeDynamicLine (dyn);
    if (!result)
	sqlite3_result_null (context);
    else
      {
	  /* builds the BLOB geometry to be returned */
	  int len;
	  unsigned char *p_result = NULL;
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (result);
      }
  end:
    gaiaFreeGeomColl (geom);
}

static void
fnct_MakeLine (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakeLine(point-geometry geom1, point-geometry geom2)
/     or
/ MakeLine(multipoint geom, boolean direction)
/
/ - builds a SEGMENT joining two POINTs 
/ - the MultiPoint version works exactely as the corresponding aggregate
/   function, but not requiring aggregation; direction=TRUE direct order,
/   direction=FALSE reverse order
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_blob;
    int n_bytes;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  /* expecting a single MultiPoint input */
	  int direction = sqlite3_value_int (argv[1]);
	  buildLineFromMultiPoint (context, geo1, direction);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo2)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    gaiaMakeLine (geo1, geo2, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
  stop:
    if (geo1)
	gaiaFreeGeomColl (geo1);
    if (geo2)
	gaiaFreeGeomColl (geo2);
}

static void
fnct_MakeCircle (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakeCircle(double cx, double cy, double radius)
/     or
/ MakeCircle(double cx, double cy, double radius, int srid)
/     or
/ MakeCircle(double cx, double cy, double radius, int srid, double step)
/
/ - builds a Linestring approximating a Circle
/ - step is the angular distance (in degrees) between points on 
/   the circurmference (by default: every 10 degs) 
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geom = NULL;
    int ival;
    double cx;
    double cy;
    double r;
    int srid = 0;
    double step = 10.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[0]);
	  cx = ival;
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	cx = sqlite3_value_double (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  cy = ival;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	cy = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  r = ival;
      }
    else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	r = sqlite3_value_double (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	      srid = sqlite3_value_int (argv[3]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 5)
      {
	  if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	    {
		ival = sqlite3_value_int (argv[4]);
		step = ival;
	    }
	  else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	      step = sqlite3_value_double (argv[4]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    geom = gaiaMakeCircle (cx, cy, r, step);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  if (srid != 0)
	      geom->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (geom)
	gaiaFreeGeomColl (geom);
}

static void
fnct_MakeArc (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakeArc(double cx, double cy, double radius, double start, double stop)
/     or
/ MakeArc(double cx, double cy, double radius, double start, double stop, 
/         int srid)
/     or
/ MakeArc(double cx, double cy, double radius, double start, double stop, 
/         int srid, double step)
/
/ - builds a Linestring approximating a Circular Arc
/ - start and stop are the initial and final angles (in degrees)
/ - step is the angular distance (in degrees) between points on 
/   the circurmference (by default: every 10 degs) 
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geom = NULL;
    int ival;
    double cx;
    double cy;
    double r;
    double start;
    double stop;
    int srid = 0;
    double step = 10.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[0]);
	  cx = ival;
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	cx = sqlite3_value_double (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  cy = ival;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	cy = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  r = ival;
      }
    else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	r = sqlite3_value_double (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[3]);
	  start = ival;
      }
    else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	start = sqlite3_value_double (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[4]);
	  stop = ival;
      }
    else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	stop = sqlite3_value_double (argv[4]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 6)
      {
	  if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
	      srid = sqlite3_value_int (argv[5]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 7)
      {
	  if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
	    {
		ival = sqlite3_value_int (argv[6]);
		step = ival;
	    }
	  else if (sqlite3_value_type (argv[6]) == SQLITE_FLOAT)
	      step = sqlite3_value_double (argv[6]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    geom = gaiaMakeArc (cx, cy, r, start, stop, step);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  if (srid != 0)
	      geom->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (geom)
	gaiaFreeGeomColl (geom);
}

static void
fnct_MakeEllipse (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakeEllipse(double cx, double cy, double x_axis, double y_axis)
/     or
/ MakeEllipse(double cx, double cy, double x_axis, double y_axis,
/            int srid)
/     or
/ MakeEllipse(double cx, double cy, double x_axis, double y_axis,
/             int srid, double step)
/
/ - builds a Linestring approximating an Ellipse
/ - step is the angular distance (in degrees) between points on 
/   the ellipse (by default: every 10 degs) 
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geom = NULL;
    int ival;
    double cx;
    double cy;
    double x_axis;
    double y_axis;
    int srid = 0;
    double step = 10.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[0]);
	  cx = ival;
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	cx = sqlite3_value_double (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  cy = ival;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	cy = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  x_axis = ival;
      }
    else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	x_axis = sqlite3_value_double (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[3]);
	  y_axis = ival;
      }
    else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	y_axis = sqlite3_value_double (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 5)
      {
	  if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	      srid = sqlite3_value_int (argv[4]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 6)
      {
	  if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
	    {
		ival = sqlite3_value_int (argv[5]);
		step = ival;
	    }
	  else if (sqlite3_value_type (argv[5]) == SQLITE_FLOAT)
	      step = sqlite3_value_double (argv[5]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    geom = gaiaMakeEllipse (cx, cy, x_axis, y_axis, step);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  if (srid != 0)
	      geom->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (geom)
	gaiaFreeGeomColl (geom);
}

static void
fnct_MakeEllipticArc (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ MakeEllipticArc(double cx, double cy, double x_axis, double y_axis, 
/                 double start, double stop)
/     or
/ MakeEllipticArc(double cx, double cy, double x_axis, double y_axis,
/                 double start, double stop, int srid)
/     or
/ MakeEllipticArc(double cx, double cy, double x_axis, double y_axis,
/                 double start, double stop, int srid, double step)
/
/ - builds a Linestring approximating an Elliptic Arc
/ - start and stop are the initial and final angles (in degrees)
/ - step is the angular distance (in degrees) between points on 
/   the ellipse (by default: every 10 degs) 
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geom = NULL;
    int ival;
    double cx;
    double cy;
    double x_axis;
    double y_axis;
    double start;
    double stop;
    int srid = 0;
    double step = 10.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[0]);
	  cx = ival;
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	cx = sqlite3_value_double (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  cy = ival;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	cy = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  x_axis = ival;
      }
    else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	x_axis = sqlite3_value_double (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[3]);
	  y_axis = ival;
      }
    else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	y_axis = sqlite3_value_double (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[4]);
	  start = ival;
      }
    else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	start = sqlite3_value_double (argv[4]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[5]);
	  stop = ival;
      }
    else if (sqlite3_value_type (argv[5]) == SQLITE_FLOAT)
	stop = sqlite3_value_double (argv[5]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 7)
      {
	  if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
	      srid = sqlite3_value_int (argv[6]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 8)
      {
	  if (sqlite3_value_type (argv[7]) == SQLITE_INTEGER)
	    {
		ival = sqlite3_value_int (argv[7]);
		step = ival;
	    }
	  else if (sqlite3_value_type (argv[7]) == SQLITE_FLOAT)
	      step = sqlite3_value_double (argv[7]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    geom = gaiaMakeEllipticArc (cx, cy, x_axis, y_axis, start, stop, step);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  if (srid != 0)
	      geom->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (geom)
	gaiaFreeGeomColl (geom);
}

static void
fnct_MakeCircularSector (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ MakeCircularSector(double cx, double cy, double radius, double start, double stop)
/     or
/ MakeCircularSector(double cx, double cy, double radius, double start, double stop, 
/         int srid)
/     or
/ MakeCircularSector(double cx, double cy, double radius, double start, double stop, 
/         int srid, double step)
/
/ - builds a Polygon approximating a Circular Sector
/ - start and stop are the initial and final angles (in degrees)
/ - step is the angular distance (in degrees) between points on 
/   the circurmference (by default: every 10 degs) 
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr sector = NULL;
    int ival;
    double cx;
    double cy;
    double r;
    double start;
    double stop;
    int srid = 0;
    double step = 10.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[0]);
	  cx = ival;
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	cx = sqlite3_value_double (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  cy = ival;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	cy = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  r = ival;
      }
    else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	r = sqlite3_value_double (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[3]);
	  start = ival;
      }
    else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	start = sqlite3_value_double (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[4]);
	  stop = ival;
      }
    else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	stop = sqlite3_value_double (argv[4]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 6)
      {
	  if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
	      srid = sqlite3_value_int (argv[5]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 7)
      {
	  if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
	    {
		ival = sqlite3_value_int (argv[6]);
		step = ival;
	    }
	  else if (sqlite3_value_type (argv[6]) == SQLITE_FLOAT)
	      step = sqlite3_value_double (argv[6]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    geom = gaiaMakeArc (cx, cy, r, start, stop, step);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  int ii;
	  int io = 0;
	  double x;
	  double y;
	  gaiaLinestringPtr in = geom->FirstLinestring;
	  gaiaPolygonPtr pg;
	  gaiaRingPtr out;
	  sector = gaiaAllocGeomColl ();
	  pg = gaiaAddPolygonToGeomColl (sector, in->Points + 2, 0);
	  out = pg->Exterior;
	  /* inserting the Centre - first point */
	  gaiaSetPoint (out->Coords, io, cx, cy);
	  io++;
	  for (ii = 0; ii < in->Points; ii++)
	    {
		/* copying the Arc's points */
		gaiaGetPoint (in->Coords, ii, &x, &y);
		gaiaSetPoint (out->Coords, io, x, y);
		io++;
	    }
	  /* inserting the Centre - last point */
	  gaiaSetPoint (out->Coords, io, cx, cy);
	  if (srid != 0)
	      sector->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (sector, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (geom)
	gaiaFreeGeomColl (geom);
    if (sector)
	gaiaFreeGeomColl (sector);
}

static void
fnct_MakeCircularStripe (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ MakeCircularStripe(double cx, double cy, double radius_1, double radius_2,
/                    double start, double stop)
/     or
/ MakeCircularStripe(double cx, double cy, double radius_1, double radius_2,
/                    double start, double stop, int srid)
/     or
/ MakeCircularStripe(double cx, double cy, double radius_1, double radius_2, 
/                    double start, double stop, int srid, double step)
/
/ - builds a Polygon approximating a Circular Stripe delimited by two
/   arcs sharing the same Centre-Point but having different radii
/ - start and stop are the initial and final angles (in degrees)
/ - step is the angular distance (in degrees) between points on 
/   the circurmference (by default: every 10 degs) 
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr arc1 = NULL;
    gaiaGeomCollPtr arc2 = NULL;
    gaiaGeomCollPtr stripe = NULL;
    int ival;
    double cx;
    double cy;
    double r1;
    double r2;
    double start;
    double stop;
    int srid = 0;
    double step = 10.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[0]);
	  cx = ival;
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	cx = sqlite3_value_double (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  cy = ival;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	cy = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  r1 = ival;
      }
    else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	r1 = sqlite3_value_double (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[3]);
	  r2 = ival;
      }
    else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	r2 = sqlite3_value_double (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[4]);
	  start = ival;
      }
    else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	start = sqlite3_value_double (argv[4]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[5]);
	  stop = ival;
      }
    else if (sqlite3_value_type (argv[5]) == SQLITE_FLOAT)
	stop = sqlite3_value_double (argv[5]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 7)
      {
	  if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
	      srid = sqlite3_value_int (argv[6]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 8)
      {
	  if (sqlite3_value_type (argv[7]) == SQLITE_INTEGER)
	    {
		ival = sqlite3_value_int (argv[7]);
		step = ival;
	    }
	  else if (sqlite3_value_type (argv[7]) == SQLITE_FLOAT)
	      step = sqlite3_value_double (argv[7]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    arc1 = gaiaMakeArc (cx, cy, r1, start, stop, step);
    arc2 = gaiaMakeArc (cx, cy, r2, start, stop, step);
    if (arc1 == NULL || arc2 == NULL)
	sqlite3_result_null (context);
    else
      {
	  int ii;
	  int io = 0;
	  double x;
	  double y;
	  gaiaLinestringPtr in1 = arc1->FirstLinestring;
	  gaiaLinestringPtr in2 = arc2->FirstLinestring;
	  gaiaPolygonPtr pg;
	  gaiaRingPtr out;
	  stripe = gaiaAllocGeomColl ();
	  pg = gaiaAddPolygonToGeomColl (stripe,
					 in1->Points + in2->Points + 1, 0);
	  out = pg->Exterior;
	  for (ii = 0; ii < in1->Points; ii++)
	    {
		/* copying the first Arc's points - direct order */
		gaiaGetPoint (in1->Coords, ii, &x, &y);
		gaiaSetPoint (out->Coords, io, x, y);
		io++;
	    }
	  for (ii = in2->Points - 1; ii >= 0; ii--)
	    {
		/* copying the second Arc's points - reverse order */
		gaiaGetPoint (in2->Coords, ii, &x, &y);
		gaiaSetPoint (out->Coords, io, x, y);
		io++;
	    }
	  /* closing the Polygon Ring */
	  gaiaGetPoint (out->Coords, 0, &x, &y);
	  gaiaSetPoint (out->Coords, io, x, y);
	  if (srid != 0)
	      stripe->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (stripe, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (arc1)
	gaiaFreeGeomColl (arc1);
    if (arc2)
	gaiaFreeGeomColl (arc2);
    if (stripe)
	gaiaFreeGeomColl (stripe);
}

static void
fnct_MakeEllipticSector (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ MakeEllipticSector(double cx, double cy, double x_axis, double y_axis, 
/                 double start, double stop)
/     or
/ MakeEllipticSector(double cx, double cy, double x_axis, double y_axis,
/                 double start, double stop, int srid)
/     or
/ MakeEllipticSector(double cx, double cy, double x_axis, double y_axis,
/                 double start, double stop, int srid, double step)
/
/ - builds a Polygon approximating an Elliptic Sector
/ - start and stop are the initial and final angles (in degrees)
/ - step is the angular distance (in degrees) between points on 
/   the ellipse (by default: every 10 degs) 
/ - or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr sector = NULL;
    int ival;
    double cx;
    double cy;
    double x_axis;
    double y_axis;
    double start;
    double stop;
    int srid = 0;
    double step = 10.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[0]);
	  cx = ival;
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	cx = sqlite3_value_double (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  cy = ival;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	cy = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  x_axis = ival;
      }
    else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	x_axis = sqlite3_value_double (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[3]);
	  y_axis = ival;
      }
    else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	y_axis = sqlite3_value_double (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[4]);
	  start = ival;
      }
    else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	start = sqlite3_value_double (argv[4]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[5]);
	  stop = ival;
      }
    else if (sqlite3_value_type (argv[5]) == SQLITE_FLOAT)
	stop = sqlite3_value_double (argv[5]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 7)
      {
	  if (sqlite3_value_type (argv[6]) == SQLITE_INTEGER)
	      srid = sqlite3_value_int (argv[6]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 8)
      {
	  if (sqlite3_value_type (argv[7]) == SQLITE_INTEGER)
	    {
		ival = sqlite3_value_int (argv[7]);
		step = ival;
	    }
	  else if (sqlite3_value_type (argv[7]) == SQLITE_FLOAT)
	      step = sqlite3_value_double (argv[7]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    geom = gaiaMakeEllipticArc (cx, cy, x_axis, y_axis, start, stop, step);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  int ii;
	  int io = 0;
	  double x;
	  double y;
	  gaiaLinestringPtr in = geom->FirstLinestring;
	  gaiaPolygonPtr pg;
	  gaiaRingPtr out;
	  sector = gaiaAllocGeomColl ();
	  pg = gaiaAddPolygonToGeomColl (sector, in->Points + 2, 0);
	  out = pg->Exterior;
	  /* inserting the Centre - first point */
	  gaiaSetPoint (out->Coords, io, cx, cy);
	  io++;
	  for (ii = 0; ii < in->Points; ii++)
	    {
		/* copying the Arc's points */
		gaiaGetPoint (in->Coords, ii, &x, &y);
		gaiaSetPoint (out->Coords, io, x, y);
		io++;
	    }
	  /* inserting the Centre - last point */
	  gaiaSetPoint (out->Coords, io, cx, cy);
	  if (srid != 0)
	      sector->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (sector, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (geom)
	gaiaFreeGeomColl (geom);
    if (sector)
	gaiaFreeGeomColl (sector);
}

static void
fnct_Collect_step (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Collect(BLOBencoded geom)
/
/ aggregate function - STEP
/
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    gaiaGeomCollPtr result;
    gaiaGeomCollPtr *p;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	return;
    p = sqlite3_aggregate_context (context, sizeof (gaiaGeomCollPtr));
    if (!(*p))
      {
	  /* this is the first row */
	  *p = geom;
      }
    else
      {
	  /* subsequent rows */
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMergeGeometries_r (data, *p, geom);
	  else
	      result = gaiaMergeGeometries (*p, geom);
	  *p = result;
	  gaiaFreeGeomColl (geom);
      }
}

static void
fnct_Collect_final (sqlite3_context * context)
{
/* SQL function:
/ Collect(BLOBencoded geom)
/
/ aggregate function - FINAL
/
*/
    gaiaGeomCollPtr result;
    gaiaGeomCollPtr *p = sqlite3_aggregate_context (context, 0);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    result = *p;
    if (!result)
	sqlite3_result_null (context);
    else if (gaiaIsEmpty (result))
      {
	  gaiaFreeGeomColl (result);
	  sqlite3_result_null (context);
      }
    else
      {
	  /* builds the BLOB geometry to be returned */
	  int len;
	  unsigned char *p_result = NULL;
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (result);
      }
}

static void
fnct_Collect (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Collect(geometry geom1, geometry geom2)
/
/ merges two generic GEOMETRIES into a single one 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
      {
	  if (geo1 != NULL)
	      gaiaFreeGeomColl (geo1);
	  if (geo2 != NULL)
	      gaiaFreeGeomColl (geo2);
	  geo1 = NULL;
	  geo2 = NULL;
	  sqlite3_result_null (context);
      }
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMergeGeometries_r (data, geo1, geo2);
	  else
	      result = gaiaMergeGeometries (geo1, geo2);
	  if (!result)
	      sqlite3_result_null (context);
	  else if (gaiaIsEmpty (result))
	    {
		gaiaFreeGeomColl (result);
		sqlite3_result_null (context);
	    }
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo2);
}

static void
geom_from_text1 (sqlite3_context * context, int argc, sqlite3_value ** argv,
		 short type)
{
/* SQL function:
/ GeomFromText(WKT encoded geometry)
/
/ returns the current geometry by parsing WKT encoded string 
/ or NULL if any error is encountered
/
/ if *type* is a negative value can accept any GEOMETRY CLASS
/ otherwise only requests conforming with required CLASS are valid
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseWkt (text, type);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
geom_from_text2 (sqlite3_context * context, int argc, sqlite3_value ** argv,
		 short type)
{
/* SQL function:
/ GeomFromText(WKT encoded geometry, SRID)
/
/ returns the current geometry by parsing WKT encoded string 
/ or NULL if any error is encountered
/
/ if *type* is a negative value can accept any GEOMETRY CLASS
/ otherwise only requests conforming with required CLASS are valid
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseWkt (text, type);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = sqlite3_value_int (argv[1]);
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static int
check_wkb (const unsigned char *wkb, int size, short type)
{
/* checking type coherency for WKB encoded GEOMETRY */
    int little_endian;
    int wkb_type;
    int endian_arch = gaiaEndianArch ();
    if (size < 5)
	return 0;		/* too short to be a WKB */
    if (*(wkb + 0) == 0x01)
	little_endian = GAIA_LITTLE_ENDIAN;
    else if (*(wkb + 0) == 0x00)
	little_endian = GAIA_BIG_ENDIAN;
    else
	return 0;		/* illegal byte ordering; neither BIG-ENDIAN nor LITTLE-ENDIAN */
    wkb_type = gaiaImport32 (wkb + 1, little_endian, endian_arch);
    if (wkb_type == GAIA_POINT || wkb_type == GAIA_LINESTRING
	|| wkb_type == GAIA_POLYGON || wkb_type == GAIA_MULTIPOINT
	|| wkb_type == GAIA_MULTILINESTRING || wkb_type == GAIA_MULTIPOLYGON
	|| wkb_type == GAIA_GEOMETRYCOLLECTION || wkb_type == GAIA_POINTZ
	|| wkb_type == GAIA_LINESTRINGZ || wkb_type == GAIA_POLYGONZ
	|| wkb_type == GAIA_MULTIPOINTZ || wkb_type == GAIA_MULTILINESTRINGZ
	|| wkb_type == GAIA_MULTIPOLYGONZ
	|| wkb_type == GAIA_GEOMETRYCOLLECTIONZ || wkb_type == GAIA_POINTM
	|| wkb_type == GAIA_LINESTRINGM || wkb_type == GAIA_POLYGONM
	|| wkb_type == GAIA_MULTIPOINTM || wkb_type == GAIA_MULTILINESTRINGM
	|| wkb_type == GAIA_MULTIPOLYGONM
	|| wkb_type == GAIA_GEOMETRYCOLLECTIONM || wkb_type == GAIA_POINTZM
	|| wkb_type == GAIA_LINESTRINGZM || wkb_type == GAIA_POLYGONZM
	|| wkb_type == GAIA_MULTIPOINTZM || wkb_type == GAIA_MULTILINESTRINGZM
	|| wkb_type == GAIA_MULTIPOLYGONZM
	|| wkb_type == GAIA_GEOMETRYCOLLECTIONZM)
	;
    else
	return 0;		/* illegal GEOMETRY CLASS */
    if (type < 0)
	;			/* no restrinction about GEOMETRY CLASS TYPE */
    else
      {
	  if (wkb_type != type)
	      return 0;		/* invalid CLASS TYPE for request */
      }
    return 1;
}

static void
geom_from_wkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv,
		short type)
{
/* SQL function:
/ GeomFromWKB(WKB encoded geometry)
/
/ returns the current geometry by parsing a WKB encoded blob 
/ or NULL if any error is encountered
/
/ if *type* is a negative value can accept any GEOMETRY CLASS
/ otherwise only requests conforming with required CLASS are valid
*/
    int len;
    int n_bytes;
    unsigned char *p_result = NULL;
    const unsigned char *wkb;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    wkb = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!check_wkb (wkb, n_bytes, type))
	return;
    geo = gaiaFromWkb (wkb, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
geom_from_wkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv,
		short type)
{
/* SQL function:
/ GeomFromWKB(WKB encoded geometry, SRID)
/
/ returns the current geometry by parsing a WKB encoded blob
/ or NULL if any error is encountered
/
/ if *type* is a negative value can accept any GEOMETRY CLASS
/ otherwise only requests conforming with required CLASS are valid
*/
    int len;
    int n_bytes;
    unsigned char *p_result = NULL;
    const unsigned char *wkb;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    wkb = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!check_wkb (wkb, n_bytes, type))
	return;
    geo = gaiaFromWkb (wkb, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = sqlite3_value_int (argv[1]);
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_GeometryFromFGF1 (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromFGF(FGF encoded geometry)
/
/ returns the current geometry by parsing an FGF encoded blob 
/ or NULL if any error is encountered
/
/ if *type* is a negative value can accept any GEOMETRY CLASS
/ otherwise only requests conforming with required CLASS are valid
*/
    int len;
    int n_bytes;
    unsigned char *p_result = NULL;
    const unsigned char *fgf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    fgf = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromFgf (fgf, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_GeometryFromFGF2 (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromFGF(FGF encoded geometry, SRID)
/
/ returns the current geometry by parsing an FGF encoded string 
/ or NULL if any error is encountered
/
/ if *type* is a negative value can accept any GEOMETRY CLASS
/ otherwise only requests conforming with required CLASS are valid
*/
    int len;
    int n_bytes;
    unsigned char *p_result = NULL;
    const unsigned char *fgf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    fgf = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromFgf (fgf, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = sqlite3_value_int (argv[1]);
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

/*
/ the following functions simply readdress the request to geom_from_text?()
/ setting the appropriate GEOMETRY CLASS TYPE
*/

static void
fnct_GeomFromText1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) -1);
}

static void
fnct_GeomFromText2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) -1);
}

static void
fnct_GeomCollFromText1 (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) GAIA_GEOMETRYCOLLECTION);
}

static void
fnct_GeomCollFromText2 (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) GAIA_GEOMETRYCOLLECTION);
}

static void
fnct_LineFromText1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) GAIA_LINESTRING);
}

static void
fnct_LineFromText2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) GAIA_LINESTRING);
}

static void
fnct_PointFromText1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) GAIA_POINT);
}

static void
fnct_PointFromText2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) GAIA_POINT);
}

static void
fnct_PolyFromText1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) GAIA_POLYGON);
}

static void
fnct_PolyFromText2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) GAIA_POLYGON);
}

static void
fnct_MLineFromText1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) GAIA_MULTILINESTRING);
}

static void
fnct_MLineFromText2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) GAIA_MULTILINESTRING);
}

static void
fnct_MPointFromText1 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) GAIA_MULTIPOINT);
}

static void
fnct_MPointFromText2 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) GAIA_MULTIPOINT);
}

static void
fnct_MPolyFromText1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text1 (context, argc, argv, (short) GAIA_MULTIPOLYGON);
}

static void
fnct_MPolyFromText2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_text2 (context, argc, argv, (short) GAIA_MULTIPOLYGON);
}

static void
fnct_WktToSql (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_WKTToSQL(WKT encoded geometry)
/
/ returns the current geometry by parsing WKT encoded string 
/ or NULL if any error is encountered
/
/ the SRID is always 0 [SQL/MM function]
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseWkt (text, -1);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = 0;
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

/*
/ the following functions simply readdress the request to geom_from_wkb?()
/ setting the appropriate GEOMETRY CLASS TYPE
*/

static void
fnct_GeomFromWkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) -1);
}

static void
fnct_GeomFromWkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) -1);
}

static void
fnct_GeomCollFromWkb1 (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) GAIA_GEOMETRYCOLLECTION);
}

static void
fnct_GeomCollFromWkb2 (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) GAIA_GEOMETRYCOLLECTION);
}

static void
fnct_LineFromWkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) GAIA_LINESTRING);
}

static void
fnct_LineFromWkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) GAIA_LINESTRING);
}

static void
fnct_PointFromWkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) GAIA_POINT);
}

static void
fnct_PointFromWkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) GAIA_POINT);
}

static void
fnct_PolyFromWkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) GAIA_POLYGON);
}

static void
fnct_PolyFromWkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) GAIA_POLYGON);
}

static void
fnct_MLineFromWkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) GAIA_MULTILINESTRING);
}

static void
fnct_MLineFromWkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) GAIA_MULTILINESTRING);
}

static void
fnct_MPointFromWkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) GAIA_MULTIPOINT);
}

static void
fnct_MPointFromWkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) GAIA_MULTIPOINT);
}

static void
fnct_MPolyFromWkb1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb1 (context, argc, argv, (short) GAIA_MULTIPOLYGON);
}

static void
fnct_MPolyFromWkb2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    geom_from_wkb2 (context, argc, argv, (short) GAIA_MULTIPOLYGON);
}

static void
fnct_WkbToSql (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_WKBToSQL(WKB encoded geometry)
/
/ returns the current geometry by parsing a WKB encoded blob 
/ or NULL if any error is encountered
/
/ the SRID is always 0 [SQL/MM function]
*/
    int len;
    int n_bytes;
    unsigned char *p_result = NULL;
    const unsigned char *wkb;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    wkb = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!check_wkb (wkb, n_bytes, -1))
	return;
    geo = gaiaFromWkb (wkb, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = 0;
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_CompressGeometry (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ CompressGeometry(BLOB encoded geometry)
/
/ returns a COMPRESSED geometry [if a valid Geometry was supplied]
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaToCompressedBlobWkb (geo, &p_result, &len);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_UncompressGeometry (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ UncompressGeometry(BLOB encoded geometry)
/
/ returns an UNCOMPRESSED geometry [if a valid Geometry was supplied] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_SanitizeGeometry (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ SanitizeGeometry(BLOB encoded geometry)
/
/ returns a SANITIZED geometry [if a valid Geometry was supplied]
/ or NULL in any other case
/
/ Sanitizing includes:
/ - repeated vertices suppression
/ - enforcing ring closure
/
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr sanitized = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  sanitized = gaiaSanitize (geo);
	  gaiaToSpatiaLiteBlobWkbEx2 (sanitized, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
    gaiaFreeGeomColl (sanitized);
}

static void
fnct_EnsureClosedRings (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ EnsureClosedRings(BLOB encoded geometry)
/
/ returns a SANITIZED geometry [if a valid Geometry was supplied]
/ or NULL in any other case
/
/ Sanitizing includes only enforcing ring closure 
/
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr sanitized = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  sanitized = gaiaEnsureClosedRings (geo);
	  gaiaToSpatiaLiteBlobWkbEx2 (sanitized, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
    gaiaFreeGeomColl (sanitized);
}

static void
fnct_RemoveRepeatedPoints (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ RemoveRepeatedPoints(BLOB encoded geometry)
/ RemoveRepeatedPoints(BLOB encoded geometry, double tolerance)
/
/ returns a SANITIZED geometry [if a valid Geometry was supplied]
/ or NULL in any other case
/
/ Sanitizing includes only repeated vertices suppression
/
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr sanitized = NULL;
    double tolerance = 0.0;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int tol = sqlite3_value_int (argv[1]);
		tolerance = tol;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	      tolerance = sqlite3_value_double (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  sanitized = gaiaRemoveRepeatedPoints (geo, tolerance);
	  gaiaToSpatiaLiteBlobWkbEx2 (sanitized, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
    gaiaFreeGeomColl (sanitized);
}

static void
cast_count (gaiaGeomCollPtr geom, int *pts, int *lns, int *pgs)
{
/* counting elementary geometries */
    int n_pts = 0;
    int n_lns = 0;
    int n_pgs = 0;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    if (geom)
      {
	  pt = geom->FirstPoint;
	  while (pt)
	    {
		n_pts++;
		pt = pt->Next;
	    }
	  ln = geom->FirstLinestring;
	  while (ln)
	    {
		n_lns++;
		ln = ln->Next;
	    }
	  pg = geom->FirstPolygon;
	  while (pg)
	    {
		n_pgs++;
		pg = pg->Next;
	    }
      }
    *pts = n_pts;
    *lns = n_lns;
    *pgs = n_pgs;
}

static void
fnct_CastAutomagic (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastAutomagic(BLOB encoded geometry)
/
/ accepts on input both a valid SpatiaLite BLOB geometry
/ or a valid GPKG BLOB geometry, thus returning a SpatiaLite
/ BLOB geometry 
/ will return NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		geo = gaiaFromGeoPackageGeometryBlob (p_blob, n_bytes);
		if (geo == NULL)
		    sqlite3_result_null (context);
		else
		  {
		      gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len,
						  gpkg_mode, tiny_point);
		      gaiaFreeGeomColl (geo);
		      sqlite3_result_blob (context, p_result, len, free);
		  }
		return;
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
      {
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_blob (context, p_result, len, free);
      }
}

static void
fnct_CastToPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToPoint(BLOB encoded geometry)
/
/ returns a POINT-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts == 1 && lns == 0 && pgs == 0)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_POINT;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToLinestring (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ CastToLinestring(BLOB encoded geometry)
/
/ returns a LINESTRING-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts == 0 && lns == 1 && pgs == 0)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_LINESTRING;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToPolygon (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToPolygon(BLOB encoded geometry)
/
/ returns a POLYGON-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts == 0 && lns == 0 && pgs == 1)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_POLYGON;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToMultiPoint (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ CastToMultiPoint(BLOB encoded geometry)
/
/ returns a MULTIPOINT-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts >= 1 && lns == 0 && pgs == 0)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_MULTIPOINT;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToMultiLinestring (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ CastToMultiLinestring(BLOB encoded geometry)
/
/ returns a MULTILINESTRING-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts == 0 && lns >= 1 && pgs == 0)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_MULTILINESTRING;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToMultiPolygon (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ CastToMultiPolygon(BLOB encoded geometry)
/
/ returns a MULTIPOLYGON-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts == 0 && lns == 0 && pgs >= 1)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_MULTIPOLYGON;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToGeometryCollection (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ CastToGeometryCollection(BLOB encoded geometry)
/
/ returns a GEOMETRYCOLLECTION-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts >= 1 || lns >= 1 || pgs >= 1)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_GEOMETRYCOLLECTION;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToMulti (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToMulti(BLOB encoded geometry)
/
/ returns a MULTIPOINT, MULTILINESTRING, MULTIPOLYGON or
/ GEOMETRYCOLLECTION-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts >= 1 || lns >= 1 || pgs >= 1)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		if (pts >= 1 && lns == 0 && pgs == 0)
		    geom2->DeclaredType = GAIA_MULTIPOINT;
		else if (pts == 0 && lns >= 1 && pgs == 0)
		    geom2->DeclaredType = GAIA_MULTILINESTRING;
		else if (pts == 0 && lns == 0 && pgs >= 1)
		    geom2->DeclaredType = GAIA_MULTIPOLYGON;
		else
		    geom2->DeclaredType = GAIA_GEOMETRYCOLLECTION;
		if (geo->DeclaredType == GAIA_GEOMETRYCOLLECTION)
		    geom2->DeclaredType = GAIA_GEOMETRYCOLLECTION;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToSingle (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToSingle(BLOB encoded geometry)
/
/ returns a POINT, LINESTRING or POLYGON-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    int ok;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  ok = 0;
	  if (pts == 1 && lns == 0 && pgs == 0)
	      ok = 1;
	  if (pts == 0 && lns == 1 && pgs == 0)
	      ok = 1;
	  if (pts == 0 && lns == 0 && pgs == 1)
	      ok = 1;
	  if (ok)
	    {
		geom2 = gaiaCloneGeomColl (geo);
		geom2->Srid = geo->Srid;
		if (pts == 1)
		    geom2->DeclaredType = GAIA_POINT;
		else if (lns == 1)
		    geom2->DeclaredType = GAIA_LINESTRING;
		else
		    geom2->DeclaredType = GAIA_POLYGON;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToXY (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToXY(BLOB encoded geometry)
/
/ returns an XY-dimension Geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  geom2 = gaiaCastGeomCollToXY (geo);
	  if (geom2)
	    {
		geom2->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToXYZ (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToXYZ(BLOB encoded geometry)
/    or
/ CastToXYZ(BLOB encoded geometry, nodata double)
/
/ returns an XY-dimension Geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    double no_data;
    int has_no_data = 0;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[1]);
		no_data = val;
		has_no_data = 1;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		no_data = sqlite3_value_double (argv[1]);
		has_no_data = 1;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (has_no_data)
	      geom2 = gaiaCastGeomCollToXYZnoData (geo, no_data);
	  else
	      geom2 = gaiaCastGeomCollToXYZ (geo);
	  if (geom2)
	    {
		geom2->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToXYM (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToXYM(BLOB encoded geometry)
/    or
/ CastToXYM(BLOB encoded geometry. nodata double)
/
/ returns an XYM-dimension Geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    double no_data;
    int has_no_data = 0;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[1]);
		no_data = val;
		has_no_data = 1;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		no_data = sqlite3_value_double (argv[1]);
		has_no_data = 1;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (has_no_data)
	      geom2 = gaiaCastGeomCollToXYMnoData (geo, no_data);
	  else
	      geom2 = gaiaCastGeomCollToXYM (geo);
	  if (geom2)
	    {
		geom2->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CastToXYZM (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToXYZM(BLOB encoded geometry)
/    or
/ CastToXYZM(BLOB encoded geometry, nodata_z double, nodata_m double)
/
/ returns an XYZM-dimension Geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    double z_no_data;
    double m_no_data;
    int has_no_data = 0;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[1]);
		z_no_data = val;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	      z_no_data = sqlite3_value_double (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[2]);
		m_no_data = val;
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      m_no_data = sqlite3_value_double (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  has_no_data = 1;
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (has_no_data)
	      geom2 = gaiaCastGeomCollToXYZMnoData (geo, z_no_data, m_no_data);
	  else
	      geom2 = gaiaCastGeomCollToXYZM (geo);
	  if (geom2)
	    {
		geom2->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ExtractMultiPoint (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ ExtractMultiPoint(BLOB encoded geometry)
/
/ returns a MULTIPOINT-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pts >= 1)
	    {
		geom2 = gaiaCloneGeomCollPoints (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_MULTIPOINT;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ExtractMultiLinestring (sqlite3_context * context, int argc,
			     sqlite3_value ** argv)
{
/* SQL function:
/ ExtractMultiLinestring(BLOB encoded geometry)
/
/ returns a MULTILINESTRING-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (lns >= 1)
	    {
		geom2 = gaiaCloneGeomCollLinestrings (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_MULTILINESTRING;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ExtractMultiPolygon (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ ExtractMultiPolygon(BLOB encoded geometry)
/
/ returns a MULTIPOLYGON-type geometry [if conversion is possible] 
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    int pts;
    int lns;
    int pgs;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  cast_count (geo, &pts, &lns, &pgs);
	  if (pgs >= 1)
	    {
		geom2 = gaiaCloneGeomCollPolygons (geo);
		geom2->Srid = geo->Srid;
		geom2->DeclaredType = GAIA_MULTIPOLYGON;
		gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
					    tiny_point);
		gaiaFreeGeomColl (geom2);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Reverse (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_Reverse(BLOB encoded geometry)
/
/ returns a new Geometry: any Linestring or Ring will be in reverse order
/ or NULL in any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  geom2 = gaiaCloneGeomCollSpecial (geo, GAIA_REVERSE_ORDER);
	  geom2->Srid = geo->Srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
				      tiny_point);
	  gaiaFreeGeomColl (geom2);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_ForcePolygonCW (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_ForcePolygonCW(BLOB encoded geometry)
/    or
/ ST_ForceLHR(BLOB encoded geometry)
/
/ returns a new Geometry: any Exterior Ring will be in clockwise orientation
/         and any Interior Ring will be in counter-clockwise orientation
/ or NULL on invalid geometries
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  geom2 = gaiaCloneGeomCollSpecial (geo, GAIA_CW_ORDER);
	  geom2->Srid = geo->Srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
				      tiny_point);
	  gaiaFreeGeomColl (geom2);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_ForcePolygonCCW (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ ST_ForcePolygonCCW(BLOB encoded geometry)
/
/ returns a new Geometry: any Exterior Ring will be in counter-clockwise 
/ orientation and any Interior Ring will be in clockwise orientation
/ or NULL on invalid geometries
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  geom2 = gaiaCloneGeomCollSpecial (geo, GAIA_CCW_ORDER);
	  geom2->Srid = geo->Srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom2, &p_result, &len, gpkg_mode,
				      tiny_point);
	  gaiaFreeGeomColl (geom2);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_IsPolygonCW (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_IsPolygonCW(BLOB encoded geometry)
/
/ returns TRUE if all Exterior Rings are in clockwise orientation 
/ and all Interior Ring are in counter-clockwise orientation,
/ FALSE if not
/ and -1 on invalid geometries
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_int (context, -1);
    else
      {
	  int retval = gaiaCheckClockwise (geo);
	  sqlite3_result_int (context, retval);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_IsPolygonCCW (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_IsPolygonCCW(BLOB encoded geometry)
/
/ returns TRUE if all Exterior Rings are in counter-clockwise 
/ orientation and all Interior Ring are in clockwise orientation,
/ FALSE if not
/ and NULL on invalid geometries
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_int (context, -1);
    else
      {
	  int retval = gaiaCheckCounterClockwise (geo);
	  sqlite3_result_int (context, retval);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_Dimension (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Dimension(BLOB encoded geometry)
/
/ returns:
/ 0 if geometry is a POINT or MULTIPOINT
/ 1 if geometry is a LINESTRING or MULTILINESTRING
/ 2 if geometry is a POLYGON or MULTIPOLYGON
/ 0, 1, 2, for GEOMETRYCOLLECTIONS according to geometries contained inside
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int dim;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  dim = gaiaDimension (geo);
	  sqlite3_result_int (context, dim);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CoordDimension (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CoordDimension(BLOB encoded geometry)
/
/ returns:
/ 'XY', 'XYM', 'XYZ', 'XYZM'
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    char *p_dim = NULL;
    char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (geo->DimensionModel == GAIA_XY)
	      p_dim = "XY";
	  else if (geo->DimensionModel == GAIA_XY_Z)
	      p_dim = "XYZ";
	  else if (geo->DimensionModel == GAIA_XY_M)
	      p_dim = "XYM";
	  else if (geo->DimensionModel == GAIA_XY_Z_M)
	      p_dim = "XYZM";
	  if (p_dim)
	    {
		len = strlen (p_dim);
		p_result = malloc (len + 1);
		strcpy (p_result, p_dim);
	    }
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	    {
		len = strlen (p_result);
		sqlite3_result_text (context, p_result, len, free);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_NDims (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_NDims(BLOB encoded geometry)
/
/ returns:
/ 2, 3 or 4
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int result = 0;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (geo->DimensionModel == GAIA_XY)
	      result = 2;
	  else if (geo->DimensionModel == GAIA_XY_Z)
	      result = 3;
	  else if (geo->DimensionModel == GAIA_XY_M)
	      result = 3;
	  else if (geo->DimensionModel == GAIA_XY_Z_M)
	      result = 4;
	  sqlite3_result_int (context, result);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_GeometryType (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeometryType(BLOB encoded geometry)
/
/ returns the class for current geometry:
/ 'POINT' or 'MULTIPOINT' [Z, M, ZM]
/ 'LINESTRING' or 'MULTILINESTRING' [Z, M, ZM]
/ 'POLYGON' or 'MULTIPOLYGON' [Z, M, ZM]
/ 'GEOMETRYCOLLECTION'  [Z, M, ZM]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    int type;
    char *p_type = NULL;
    char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		char *gpb_type = gaiaGetGeometryTypeFromGPB (p_blob, n_bytes);
		if (gpb_type == NULL)
		    sqlite3_result_null (context);
		else
		  {
		      len = strlen (gpb_type);
		      sqlite3_result_text (context, gpb_type, len, free);
		  }
		return;
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
      {
	  type = gaiaGeometryType (geo);
	  switch (type)
	    {
	    case GAIA_POINT:
		p_type = "POINT";
		break;
	    case GAIA_POINTZ:
		p_type = "POINT Z";
		break;
	    case GAIA_POINTM:
		p_type = "POINT M";
		break;
	    case GAIA_POINTZM:
		p_type = "POINT ZM";
		break;
	    case GAIA_MULTIPOINT:
		p_type = "MULTIPOINT";
		break;
	    case GAIA_MULTIPOINTZ:
		p_type = "MULTIPOINT Z";
		break;
	    case GAIA_MULTIPOINTM:
		p_type = "MULTIPOINT M";
		break;
	    case GAIA_MULTIPOINTZM:
		p_type = "MULTIPOINT ZM";
		break;
	    case GAIA_LINESTRING:
	    case GAIA_COMPRESSED_LINESTRING:
		p_type = "LINESTRING";
		break;
	    case GAIA_LINESTRINGZ:
	    case GAIA_COMPRESSED_LINESTRINGZ:
		p_type = "LINESTRING Z";
		break;
	    case GAIA_LINESTRINGM:
	    case GAIA_COMPRESSED_LINESTRINGM:
		p_type = "LINESTRING M";
		break;
	    case GAIA_LINESTRINGZM:
	    case GAIA_COMPRESSED_LINESTRINGZM:
		p_type = "LINESTRING ZM";
		break;
	    case GAIA_MULTILINESTRING:
		p_type = "MULTILINESTRING";
		break;
	    case GAIA_MULTILINESTRINGZ:
		p_type = "MULTILINESTRING Z";
		break;
	    case GAIA_MULTILINESTRINGM:
		p_type = "MULTILINESTRING M";
		break;
	    case GAIA_MULTILINESTRINGZM:
		p_type = "MULTILINESTRING ZM";
		break;
	    case GAIA_POLYGON:
	    case GAIA_COMPRESSED_POLYGON:
		p_type = "POLYGON";
		break;
	    case GAIA_POLYGONZ:
	    case GAIA_COMPRESSED_POLYGONZ:
		p_type = "POLYGON Z";
		break;
	    case GAIA_POLYGONM:
	    case GAIA_COMPRESSED_POLYGONM:
		p_type = "POLYGON M";
		break;
	    case GAIA_POLYGONZM:
	    case GAIA_COMPRESSED_POLYGONZM:
		p_type = "POLYGON ZM";
		break;
	    case GAIA_MULTIPOLYGON:
		p_type = "MULTIPOLYGON";
		break;
	    case GAIA_MULTIPOLYGONZ:
		p_type = "MULTIPOLYGON Z";
		break;
	    case GAIA_MULTIPOLYGONM:
		p_type = "MULTIPOLYGON M";
		break;
	    case GAIA_MULTIPOLYGONZM:
		p_type = "MULTIPOLYGON ZM";
		break;
	    case GAIA_GEOMETRYCOLLECTION:
		p_type = "GEOMETRYCOLLECTION";
		break;
	    case GAIA_GEOMETRYCOLLECTIONZ:
		p_type = "GEOMETRYCOLLECTION Z";
		break;
	    case GAIA_GEOMETRYCOLLECTIONM:
		p_type = "GEOMETRYCOLLECTION M";
		break;
	    case GAIA_GEOMETRYCOLLECTIONZM:
		p_type = "GEOMETRYCOLLECTION ZM";
		break;
	    };
	  if (p_type)
	    {
		len = strlen (p_type);
		p_result = malloc (len + 1);
		strcpy (p_result, p_type);
	    }
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	    {
		len = strlen (p_result);
		sqlite3_result_text (context, p_result, len, free);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_GeometryAliasType (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ GeometryAliasType(BLOB encoded geometry)
/
/ returns the alias-class for current geometry:
/ 'POINT'
/ 'LINESTRING'
/ 'POLYGON'
/ 'MULTIPOINT'
/ 'MULTILINESTRING'
/ 'MULTIPOLYGON'
/ 'GEOMETRYCOLLECTION' 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    int type;
    char *p_type = NULL;
    char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  type = gaiaGeometryAliasType (geo);
	  switch (type)
	    {
	    case GAIA_POINT:
		p_type = "POINT";
		break;
	    case GAIA_MULTIPOINT:
		p_type = "MULTIPOINT";
		break;
	    case GAIA_LINESTRING:
		p_type = "LINESTRING";
		break;
	    case GAIA_MULTILINESTRING:
		p_type = "MULTILINESTRING";
		break;
	    case GAIA_POLYGON:
		p_type = "POLYGON";
		break;
	    case GAIA_MULTIPOLYGON:
		p_type = "MULTIPOLYGON";
		break;
	    case GAIA_GEOMETRYCOLLECTION:
		p_type = "GEOMETRYCOLLECTION";
		break;
	    };
	  if (p_type)
	    {
		len = strlen (p_type);
		p_result = malloc (len + 1);
		strcpy (p_result, p_type);
	    }
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	    {
		len = strlen (p_result);
		sqlite3_result_text (context, p_result, len, free);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_SridFromAuthCRS (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ SridFromAuthCRS(auth_name, auth_srid)
/
/ returns the SRID
/ or NULL if any error is encountered
*/
    const unsigned char *auth_name;
    int auth_srid;
    int srid = -1;
    char *sql;
    char **results;
    int n_rows;
    int n_columns;
    char *err_msg = NULL;
    int ret;
    int i;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    auth_name = sqlite3_value_text (argv[0]);
    auth_srid = sqlite3_value_int (argv[1]);

    sql = sqlite3_mprintf ("SELECT srid FROM spatial_ref_sys "
			   "WHERE Upper(auth_name) = Upper(%Q) AND auth_srid = %d",
			   auth_name, auth_srid);
    ret = sqlite3_get_table (sqlite, sql, &results, &n_rows, &n_columns,
			     &err_msg);
    sqlite3_free (sql);
    if (ret != SQLITE_OK)
	goto done;
    if (n_rows >= 1)
      {
	  for (i = 1; i <= n_rows; i++)
	      srid = atoi (results[(i * n_columns) + 0]);
      }
    sqlite3_free_table (results);
  done:
    sqlite3_result_int (context, srid);
}

static void
fnct_SRID (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Srid(BLOB encoded geometry)
/
/ returns the SRID
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		int srid = gaiaGetSridFromGPB (p_blob, n_bytes);
		sqlite3_result_int (context, srid);
	    }
	  else
	      sqlite3_result_null (context);
	  return;
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	  sqlite3_result_null (context);
      }
    else
	sqlite3_result_int (context, geo->Srid);
    gaiaFreeGeomColl (geo);
}

static void
fnct_SetSRID (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SetSrid(BLOBencoded geometry, srid)
/
/ returns a new geometry that is the original one received, but with the new SRID [no coordinates translation is applied]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    int srid;
    unsigned char *p_result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  geo->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &n_bytes, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, n_bytes, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_IsEmpty (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsEmpty(BLOB encoded geometry)
/
/ returns:
/ 1 if this geometry contains no elementary geometries
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		int is_empty = gaiaIsEmptyGPB (p_blob, n_bytes);
		sqlite3_result_int (context, is_empty);
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_int (context, -1);
      }
    else
	sqlite3_result_int (context, gaiaIsEmpty (geo));
    gaiaFreeGeomColl (geo);
}

static void
fnct_Is3D (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Is3D(BLOB encoded geometry)
/
/ returns:
/ 1 if this geometry has Z coords
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      sqlite3_result_int (context, has_z);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_int (context, -1);
      }
    else
      {
	  if (geo->DimensionModel == GAIA_XY_Z
	      || geo->DimensionModel == GAIA_XY_Z_M)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_IsMeasured (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsMeasured(BLOB encoded geometry)
/
/ returns:
/ 1 if this geometry has M coords
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      sqlite3_result_int (context, has_m);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_int (context, -1);
      }
    else
      {
	  if (geo->DimensionModel == GAIA_XY_M
	      || geo->DimensionModel == GAIA_XY_Z_M)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_MinZ (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_MinZ(BLOB encoded GEMETRY)
/    or
/ ST_MinZ(BLOB encoded GEOMETRY, DOUBLE nodata-value)
/
/ returns the MinZ coordinate for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double min;
    double max;
    double nodata = DBL_MAX;
    int hasNodata = 0;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		nodata = sqlite3_value_double (argv[1]);
		hasNodata = 1;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int intval = sqlite3_value_int (argv[1]);
		nodata = intval;
		hasNodata = 1;
	    }
	  else
	      sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      if (has_z)
			  sqlite3_result_double (context, min_z);
		      else
			  sqlite3_result_null (context);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
      {
	  if (geo->DimensionModel == GAIA_XY_Z
	      || geo->DimensionModel == GAIA_XY_Z_M)
	    {
		if (hasNodata)
		    gaiaZRangeGeometryEx (geo, nodata, &min, &max);
		else
		    gaiaZRangeGeometry (geo, &min, &max);
		sqlite3_result_double (context, min);
	    }
	  else
	      sqlite3_result_null (context);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_MaxZ (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_MaxZ(BLOB encoded GEMETRY)
/    or
/ ST_MaxZ(BLOB encoded GEOMETRY, DOUBLE nodata-value)
/
/ returns the MaxZ coordinate for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double min;
    double max;
    double nodata = DBL_MAX;
    int hasNodata = 0;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		nodata = sqlite3_value_double (argv[1]);
		hasNodata = 1;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int intval = sqlite3_value_int (argv[1]);
		nodata = intval;
		hasNodata = 1;
	    }
	  else
	      sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      if (has_z)
			  sqlite3_result_double (context, max_z);
		      else
			  sqlite3_result_null (context);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
      {
	  if (geo->DimensionModel == GAIA_XY_Z
	      || geo->DimensionModel == GAIA_XY_Z_M)
	    {
		if (hasNodata)
		    gaiaZRangeGeometryEx (geo, nodata, &min, &max);
		else
		    gaiaZRangeGeometry (geo, &min, &max);
		sqlite3_result_double (context, max);
	    }
	  else
	      sqlite3_result_null (context);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_MinM (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_MinM(BLOB encoded GEMETRY)
/    or
/ ST_MinM(BLOB encoded GEOMETRY, DOUBLE nodata-value)
/
/ returns the MinM coordinate for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double min;
    double max;
    double nodata = DBL_MAX;
    int hasNodata = 0;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		nodata = sqlite3_value_double (argv[1]);
		hasNodata = 1;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int intval = sqlite3_value_int (argv[1]);
		nodata = intval;
		hasNodata = 1;
	    }
	  else
	      sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      if (has_m)
			  sqlite3_result_double (context, min_m);
		      else
			  sqlite3_result_null (context);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
      {
	  if (geo->DimensionModel == GAIA_XY_M
	      || geo->DimensionModel == GAIA_XY_Z_M)
	    {
		if (hasNodata)
		    gaiaMRangeGeometryEx (geo, nodata, &min, &max);
		else
		    gaiaMRangeGeometry (geo, &min, &max);
		sqlite3_result_double (context, min);
	    }
	  else
	      sqlite3_result_null (context);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_MaxM (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_MaxM(BLOB encoded GEMETRY)
/    or
/ ST_MaxM(BLOB encoded GEOMETRY, DOUBLE nodata-value)
/
/ returns the MaxM coordinate for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double min;
    double max;
    double nodata = DBL_MAX;
    int hasNodata = 0;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		nodata = sqlite3_value_double (argv[1]);
		hasNodata = 1;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int intval = sqlite3_value_int (argv[1]);
		nodata = intval;
		hasNodata = 1;
	    }
	  else
	      sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
    if (!geo)
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      if (has_m)
			  sqlite3_result_double (context, max_m);
		      else
			  sqlite3_result_null (context);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
      {
	  if (geo->DimensionModel == GAIA_XY_M
	      || geo->DimensionModel == GAIA_XY_Z_M)
	    {
		if (hasNodata)
		    gaiaMRangeGeometryEx (geo, nodata, &min, &max);
		else
		    gaiaMRangeGeometry (geo, &min, &max);
		sqlite3_result_double (context, max);
	    }
	  else
	      sqlite3_result_null (context);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_Envelope (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Envelope(BLOB encoded geometry)
/
/ returns the MBR for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr bbox;
    gaiaPolygonPtr polyg;
    gaiaRingPtr rect;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaMbrGeometry (geo);
	  bbox = gaiaAllocGeomColl ();
	  bbox->Srid = geo->Srid;
	  polyg = gaiaAddPolygonToGeomColl (bbox, 5, 0);
	  rect = polyg->Exterior;
	  gaiaSetPoint (rect->Coords, 0, geo->MinX, geo->MinY);	/* vertex # 1 */
	  gaiaSetPoint (rect->Coords, 1, geo->MaxX, geo->MinY);	/* vertex # 2 */
	  gaiaSetPoint (rect->Coords, 2, geo->MaxX, geo->MaxY);	/* vertex # 3 */
	  gaiaSetPoint (rect->Coords, 3, geo->MinX, geo->MaxY);	/* vertex # 4 */
	  gaiaSetPoint (rect->Coords, 4, geo->MinX, geo->MinY);	/* vertex # 5 [same as vertex # 1 to close the polygon] */
	  gaiaToSpatiaLiteBlobWkbEx2 (bbox, &p_result, &len, gpkg_mode,
				      tiny_point);
	  gaiaFreeGeomColl (bbox);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Expand (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_Expand(BLOB encoded geometry, double amount)
/
/ returns the MBR for current geometry expanded by "amount" in each direction
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr bbox;
    gaiaPolygonPtr polyg;
    gaiaRingPtr rect;
    double tic;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	tic = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  tic = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaMbrGeometry (geo);
	  bbox = gaiaAllocGeomColl ();
	  bbox->Srid = geo->Srid;
	  polyg = gaiaAddPolygonToGeomColl (bbox, 5, 0);
	  rect = polyg->Exterior;
	  gaiaSetPoint (rect->Coords, 0, geo->MinX - tic, geo->MinY - tic);	/* vertex # 1 */
	  gaiaSetPoint (rect->Coords, 1, geo->MaxX + tic, geo->MinY - tic);	/* vertex # 2 */
	  gaiaSetPoint (rect->Coords, 2, geo->MaxX + tic, geo->MaxY + tic);	/* vertex # 3 */
	  gaiaSetPoint (rect->Coords, 3, geo->MinX - tic, geo->MaxY + tic);	/* vertex # 4 */
	  gaiaSetPoint (rect->Coords, 4, geo->MinX - tic, geo->MinY - tic);	/* vertex # 5 [same as vertex # 1 to close the polygon] */
	  gaiaToSpatiaLiteBlobWkbEx2 (bbox, &p_result, &len, gpkg_mode,
				      tiny_point);
	  gaiaFreeGeomColl (bbox);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
build_filter_mbr (sqlite3_context * context, int argc,
		  sqlite3_value ** argv, int mode)
{
/* SQL functions:
/ BuildMbrFilter(double X1, double Y1, double X2, double Y2)
/ FilterMBRWithin(double X1, double Y1, double X2, double Y2)
/ FilterMBRContain(double X1, double Y1, double X2, double Y2)
/ FilterMBRIntersects(double X1, double Y1, double X2, double Y2)
/
/ builds a generic filter for MBR from two points (identifying a rectangle's diagonal) 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    double x1;
    double y1;
    double x2;
    double y2;
    int int_value;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x1 = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x1 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y1 = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y1 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	x2 = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  x2 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	y2 = sqlite3_value_double (argv[3]);
    else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[3]);
	  y2 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaBuildFilterMbr (x1, y1, x2, y2, mode, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

/*
/ the following functions simply readdress the request to build_filter_mbr()
/ setting the appropriate MODe
*/

static void
fnct_BuildMbrFilter (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    build_filter_mbr (context, argc, argv, GAIA_FILTER_MBR_DECLARE);
}

static void
fnct_FilterMbrWithin (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
    build_filter_mbr (context, argc, argv, GAIA_FILTER_MBR_WITHIN);
}

static void
fnct_FilterMbrContains (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
    build_filter_mbr (context, argc, argv, GAIA_FILTER_MBR_CONTAINS);
}

static void
fnct_FilterMbrIntersects (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
    build_filter_mbr (context, argc, argv, GAIA_FILTER_MBR_INTERSECTS);
}

static void
fnct_BuildMbr1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ BuildMBR(double X1, double Y1, double X2, double Y2)
/
/ builds an MBR from two points (identifying a rectangle's diagonal) 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    double x1;
    double y1;
    double x2;
    double y2;
    int int_value;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x1 = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x1 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y1 = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y1 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	x2 = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  x2 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	y2 = sqlite3_value_double (argv[3]);
    else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[3]);
	  y2 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaBuildMbr (x1, y1, x2, y2, -1, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_BuildMbr2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ BuildMBR(double X1, double Y1, double X2, double Y2, int SRID)
/
/ builds an MBR from two points (identifying a rectangle's diagonal) 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    double x1;
    double y1;
    double x2;
    double y2;
    int int_value;
    int srid;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x1 = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x1 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y1 = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y1 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	x2 = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  x2 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	y2 = sqlite3_value_double (argv[3]);
    else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[3]);
	  y2 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[4]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaBuildMbr (x1, y1, x2, y2, srid, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_BuildCircleMbr1 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ BuildCircleMBR(double X, double Y, double radius)
/
/ builds an MBR from two points (identifying a rectangle's diagonal) 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double radius;
    int int_value;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	radius = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  radius = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaBuildCircleMbr (x, y, radius, -1, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_BuildCircleMbr2 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ BuildCircleMBR(double X, double Y, double radius, int SRID)
/
/ builds an MBR from two points (identifying a rectangle's diagonal) 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    double x;
    double y;
    double radius;
    int int_value;
    int srid;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	radius = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  radius = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	srid = sqlite3_value_int (argv[3]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaBuildCircleMbr (x, y, radius, srid, &p_result, &len);
    if (!p_result)
	sqlite3_result_null (context);
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_Extent_step (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Extent(BLOBencoded geom)
/
/ aggregate function - STEP
/
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    double **p;
    double *max_min;
    int *srid_check;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	return;
    gaiaMbrGeometry (geom);
    p = sqlite3_aggregate_context (context, sizeof (double **));
    if (!(*p))
      {
	  /* this is the first row */
	  max_min = malloc ((sizeof (double) * 5));
	  *(max_min + 0) = geom->MinX;
	  *(max_min + 1) = geom->MinY;
	  *(max_min + 2) = geom->MaxX;
	  *(max_min + 3) = geom->MaxY;
	  srid_check = (int *) (max_min + 4);
	  *(srid_check + 0) = geom->Srid;
	  *(srid_check + 1) = geom->Srid;
	  *p = max_min;
      }
    else
      {
	  /* subsequent rows */
	  max_min = *p;
	  if (geom->MinX < *(max_min + 0))
	      *(max_min + 0) = geom->MinX;
	  if (geom->MinY < *(max_min + 1))
	      *(max_min + 1) = geom->MinY;
	  if (geom->MaxX > *(max_min + 2))
	      *(max_min + 2) = geom->MaxX;
	  if (geom->MaxY > *(max_min + 3))
	      *(max_min + 3) = geom->MaxY;
	  srid_check = (int *) (max_min + 4);
	  if (*(srid_check + 1) != geom->Srid)
	      *(srid_check + 1) = geom->Srid;
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_Extent_final (sqlite3_context * context)
{
/* SQL function:
/ Extent(BLOBencoded geom)
/
/ aggregate function - FINAL
/
*/
    gaiaGeomCollPtr result;
    gaiaPolygonPtr polyg;
    gaiaRingPtr rect;
    double *max_min;
    double minx;
    double miny;
    double maxx;
    double maxy;
    int *srid_check;
    double **p = sqlite3_aggregate_context (context, 0);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    max_min = *p;
    if (!max_min)
      {
	  sqlite3_result_null (context);
	  return;
      }
    srid_check = (int *) (max_min + 4);
    if (*(srid_check + 0) != *(srid_check + 1))
      {
	  sqlite3_result_null (context);
	  return;
      }
    result = gaiaAllocGeomColl ();
    if (!result)
	sqlite3_result_null (context);
    else
      {
	  /* builds the BLOB geometry to be returned */
	  int len;
	  unsigned char *p_result = NULL;
	  result->Srid = *(srid_check + 0);
	  polyg = gaiaAddPolygonToGeomColl (result, 5, 0);
	  rect = polyg->Exterior;
	  minx = *(max_min + 0);
	  miny = *(max_min + 1);
	  maxx = *(max_min + 2);
	  maxy = *(max_min + 3);
	  gaiaSetPoint (rect->Coords, 0, minx, miny);	/* vertex # 1 */
	  gaiaSetPoint (rect->Coords, 1, maxx, miny);	/* vertex # 2 */
	  gaiaSetPoint (rect->Coords, 2, maxx, maxy);	/* vertex # 3 */
	  gaiaSetPoint (rect->Coords, 3, minx, maxy);	/* vertex # 4 */
	  gaiaSetPoint (rect->Coords, 4, minx, miny);	/* vertex # 5 [same as vertex # 1 to close the polygon] */
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (result);
      }
    free (max_min);
}

static void
fnct_MbrMinX (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MbrMinX(BLOB encoded GEMETRY)
/
/ returns the MinX coordinate for current geometry's MBR 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double coord;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!gaiaGetMbrMinX (p_blob, n_bytes, &coord))
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      sqlite3_result_double (context, min_x);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
	sqlite3_result_double (context, coord);
}

static void
fnct_MbrMaxX (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MbrMaxX(BLOB encoded GEMETRY)
/
/ returns the MaxX coordinate for current geometry's MBR 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double coord;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!gaiaGetMbrMaxX (p_blob, n_bytes, &coord))
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      sqlite3_result_double (context, max_x);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
	sqlite3_result_double (context, coord);
}

static void
fnct_MbrMinY (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MbrMinY(BLOB encoded GEMETRY)
/
/ returns the MinY coordinate for current geometry's MBR 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double coord;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!gaiaGetMbrMinY (p_blob, n_bytes, &coord))
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      sqlite3_result_double (context, min_y);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
	sqlite3_result_double (context, coord);
}

static void
fnct_MbrMaxY (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MbrMaxY(BLOB encoded GEMETRY)
/
/ returns the MaxY coordinate for current geometry's MBR 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double coord;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!gaiaGetMbrMaxY (p_blob, n_bytes, &coord))
      {
#ifdef ENABLE_GEOPACKAGE	/* GEOPACKAGE enabled: supporting GPKG geometries */
	  if (gaiaIsValidGPB (p_blob, n_bytes))
	    {
		double min_x;
		double max_x;
		double min_y;
		double max_y;
		int has_z;
		double min_z;
		double max_z;
		int has_m;
		double min_m;
		double max_m;
		if (gaiaGetEnvelopeFromGPB
		    (p_blob, n_bytes, &min_x, &max_x, &min_y, &max_y, &has_z,
		     &min_z, &max_z, &has_m, &min_m, &max_m))
		  {
		      sqlite3_result_double (context, max_y);
		  }
	    }
	  else
#endif /* end GEOPACKAGE: supporting GPKG geometries */
	      sqlite3_result_null (context);
      }
    else
	sqlite3_result_double (context, coord);
}

static void
fnct_X (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ X(BLOB encoded POINT)
/
/ returns the X coordinate for current POINT geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaPointPtr point;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  point = simplePoint (geo);
	  if (!point)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, point->X);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Y (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Y(BLOB encoded POINT)
/
/ returns the Y coordinate for current POINT geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaPointPtr point;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  point = simplePoint (geo);
	  if (!point)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, point->Y);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Z (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Z(BLOB encoded POINT)
/
/ returns the Z coordinate for current POINT geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaPointPtr point;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  point = simplePoint (geo);
	  if (!point)
	      sqlite3_result_null (context);
	  else
	    {
		if (point->DimensionModel == GAIA_XY_Z
		    || point->DimensionModel == GAIA_XY_Z_M)
		    sqlite3_result_double (context, point->Z);
		else
		    sqlite3_result_null (context);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_M (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ M(BLOB encoded POINT)
/
/ returns the M coordinate for current POINT geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaPointPtr point;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  point = simplePoint (geo);
	  if (!point)
	      sqlite3_result_null (context);
	  else
	    {
		if (point->DimensionModel == GAIA_XY_M
		    || point->DimensionModel == GAIA_XY_Z_M)
		    sqlite3_result_double (context, point->M);
		else
		    sqlite3_result_null (context);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_NumPoints (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ NumPoints(BLOB encoded LINESTRING)
/
/ returns the number of vertices for current LINESTRING geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaLinestringPtr line;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  line = simpleLinestring (geo);
	  if (!line)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_int (context, line->Points);
      }
    gaiaFreeGeomColl (geo);
}

static void
point_n (sqlite3_context * context, int argc, sqlite3_value ** argv,
	 int request)
{
/* SQL functions:
/ StartPoint(BLOB encoded LINESTRING geometry)
/ EndPoint(BLOB encoded LINESTRING geometry)
/ PointN(BLOB encoded LINESTRING geometry, integer point_no)
/
/ returns the Nth POINT for current LINESTRING geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int vertex;
    int len;
    double x;
    double y;
    double z;
    double m;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    gaiaLinestringPtr line;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (request == GAIA_POINTN)
      {
	  /* PointN() requires point index to be defined as an SQL function argument */
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  vertex = sqlite3_value_int (argv[1]);
      }
    else if (request == GAIA_END_POINT)
	vertex = -1;		/* EndPoint() specifies a negative point index */
    else
	vertex = 1;		/* StartPoint() */
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  line = simpleLinestring (geo);
	  if (!line)
	      sqlite3_result_null (context);
	  else
	    {
		if (vertex < 0)
		    vertex = line->Points - 1;
		else
		    vertex -= 1;	/* decreasing the point index by 1, because PointN counts starting at index 1 */
		if (vertex >= 0 && vertex < line->Points)
		  {
		      if (line->DimensionModel == GAIA_XY_Z)
			{
			    gaiaGetPointXYZ (line->Coords, vertex, &x, &y, &z);
			    result = gaiaAllocGeomCollXYZ ();
			    result->Srid = geo->Srid;
			    gaiaAddPointToGeomCollXYZ (result, x, y, z);
			}
		      else if (line->DimensionModel == GAIA_XY_M)
			{
			    gaiaGetPointXYM (line->Coords, vertex, &x, &y, &m);
			    result = gaiaAllocGeomCollXYM ();
			    result->Srid = geo->Srid;
			    gaiaAddPointToGeomCollXYM (result, x, y, m);
			}
		      else if (line->DimensionModel == GAIA_XY_Z_M)
			{
			    gaiaGetPointXYZM (line->Coords, vertex, &x, &y,
					      &z, &m);
			    result = gaiaAllocGeomCollXYZM ();
			    result->Srid = geo->Srid;
			    gaiaAddPointToGeomCollXYZM (result, x, y, z, m);
			}
		      else
			{
			    gaiaGetPoint (line->Coords, vertex, &x, &y);
			    result = gaiaAllocGeomColl ();
			    result->Srid = geo->Srid;
			    gaiaAddPointToGeomColl (result, x, y);
			}
		  }
		else
		    result = NULL;
		if (!result)
		    sqlite3_result_null (context);
		else
		  {
		      gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
						  gpkg_mode, tiny_point);
		      gaiaFreeGeomColl (result);
		      sqlite3_result_blob (context, p_result, len, free);
		  }
	    }
      }
    gaiaFreeGeomColl (geo);
}

/*
/ the following functions simply readdress the request to point_n()
/ setting the appropriate request mode
*/

static void
fnct_StartPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    point_n (context, argc, argv, GAIA_START_POINT);
}

static void
fnct_EndPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    point_n (context, argc, argv, GAIA_END_POINT);
}

static void
fnct_PointN (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    point_n (context, argc, argv, GAIA_POINTN);
}

static void
fnct_ExteriorRing (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ExteriorRing(BLOB encoded POLYGON geometry)
/
/ returns the EXTERIOR RING for current POLYGON geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int iv;
    double x;
    double y;
    double z;
    double m;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    gaiaPolygonPtr polyg;
    gaiaRingPtr ring;
    gaiaLinestringPtr line;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  polyg = simplePolygon (geo);
	  if (!polyg)
	      sqlite3_result_null (context);
	  else
	    {
		ring = polyg->Exterior;
		if (ring->DimensionModel == GAIA_XY_Z)
		    result = gaiaAllocGeomCollXYZ ();
		else if (ring->DimensionModel == GAIA_XY_M)
		    result = gaiaAllocGeomCollXYM ();
		else if (ring->DimensionModel == GAIA_XY_Z_M)
		    result = gaiaAllocGeomCollXYZM ();
		else
		    result = gaiaAllocGeomColl ();
		result->Srid = geo->Srid;
		line = gaiaAddLinestringToGeomColl (result, ring->Points);
		for (iv = 0; iv < line->Points; iv++)
		  {
		      if (ring->DimensionModel == GAIA_XY_Z)
			{
			    gaiaGetPointXYZ (ring->Coords, iv, &x, &y, &z);
			    gaiaSetPointXYZ (line->Coords, iv, x, y, z);
			}
		      else if (ring->DimensionModel == GAIA_XY_M)
			{
			    gaiaGetPointXYM (ring->Coords, iv, &x, &y, &m);
			    gaiaSetPointXYM (line->Coords, iv, x, y, m);
			}
		      else if (ring->DimensionModel == GAIA_XY_Z_M)
			{
			    gaiaGetPointXYZM (ring->Coords, iv, &x, &y, &z, &m);
			    gaiaSetPointXYZM (line->Coords, iv, x, y, z, m);
			}
		      else
			{
			    gaiaGetPoint (ring->Coords, iv, &x, &y);
			    gaiaSetPoint (line->Coords, iv, x, y);
			}
		  }
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		gaiaFreeGeomColl (result);
		sqlite3_result_blob (context, p_result, len, free);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_NumInteriorRings (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ NumInteriorRings(BLOB encoded POLYGON)
/
/ returns the number of INTERIOR RINGS for current POLYGON geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaPolygonPtr polyg;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  polyg = simplePolygon (geo);
	  if (!polyg)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_int (context, polyg->NumInteriors);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_InteriorRingN (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ InteriorRingN(BLOB encoded POLYGON geometry)
/
/ returns the Nth INTERIOR RING for current POLYGON geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int border;
    int iv;
    double x;
    double y;
    double z;
    double m;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    gaiaPolygonPtr polyg;
    gaiaRingPtr ring;
    gaiaLinestringPtr line;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    border = sqlite3_value_int (argv[1]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  polyg = simplePolygon (geo);
	  if (!polyg)
	      sqlite3_result_null (context);
	  else
	    {
		if (border >= 1 && border <= polyg->NumInteriors)
		  {
		      ring = polyg->Interiors + (border - 1);
		      if (ring->DimensionModel == GAIA_XY_Z)
			  result = gaiaAllocGeomCollXYZ ();
		      else if (ring->DimensionModel == GAIA_XY_M)
			  result = gaiaAllocGeomCollXYM ();
		      else if (ring->DimensionModel == GAIA_XY_Z_M)
			  result = gaiaAllocGeomCollXYZM ();
		      else
			  result = gaiaAllocGeomColl ();
		      result->Srid = geo->Srid;
		      line = gaiaAddLinestringToGeomColl (result, ring->Points);
		      for (iv = 0; iv < line->Points; iv++)
			{
			    if (ring->DimensionModel == GAIA_XY_Z)
			      {
				  gaiaGetPointXYZ (ring->Coords, iv, &x,
						   &y, &z);
				  gaiaSetPointXYZ (line->Coords, iv, x, y, z);
			      }
			    else if (ring->DimensionModel == GAIA_XY_M)
			      {
				  gaiaGetPointXYM (ring->Coords, iv, &x,
						   &y, &m);
				  gaiaSetPointXYM (line->Coords, iv, x, y, m);
			      }
			    else if (ring->DimensionModel == GAIA_XY_Z_M)
			      {
				  gaiaGetPointXYZM (ring->Coords, iv, &x,
						    &y, &z, &m);
				  gaiaSetPointXYZM (line->Coords, iv, x,
						    y, z, m);
			      }
			    else
			      {
				  gaiaGetPoint (ring->Coords, iv, &x, &y);
				  gaiaSetPoint (line->Coords, iv, x, y);
			      }
			}
		      gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
						  gpkg_mode, tiny_point);
		      gaiaFreeGeomColl (result);
		      sqlite3_result_blob (context, p_result, len, free);
		  }
		else
		    sqlite3_result_null (context);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_NumGeometries (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ NumGeometries(BLOB encoded GEOMETRYCOLLECTION)
/
/ returns the number of elementary geometries for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int cnt = 0;
    gaiaPointPtr point;
    gaiaLinestringPtr line;
    gaiaPolygonPtr polyg;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  point = geo->FirstPoint;
	  while (point)
	    {
		/* counts how many points are there */
		cnt++;
		point = point->Next;
	    }
	  line = geo->FirstLinestring;
	  while (line)
	    {
		/* counts how many linestrings are there */
		cnt++;
		line = line->Next;
	    }
	  polyg = geo->FirstPolygon;
	  while (polyg)
	    {
		/* counts how many polygons are there */
		cnt++;
		polyg = polyg->Next;
	    }
	  sqlite3_result_int (context, cnt);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_NPoints (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_NPoints(BLOB encoded GEOMETRYCOLLECTION)
/
/ returns the total number of points/vertices for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int cnt = 0;
    int ib;
    gaiaPointPtr point;
    gaiaLinestringPtr line;
    gaiaPolygonPtr polyg;
    gaiaRingPtr rng;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  point = geo->FirstPoint;
	  while (point)
	    {
		/* counts how many points are there */
		cnt++;
		point = point->Next;
	    }
	  line = geo->FirstLinestring;
	  while (line)
	    {
		/* counts how many points are there */
		cnt += line->Points;
		line = line->Next;
	    }
	  polyg = geo->FirstPolygon;
	  while (polyg)
	    {
		/* counts how many points are in the exterior ring */
		rng = polyg->Exterior;
		cnt += rng->Points;
		for (ib = 0; ib < polyg->NumInteriors; ib++)
		  {
		      /* processing any interior ring */
		      rng = polyg->Interiors + ib;
		      cnt += rng->Points;
		  }
		polyg = polyg->Next;
	    }
	  sqlite3_result_int (context, cnt);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_NRings (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_NRings(BLOB encoded GEOMETRYCOLLECTION)
/
/ returns the total number of rings for current geometry 
/ (this including both interior and exterior rings)
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int cnt = 0;
    gaiaPolygonPtr polyg;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  polyg = geo->FirstPolygon;
	  while (polyg)
	    {
		/* counts how many rings are there */
		cnt += polyg->NumInteriors + 1;
		polyg = polyg->Next;
	    }
	  sqlite3_result_int (context, cnt);
      }
    gaiaFreeGeomColl (geo);
}

static int
is_single_point (gaiaGeomCollPtr geom)
{
/* check if this geometry is a simple Point */
    int pts = 0;
    int lns = 0;
    int pgs = 0;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    pt = geom->FirstPoint;
    while (pt)
      {
	  pts++;
	  pt = pt->Next;
      }
    ln = geom->FirstLinestring;
    while (ln)
      {
	  lns++;
	  ln = ln->Next;
      }
    pg = geom->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pts == 1 && lns == 0 && pgs == 0)
	return 1;
    return 0;
}

static int
is_single_linestring (gaiaGeomCollPtr geom)
{
/* check if this geometry is a simple Linestring */
    int pts = 0;
    int lns = 0;
    int pgs = 0;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    pt = geom->FirstPoint;
    while (pt)
      {
	  pts++;
	  pt = pt->Next;
      }
    ln = geom->FirstLinestring;
    while (ln)
      {
	  lns++;
	  ln = ln->Next;
      }
    pg = geom->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pts == 0 && lns == 1 && pgs == 0)
	return 1;
    return 0;
}

static void
fnct_AddPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_AddPoint(BLOB encoded LINESTRING line, BLOB encoded POINT point)
/ ST_AddPoint(BLOB encoded LINESTRING line, BLOB encoded POINT point, INTEGER position)
/
/ returns a new Linestring by adding a new Point before "position" (zero-based index)
/ a negative "position" (default) means appending to the end 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaLinestringPtr ln;
    gaiaLinestringPtr out_ln;
    gaiaPointPtr pt;
    int position = -1;
    gaiaGeomCollPtr line = NULL;
    gaiaGeomCollPtr point = NULL;
    gaiaGeomCollPtr out;
    int len;
    unsigned char *p_result = NULL;
    int iv;
    int out_iv;
    double x;
    double y;
    double m;
    double z;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!line)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    point =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!point)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		goto stop;
	    }
	  position = sqlite3_value_int (argv[2]);
      }
    if (is_single_linestring (line) && is_single_point (point))
	;
    else
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    ln = line->FirstLinestring;
    pt = point->FirstPoint;
    if (position >= 0 && position >= ln->Points)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
/* creating the output Geometry */
    if (line->DimensionModel == GAIA_XY_Z)
	out = gaiaAllocGeomCollXYZ ();
    else if (line->DimensionModel == GAIA_XY_M)
	out = gaiaAllocGeomCollXYM ();
    else if (line->DimensionModel == GAIA_XY_Z_M)
	out = gaiaAllocGeomCollXYZM ();
    else
	out = gaiaAllocGeomColl ();
    out->Srid = line->Srid;
    out->DeclaredType = line->DeclaredType;
    out_ln = gaiaAddLinestringToGeomColl (out, ln->Points + 1);
    if (position < 0)
      {
	  /* appending the new Point */
	  for (iv = 0; iv < ln->Points; iv++)
	    {
		if (line->DimensionModel == GAIA_XY_Z)
		  {
		      gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
		      gaiaSetPointXYZ (out_ln->Coords, iv, x, y, z);
		  }
		else if (line->DimensionModel == GAIA_XY_M)
		  {
		      gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
		      gaiaSetPointXYM (out_ln->Coords, iv, x, y, m);
		  }
		else if (line->DimensionModel == GAIA_XY_Z_M)
		  {
		      gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
		      gaiaSetPointXYZM (out_ln->Coords, iv, x, y, z, m);
		  }
		else
		  {
		      gaiaGetPoint (ln->Coords, iv, &x, &y);
		      gaiaSetPoint (out_ln->Coords, iv, x, y);
		  }
	    }
	  /* appending the new Point */
	  x = pt->X;
	  y = pt->Y;
	  z = pt->Z;
	  m = pt->M;
	  out_iv = ln->Points;
	  if (line->DimensionModel == GAIA_XY_Z)
	    {
		gaiaSetPointXYZ (out_ln->Coords, out_iv, x, y, z);
	    }
	  else if (line->DimensionModel == GAIA_XY_M)
	    {
		gaiaSetPointXYM (out_ln->Coords, out_iv, x, y, m);
	    }
	  else if (line->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaSetPointXYZM (out_ln->Coords, out_iv, x, y, z, m);
	    }
	  else
	    {
		gaiaSetPoint (out_ln->Coords, out_iv, x, y);
	    }
      }
    else
      {
	  /* inserting the new Point before "position" */
	  out_iv = 0;
	  for (iv = 0; iv < position; iv++)
	    {
		/* copying all Points before "position" */
		if (line->DimensionModel == GAIA_XY_Z)
		  {
		      gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
		      gaiaSetPointXYZ (out_ln->Coords, out_iv, x, y, z);
		  }
		else if (line->DimensionModel == GAIA_XY_M)
		  {
		      gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
		      gaiaSetPointXYM (out_ln->Coords, out_iv, x, y, m);
		  }
		else if (line->DimensionModel == GAIA_XY_Z_M)
		  {
		      gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
		      gaiaSetPointXYZM (out_ln->Coords, out_iv, x, y, z, m);
		  }
		else
		  {
		      gaiaGetPoint (ln->Coords, iv, &x, &y);
		      gaiaSetPoint (out_ln->Coords, out_iv, x, y);
		  }
		out_iv++;
	    }
	  /* inserting the new Point */
	  x = pt->X;
	  y = pt->Y;
	  z = pt->Z;
	  m = pt->M;
	  if (line->DimensionModel == GAIA_XY_Z)
	    {
		gaiaSetPointXYZ (out_ln->Coords, out_iv, x, y, z);
	    }
	  else if (line->DimensionModel == GAIA_XY_M)
	    {
		gaiaSetPointXYM (out_ln->Coords, out_iv, x, y, m);
	    }
	  else if (line->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaSetPointXYZM (out_ln->Coords, out_iv, x, y, z, m);
	    }
	  else
	    {
		gaiaSetPoint (out_ln->Coords, out_iv, x, y);
	    }
	  out_iv++;
	  for (iv = position; iv < ln->Points; iv++)
	    {
		/* copying all Points after "position" */
		if (line->DimensionModel == GAIA_XY_Z)
		  {
		      gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
		      gaiaSetPointXYZ (out_ln->Coords, out_iv, x, y, z);
		  }
		else if (line->DimensionModel == GAIA_XY_M)
		  {
		      gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
		      gaiaSetPointXYM (out_ln->Coords, out_iv, x, y, m);
		  }
		else if (line->DimensionModel == GAIA_XY_Z_M)
		  {
		      gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
		      gaiaSetPointXYZM (out_ln->Coords, out_iv, x, y, z, m);
		  }
		else
		  {
		      gaiaGetPoint (ln->Coords, iv, &x, &y);
		      gaiaSetPoint (out_ln->Coords, out_iv, x, y);
		  }
		out_iv++;
	    }
      }
    gaiaToSpatiaLiteBlobWkbEx2 (out, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (out);
    sqlite3_result_blob (context, p_result, len, free);
  stop:
    gaiaFreeGeomColl (line);
    gaiaFreeGeomColl (point);
}

static void
common_set_point (sqlite3_context * context, gaiaGeomCollPtr line,
		  int position, gaiaGeomCollPtr point)
{
/* SetPoint - common implementation */
    gaiaGeomCollPtr out;
    gaiaLinestringPtr ln;
    gaiaLinestringPtr out_ln;
    gaiaPointPtr pt;
    unsigned char *p_result = NULL;
    int len;
    int iv;
    double x = 0.0;
    double y = 0.0;
    double m = 0.0;
    double z = 0.0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }

    if (is_single_linestring (line) && is_single_point (point))
	;
    else
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    ln = line->FirstLinestring;
    pt = point->FirstPoint;
    if (position < 0 || position >= ln->Points)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
/* creating the output Geometry */
    if (line->DimensionModel == GAIA_XY_Z)
	out = gaiaAllocGeomCollXYZ ();
    else if (line->DimensionModel == GAIA_XY_M)
	out = gaiaAllocGeomCollXYM ();
    else if (line->DimensionModel == GAIA_XY_Z_M)
	out = gaiaAllocGeomCollXYZM ();
    else
	out = gaiaAllocGeomColl ();
    out->Srid = line->Srid;
    out->DeclaredType = line->DeclaredType;
    out_ln = gaiaAddLinestringToGeomColl (out, ln->Points);
    for (iv = 0; iv < ln->Points; iv++)
      {
	  if (line->DimensionModel == GAIA_XY_Z)
	    {
		gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
	    }
	  else if (line->DimensionModel == GAIA_XY_M)
	    {
		gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
	    }
	  else if (line->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
	    }
	  else
	    {
		gaiaGetPoint (ln->Coords, iv, &x, &y);
	    }
	  if (iv == position)
	    {
		/* replacing the new Point */
		x = pt->X;
		y = pt->Y;
		z = pt->Z;
		m = pt->M;
	    }
	  if (line->DimensionModel == GAIA_XY_Z)
	    {
		gaiaSetPointXYZ (out_ln->Coords, iv, x, y, z);
	    }
	  else if (line->DimensionModel == GAIA_XY_M)
	    {
		gaiaSetPointXYM (out_ln->Coords, iv, x, y, m);
	    }
	  else if (line->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaSetPointXYZM (out_ln->Coords, iv, x, y, z, m);
	    }
	  else
	    {
		gaiaSetPoint (out_ln->Coords, iv, x, y);
	    }
      }
    gaiaToSpatiaLiteBlobWkbEx2 (out, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (out);
    sqlite3_result_blob (context, p_result, len, free);
  stop:
    gaiaFreeGeomColl (line);
    gaiaFreeGeomColl (point);
}

static void
get_point_index (sqlite3_context * context, gaiaGeomCollPtr line,
		 gaiaGeomCollPtr point, int check_multiple)
{
/* actual implementation of get_point_index */
    gaiaLinestringPtr ln;
    gaiaPointPtr pt;
    int iv;
    double x = 0.0;
    double y = 0.0;
    double m = 0.0;
    double z = 0.0;
    double sv_x = 0.0;
    double sv_y = 0.0;
    double sv_m = 0.0;
    double sv_z = 0.0;
    double dist;
    double min_dist = DBL_MAX;
    int position;
    int multiple = 0;

    if (is_single_linestring (line) && is_single_point (point))
	;
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    ln = line->FirstLinestring;
    pt = point->FirstPoint;
    for (iv = 0; iv < ln->Points; iv++)
      {
	  if (line->DimensionModel == GAIA_XY_Z)
	    {
		gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
	    }
	  else if (line->DimensionModel == GAIA_XY_M)
	    {
		gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
	    }
	  else if (line->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
	    }
	  else
	    {
		gaiaGetPoint (ln->Coords, iv, &x, &y);
	    }
	  dist =
	      sqrt (((x - pt->X) * (x - pt->X)) + ((y - pt->Y) * (y - pt->Y)));
	  if (dist < min_dist)
	    {
		min_dist = dist;
		position = iv;
		sv_x = x;
		sv_y = y;
		sv_z = z;
		sv_m = m;
	    }
      }
    if (check_multiple)
      {
	  /* checking for multiple points */
	  for (iv = 0; iv < ln->Points; iv++)
	    {
		if (iv == position)
		    continue;	/* ignoring the first occurence */
		if (line->DimensionModel == GAIA_XY_Z)
		  {
		      gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
		  }
		else if (line->DimensionModel == GAIA_XY_M)
		  {
		      gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
		  }
		else if (line->DimensionModel == GAIA_XY_Z_M)
		  {
		      gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
		  }
		else
		  {
		      gaiaGetPoint (ln->Coords, iv, &x, &y);
		  }
		if (x == sv_x && y == sv_y && z == sv_z && m == sv_m)
		    multiple++;
	    }
      }
    if (multiple)
	sqlite3_result_int (context, -1);
    else
	sqlite3_result_int (context, position);
}

static void
fnct_GetPointIndex (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_GetPointIndex(BLOB encoded LINESTRING line, BLOB encoded POINT point)
/ or
/ ST_GetPointIndex(BLOB encoded LINESTRING line, BLOB encoded POINT point,
/                  BOOLEAN check_multiple)
/
/ returns the "position" (zero-based index) of the Linestring's vertex
/ nearest to the given Point
/ or NULL if any error is encountered
/ or a NEGATIVE value if there are multiple definitions of the same vertex
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr line = NULL;
    gaiaGeomCollPtr point = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int check_multiple = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!line)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    point =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!point)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		gaiaFreeGeomColl (line);
		gaiaFreeGeomColl (point);
		return;
	    }
	  check_multiple = sqlite3_value_int (argv[2]);
      }
    get_point_index (context, line, point, check_multiple);
    gaiaFreeGeomColl (line);
    gaiaFreeGeomColl (point);
}

static void
fnct_SetMultiplePoints (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ ST_SetMultiplePoints(BLOB encoded LINESTRING line, INTEGER pk_value, TEXT table_name,
/                      TEXT point_name, TEXT pk_name, TEXT pos_name)
/
/ returns a new Linestring by replacing all Points found into an helper table
/                      - table_name is the name of the helper table
/                      - point_name is the name of the column containing Points
/                      - pk_name is the name of the column containig FIDs
/                      - pos_name is the name of the column containing 
/                        "positions" (zero-based index) of the Points to
/                        be replaced
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr line = NULL;
    sqlite3_int64 pk_value;
    const char *table_name;
    const char *point_name;
    const char *pk_name;
    const char *pos_name;
    int ret;
    char *err_msg;
    unsigned char *p_result = NULL;
    int len;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
	goto invalid_linestring;
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!line)
	goto invalid_linestring;
    if (!is_single_linestring (line))
	goto invalid_linestring;
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	goto invalid_pk_value;
    else
	pk_value = sqlite3_value_int64 (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
	goto invalid_table_name;
    else
	table_name = (const char *) sqlite3_value_text (argv[2]);
    if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
	goto invalid_point_name;
    else
	point_name = (const char *) sqlite3_value_text (argv[3]);
    if (sqlite3_value_type (argv[4]) != SQLITE_TEXT)
	goto invalid_pk_name;
    else
	pk_name = (const char *) sqlite3_value_text (argv[4]);
    if (sqlite3_value_type (argv[5]) != SQLITE_TEXT)
	goto invalid_pos_name;
    else
	pos_name = (const char *) sqlite3_value_text (argv[5]);

    ret =
	do_set_multiple_points (sqlite, line, pk_value, table_name, point_name,
				pk_name, pos_name);
    switch (ret)
      {
      case MULTIPLE_POINTS_TABLE:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: no such table \"%s\".",
	       table_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_POINT:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: table \"%s\" has no column \"%s\".",
	       table_name, point_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_PK:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: table \"%s\" has no column \"%s\".",
	       table_name, pk_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_POS:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: table \"%s\" has no column \"%s\".",
	       table_name, pos_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_NOGEOM:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: \"%s\".\"%s\" is not a registered Geometry.",
	       table_name, point_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_SRID:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: \"%s\".\"%s\" mismatching SRID.",
	       table_name, point_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_NOPOINT:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: \"%s\".\"%s\" is not a Geometry of the POINT type.",
	       table_name, point_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_DIMS:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: \"%s\".\"%s\" mismatching dimensions.",
	       table_name, point_name);
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_SQL:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: unexpected SQL error.");
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_DUPL:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: duplicate position found.");
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_GEOM:
	  err_msg =
	      sqlite3_mprintf
	      ("SetMultiplePoints() exception: illegal Geometry found.");
	  sqlite3_result_error (context, err_msg, -1);
	  sqlite3_free (err_msg);
	  break;
      case MULTIPLE_POINTS_OK:
	  gaiaToSpatiaLiteBlobWkbEx2 (line, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  break;
      };
    gaiaFreeGeomColl (line);
    return;

  invalid_linestring:
    if (line != NULL)
	gaiaFreeGeomColl (line);
    sqlite3_result_error (context,
			  "SetMultiplePoints() exception: first argument is not a valid Linestring.",
			  -1);
    return;

  invalid_pk_value:
    gaiaFreeGeomColl (line);
    sqlite3_result_error (context,
			  "SetMultiplePoints() exception: second argument is not of the INTEGER type.",
			  -1);
    return;

  invalid_table_name:
    gaiaFreeGeomColl (line);
    sqlite3_result_error (context,
			  "SetMultiplePoints() exception: third argument is not of the TEXT type.",
			  -1);
    return;

  invalid_point_name:
    gaiaFreeGeomColl (line);
    sqlite3_result_error (context,
			  "SetMultiplePoints() exception: fourth argument is not of the TEXT type.",
			  -1);
    return;

  invalid_pk_name:
    gaiaFreeGeomColl (line);
    sqlite3_result_error (context,
			  "SetMultiplePoints() exception: fifth argument is not of the TEXT type.",
			  -1);
    return;

  invalid_pos_name:
    sqlite3_result_error (context,
			  "SetMultiplePoints() exception: sixth argument is not of the TEXT type.",
			  -1);
    return;
}

static void
fnct_SetPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_SetPoint(BLOB encoded LINESTRING line, INTEGER position, BLOB encoded POINT point)
/
/ returns a new Linestring by replacing the Point at "position" (zero-based index)
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int position;
    gaiaGeomCollPtr line = NULL;
    gaiaGeomCollPtr point = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!line)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    position = sqlite3_value_int (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_BLOB)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[2]);
    n_bytes = sqlite3_value_bytes (argv[2]);
    point =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!point)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    common_set_point (context, line, position, point);
}

static void
fnct_SetStartPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_SetStartPoint(BLOB encoded LINESTRING line, BLOB encoded POINT point)
/
/ returns a new Linestring by replacing its StartPoint
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr line = NULL;
    gaiaGeomCollPtr point = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!line)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    point =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!point)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    common_set_point (context, line, 0, point);
}

static void
fnct_SetEndPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_SetEndPoint(BLOB encoded LINESTRING line, BLOB encoded POINT point)
/
/ returns a new Linestring by replacing its EndPoint
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaLinestringPtr ln;
    gaiaGeomCollPtr line = NULL;
    gaiaGeomCollPtr point = NULL;
    int position;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!line)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    point =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!point)
      {
	  gaiaFreeGeomColl (line);
	  sqlite3_result_null (context);
	  return;
      }
    if (is_single_linestring (line) && is_single_point (point))
	;
    else
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    ln = line->FirstLinestring;
    position = ln->Points - 1;
    common_set_point (context, line, position, point);
    return;
  stop:
    gaiaFreeGeomColl (line);
    gaiaFreeGeomColl (point);
}

static void
fnct_RemovePoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_RemovePoint(BLOB encoded LINESTRING line, INTEGER position)
/
/ returns a new Linestring by removing the Point at "position" (zero-based index)
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaLinestringPtr ln;
    gaiaLinestringPtr out_ln;
    int position;
    gaiaGeomCollPtr line = NULL;
    gaiaGeomCollPtr out;
    int len;
    unsigned char *p_result = NULL;
    int iv;
    int out_iv;
    double x;
    double y;
    double m;
    double z;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!line)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    position = sqlite3_value_int (argv[1]);
    if (!is_single_linestring (line))
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    ln = line->FirstLinestring;
    if (position < 0 || position >= ln->Points)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
/* creating the output Geometry */
    if (line->DimensionModel == GAIA_XY_Z)
	out = gaiaAllocGeomCollXYZ ();
    else if (line->DimensionModel == GAIA_XY_M)
	out = gaiaAllocGeomCollXYM ();
    else if (line->DimensionModel == GAIA_XY_Z_M)
	out = gaiaAllocGeomCollXYZM ();
    else
	out = gaiaAllocGeomColl ();
    out->Srid = line->Srid;
    out->DeclaredType = line->DeclaredType;
    out_ln = gaiaAddLinestringToGeomColl (out, ln->Points - 1);
    out_iv = 0;
    for (iv = 0; iv < position; iv++)
      {
	  /* copying all Points before "position" */
	  if (line->DimensionModel == GAIA_XY_Z)
	    {
		gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
		gaiaSetPointXYZ (out_ln->Coords, out_iv, x, y, z);
	    }
	  else if (line->DimensionModel == GAIA_XY_M)
	    {
		gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
		gaiaSetPointXYM (out_ln->Coords, out_iv, x, y, m);
	    }
	  else if (line->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
		gaiaSetPointXYZM (out_ln->Coords, out_iv, x, y, z, m);
	    }
	  else
	    {
		gaiaGetPoint (ln->Coords, iv, &x, &y);
		gaiaSetPoint (out_ln->Coords, out_iv, x, y);
	    }
	  out_iv++;
      }
    for (iv = position + 1; iv < ln->Points; iv++)
      {
	  /* copying all Points after "position" */
	  if (line->DimensionModel == GAIA_XY_Z)
	    {
		gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
		gaiaSetPointXYZ (out_ln->Coords, out_iv, x, y, z);
	    }
	  else if (line->DimensionModel == GAIA_XY_M)
	    {
		gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
		gaiaSetPointXYM (out_ln->Coords, out_iv, x, y, m);
	    }
	  else if (line->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
		gaiaSetPointXYZM (out_ln->Coords, out_iv, x, y, z, m);
	    }
	  else
	    {
		gaiaGetPoint (ln->Coords, iv, &x, &y);
		gaiaSetPoint (out_ln->Coords, out_iv, x, y);
	    }
	  out_iv++;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (out, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (out);
    sqlite3_result_blob (context, p_result, len, free);
  stop:
    gaiaFreeGeomColl (line);
}

static void
fnct_MakePolygon (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_MakePolygon(BLOB encoded LINESTRING line)
/ ST_MakePolygon(BLOB encoded LINESTRING line, BLOB encoded (MULTI)LINESTING holes)
/
/ returns a new POLYGON from the given exterior and interior rings
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr exterior = NULL;
    gaiaGeomCollPtr interiors = NULL;
    gaiaGeomCollPtr out;
    int len;
    unsigned char *p_result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    exterior =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!exterior)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		goto stop;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
	  n_bytes = sqlite3_value_bytes (argv[1]);
	  interiors =
	      gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
					   gpkg_amphibious);
	  if (!interiors)
	    {
		sqlite3_result_null (context);
		goto stop;
	    }
      }
    out = gaiaMakePolygon (exterior, interiors);
    if (!out)
      {
	  sqlite3_result_null (context);
	  goto stop;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (out, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (out);
    sqlite3_result_blob (context, p_result, len, free);
  stop:
    gaiaFreeGeomColl (exterior);
    gaiaFreeGeomColl (interiors);
}

static int
getXYZMSinglePoint (gaiaGeomCollPtr geom, double *x, double *y, double *z,
		    double *m)
{
/* check if this geometry is a simple Point (returning full coords) */
    int pts = 0;
    int lns = 0;
    int pgs = 0;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    pt = geom->FirstPoint;
    while (pt)
      {
	  pts++;
	  pt = pt->Next;
      }
    ln = geom->FirstLinestring;
    while (ln)
      {
	  lns++;
	  ln = ln->Next;
      }
    pg = geom->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pts == 1 && lns == 0 && pgs == 0)
	;
    else
	return 0;
    *x = geom->FirstPoint->X;
    *y = geom->FirstPoint->Y;
    if (geom->DimensionModel == GAIA_XY_Z
	|| geom->DimensionModel == GAIA_XY_Z_M)
	*z = geom->FirstPoint->Z;
    else
	*z = 0.0;
    if (geom->DimensionModel == GAIA_XY_M
	|| geom->DimensionModel == GAIA_XY_Z_M)
	*m = geom->FirstPoint->M;
    else
	*m = 0.0;
    return 1;
}

static void
fnct_SnapToGrid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_SnapToGrid(BLOBencoded geom, double size)
/ ST_SnapToGrid(BLOBencoded geom, double sizeX, double sizeY)
/ ST_SnapToGrid(BLOBencoded geom, double originX, double originY, 
/               double sizeX, double sizeY)
/
/ Snap all points of the input geometry to the grid defined by its 
/ origin and cell size. Remove consecutive points falling on the same
/ cell. Collapsed geometries in a collection are stripped from it.
/
/
/ ST_SnapToGrid(BLOBencoded geom, BLOBencoded point, double sizeX,
/               double sizeY, double sizeZ, double sizeM)
/
/ Snap all points of the input geometry to the grid defined by its 
/ origin (the second argument, must be a point) and cell sizes.
/ 
/ Specify 0 as size for any dimension you don't want to snap to
/ a grid.
/ return NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double origin_x = 0.0;
    double origin_y = 0.0;
    double origin_z = 0.0;
    double origin_m = 0.0;
    double size_x = 0.0;
    double size_y = 0.0;
    double size_z = 0.0;
    double size_m = 0.0;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr point = NULL;
    gaiaGeomCollPtr result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  /* ST_SnapToGrid(BLOBencoded geom, double size) */
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[1]);
		size_x = int_value;
		size_y = size_x;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		size_x = sqlite3_value_double (argv[1]);
		size_y = size_x;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 3)
      {
	  /* ST_SnapToGrid(BLOBencoded geom, double sizeX, double sizeY) */
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[1]);
		size_x = int_value;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		size_x = sqlite3_value_double (argv[1]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[2]);
		size_y = int_value;
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	    {
		size_y = sqlite3_value_double (argv[2]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 5)
      {
	  /*
	     / ST_SnapToGrid(BLOBencoded geom, double originX, double originY, 
	     /               double sizeX, double sizeY)
	   */
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[1]);
		origin_x = int_value;
	    }
	  else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	    {
		origin_x = sqlite3_value_double (argv[1]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[2]);
		origin_y = int_value;
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	    {
		origin_y = sqlite3_value_double (argv[2]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[3]);
		size_x = int_value;
	    }
	  else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	    {
		size_x = sqlite3_value_double (argv[3]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[4]);
		size_y = int_value;
	    }
	  else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	    {
		size_y = sqlite3_value_double (argv[4]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 6)
      {
	  /*
	     / ST_SnapToGrid(BLOBencoded geom, BLOBencoded point, double sizeX,
	     /               double sizeY, double sizeZ, double sizeM)
	   */
	  if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
	  n_bytes = sqlite3_value_bytes (argv[1]);
	  point =
	      gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
					   gpkg_amphibious);
	  if (!point)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (!getXYZMSinglePoint
	      (point, &origin_x, &origin_y, &origin_z, &origin_m))
	    {
		gaiaFreeGeomColl (point);
		sqlite3_result_null (context);
		return;
	    }
	  gaiaFreeGeomColl (point);
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[2]);
		size_x = int_value;
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	    {
		size_x = sqlite3_value_double (argv[2]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[3]);
		size_y = int_value;
	    }
	  else if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	    {
		size_y = sqlite3_value_double (argv[3]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[4]);
		size_z = int_value;
	    }
	  else if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	    {
		size_z = sqlite3_value_double (argv[4]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[5]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[5]);
		size_m = int_value;
	    }
	  else if (sqlite3_value_type (argv[5]) == SQLITE_FLOAT)
	    {
		size_m = sqlite3_value_double (argv[5]);
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  result =
	      gaiaSnapToGrid (geo, origin_x, origin_y, origin_z, origin_m,
			      size_x, size_y, size_z, size_m);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static char garsMapping[24] = { 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J',
    'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'
};

static char
garsLetterCode (int value)
{
    return garsMapping[value];
}

static int
garsMappingIndex (const char letter)
{
    int i = 0;
    for (i = 0; i < 24; ++i)
      {
	  if (letter == garsMapping[i])
	      return i;
      }
    return -1;
}

static double
garsLetterToDegreesLat (char msd, char lsd)
{
    double high = garsMappingIndex (msd) * 24.0;
    double low = garsMappingIndex (lsd);
    if ((high < 0) || (low < 0))
      {
	  return -100.0;
      }
    return (((high + low) * 0.5) - 90.0);
}


static void
fnct_GARSMbr (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GARSMbr(Text)
/
/ converts the Text (which should be a valid GARS area) to the corresponding
/ MBR geometry.
/ This function will return NULL if an error occurs
*/
    const char *text = NULL;
    int len = 0;
    unsigned char *p_result = NULL;
    double x1 = 0.0;
    double y1 = 0.0;
    double x2 = 0.0;
    double y2 = 0.0;

    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = (const char *) sqlite3_value_text (argv[0]);
    if ((strlen (text) < 5) || (strlen (text) > 7))
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (strlen (text) == 5)
      {
	  int numMatch = 0;
	  unsigned int digit100 = 0;
	  char letterMSD = '\0';
	  char letterLSD = '\0';
	  numMatch = sscanf (text, "%u%c%c", &digit100, &letterMSD, &letterLSD);
	  if (numMatch != 3)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  x1 = ((digit100 - 1) * 0.5) - 180.0;
	  y1 = garsLetterToDegreesLat (letterMSD, letterLSD);
	  if ((x1 < -180.0) || (x1 > 179.5) || (y1 < -90.0) || (y1 > 89.5))
	    {
		sqlite3_result_null (context);
		return;
	    }
	  x2 = x1 + 0.5;
	  y2 = y1 + 0.5;
      }
    if (strlen (text) == 6)
      {
	  unsigned int numMatch = 0;
	  unsigned int digit100 = 0;
	  char letterMSD = '\0';
	  char letterLSD = '\0';
	  unsigned int digitSegment = 0;
	  numMatch =
	      sscanf (text, "%u%c%c%u", &digit100, &letterMSD, &letterLSD,
		      &digitSegment);
	  if (numMatch != 4)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if ((digitSegment < 1) || (digitSegment > 4))
	    {
		sqlite3_result_null (context);
		return;
	    }
	  x1 = ((digit100 - 1) * 0.5) - 180.0;
	  if ((digitSegment == 2) || (digitSegment == 4))
	    {
		x1 += 0.25;
	    }
	  y1 = garsLetterToDegreesLat (letterMSD, letterLSD);
	  if ((digitSegment == 1) || (digitSegment == 2))
	    {
		y1 += 0.25;
	    }
	  if ((x1 < -180.0) || (x1 > 179.75) || (y1 < -90.0) || (y1 > 89.75))
	    {
		sqlite3_result_null (context);
		return;
	    }
	  x2 = x1 + 0.25;
	  y2 = y1 + 0.25;
      }
    if (strlen (text) == 7)
      {
	  unsigned int numMatch = 0;
	  unsigned int digit100 = 0;
	  char letterMSD = '\0';
	  char letterLSD = '\0';
	  unsigned int digitAndKeypad = 0;
	  unsigned int digitSegment = 0;
	  unsigned int keypadNumber = 0;
	  numMatch =
	      sscanf (text, "%u%c%c%u", &digit100, &letterMSD, &letterLSD,
		      &digitAndKeypad);
	  if (numMatch != 4)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  digitSegment = digitAndKeypad / 10;
	  keypadNumber = digitAndKeypad % 10;
	  if ((digitSegment < 1) || (digitSegment > 4))
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (keypadNumber < 1)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  x1 = ((digit100 - 1) * 0.5) - 180.0;
	  if ((digitSegment == 2) || (digitSegment == 4))
	    {
		x1 += 0.25;
	    }
	  y1 = garsLetterToDegreesLat (letterMSD, letterLSD);
	  if ((digitSegment == 1) || (digitSegment == 2))
	    {
		y1 += 0.25;
	    }
	  switch (keypadNumber)
	    {
	    case 1:
		x1 += 0 * 0.25 / 3;
		y1 += 2 * 0.25 / 3;
		break;
	    case 2:
		x1 += 1 * 0.25 / 3;
		y1 += 2 * 0.25 / 3;
		break;
	    case 3:
		x1 += 2 * 0.25 / 3;
		y1 += 2 * 0.25 / 3;
		break;
	    case 4:
		x1 += 0 * 0.25 / 3;
		y1 += 1 * 0.25 / 3;
		break;
	    case 5:
		x1 += 1 * 0.25 / 3;
		y1 += 1 * 0.25 / 3;
		break;
	    case 6:
		x1 += 2 * 0.25 / 3;
		y1 += 1 * 0.25 / 3;
		break;
	    case 7:
		x1 += 0 * 0.25 / 3;
		y1 += 0 * 0.25 / 3;
		break;
	    case 8:
		x1 += 1 * 0.25 / 3;
		y1 += 0 * 0.25 / 3;
		break;
	    case 9:
		x1 += 2 * 0.25 / 3;
		y1 += 0 * 0.25 / 3;
		break;
	    }
	  if ((x1 < -180.0) || (x1 >= 180.0) || (y1 < -90.0) || (y1 >= 90.0))
	    {
		sqlite3_result_null (context);
		return;
	    }
	  x2 = x1 + (0.25 / 3);
	  y2 = y1 + (0.25 / 3);
      }
    gaiaBuildMbr (x1, y1, x2, y2, 4326, &p_result, &len);
    if (!p_result)
      {
	  sqlite3_result_null (context);
	  spatialite_e ("bad p_result\n");
      }
    else
	sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_ToGARS (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ToGARS(BLOB encoded POINT)
/
/ returns the Global Area Reference System coordinate area for a given point,
/ or NULL if an error occurs
*/
    unsigned char *p_blob;
    int n_bytes;
    int pts = 0;
    int lns = 0;
    int pgs = 0;
    gaiaPointPtr point;
    gaiaLinestringPtr line;
    gaiaPolygonPtr polyg;
    gaiaGeomCollPtr geo = NULL;
    char p_result[8];
    int lon_band = 0;
    double lon_minutes = 0;
    int segmentNumber = 0;
    int lat_band = 0;
    double lat_minutes = 0;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaNormalizeLonLat (geo);
    point = geo->FirstPoint;
    while (point != NULL)
      {
	  pts++;
	  point = point->Next;
      }
    line = geo->FirstLinestring;
    while (line != NULL)
      {
	  lns++;
	  line = line->Next;
      }
    polyg = geo->FirstPolygon;
    while (polyg != NULL)
      {
	  pgs++;
	  polyg = polyg->Next;
      }
    if (pts == 1 && lns == 0 && pgs == 0)
	point = geo->FirstPoint;
    else
      {
	  /* not a single Point */
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    /* longitude band */
    lon_band = 1 + (int) ((point->X + 180.0) * 2);
    sprintf (p_result, "%03i", lon_band);
    /* latitude band */
    lat_band = (int) ((point->Y + 90.0) * 2);
    p_result[3] = garsLetterCode (lat_band / 24);
    p_result[4] = garsLetterCode (lat_band % 24);
    /* quadrant */
    lon_minutes = fmod ((point->X + 180.0), 0.5) * 60.0;
    if (lon_minutes < 15.0)
      {
	  segmentNumber = 1;
      }
    else
      {
	  segmentNumber = 2;
	  lon_minutes -= 15.0;
      }
    lat_minutes = fmod ((point->Y + 90.0), 0.5) * 60.0;
    if (lat_minutes < 15.0)
      {
	  segmentNumber += 2;
      }
    else
      {
	  /* we already have the right segment */
	  lat_minutes -= 15.0;
      }
    sprintf (&(p_result[5]), "%i", segmentNumber);
    /* area */
    segmentNumber = 0;
    if (lon_minutes >= 10.0)
      {
	  segmentNumber = 3;
      }
    else if (lon_minutes >= 5.0)
      {
	  segmentNumber = 2;
      }
    else
      {
	  segmentNumber = 1;
      }
    if (lat_minutes >= 10.0)
      {
	  /* nothing to add */
      }
    else if (lat_minutes >= 5.0)
      {
	  segmentNumber += 3;
      }
    else
      {
	  segmentNumber += 6;
      }
    sprintf (&(p_result[6]), "%i", segmentNumber);
    sqlite3_result_text (context, p_result, 7, SQLITE_TRANSIENT);
    gaiaFreeGeomColl (geo);
}

static void
fnct_GeometryN (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeometryN(BLOB encoded GEOMETRYCOLLECTION geometry)
/
/ returns the Nth geometry for current GEOMETRYCOLLECTION or MULTIxxxx geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int entity;
    int len;
    int cnt = 0;
    int iv;
    int ib;
    double x;
    double y;
    double z;
    double m;
    gaiaPointPtr point;
    gaiaLinestringPtr line;
    gaiaLinestringPtr line2;
    gaiaPolygonPtr polyg;
    gaiaPolygonPtr polyg2;
    gaiaRingPtr ring_in;
    gaiaRingPtr ring_out;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    entity = sqlite3_value_int (argv[1]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  point = geo->FirstPoint;
	  while (point)
	    {
		/* counts how many points are there */
		cnt++;
		if (cnt == entity)
		  {
		      /* ok, required elementary geometry is this POINT */
		      if (point->DimensionModel == GAIA_XY_Z)
			  result = gaiaAllocGeomCollXYZ ();
		      else if (point->DimensionModel == GAIA_XY_M)
			  result = gaiaAllocGeomCollXYM ();
		      else if (point->DimensionModel == GAIA_XY_Z_M)
			  result = gaiaAllocGeomCollXYZM ();
		      else
			  result = gaiaAllocGeomColl ();
		      result->Srid = geo->Srid;
		      if (point->DimensionModel == GAIA_XY_Z)
			  gaiaAddPointToGeomCollXYZ (result, point->X,
						     point->Y, point->Z);
		      else if (point->DimensionModel == GAIA_XY_M)
			  gaiaAddPointToGeomCollXYM (result, point->X,
						     point->Y, point->M);
		      else if (point->DimensionModel == GAIA_XY_Z_M)
			  gaiaAddPointToGeomCollXYZM (result, point->X,
						      point->Y, point->Z,
						      point->M);
		      else
			  gaiaAddPointToGeomColl (result, point->X, point->Y);
		      goto skip;
		  }
		point = point->Next;
	    }
	  line = geo->FirstLinestring;
	  while (line)
	    {
		/* counts how many linestrings are there */
		cnt++;
		if (cnt == entity)
		  {
		      /* ok, required elementary geometry is this LINESTRING */
		      if (line->DimensionModel == GAIA_XY_Z)
			  result = gaiaAllocGeomCollXYZ ();
		      else if (line->DimensionModel == GAIA_XY_M)
			  result = gaiaAllocGeomCollXYM ();
		      else if (line->DimensionModel == GAIA_XY_Z_M)
			  result = gaiaAllocGeomCollXYZM ();
		      else
			  result = gaiaAllocGeomColl ();
		      result->Srid = geo->Srid;
		      line2 =
			  gaiaAddLinestringToGeomColl (result, line->Points);
		      for (iv = 0; iv < line2->Points; iv++)
			{
			    if (line->DimensionModel == GAIA_XY_Z)
			      {
				  gaiaGetPointXYZ (line->Coords, iv, &x,
						   &y, &z);
				  gaiaSetPointXYZ (line2->Coords, iv, x, y, z);
			      }
			    else if (line->DimensionModel == GAIA_XY_M)
			      {
				  gaiaGetPointXYM (line->Coords, iv, &x,
						   &y, &m);
				  gaiaSetPointXYM (line2->Coords, iv, x, y, m);
			      }
			    else if (line->DimensionModel == GAIA_XY_Z_M)
			      {
				  gaiaGetPointXYZM (line->Coords, iv, &x,
						    &y, &z, &m);
				  gaiaSetPointXYZM (line2->Coords, iv, x,
						    y, z, m);
			      }
			    else
			      {
				  gaiaGetPoint (line->Coords, iv, &x, &y);
				  gaiaSetPoint (line2->Coords, iv, x, y);
			      }
			}
		      goto skip;
		  }
		line = line->Next;
	    }
	  polyg = geo->FirstPolygon;
	  while (polyg)
	    {
		/* counts how many polygons are there */
		cnt++;
		if (cnt == entity)
		  {
		      /* ok, required elementary geometry is this POLYGON */
		      if (polyg->DimensionModel == GAIA_XY_Z)
			  result = gaiaAllocGeomCollXYZ ();
		      else if (polyg->DimensionModel == GAIA_XY_M)
			  result = gaiaAllocGeomCollXYM ();
		      else if (polyg->DimensionModel == GAIA_XY_Z_M)
			  result = gaiaAllocGeomCollXYZM ();
		      else
			  result = gaiaAllocGeomColl ();
		      result->Srid = geo->Srid;
		      ring_in = polyg->Exterior;
		      polyg2 =
			  gaiaAddPolygonToGeomColl (result,
						    ring_in->Points,
						    polyg->NumInteriors);
		      ring_out = polyg2->Exterior;
		      for (iv = 0; iv < ring_out->Points; iv++)
			{
			    /* copying the exterior ring POINTs */
			    if (ring_in->DimensionModel == GAIA_XY_Z)
			      {
				  gaiaGetPointXYZ (ring_in->Coords, iv,
						   &x, &y, &z);
				  gaiaSetPointXYZ (ring_out->Coords, iv,
						   x, y, z);
			      }
			    else if (ring_in->DimensionModel == GAIA_XY_M)
			      {
				  gaiaGetPointXYM (ring_in->Coords, iv,
						   &x, &y, &m);
				  gaiaSetPointXYM (ring_out->Coords, iv,
						   x, y, m);
			      }
			    else if (ring_in->DimensionModel == GAIA_XY_Z_M)
			      {
				  gaiaGetPointXYZM (ring_in->Coords, iv,
						    &x, &y, &z, &m);
				  gaiaSetPointXYZM (ring_out->Coords, iv,
						    x, y, z, m);
			      }
			    else
			      {
				  gaiaGetPoint (ring_in->Coords, iv, &x, &y);
				  gaiaSetPoint (ring_out->Coords, iv, x, y);
			      }
			}
		      for (ib = 0; ib < polyg2->NumInteriors; ib++)
			{
			    /* processing the interior rings */
			    ring_in = polyg->Interiors + ib;
			    ring_out =
				gaiaAddInteriorRing (polyg2, ib,
						     ring_in->Points);
			    for (iv = 0; iv < ring_out->Points; iv++)
			      {
				  if (ring_in->DimensionModel == GAIA_XY_Z)
				    {
					gaiaGetPointXYZ (ring_in->Coords,
							 iv, &x, &y, &z);
					gaiaSetPointXYZ (ring_out->Coords, iv,
							 x, y, z);
				    }
				  else if (ring_in->DimensionModel == GAIA_XY_M)
				    {
					gaiaGetPointXYM (ring_in->Coords,
							 iv, &x, &y, &m);
					gaiaSetPointXYM (ring_out->Coords, iv,
							 x, y, m);
				    }
				  else if (ring_in->DimensionModel ==
					   GAIA_XY_Z_M)
				    {
					gaiaGetPointXYZM (ring_in->Coords, iv,
							  &x, &y, &z, &m);
					gaiaSetPointXYZM (ring_out->Coords,
							  iv, x, y, z, m);
				    }
				  else
				    {
					gaiaGetPoint (ring_in->Coords,
						      iv, &x, &y);
					gaiaSetPoint (ring_out->Coords,
						      iv, x, y);
				    }
			      }
			}
		      goto skip;
		  }
		polyg = polyg->Next;
	    }
	skip:
	  if (result)
	    {
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		gaiaFreeGeomColl (result);
		sqlite3_result_blob (context, p_result, len, free);
	    }
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo);
}

static void
mbrs_eval (sqlite3_context * context, int argc, sqlite3_value ** argv,
	   int request)
{
/* SQL function:
/ MBRsomething(BLOB encoded GEOMETRY-1, BLOB encoded GEOMETRY-2)
/
/ returns:
/ 1 if the required spatial relationship between the two MBRs is TRUE
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int ret;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 = gaiaFromSpatiaLiteBlobMbr (p_blob, n_bytes);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 = gaiaFromSpatiaLiteBlobMbr (p_blob, n_bytes);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  ret = 0;
	  gaiaMbrGeometry (geo1);
	  gaiaMbrGeometry (geo2);
	  switch (request)
	    {
	    case GAIA_MBR_CONTAINS:
		ret = gaiaMbrsContains (geo1, geo2);
		break;
	    case GAIA_MBR_DISJOINT:
		ret = gaiaMbrsDisjoint (geo1, geo2);
		break;
	    case GAIA_MBR_EQUAL:
		ret = gaiaMbrsEqual (geo1, geo2);
		break;
	    case GAIA_MBR_INTERSECTS:
		ret = gaiaMbrsIntersects (geo1, geo2);
		break;
	    case GAIA_MBR_OVERLAPS:
		ret = gaiaMbrsOverlaps (geo1, geo2);
		break;
	    case GAIA_MBR_TOUCHES:
		ret = gaiaMbrsTouches (geo1, geo2);
		break;
	    case GAIA_MBR_WITHIN:
		ret = gaiaMbrsWithin (geo1, geo2);
		break;
	    }
	  if (ret < 0)
	      sqlite3_result_int (context, -1);
	  else
	      sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

/*
/ the following functions simply readdress the mbr_eval()
/ setting the appropriate request mode
*/

static void
fnct_MbrContains (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    mbrs_eval (context, argc, argv, GAIA_MBR_CONTAINS);
}

static void
fnct_MbrDisjoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    mbrs_eval (context, argc, argv, GAIA_MBR_DISJOINT);
}

static void
fnct_MbrEqual (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    mbrs_eval (context, argc, argv, GAIA_MBR_EQUAL);
}

static void
fnct_MbrIntersects (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    mbrs_eval (context, argc, argv, GAIA_MBR_INTERSECTS);
}

static void
fnct_EnvIntersects (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_EnvIntersects(Geometry geom, double X1, double Y1, double X2, double Y2)
/ ST_EnvelopesIntersects(Geometry geom, double X1, double Y1, double X2, double Y2)
/
/ the second MBR is defined by two points (identifying a rectangle's diagonal) 
/
/ returns:
/ 1 if the required spatial intersection is TRUE
/ 0 otherwise
/ or -1 if any error is encountered
*/
    double x1;
    double y1;
    double x2;
    double y2;
    int int_value;
    unsigned char *p_blob;
    int n_bytes;
    int ret = 0;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaLinestringPtr ln;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	x1 = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  x1 = int_value;
      }
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	y1 = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  y1 = int_value;
      }
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	x2 = sqlite3_value_double (argv[3]);
    else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[3]);
	  x2 = int_value;
      }
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_FLOAT)
	y2 = sqlite3_value_double (argv[4]);
    else if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[4]);
	  y2 = int_value;
      }
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1)
	sqlite3_result_int (context, -1);
    else
      {
	  gaiaMbrGeometry (geo1);
	  geo2 = gaiaAllocGeomColl ();
	  ln = gaiaAddLinestringToGeomColl (geo2, 2);
	  gaiaSetPoint (ln->Coords, 0, x1, y1);
	  gaiaSetPoint (ln->Coords, 1, x2, y2);
	  gaiaMbrGeometry (geo2);
	  ret = gaiaMbrsIntersects (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}


static void
fnct_MbrOverlaps (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    mbrs_eval (context, argc, argv, GAIA_MBR_OVERLAPS);
}

static void
fnct_MbrTouches (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    mbrs_eval (context, argc, argv, GAIA_MBR_TOUCHES);
}

static void
fnct_MbrWithin (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    mbrs_eval (context, argc, argv, GAIA_MBR_WITHIN);
}

static void
fnct_ShiftCoords (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ShiftCoords(BLOBencoded geometry, shiftX, shiftY)
/
/ returns a new geometry that is the original one received, but with shifted coordinates
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    double shift_x;
    double shift_y;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	shift_x = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  shift_x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	shift_y = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  shift_y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaShiftCoords (geo, shift_x, shift_y);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Translate (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Translate(BLOBencoded geometry, shiftX, shiftY, shiftZ)
/
/ returns a new geometry that is the original one received, but with shifted coordinates
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    double shift_x;
    double shift_y;
    double shift_z;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	shift_x = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  shift_x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	shift_y = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  shift_y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	shift_z = sqlite3_value_double (argv[3]);
    else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[3]);
	  shift_z = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaShiftCoords3D (geo, shift_x, shift_y, shift_z);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}


static void
fnct_ShiftLongitude (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ShiftLongitude(BLOBencoded geometry)
/
/ returns a new geometry that is the original one received, but with negative
/ longitudes shifted by 360
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaShiftLongitude (geo);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_NormalizeLonLat (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ NormalizeLonLat (BLOBencoded geometry)
/
/ returns a new geometry that is the original one received, but with longitude
/ and latitude values shifted into the range [-180 - 180, -90 - 90]. 
/ NULL is returned if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaNormalizeLonLat (geo);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ScaleCoords (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ScaleCoords(BLOBencoded geometry, scale_factor_x [, scale_factor_y])
/
/ returns a new geometry that is the original one received, but with scaled coordinates
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    double scale_x;
    double scale_y;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	scale_x = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  scale_x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
	scale_y = scale_x;	/* this one is an isotropic scaling request */
    else
      {
	  /* an anisotropic scaling is requested */
	  if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      scale_y = sqlite3_value_double (argv[2]);
	  else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[2]);
		scale_y = int_value;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaScaleCoords (geo, scale_x, scale_y);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_RotateCoords (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ RotateCoords(BLOBencoded geometry, angle)
/
/ returns a new geometry that is the original one received, but with rotated coordinates
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    double angle;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	angle = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  angle = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaRotateCoords (geo, angle);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ReflectCoords (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ReflectCoords(BLOBencoded geometry, x_axis,  y_axis)
/
/ returns a new geometry that is the original one received, but with mirrored coordinates
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int x_axis;
    int y_axis;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	x_axis = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	y_axis = sqlite3_value_int (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaReflectCoords (geo, x_axis, y_axis);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_SwapCoords (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SwapCoords(BLOBencoded geometry)
/
/ returns a new geometry that is the original one received, but with swapped x- and y-coordinate
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaSwapCoords (geo);
	  gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode,
				      tiny_point);
	  if (!p_result)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (geo);
}

SPATIALITE_PRIVATE int
getEllipsoidParams (void *p_sqlite, int srid, double *a, double *b, double *rf)
{
/* 
/ retrieves the PROJ +ellps=xx [+a=xx +b=xx] params 
/from SPATIAL_SYS_REF table, if possible 
*/
    sqlite3 *sqlite = (sqlite3 *) p_sqlite;
    char *proj4text;
    char *p_proj;
    char *p_ellps;
    char *p_datum;
    char *p_a;
    char *p_b;
    char *p_end;

    if (srid == 0)
      {
	  /* 
	     / SRID=0 is formally defined as "Undefined Geographic"
	     / so will default to SRID=4326 (WGS84 Long/Lat)
	   */
	  srid = 4326;
      }
    getProjParams (sqlite, srid, &proj4text);
    if (proj4text == NULL)
	return 0;
/* parsing the proj4text geodesic string */
    p_proj = strstr (proj4text, "+proj=");
    p_datum = strstr (proj4text, "+datum=");
    p_ellps = strstr (proj4text, "+ellps=");
    p_a = strstr (proj4text, "+a=");
    p_b = strstr (proj4text, "+b=");
/* checking if +proj=longlat is true */
    if (!p_proj)
	goto invalid;
    p_end = strchr (p_proj, ' ');
    if (p_end)
	*p_end = '\0';
    if (strcmp (p_proj + 6, "longlat") != 0)
	goto invalid;
    if (p_ellps)
      {
	  /* trying to retrieve the ellipsoid params by name */
	  p_end = strchr (p_ellps, ' ');
	  if (p_end)
	      *p_end = '\0';
	  if (gaiaEllipseParams (p_ellps + 7, a, b, rf))
	      goto valid;
      }
    else if (p_datum)
      {
	  /*
	     / starting since GDAL 1.9.0 the WGS84 [4326] PROJ.4 def doesn't 
	     / declares any longer the "+ellps=" param
	     / in this case we'll attempt to recover using "+datum=".
	   */
	  p_end = strchr (p_datum, ' ');
	  if (p_end)
	      *p_end = '\0';
	  if (gaiaEllipseParams (p_datum + 7, a, b, rf))
	      goto valid;
      }
    if (p_a && p_b)
      {
	  /* trying to retrieve the +a=xx and +b=xx args */
	  p_end = strchr (p_a, ' ');
	  if (p_end)
	      *p_end = '\0';
	  p_end = strchr (p_b, ' ');
	  if (p_end)
	      *p_end = '\0';
	  *a = atof (p_a + 3);
	  *b = atof (p_b + 3);
	  *rf = 1.0 / ((*a - *b) / *a);
	  goto valid;
      }

  valid:
    free (proj4text);
    return 1;

  invalid:
    free (proj4text);
    return 0;
}

static void
fnct_FromEWKB (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromEWKB(EWKB encoded geometry)
/
/ returns the current geometry by parsing Geos/PostGis EWKB encoded string 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaFromEWKB (text);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_ToEWKB (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsEWKB(BLOB encoded geometry)
/
/ returns a text string corresponding to Geos/PostGIS EWKB notation 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  gaiaOutBufferInitialize (&out_buf);
	  gaiaToEWKB (&out_buf, geo);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_ToEWKT (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsEWKT(BLOB encoded geometry)
/
/ returns the corresponding PostGIS EWKT encoded value
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaOutBuffer out_buf;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    gaiaOutBufferInitialize (&out_buf);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  gaiaToEWKT (&out_buf, geo);
	  if (out_buf.Error || out_buf.Buffer == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		len = out_buf.WriteOffset;
		sqlite3_result_text (context, out_buf.Buffer, len, free);
		out_buf.Buffer = NULL;
	    }
      }
    gaiaFreeGeomColl (geo);
    gaiaOutBufferReset (&out_buf);
}

static void
fnct_FromEWKT (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromEWKT(EWKT encoded geometry)
/
/ returns the current geometry by parsing EWKT  (PostGIS) encoded string 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseEWKT (text);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_FromGeoJSON (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromGeoJSON(GeoJSON encoded geometry)
/
/ returns the current geometry by parsing GeoJSON encoded string 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseGeoJSON (text);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_FromKml (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromKml(KML encoded geometry)
/
/ returns the current geometry by parsing KML encoded string 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseKml (text);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_FromGml (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromGml(GML encoded geometry)
/
/ returns the current geometry by parsing GML encoded string 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    void *data = sqlite3_user_data (context);
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    if (data != NULL)
	geo = gaiaParseGml_r (data, text, sqlite);
    else
	geo = gaiaParseGml (text, sqlite);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_LinesFromRings (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ LinesFromRings(BLOBencoded geometry, BOOL multi_linestring)
/
/ returns a new geometry [LINESTRING or MULTILINESTRING] representing 
/ the linearization for current (MULTI)POLYGON geometry
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr geom_new = NULL;
    int len;
    int multi_linestring = 0;
    unsigned char *p_result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      multi_linestring = sqlite3_value_int (argv[1]);
      }
    geom_new = gaiaLinearize (geo, multi_linestring);
    if (!geom_new)
	goto invalid;
    gaiaFreeGeomColl (geo);
    gaiaToSpatiaLiteBlobWkbEx2 (geom_new, &p_result, &len, gpkg_mode,
				tiny_point);
    gaiaFreeGeomColl (geom_new);
    sqlite3_result_blob (context, p_result, len, free);
    return;
  invalid:
    if (geo)
	gaiaFreeGeomColl (geo);
    sqlite3_result_null (context);
}

#ifndef OMIT_GEOS		/* including GEOS */

static void
fnct_BuildArea (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ BuildArea(BLOBencoded geometry)
/
/ Assuming that Geometry represents a set of sparse Linestrings,
/ this function will attempt to reassemble a single Polygon
/ (or a set of Polygons)
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaPolygonize_r (data, geo, 0);
	  else
	      result = gaiaPolygonize (geo, 0);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}


static void
fnct_Polygonize_step (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ Polygonize(BLOBencoded geom)
/
/ aggregate function - STEP
/
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    gaiaGeomCollPtr result;
    gaiaGeomCollPtr *p;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	return;
    p = sqlite3_aggregate_context (context, sizeof (gaiaGeomCollPtr));
    if (!(*p))
      {
	  /* this is the first row */
	  *p = geom;
      }
    else
      {
	  /* subsequent rows */
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMergeGeometries_r (data, *p, geom);
	  else
	      result = gaiaMergeGeometries (*p, geom);
	  *p = result;
	  gaiaFreeGeomColl (geom);
      }
}

static void
fnct_Polygonize_final (sqlite3_context * context)
{
/* SQL function:
/ Polygonize(BLOBencoded geom)
/
/ aggregate function - FINAL
/
*/
    gaiaGeomCollPtr result;
    gaiaGeomCollPtr geom;
    gaiaGeomCollPtr *p = sqlite3_aggregate_context (context, 0);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    result = *p;
    if (!result)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      geom = gaiaPolygonize_r (data, result, 0);
	  else
	      geom = gaiaPolygonize (result, 0);
	  if (geom == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		geom->Srid = result->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (geom, &p_result, &len, gpkg_mode,
					    tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (geom);
	    }
	  gaiaFreeGeomColl (result);
      }
}

#endif /* end including GEOS */

static void
fnct_DissolveSegments (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ DissolveSegments(BLOBencoded geometry)
/
/ Dissolves any LINESTRING or RING into elementary segments
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  result = gaiaDissolveSegments (geo);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_DissolvePoints (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ DissolvePoints(BLOBencoded geometry)
/
/ Dissolves any LINESTRING or RING into elementary Vertices
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  result = gaiaDissolvePoints (geo);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_CollectionExtract (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ CollectionExtract(BLOBencoded geometry, Integer type)
/
/ Extracts from a GEOMETRYCOLLECTION any item of the required TYPE
/ 1=Point - 2=Linestring - 3=Polygon
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    int type;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	type = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (type == 1 || type == 2 || type == 3)
	;
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  switch (type)
	    {
	    case 1:
		result = gaiaExtractPointsFromGeomColl (geo);
		break;
	    case 2:
		result = gaiaExtractLinestringsFromGeomColl (geo);
		break;
	    case 3:
		result = gaiaExtractPolygonsFromGeomColl (geo);
		break;
	    };
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_AddMeasure (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL functions:
/ ST_AddMeasure(BLOBencoded geometry, Double m_start, Double m_end)
/
/ Will return a new GEOMETRY (supporting M) with measures linearly
/ interpolated between the start and end points.
/ the input Geometry is expected to be a Linestring or MultiLinestring
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    double m_start;
    double m_end;
    int intval;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	m_start = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  intval = sqlite3_value_int (argv[1]);
	  m_start = intval;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	m_end = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  intval = sqlite3_value_int (argv[2]);
	  m_end = intval;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  result = gaiaAddMeasure (geo, m_start, m_end);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_InterpolatePoint (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL functions:
/ ST_InterpolatePoint(BLOBencoded line-geometry, BLOBencoded point-geometry)
/
/ Will return the M-value at the interpolated point nearest to the given point
/ the first input Geometry is expected to be a Linestring supporting M,
/ the second input Geometry is expected to be a Point
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob1;
    int n_bytes1;
    unsigned char *p_blob2;
    int n_bytes2;
    gaiaGeomCollPtr line = NULL;
    gaiaGeomCollPtr point = NULL;
    double m_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes1 = sqlite3_value_bytes (argv[0]);
    line =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob1, n_bytes1, gpkg_mode,
				     gpkg_amphibious);
    p_blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes2 = sqlite3_value_bytes (argv[1]);
    point =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob2, n_bytes2, gpkg_mode,
				     gpkg_amphibious);
    if (line == NULL || point == NULL)
	sqlite3_result_null (context);
    else
      {
	  if (!gaiaInterpolatePoint (cache, line, point, &m_value))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, m_value);
      }
    if (line != NULL)
	gaiaFreeGeomColl (line);
    if (point != NULL)
	gaiaFreeGeomColl (point);
}

static void
fnct_LocateBetweenMeasures (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL functions:
/ ST_Locate_Along_Measure(BLOBencoded geometry, Double m_value)
/ ST_LocateAlong(BLOBencoded geometry, Double m_value)
/ ST_Locate_Between_Measures(BLOBencoded geometry, Double m_start, Double m_end)
/ ST_LocateBetween(BLOBencoded geometry, Double m_start, Double m_end)
/
/ Extracts from a GEOMETRY (supporting M) any Point/Linestring
/ matching the range of measures
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    double m_start;
    double m_end;
    int intval;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	m_start = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  intval = sqlite3_value_int (argv[1]);
	  m_start = intval;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 2)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      m_end = sqlite3_value_double (argv[2]);
	  else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		intval = sqlite3_value_int (argv[2]);
		m_end = intval;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    else
	m_end = m_start;
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  result = gaiaLocateBetweenMeasures (geo, m_start, m_end);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_IsValidTrajectory (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ ST_IsValidTrajectory(BLOBencoded geometry)
/
/ returns:
/ 1 if this GEOMETRY is a LINESTRING supporting M-values and
/   presenting M-values growing from o vertex to the next
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int intval;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_int (context, -1);
    else
      {
	  intval = gaiaIsValidTrajectory (geo);
	  sqlite3_result_int (context, intval);
	  gaiaFreeGeomColl (geo);
      }
}

static void
fnct_TrajectoryInterpolatePoint (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ ST_TrajectoryInterpolatePoint(BLOBencoded geometry, DOUBLE m)
/
/ returns a new geometry representing a point interpolated along a Trajectory
/ accordingly to given M-Value
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int intval;
    double m;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  intval = sqlite3_value_int (argv[1]);
	  m = intval;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	m = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_int (context, -1);
    else
      {
	  intval = gaiaIsValidTrajectory (geo);
	  result = gaiaTrajectoryInterpolatePoint (geo, m);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
	  gaiaFreeGeomColl (geo);
      }
}

#ifndef OMIT_PROJ		/* including PROJ.4 */

static void
fnct_Transform (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Transform(BLOBencoded geometry, srid)
/ Transform(BLOBencoded geometry, srid, BBOXgeometry)
/ Transform(BLOBencoded geometry, srid, BBOXgeometry, string_1)
/ Transform(BLOBencoded geometry, srid, BBOXgeometry, string_1, string_2)
/
/ returns a new geometry that is the original one received, but 
/ transformed / translated to the new SRID [coordinates translation 
/ is applied]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int srid_from;
    int srid_to;
    char *proj_from = NULL;
    char *proj_to = NULL;
    const char *proj_string_1 = NULL;
    const char *proj_string_2 = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    const char *msg;
    int check_origin_destination = 0;
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
    char *msg2;
    gaiaGeomCollPtr bbox = NULL;
    gaiaProjArea proj_area;
    gaiaProjAreaPtr proj_bbox = NULL;
#else /* supporting old PROJ.4 */
#endif
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	srid_to = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 3)
      {
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
	  if (sqlite3_value_type (argv[2]) == SQLITE_NULL)
	      ;
	  else if (sqlite3_value_type (argv[2]) != SQLITE_BLOB)
	    {
		msg =
		    "ST_Transform exception - 3rd argument is neither a BLOB nor NULL.";
		sqlite3_result_error (context, msg, -1);
		return;
	    }
	  else
	    {
		p_blob = (unsigned char *) sqlite3_value_blob (argv[2]);
		n_bytes = sqlite3_value_bytes (argv[2]);
		bbox = gaiaFromSpatiaLiteBlobWkb (p_blob, n_bytes);
		if (!bbox)
		  {
		      gaiaFreeGeomColl (bbox);
		      msg =
			  "ST_Transform exception - 3rd argument is not a valid BLOB Geometry.";
		      sqlite3_result_error (context, msg, -1);
		      return;
		  }
		if (bbox->Srid != 4326)
		  {
		      gaiaFreeGeomColl (bbox);
		      msg =
			  "ST_Transform exception - 3rd argument is not a SRID=4326 Geometry.";
		      sqlite3_result_error (context, msg, -1);
		      return;
		  }
		proj_area.WestLongitude = bbox->MinX;
		proj_area.EastLongitude = bbox->MaxX;
		proj_area.SouthLatitude = bbox->MinY;
		proj_area.NorthLatitude = bbox->MaxY;
		proj_bbox = &proj_area;
		gaiaFreeGeomColl (bbox);
	    }
#else /* supporting old PROJ.4 */
	  msg =
	      "ST_Transform exception - extra arguments require using PROJ.6 or later.";
	  sqlite3_result_error (context, msg, -1);
	  return;
#endif
      }
    if (argc >= 4)
      {
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
	  if (sqlite3_value_type (argv[3]) == SQLITE_NULL)
	      ;
	  else if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
	    {
		msg =
		    "ST_Transform exception - 4th argument is neither a TEXT string nor NULL.";
		sqlite3_result_error (context, msg, -1);
		return;
	    }
	  else
	      proj_string_1 = (const char *) sqlite3_value_text (argv[3]);
#else /* supporting old PROJ.4 */
	  msg =
	      "ST_Transform exception - extra arguments require using PROJ.6 or later.";
	  sqlite3_result_error (context, msg, -1);
	  return;
#endif
      }
    if (argc >= 5)
      {
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
	  if (sqlite3_value_type (argv[4]) == SQLITE_NULL)
	      ;
	  else if (sqlite3_value_type (argv[4]) != SQLITE_TEXT)
	    {
		msg =
		    "ST_Transform exception - 5th argument is neither a TEXT string nor NULL.";
		sqlite3_result_error (context, msg, -1);
		return;
	    }
	  else
	      proj_string_2 = (const char *) sqlite3_value_text (argv[4]);
#else /* supporting old PROJ.4 */
	  msg =
	      "ST_Transform exception - extra arguments require using PROJ.6 or later.";
	  sqlite3_result_error (context, msg, -1);
	  return;
#endif
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  srid_from = geo->Srid;
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
	  if (proj_string_1 == NULL && proj_string_2 == NULL)
	    {
		getProjAuthNameSrid (sqlite, srid_from, &proj_from);
		getProjAuthNameSrid (sqlite, srid_to, &proj_to);
		proj_string_1 = proj_from;
		proj_string_2 = proj_to;
		check_origin_destination = 1;
	    }
	  else if (proj_string_1 != NULL && proj_string_2 != NULL)
	      check_origin_destination = 0;
	  else if (proj_string_1 != NULL && proj_string_2 == NULL)
	      check_origin_destination = 0;
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
#else /* supporting old PROJ.4 */
	  getProjParams (sqlite, srid_from, &proj_from);
	  getProjParams (sqlite, srid_to, &proj_to);
	  proj_string_1 = proj_from;
	  proj_string_2 = proj_to;
	  check_origin_destination = 1;
#endif
	  if (check_origin_destination)
	    {
		if (proj_to == NULL || proj_from == NULL)
		  {
		      if (proj_from)
			  free (proj_from);
		      if (proj_to)
			  free (proj_to);
		      gaiaFreeGeomColl (geo);
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
		      if (proj_string_1 == NULL)
			{
			    msg =
				"ST_Transform exception - unable to find the origin SRID.";
			    sqlite3_result_error (context, msg, -1);
			    return;
			}
		      if (proj_string_2 == NULL)
			{
			    msg =
				"ST_Transform exception - unable to find the destination SRID.";
			    sqlite3_result_error (context, msg, -1);
			    return;
			}
#else /* supporting old PROJ.4 */
		      sqlite3_result_null (context);
		      return;
#endif
		  }
	    }

#ifdef PROJ_NEW			/* supporting new PROJ.6 */
	  if (cache != NULL)
	    {
		gaiaResetProjErrorMsg_r (cache);
		result =
		    gaiaTransformEx_r (cache, geo, proj_string_1, proj_string_2,
				       proj_bbox);
	    }
	  else
	      result =
		  gaiaTransformEx (geo, proj_string_1, proj_string_2,
				   proj_bbox);
	  if (result == NULL)
	    {
		msg2 =
		    sqlite3_mprintf
		    ("ST_Transform exception - PROJ reports \"%s\".",
		     gaiaGetProjErrorMsg_r (cache));
		sqlite3_result_error (context, msg2, -1);
		sqlite3_free (msg2);
		if (proj_from != NULL)
		    free (proj_from);
		if (proj_to != NULL)
		    free (proj_to);
		gaiaFreeGeomColl (geo);
		return;
	    }
#else /* supporting old PROJ.4 */
	  if (cache != NULL)
	      result =
		  gaiaTransform_r (cache, geo, proj_string_1, proj_string_2);
	  else
	      result = gaiaTransform (geo, proj_string_1, proj_string_2);
#endif
	  if (proj_from != NULL)
	      free (proj_from);
	  if (proj_to != NULL)
	      free (proj_to);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = srid_to;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_TransformXY (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ TransformXY(BLOBencoded geometry, srid)
/
/ returns a new geometry that is the original one received, but 
/ transformed / translated to the new SRID [coordinates translation 
/ is applied]
/
/ NOTE: this is a special "flavor" of ST_Transform()
/       just X and Y coordinates will be transformed,
/       Z and M values (if eventually present) will be 
/       left untouched.
/       Mainly intended as a workaround possibily useful
/       when facing partially broken PROJ.4 definitions.
/
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int srid_from;
    int srid_to;
    char *proj_from = NULL;
    char *proj_to = NULL;
    void *data = sqlite3_user_data (context);
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	srid_to = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  srid_from = geo->Srid;
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
	  getProjAuthNameSrid (sqlite, srid_from, &proj_from);
	  getProjAuthNameSrid (sqlite, srid_to, &proj_to);
#else /* supporting old PROJ.4 */
	  getProjParams (sqlite, srid_from, &proj_from);
	  getProjParams (sqlite, srid_to, &proj_to);
#endif
	  if (proj_to == NULL || proj_from == NULL)
	    {
		if (proj_from)
		    free (proj_from);
		if (proj_to)
		    free (proj_to);
		gaiaFreeGeomColl (geo);
		sqlite3_result_null (context);
		return;
	    }
	  if (data != NULL)
	      result = gaiaTransformXY_r (data, geo, proj_from, proj_to);
	  else
	      result = gaiaTransformXY (geo, proj_from, proj_to);
	  free (proj_from);
	  free (proj_to);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = srid_to;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_TransformXYZ (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ TransformXYZ(BLOBencoded geometry, srid)
/
/ returns a new geometry that is the original one received, but 
/ transformed / translated to the new SRID [coordinates translation 
/ is applied]
/
/ NOTE: this is a special "flavor" of ST_Transform()
/       just X, Y and Z coordinates will be transformed,
/       M values (if eventually present) will be 
/       left untouched.
/       Mainly intended as a workaround possibily useful
/       when handling 4D geometries having M-values not
/		corresponding to Time.
/
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int srid_from;
    int srid_to;
    char *proj_from = NULL;
    char *proj_to = NULL;
    void *data = sqlite3_user_data (context);
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	srid_to = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  srid_from = geo->Srid;
#ifdef PROJ_NEW			/* supporting new PROJ.6 */
	  getProjAuthNameSrid (sqlite, srid_from, &proj_from);
	  getProjAuthNameSrid (sqlite, srid_to, &proj_to);
#else /* supporting old PROJ.4 */
	  getProjParams (sqlite, srid_from, &proj_from);
	  getProjParams (sqlite, srid_to, &proj_to);
#endif
	  if (proj_to == NULL || proj_from == NULL)
	    {
		if (proj_from)
		    free (proj_from);
		if (proj_to)
		    free (proj_to);
		gaiaFreeGeomColl (geo);
		sqlite3_result_null (context);
		return;
	    }
	  if (data != NULL)
	      result = gaiaTransformXYZ_r (data, geo, proj_from, proj_to);
	  else
	      result = gaiaTransformXYZ (geo, proj_from, proj_to);
	  free (proj_from);
	  free (proj_to);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = srid_to;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

#ifdef PROJ_NEW			/* only if PROJ.6 is supported */
static void
fnct_PROJ_GetLastErrorMsg (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_GetLastErrorMsg()
/
/ return the most recent PROJ error message (if any)
/ return NULL on any other case
*/
    const char *msg;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (data != NULL)
	msg = gaiaGetProjErrorMsg_r (data);
    else
	msg = NULL;
    if (msg == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, msg, strlen (msg), SQLITE_STATIC);
}

static void
fnct_PROJ_GetDatabasePath (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_GetDatabasePath()
/
/ return the currently set PATH leading to the private PROJ.6 database
/ return NULL on any other case
*/
    const char *path;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    path = gaiaGetProjDatabasePath (data);
    if (path == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, path, strlen (path), SQLITE_STATIC);
}

static void
fnct_PROJ_SetDatabasePath (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_SetDatabasePath(path TEXT)
/
/ sets the PATH leading to the private PROJ.6 database
/ return the new path on success
/ or NULL on any other case
*/
    const char *path;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
	path = (const char *) sqlite3_value_text (argv[0]);
    path = gaiaSetProjDatabasePath (data, path);
    if (path == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, path, strlen (path), SQLITE_STATIC);
}

static void
fnct_PROJ_AsProjString (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_AsProjString(auth_name TEXT, auth_srid INTEGER)
/
/ return the representation of some CRS as a proj-string
/ or NULL on error or invalid arguments
*/
    const char *auth_name = "EPSG";
    int auth_srid;
    char *proj_string = NULL;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	auth_name = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	auth_srid = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    proj_string = gaiaGetProjString (data, auth_name, auth_srid);
    if (proj_string == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, proj_string, strlen (proj_string), free);
}

static void
fnct_PROJ_AsWKT (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_AsWKT(auth_name TEXT, auth_srid INTEGER)
/ PROJ_AsWKT(auth_name TEXT, auth_srid INTEGER, style TEXT)
/ PROJ_AsWKT(auth_name TEXT, auth_srid INTEGER, style TEXT, indented BOOLEAN)
/ PROJ_AsWKT(auth_name TEXT, auth_srid INTEGER, style TEXT, 
/            indented BOOLEAN, indentation INTEGER)
/
/ return the WKT representation of some CRS
/ or NULL on error or invalid arguments
*/
    const char *auth_name = "EPSG";
    int auth_srid;
    int style = GAIA_PROJ_WKT_ISO_2018;
    int indented = 1;
    int indentation = 4;
    char *wkt_expr = NULL;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	auth_name = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	auth_srid = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	    {
		const char *wkt = (const char *) sqlite3_value_text (argv[2]);
		if (strcasecmp (wkt, "ISO-2015") == 0)
		    style = GAIA_PROJ_WKT_ISO_2015;
		if (strcasecmp (wkt, "GDAL") == 0)
		    style = GAIA_PROJ_WKT_GDAL;
		if (strcasecmp (wkt, "ESRI") == 0)
		    style = GAIA_PROJ_WKT_ESRI;
		else
		    style = GAIA_PROJ_WKT_ISO_2018;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc >= 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	      indented = sqlite3_value_int (argv[3]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc >= 5)
      {
	  if (sqlite3_value_type (argv[4]) == SQLITE_INTEGER)
	      indentation = sqlite3_value_int (argv[4]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    wkt_expr =
	gaiaGetProjWKT (data, auth_name, auth_srid, style, indented,
			indentation);
    if (wkt_expr == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, wkt_expr, strlen (wkt_expr), free);
}

static void
fnct_PROJ_GuessSridFromWKT (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_GuessSridFromWKT(wkt TEXT)
/
/ return the SRID corresponding to a given WKT expression
/ -1 if not matching SRID exists
/ or NULL on error or invalid arguments
*/
    const char *wkt;
    int srid;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	wkt = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!gaiaGuessSridFromWKT (sqlite, data, wkt, &srid))
	sqlite3_result_int (context, -1);
    else
	sqlite3_result_int (context, srid);
}

static void
fnct_PROJ_GuessSridFromSHP (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_GuessSridFromSHP(filename TEXT)
/
/ return the SRID corresponding to a given Shapefile
/ -1 if not matching SRID exists
/ or NULL on error or invalid arguments
*/
    const char *basepath;
    char *path;
    char *wkt = NULL;
    int srid;
    FILE *in;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	basepath = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* loocking for an eventual .PRJ file */
    path = sqlite3_mprintf ("%s.prj", basepath);
#ifdef _WIN32
    in = gaia_win_fopen (path, "rb");
#else
    in = fopen (path, "rb");
#endif
    if (in != NULL)
      {
	  /* reading the WKT expression from the PRJ file */
	  if (fseek (in, 0, SEEK_END) != -1)
	    {
		int rd;
		int len = ftell (in);
		rewind (in);
		wkt = malloc (len + 1);
		rd = fread (wkt, 1, len, in);
		if (len != rd)
		  {
		      free (wkt);
		      wkt = NULL;
		  }
		*(wkt + len) = '\0';
	    }
	  fclose (in);
      }
    sqlite3_free (path);
    if (wkt == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!gaiaGuessSridFromWKT (sqlite, data, wkt, &srid))
	sqlite3_result_int (context, -1);
    else
	sqlite3_result_int (context, srid);
    free (wkt);
}

#ifdef ENABLE_MINIZIP		/* only id MINIZIP is enabled */
static void
fnct_PROJ_GuessSridFromZipSHP (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ PROJ_GuessSridFromZipSHP(zip_path TEXT, filename TEXT)
/
/ return the SRID corresponding to a given Shapefile
/ -1 if not matching SRID exists
/ or NULL on error or invalid arguments
*/
    const char *zip_path;
    const char *basepath;
    char *wkt = NULL;
    int srid;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	zip_path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	basepath = (const char *) sqlite3_value_text (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* loocking for an eventual .PRJ file */
    wkt = gaiaReadWktFromZipShp (zip_path, basepath);
    if (wkt == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!gaiaGuessSridFromWKT (sqlite, data, wkt, &srid))
	sqlite3_result_int (context, -1);
    else
	sqlite3_result_int (context, srid);
    free (wkt);
}
#endif /* end MINIZIP */

#endif /* end PROJ_NEW */

#endif /* end including PROJ.4 */

#ifndef OMIT_GEOS		/* including GEOS */

static void
fnct_GEOS_GetLastWarningMsg (sqlite3_context * context, int argc,
			     sqlite3_value ** argv)
{
/* SQL function:
/ GEOS_GetLastWarningMsg()
/
/ return the most recent GEOS warning message (if any)
/ return NULL on any other case
*/
    const char *msg;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (data != NULL)
	msg = gaiaGetGeosWarningMsg_r (data);
    else
	msg = gaiaGetGeosWarningMsg ();
    if (msg == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, msg, strlen (msg), SQLITE_STATIC);
}

static void
fnct_GEOS_GetLastErrorMsg (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ GEOS_GetLastErrorMsg()
/
/ return the most recent GEOS error message (if any)
/ return NULL on any other case
*/
    const char *msg;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (data != NULL)
	msg = gaiaGetGeosErrorMsg_r (data);
    else
	msg = gaiaGetGeosErrorMsg ();
    if (msg == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, msg, strlen (msg), SQLITE_STATIC);
}

static void
fnct_GEOS_GetLastAuxErrorMsg (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ GEOS_GetLastAuxErrorMsg()
/
/ return the most recent GEOS error message (if any)
/ return NULL on any other case
*/
    const char *msg;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (data != NULL)
	msg = gaiaGetGeosAuxErrorMsg_r (data);
    else
	msg = gaiaGetGeosAuxErrorMsg ();
    if (msg == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, msg, strlen (msg), SQLITE_STATIC);
}

static void
fnct_GEOS_GetCriticalPointFromMsg (sqlite3_context * context, int argc,
				   sqlite3_value ** argv)
{
/* SQL function:
/ GEOS_GetCriticalPointFromMsg()
/
/ return a Point Geometry by (possibly) parsing the most recent GEOS error/warning message 
/ return NULL on any other case
*/
    int srid = -1;
    gaiaGeomCollPtr geom;
    void *data = sqlite3_user_data (context);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (argc == 1)
      {
	  if (sqlite3_value_type (argv[0]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  srid = sqlite3_value_int (argv[0]);
      }
    if (data != NULL)
	geom = gaiaCriticalPointFromGEOSmsg_r (data);
    else
	geom = gaiaCriticalPointFromGEOSmsg ();
    if (geom == NULL)
	sqlite3_result_null (context);
    else
      {
	  unsigned char *blob;
	  int len;
	  geom->Srid = srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom, &blob, &len, gpkg_mode, tiny_point);
	  gaiaFreeGeomColl (geom);
	  sqlite3_result_blob (context, blob, len, free);
      }
}

static void
fnct_IsValidReason (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsValidReason(geom [ , esri_flag] )
/ ST_IsValidReason(geom [ , esri_flag] )
/
/ return a TEXT string stating if a Geometry is valid
/ and if not valid, a reason why
/ return NULL on any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaGeomCollPtr geom;
    int esri_flag = 0;
    char *str;
    void *data = sqlite3_user_data (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  esri_flag = sqlite3_value_int (argv[1]);
      }
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (esri_flag)
      {
	  gaiaGeomCollPtr detail;
	  if (data != NULL)
	      detail = gaiaIsValidDetailEx_r (data, geom, esri_flag);
	  else
	      detail = gaiaIsValidDetailEx (geom, esri_flag);
	  if (detail == NULL)
	    {
		/* performing extra checks */
		if (data != NULL)
		  {
		      if (gaiaIsToxic_r (data, geom))
			  sqlite3_result_text (context,
					       "Invalid: Toxic Geometry ... too few points",
					       -1, SQLITE_TRANSIENT);
		      else if (gaiaIsNotClosedGeomColl_r (data, geom))
			  sqlite3_result_text (context,
					       "Invalid: Unclosed Rings were detected",
					       -1, SQLITE_TRANSIENT);
		      else
			  sqlite3_result_text (context, "Valid Geometry", -1,
					       SQLITE_TRANSIENT);
		  }
		else
		  {
		      if (gaiaIsToxic (geom))
			  sqlite3_result_text (context,
					       "Invalid: Toxic Geometry ... too few points",
					       -1, SQLITE_TRANSIENT);
		      else if (gaiaIsNotClosedGeomColl (geom))
			  sqlite3_result_text (context,
					       "Invalid: Unclosed Rings were detected",
					       -1, SQLITE_TRANSIENT);
		      else
			  sqlite3_result_text (context, "Valid Geometry", -1,
					       SQLITE_TRANSIENT);
		  }
		goto end;
	    }
	  else
	      gaiaFreeGeomColl (detail);
      }
    if (data != NULL)
	str = gaiaIsValidReason_r (data, geom);
    else
	str = gaiaIsValidReason (geom);
    if (str == NULL)
	sqlite3_result_null (context);
    else
      {
	  len = strlen (str);
	  sqlite3_result_text (context, str, len, free);
      }
  end:
    if (geom != NULL)
	gaiaFreeGeomColl (geom);
}

static void
fnct_IsValidDetail (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsValidDetail(geom [ , esri_flag] )
/ ST_IsValidDetail(geom [ , esri_flag] )
/
/ return a Geometry detail causing a Geometry to be invalid
/ return NULL on any other case
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    gaiaGeomCollPtr geom;
    gaiaGeomCollPtr detail;
    int esri_flag = 0;
    unsigned char *p_result = NULL;
    void *data = sqlite3_user_data (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  esri_flag = sqlite3_value_int (argv[1]);
      }
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (data != NULL)
	detail = gaiaIsValidDetailEx_r (data, geom, esri_flag);
    else
	detail = gaiaIsValidDetailEx (geom, esri_flag);
    if (detail == NULL)
	sqlite3_result_null (context);
    else
      {
	  detail->Srid = geom->Srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (detail, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    if (geom != NULL)
	gaiaFreeGeomColl (geom);
    if (detail != NULL)
	gaiaFreeGeomColl (detail);
}

static void
fnct_Boundary (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Boundary(BLOB encoded geometry)
/
/ returns the combinatorial boundary for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr boundary;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (gaiaIsEmpty (geo))
	      sqlite3_result_null (context);
	  else
	    {
		void *data = sqlite3_user_data (context);
		if (data != NULL)
		    boundary = gaiaBoundary_r (data, geo);
		else
		    boundary = gaiaBoundary (geo);
		if (!boundary)
		    sqlite3_result_null (context);
		else
		  {
		      gaiaToSpatiaLiteBlobWkbEx2 (boundary, &p_result, &len,
						  gpkg_mode, tiny_point);
		      gaiaFreeGeomColl (boundary);
		      sqlite3_result_blob (context, p_result, len, free);
		  }
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_IsClosed (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsClosed(BLOB encoded LINESTRING or MULTILINESTRING geometry)
/
/ returns:
/ 1 if this LINESTRING is closed [or if this is a MULTILINESTRING and every LINESTRINGs are closed] 
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_int (context, -1);
    else
      {
	  sqlite3_result_int (context, gaiaIsClosedGeom (geo));
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_IsSimple (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsSimple(BLOB encoded GEOMETRY)
/
/ returns:
/ 1 if this GEOMETRY is simple
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int ret;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaIsSimple_r (data, geo);
	  else
	      ret = gaiaIsSimple (geo);
	  if (ret < 0)
	      sqlite3_result_int (context, -1);
	  else
	      sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_IsRing (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsRing(BLOB encoded LINESTRING geometry)
/
/ returns:
/ 1 if this LINESTRING is a valid RING
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int ret;
    gaiaGeomCollPtr geo = NULL;
    gaiaLinestringPtr line;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_int (context, -1);
    else
      {
	  line = simpleLinestring (geo);
	  if (line == NULL)
	      sqlite3_result_int (context, -1);
	  else
	    {
		void *data = sqlite3_user_data (context);
		if (data != NULL)
		    ret = gaiaIsRing_r (data, line);
		else
		    ret = gaiaIsRing (line);
		sqlite3_result_int (context, ret);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_IsValid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsValid(BLOB encoded GEOMETRY [ , BOOLEAN esri_flag] )
/
/ returns:
/ 1 if this GEOMETRY is a valid one
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int ret;
    gaiaGeomCollPtr geo = NULL;
    int esri_flag = 0;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  esri_flag = sqlite3_value_int (argv[1]);
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (esri_flag)
	    {
		gaiaGeomCollPtr detail;
		if (data != NULL)
		    detail = gaiaIsValidDetailEx_r (data, geo, esri_flag);
		else
		    detail = gaiaIsValidDetailEx (geo, esri_flag);
		if (detail == NULL)
		  {
		      /* extra checks */
		      int extra = 0;
		      if (data != NULL)
			{
			    if (gaiaIsToxic_r (data, geo))
				extra = 1;
			    if (gaiaIsNotClosedGeomColl_r (data, geo))
				extra = 1;
			}
		      else
			{
			    if (gaiaIsToxic (geo))
				extra = 1;
			    if (gaiaIsNotClosedGeomColl (geo))
				extra = 1;
			}
		      if (extra)
			  sqlite3_result_int (context, 0);
		      else
			  sqlite3_result_int (context, 1);
		  }
		else
		  {
		      gaiaFreeGeomColl (detail);
		      sqlite3_result_int (context, 0);
		  }
		goto end;
	    }
	  if (data != NULL)
	      ret = gaiaIsValid_r (data, geo);
	  else
	      ret = gaiaIsValid (geo);
	  if (ret < 0)
	      sqlite3_result_int (context, -1);
	  else
	      sqlite3_result_int (context, ret);
      }
  end:
    gaiaFreeGeomColl (geo);
}

static void
length_common (const void *p_cache, sqlite3_context * context, int argc,
	       sqlite3_value ** argv, int is_perimeter)
{
/* common implementation supporting both ST_Length and ST_Perimeter */
    unsigned char *p_blob;
    int n_bytes;
    double length = 0.0;
    int ret;
    int use_ellipsoid = -1;
    double a;
    double b;
    double rf;
    gaiaGeomCollPtr geo = NULL;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  use_ellipsoid = sqlite3_value_int (argv[1]);
	  if (use_ellipsoid != 0)
	      use_ellipsoid = 1;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (use_ellipsoid >= 0)
	    {
		/* attempting to identify the corresponding ellipsoid */
		if (getEllipsoidParams (sqlite, geo->Srid, &a, &b, &rf))
		  {
		      double l;
		      int ib;
		      gaiaLinestringPtr line;
		      gaiaPolygonPtr polyg;
		      gaiaRingPtr ring;
		      if (use_ellipsoid)
			{
			    /* measuring on the Ellipsoid */
			    if (!is_perimeter)
			      {
				  line = geo->FirstLinestring;
				  while (line)
				    {
					/* Linestrings */
					l = gaiaGeodesicTotalLength (a,
								     b,
								     rf,
								     line->
								     DimensionModel,
								     line->Coords,
								     line->Points);
					if (l < 0.0)
					  {
					      length = -1.0;
					      break;
					  }
					length += l;
					line = line->Next;
				    }
			      }
			    if (length >= 0)
			      {
				  if (is_perimeter)
				    {
					/* Polygons */
					polyg = geo->FirstPolygon;
					while (polyg)
					  {
					      /* exterior Ring */
					      ring = polyg->Exterior;
					      l = gaiaGeodesicTotalLength (a,
									   b,
									   rf,
									   ring->DimensionModel,
									   ring->Coords,
									   ring->Points);
					      if (l < 0.0)
						{
						    length = -1.0;
						    break;
						}
					      length += l;
					      for (ib = 0;
						   ib <
						   polyg->NumInteriors; ib++)
						{
						    /* interior Rings */
						    ring =
							polyg->Interiors + ib;
						    l = gaiaGeodesicTotalLength
							(a, b, rf,
							 ring->DimensionModel,
							 ring->Coords,
							 ring->Points);
						    if (l < 0.0)
						      {
							  length = -1.0;
							  break;
						      }
						    length += l;
						}
					      if (length < 0.0)
						  break;
					      polyg = polyg->Next;
					  }
				    }
			      }
			}
		      else
			{
			    /* measuring on the Great Circle */
			    if (!is_perimeter)
			      {
				  line = geo->FirstLinestring;
				  while (line)
				    {
					/* Linestrings */
					length +=
					    gaiaGreatCircleTotalLength
					    (a, b, line->DimensionModel,
					     line->Coords, line->Points);
					line = line->Next;
				    }
			      }
			    if (length >= 0)
			      {
				  if (is_perimeter)
				    {
					/* Polygons */
					polyg = geo->FirstPolygon;
					while (polyg)
					  {
					      /* exterior Ring */
					      ring = polyg->Exterior;
					      length +=
						  gaiaGreatCircleTotalLength
						  (a, b,
						   ring->DimensionModel,
						   ring->Coords, ring->Points);
					      for (ib = 0;
						   ib < polyg->NumInteriors;
						   ib++)
						{
						    /* interior Rings */
						    ring =
							polyg->Interiors + ib;
						    length +=
							gaiaGreatCircleTotalLength
							(a, b,
							 ring->DimensionModel,
							 ring->Coords,
							 ring->Points);
						}
					      polyg = polyg->Next;
					  }
				    }
			      }
			}
		      if (length < 0.0)
			{
			    /* invalid distance */
			    sqlite3_result_null (context);
			}
		      else
			  sqlite3_result_double (context, length);
		  }
		else
		    sqlite3_result_null (context);
		goto stop;
	    }
	  else if (p_cache != NULL)
	      ret =
		  gaiaGeomCollLengthOrPerimeter_r (p_cache, geo, is_perimeter,
						   &length);
	  else
	      ret = gaiaGeomCollLengthOrPerimeter (geo, is_perimeter, &length);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, length);
      }
  stop:
    gaiaFreeGeomColl (geo);
}

static void
fnct_Length (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_Length(BLOB encoded GEOMETRYCOLLECTION)
/ ST_Length(BLOB encoded GEOMETRYCOLLECTION, Boolean use_ellipsoid)
/
/ returns  the total length for current geometry 
/ or NULL if any error is encountered
/
/ Please note: starting since 4.0.0 this function will ignore
/ any Polygon (only Linestrings will be considered)
/
*/
    void *data = sqlite3_user_data (context);
    length_common (data, context, argc, argv, 0);
}

static void
fnct_Perimeter (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_Perimeter(BLOB encoded GEOMETRYCOLLECTION)
/ ST_Perimeter(BLOB encoded GEOMETRYCOLLECTION, Boolean use_ellipsoid)
/
/ returns  the total perimeter length for current geometry 
/ or NULL if any error is encountered
/
/ Please note: starting since 4.0.0 this function will ignore
/ any Linestring (only Polygons will be considered)
/
*/
    void *data = sqlite3_user_data (context);
    length_common (data, context, argc, argv, 1);
}

static void
linestring_segment_length_common (sqlite3_context * context, int argc,
				  sqlite3_value ** argv, int mode)
{
/* common implementation supporting LinestringXxxSegmentLenght */
    unsigned char *p_blob;
    int n_bytes;
    int ignore_repeated_vertices = 1;
    int iv;
    double x;
    double y;
    double z;
    double m;
    double last_x;
    double last_y;
    double min = DBL_MAX;
    double max = 0.0;
    double tot = 0.0;
    int n = 0;
    gaiaLinestringPtr ln;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  ignore_repeated_vertices = sqlite3_value_int (argv[1]);
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!is_single_linestring (geo))
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }

    ln = geo->FirstLinestring;
    for (iv = 0; iv < ln->Points; iv++)
      {
	  if (geo->DimensionModel == GAIA_XY_Z)
	    {
		gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
	    }
	  else if (geo->DimensionModel == GAIA_XY_M)
	    {
		gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
	    }
	  else if (geo->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
	    }
	  else
	    {
		gaiaGetPoint (ln->Coords, iv, &x, &y);
	    }
	  if (iv > 0)
	    {
		int ok = 1;
		if (ignore_repeated_vertices)
		  {
		      if (last_x == x && last_y == y)
			  ok = 0;
		  }
		if (ok)
		  {
		      double l =
			  sqrt (((last_x - x) * (last_x - x)) +
				((last_y - y) * (last_y - y)));
		      if (l < min)
			  min = l;
		      if (l > max)
			  max = l;
		      tot += l;
		      n++;
		  }
	    }
	  last_x = x;
	  last_y = y;
      }
    if (mode == LINESTRING_MIN_SEGMENT_LENGTH)
	sqlite3_result_double (context, min);
    else if (mode == LINESTRING_MAX_SEGMENT_LENGTH)
	sqlite3_result_double (context, max);
    else
	sqlite3_result_double (context, tot / (double) n);
    gaiaFreeGeomColl (geo);
}

static void
fnct_LinestringMinSegmentLength (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ ST_LinestringMinSegmentLength(BLOB encoded LINESTRING)
/ ST_LinestringMinSegmentLength(BLOB encoded LINESTRING, BOOL ignore_repeated_vertices)
/
/ returns the length of the shortest segment in the Linestring
/ or NULL if any error is encountered
/
*/
    linestring_segment_length_common (context, argc, argv,
				      LINESTRING_MIN_SEGMENT_LENGTH);
}

static void
fnct_LinestringMaxSegmentLength (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ ST_LinsetringMaxSegmentLength(BLOB encoded LINESTRING)
/
/ returns the length of the longest segment in the Linestring
/ or NULL if any error is encountered
/
*/
    linestring_segment_length_common (context, argc, argv,
				      LINESTRING_MAX_SEGMENT_LENGTH);
}

static void
fnct_LinestringAvgSegmentLength (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ ST_LinestringMaxSegmentLength(BLOB encoded LINESTRING)
/
/ returns the average segment length in the Linsetring
/ or NULL if any error is encountered
/
*/
    linestring_segment_length_common (context, argc, argv,
				      LINESTRING_AVG_SEGMENT_LENGTH);
}

static void
fnct_CurvosityIndex (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_CurvosityIndex(BLOB encoded LINESTRING)
/    or
/ ST_CurvosityIndex(BLOB encoded LINESTRING, points INTEGER)
/
/ returns the CurvosityIndex of some Linestring
/ or NULL if any error is encountered
/
*/
    unsigned char *p_blob;
    int n_bytes;
    int extra_points = 0;
    double index;
    gaiaLinestringPtr ln;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  extra_points = sqlite3_value_int (argv[1]);
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!is_single_linestring (geo))
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }

    ln = geo->FirstLinestring;
    index = gaiaCurvosityIndex (cache, ln, extra_points);
    gaiaFreeGeomColl (geo);
    sqlite3_result_double (context, index);
}

static void
fnct_UphillHeight (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_UphillHeight(BLOB encoded LINESTRING)
/
/ returns the cumulative Uphill Height of some 3D Linestring
/ or NULL if any error is encountered
/
*/
    unsigned char *p_blob;
    int n_bytes;
    double up;
    double down;
    gaiaLinestringPtr ln;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!is_single_linestring (geo))
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }

    ln = geo->FirstLinestring;
    gaiaUpDownHeight (ln, &up, &down);
    gaiaFreeGeomColl (geo);
    sqlite3_result_double (context, up);
}

static void
fnct_DownhillHeight (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_DownhillHeight(BLOB encoded LINESTRING)
/
/ returns the cumulative Downhill Height of some 3D Linestring
/ or NULL if any error is encountered
/
*/
    unsigned char *p_blob;
    int n_bytes;
    double up;
    double down;
    gaiaLinestringPtr ln;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!is_single_linestring (geo))
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }

    ln = geo->FirstLinestring;
    gaiaUpDownHeight (ln, &up, &down);
    gaiaFreeGeomColl (geo);
    sqlite3_result_double (context, down);
}

static void
fnct_UpDownHeight (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_UpDownHeight(BLOB encoded LINESTRING)
/
/ returns the cumulative UpDown Height of some 3D Linestring
/ or NULL if any error is encountered
/
*/
    unsigned char *p_blob;
    int n_bytes;
    double up;
    double down;
    gaiaLinestringPtr ln;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!is_single_linestring (geo))
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }

    ln = geo->FirstLinestring;
    gaiaUpDownHeight (ln, &up, &down);
    sqlite3_result_double (context, up + down);
    gaiaFreeGeomColl (geo);
}

#ifdef ENABLE_RTTOPO		/* only if RTTOPO is enabled */

static void
fnct_3dLength (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_3dLength(BLOB encoded GEOMETRYCOLLECTION)
/
/ returns  the total 2D or 3D length for current geometry
/ accordingly to the Geometry dimensions 
/ returns NULL if any error is encountered
/
/ Please note: this function will ignore
/ any Polygon (only Linestrings will be considered)
/
*/
    unsigned char *p_blob;
    int n_bytes;
    double length = 0.0;
    int ret;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  ret = gaia3dLength (cache, geo, &length);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, length);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_FromTWKB (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromTWKB(TWKB encoded geometry)
/   or
/ GeomFromTWKB(TWKB encoded geometry, INT srid)
/
/ returns the current geometry by parsing a TWKB encoded string 
/ or NULL if any error is encountered
*/
    int len;
    unsigned char *p_result = NULL;
    const unsigned char *twkb;
    int twkb_size;
    int srid = -1;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  twkb = sqlite3_value_blob (argv[0]);
	  twkb_size = sqlite3_value_bytes (argv[0]);
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  srid = sqlite3_value_int (argv[1]);
	  if (srid < 0)
	      srid = -1;
      }
    geo = gaiaFromTWKB (cache, twkb, twkb_size, srid);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &len, gpkg_mode, tiny_point);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, len, free);
}

static void
fnct_ToTWKB (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsTWKB(BLOB encoded geometry)
/   or
/ AsTWKB(BLOB encoded geometry, INT precision_xy)
/   or
/ AsTWKB(BLOB encoded geometry, INT precision_xy, INT precision_z)
/   or
/ AsTWKB(BLOB encoded geometry, INT precision_xy, INT precision_z, 
/        INT precision_m)
/   or
/ AsTWKB(BLOB encoded geometry, INT precision_xy, INT precision_z, 
/        INT precision_m, INT with_size)
/   or
/ AsTWKB(BLOB encoded geometry, INT precision_xy, INT precision_z, 
/        INT precision_m, INT with_size, INT with_bbox)
/
/ returns a text string corresponding to compressed TWKB notation 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int value;
    unsigned char precision_xy = 0;
    unsigned char precision_z = 0;
    unsigned char precision_m = 0;
    int with_size = 0;
    int with_bbox = 0;
    int ret;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = sqlite3_value_int (argv[1]);
	  if (value < 0)
	      precision_xy = 0;
	  else if (value > 20)
	      precision_xy = 20;
	  else
	      precision_xy = value;
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = sqlite3_value_int (argv[2]);
	  if (value < 0)
	      precision_z = 0;
	  else if (value > 20)
	      precision_z = 20;
	  else
	      precision_z = value;
      }
    if (argc >= 4)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = sqlite3_value_int (argv[3]);
	  if (value < 0)
	      precision_m = 0;
	  else if (value > 20)
	      precision_m = 20;
	  else
	      precision_m = value;
      }
    if (argc >= 5)
      {
	  if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  with_size = sqlite3_value_int (argv[4]);
	  if (with_size != 0)
	      with_size = 1;
      }
    if (argc >= 6)
      {
	  if (sqlite3_value_type (argv[5]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  with_bbox = sqlite3_value_int (argv[5]);
	  if (with_bbox != 0)
	      with_bbox = 1;
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  unsigned char *twkb;
	  int size_twkb;
	  ret =
	      gaiaToTWKB (cache, geo, precision_xy, precision_z, precision_m,
			  with_size, with_bbox, &twkb, &size_twkb);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, twkb, size_twkb, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_AsEncodedPolyline (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ ST_AsEncodedPolyline(BLOB encoded geometry)
/   or
/ ST_AsEncodedPolyline(BLOB encoded geometry, INT precision)
/
/ returns a text string corresponding to GoogleMaps encoded Polyline
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int value;
    unsigned char precision = 5;
    int ret;
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = sqlite3_value_int (argv[1]);
	  if (value < 0)
	      precision = 0;
	  else if (value > 20)
	      precision = 20;
	  else
	      precision = value;
      }
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  int invalid = 0;
	  int geographic = 0;
	  char *encoded;
	  int size_encoded;
	  if (geo->FirstPoint == NULL && geo->FirstPolygon == NULL
	      && geo->FirstLinestring != NULL
	      && geo->FirstLinestring == geo->LastLinestring)
	      ;
	  else
	      invalid = 1;
	  if (!srid_is_geographic (sqlite, geo->Srid, &geographic))
	      invalid = 1;
	  if (!geographic)
	      invalid = 1;
	  if (invalid)
	    {
		gaiaFreeGeomColl (geo);
		sqlite3_result_null (context);
		return;
	    }
	  ret =
	      gaiaAsEncodedPolyLine (cache, geo, precision, &encoded,
				     &size_encoded);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_text (context, encoded, size_encoded, free);
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_LineFromEncodedPolyline (sqlite3_context * context, int argc,
			      sqlite3_value ** argv)
{
/* SQL function:
/ ST_LineFromEncodedPolyline(TEXT encoded geometry)
/   or
/ ST_LineFromEncodedPolyline(TEXT encoded geometry, INT precision)
/
/ returns a Linestring Geometry from a text string corresponding to
/ GoogleMaps encoded Polyline
/ or NULL if any error is encountered
*/
    unsigned char *p_result = NULL;
    int size;
    const char *encoded;
    gaiaGeomCollPtr geo = NULL;
    int value;
    unsigned char precision = 5;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	gpkg_mode = cache->gpkg_mode;
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
	encoded = (const char *) sqlite3_value_text (argv[0]);
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = sqlite3_value_int (argv[1]);
	  if (value < 0)
	      precision = 0;
	  else if (value > 20)
	      precision = 20;
	  else
	      precision = value;
      }
    geo = gaiaLineFromEncodedPolyline (cache, encoded, precision);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geo, &p_result, &size, gpkg_mode, 0);
    gaiaFreeGeomColl (geo);
    sqlite3_result_blob (context, p_result, size, free);
}

#endif /* end RTTOPO conditional */

static void
fnct_Area (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Area(BLOB encoded GEOMETRYCOLLECTION)
/
/ returns the total area for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double area = 0.0;
    int ret;
    int use_ellipsoid = -1;
#ifdef ENABLE_RTTOPO		/* only if RTTOPO is enabled */
    double a;
    double b;
    double rf;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
#endif /* end RTTOPO conditional */
    gaiaGeomCollPtr geo = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  use_ellipsoid = sqlite3_value_int (argv[1]);
	  if (use_ellipsoid != 0)
	      use_ellipsoid = 1;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (use_ellipsoid >= 0)
	    {
#ifdef ENABLE_RTTOPO		/* only if RTTOPO is enabled */
		/* attempting to identify the corresponding ellipsoid */
		if (getEllipsoidParams (sqlite, geo->Srid, &a, &b, &rf))
		    ret =
			gaiaGeodesicArea (cache, geo, a, b, use_ellipsoid,
					  &area);
		else
		    ret = 0;
#else
		ret = 0;
#endif /* end RTTOPO conditional */
	    }
	  else
	    {
		void *data = sqlite3_user_data (context);
		if (data != NULL)
		    ret = gaiaGeomCollArea_r (data, geo, &area);
		else
		    ret = gaiaGeomCollArea (geo, &area);
	    }
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, area);
      }
    gaiaFreeGeomColl (geo);
}

static gaiaGeomCollPtr
circularity_polygon (int srid, int dims, gaiaPolygonPtr pg)
{
/* building an individual Polygon for Circularity */
    gaiaGeomCollPtr geom = NULL;
    gaiaPolygonPtr pg2;
    gaiaRingPtr i_rng;
    gaiaRingPtr o_rng;
    if (dims == GAIA_XY_Z)
	geom = gaiaAllocGeomCollXYZ ();
    else if (dims == GAIA_XY_M)
	geom = gaiaAllocGeomCollXYM ();
    else if (dims == GAIA_XY_Z_M)
	geom = gaiaAllocGeomCollXYZM ();
    else
	geom = gaiaAllocGeomColl ();
    geom->Srid = srid;
    i_rng = pg->Exterior;
    pg2 = gaiaAddPolygonToGeomColl (geom, i_rng->Points, 0);
    o_rng = pg2->Exterior;
    /* copying points (only EXTERIOR RING) */
    gaiaCopyRingCoords (o_rng, i_rng);
    return geom;
}

static void
fnct_Circularity (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Circularity(BLOB encoded GEOMETRYCOLLECTION)
/
/ returns the Circularity Index for current geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    double pi = 3.14159265358979323846;
    double area = 0.0;
    double perimeter = 0.0;
    double sum_area = 0.0;
    double sum_perimeter = 0.0;
    int nlns = 0;
    int npgs = 0;
    int ret;
    int use_ellipsoid = -1;
#ifdef ENABLE_RTTOPO		/* only if RTTOPO is enabled */
    double a;
    double b;
    double rf;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
#endif /* end RTTOPO conditional */
    gaiaGeomCollPtr geo = NULL;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  use_ellipsoid = sqlite3_value_int (argv[1]);
	  if (use_ellipsoid != 0)
	      use_ellipsoid = 1;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (use_ellipsoid >= 0)
	    {
#ifdef ENABLE_RTTOPO		/* only if RTTOPO is enabled */
		/* attempting to identify the corresponding ellipsoid */
		if (getEllipsoidParams (sqlite, geo->Srid, &a, &b, &rf))
		    ret = 1;
		else
		    ret = 0;
#else
		ret = 0;
#endif /* end RTTOPO conditional */
		if (!ret)
		  {
		      sqlite3_result_null (context);
		      goto end;
		  }
	    }
	  ln = geo->FirstLinestring;
	  while (ln != NULL)
	    {
		nlns++;
		ln = ln->Next;
	    }

	  pg = geo->FirstPolygon;
	  while (pg != NULL)
	    {
		/* looping on individual polygons */
		gaiaGeomCollPtr geo2 =
		    circularity_polygon (geo->Srid, geo->DimensionModel, pg);
		if (use_ellipsoid >= 0)
		  {
#ifdef ENABLE_RTTOPO		/* only if RTTOPO is enabled */
		      /* attempting to identify the corresponding ellipsoid */
		      ret =
			  gaiaGeodesicArea (cache, geo2, a, b, use_ellipsoid,
					    &area);
#else
		      ret = 0;
#endif /* end RTTOPO conditional */
		  }
		else
		  {
		      if (data != NULL)
			  ret = gaiaGeomCollArea_r (data, geo2, &area);
		      else
			  ret = gaiaGeomCollArea (geo2, &area);
		  }
		if (ret)
		  {
		      sum_area += area;
		      npgs++;
		  }
		else
		  {
		      gaiaFreeGeomColl (geo2);
		      npgs = 0;
		      break;
		  }

		if (use_ellipsoid >= 0)
		  {
#ifdef ENABLE_RTTOPO		/* only if RTTOPO is enabled */
		      perimeter = gaiaGeodesicTotalLength (a, b, rf,
							   pg->
							   Exterior->DimensionModel,
							   pg->Exterior->Coords,
							   pg->
							   Exterior->Points);
		      if (perimeter < 0.0)
			  ret = 0;
		      else
			  ret = 1;
#else
		      ret = 0;
#endif /* end RTTOPO conditional */
		  }
		else
		  {
		      if (data != NULL)
			  ret =
			      gaiaGeomCollLengthOrPerimeter_r (data, geo2, 1,
							       &perimeter);
		      else
			  ret =
			      gaiaGeomCollLengthOrPerimeter (geo2, 1,
							     &perimeter);
		  }
		if (ret)
		    sum_perimeter += perimeter;
		else
		  {
		      gaiaFreeGeomColl (geo2);
		      npgs = 0;
		      break;
		  }
		gaiaFreeGeomColl (geo2);
		pg = pg->Next;
	    }
	  if (!npgs)
	    {
		if (nlns)
		    sqlite3_result_double (context, 0.0);
		else
		    sqlite3_result_null (context);
	    }
	  else
	    {
		double index =
		    (4.0 * pi * sum_area) / (sum_perimeter * sum_perimeter);
		sqlite3_result_double (context, index);
	    }
      }
  end:
    gaiaFreeGeomColl (geo);
}

static void
fnct_Centroid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Centroid(BLOBencoded POLYGON or MULTIPOLYGON geometry)
/
/ returns a POINT representing the centroid for current POLYGON / MULTIPOLYGON geometry 
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    int ret;
    double x;
    double y;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  if (gaiaIsEmpty (geo))
	      sqlite3_result_null (context);
	  else
	    {
		void *data = sqlite3_user_data (context);
		if (data != NULL)
		    ret = gaiaGeomCollCentroid_r (data, geo, &x, &y);
		else
		    ret = gaiaGeomCollCentroid (geo, &x, &y);
		if (!ret)
		    sqlite3_result_null (context);
		else
		  {
		      result = gaiaAllocGeomColl ();
		      result->Srid = geo->Srid;
		      gaiaAddPointToGeomColl (result, x, y);
		      gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
						  gpkg_mode, tiny_point);
		      gaiaFreeGeomColl (result);
		      sqlite3_result_blob (context, p_result, len, free);
		  }
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_PointOnSurface (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ PointOnSurface(BLOBencoded POLYGON or MULTIPOLYGON geometry)
/
/ returns a POINT guaranteed to lie on the Surface
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    double x;
    double y;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  int posret;
	  if (data != NULL)
	      posret = gaiaGetPointOnSurface_r (data, geo, &x, &y);
	  else
	      posret = gaiaGetPointOnSurface (geo, &x, &y);
	  if (!posret)
	      sqlite3_result_null (context);
	  else
	    {
		if (geo->DimensionModel == GAIA_XY_Z)
		  {
		      result = gaiaAllocGeomCollXYZ ();
		      gaiaAddPointToGeomCollXYZ (result, x, y, 0.0);
		  }
		else if (geo->DimensionModel == GAIA_XY_M)
		  {
		      result = gaiaAllocGeomCollXYM ();
		      gaiaAddPointToGeomCollXYM (result, x, y, 0.0);
		  }
		else if (geo->DimensionModel == GAIA_XY_Z_M)
		  {
		      result = gaiaAllocGeomCollXYZM ();
		      gaiaAddPointToGeomCollXYZM (result, x, y, 0.0, 0.0);
		  }
		else
		  {
		      result = gaiaAllocGeomColl ();
		      gaiaAddPointToGeomColl (result, x, y);
		  }
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		gaiaFreeGeomColl (result);
		sqlite3_result_blob (context, p_result, len, free);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Simplify (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Simplify(BLOBencoded geometry, tolerance)
/
/ returns a new geometry that is a caricature of the original one received, but simplified using the Douglas-Peuker algorihtm
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int int_value;
    double tolerance;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	tolerance = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  tolerance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeomCollSimplify_r (data, geo, tolerance);
	  else
	      result = gaiaGeomCollSimplify (geo, tolerance);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_SimplifyPreserveTopology (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ SimplifyPreserveTopology(BLOBencoded geometry, tolerance)
/
/ returns a new geometry that is a caricature of the original one received, but simplified using the Douglas-Peuker algorihtm [preserving topology]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int int_value;
    double tolerance;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	tolerance = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  tolerance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaGeomCollSimplifyPreserveTopology_r (data, geo, tolerance);
	  else
	      result = gaiaGeomCollSimplifyPreserveTopology (geo, tolerance);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ConvexHull (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ConvexHull(BLOBencoded geometry)
/
/ returns a new geometry representing the CONVEX HULL for current geometry
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int len;
    unsigned char *p_result = NULL;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaConvexHull_r (data, geo);
	  else
	      result = gaiaConvexHull (geo);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Buffer (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Buffer(BLOBencoded geometry, radius)
/ Buffer(BLOBencoded geometry, radius, quadrantsegments)
/
/ returns a new geometry representing the BUFFER for current geometry
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    double radius;
    int int_value;
    int quadrantsegments = -1;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	radius = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  radius = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  quadrantsegments = sqlite3_value_int (argv[2]);
	  if (quadrantsegments <= 0)
	      quadrantsegments = 1;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaGeomCollBuffer_r (data, geo, radius, quadrantsegments);
	  else
	    {
		if (quadrantsegments <= 0)
		    quadrantsegments = 30;
		result = gaiaGeomCollBuffer (geo, radius, quadrantsegments);
	    }
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Intersection (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Intersection(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns a new geometry representing the INTERSECTION of both geometries
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeometryIntersection_r (data, geo1, geo2);
	  else
	      result = gaiaGeometryIntersection (geo1, geo2);
	  if (!result)
	      sqlite3_result_null (context);
	  else if (gaiaIsEmpty (result))
	    {
		gaiaFreeGeomColl (result);
		sqlite3_result_null (context);
	    }
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static int
gaia_union_polygs (gaiaGeomCollPtr geom)
{
/* testing if this geometry simply contains Polygons */
    int pts = 0;
    int lns = 0;
    int pgs = 0;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    pt = geom->FirstPoint;
    while (pt)
      {
	  pts++;
	  pt = pt->Next;
      }
    ln = geom->FirstLinestring;
    while (ln)
      {
	  lns++;
	  ln = ln->Next;
      }
    pg = geom->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pts || lns)
	return 0;
    if (!pgs)
	return 0;
    return 1;
}

static void
fnct_Union_step (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Union(BLOBencoded geom)
/
/ aggregate function - STEP
/
*/
    struct gaia_geom_chain *chain;
    struct gaia_geom_chain_item *item;
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    struct gaia_geom_chain **p;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	return;
    p = sqlite3_aggregate_context (context, sizeof (struct gaia_geom_chain **));
    if (!(*p))
      {
	  /* this is the first row */
	  chain = malloc (sizeof (struct gaia_geom_chain));
	  *p = chain;
	  item = malloc (sizeof (struct gaia_geom_chain_item));
	  item->geom = geom;
	  item->next = NULL;
	  chain->all_polygs = gaia_union_polygs (geom);
	  chain->first = item;
	  chain->last = item;
      }
    else
      {
	  /* subsequent rows */
	  chain = *p;
	  item = malloc (sizeof (struct gaia_geom_chain_item));
	  item->geom = geom;
	  item->next = NULL;
	  if (!gaia_union_polygs (geom))
	      chain->all_polygs = 0;
	  chain->last->next = item;
	  chain->last = item;
      }
}

static void
gaia_free_geom_chain (struct gaia_geom_chain *chain)
{
    struct gaia_geom_chain_item *p = chain->first;
    struct gaia_geom_chain_item *pn;
    while (p)
      {
	  pn = p->next;
	  gaiaFreeGeomColl (p->geom);
	  free (p);
	  p = pn;
      }
    free (chain);
}

static void
fnct_Union_final (sqlite3_context * context)
{
/* SQL function:
/ Union(BLOBencoded geom)
/
/ aggregate function - FINAL
/
*/
    gaiaGeomCollPtr tmp;
    struct gaia_geom_chain *chain;
    struct gaia_geom_chain_item *item;
    gaiaGeomCollPtr aggregate = NULL;
    gaiaGeomCollPtr result;
    void *data = sqlite3_user_data (context);
    struct gaia_geom_chain **p = sqlite3_aggregate_context (context, 0);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    chain = *p;

/* applying UnaryUnion */
    item = chain->first;
    while (item)
      {
	  gaiaGeomCollPtr geom = item->geom;
	  if (item == chain->first)
	    {
		/* initializing the aggregate geometry */
		aggregate = geom;
		item->geom = NULL;
		item = item->next;
		continue;
	    }
	  if (data != NULL)
	      tmp = gaiaMergeGeometries_r (data, aggregate, geom);
	  else
	      tmp = gaiaMergeGeometries (aggregate, geom);
	  gaiaFreeGeomColl (geom);
	  item->geom = NULL;
	  aggregate = tmp;
	  item = item->next;
      }
    if (data != NULL)
	result = gaiaUnaryUnion_r (data, aggregate);
    else
	result = gaiaUnaryUnion (aggregate);
    gaiaFreeGeomColl (aggregate);
    gaia_free_geom_chain (chain);

    if (result == NULL)
	sqlite3_result_null (context);
    else if (gaiaIsEmpty (result))
	sqlite3_result_null (context);
    else
      {
	  /* builds the BLOB geometry to be returned */
	  int len;
	  unsigned char *p_result = NULL;
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
      }
    gaiaFreeGeomColl (result);
}

static void
fnct_Union (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Union(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns a new geometry representing the UNION of both geometries
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeometryUnion_r (data, geo1, geo2);
	  else
	      result = gaiaGeometryUnion (geo1, geo2);
	  if (!result)
	      sqlite3_result_null (context);
	  else if (gaiaIsEmpty (result))
	    {
		gaiaFreeGeomColl (result);
		sqlite3_result_null (context);
	    }
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Difference (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Difference(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns a new geometry representing the DIFFERENCE of both geometries
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeometryDifference_r (data, geo1, geo2);
	  else
	      result = gaiaGeometryDifference (geo1, geo2);
	  if (!result)
	      sqlite3_result_null (context);
	  else if (gaiaIsEmpty (result))
	    {
		gaiaFreeGeomColl (result);
		sqlite3_result_null (context);
	    }
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_SymDifference (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SymDifference(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns a new geometry representing the SYMMETRIC DIFFERENCE of both geometries
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeometrySymDifference_r (data, geo1, geo2);
	  else
	      result = gaiaGeometrySymDifference (geo1, geo2);
	  if (!result)
	      sqlite3_result_null (context);
	  else if (gaiaIsEmpty (result))
	    {
		gaiaFreeGeomColl (result);
		sqlite3_result_null (context);
	    }
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Equals (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Equals(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if the two geometries are "spatially equal"
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollEquals_r (data, geo1, geo2);
	  else
	      ret = gaiaGeomCollEquals (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Intersects (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Intersects(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if the two geometries do "spatially intersects"
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedIntersects (data,
						    geo1, blob1, bytes1, geo2,
						    blob2, bytes2);
	  else
	      ret = gaiaGeomCollIntersects (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Disjoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Disjoint(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if the two geometries are "spatially disjoint"
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedDisjoint (data,
						  geo1, blob1, bytes1, geo2,
						  blob2, bytes2);
	  else
	      ret = gaiaGeomCollDisjoint (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Overlaps (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Overlaps(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if the two geometries do "spatially overlaps"
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedOverlaps (data,
						  geo1, blob1, bytes1, geo2,
						  blob2, bytes2);
	  else
	      ret = gaiaGeomCollOverlaps (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Crosses (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Crosses(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if the two geometries do "spatially crosses"
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedCrosses (data,
						 geo1, blob1, bytes1, geo2,
						 blob2, bytes2);
	  else
	      ret = gaiaGeomCollCrosses (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Touches (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Touches(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if the two geometries do "spatially touches"
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedTouches (data,
						 geo1, blob1, bytes1, geo2,
						 blob2, bytes2);
	  else
	      ret = gaiaGeomCollTouches (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Within (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Within(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if GEOM-1 is completely contained within GEOM-2
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedWithin (data, geo1,
						blob1, bytes1, geo2, blob2,
						bytes2);
	  else
	      ret = gaiaGeomCollWithin (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Contains (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Contains(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if GEOM-1 completely contains GEOM-2
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedContains (data,
						  geo1, blob1, bytes1, geo2,
						  blob2, bytes2);
	  else
	      ret = gaiaGeomCollContains (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Relate (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Relate(BLOBencoded geom1, BLOBencoded geom2, string pattern)
/
/ returns:
/ 1 if GEOM-1 and GEOM-2 have a spatial relationship as specified by the patternMatrix 
/ 0 otherwise
/ or -1 if any error is encountered
/
/ or alternatively:
/
/ Relate(BLOBencoded geom1, BLOBencoded geom2)
/ Relate(BLOBencoded geom1, BLOBencoded geom2, int bnr)
/
/ returns:
/ an intersection matrix [DE-9IM]
/ NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    const char *pattern = NULL;
    int bnr = 1;
    char *matrix;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  if (argc < 3)
	      sqlite3_result_null (context);
	  else
	    {
		if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
		    sqlite3_result_null (context);
		else
		    sqlite3_result_int (context, -1);
	    }
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  if (argc < 3)
	      sqlite3_result_null (context);
	  else
	    {
		if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
		    sqlite3_result_null (context);
		else
		    sqlite3_result_int (context, -1);
	    }
	  return;
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	      pattern = (const char *) sqlite3_value_text (argv[2]);
	  else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      bnr = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
      {
	  if (pattern == NULL)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_int (context, -1);
      }
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (pattern != NULL)
	    {
		/* evaluating the given intersection matrix pattern */
		if (data != NULL)
		    ret = gaiaGeomCollRelate_r (data, geo1, geo2, pattern);
		else
		    ret = gaiaGeomCollRelate (geo1, geo2, pattern);
		sqlite3_result_int (context, ret);
	    }
	  else
	    {
		/* returning an intersection matrix */
		if (data != NULL)
		    matrix =
			gaiaGeomCollRelateBoundaryNodeRule_r (data, geo1,
							      geo2, bnr);
		else
		    matrix =
			gaiaGeomCollRelateBoundaryNodeRule (geo1, geo2, bnr);
		if (matrix == NULL)
		    sqlite3_result_null (context);
		else
		    sqlite3_result_text (context, matrix, strlen (matrix),
					 free);
	    }
      }
    if (geo1 != NULL)
	gaiaFreeGeomColl (geo1);
    if (geo2 != NULL)
	gaiaFreeGeomColl (geo2);
}

static void
fnct_RelateMatch (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_RelateMatch(string matrix, string pattern)
/
/ returns:
/ 1 if the intersection matrix satisfies the intersection pattern
/ 0 otherwise
/ or -1 if any error is encountered
*/
    int ret;
    const char *matrix;
    const char *pattern;
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    else
	matrix = (char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    else
	pattern = (char *) sqlite3_value_text (argv[1]);
    if (data != NULL)
	ret = gaiaIntersectionMatrixPatternMatch_r (data, matrix, pattern);
    else
	ret = gaiaIntersectionMatrixPatternMatch (matrix, pattern);
    sqlite3_result_int (context, ret);
}

static void
fnct_Distance (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Distance(BLOBencoded geom1, BLOBencoded geom2)
/ Distance(BLOBencoded geom1, BLOBencoded geom2, Boolen use_ellipsoid)
/
/ returns the distance between GEOM-1 and GEOM-2
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double dist;
    int use_ellipsoid = -1;
    double a;
    double b;
    double rf;
    int ret;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    void *data = sqlite3_user_data (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  use_ellipsoid = sqlite3_value_int (argv[2]);
	  if (use_ellipsoid != 0)
	      use_ellipsoid = 1;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  if (use_ellipsoid >= 0)
	    {
		/* checking first if an intersection exists */
		if (data != NULL)
		    ret = gaiaGeomCollIntersects_r (data, geo1, geo2);
		else
		    ret = gaiaGeomCollIntersects (geo1, geo2);
		if (ret)
		  {
		      /* if an intersection exists the distance is always ZERO */
		      sqlite3_result_double (context, 0.0);
		      goto stop;
		  }

		/* attempting to identify the corresponding ellipsoid */
		if (getEllipsoidParams (sqlite, geo1->Srid, &a, &b, &rf))
		  {
		      gaiaGeomCollPtr shortest;
		      if (data != NULL)
			  shortest = gaiaShortestLine_r (data, geo1, geo2);
		      else
			  shortest = gaiaShortestLine (geo1, geo2);
		      if (shortest == NULL)
			  sqlite3_result_null (context);
		      else if (shortest->FirstLinestring == NULL)
			{
			    gaiaFreeGeomColl (shortest);
			    sqlite3_result_null (context);
			}
		      else
			{
			    /* computes the metric distance */
			    double x0;
			    double y0;
			    double x1;
			    double y1;
			    double z;
			    double m;
			    gaiaLinestringPtr ln = shortest->FirstLinestring;
			    dist = -1.0;
			    if (ln->Points == 2)
			      {
				  if (ln->DimensionModel == GAIA_XY_Z)
				    {
					gaiaGetPointXYZ (ln->Coords, 0,
							 &x0, &y0, &z);
				    }
				  else if (ln->DimensionModel == GAIA_XY_M)
				    {
					gaiaGetPointXYM (ln->Coords, 0,
							 &x0, &y0, &m);
				    }
				  else if (ln->DimensionModel == GAIA_XY_Z_M)
				    {
					gaiaGetPointXYZM (ln->Coords, 0,
							  &x0, &y0, &z, &m);
				    }
				  else
				    {
					gaiaGetPoint (ln->Coords, 0, &x0, &y0);
				    }
				  if (ln->DimensionModel == GAIA_XY_Z)
				    {
					gaiaGetPointXYZ (ln->Coords, 1,
							 &x1, &y1, &z);
				    }
				  else if (ln->DimensionModel == GAIA_XY_M)
				    {
					gaiaGetPointXYM (ln->Coords, 1,
							 &x1, &y1, &m);
				    }
				  else if (ln->DimensionModel == GAIA_XY_Z_M)
				    {
					gaiaGetPointXYZM (ln->Coords, 1,
							  &x1, &y1, &z, &m);
				    }
				  else
				    {
					gaiaGetPoint (ln->Coords, 1, &x1, &y1);
				    }
				  if (use_ellipsoid)
				      dist =
					  gaiaGeodesicDistance (a, b,
								rf, y0,
								x0, y1, x1);
				  else
				    {
					a = 6378137.0;
					rf = 298.257223563;
					b = (a * (1.0 - (1.0 / rf)));
					dist =
					    gaiaGreatCircleDistance (a,
								     b,
								     y0,
								     x0,
								     y1, x1);
				    }
				  if (dist < 0.0)
				    {
					/* invalid distance */
					sqlite3_result_null (context);
				    }
				  else
				      sqlite3_result_double (context, dist);
			      }
			    else
				sqlite3_result_null (context);
			    gaiaFreeGeomColl (shortest);
			}
		  }
		else
		    sqlite3_result_null (context);
		goto stop;
	    }
	  else
	    {
		if (data != NULL)
#ifdef GEOS_3100		/* only if GEOS_3100 support is available */
		    ret =
			gaiaGeomCollPreparedDistance (data, geo1, blob1, bytes1,
						      geo2, blob2, bytes2,
						      &dist);
#else
		    ret = gaiaGeomCollDistance_r (data, geo1, geo2, &dist);
#endif
		else
		    ret = gaiaGeomCollDistance (geo1, geo2, &dist);
		if (!ret)
		    sqlite3_result_null (context);
		else
		    sqlite3_result_double (context, dist);
	    }
      }
  stop:
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_PtDistWithin (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ PtDistWithin(BLOBencoded geom1, BLOBencoded geom2, double dist 
/ [, boolen use_spheroid])
/
/ returns TRUE if the distance between GEOM-1 and GEOM-2
/ is less or equal to dist
/
/ - if both geom1 and geom2 are in the 4326 (WGS84) SRID,
/   (and does actually contains a single POINT each one)
/   dist is assumed to be measured in Meters
/ - in this case the optional arg use_spheroid is
/   checked to determine if geodesic distance has to be
/   computed on the sphere (quickest) or on the spheroid 
/   default: use_spheroid = FALSE
/ 
/ in any other case the "plain" distance is evaluated
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    double ref_dist;
    int use_spheroid = 0;
    double x0 = 0.0;
    double y0 = 0.0;
    double x1 = 0.0;
    double y1 = 0.0;
    int pt0 = 0;
    int ln0 = 0;
    int pg0 = 0;
    int pt1 = 0;
    int ln1 = 0;
    int pg1 = 0;
    double dist;
    double a;
    double b;
    double rf;
    int ret;
    void *data = sqlite3_user_data (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER
	|| sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	;
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 4)
      {
	  /* optional use_spheroid arg */
	  if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int dst = sqlite3_value_int (argv[2]);
	  ref_dist = dst;
      }
    else
	ref_dist = sqlite3_value_double (argv[2]);
    if (argc == 4)
	use_spheroid = sqlite3_value_int (argv[3]);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  if (geo1->Srid == 4326 && geo2->Srid == 4326)
	    {
		/* checking for single points */
		pt = geo1->FirstPoint;
		while (pt)
		  {
		      x0 = pt->X;
		      y0 = pt->Y;
		      pt0++;
		      pt = pt->Next;
		  }
		ln = geo1->FirstLinestring;
		while (ln)
		  {
		      ln0++;
		      ln = ln->Next;
		  }
		pg = geo1->FirstPolygon;
		while (pg)
		  {
		      pg0++;
		      pg = pg->Next;
		  }
		pt = geo2->FirstPoint;
		while (pt)
		  {
		      x1 = pt->X;
		      y1 = pt->Y;
		      pt1++;
		      pt = pt->Next;
		  }
		ln = geo2->FirstLinestring;
		while (ln)
		  {
		      ln1++;
		      ln = ln->Next;
		  }
		pg = geo2->FirstPolygon;
		while (pg)
		  {
		      pg1++;
		      pg = pg->Next;
		  }
		if (pt0 == 1 && pt1 == 1 && ln0 == 0 && ln1 == 0 && pg0 == 0
		    && pg1 == 0)
		  {
		      /* using geodesic distance */
		      a = 6378137.0;
		      rf = 298.257223563;
		      b = (a * (1.0 - (1.0 / rf)));
		      if (use_spheroid)
			{
			    dist =
				gaiaGeodesicDistance (a, b, rf, y0, x0, y1, x1);
			    if (dist <= ref_dist)
				sqlite3_result_int (context, 1);
			    else
				sqlite3_result_int (context, 0);
			}
		      else
			{
			    dist =
				gaiaGreatCircleDistance (a, b, y0, x0, y1, x1);
			    if (dist <= ref_dist)
				sqlite3_result_int (context, 1);
			    else
				sqlite3_result_int (context, 0);
			}
		      goto stop;
		  }
	    }
/* defaulting to flat distance */
	  if (data != NULL)
	      ret = gaiaGeomCollDistance_r (data, geo1, geo2, &dist);
	  else
	      ret = gaiaGeomCollDistance (geo1, geo2, &dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  if (dist <= ref_dist)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
      }
  stop:
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

SPATIALITE_PRIVATE void
geos_error (const char *fmt, ...)
{
/* reporting some GEOS error */
    va_list ap;
    char *msg;
    va_start (ap, fmt);
    msg = sqlite3_vmprintf (fmt, ap);
    va_end (ap);
    if (msg)
      {
	  spatialite_e ("GEOS error: %s\n", msg);
	  gaiaSetGeosErrorMsg (msg);
	  sqlite3_free (msg);
      }
    else
	gaiaSetGeosErrorMsg (NULL);
}

SPATIALITE_PRIVATE void
geos_warning (const char *fmt, ...)
{
/* reporting some GEOS warning */
    va_list ap;
    char *msg;
    va_start (ap, fmt);
    msg = sqlite3_vmprintf (fmt, ap);
    va_end (ap);
    if (msg)
      {
	  spatialite_e ("GEOS warning: %s\n", msg);
	  gaiaSetGeosWarningMsg (msg);
	  sqlite3_free (msg);
      }
    else
	gaiaSetGeosWarningMsg (NULL);
}

static void
fnct_aux_polygonize (sqlite3_context * context, gaiaGeomCollPtr geom_org,
		     int force_multipolygon, int allow_multipolygon)
{
/* a  common function performing any kind of polygonization op */
    gaiaGeomCollPtr geom_new = NULL;
    int len;
    unsigned char *p_result = NULL;
    gaiaPolygonPtr pg;
    int pgs = 0;
    void *data = sqlite3_user_data (context);
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (!geom_org)
	goto invalid;
    if (data != NULL)
	geom_new = gaiaPolygonize_r (data, geom_org, force_multipolygon);
    else
	geom_new = gaiaPolygonize (geom_org, force_multipolygon);
    if (!geom_new)
	goto invalid;
    gaiaFreeGeomColl (geom_org);
    pg = geom_new->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pgs > 1 && allow_multipolygon == 0)
      {
	  /* invalid: a POLYGON is expected !!! */
	  gaiaFreeGeomColl (geom_new);
	  sqlite3_result_null (context);
	  return;
      }
    gaiaToSpatiaLiteBlobWkbEx2 (geom_new, &p_result, &len, gpkg_mode,
				tiny_point);
    gaiaFreeGeomColl (geom_new);
    sqlite3_result_blob (context, p_result, len, free);
    return;
  invalid:
    if (geom_org)
	gaiaFreeGeomColl (geom_org);
    sqlite3_result_null (context);
}

/*
/ the following functions performs initial argument checking, 
/ and then readdressing the request to fnct_aux_polygonize()
/ for actual processing
*/

static void
fnct_BdPolyFromText1 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ BdPolyFromText(WKT encoded MULTILINESTRING)
/
/ returns the current geometry [POLYGON] by parsing a WKT encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseWkt (text, -1);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = 0;
    fnct_aux_polygonize (context, geo, 0, 0);
    return;
}

static void
fnct_BdPolyFromText2 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ BdPolyFromText(WKT encoded MULTILINESTRING, SRID)
/
/ returns the current geometry [POLYGON] by parsing a WKT encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseWkt (text, -1);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = sqlite3_value_int (argv[1]);
    fnct_aux_polygonize (context, geo, 0, 0);
    return;
}

static void
fnct_BdMPolyFromText1 (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ BdMPolyFromText(WKT encoded MULTILINESTRING)
/
/ returns the current geometry [MULTIPOLYGON] by parsing a WKT encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseWkt (text, -1);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = 0;
    fnct_aux_polygonize (context, geo, 1, 1);
    return;
}

static void
fnct_BdMPolyFromText2 (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ BdMPolyFromText(WKT encoded MULTILINESTRING, SRID)
/
/ returns the current geometry [MULTIPOLYGON] by parsing a WKT encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    const unsigned char *text;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    text = sqlite3_value_text (argv[0]);
    geo = gaiaParseWkt (text, -1);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = sqlite3_value_int (argv[1]);
    fnct_aux_polygonize (context, geo, 1, 1);
    return;
}

static void
fnct_BdPolyFromWKB1 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ BdPolyFromWKB(WKB encoded MULTILINESTRING)
/
/ returns the current geometry [POLYGON] by parsing a WKB encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    int n_bytes;
    const unsigned char *wkb;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    wkb = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!check_wkb (wkb, n_bytes, -1))
	return;
    geo = gaiaFromWkb (wkb, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = 0;
    fnct_aux_polygonize (context, geo, 0, 0);
    return;
}

static void
fnct_BdPolyFromWKB2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ BdPolyFromWKB(WKB encoded MULTILINESTRING)
/
/ returns the current geometry [POLYGON] by parsing a WKB encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    int n_bytes;
    const unsigned char *wkb;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    wkb = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!check_wkb (wkb, n_bytes, -1))
	return;
    geo = gaiaFromWkb (wkb, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = sqlite3_value_int (argv[1]);
    fnct_aux_polygonize (context, geo, 0, 0);
    return;
}

static void
fnct_BdMPolyFromWKB1 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ BdMPolyFromWKB(WKB encoded MULTILINESTRING)
/
/ returns the current geometry [MULTIPOLYGON] by parsing a WKB encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    int n_bytes;
    const unsigned char *wkb;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    wkb = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!check_wkb (wkb, n_bytes, -1))
	return;
    geo = gaiaFromWkb (wkb, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = 0;
    fnct_aux_polygonize (context, geo, 1, 1);
    return;
}

static void
fnct_BdMPolyFromWKB2 (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ BdMPolyFromWKB(WKB encoded MULTILINESTRING)
/
/ returns the current geometry [MULTIPOLYGON] by parsing a WKB encoded MULTILINESTRING 
/ or NULL if any error is encountered
/
*/
    int n_bytes;
    const unsigned char *wkb;
    gaiaGeomCollPtr geo = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
      {
	  sqlite3_result_null (context);
	  return;
      }
    wkb = sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (!check_wkb (wkb, n_bytes, -1))
	return;
    geo = gaiaFromWkb (wkb, n_bytes);
    if (geo == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geo->DeclaredType != GAIA_MULTILINESTRING)
      {
	  gaiaFreeGeomColl (geo);
	  sqlite3_result_null (context);
	  return;
      }
    geo->Srid = sqlite3_value_int (argv[1]);
    fnct_aux_polygonize (context, geo, 1, 1);
    return;
}

static void
fnct_OffsetCurve (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ OffsetCurve(BLOBencoded geometry, DOUBLE radius)
/
/ returns a new geometry representing the OFFSET-CURVE for current geometry
/ [a LINESTRING is expected]
/ or NULL if any error is encountered
/
/ negative radius: right-side / positive radius: left-side
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    double radius;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	radius = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  radius = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaOffsetCurve_r (data, geo, radius, 16, 0);
	  else
	      result = gaiaOffsetCurve (geo, radius, 16, 0);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_SingleSidedBuffer (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ SingleSidedBuffer(BLOBencoded geometry, radius, left-or-right-side)
/
/ returns a new geometry representing the SingleSided BUFFER 
/ for current geometry [a LINESTRING is expected]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    double radius;
    int int_value;
    int left_right;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	radius = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  radius = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	left_right = sqlite3_value_int (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaSingleSidedBuffer_r (data, geo, radius, -1, left_right);
	  else
	      result = gaiaSingleSidedBuffer (geo, radius, 16, left_right);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_HausdorffDistance (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ HausdorffDistance(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns the discrete Hausdorff distance between GEOM-1 and GEOM-2
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double dist;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaHausdorffDistance_r (data, geo1, geo2, &dist);
	  else
	      ret = gaiaHausdorffDistance (geo1, geo2, &dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  sqlite3_result_double (context, dist);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

#ifdef GEOS_370			/* only if GEOS_370 support is available */

static void
fnct_HausdorffDistanceDensify (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ HausdorffDistance(BLOBencoded geom1, BLOBencoded geom2, double densify_fract)
/
/ returns the discrete Hausdorff distance between GEOM-1 and GEOM-2
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double densify_fract;
    double dist;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
      {
	  densify_fract = sqlite3_value_double (argv[2]);
	  if (densify_fract > 0.0 && densify_fract < 1.0)
	      ;
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret =
		  gaiaHausdorffDistanceDensify_r (data, geo1, geo2,
						  densify_fract, &dist);
	  else
	      ret =
		  gaiaHausdorffDistanceDensify (geo1, geo2, densify_fract,
						&dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  sqlite3_result_double (context, dist);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_FrechetDistance (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ FrechetDistance(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns the discrete Frechet distance between GEOM-1 and GEOM-2
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double dist;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaFrechetDistance_r (data, geo1, geo2, &dist);
	  else
	      ret = gaiaFrechetDistance (geo1, geo2, &dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  sqlite3_result_double (context, dist);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_FrechetDistanceDensify (sqlite3_context * context, int argc,
			     sqlite3_value ** argv)
{
/* SQL function:
/ FrechetDistance(BLOBencoded geom1, BLOBencoded geom2, double densify_fract)
/
/ returns the discrete Frechet distance between GEOM-1 and GEOM-2
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double densify_fract;
    double dist;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
      {
	  densify_fract = sqlite3_value_double (argv[2]);
	  if (densify_fract > 0.0 && densify_fract < 1.0)
	      ;
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret =
		  gaiaFrechetDistanceDensify_r (data, geo1, geo2, densify_fract,
						&dist);
	  else
	      ret =
		  gaiaFrechetDistanceDensify (geo1, geo2, densify_fract, &dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  sqlite3_result_double (context, dist);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_GEOSMinimumRotatedRectangle (sqlite3_context * context, int argc,
				  sqlite3_value ** argv)
{
/* SQL function:
/ GEOSMinimumRotatedRectangle(BLOBencoded geom)
/
/ Returns the minimum rotated rectangular POLYGON which encloses 
/ the input geometry
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMinimumRotatedRectangle_r (data, geom);
	  else
	      result = gaiaMinimumRotatedRectangle (geom);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geom->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_GEOSMinimumBoundingCircle (sqlite3_context * context, int argc,
				sqlite3_value ** argv)
{
/* SQL function:
/ GEOSMinimumBoundingCircle(BLOBencoded geom)
/
/ Constructs the Minimum Bounding Circle for a  generic geometry
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMinimumBoundingCircle_r (data, geom, NULL, NULL);
	  else
	      result = gaiaMinimumBoundingCircle (geom, NULL, NULL);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geom->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_GEOSMinimumBoundingRadius (sqlite3_context * context, int argc,
				sqlite3_value ** argv)
{
/* SQL function:
/ ST_MinimumBoundingRadius(BLOBencoded geom)
/
/ Returns the Radius of the Minimum Bounding Circle for a  generic geometry
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr result;
    double radius;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMinimumBoundingCircle_r (data, geom, &radius, NULL);
	  else
	      result = gaiaMinimumBoundingCircle (geom, &radius, NULL);
	  if (result)
	      gaiaFreeGeomColl (result);
	  sqlite3_result_null (context);
	  sqlite3_result_double (context, radius);
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_GEOSMinimumBoundingCenter (sqlite3_context * context, int argc,
				sqlite3_value ** argv)
{
/* SQL function:
/ ST_MinimumBoundingCenter(BLOBencoded geom)
/
/ Returns the Center of the Minimum Bounding Circle for a  generic geometry
/ (POINT Geometry)
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr center = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMinimumBoundingCircle_r (data, geom, NULL, &center);
	  else
	      result = gaiaMinimumBoundingCircle (geom, NULL, &center);
	  gaiaFreeGeomColl (result);
	  if (!center)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		center->Srid = geom->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (center, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (center);
	    }
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_GEOSMinimumWidth (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ GEOSGEOSMinimumWidth(BLOBencoded geom)
/
/ Constructs the Minimum Width (a Linestring) for a  generic geometry
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMinimumWidth_r (data, geom);
	  else
	      result = gaiaMinimumWidth (geom);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geom->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_GEOSMinimumClearance (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ GEOSMinimumClearance(BLOBencoded geom) 
/
/ Computes the minimum clearance of a geometry 
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom = NULL;
    double result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  int ret;
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaMinimumClearance_r (data, geom, &result);
	  else
	      ret = gaiaMinimumClearance (geom, &result);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, result);
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_GEOSMinimumClearanceLine (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ GEOSMinimumClearanceLine(BLOBencoded geom)
/
/ Constructs a LineString whose endpoints define the minimum clearance of a geometry 
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMinimumClearanceLine_r (data, geom);
	  else
	      result = gaiaMinimumClearanceLine (geom);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geom->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geom);
}

#endif /* end GEOS_370 conditional */

#ifdef GEOS_3100		/* only if GEOS_3100 support is available */

static void
fnct_GeosDensify (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeosDensify(BLOBencoded geometry, DOUBLE tolerance)
/
/ return a densified geometry using a given distance tolerance.
/ or NULL if any error is encountered
/
/ Additional vertices will be added to every line segment
/ that is greater this tolerance; these vertices will
/ evenly subdivide that segment.
/ Only linear components of input geometry are densified.
/
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    double tolerance;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	tolerance = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  tolerance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeosDensify_r (data, geo, tolerance);
	  else
	      result = gaiaGeosDensify (geo, tolerance);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ConstrainedDelaunayTriangulation (sqlite3_context * context, int argc,
				       sqlite3_value ** argv)
{
/* SQL function:
/ ConstrainedDelaunayTriangulation(BLOBencoded geometry)
/
/ Return a constrained Delaunay triangulation of the vertices of the
* given polygon(s).
/ NULL is returned for invalid arguments
* For non-polygonal inputs, returns a NULL geometry collection.
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaConstrainedDelaunayTriangulation_r (data, geo);
	  else
	      result = gaiaConstrainedDelaunayTriangulation (geo);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    if (geo != NULL)
	gaiaFreeGeomColl (geo);
}

static void
fnct_GeosMakeValid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeosMakeValid(BLOBencoded geometry)
/ GeosMakeValid(BLOBencoded geometry, BOOL keep_collapsed)
/
/ Attempts to make an invalid geometry valid (the GEOS way).
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    int keep_collapsed = 1;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      keep_collapsed = sqlite3_value_int (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeosMakeValid_r (data, geo, keep_collapsed);
	  else
	      result = gaiaGeosMakeValid (geo, keep_collapsed);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_GEOSMaximumInscribedCircle (sqlite3_context * context, int argc,
				 sqlite3_value ** argv)
{
/* SQL function:
/ GEOSMaximumInscribedCircle(BLOBencoded geom, double tolerance)
/
/ Constructs the Maximum Inscribed Circle for a  polygonal geometry, 
/ up to a specified tolerance.
*/
    unsigned char *p_blob;
    int n_bytes;
    double tolerance;
    int int_value;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	tolerance = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  tolerance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geom)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaMaximumInscribedCircle_r (data, geom, tolerance);
	  else
	      result = gaiaMaximumInscribedCircle (geom, tolerance);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geom->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_GEOSLargestEmptyCircle (sqlite3_context * context, int argc,
			     sqlite3_value ** argv)
{
/* SQL function:
/ GEOSLargestEmptyCircle(BLOBencoded geom, double tolerance)
/
/ Constructs the Largest Empty Circle for a set of obstacle geometries, 
/ up to a specified tolerance.
*/
    unsigned char *p_blob;
    int n_bytes;
    double tolerance;
    int int_value;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr boundary = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	tolerance = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  tolerance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom)
      {
	  /* building the Boundary */
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      boundary = gaiaConvexHull_r (data, geom);
	  else
	      boundary = gaiaConvexHull (geom);
      }
    if (!boundary)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaLargestEmptyCircle_r (data, geom, boundary, tolerance);
	  else
	      result = gaiaLargestEmptyCircle (geom, boundary, tolerance);
	  gaiaFreeGeomColl (boundary);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geom->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geom);
}

static void
fnct_ReducePrecision (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ ReducePrecision(BLOBencoded geometry, DOUBLE grid_size)
/
/ Change the coordinate precision of a geometry. This will
/ affect the precision of the existing geometry as well as
/ any geometries derived from this geometry using overlay
/ functions. The output will be a valid Geometry.
/
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    double grid_size;
    int int_value;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	grid_size = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  grid_size = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaReducePrecision_r (data, geo, grid_size);
	  else
	      result = gaiaReducePrecision (geo, grid_size);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

#endif /* end GEOS_3100 conditional */

#ifdef GEOS_3110		/* only if GEOS_3100 support is available */

static void
fnct_HilbertCode (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ HilbertCode(BLOBencoded geometry, BLOBencoded extent, int level)
/
/ Calculate the Hilbert Code of the centroid of a Geometry
/ realtive to an extent.
/ Level is the precision of Hilbert Curve [1-16]
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int level = 0;
    unsigned int code;
    int result;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    void *data = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	level = sqlite3_value_int (argv[2]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo1 == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo2 == NULL)
      {
	  sqlite3_result_null (context);
	  gaiaFreeGeomColl (geo1);
	  return;
      }

    if (level < 1)
	level = 1;
    if (level > 16)
	level = 16;

    if (data != NULL)
	result = gaiaHilbertCode_r (data, geo1, geo2, level, &code);
    else
	result = gaiaHilbertCode (geo1, geo2, level, &code);
    if (result == 0)
	sqlite3_result_null (context);
    else
	sqlite3_result_int64 (context, code);

    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_GeosConcaveHull (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ GeosConcaveHull(BLOBencoded geometry, DOUBLE ratio)
/ GeosConcaveHull(BLOBencoded geometry, DOUBLE ratio, BOOL allow_holes)
/
/ Attempts to build a ConcaveHull using all points/vertices 
/ found in the input geometry (the GEOS way).
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    double ratio;
    int allow_holes = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int intval = sqlite3_value_int (argv[1]);
	  ratio = intval;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	ratio = sqlite3_value_double (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      allow_holes = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (ratio < 0.0)
	ratio = 0.0;
    if (ratio > 1.0)
	ratio = 1.0;
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaGeosConcaveHull_r (data, geo, ratio, allow_holes);
	  else
	      result = gaiaGeosConcaveHull (geo, ratio, allow_holes);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

#endif /* end GEOS_3110 conditional */


static void
fnct_DistanceWithin (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ DistanceWithin(BLOBencoded geom1, BLOBencoded geom2, double dist)
/
/ Test whether the distance between two geometries is
/ within the given dist.
/ Will return TRUE or FALSE; -1 on invalid args
/
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int int_val;
    double dist;
    double xdist;
    int ret;
    void *data = sqlite3_user_data (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int_val = sqlite3_value_int (argv[2]);
		dist = int_val;
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      dist = sqlite3_value_double (argv[2]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  int result = 0;
	  if (data != NULL)
	    {
#ifdef GEOS_3100		/* only if GEOS_3100 support is available */
		ret =
		    gaiaGeomCollPreparedDistanceWithin (data, geo1, blob1,
							bytes1, geo2, blob2,
							bytes2, dist);
		result = ret;
#else
		ret = gaiaGeomCollDistance_r (data, geo1, geo2, &xdist);
		if (ret && xdist <= dist)
		    result = 1;
#endif
	    }
	  else
	    {
		ret = gaiaGeomCollDistance (geo1, geo2, &xdist);
		if (ret && xdist <= dist)
		    result = 1;
	    }
	  sqlite3_result_int (context, result);
      }
    if (geo1 != NULL)
	gaiaFreeGeomColl (geo1);
    if (geo2 != NULL)
	gaiaFreeGeomColl (geo2);
}

static void
fnct_SharedPaths (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SharedPaths(BLOBencoded geometry1, BLOBencoded geometry2)
/
/ returns a new geometry representing common (shared) Edges
/ [two LINESTRINGs/MULTILINESTRINGs are expected]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo1 == NULL || geo2 == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaSharedPaths_r (data, geo1, geo2);
	  else
	      result = gaiaSharedPaths (geo1, geo2);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo1->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Covers (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Covers(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if GEOM-1 "spatially covers" GEOM-2
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedCovers (data, geo1,
						blob1, bytes1, geo2, blob2,
						bytes2);
	  else
	      ret = gaiaGeomCollCovers (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_CoveredBy (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CoveredBy(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns:
/ 1 if GEOM-1 is "spatially covered by" GEOM-2
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *blob1;
    unsigned char *blob2;
    int bytes1;
    int bytes2;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    blob1 = (unsigned char *) sqlite3_value_blob (argv[0]);
    bytes1 = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (blob1, bytes1, gpkg_mode, gpkg_amphibious);
    blob2 = (unsigned char *) sqlite3_value_blob (argv[1]);
    bytes2 = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (blob2, bytes2, gpkg_mode, gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_int (context, -1);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      ret = gaiaGeomCollPreparedCoveredBy (data,
						   geo1, blob1, bytes1, geo2,
						   blob2, bytes2);
	  else
	      ret = gaiaGeomCollCoveredBy (geo1, geo2);
	  sqlite3_result_int (context, ret);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_LineInterpolatePoint (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ LineInterpolatePoint(BLOBencoded geometry1, double fraction)
/
/ returns a new geometry representing a point interpolated along a line
/ [a LINESTRING is expected / fraction ranging from 0.0 to 1.0]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double fraction;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	fraction = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  fraction = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaLineInterpolatePoint_r (data, geo, fraction);
	  else
	      result = gaiaLineInterpolatePoint (geo, fraction);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_LineInterpolateEquidistantPoints (sqlite3_context * context, int argc,
				       sqlite3_value ** argv)
{
/* SQL function:
/ LineInterpolateEquidistantPointS(BLOBencoded geometry1, double distance)
/
/ returns a new geometry representing a point interpolated along a line
/ [a LINESTRING is expected / fraction ranging from 0.0 to 1.0]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double distance;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	distance = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  distance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaLineInterpolateEquidistantPoints_r (data, geo, distance);
	  else
	      result = gaiaLineInterpolateEquidistantPoints (geo, distance);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_LineLocatePoint (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ LineLocatePoint(BLOBencoded geometry1, BLOBencoded geometry2)
/
/ return a number (between 0.0 and 1.0) representing the location 
/ of the closest point on LineString to the given Point, as a fraction 
/ of total 2d line length
/
/ - geom1 is expected to represent some LINESTRING
/ - geom2 is expected to represent some POINT
*/
    unsigned char *p_blob;
    int n_bytes;
    double fraction;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo1 == NULL || geo2 == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      fraction = gaiaLineLocatePoint_r (data, geo1, geo2);
	  else
	      fraction = gaiaLineLocatePoint (geo1, geo2);
	  if (fraction >= 0.0 && fraction <= 1.0)
	      sqlite3_result_double (context, fraction);
	  else
	      sqlite3_result_null (context);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_LineSubstring (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ LineSubstring(BLOBencoded geometry1, double start_fraction, double end_fraction)
/
/ Return a Linestring being a substring of the input one starting and ending at 
/ the given fractions of total 2d length [fractions ranging from 0.0 to 1.0]
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double fraction1;
    double fraction2;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	fraction1 = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  fraction1 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	fraction2 = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  fraction2 = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaLineSubstring_r (data, geo, fraction1, fraction2);
	  else
	      result = gaiaLineSubstring (geo, fraction1, fraction2);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ClosestPoint (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ClosestPoint(BLOBencoded geometry1, BLOBencoded geometry2)
/
/ Returns the Point on geom1 that is closest to geom2
/ NULL is returned for invalid arguments (or if distance is ZERO)
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo1 == NULL || geo2 == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaShortestLine_r (data, geo1, geo2);
	  else
	      result = gaiaShortestLine (geo1, geo2);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else if (result->FirstLinestring == NULL)
	    {
		gaiaFreeGeomColl (result);
		sqlite3_result_null (context);
	    }
	  else
	    {
		/* builds the BLOB geometry to be returned */
		double x;
		double y;
		double z;
		double m;
		int len;
		unsigned char *p_result = NULL;
		gaiaGeomCollPtr pt = NULL;
		gaiaLinestringPtr ln = result->FirstLinestring;
		if (ln->DimensionModel == GAIA_XY_Z)
		    pt = gaiaAllocGeomCollXYZ ();
		else if (ln->DimensionModel == GAIA_XY_M)
		    pt = gaiaAllocGeomCollXYM ();
		else if (ln->DimensionModel == GAIA_XY_Z_M)
		    pt = gaiaAllocGeomCollXYZM ();
		else
		    pt = gaiaAllocGeomColl ();
		if (ln->DimensionModel == GAIA_XY_Z)
		  {
		      gaiaGetPointXYZ (ln->Coords, 0, &x, &y, &z);
		      gaiaAddPointToGeomCollXYZ (pt, x, y, z);
		  }
		else if (ln->DimensionModel == GAIA_XY_M)
		  {
		      gaiaGetPointXYM (ln->Coords, 0, &x, &y, &m);
		      gaiaAddPointToGeomCollXYM (pt, x, y, m);
		  }
		else if (ln->DimensionModel == GAIA_XY_Z_M)
		  {
		      gaiaGetPointXYZM (ln->Coords, 0, &x, &y, &z, &m);
		      gaiaAddPointToGeomCollXYZM (pt, x, y, z, m);
		  }
		else
		  {
		      gaiaGetPoint (ln->Coords, 0, &x, &y);
		      gaiaAddPointToGeomColl (pt, x, y);
		  }
		pt->Srid = geo1->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (pt, &p_result, &len, gpkg_mode,
					    tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
		gaiaFreeGeomColl (pt);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_ShortestLine (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ShortestLine(BLOBencoded geometry1, BLOBencoded geometry2)
/
/ Returns the shortest line between two geometries
/ NULL is returned for invalid arguments (or if distance is ZERO)
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo1 == NULL || geo2 == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaShortestLine_r (data, geo1, geo2);
	  else
	      result = gaiaShortestLine (geo1, geo2);
	  sqlite3_result_null (context);
	  if (!result)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo1->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Snap (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Snap(BLOBencoded geometry1, BLOBencoded geometry2, double tolerance)
/
/ Returns a new Geometry corresponding to geom1 snapped to geom2
/ and using the given tolerance
/ NULL is returned for invalid arguments (or if distance is ZERO)
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double tolerance;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	tolerance = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  tolerance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo1 == NULL || geo2 == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaSnap_r (data, geo1, geo2, tolerance);
	  else
	      result = gaiaSnap (geo1, geo2, tolerance);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo1->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_LineMerge (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ LineMerge(BLOBencoded geometry)
/
/ Assuming that Geometry represents a set of sparse Linestrings,
/ this function will attempt to reassemble a single line
/ (or a set of lines)
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaLineMerge_r (data, geo);
	  else
	      result = gaiaLineMerge (geo);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_UnaryUnion (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ UnaryUnion(BLOBencoded geometry)
/
/ exactly like Union, but using a single Collection
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result = gaiaUnaryUnion_r (data, geo);
	  else
	      result = gaiaUnaryUnion (geo);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_SquareGrid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_SquareGrid(BLOBencoded geom, double size)
/ ST_SquareGrid(BLOBencoded geom, double size, int mode)
/ ST_SquareGrid(BLOBencoded geom, double size, int mode, BLOBencoded origin)
/
/ mode interpretation:
/ ==============================================
/ positive = a MULTILINESTRING will be returned
/        0 = a MULTIPOLYGON will be returned
/ negative = a MULTIPOINT will be returned
/ ==============================================
/
/ Builds a regular grid (Square cells) covering the geom.
/ each cell has the edges's length as defined by the size argument
/ an arbitrary origin is supported (0,0 is assumed by default)
/ return NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double origin_x = 0.0;
    double origin_y = 0.0;
    double size;
    int mode = 0;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr point = NULL;
    gaiaGeomCollPtr result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  size = int_value;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
      {
	  size = sqlite3_value_double (argv[1]);
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (size <= 0.0)
      {
	  /* negative side size */
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      mode = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[3]);
	  n_bytes = sqlite3_value_bytes (argv[3]);
	  point =
	      gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
					   gpkg_amphibious);
	  if (!point)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (point->FirstLinestring != NULL)
	      goto no_point;
	  if (point->FirstPolygon != NULL)
	      goto no_point;
	  if (point->FirstPoint != NULL)
	    {
		if (point->FirstPoint == point->LastPoint)
		  {
		      origin_x = point->FirstPoint->X;
		      origin_y = point->FirstPoint->Y;
		      gaiaFreeGeomColl (point);
		  }
		else
		    goto no_point;
	    }
	  else
	      goto no_point;

      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (geo->FirstPoint != NULL)
	      goto no_polygon;
	  if (geo->FirstLinestring != NULL)
	      goto no_polygon;
	  if (geo->FirstPolygon == NULL)
	      goto no_polygon;
	  if (data != NULL)
	      result =
		  gaiaSquareGrid_r (data, geo, origin_x, origin_y, size, mode);
	  else
	      result = gaiaSquareGrid (geo, origin_x, origin_y, size, mode);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
    return;

  no_point:
    gaiaFreeGeomColl (point);
    sqlite3_result_null (context);
    return;

  no_polygon:
    gaiaFreeGeomColl (geo);
    sqlite3_result_null (context);
    return;
}

static void
fnct_TriangularGrid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_TriangularGrid(BLOBencoded geom, double size)
/ ST_TriangularGrid(BLOBencoded geom, double size, int mode)
/ ST_TriangularGrid(BLOBencoded geom, double size, int mode, BLOBencoded origin)
/
/ mode interpretation:
/ ==============================================
/ positive = a MULTILINESTRING will be returned
/        0 = a MULTIPOLYGON will be returned
/ negative = a MULTIPOINT will be returned
/ ==============================================
/
/ Builds a regular grid (Triangular cells) covering the geom.
/ each cell has the edge's length as defined by the size argument
/ an arbitrary origin is supported (0,0 is assumed by default)
/ return NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double origin_x = 0.0;
    double origin_y = 0.0;
    double size;
    int mode = 0;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr point = NULL;
    gaiaGeomCollPtr result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  size = int_value;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
      {
	  size = sqlite3_value_double (argv[1]);
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (size <= 0.0)
      {
	  /* negative side size */
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      mode = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[3]);
	  n_bytes = sqlite3_value_bytes (argv[3]);
	  point =
	      gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
					   gpkg_amphibious);
	  if (!point)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (point->FirstLinestring != NULL)
	      goto no_point;
	  if (point->FirstPolygon != NULL)
	      goto no_point;
	  if (point->FirstPoint != NULL)
	    {
		if (point->FirstPoint == point->LastPoint)
		  {
		      origin_x = point->FirstPoint->X;
		      origin_y = point->FirstPoint->Y;
		      gaiaFreeGeomColl (point);
		  }
		else
		    goto no_point;
	    }
	  else
	      goto no_point;

      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (geo->FirstPoint != NULL)
	      goto no_polygon;
	  if (geo->FirstLinestring != NULL)
	      goto no_polygon;
	  if (geo->FirstPolygon == NULL)
	      goto no_polygon;
	  if (data != NULL)
	      result =
		  gaiaTriangularGrid_r (data, geo, origin_x, origin_y, size,
					mode);
	  else
	      result = gaiaTriangularGrid (geo, origin_x, origin_y, size, mode);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
    return;

  no_point:
    gaiaFreeGeomColl (point);
    sqlite3_result_null (context);
    return;

  no_polygon:
    gaiaFreeGeomColl (geo);
    sqlite3_result_null (context);
    return;
}

static void
fnct_HexagonalGrid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_HexagonalGrid(BLOBencoded geom, double size)
/ ST_HexagonalGrid(BLOBencoded geom, double size, int mode)
/ ST_HexagonalGrid(BLOBencoded geom, double size, int mode, BLOBencoded origin)
/
/ mode interpretation:
/ ==============================================
/ positive = a MULTILINESTRING will be returned
/        0 = a MULTIPOLYGON will be returned
/ negative = a MULTIPOINT will be returned
/ ==============================================
/
/ Builds a regular grid (Hexagonal cells) covering the geom.
/ each cell has the edges's length as defined by the size argument
/ an arbitrary origin is supported (0,0 is assumed by default)
/ return NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double origin_x = 0.0;
    double origin_y = 0.0;
    double size;
    int mode = 0;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr point = NULL;
    gaiaGeomCollPtr result = NULL;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  size = int_value;
      }
    else if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
      {
	  size = sqlite3_value_double (argv[1]);
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (size <= 0.0)
      {
	  /* negative side size */
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      mode = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_BLOB)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  p_blob = (unsigned char *) sqlite3_value_blob (argv[3]);
	  n_bytes = sqlite3_value_bytes (argv[3]);
	  point =
	      gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
					   gpkg_amphibious);
	  if (!point)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (point->FirstLinestring != NULL)
	      goto no_point;
	  if (point->FirstPolygon != NULL)
	      goto no_point;
	  if (point->FirstPoint != NULL)
	    {
		if (point->FirstPoint == point->LastPoint)
		  {
		      origin_x = point->FirstPoint->X;
		      origin_y = point->FirstPoint->Y;
		      gaiaFreeGeomColl (point);
		  }
		else
		    goto no_point;
	    }
	  else
	      goto no_point;

      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (geo->FirstPoint != NULL)
	      goto no_polygon;
	  if (geo->FirstLinestring != NULL)
	      goto no_polygon;
	  if (geo->FirstPolygon == NULL)
	      goto no_polygon;
	  if (data != NULL)
	      result =
		  gaiaHexagonalGrid_r (data, geo, origin_x, origin_y, size,
				       mode);
	  else
	      result = gaiaHexagonalGrid (geo, origin_x, origin_y, size, mode);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
    return;

  no_point:
    gaiaFreeGeomColl (point);
    sqlite3_result_null (context);
    return;

  no_polygon:
    gaiaFreeGeomColl (geo);
    sqlite3_result_null (context);
    return;
}

static void
fnct_LinesCutAtNodes (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ LinesCutAtNodes(BLOBencoded geometry1, BLOBencoded geometry2)
/
/ Assuming that Geometry-1 represents a set of arbitray Linestrings,
/ and that Geometry-2 represents of arbitrary Points, this function 
/ will then attempt to cut lines accordingly to given nodes.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom1 = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geom2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom1 == NULL || geom2 == NULL)
      {
	  if (geom1)
	      gaiaFreeGeomColl (geom1);
	  if (geom2)
	      gaiaFreeGeomColl (geom2);
	  sqlite3_result_null (context);
	  return;
      }
    result = gaiaLinesCutAtNodes (geom1, geom2);
    if (result == NULL)
      {
	  sqlite3_result_null (context);
      }
    else
      {
	  /* builds the BLOB geometry to be returned */
	  int len;
	  unsigned char *p_result = NULL;
	  result->Srid = geom1->Srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (result);
      }
    gaiaFreeGeomColl (geom1);
    gaiaFreeGeomColl (geom2);
}

static int
cmp_pt_coords (const void *p1, const void *p2)
{
/* compares two nodes  by ID [for QSORT] */
    gaiaPointPtr pt1 = *((gaiaPointPtr *) p1);
    gaiaPointPtr pt2 = *((gaiaPointPtr *) p2);
    if (pt1->X == pt2->X && pt1->Y == pt2->Y && pt1->Z == pt2->Z)
	return 0;
    if (pt1->X > pt2->X)
	return 1;
    if (pt1->X == pt2->X && pt1->Y > pt2->Y)
	return 1;
    if (pt1->X == pt2->X && pt1->Y == pt2->Y && pt1->Z > pt2->Z)
	return 1;
    return -1;
}

static gaiaGeomCollPtr
auxPolygNodes (gaiaGeomCollPtr geom)
{
/* attempting to identify Ring-Nodes */
    gaiaGeomCollPtr result = NULL;
    gaiaPolygonPtr pg;
    gaiaRingPtr rng;
    gaiaPointPtr pt;
    gaiaPointPtr prev_pt;
    gaiaPointPtr *sorted = NULL;
    int count = 0;
    int iv;
    int ib;
    double x;
    double y;
    double z;
    double m;

/* inserting all Points into a Dynamic Line */
    gaiaDynamicLinePtr dyn = gaiaAllocDynamicLine ();
    pg = geom->FirstPolygon;
    while (pg)
      {
	  rng = pg->Exterior;
	  /* CAVEAT: first point needs to be skipped (closed ring) */
	  for (iv = 1; iv < rng->Points; iv++)
	    {
		/* exterior ring */
		if (geom->DimensionModel == GAIA_XY_Z)
		  {
		      gaiaGetPointXYZ (rng->Coords, iv, &x, &y, &z);
		      gaiaAppendPointZToDynamicLine (dyn, x, y, z);
		  }
		else if (geom->DimensionModel == GAIA_XY_M)
		  {
		      gaiaGetPointXYM (rng->Coords, iv, &x, &y, &m);
		      gaiaAppendPointMToDynamicLine (dyn, x, y, m);
		  }
		else if (geom->DimensionModel == GAIA_XY_Z_M)
		  {
		      gaiaGetPointXYZM (rng->Coords, iv, &x, &y, &z, &m);
		      gaiaAppendPointZMToDynamicLine (dyn, x, y, z, m);
		  }
		else
		  {
		      gaiaGetPoint (rng->Coords, iv, &x, &y);
		      gaiaAppendPointToDynamicLine (dyn, x, y);
		  }
	    }

	  for (ib = 0; ib < pg->NumInteriors; ib++)
	    {
		rng = pg->Interiors + ib;
		/* CAVEAT: first point needs to be skipped (closed ring) */
		for (iv = 1; iv < rng->Points; iv++)
		  {
		      /* interior ring */
		      if (geom->DimensionModel == GAIA_XY_Z)
			{
			    gaiaGetPointXYZ (rng->Coords, iv, &x, &y, &z);
			    gaiaAppendPointZToDynamicLine (dyn, x, y, z);
			}
		      else if (geom->DimensionModel == GAIA_XY_M)
			{
			    gaiaGetPointXYM (rng->Coords, iv, &x, &y, &m);
			    gaiaAppendPointMToDynamicLine (dyn, x, y, m);
			}
		      else if (geom->DimensionModel == GAIA_XY_Z_M)
			{
			    gaiaGetPointXYZM (rng->Coords, iv, &x, &y, &z, &m);
			    gaiaAppendPointZMToDynamicLine (dyn, x, y, z, m);
			}
		      else
			{
			    gaiaGetPoint (rng->Coords, iv, &x, &y);
			    gaiaAppendPointToDynamicLine (dyn, x, y);
			}
		  }
	    }
	  pg = pg->Next;
      }

    pt = dyn->First;
    while (pt)
      {
	  /* counting how many points */
	  count++;
	  pt = pt->Next;
      }
    if (count == 0)
      {
	  gaiaFreeDynamicLine (dyn);
	  return NULL;
      }

/* allocating and initializing an array of pointers */
    sorted = malloc (sizeof (gaiaPointPtr) * count);
    iv = 0;
    pt = dyn->First;
    while (pt)
      {
	  *(sorted + iv++) = pt;
	  pt = pt->Next;
      }

/* sorting points by coords */
    qsort (sorted, count, sizeof (gaiaPointPtr), cmp_pt_coords);

    if (geom->DimensionModel == GAIA_XY_Z)
	result = gaiaAllocGeomCollXYZ ();
    else if (geom->DimensionModel == GAIA_XY_M)
	result = gaiaAllocGeomCollXYM ();
    else if (geom->DimensionModel == GAIA_XY_Z_M)
	result = gaiaAllocGeomCollXYZM ();
    else
	result = gaiaAllocGeomColl ();
    result->Srid = geom->Srid;

/* identifying nodes */
    prev_pt = NULL;
    for (iv = 0; iv < count; iv++)
      {
	  pt = *(sorted + iv);
	  if (prev_pt != NULL)
	    {
		if (prev_pt->X == pt->X && prev_pt->Y == pt->Y
		    && prev_pt->Z == pt->Z)
		  {
		      if (result->LastPoint != NULL)
			{
			    if (result->LastPoint->X == pt->X
				&& result->LastPoint->Y == pt->Y
				&& result->LastPoint->Z == pt->Z)
				continue;
			}
		      /* Node found */
		      if (result->DimensionModel == GAIA_XY_Z)
			  gaiaAddPointToGeomCollXYZ (result, pt->X, pt->Y,
						     pt->Z);
		      else if (result->DimensionModel == GAIA_XY_M)
			  gaiaAddPointToGeomCollXYM (result, pt->X, pt->Y,
						     pt->M);
		      else if (result->DimensionModel == GAIA_XY_Z_M)
			  gaiaAddPointToGeomCollXYZM (result, pt->X,
						      pt->Y, pt->Z, pt->M);
		      else
			  gaiaAddPointToGeomColl (result, pt->X, pt->Y);
		  }
	    }
	  prev_pt = pt;
      }

    if (result->FirstPoint == NULL)
      {
	  gaiaFreeGeomColl (result);
	  result = NULL;
      }
    free (sorted);
    gaiaFreeDynamicLine (dyn);
    return result;
}

static void
fnct_RingsCutAtNodes (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ RingsCutAtNodes(BLOBencoded geometry)
/
/ This function will attempt to return a collection of lines
/ representing Polygon/Rings: the input geometry is expected
/ to be a Polygon or MultiPolygon.
/ Each Ring will be cut accordingly to any identified "node"
/ i.e. self-intersections or intersections between two Rings.
/
/ NULL is returned for invalid arguments
*/
    int pts = 0;
    int lns = 0;
    int pgs = 0;
    unsigned char *p_blob;
    int n_bytes;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;
    gaiaGeomCollPtr geom = NULL;
    gaiaGeomCollPtr geom1 = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }

/* checking if Geometry is a Polygon or MultiPolyhon */
    pt = geom->FirstPoint;
    while (pt)
      {
	  pts++;
	  pt = pt->Next;
      }
    ln = geom->FirstLinestring;
    while (ln)
      {
	  lns++;
	  ln = ln->Next;
      }
    pg = geom->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pts > 0 || lns > 0 || pgs == 0)
      {
	  /* not Polygon/MultiPolygon */
	  gaiaFreeGeomColl (geom);
	  sqlite3_result_null (context);
	  return;
      }

/* transforming Rings into Linestrings */
    geom1 = gaiaLinearize (geom, 1);
    if (geom1 == NULL)
      {
	  gaiaFreeGeomColl (geom);
	  sqlite3_result_null (context);
	  return;
      }

/* identifying the Nodes */
    geom2 = auxPolygNodes (geom);
    if (geom2 == NULL)
      {
	  /* there is no need to cut any Ring [no Nodes] */
	  int len;
	  unsigned char *p_result = NULL;
	  geom1->Srid = geom->Srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (geom1, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (geom);
	  gaiaFreeGeomColl (geom1);
	  return;
      }

/* attempting to cut Rings */
    result = gaiaLinesCutAtNodes (geom1, geom2);
    if (result == NULL)
	sqlite3_result_null (context);
    else
      {
	  /* builds the BLOB geometry to be returned */
	  int len;
	  unsigned char *p_result = NULL;
	  result->Srid = geom->Srid;
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_result, len, free);
	  gaiaFreeGeomColl (result);
      }
    gaiaFreeGeomColl (geom);
    gaiaFreeGeomColl (geom1);
    gaiaFreeGeomColl (geom2);
}

#ifdef GEOS_ADVANCED		/* GEOS advanced features - 3.4.0 */

static void
fnct_DelaunayTriangulation (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ DelaunayTriangulation(BLOBencoded geometry)
/ DelaunayTriangulation(BLOBencoded geometry, boolean onlyEdges)
/ DelaunayTriangulation(BLOBencoded geometry, boolean onlyEdges, double tolerance)
/
/ Attempts to build a Delaunay Triangulation using all points/vertices 
/ found in the input geometry.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int int_value;
    double tolerance = 0.0;
    int only_edges = 0;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      only_edges = sqlite3_value_int (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      tolerance = sqlite3_value_double (argv[2]);
	  else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[2]);
		tolerance = int_value;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaDelaunayTriangulation_r (data, geo, tolerance,
					       only_edges);
	  else
	      result = gaiaDelaunayTriangulation (geo, tolerance, only_edges);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_VoronojDiagram (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ VoronojDiagram(BLOBencoded geometry)
/ VoronojDiagram(BLOBencoded geometry, boolean onlyEdges)
/ VoronojDiagram(BLOBencoded geometry, boolean onlyEdges, 
/        double extra_frame_size)
/ VoronojDiagram(BLOBencoded geometry, boolean onlyEdges,
/        double extra_frame_size, double tolerance)
/
/ Attempts to build a Voronoj Diagram using all points/vertices 
/ found in the input geometry.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int int_value;
    double tolerance = 0.0;
    double extra_frame_size = -1.0;
    int only_edges = 0;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      only_edges = sqlite3_value_int (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      extra_frame_size = sqlite3_value_double (argv[2]);
	  else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[2]);
		extra_frame_size = int_value;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	      tolerance = sqlite3_value_double (argv[3]);
	  else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[3]);
		tolerance = int_value;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaVoronojDiagram_r (data, geo, extra_frame_size,
					tolerance, only_edges);
	  else
	      result =
		  gaiaVoronojDiagram (geo, extra_frame_size, tolerance,
				      only_edges);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_ConcaveHull (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ConcaveHull(BLOBencoded geometry)
/ ConcaveHull(BLOBencoded geometry, double factor)
/ ConcaveHull(BLOBencoded geometry, double factor, boolean allow_holes)
/ ConcaveHull(BLOBencoded geometry, double factor,
/        boolean allow_holes, double tolerance)
/
/ Attempts to build a ConcaveHull using all points/vertices 
/ found in the input geometry.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int int_value;
    double tolerance = 0.0;
    double factor = 3.0;
    int allow_holes = 0;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	      factor = sqlite3_value_double (argv[1]);
	  else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[1]);
		factor = int_value;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      allow_holes = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_FLOAT)
	      tolerance = sqlite3_value_double (argv[3]);
	  else if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	    {
		int_value = sqlite3_value_int (argv[3]);
		tolerance = int_value;
	    }
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result =
		  gaiaConcaveHull_r (data, geo, factor, tolerance, allow_holes);
	  else
	      result = gaiaConcaveHull (geo, factor, tolerance, allow_holes);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

#endif /* end GEOS advanced features */

#ifdef ENABLE_RTTOPO		/* enabling RTTOPO support */

static void
fnct_RTTOPO_GetLastWarningMsg (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
/* SQL function:
/ RTTOPO_GetLastWarningMsg()
/
/ return the most recent RTTOPO warning message (if any)
/ return NULL on any other case
*/
    const char *msg;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    msg = gaiaGetRtTopoWarningMsg (cache);
    if (msg == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, msg, strlen (msg), SQLITE_STATIC);
}

static void
fnct_RTTOPO_GetLastErrorMsg (sqlite3_context * context, int argc,
			     sqlite3_value ** argv)
{
/* SQL function:
/ RTTOPO_GetLastErrorMsg()
/
/ return the most recent RTEOM error message (if any)
/ return NULL on any other case
*/
    const char *msg;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    msg = gaiaGetRtTopoErrorMsg (cache);
    if (msg == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, msg, strlen (msg), SQLITE_STATIC);
}

static void
fnct_MakeValid (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MakeValid(BLOBencoded geometry)
/
/ Attempts to make an invalid geometry valid without loosing vertices.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  result = gaiaMakeValid (cache, geo);
	  if (result == NULL)
	    {
		char *msg;
		const char *lw_err = gaiaGetRtTopoErrorMsg (cache);
		if (lw_err)
		    msg = sqlite3_mprintf
			("MakeValid error - RTTOPO reports: %s\n", lw_err);
		else
		    msg = sqlite3_mprintf
			("MakeValid error - RTTOPO reports: Unknown Reason\n");
		sqlite3_result_error (context, msg, strlen (msg));
		sqlite3_free (msg);
	    }
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_MakeValidDiscarded (sqlite3_context * context, int argc,
			 sqlite3_value ** argv)
{
/* SQL function:
/ MakeValidDiscarded(BLOBencoded geometry)
/
/ Strictly related to MakeValid(); useful to collect any offending item
/ discarded during the validation process.
/ NULL is returned for invalid arguments (or if no discarded items are found)
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  result = gaiaMakeValidDiscarded (cache, geo);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static void
fnct_Segmentize (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Segmentize(BLOBencoded geometry, double dist)
/
/ Ensure every segment is at most 'dist' long
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double dist;
    gaiaGeomCollPtr geo = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	dist = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  dist = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo == NULL)
	sqlite3_result_null (context);
    else
      {
	  result = gaiaSegmentize (cache, geo, dist);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = geo->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (geo);
}

static int
is_line (gaiaGeomCollPtr geom)
{
/* checking for a Linestring or MultiLinestring */
    if (geom->FirstPoint != NULL || geom->FirstPolygon != NULL)
	return 0;
    if (geom->FirstLinestring != NULL)
	return 1;
    return 0;
}

static int
is_point_blade (gaiaGeomCollPtr geom)
{
/* checking for a Point or MultiPoint */
    if (geom->FirstLinestring != NULL || geom->FirstPolygon != NULL)
	return 0;
    if (geom->FirstPoint != NULL)
	return 1;
    return 0;
}

static void
fnct_SnapAndSplit (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SnapAndSplit(BLOBencoded geometry1, BLOBencoded geometry2, double tolerance)
/
/ - geometry1 is expected to be a LINESTRING or MULTILINESTRING
/ - geometry2 is expected to be a MULTIPOINT
/ - in a first pass geometry1 will be snapped against geometry2
/ - then the intermediate result of Snap will be Split using
/   geometry2 as the cutting blade.
/ .- 
/ Returns a new Geometry of the MULTILINESTRING type
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    int int_value;
    double tolerance;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    gaiaGeomCollPtr result_snap;
    gaiaGeomCollPtr result_split;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	tolerance = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[2]);
	  tolerance = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geo1 == NULL || geo2 == NULL)
	sqlite3_result_null (context);
    else if (is_line (geo1) == 0)
	sqlite3_result_null (context);
    else if (is_point_blade (geo2) == 0)
	sqlite3_result_null (context);
    else
      {
	  void *data = sqlite3_user_data (context);
	  if (data != NULL)
	      result_snap = gaiaSnap_r (data, geo1, geo2, tolerance);
	  else
	      result_snap = gaiaSnap (geo1, geo2, tolerance);
	  if (result_snap == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		result_split = gaiaSplit (cache, result_snap, geo2);
		gaiaFreeGeomColl (result_snap);
		if (result_split == NULL)
		    sqlite3_result_null (context);
		else
		  {
		      /* builds the BLOB geometry to be returned */
		      int len;
		      unsigned char *p_result = NULL;
		      result_split->Srid = geo1->Srid;
		      gaiaToSpatiaLiteBlobWkbEx2 (result_split, &p_result, &len,
						  gpkg_mode, tiny_point);
		      sqlite3_result_blob (context, p_result, len, free);
		      gaiaFreeGeomColl (result_split);
		  }
	    }
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Split (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Split(BLOBencoded input, BLOBencoded blade)
/
/ Returns a collection of geometries resulting by splitting a geometry
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr input = NULL;
    gaiaGeomCollPtr blade = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    input =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (input == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    blade =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (blade == NULL)
      {
	  gaiaFreeGeomColl (input);
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  result = gaiaSplit (cache, input, blade);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = input->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (input);
    gaiaFreeGeomColl (blade);
}

static void
fnct_SplitLeft (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SplitLeft(BLOBencoded input, BLOBencoded blade)
/
/ Returns a collection of geometries resulting by splitting a geometry [left half]
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr input = NULL;
    gaiaGeomCollPtr blade = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    input =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (input == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    blade =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (blade == NULL)
      {
	  gaiaFreeGeomColl (input);
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  result = gaiaSplitLeft (cache, input, blade);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = input->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (input);
    gaiaFreeGeomColl (blade);
}

static void
fnct_SplitRight (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ SplitRight(BLOBencoded input, BLOBencoded blade)
/
/ Returns a collection of geometries resulting by splitting a geometry [right half]
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr input = NULL;
    gaiaGeomCollPtr blade = NULL;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    input =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (input == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    blade =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (blade == NULL)
      {
	  gaiaFreeGeomColl (input);
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  result = gaiaSplitRight (cache, input, blade);
	  if (result == NULL)
	      sqlite3_result_null (context);
	  else
	    {
		/* builds the BLOB geometry to be returned */
		int len;
		unsigned char *p_result = NULL;
		result->Srid = input->Srid;
		gaiaToSpatiaLiteBlobWkbEx2 (result, &p_result, &len,
					    gpkg_mode, tiny_point);
		sqlite3_result_blob (context, p_result, len, free);
		gaiaFreeGeomColl (result);
	    }
      }
    gaiaFreeGeomColl (input);
    gaiaFreeGeomColl (blade);
}

static int
getXYSinglePoint (gaiaGeomCollPtr geom, double *x, double *y)
{
/* check if this geometry is a simple Point (returning 2D coords) */
    double z;
    double m;
    return getXYZMSinglePoint (geom, x, y, &z, &m);
}

static void
fnct_Azimuth (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Azimuth(BLOBencoded pointA, BLOBencoded pointB)
/
/ Returns the angle in radians from the horizontal of the vector 
/ defined by pointA and pointB. 
/ Angle is computed clockwise from down-to-up: on the clock: 
/ 12=0; 3=PI/2; 6=PI; 9=3PI/2.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    double x1;
    double y1;
    double x2;
    double y2;
    double a;
    double b;
    double rf;
    double azimuth;
    int srid;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }

/* retrieving and validating the first point */
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!getXYSinglePoint (geom, &x1, &y1))
      {
	  gaiaFreeGeomColl (geom);
	  sqlite3_result_null (context);
	  return;
      }
    srid = geom->Srid;
    gaiaFreeGeomColl (geom);

/* retrieving and validating the second point */
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!getXYSinglePoint (geom, &x2, &y2))
      {
	  gaiaFreeGeomColl (geom);
	  sqlite3_result_null (context);
	  return;
      }
    gaiaFreeGeomColl (geom);

    if (getEllipsoidParams (sqlite, srid, &a, &b, &rf))
      {
	  if (gaiaEllipsoidAzimuth (cache, x1, y1, x2, y2, a, b, &azimuth))
	      sqlite3_result_double (context, azimuth);
	  else
	      sqlite3_result_null (context);
	  return;
      }

    if (gaiaAzimuth (cache, x1, y1, x2, y2, &azimuth))
	sqlite3_result_double (context, azimuth);
    else
	sqlite3_result_null (context);
}

static void
fnct_Project (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Project(BLOBencoded point, distance Double, bearing Double)
/
/ Returns a new Point projected from a start point given a
/ distance and a bearing. 
/ - Point is expected to be Long/Lat
/ - Distance is in meters
/ - Bearing is in radians; on the clock: 
/   12=0; 3=PI/2; 6=PI; 9=3PI/2.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    double x1;
    double y1;
    double x2;
    double y2;
    int ival;
    double distance;
    double azimuth;
    double a;
    double b;
    double rf;
    int srid;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	distance = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[1]);
	  distance = ival;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	azimuth = sqlite3_value_double (argv[2]);
    else if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
      {
	  ival = sqlite3_value_int (argv[2]);
	  azimuth = ival;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }

/* retrieving and validating the start point */
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (!getXYSinglePoint (geom, &x1, &y1))
      {
	  gaiaFreeGeomColl (geom);
	  sqlite3_result_null (context);
	  return;
      }
    srid = geom->Srid;
    gaiaFreeGeomColl (geom);
    if (!getEllipsoidParams (sqlite, srid, &a, &b, &rf))
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (distance == 0.0)
      {
	  /* returning the Start Point */
	  gaiaMakePointEx (tiny_point, x1, y1, srid, &p_blob, &n_bytes);
	  if (!p_blob)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_blob, n_bytes, free);
	  return;
      }

    if (gaiaProjectedPoint (cache, x1, y1, a, b, distance, azimuth, &x2, &y2))
      {
	  gaiaMakePointEx (tiny_point, x2, y2, srid, &p_blob, &n_bytes);
	  if (!p_blob)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_blob (context, p_blob, n_bytes, free);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_GeoHash (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GeoHash(BLOBencoded geom)
/ GeoHash(BLOBencoded geom, Integer precision)
/
/ Returns a GeoHash representation for input geometry
/ (expected to be in longitude/latitude coords)
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    int precision = 0;
    char *geo_hash;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      precision = sqlite3_value_int (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    geo_hash = gaiaGeoHash (cache, geom, precision);
    if (geo_hash != NULL)
      {
	  int len = strlen (geo_hash);
	  sqlite3_result_text (context, geo_hash, len, free);
      }
    else
	sqlite3_result_null (context);
    gaiaFreeGeomColl (geom);
}

static char *
get_srs_by_srid (sqlite3 * sqlite, int srid, int longsrs)
{
/* retrieves the short- or long- srs reference for the given srid */
    char sql[1024];
    int ret;
    const char *name;
    int i;
    char **results;
    int rows;
    int columns;
    int len;
    char *srs = NULL;

    if (longsrs)
	sprintf (sql,
		 "SELECT 'urn:ogc:def:crs:' || auth_name || '::' || auth_srid "
		 "FROM spatial_ref_sys WHERE srid = %d", srid);
    else
	sprintf (sql, "SELECT auth_name || ':' || auth_srid "
		 "FROM spatial_ref_sys WHERE srid = %d", srid);
    ret = sqlite3_get_table (sqlite, sql, &results, &rows, &columns, NULL);
    if (ret != SQLITE_OK)
	return NULL;
    if (rows < 1)
	;
    else
      {
	  for (i = 1; i <= rows; i++)
	    {
		name = results[(i * columns) + 0];
		len = strlen (name);
		srs = malloc (len + 1);
		strcpy (srs, name);
	    }
      }
    sqlite3_free_table (results);
    return srs;
}

static void
fnct_AsX3D (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ AsX3D(BLOBencoded geom)
/ AsX3D(BLOBencoded geom, Integer precision)
/ AsX3D(BLOBencoded geom, Integer precision, Integer options)
/ AsX3D(BLOBencoded geom, Integer precision, Integer options, Text refid)
/
/ Returns an X3D representation for input geometry
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    int precision = 15;
    int options = 0;
    const char *refid = "";
    char *srs = NULL;
    char *x3d;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	      precision = sqlite3_value_int (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	      options = sqlite3_value_int (argv[2]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
    if (argc == 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_TEXT)
	      refid = (const char *) sqlite3_value_text (argv[3]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geom =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (geom == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (geom->Srid > 0)
      {
	  int longshort = 0;
	  if (options & 1)
	      longshort = 1;
	  srs = get_srs_by_srid (sqlite, geom->Srid, longshort);
      }
    x3d = gaiaAsX3D (cache, geom, srs, precision, options, refid);
    if (x3d != NULL)
      {
	  int len = strlen (x3d);
	  sqlite3_result_text (context, x3d, len, free);
      }
    else
	sqlite3_result_null (context);
    gaiaFreeGeomColl (geom);
    if (srs)
	free (srs);
}

static void
fnct_3DDistance (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ 3DDistance(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns the 3D distance between GEOM-1 and GEOM-2
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double dist;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  ret = gaia3DDistance (cache, geo1, geo2, &dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, dist);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_MaxDistance (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MaxDistance(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns the max 2D distance between GEOM-1 and GEOM-2
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double dist;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  ret = gaiaMaxDistance (cache, geo1, geo2, &dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, dist);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_3DMaxDistance (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ 3DMaxDistance(BLOBencoded geom1, BLOBencoded geom2)
/
/ returns the max 3D distance between GEOM-1 and GEOM-2
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geo1 = NULL;
    gaiaGeomCollPtr geo2 = NULL;
    double dist;
    int ret;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    geo1 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    p_blob = (unsigned char *) sqlite3_value_blob (argv[1]);
    n_bytes = sqlite3_value_bytes (argv[1]);
    geo2 =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (!geo1 || !geo2)
	sqlite3_result_null (context);
    else
      {
	  ret = gaia3DMaxDistance (cache, geo1, geo2, &dist);
	  if (!ret)
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, dist);
      }
    gaiaFreeGeomColl (geo1);
    gaiaFreeGeomColl (geo2);
}

static void
fnct_Node (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_Node(BLOBencoded linestring(s))
/
/ Returns a new new (Multi)Linestring by re-noding the input linestring(s)
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr input;
    gaiaGeomCollPtr result;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }

/* retrieving the input geometry */
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    input =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (input == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }

    result = gaiaNodeLines (cache, input);
    if (result != NULL)
      {
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_blob, &n_bytes, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_blob, n_bytes, free);
	  gaiaFreeGeomColl (result);
      }
    else
	sqlite3_result_null (context);
    gaiaFreeGeomColl (input);
}

static int
check_all_linestrings (gaiaGeomCollPtr in)
{
/* check id this is a collection of Linestrings */
    int pts = 0;
    int lns = 0;
    int pgs = 0;
    gaiaPointPtr pt;
    gaiaLinestringPtr ln;
    gaiaPolygonPtr pg;

    if (in == NULL)
	return 0;
/* checking if we have any POINT */
    pt = in->FirstPoint;
    while (pt)
      {
	  pts++;
	  pt = pt->Next;
      }
    if (pts > 0)
	return 0;
/* checking if we have any POLYGON */
    pg = in->FirstPolygon;
    while (pg)
      {
	  pgs++;
	  pg = pg->Next;
      }
    if (pgs > 0)
	return 0;
/* checking if we have any LINESTRING */
    ln = in->FirstLinestring;
    while (ln)
      {
	  lns++;
	  ln = ln->Next;
      }
    if (lns == 0)
	return 0;
    return 1;
}

static gaiaGeomCollPtr
get_nodes (gaiaGeomCollPtr in)
{
/* extracts all Nodes (Linestring extermities) */
    int iv;
    double x;
    double y;
    double m;
    double z;
    gaiaLinestringPtr ln;
    gaiaGeomCollPtr out;

    if (in == NULL)
	return NULL;

    if (in->DimensionModel == GAIA_XY_M)
	out = gaiaAllocGeomCollXYM ();
    else if (in->DimensionModel == GAIA_XY_Z)
	out = gaiaAllocGeomCollXYZ ();
    else if (in->DimensionModel == GAIA_XY_Z_M)
	out = gaiaAllocGeomCollXYZM ();
    else
	out = gaiaAllocGeomColl ();
    out->Srid = in->Srid;

    ln = in->FirstLinestring;
    while (ln)
      {
	  /* saving the extreme points - Start */
	  if (ln->DimensionModel == GAIA_XY_Z)
	    {
		gaiaGetPointXYZ (ln->Coords, 0, &x, &y, &z);
		gaiaAddPointToGeomCollXYZ (out, x, y, z);
	    }
	  else if (ln->DimensionModel == GAIA_XY_M)
	    {
		gaiaGetPointXYM (ln->Coords, 0, &x, &y, &m);
		gaiaAddPointToGeomCollXYM (out, x, y, m);
	    }
	  else if (ln->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaGetPointXYZM (ln->Coords, 0, &x, &y, &z, &m);
		gaiaAddPointToGeomCollXYZM (out, x, y, z, m);
	    }
	  else
	    {
		gaiaGetPoint (ln->Coords, 0, &x, &y);
		gaiaAddPointToGeomColl (out, x, y);
	    }
	  /* saving the extreme points - End */
	  iv = ln->Points - 1;
	  if (ln->DimensionModel == GAIA_XY_Z)
	    {
		gaiaGetPointXYZ (ln->Coords, iv, &x, &y, &z);
		gaiaAddPointToGeomCollXYZ (out, x, y, z);
	    }
	  else if (ln->DimensionModel == GAIA_XY_M)
	    {
		gaiaGetPointXYM (ln->Coords, iv, &x, &y, &m);
		gaiaAddPointToGeomCollXYM (out, x, y, m);
	    }
	  else if (ln->DimensionModel == GAIA_XY_Z_M)
	    {
		gaiaGetPointXYZM (ln->Coords, iv, &x, &y, &z, &m);
		gaiaAddPointToGeomCollXYZM (out, x, y, z, m);
	    }
	  else
	    {
		gaiaGetPoint (ln->Coords, iv, &x, &y);
		gaiaAddPointToGeomColl (out, x, y);
	    }
	  ln = ln->Next;
      }

    return out;
}

static int
point_is_defined (gaiaPointPtr in, gaiaGeomCollPtr geom)
{
/* checking if a Point is already defined */
    gaiaPointPtr pt = geom->FirstPoint;
    while (pt)
      {
	  if (geom->DimensionModel == GAIA_XY_Z)
	    {
		if (pt->X == in->X && pt->Y == in->Y && pt->Z == in->Z)
		    return 1;
	    }
	  else if (geom->DimensionModel == GAIA_XY_M)
	    {
		if (pt->X == in->X && pt->Y == in->Y && pt->M == in->M)
		    return 1;
	    }
	  else if (geom->DimensionModel == GAIA_XY_Z_M)
	    {
		if (pt->X == in->X && pt->Y == in->Y && pt->Z == in->Z
		    && pt->M == in->M)
		    return 1;
	    }
	  else
	    {
		if (pt->X == in->X && pt->Y == in->Y)
		    return 1;
	    }
	  pt = pt->Next;
      }
    return 0;
}

static gaiaGeomCollPtr
get_self_intersections (gaiaGeomCollPtr in_old, gaiaGeomCollPtr in_new)
{
/* extracting the self-intersection points */
    gaiaPointPtr pt;
    gaiaGeomCollPtr out;

    if (in_old->DimensionModel == GAIA_XY_M)
	out = gaiaAllocGeomCollXYM ();
    else if (in_old->DimensionModel == GAIA_XY_Z)
	out = gaiaAllocGeomCollXYZ ();
    else if (in_old->DimensionModel == GAIA_XY_Z_M)
	out = gaiaAllocGeomCollXYZM ();
    else
	out = gaiaAllocGeomColl ();
    out->Srid = in_old->Srid;

    pt = in_new->FirstPoint;
    while (pt)
      {
	  int ok1 = point_is_defined (pt, in_old);
	  int ok2 = point_is_defined (pt, out);
	  if (!ok1 && !ok2)
	    {
		/* inserting a Point into the result collection */
		if (out->DimensionModel == GAIA_XY_Z)
		    gaiaAddPointToGeomCollXYZ (out, pt->X, pt->Y, pt->Z);
		else if (out->DimensionModel == GAIA_XY_M)
		    gaiaAddPointToGeomCollXYM (out, pt->X, pt->Y, pt->M);
		else if (out->DimensionModel == GAIA_XY_Z_M)
		    gaiaAddPointToGeomCollXYZM (out, pt->X, pt->Y, pt->Z,
						pt->M);
		else
		    gaiaAddPointToGeomColl (out, pt->X, pt->Y);
	    }
	  pt = pt->Next;
      }

    if (out->FirstPoint == NULL)
      {
	  /* no self-intersections were found */
	  gaiaFreeGeomColl (out);
	  return NULL;
      }

    return out;
}

static void
fnct_SelfIntersections (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ ST_SelfIntersections(BLOBencoded linestring(s))
/
/ Returns a MultiPoint Geometry representing any self-intersection
/ found within the input geometry [linestring(s)]
/ NULL is returned for invalid arguments, or when no self-intersections
/ were found
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr input;
    gaiaGeomCollPtr noded;
    gaiaGeomCollPtr result;
    gaiaGeomCollPtr nodes_in;
    gaiaGeomCollPtr nodes_out;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }

/* retrieving the input geometry */
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    input =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (input == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }

/* checking the input (Linestrings only) */
    if (!check_all_linestrings (input))
      {
	  gaiaFreeGeomColl (input);
	  sqlite3_result_null (context);
	  return;
      }
/* extracting all input nodes */
    nodes_in = get_nodes (input);

    noded = gaiaNodeLines (cache, input);
    gaiaFreeGeomColl (input);
/* extracting all output nodes */
    nodes_out = get_nodes (noded);
    gaiaFreeGeomColl (noded);

/* identifying the intersections */
    result = get_self_intersections (nodes_in, nodes_out);
    gaiaFreeGeomColl (nodes_in);
    gaiaFreeGeomColl (nodes_out);
    if (result != NULL)
      {
	  result->DeclaredType = GAIA_MULTIPOINT;
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_blob, &n_bytes, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_blob, n_bytes, free);
	  gaiaFreeGeomColl (result);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_Subdivide (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_Subdivide(BLOBencoded geom)
/    or
/ ST_Subdivide(BLOBencoded geom, int max_vertices)
/
/ Divides geometry into parts until a part can be represented using no
/ more than max_vertices.
/ NULL is returned for invalid arguments
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr input;
    gaiaGeomCollPtr result;
    int max_vertices = 128;
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  max_vertices = sqlite3_value_int (argv[1]);
      }

/* retrieving the input geometry */
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    input =
	gaiaFromSpatiaLiteBlobWkbEx (p_blob, n_bytes, gpkg_mode,
				     gpkg_amphibious);
    if (input == NULL)
      {
	  sqlite3_result_null (context);
	  goto end;
      }

/* subdiving the input geometry */
    result = gaiaSubdivide (cache, input, max_vertices);
    if (result != NULL)
      {
	  gaiaToSpatiaLiteBlobWkbEx2 (result, &p_blob, &n_bytes, gpkg_mode,
				      tiny_point);
	  sqlite3_result_blob (context, p_blob, n_bytes, free);
	  gaiaFreeGeomColl (result);
      }
    else
	sqlite3_result_null (context);
  end:
    gaiaFreeGeomColl (input);
}

#endif /* end RTTOPO support */


static void
fnct_Cutter (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_Cutter(TEXT in_db_prefix, TEXT input_table, TEXT input_geom,
/              TEXT blade_db_prefix, TEXT blade_table, TEXT blade_geom,
/              TEXT output_table)
/ ST_Cutter(TEXT in_db_prefix, TEXT input_table, TEXT input_geom,
/              TEXT blade_db_prefix, TEXT blade_table, TEXT blade_geom,
/              TEXT output_table, INT transaction)
/ ST_Cutter(TEXT in_db_prefix, TEXT input_table, TEXT input_geom,
/              TEXT blade_db_prefix, TEXT blade_table, TEXT blade_geom,
/              TEXT output_table, INT transaction, INT ram_temp_store)
/
/ the "input" table-geometry is expected to be declared as POINT,
/ LINESTRING, POLYGON, MULTIPOINT, MULTILINESTRING or MULTIPOLYGON
/ and can be of any 2D or 3D dimension
/
/ the "blade" table-geometry is expected to be declared as POLYGON
/ or MULTIPOLYGON, and will always be casted to a pure(X,Y) dimension
/
/ the "output" table *must* not exists, and will be automatically
/ created within the MAIN database.
/ 
/ in_db_prefix and/or blade_db_prefix can eventually be NULL, and
/ in this case the MAIN db will be assumed
/
/ input_geom and/or blade_geom can eventually be NULL, and in this
/ case the geometry column name will be automatically determined.
/ anyway when a table defines two or more Geometries declaring a
/ NULL geometry name will cause a failure.
/
///////////////////////////////////////////////////////////////////
/
/ will precisely cut the input dataset against polygonal blade(s)
/ and will consequently create and populate an output dataset
/
/
/ returns 1 on success
/ 0 on failure, -1 on invalid arguments
*/
    sqlite3 *sqlite;
    int ret = 0;
    const char *in_db_prefix = NULL;
    const char *input_table = NULL;
    const char *input_geom = NULL;
    const char *blade_db_prefix = NULL;
    const char *blade_table = NULL;
    const char *blade_geom = NULL;
    const char *output_table = NULL;
    int transaction = 0;
    int ram_tmp_store = 0;
    char **message = NULL;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	message = &(cache->cutterMessage);

    if (sqlite3_value_type (argv[0]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	in_db_prefix = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	input_table = (const char *) sqlite3_value_text (argv[1]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[2]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[2]) == SQLITE_TEXT)
	input_geom = (const char *) sqlite3_value_text (argv[2]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[3]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[3]) == SQLITE_TEXT)
	blade_db_prefix = (const char *) sqlite3_value_text (argv[3]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[4]) == SQLITE_TEXT)
	blade_table = (const char *) sqlite3_value_text (argv[4]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[5]) == SQLITE_NULL)
	;
    else if (sqlite3_value_type (argv[5]) == SQLITE_TEXT)
	blade_geom = (const char *) sqlite3_value_text (argv[5]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (sqlite3_value_type (argv[6]) == SQLITE_TEXT)
	output_table = (const char *) sqlite3_value_text (argv[6]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    if (argc >= 8)
      {
	  if (sqlite3_value_type (argv[7]) == SQLITE_INTEGER)
	      transaction = sqlite3_value_int (argv[7]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }
    if (argc == 9)
      {
	  if (sqlite3_value_type (argv[8]) == SQLITE_INTEGER)
	      ram_tmp_store = sqlite3_value_int (argv[8]);
	  else
	    {
		sqlite3_result_int (context, -1);
		return;
	    }
      }

    sqlite = sqlite3_context_db_handle (context);
    ret =
	gaiaCutter (sqlite, cache, in_db_prefix, input_table, input_geom,
		    blade_db_prefix, blade_table, blade_geom, output_table,
		    transaction, ram_tmp_store, message);

    sqlite3_result_int (context, ret);
}

static void
fnct_GetCutterMessage (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ GetCutterMessage( void )
/
/ will return the last diagnostic message from Cutter
/ NULL if there is no pending message
*/
    char *message = NULL;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
	message = cache->cutterMessage;

    if (message == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, message, strlen (message), SQLITE_STATIC);
}

static void
fnct_DrapeLine (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ST_DrapeLine( line-1 Linestring, line-2 Linestring )
/    or
/ ST_DrapeLine( line-1 Linestring, line-2 Linestring, tolerance Double )
/
/ will drape line-1 over line-2 returning a new linestring
/ NULL on invalid arguments or if any error is encountered
*/
    unsigned char *blob;
    int bytes;
    gaiaGeomCollPtr geom1 = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    gaiaGeomCollPtr geom3 = NULL;
    double tolerance = 0.0;
    sqlite3 *db_handle = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }

    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  blob = (unsigned char *) sqlite3_value_blob (argv[0]);
	  bytes = sqlite3_value_bytes (argv[0]);
	  geom1 =
	      gaiaFromSpatiaLiteBlobWkbEx (blob, bytes, gpkg_mode,
					   gpkg_amphibious);
      }
    else
	goto error;
    if (sqlite3_value_type (argv[1]) == SQLITE_BLOB)
      {
	  blob = (unsigned char *) sqlite3_value_blob (argv[1]);
	  bytes = sqlite3_value_bytes (argv[1]);
	  geom2 =
	      gaiaFromSpatiaLiteBlobWkbEx (blob, bytes, gpkg_mode,
					   gpkg_amphibious);
      }
    else
	goto error;
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[2]);
		tolerance = val;
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      tolerance = sqlite3_value_double (argv[2]);
	  else
	      goto error;
      }

/* checking the arguments for validity */
    if (geom1 == NULL || geom2 == NULL)
	goto error;
    if (geom1->Srid != geom2->Srid)
	goto error;
    if (geom1->DimensionModel == GAIA_XY || geom1->DimensionModel == GAIA_XY_M)
	;
    else
	goto error;
    if (geom2->DimensionModel == GAIA_XY_Z
	|| geom2->DimensionModel == GAIA_XY_Z_M)
	;
    else
	goto error;
    if (!gaia_do_check_linestring (geom1))
	goto error;
    if (!gaia_do_check_linestring (geom2))
	goto error;
    if (tolerance < 0.0)
	goto error;

    geom3 = gaiaDrapeLine (db_handle, geom1, geom2, tolerance);
    if (geom3 == NULL)
	goto error;

    gaiaToSpatiaLiteBlobWkb (geom3, &blob, &bytes);
    sqlite3_result_blob (context, blob, bytes, free);
    gaiaFreeGeomColl (geom1);
    gaiaFreeGeomColl (geom2);
    gaiaFreeGeomColl (geom3);
    return;

  error:
    if (geom1 != NULL)
	gaiaFreeGeomColl (geom1);
    if (geom2 != NULL)
	gaiaFreeGeomColl (geom2);
    if (geom3 != NULL)
	gaiaFreeGeomColl (geom3);
    sqlite3_result_null (context);
}

static void
fnct_DrapeLineExceptions (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ ST_DrapeLineExceptions( line-1 Linestring, line-2 Linestring )
/    or
/ ST_DrapeLineExceptions( line-1 Linestring, line-2 Linestring, 
/                         tolerance  Double )
/    or
/ ST_DrapeLineExceptions( line-1 Linestring, line-2 Linestring, 
/                         tolerance  Double, interpolated Integer )
/
/ will drape line-1 over line-2 returning a MultiPoint containing
/ all Points from line-1 not correctly draped.
/ NULL on invalid arguments or if any error is encountered or
/ if all Point are correctly draped.
*/
    unsigned char *blob;
    int bytes;
    gaiaGeomCollPtr geom1 = NULL;
    gaiaGeomCollPtr geom2 = NULL;
    gaiaGeomCollPtr geom3 = NULL;
    double tolerance = 0.0;
    int interpolated = 1;
    sqlite3 *db_handle = sqlite3_context_db_handle (context);
    int gpkg_amphibious = 0;
    int gpkg_mode = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_amphibious = cache->gpkg_amphibious_mode;
	  gpkg_mode = cache->gpkg_mode;
      }

    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  blob = (unsigned char *) sqlite3_value_blob (argv[0]);
	  bytes = sqlite3_value_bytes (argv[0]);
	  geom1 =
	      gaiaFromSpatiaLiteBlobWkbEx (blob, bytes, gpkg_mode,
					   gpkg_amphibious);
      }
    else
	goto error;
    if (sqlite3_value_type (argv[1]) == SQLITE_BLOB)
      {
	  blob = (unsigned char *) sqlite3_value_blob (argv[1]);
	  bytes = sqlite3_value_bytes (argv[1]);
	  geom2 =
	      gaiaFromSpatiaLiteBlobWkbEx (blob, bytes, gpkg_mode,
					   gpkg_amphibious);
      }
    else
	goto error;
    if (argc >= 3)
      {
	  if (sqlite3_value_type (argv[2]) == SQLITE_INTEGER)
	    {
		int val = sqlite3_value_int (argv[2]);
		tolerance = val;
	    }
	  else if (sqlite3_value_type (argv[2]) == SQLITE_FLOAT)
	      tolerance = sqlite3_value_double (argv[2]);
	  else
	      goto error;
      }
    if (argc >= 4)
      {
	  if (sqlite3_value_type (argv[3]) == SQLITE_INTEGER)
	      interpolated = sqlite3_value_int (argv[3]);
	  else
	      goto error;
      }

/* checking the arguments for validity */
    if (geom1 == NULL || geom2 == NULL)
	goto error;
    if (geom1->Srid != geom2->Srid)
	goto error;
    if (geom1->DimensionModel == GAIA_XY || geom1->DimensionModel == GAIA_XY_M)
	;
    else
	goto error;
    if (geom2->DimensionModel == GAIA_XY_Z
	|| geom2->DimensionModel == GAIA_XY_Z_M)
	;
    else
	goto error;
    if (!gaia_do_check_linestring (geom1))
	goto error;
    if (!gaia_do_check_linestring (geom2))
	goto error;
    if (tolerance < 0.0)
	goto error;

    geom3 =
	gaiaDrapeLineExceptions (db_handle, geom1, geom2, tolerance,
				 interpolated);
    if (geom3 == NULL)
	goto error;

    gaiaToSpatiaLiteBlobWkb (geom3, &blob, &bytes);
    sqlite3_result_blob (context, blob, bytes, free);
    gaiaFreeGeomColl (geom1);
    gaiaFreeGeomColl (geom2);
    gaiaFreeGeomColl (geom3);
    return;

  error:
    if (geom1 != NULL)
	gaiaFreeGeomColl (geom1);
    if (geom2 != NULL)
	gaiaFreeGeomColl (geom2);
    if (geom3 != NULL)
	gaiaFreeGeomColl (geom3);
    sqlite3_result_null (context);
}

#endif /* end including GEOS */

static int
text2double (const unsigned char *str, double *val)
{
/* checks for a valid number, eventually returning a DOUBLE */
    int err = 0;
    int sign = 0;
    int decimal = 0;
    int exp = 0;
    int expsign = 0;
    const unsigned char *p = str;
    while (*p != '\0')
      {
	  switch (*p)
	    {
	    case '0':
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
		break;
	    case '-':
	    case '+':
		if (!exp)
		    sign++;
		else
		    expsign++;
		break;
	    case '.':
		decimal++;
		break;
	    case 'e':
	    case 'E':
		exp++;
		break;
	    default:
		err = 1;
		break;
	    };
	  p++;
      }
    if (sign > 1 || expsign > 1 || decimal > 1 || (exp == 0 && expsign > 0))
	err = 1;
    if (err)
	return 0;
    *val = atof ((const char *) str);
    return 1;
}

static int
is_integer (const unsigned char *value)
{
/* checking if a text string is an Integer Number */
    const unsigned char *p = value;
    while (*p != '\0')
      {
	  /* skipping any leading whitespace */
	  if (*p == ' ')
	    {
		p++;
		continue;
	    }
	  break;
      }
    if (*p == '\0')
	return 0;
    if (*p == '-' || *p == '+')
	p++;			/* skipping an eventual sign */
    if (*p == '\0')
	return 0;
    while (*p != '\0')
      {
	  if (*p >= '0' && *p <= '9')
	    {
		p++;
		continue;
	    }
	  return 0;
      }
    return 1;
}

static int
is_decimal_number (const unsigned char *value)
{
/* checking if a text string is a Decimal Number */
    const unsigned char *p = value;
    while (*p != '\0')
      {
	  /* skipping any leading whitespace */
	  if (*p == ' ')
	    {
		p++;
		continue;
	    }
	  break;
      }
    if (*p == '\0')
	return 0;
    if (*p == '-' || *p == '+')
	p++;			/* skipping an eventual sign */
    if (*p == '\0')
	return 0;
    while (*p != '\0')
      {
	  /* integer part */
	  if (*p == '.')
	    {
		p++;
		break;
	    }
	  if (*p >= '0' && *p <= '9')
	    {
		p++;
		continue;
	    }
	  return 0;
      }
    if (*p == '\0')
	return 0;
    while (*p != '\0')
      {
	  /* fractional part */
	  if (*p == 'e' || *p == 'E')
	      break;
	  if (*p >= '0' && *p <= '9')
	    {
		p++;
		continue;
	    }
	  return 0;
      }
    if (*p == '\0')
	return 1;		/* valid decimal number without exponent */

/* checking the exponent */
    if (*p == 'e' || *p == 'E')
	p++;			/* skipping an eventual exponent marker */
    else
	return 0;
    if (*p == '\0')
	return 0;
    if (*p == '-' || *p == '+')
	p++;			/* skipping an eventual exponent sign */
    if (*p == '\0')
	return 0;
    while (*p != '\0')
      {
	  /* exponent */
	  if (*p >= '0' && *p <= '9')
	    {
		p++;
		continue;
	    }
	  return 0;
      }
    return 1;
}

static void
fnct_IsInteger (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsInteger(TEXT value)
/
/ returns 1 if yes, 0 if  not and -1 on any non-text value
*/
    int ret;
    const unsigned char *value;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	value = sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    ret = is_integer (value);
    sqlite3_result_int (context, ret);
}

static void
fnct_IsDecimalNumber (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ IsDecimalNumber(TEXT value)
/
/ returns 1 if yes, 0 if  not and -1 on any non-text value
*/
    int ret;
    const unsigned char *value;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	value = sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    ret = is_decimal_number (value);
    sqlite3_result_int (context, ret);
}

static void
fnct_IsNumber (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ IsNumber(TEXT value)
/
/ returns 1 if yes, 0 if  not and -1 on any non-text value
*/
    int ret;
    const unsigned char *value;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	value = sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    ret = is_integer (value);
    if (ret)
      {
	  sqlite3_result_int (context, 1);
	  return;
      }
    ret = is_decimal_number (value);
    sqlite3_result_int (context, ret);
}

static void
fnct_CastToInteger (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToInteger(generic value)
/
/ returns an INTEGER value [if conversion is possible] 
/ or NULL in any other case
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  sqlite3_int64 val = sqlite3_value_int64 (argv[0]);
	  sqlite3_result_int64 (context, val);
	  return;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  sqlite3_int64 val;
	  double dval = sqlite3_value_double (argv[0]);
	  double diff = dval - floor (dval);
	  val = (sqlite3_int64) sqlite3_value_double (argv[0]);
	  if (diff >= 0.5)
	      val++;
	  sqlite3_result_int64 (context, val);
	  return;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
      {
	  const unsigned char *txt = sqlite3_value_text (argv[0]);
	  double dval;
	  if (text2double (txt, &dval))
	    {
		sqlite3_int64 val;
		double dval = sqlite3_value_double (argv[0]);
		double diff = dval - floor (dval);
		val = (sqlite3_int64) sqlite3_value_double (argv[0]);
		if (diff >= 0.5)
		    val++;
		sqlite3_result_int64 (context, val);
		return;
	    }
      }
    sqlite3_result_null (context);
}

static void
fnct_CastToDouble (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToDouble(generic value)
/
/ returns a DOUBLE value [if conversion is possible] 
/ or NULL in any other case
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  double val = (double) sqlite3_value_int64 (argv[0]);
	  sqlite3_result_double (context, val);
	  return;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  double val = sqlite3_value_double (argv[0]);
	  sqlite3_result_double (context, val);
	  return;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
      {
	  const unsigned char *txt = sqlite3_value_text (argv[0]);
	  double val;
	  if (text2double (txt, &val))
	    {
		sqlite3_result_double (context, val);
		return;
	    }
      }
    sqlite3_result_null (context);
}

static void
fnct_CastToText (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToText(generic value)
/ CastToText(generic value, Integer left-aligned-length)
/
/ returns a TEXT value [if conversion is possible] 
/ or NULL in any other case
*/
    char *txt;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  char format[32];
	  const char *fmt = "%lld";
	  sqlite3_int64 val;
	  if (argc == 2)
	    {
		int length;
		if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
		  {
		      sqlite3_result_null (context);
		      return;
		  }
		length = sqlite3_value_int (argv[1]);
		if (length > 0)
		  {
		      sprintf (format, "%%0%dlld", length);
		      fmt = format;
		  }
	    }
	  val = sqlite3_value_int64 (argv[0]);
	  txt = sqlite3_mprintf (fmt, val);
	  sqlite3_result_text (context, txt, strlen (txt), sqlite3_free);
	  return;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  int i;
	  int len;
	  double val = sqlite3_value_double (argv[0]);
	  char format[32];
	  const char *fmt = "%1.18f";
	  if (argc == 2)
	    {
		int length;
		if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
		  {
		      sqlite3_result_null (context);
		      return;
		  }
		length = sqlite3_value_int (argv[1]);
		if (length > 0)
		  {
		      sprintf (format, "%%0%d.18f", length + 19);
		      fmt = format;
		  }
	    }
	  txt = sqlite3_mprintf (fmt, val);
	  len = strlen (txt);
	  for (i = len - 1; i > 0; i--)
	    {
		/* suppressing meaningless trailing zeroes */
		if (txt[i] >= '1' && txt[i] <= '9')
		    break;
		if (txt[i] == '.')
		  {
		      txt[i + 1] = '0';
		      break;
		  }
		if (txt[i] == '0')
		    txt[i] = '\0';
	    }
	  sqlite3_result_text (context, txt, strlen (txt), sqlite3_free);
	  return;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
      {
	  int n_bytes;
	  txt = (char *) sqlite3_value_text (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
	  sqlite3_result_text (context, txt, n_bytes, SQLITE_TRANSIENT);
	  return;
      }
    sqlite3_result_null (context);
}

static int
parseHexByte (unsigned char hi, unsigned char lo, unsigned char *val)
{
/* converting a byte for its Hex representation */
    unsigned char x;
    switch (hi)
      {
      case '0':
	  x = 0;
	  break;
      case '1':
	  x = 16;
	  break;
      case '2':
	  x = 16 * 2;
	  break;
      case '3':
	  x = 16 * 3;
	  break;
      case '4':
	  x = 16 * 4;
	  break;
      case '5':
	  x = 16 * 5;
	  break;
      case '6':
	  x = 16 * 6;
	  break;
      case '7':
	  x = 16 * 7;
	  break;
      case '8':
	  x = 16 * 8;
	  break;
      case '9':
	  x = 16 * 9;
	  break;
      case 'a':
      case 'A':
	  x = 16 * 10;
	  break;
      case 'b':
      case 'B':
	  x = 16 * 11;
	  break;
      case 'c':
      case 'C':
	  x = 16 * 12;
	  break;
      case 'd':
      case 'D':
	  x = 16 * 13;
	  break;
      case 'e':
      case 'E':
	  x = 16 * 14;
	  break;
      case 'f':
      case 'F':
	  x = 16 * 15;
	  break;
      default:
	  return 0;
      };
    switch (lo)
      {
      case '0':
	  x += 0;
	  break;
      case '1':
	  x += 1;
	  break;
      case '2':
	  x += 2;
	  break;
      case '3':
	  x += 3;
	  break;
      case '4':
	  x += 4;
	  break;
      case '5':
	  x += 5;
	  break;
      case '6':
	  x += 6;
	  break;
      case '7':
	  x += 7;
	  break;
      case '8':
	  x += 8;
	  break;
      case '9':
	  x += 9;
	  break;
      case 'a':
      case 'A':
	  x += 10;
	  break;
      case 'b':
      case 'B':
	  x += 11;
	  break;
      case 'c':
      case 'C':
	  x += 12;
	  break;
      case 'd':
      case 'D':
	  x += 13;
	  break;
      case 'e':
      case 'E':
	  x += 14;
	  break;
      case 'f':
      case 'F':
	  x += 15;
	  break;
      default:
	  return 0;
      };
    *val = x;
    return 1;
}

static int
parseHexString (const unsigned char *in, int in_len, unsigned char **out,
		int *out_len)
{
/* parsing an Hexadecimal string */
    unsigned char *buf;
    unsigned char *p_out;
    unsigned char byteval;
    int i;
    int len;
    *out = NULL;
    *out_len = 0;
    if (in == NULL)
	return 0;
    len = in_len / 2;
    if (len * 2 != in_len)	/* # digits is an odd number */
	return 0;
    buf = malloc (len);
    p_out = buf;
    for (i = 0; i < in_len; i += 2)
      {
	  if (!parseHexByte (in[i], in[i + 1], &byteval))
	      goto error;
	  *p_out++ = byteval;
      }
    *out = buf;
    *out_len = len;
    return 1;
  error:
    free (buf);
    return 0;
}

static void
fnct_CastToBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CastToBlob(generic value)
/   or
/ CastToBlob(generic value, boolen hex_input)
/
/ returns a BLOB value [if conversion is possible] 
/ or NULL in any other case
*/
    const unsigned char *p_blob;
    int n_bytes;
    int is_hex = 0;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (argc == 2)
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  is_hex = sqlite3_value_int (argv[1]);
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  p_blob = sqlite3_value_blob (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
	  if (is_hex)
	    {
		/* attempting to convert Hexadecimal input */
		unsigned char *blob;
		int bytes;
		if (!parseHexString (p_blob, n_bytes, &blob, &bytes))
		  {
		      sqlite3_result_null (context);
		      return;
		  }
		sqlite3_result_blob (context, blob, bytes, free);
		return;
	    }
	  sqlite3_result_blob (context, p_blob, n_bytes, SQLITE_TRANSIENT);
	  return;
      }
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
      {
	  p_blob = sqlite3_value_text (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
	  if (is_hex)
	    {
		/* attempting to convert Hexadecimal input */
		unsigned char *blob;
		int bytes;
		if (!parseHexString (p_blob, n_bytes, &blob, &bytes))
		  {
		      sqlite3_result_null (context);
		      return;
		  }
		sqlite3_result_blob (context, blob, bytes, free);
		return;
	    }
	  sqlite3_result_blob (context, p_blob, n_bytes, SQLITE_TRANSIENT);
	  return;
      }
    sqlite3_result_null (context);
}

static void
fnct_ForceAsNull (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ForceAsNull(generic val1, generic val2)
/
/ returns a NULL value if val1 and val2 are equal 
/ (and exactly of the same type)
/ return val1 in any other case
*/
    int type1;
    int type2;
    const unsigned char *p_blob;
    int n_bytes;
    const unsigned char *p_blob2;
    int n_bytes2;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    type1 = sqlite3_value_type (argv[0]);
    type2 = sqlite3_value_type (argv[1]);
    if (type1 == type2)
      {
	  switch (type1)
	    {
	    case SQLITE_INTEGER:
		if (sqlite3_value_int64 (argv[0]) ==
		    sqlite3_value_int64 (argv[1]))
		  {
		      sqlite3_result_null (context);
		      return;
		  }
		break;
	    case SQLITE_FLOAT:
		if (sqlite3_value_double (argv[0]) ==
		    sqlite3_value_double (argv[1]))
		  {
		      sqlite3_result_null (context);
		      return;
		  }
		break;
	    case SQLITE_TEXT:
		p_blob = sqlite3_value_text (argv[0]);
		n_bytes = sqlite3_value_bytes (argv[0]);
		p_blob2 = sqlite3_value_text (argv[1]);
		n_bytes2 = sqlite3_value_bytes (argv[1]);
		if (n_bytes == n_bytes2)
		  {
		      if (strcasecmp
			  ((const char *) p_blob, (const char *) p_blob2) == 0)
			{
			    sqlite3_result_null (context);
			    return;
			}
		  }
		break;
	    case SQLITE_BLOB:
		p_blob = sqlite3_value_blob (argv[0]);
		n_bytes = sqlite3_value_bytes (argv[0]);
		p_blob2 = sqlite3_value_blob (argv[1]);
		n_bytes2 = sqlite3_value_bytes (argv[1]);
		if (n_bytes == n_bytes2)
		  {
		      if (memcmp (p_blob, p_blob2, n_bytes) == 0)
			{
			    sqlite3_result_null (context);
			    return;
			}
		  }
		break;
	    case SQLITE_NULL:
		sqlite3_result_null (context);
		return;
	    };
      }
/* returning the first argument */
    switch (type1)
      {
      case SQLITE_INTEGER:
	  sqlite3_result_int64 (context, sqlite3_value_int64 (argv[0]));
	  break;
      case SQLITE_FLOAT:
	  sqlite3_result_double (context, sqlite3_value_double (argv[0]));
	  break;
      case SQLITE_TEXT:
	  p_blob = sqlite3_value_text (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
	  sqlite3_result_text (context, (const char *) p_blob, n_bytes,
			       SQLITE_TRANSIENT);
	  break;
      case SQLITE_BLOB:
	  p_blob = sqlite3_value_blob (argv[0]);
	  n_bytes = sqlite3_value_bytes (argv[0]);
	  sqlite3_result_blob (context, p_blob, n_bytes, SQLITE_TRANSIENT);
	  break;
      default:
	  sqlite3_result_null (context);
	  break;
      };
}

static void
fnct_CreateUUID (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ CreateUUID()
/
/ returns a TEXT value containing an UUID
/ [xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxxxxxx]
*/
    unsigned char rnd[16];
    char uuid[64];
    char *p = uuid;
    int i;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_randomness (16, rnd);
    for (i = 0; i < 16; i++)
      {
	  if (i == 4 || i == 6 || i == 8 || i == 10)
	      *p++ = '-';
	  sprintf (p, "%02x", rnd[i]);
	  p += 2;
      }
    *p = '\0';
    uuid[14] = '4';
    uuid[19] = '8';
    sqlite3_result_text (context, uuid, strlen (uuid), SQLITE_TRANSIENT);
}

static void
fnct_MD5Checksum (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ MD5Checksum(blob)
/
/ returns a TEXT value containing an hex MD5 checksum
/ [xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx]
/      or
/ NULL on invalid arguments
*/
    void *md5;
    char *checksum;
    const unsigned char *blob;
    int blob_len;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  blob = sqlite3_value_blob (argv[0]);
	  blob_len = sqlite3_value_bytes (argv[0]);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
      {
	  blob = sqlite3_value_text (argv[0]);
	  blob_len = sqlite3_value_bytes (argv[0]);
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* creating an MD5 object */
    md5 = gaiaCreateMD5Checksum ();
/* evaluating the BLOB */
    gaiaUpdateMD5Checksum (md5, blob, blob_len);
    checksum = gaiaFinalizeMD5Checksum (md5);
    gaiaFreeMD5Checksum (md5);
    if (checksum == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, checksum, strlen (checksum), free);
}

static void
fnct_MD5TotalChecksum_step (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ MD5TotalChecksum(BLOB)
/
/ aggregate function - STEP
/
*/
    void **p;
    void *md5;
    const unsigned char *blob;
    int blob_len;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_BLOB)
      {
	  blob = sqlite3_value_blob (argv[0]);
	  blob_len = sqlite3_value_bytes (argv[0]);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
      {
	  blob = sqlite3_value_text (argv[0]);
	  blob_len = sqlite3_value_bytes (argv[0]);
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p = sqlite3_aggregate_context (context, sizeof (void *));
    if (!(*p))
      {
	  /* this is the first row - creating an MD5 object */
	  md5 = gaiaCreateMD5Checksum ();
	  gaiaUpdateMD5Checksum (md5, blob, blob_len);
	  *p = md5;
      }
    else
      {
	  /* subsequent rows */
	  md5 = *p;
	  gaiaUpdateMD5Checksum (md5, blob, blob_len);
      }
}

static void
fnct_MD5TotalChecksum_final (sqlite3_context * context)
{
/* SQL function:
/ MD5TotalChecksum(BLOB)
/
/ aggregate function - FINAL
/
*/
    void **p;
    void *md5;
    char *checksum;
    p = sqlite3_aggregate_context (context, 0);
    if (!(*p))
      {
	  sqlite3_result_null (context);
	  return;
      }
    md5 = *p;
    checksum = gaiaFinalizeMD5Checksum (md5);
    gaiaFreeMD5Checksum (md5);
    if (checksum == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, checksum, strlen (checksum), free);
}

#if OMIT_ICONV == 0		/* ICONV is absolutely required */

static void
fnct_ExportGeoJSON (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ExportGeoJSON(TEXT table, TEXT geom_column, TEXT filename)
/ ExportGeoJSON(TEXT table, TEXT geom_column, TEXT filename, 
/               TEXT format)
/ ExportGeoJSON(TEXT table, TEXT geom_column, TEXT filename, 
/               TEXT format, INT precision)
/
/ returns:
/ the number of exported rows
/ NULL on invalid arguments
*/
    int ret;
    char *table;
    char *geom_col;
    char *path;
    int format = 0;
    int precision = 8;
    char *fmt = NULL;
    int rows;
    sqlite3 *db_handle = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    table = (char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    geom_col = (char *) sqlite3_value_text (argv[1]);
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    path = (char *) sqlite3_value_text (argv[2]);
    if (argc > 3)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  else
	    {
		fmt = (char *) sqlite3_value_text (argv[3]);
		if (strcasecmp (fmt, "none") == 0)
		    format = 0;
		else if (strcasecmp (fmt, "MBR") == 0)
		    format = 1;
		else if (strcasecmp (fmt, "withShortCRS") == 0)
		    format = 2;
		else if (strcasecmp (fmt, "MBRwithShortCRS") == 0)
		    format = 3;
		else if (strcasecmp (fmt, "withLongCRS") == 0)
		    format = 4;
		else if (strcasecmp (fmt, "MBRwithLongCRS") == 0)
		    format = 5;
		else
		  {
		      sqlite3_result_null (context);
		      return;
		  }
	    }
      }
    if (argc > 4)
      {
	  if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  else
	      precision = sqlite3_value_int (argv[4]);
      }

    ret =
	dump_geojson_ex (db_handle, table, geom_col, path, precision, format,
			 &rows);

    if (rows < 0 || !ret)
	sqlite3_result_null (context);
    else
	sqlite3_result_int (context, rows);
}

static void
fnct_ExportGeoJSON2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ExportGeoJSON2(TEXT table, TEXT geom_column, TEXT filename)
/ ExportGeoJSON2(TEXT table, TEXT geom_column, TEXT filename, 
/                INT precision)
/ ExportGeoJSON2(TEXT table, TEXT geom_column, TEXT filename, 
/                INT precision, INT lon_lat)
/ ExportGeoJSON2(TEXT table, TEXT geom_column, TEXT filename, 
/                INT precision, INT lon_lat, INT M_coords)
/ ExportGeoJSON2(TEXT table, TEXT geom_column, TEXT filename, 
/                INT precision, INT lon_lat, INT M_coords,
/                INT indented)
/ ExportGeoJSON2(TEXT table, TEXT geom_column, TEXT filename, 
/                INT precision, INT lon_lat, INT M_coords,
/                INT indented, TEXT colname_case)
/
/ returns:
/ the number of exported rows
/ NULL on invalid arguments
*/
    int ret;
    char *table;
    char *geom_col;
    char *path;
    int precision = 8;
    int lon_lat = 1;
    int m_coords = 0;
    int indented = 1;
    int colname_case = GAIA_DBF_COLNAME_LOWERCASE;
    int rows;
    char *errmsg = NULL;
    sqlite3 *db_handle = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    table = (char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) == SQLITE_NULL)
	geom_col = NULL;
    else
      {
	  if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  geom_col = (char *) sqlite3_value_text (argv[1]);
      }
    if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    path = (char *) sqlite3_value_text (argv[2]);
    if (argc > 3)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  precision = sqlite3_value_int (argv[3]);
      }
    if (argc > 4)
      {
	  if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  lon_lat = sqlite3_value_int (argv[4]);
      }
    if (argc > 5)
      {
	  if (sqlite3_value_type (argv[5]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  m_coords = sqlite3_value_int (argv[5]);
      }
    if (argc > 6)
      {
	  if (sqlite3_value_type (argv[6]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  indented = sqlite3_value_int (argv[6]);
      }
    if (argc > 7)
      {
	  if (sqlite3_value_type (argv[7]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  else
	    {
		const char *val = (char *) sqlite3_value_text (argv[7]);
		if (strcasecmp (val, "UPPER") == 0
		    || strcasecmp (val, "UPPERCASE") == 0)
		    colname_case = GAIA_DBF_COLNAME_UPPERCASE;
		else if (strcasecmp (val, "SAME") == 0
			 || strcasecmp (val, "SAMECASE") == 0)
		    colname_case = GAIA_DBF_COLNAME_CASE_IGNORE;
		else
		    colname_case = GAIA_DBF_COLNAME_LOWERCASE;
	    }
      }

    ret =
	dump_geojson2 (db_handle, table, geom_col, path, precision, lon_lat,
		       m_coords, indented, colname_case, &rows, &errmsg);
    if (errmsg != NULL)
      {
	  spatialite_e ("%s", errmsg);
	  sqlite3_free (errmsg);
      }

    if (rows < 0 || !ret)
	sqlite3_result_null (context);
    else
	sqlite3_result_int (context, rows);
}

static void
fnct_ImportGeoJSON (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ImportGeoJSON(TEXT filename, TEXT table)
/ ImportGeoJSON(TEXT filename, TEXT table, TEXT geom_column)
/ ImportGeoJSON(TEXT filename, TEXT table, TEXT geom_column,
/               INT spatial_index)
/ ImportGeoJSON(TEXT filename, TEXT table, TEXT geom_column,
/               INT spatial_index, INT srid)
/ ImportGeoJSON(TEXT filename, TEXT table, TEXT geom_column,
/               INT spatial_index, INT srid, TEXT colname_case)
/
/ returns:
/ the number of imported rows
/ NULL on invalid arguments
*/
    int ret;
    char *table;
    char *geom_col = "geometry";
    char *path;
    int spatial_index = 0;
    int srid = 4326;
    int colname_case = GAIA_DBF_COLNAME_LOWERCASE;
    int rows;
    char *errmsg = NULL;
    sqlite3 *db_handle = sqlite3_context_db_handle (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    path = (char *) sqlite3_value_text (argv[0]);
    if (sqlite3_value_type (argv[1]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    table = (char *) sqlite3_value_text (argv[1]);
    if (argc > 2)
      {
	  if (sqlite3_value_type (argv[2]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  geom_col = (char *) sqlite3_value_text (argv[2]);
      }
    if (argc > 3)
      {
	  if (sqlite3_value_type (argv[3]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  spatial_index = sqlite3_value_int (argv[3]);
      }
    if (argc > 4)
      {
	  if (sqlite3_value_type (argv[4]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  srid = sqlite3_value_int (argv[4]);
      }
    if (argc > 5)
      {
	  if (sqlite3_value_type (argv[5]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  else
	    {
		const char *val = (char *) sqlite3_value_text (argv[5]);
		if (strcasecmp (val, "UPPER") == 0
		    || strcasecmp (val, "UPPERCASE") == 0)
		    colname_case = GAIA_DBF_COLNAME_UPPERCASE;
		else if (strcasecmp (val, "SAME") == 0
			 || strcasecmp (val, "SAMECASE") == 0)
		    colname_case = GAIA_DBF_COLNAME_CASE_IGNORE;
		else
		    colname_case = GAIA_DBF_COLNAME_LOWERCASE;
	    }
      }

    ret =
	load_geojson (db_handle, path, table, geom_col, spatial_index, srid,
		      colname_case, &rows, &errmsg);
    if (errmsg != NULL)
      {
	  spatialite_e ("%s", errmsg);
	  sqlite3_free (errmsg);
      }

    if (rows < 0 || !ret)
	sqlite3_result_null (context);
    else
	sqlite3_result_int (context, rows);
}

static void
fnct_EncodeURL (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ EncodeURL(text url)
/   or
/ EncodeURL(text url, text out_charset)
/
/ returns a TEXT value containing the percent-encoded URL
/      or
/ NULL on invalid arguments
*/
    const char *url;
    const char *out_charset = "UTF-8";
    char *encoded;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	url = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 1)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	      out_charset = (const char *) sqlite3_value_text (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
/* encoding the URL */
    encoded = gaiaEncodeURL (url, out_charset);
    if (encoded == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, encoded, strlen (encoded), free);
}

static void
fnct_DecodeURL (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ DecodeURL(text)
/   or
/ DecodeURL(text url, text in_charset)
/
/ returns a TEXT value containing the URL cleaned from percent-encoding
/      or
/ NULL on invalid arguments
*/
    char *url;
    const char *encoded;
    const char *in_charset = "UTF-8";
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	encoded = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (argc > 1)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	      in_charset = (const char *) sqlite3_value_text (argv[1]);
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
      }
/* decoding the URL */
    url = gaiaDecodeURL (encoded, in_charset);
    if (url == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, url, strlen (url), free);
}

#ifdef ENABLE_MINIZIP		/* only id MINIZIP is enabled */

static void
fnct_Zipfile_NumSHP (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Zipfile_NumSHP(zip_path TEXT)
/
/ return the total number of Shapefiles contained within a given Zipfile
/ 0 if not Shapefile exists
/ or NULL on error or invalid arguments
*/
    const char *zip_path;
    int count;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	zip_path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* searching for Shapefiles */
    if (!gaiaZipfileNumSHP (zip_path, &count))
      {
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_result_int (context, count);
}

static void
fnct_Zipfile_ShpN (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Zipfile_ShpN(zip_path TEXT, idx INTEGER)
/
/ return the name of the Nth Shapefile from within a Zipfile
/ or NULL on error or invalid arguments
*/
    const char *zip_path;
    int idx;
    char *basename;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	zip_path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	idx = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* searching the Nth Shapefile */
    basename = gaiaZipfileShpN (zip_path, idx);
    if (basename == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_result_text (context, basename, strlen (basename), free);
}

static void
fnct_Zipfile_NumDBF (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Zipfile_NumDBF(zip_path TEXT)
/
/ return the total number of DBF files contained within a given Zipfile
/ 0 if not DBF file exists
/ or NULL on error or invalid arguments
*/
    const char *zip_path;
    int count;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	zip_path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* searching for DBF files */
    if (!gaiaZipfileNumDBF (zip_path, &count))
      {
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_result_int (context, count);
}

static void
fnct_Zipfile_DbfN (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ Zipfile_DbfN(zip_path TEXT, idx INTEGER)
/
/ return the name of the Nth DBF file from within a Zipfile
/ or NULL on error or invalid arguments
*/
    const char *zip_path;
    int idx;
    char *filename;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	zip_path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
	idx = sqlite3_value_int (argv[1]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* searching the Nth DBF file */
    filename = gaiaZipfileDbfN (zip_path, idx);
    if (filename == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_result_text (context, filename, strlen (filename), free);
}

#endif /* end MINIZIP */

#endif /* ICONV enabled/disabled */

static void
fnct_DirNameFromPath (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ DirNameFromPath(text)
/
/ returns a TEXT value containing the Directory Name from a Path
/      or
/ NULL on invalid arguments
*/
    char *dir;
    const char *path;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* breaking the Path */
    dir = gaiaDirNameFromPath (path);
    if (dir == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, dir, strlen (dir), free);
}

static void
fnct_FullFileNameFromPath (sqlite3_context * context, int argc,
			   sqlite3_value ** argv)
{
/* SQL function:
/ FullFileNameFromPath(text)
/
/ returns a TEXT value containing the Full FileName (including extension)
/ from a Path
/      or
/ NULL on invalid arguments
*/
    char *name;
    const char *path;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* breaking the Path */
    name = gaiaFullFileNameFromPath (path);
    if (name == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, name, strlen (name), free);
}

static void
fnct_FileNameFromPath (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ FileNameFromPath(text)
/
/ returns a TEXT value containing the FileName (excluding extension)
/ from a Path
/      or
/ NULL on invalid arguments
*/
    char *name;
    const char *path;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* breaking the Path */
    name = gaiaFileNameFromPath (path);
    if (name == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, name, strlen (name), free);
}

static void
fnct_FileExtFromPath (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
/* SQL function:
/ FileExtFromPath(text)
/
/ returns a TEXT value containing the Extension (if any) from a Path
/      or
/ NULL on invalid arguments
*/
    char *ext;
    const char *path;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	path = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* breaking the Path */
    ext = gaiaFileExtFromPath (path);
    if (ext == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, ext, strlen (ext), free);
}

static void
fnct_RemoveExtraSpaces (sqlite3_context * context, int argc,
			sqlite3_value ** argv)
{
/* SQL function:
/ RemoveExtraSpaces(TEXT string)
/
/ returns a TEXT value containing no repeated whitespaces
/      or
/ NULL on invalid arguments
*/
    const char *dirty;
    char *clean;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */

    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	dirty = (const char *) sqlite3_value_text (argv[0]);
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
/* removing whitespaces from the string */
    clean = gaiaRemoveExtraSpaces (dirty);
    if (clean == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, clean, strlen (clean), free);
}

static void
fnct_make_string_list_step (sqlite3_context * context, int argc,
			    sqlite3_value ** argv)
{
/* SQL function:
/ MakeStringList(X)
/ MakeStringList(X, separator)
/
/ aggregate function - STEP
/
*/
    struct string_list_str *p;
    char buf[1024];
    const char *str = buf;
    char separator = ',';
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	str = (const char *) sqlite3_value_text (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
#if defined(_WIN32) && !defined(__MINGW32__)
	sprintf (buf, "%I64d", sqlite3_value_int64 (argv[0]));
#else
	sprintf (buf, "%lld", sqlite3_value_int64 (argv[0]));
#endif
    else
	strcpy (buf, "ILLEGAL_VALUE");
    if (argc >= 2)
      {
	  if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	    {
		const char *sep = (const char *) sqlite3_value_text (argv[1]);
		separator = *sep;
	    }
	  else
	      return;
      }
    p = sqlite3_aggregate_context (context, sizeof (struct string_list_str));
    if (p->separator == '\0' && p->string == NULL)
      {
	  /* first item */
	  p->separator = separator;
	  if (separator == '\0')
	      separator = ',';
	  p->string = sqlite3_mprintf ("%s", str);
      }
    else
      {
	  char *oldstr = p->string;
	  p->string = sqlite3_mprintf ("%s%c%s", oldstr, p->separator, str);
	  sqlite3_free (oldstr);
      }
}

static void
fnct_make_string_list_final (sqlite3_context * context)
{
/* SQL function:
/ MakeStringList(X)
/ MakeStringList(X, separator)
/ aggregate function -  FINAL
/
*/
    struct string_list_str *p = sqlite3_aggregate_context (context, 0);
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_result_text (context, p->string, -1, sqlite3_free);
}


#ifndef OMIT_MATHSQL		/* supporting SQL math functions */

static int
testInvalidFP (double x)
{
/* testing if this one is an invalid Floating Point */
#ifdef _WIN32
    if (_fpclass (x) == _FPCLASS_NN || _fpclass (x) == _FPCLASS_PN ||
	_fpclass (x) == _FPCLASS_NZ || _fpclass (x) == _FPCLASS_PZ)
	;
    else
	return 1;
#else
    if (fpclassify (x) == FP_NORMAL || fpclassify (x) == FP_ZERO)
	;
    else
	return 1;
#endif
    return 0;
}

static void
fnct_math_acos (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ acos(double X)
/
/ Returns the arc cosine of X, that is, the value whose cosine is X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = acos (sqlite3_value_double (argv[0]));
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = acos (x);
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_asin (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ asin(double X)
/
/ Returns the arc sine of X, that is, the value whose sine is X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = asin (sqlite3_value_double (argv[0]));
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = asin (x);
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_atan (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ atan(double X)
/
/ Returns the arc tangent of X, that is, the value whose tangent is X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = atan (sqlite3_value_double (argv[0]));
	  sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = atan (x);
	  sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_atan2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ atan2(double Y, double X)
/
/ Returns  the principal value of the arc tangent of Y/X, using
/ the signs of the two arguments to determine the quadrant of 
/ the result.
/ or NULL if any error is encountered
*/
    int int_value;
    double x = 0.0;
    double y = 0.0;
    double t;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    t = atan2 (y, x);
    sqlite3_result_double (context, t);
}

static void
fnct_math_ceil (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ceil(double X)
/
/ Returns the smallest integer value not less than X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = ceil (sqlite3_value_double (argv[0]));
	  sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = ceil (x);
	  sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_cos (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ cos(double X)
/
/ Returns the cosine of X, where X is given in radians
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = cos (sqlite3_value_double (argv[0]));
	  sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = cos (x);
	  sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_cot (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ cot(double X)
/
/ Returns the cotangent of X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    double tang;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    tang = tan (x);
    if (tang == 0.0)
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = 1.0 / tang;
    sqlite3_result_double (context, x);
}

static void
fnct_math_degrees (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ degrees(double X)
/
/ Returns the argument X, converted from radians to degrees
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = x * 57.29577951308232;
    sqlite3_result_double (context, x);
}

static void
fnct_math_exp (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ exp(double X)
/
/ Returns the value of e (the base of natural logarithms) raised to the power of X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = exp (sqlite3_value_double (argv[0]));
	  sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = exp (x);
	  sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_floor (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ floor(double X)
/
/ Returns the largest integer value not greater than X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = floor (sqlite3_value_double (argv[0]));
	  sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = floor (x);
	  sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_logn (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ log(double X)
/
/ Returns the natural logarithm of X; that is, the base-e logarithm of X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = log (sqlite3_value_double (argv[0]));
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = log (x);
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_logn2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ log(double X, double B)
/
/ Returns the logarithm of X to the base B
/ or NULL if any error is encountered
*/
    int int_value;
    double x = 0.0;
    double b = 1.0;
    double log1;
    double log2;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	b = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  b = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (x <= 0.0 || b <= 1.0)
      {
	  sqlite3_result_null (context);
	  return;
      }
    log1 = log (x);
    if (testInvalidFP (log1))
      {
	  sqlite3_result_null (context);
	  return;
      }
    log2 = log (b);
    if (testInvalidFP (log2))
      {
	  sqlite3_result_null (context);
	  return;
      }
    sqlite3_result_double (context, log1 / log2);
}

static void
fnct_math_log_2 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ log2(double X)
/
/ Returns the base-2 logarithm of X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    double log1;
    double log2;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    log1 = log (x);
    if (testInvalidFP (log1))
      {
	  sqlite3_result_null (context);
	  return;
      }
    log2 = log (2.0);
    sqlite3_result_double (context, log1 / log2);
}

static void
fnct_math_log_10 (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ log10(double X)
/
/ Returns the base-10 logarithm of X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    double log1;
    double log2;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    log1 = log (x);
    if (testInvalidFP (log1))
      {
	  sqlite3_result_null (context);
	  return;
      }
    log2 = log (10.0);
    sqlite3_result_double (context, log1 / log2);
}

static void
fnct_math_pi (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ pi(void)
/
/ Returns the value of (pi)
*/
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    sqlite3_result_double (context, 3.14159265358979323846);
}

static void
fnct_math_pow (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ pow(double X, double Y)
/
/ Returns the value of X raised to the power of Y.
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    double y;
    double p;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_FLOAT)
	y = sqlite3_value_double (argv[1]);
    else if (sqlite3_value_type (argv[1]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[1]);
	  y = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    p = pow (x, y);
    if (testInvalidFP (p))
	sqlite3_result_null (context);
    else
	sqlite3_result_double (context, p);
}

static void
fnct_math_stddev_step (sqlite3_context * context, int argc,
		       sqlite3_value ** argv)
{
/* SQL function:
/ stddev_pop(double X)
/ stddev_samp(double X)
/ var_pop(double X)
/ var_samp(double X)
/
/ aggregate function - STEP
/
*/
    struct stddev_str *p;
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
	return;
    p = sqlite3_aggregate_context (context, sizeof (struct stddev_str));
    if (!(p->cleaned))
      {
	  p->cleaned = 1;
	  p->mean = x;
	  p->quot = 0.0;
	  p->count = 0.0;
      }
    p->count += 1.0;
    p->quot =
	p->quot +
	(((p->count - 1.0) * ((x - p->mean) * (x - p->mean))) / p->count);
    p->mean = p->mean + ((x - p->mean) / p->count);
}

static void
fnct_math_stddev_pop_final (sqlite3_context * context)
{
/* SQL function:
/ stddev_pop(double X)
/ aggregate function -  FINAL
/
*/
    double x;
    struct stddev_str *p = sqlite3_aggregate_context (context, 0);
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = sqrt (p->quot / p->count);
    sqlite3_result_double (context, x);
}

static void
fnct_math_stddev_samp_final (sqlite3_context * context)
{
/* SQL function:
/ stddev_samp(double X)
/ aggregate function -  FINAL
/
*/
    double x;
    struct stddev_str *p = sqlite3_aggregate_context (context, 0);
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = sqrt (p->quot / (p->count - 1.0));
    sqlite3_result_double (context, x);
}

static void
fnct_math_var_pop_final (sqlite3_context * context)
{
/* SQL function:
/ var_pop(double X)
/ aggregate function -  FINAL
/
*/
    double x;
    struct stddev_str *p = sqlite3_aggregate_context (context, 0);
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = p->quot / p->count;
    sqlite3_result_double (context, x);
}

static void
fnct_math_var_samp_final (sqlite3_context * context)
{
/* SQL function:
/ var_samp(double X)
/ aggregate function -  FINAL
/
*/
    double x;
    struct stddev_str *p = sqlite3_aggregate_context (context, 0);
    if (!p)
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = p->quot / (p->count - 1.0);
    sqlite3_result_double (context, x);
}

static void
fnct_math_radians (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ radians(double X)
/
/ Returns the argument X, converted from degrees to radians
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    x = x * .0174532925199432958;
    sqlite3_result_double (context, x);
}

static void
fnct_math_sign (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ sign(double X)
/
/ Returns the sign of the argument as -1, 0, or 1, depending on whether X is negative, zero, or positive
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
	x = sqlite3_value_double (argv[0]);
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
      }
    else
      {
	  sqlite3_result_null (context);
	  return;
      }
    if (x > 0.0)
	sqlite3_result_double (context, 1.0);
    else if (x < 0.0)
	sqlite3_result_double (context, -1.0);
    else
	sqlite3_result_double (context, 0.0);
}

static void
fnct_math_sin (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ sin(double X)
/
/ Returns the sine of X, where X is given in radians
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = sin (sqlite3_value_double (argv[0]));
	  sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = sin (x);
	  sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_sqrt (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ sqrt(double X)
/
/ Returns the square root of a non-negative number X
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = sqrt (sqlite3_value_double (argv[0]));
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = sqrt (x);
	  if (testInvalidFP (x))
	      sqlite3_result_null (context);
	  else
	      sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

static void
fnct_math_tan (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ tan(double X)
/
/ Returns the tangent of X, where X is given in radians
/ or NULL if any error is encountered
*/
    int int_value;
    double x;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_FLOAT)
      {
	  x = tan (sqlite3_value_double (argv[0]));
	  sqlite3_result_double (context, x);
      }
    else if (sqlite3_value_type (argv[0]) == SQLITE_INTEGER)
      {
	  int_value = sqlite3_value_int (argv[0]);
	  x = int_value;
	  x = tan (x);
	  sqlite3_result_double (context, x);
      }
    else
	sqlite3_result_null (context);
}

#endif /* end supporting SQL math functions */

static void
fnct_GeomFromExifGpsBlob (sqlite3_context * context, int argc,
			  sqlite3_value ** argv)
{
/* SQL function:
/ GeomFromExifGpsBlob(BLOB encoded image)
/
/ returns:
/ a POINT geometry
/ or NULL if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    gaiaGeomCollPtr geom;
    unsigned char *geoblob;
    int geosize;
    double longitude;
    double latitude;
    int gpkg_mode = 0;
    int tiny_point = 0;
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (cache != NULL)
      {
	  gpkg_mode = cache->gpkg_mode;
	  tiny_point = cache->tinyPointEnabled;
      }
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    if (gaiaGetGpsCoords (p_blob, n_bytes, &longitude, &latitude))
      {
	  geom = gaiaAllocGeomColl ();
	  geom->Srid = 4326;
	  gaiaAddPointToGeomColl (geom, longitude, latitude);
	  gaiaToSpatiaLiteBlobWkbEx2 (geom, &geoblob, &geosize, gpkg_mode,
				      tiny_point);
	  gaiaFreeGeomColl (geom);
	  sqlite3_result_blob (context, geoblob, geosize, free);
      }
    else
	sqlite3_result_null (context);
}

static char *
guess_mime_type (const unsigned char *p_blob, int n_bytes)
{
/* guessing the mime-type corresponding to some BLOB */
    int blob_type;
    const char *mime = NULL;
    int len;
    char *string = NULL;
    blob_type = gaiaGuessBlobType (p_blob, n_bytes);
    switch (blob_type)
      {
      case GAIA_ZIP_BLOB:
	  mime = "application/zip";
	  break;
      case GAIA_PDF_BLOB:
	  mime = "application/pdf";
	  break;
      case GAIA_TIFF_BLOB:
	  mime = "image/tiff";
	  break;
      case GAIA_GIF_BLOB:
	  mime = "image/gif";
	  break;
      case GAIA_PNG_BLOB:
	  mime = "image/png";
	  break;
      case GAIA_JP2_BLOB:
	  mime = "image/jp2";
	  break;
      case GAIA_JPEG_BLOB:
      case GAIA_EXIF_BLOB:
      case GAIA_EXIF_GPS_BLOB:
	  mime = "image/jpeg";
	  break;
#ifdef ENABLE_LIBXML2		/* including LIBXML2 */
      case GAIA_XML_BLOB:
	  mime = "application/xml";
	  if (gaiaIsSvgXmlBlob (p_blob, n_bytes))
	      mime = "image/svg+xml";
	  break;
#endif /* end including LIBXML2 */
      };
    if (mime != NULL)
      {
	  len = strlen (mime);
	  string = malloc (len + 1);
	  strcpy (string, mime);
      }
    return string;
}

static void
fnct_GetMimeType (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ GetMimeType(BLOB)
/
/ returns:
/ the Mime-Type corresponding to the BLOB
/ or NULL if any error is encountered or no valid mime is defined
*/
    unsigned char *p_blob;
    int n_bytes;
    char *mime = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    mime = guess_mime_type (p_blob, n_bytes);
    if (mime == NULL)
	sqlite3_result_null (context);
    else
	sqlite3_result_text (context, mime, strlen (mime), free);
}

static void
blob_guess (sqlite3_context * context, int argc, sqlite3_value ** argv,
	    int request)
{
/* SQL function:
/ IsGifBlob(BLOB encoded image)
/ IsPngBlob, IsJpegBlob, IsExifBlob, IsExifGpsBlob, IsTiffBlob,
/ IsZipBlob, IsPdfBlob, IsJP2Blob, IsGeometryBlob, 
/ IsCompressedGeometryBlob, IsTinyPointBlob
/
/ returns:
/ 1 if the required BLOB_TYPE is TRUE
/ 0 otherwise
/ or -1 if any error is encountered
*/
    unsigned char *p_blob;
    int n_bytes;
    int blob_type;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, -1);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
    blob_type = gaiaGuessBlobType (p_blob, n_bytes);
    if (request == GAIA_GEOMETRY_BLOB)
      {
	  if (blob_type == GAIA_GEOMETRY_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_COMPRESSED_GEOMETRY_BLOB)
      {
	  if (blob_type == GAIA_COMPRESSED_GEOMETRY_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_TINYPOINT_BLOB)
      {
	  if (blob_type == GAIA_TINYPOINT_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_ZIP_BLOB)
      {
	  if (blob_type == GAIA_ZIP_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_PDF_BLOB)
      {
	  if (blob_type == GAIA_PDF_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_TIFF_BLOB)
      {
	  if (blob_type == GAIA_TIFF_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_GIF_BLOB)
      {
	  if (blob_type == GAIA_GIF_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_PNG_BLOB)
      {
	  if (blob_type == GAIA_PNG_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_JPEG_BLOB)
      {
	  if (blob_type == GAIA_JPEG_BLOB || blob_type == GAIA_EXIF_BLOB
	      || blob_type == GAIA_EXIF_GPS_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_EXIF_BLOB)
      {
	  if (blob_type == GAIA_EXIF_BLOB || blob_type == GAIA_EXIF_GPS_BLOB)
	    {
		sqlite3_result_int (context, 1);
	    }
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_EXIF_GPS_BLOB)
      {
	  if (blob_type == GAIA_EXIF_GPS_BLOB)
	    {
		sqlite3_result_int (context, 1);
	    }
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_WEBP_BLOB)
      {
	  if (blob_type == GAIA_WEBP_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    if (request == GAIA_JP2_BLOB)
      {
	  if (blob_type == GAIA_JP2_BLOB)
	      sqlite3_result_int (context, 1);
	  else
	      sqlite3_result_int (context, 0);
	  return;
      }
    sqlite3_result_int (context, -1);
}

/*
/ the following functions simply readdress the blob_guess()
/ setting the appropriate request mode
*/

static void
fnct_IsGeometryBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_GEOMETRY_BLOB);
}

static void
fnct_IsCompressedGeometryBlob (sqlite3_context * context, int argc,
			       sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_COMPRESSED_GEOMETRY_BLOB);
}

static void
fnct_IsTinyPointBlob (sqlite3_context * context, int argc,
		      sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_TINYPOINT_BLOB);
}

static void
fnct_IsZipBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_ZIP_BLOB);
}

static void
fnct_IsPdfBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_PDF_BLOB);
}

static void
fnct_IsTiffBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_TIFF_BLOB);
}

static void
fnct_IsGifBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_GIF_BLOB);
}

static void
fnct_IsPngBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_PNG_BLOB);
}

static void
fnct_IsJpegBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_JPEG_BLOB);
}

static void
fnct_IsExifBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_EXIF_BLOB);
}

static void
fnct_IsExifGpsBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_EXIF_GPS_BLOB);
}

static void
fnct_IsWebPBlob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_WEBP_BLOB);
}

static void
fnct_IsJP2Blob (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
    blob_guess (context, argc, argv, GAIA_JP2_BLOB);
}

static void
fnct_BlobFromFile (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ BlobFromFile(TEXT filepath)
/
/ returns:
/ some BLOB on success
/ or NULL on failure
*/
    unsigned char *p_blob;
    int n_bytes;
    int max_blob;
    int rd;
    sqlite3 *sqlite = sqlite3_context_db_handle (context);
    const char *path = NULL;
    FILE *in = NULL;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) == SQLITE_TEXT)
	path = (const char *) sqlite3_value_text (argv[0]);
    if (path == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
#ifdef _WIN32
    in = gaia_win_fopen (path, "rb");
#else
    in = fopen (path, "rb");
#endif
    if (in == NULL)
      {
	  sqlite3_result_null (context);
	  return;
      }
    else
      {
	  /* querying the file length */
	  if (fseek (in, 0, SEEK_END) < 0)
	    {
		sqlite3_result_null (context);
		fclose (in);
		return;
	    }
	  n_bytes = ftell (in);
	  max_blob = sqlite3_limit (sqlite, SQLITE_LIMIT_LENGTH, -1);
	  if (n_bytes > max_blob)
	    {
		/* too big; cannot be stored into a BLOB */
		sqlite3_result_null (context);
		fclose (in);
		return;
	    }
	  rewind (in);
	  p_blob = malloc (n_bytes);
	  /* attempting to load the BLOB from the file */
	  rd = fread (p_blob, 1, n_bytes, in);
	  fclose (in);
	  if (rd != n_bytes)
	    {
		/* read error */
		free (p_blob);
		sqlite3_result_null (context);
		return;
	    }
	  sqlite3_result_blob (context, p_blob, n_bytes, free);
      }
}

static void
fnct_BlobToFile (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ BlobToFile(BLOB payload, TEXT filepath)
/
/ returns:
/ 1 on success
/ or 0 on failure
*/
    unsigned char *p_blob;
    int n_bytes;
    const char *path = NULL;
    FILE *out = NULL;
    int ret = 1;
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_BLOB)
      {
	  sqlite3_result_int (context, 0);
	  return;
      }
    if (sqlite3_value_type (argv[1]) == SQLITE_TEXT)
	path = (const char *) sqlite3_value_text (argv[1]);
    if (path == NULL)
      {
	  sqlite3_result_int (context, 0);
	  return;
      }
    p_blob = (unsigned char *) sqlite3_value_blob (argv[0]);
    n_bytes = sqlite3_value_bytes (argv[0]);
#ifdef _WIN32
    out = gaia_win_fopen (path, "wb");
#else
    out = fopen (path, "wb");
#endif
    if (out == NULL)
	ret = 0;
    else
      {
	  /* exporting the BLOB into the file */
	  int wr = fwrite (p_blob, 1, n_bytes, out);
	  if (wr != n_bytes)
	      ret = 0;
	  fclose (out);
      }
    sqlite3_result_int (context, ret);
}

#ifndef OMIT_GEOS		/* only if GEOS is enabled */

static int
load_dxf (sqlite3 * db_handle, struct splite_internal_cache *cache,
	  char *filename, int srid, int append, int force_dims, int mode,
	  int special_rings, char *prefix, char *layer_name)
{
/* scanning a Directory and processing all DXF files */
    int ret;
    gaiaDxfParserPtr dxf = NULL;

/* creating a DXF parser */
    dxf = gaiaCreateDxfParser (srid, force_dims, prefix, layer_name,
			       special_rings);
    if (dxf == NULL)
      {
	  ret = 0;
	  goto stop_dxf;
      }
/* attempting to parse the DXF input file */
    if (gaiaParseDxfFile_r (cache, dxf, filename))
      {
	  /* loading into the DB */
	  if (!gaiaLoadFromDxfParser (db_handle, dxf, mode, append))
	    {
		ret = 0;
		spatialite_e ("DB error while loading: %s\n", filename);
	    }
      }
    else
      {
	  ret = 0;
	  spatialite_e ("Unable to parse: %s\n", filename);
	  goto stop_dxf;
      }
    spatialite_e ("\n*** DXF file successfully loaded\n");
    ret = 1;

  stop_dxf:
    /* destroying the DXF parser */
    gaiaDestroyDxfParser (dxf);
    return ret;
}

static void
fnct_ImportDXF (sqlite3_context * context, int argc, sqlite3_value ** argv)
{
/* SQL function:
/ ImportDXF(TEXT filename)
/     or
/ InportDXF(TEXT filename, INT srid, INT append, TEXT dims,
/           TEXT mode, TEXT special_rings, TEXT table_prefix,
/           TEXT layer_name)
/
/ returns:
/ 1 on success
/ or 0 on failure
/ NULL on invalid arguments
*/
    int ret;
    char *filename;
    int srid = -1;
    int append = 0;
    int special_rings = GAIA_DXF_RING_NONE;
    int mode = GAIA_DXF_IMPORT_BY_LAYER;
    int force_dims = GAIA_DXF_AUTO_2D_3D;
    char *prefix = NULL;
    char *layer_name = NULL;
    sqlite3 *db_handle = sqlite3_context_db_handle (context);
    struct splite_internal_cache *cache = sqlite3_user_data (context);
    GAIA_UNUSED ();		/* LCOV_EXCL_LINE */
    if (sqlite3_value_type (argv[0]) != SQLITE_TEXT)
      {
	  sqlite3_result_null (context);
	  return;
      }
    filename = (char *) sqlite3_value_text (argv[0]);
    if (argc > 7)
      {
	  const char *value;
	  if (sqlite3_value_type (argv[1]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  srid = sqlite3_value_int (argv[1]);
	  if (sqlite3_value_type (argv[2]) != SQLITE_INTEGER)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  append = sqlite3_value_int (argv[2]);
	  if (sqlite3_value_type (argv[3]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = (const char *) sqlite3_value_text (argv[3]);
	  if (strcasecmp (value, "2D") == 0)
	      force_dims = GAIA_DXF_FORCE_2D;
	  else if (strcasecmp (value, "3D") == 0)
	      force_dims = GAIA_DXF_FORCE_3D;
	  else if (strcasecmp (value, "AUTO") == 0)
	      force_dims = GAIA_DXF_AUTO_2D_3D;
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[4]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = (const char *) sqlite3_value_text (argv[4]);
	  if (strcasecmp (value, "MIXED") == 0)
	      mode = GAIA_DXF_IMPORT_MIXED;
	  else if (strcasecmp (value, "DISTINCT") == 0)
	      mode = GAIA_DXF_IMPORT_BY_LAYER;
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[5]) != SQLITE_TEXT)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  value = (const char *) sqlite3_value_text (argv[5]);
	  if (strcasecmp (value, "LINKED") == 0)
	      special_rings = GAIA_DXF_RING_LINKED;
	  else if (strcasecmp (value, "UNLINKED") == 0)
	      special_rings = GAIA_DXF_RING_UNLINKED;
	  else if (strcasecmp (value, "NONE") == 0)
	      special_rings = GAIA_DXF_RING_NONE;
	  else
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[6]) == SQLITE_TEXT)
	      prefix = (char *) sqlite3_value_text (argv[6]);
	  else if (sqlite3_value_type (argv[6]) != SQLITE_NULL)
	    {
		sqlite3_result_null (context);
		return;
	    }
	  if (sqlite3_value_type (argv[7]) == SQLITE_TEXT)
	      layer_name = (char *) sqlite3_value_text (argv[7]);
	  else if (sqlite3_value_type (argv[7]) != SQLITE_NULL)
	    {
		sqlite3_result_null (context);
		return;
	    }
      }

    ret =
	load_dxf (db_handle, cache, filename, srid, append, force_dims, mode,
		  special_rings, prefix, layer_name);
    sqlite3_result_int (context, ret);
}

static int
is_dxf_file (const char *filename)
{
/* testing if a FileName ends with the expected suffix */
    int len = strlen (filename);
    int off = len - 4;
    if (off >= 1)
      {
	  if (strcasecmp (filename + off, ".dxf") == 0)
	      return 1;
      }
    return 0;
}

static int
scan_dxf_dir (sqlite3 * db_handle, struct splite_internal_cache *cache,
	      char *dir_path, int srid, int append, int force_dims, int mode,
	      int special_rings, char *prefix, char *layer_name)
{
/* scanning a Directory and processing all DXF files */
    int cnt = 0;
    char *filepath;
#if defined(_WIN32) && !defined(__MINGW32__)
/* Visual Studio .NET */
    struct _finddata_t c_file;
    intptr_t hFile;
    if (_chdir (dir_path) < 0)
	return 0;
    if ((hFile = _findfirst ("*.*", &c