/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_account.c
    \brief  Utility functions to initialize, fill and clean an lcmaps account structure.
    \author Martijn Steenbakkers for the EU DataGrid.

    Utility functions to initialize, fill and clean an lcmaps account structure.
    This account structure is used by several lcmaps interfaces.
*/

#define _XOPEN_SOURCE	500

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_config.h"

/* Needed for NULL */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "lcmaps_account.h"

/******************************************************************************
Function:       lcmaps_account_info_init()
Description:    Initialize the account info structure
Parameters:
                plcmaps_account: pointer to the account info struct to be filled
Returns:        0: succes
                1: failed to initialize the account info structure
******************************************************************************/
int lcmaps_account_info_init(
        lcmaps_account_info_t * plcmaps_account
)
{
    if (plcmaps_account == NULL)
    {
        /* nothing to initialize */
        return 0;
    }
    plcmaps_account->uid = (uid_t)-1;
    plcmaps_account->pgid_list = NULL;
    plcmaps_account->npgid = 0;
    plcmaps_account->sgid_list = NULL;
    plcmaps_account->nsgid = 0;
    plcmaps_account->poolindex = NULL;

    return 0;
}


/******************************************************************************

Function:       lcmaps_account_info_fill()
Description:    Fill structure with account info
Parameters:
                puid:            ptr to the uid of the local account
                ppgid_list:      ptr to the list of primary gids
                pnpgid:          ptr to the number of primary gids found
                psgid_list:      ptr to the list of secondary gids
                pnsgid:          ptr to the number of secondary gids found
                ppoolindex:      ptr to the pool index
                plcmaps_account: pointer to the account info struct to be filled
Returns:        0: succes
                1: failed to fill the account info structure
                -1: other failure
******************************************************************************/
int lcmaps_account_info_fill(
        uid_t *                 puid,
        gid_t **                ppgid_list,
        int *                   pnpgid,
        gid_t **                psgid_list,
        int *                   pnsgid,
        char **                 ppoolindex,
        lcmaps_account_info_t * plcmaps_account
)
{
    int i = 0;

    if (plcmaps_account == NULL)
    {
        /* lcmaps_account struct has to exist before invocation */
        return -1;
    }
    /* initialize the struct */
    plcmaps_account->uid = (uid_t)-1;
    plcmaps_account->pgid_list = NULL;
    plcmaps_account->npgid = 0;
    plcmaps_account->sgid_list = NULL;
    plcmaps_account->nsgid = 0;
    plcmaps_account->poolindex = NULL;

    if (puid && ppgid_list && pnpgid && psgid_list && pnsgid && ppoolindex)
    {
        plcmaps_account->uid = *puid;
        if (*pnpgid > 0)
        {
            if (*ppgid_list == NULL)
                return -1;
            plcmaps_account->pgid_list = (gid_t *) malloc((size_t)(*pnpgid) * sizeof(gid_t));
            if (plcmaps_account->pgid_list == NULL)
            {
                /* malloc error */
                return -1;
            }
            for (i = 0; i < *pnpgid; i++)
            {
                (plcmaps_account->pgid_list)[i] = (*ppgid_list)[i];
            }
        }
        plcmaps_account->npgid = *pnpgid;
        if (*pnsgid > 0)
        {
            if (*psgid_list == NULL)
                return -1;
            plcmaps_account->sgid_list = (gid_t *) malloc((size_t)(*pnsgid) * sizeof(gid_t));
            if (plcmaps_account->sgid_list == NULL)
            {
                /* malloc error */
                return -1;
            }
            for (i = 0; i < *pnsgid; i++)
            {
                (plcmaps_account->sgid_list)[i] = (*psgid_list)[i];
            }
        }
        plcmaps_account->nsgid = *pnsgid;
        if (*ppoolindex != NULL)    {
	    plcmaps_account->poolindex = strdup(*ppoolindex);
	    if (plcmaps_account->poolindex == NULL)
	    {
                /* malloc error */
		return -1;
	    }
	}
    }
    else
    {
        return 1;
    }
    return 0;
}

/******************************************************************************
Function:       lcmaps_account_info_clean()
Description:    Clean the account info structure
Parameters:
                plcmaps_account: pointer to the account info struct to be filled
Returns:        0: succes
                1: failed to clean the account info structure
******************************************************************************/
int lcmaps_account_info_clean(
        lcmaps_account_info_t * plcmaps_account
)
{
    if (plcmaps_account == NULL)
    {
        /* nothing to clean */
        return 0;
    }
    if (plcmaps_account->pgid_list)
        free(plcmaps_account->pgid_list);
    if (plcmaps_account->sgid_list)
        free(plcmaps_account->sgid_list);
    if (plcmaps_account->poolindex)
        free(plcmaps_account->poolindex);

    return 0;
}

