use strict;
use lib 'lib', '../lib';
use Test::More;
use Log::Log4perl qw/:easy/;
use Exception::Class;
use Data::Dumper;
use DateTime;
use List::Util qw/first/;

BEGIN { 
    if (! $ENV{AWS_ACCESS_KEY_ID} || ! $ENV{AWS_ACCESS_KEY_SECRET} ) {
        plan skip_all => "Set AWS_ACCESS_KEY_ID and AWS_ACCESS_KEY_SECRET environment variables to run these _LIVE_ tests (NOTE: they may incur costs on EMR)";
    }
    else {
        Log::Log4perl->easy_init($ERROR);
        plan tests => 22;
        use_ok( 'Net::Amazon::EMR' );
    }
};


#try ssl first
my $emr = eval {
    Net::Amazon::EMR->new(
	AWSAccessKeyId  => $ENV{AWS_ACCESS_KEY_ID},
	SecretAccessKey => $ENV{AWS_ACCESS_KEY_SECRET},
	ssl             => 1,
    );
};

$emr = Net::Amazon::EMR->new(
	AWSAccessKeyId  => $ENV{AWS_ACCESS_KEY_ID},
	SecretAccessKey => $ENV{AWS_ACCESS_KEY_SECRET},
) if $@;

isa_ok($emr, 'Net::Amazon::EMR');

my $id = $emr->run_job_flow(Name => "API Test Job " . DateTime->now,
                            AmiVersion => '2.4.2',
                            Instances => {
                                Ec2KeyName => 'panda1',
                                InstanceCount => 2,
                                KeepJobFlowAliveWhenNoSteps => 1,
                                MasterInstanceType => 'm1.small',
                                Placement => { AvailabilityZone => 'us-east-1d' },
                                SlaveInstanceType => 'm1.small',
                            },
                            BootstrapActions => {
                                Name => 'Test Bootstrap Action',
                                ScriptBootstrapAction => {
                                    Path => 'file:///bin/some-bootstrap-action',
                                    Args => [ qw/1 2 3 4 5/ ],
                                }
                            },
                            JobFlowRole => "EMR_EC2_DefaultRole",
                            ServiceRole => "EMR_DefaultRole"
                        );
$SIG{__DIE__} = sub {
    die @_ if $^S;
    $emr->terminate_job_flows(JobFlowIds => [ $id->JobFlowId ]);
    print STDERR "*** Job Terminated ***\n";
};

isa_ok($id, 'Net::Amazon::EMR::RunJobFlowResult', 'run_job_flow type');
#print "Job flow id = ". Dumper($id);

#my $result = $emr->describe_job_flows(CreatedAfter => DateTime->new(year => 2012, month => 12, day => 17));
#my $result = $emr->describe_job_flows(JobFlowIds => [ $id->JobFlowId ]);
my $result = $emr->list_clusters(CreatedAfter => DateTime->today);
isa_ok($result, 'Net::Amazon::EMR::ListClustersResult', 'list_clusters type');
#use Data::Dumper; print Dumper($result);

my $current_cluster = first { $_->Id eq $id->JobFlowId } @{$result->Clusters};

ok(defined($current_cluster), "list_clusters current cluster");

$result = $emr->describe_cluster(ClusterId => $id->JobFlowId);
isa_ok($result, 'Net::Amazon::EMR::DescribeClusterResult', 'describe_cluster type');
ok($result->Cluster->Id eq $id->JobFlowId, "describe_cluster id");

$result = $emr->list_instance_groups(ClusterId => $id->JobFlowId);

ok(@{$result->InstanceGroups} == 2, "run_job_flow 2 instance groups");
ok($result->InstanceGroups->[0]->RequestedInstanceCount == 1, "run_job_flow 2 instance count");

$result = $emr->list_bootstrap_actions(ClusterId => $id->JobFlowId);
isa_ok($result, 'Net::Amazon::EMR::ListBootstrapActionsResult', 'list_bootstrap_actions type');
ok((scalar grep { $_->Name eq 'Test Bootstrap Action' } @{$result->BootstrapActions}) == 1, "bootstrap actions list");

$result = $emr->list_instances(ClusterId => $id->JobFlowId);
isa_ok($result, 'Net::Amazon::EMR::ListInstancesResult', 'list_instances type');

$result = $emr->list_steps(ClusterId => $id->JobFlowId);
isa_ok($result, 'Net::Amazon::EMR::ListStepsResult', 'list_steps type');


my $igs = $emr->add_instance_groups(JobFlowId => $id->JobFlowId,
                                    InstanceGroups => [
                                        { InstanceCount => 1,
                                          InstanceRole => 'TASK',
                                          InstanceType => 'm1.small',
                                          Market => 'ON_DEMAND',
                                          Name => 'API Test Group',
                                        }]);
isa_ok($igs, 'Net::Amazon::EMR::AddInstanceGroupsResult', 'add_instance_groups type');

$result = $emr->list_instance_groups(ClusterId => $id->JobFlowId);

ok(@{$result->InstanceGroups} == 3, "add_instance_group 3 instance groups");

# Expect an error - instance group may not be modified while in startup
eval {
    $emr->modify_instance_groups(InstanceGroups => [
                                     { InstanceGroupId => $result->InstanceGroups->[2]->Id,
                                       InstanceCount => 2,
                                     }]);
};
my $e = Exception::Class->caught('Net::Amazon::EMR::Exception');
ok($e && $e->message =~m/An instance group may only be modified when the job flow is running or waiting/, 'modify_instance_groups (expecting an ERROR)');


ok($emr->set_visible_to_all_users(JobFlowIds => [ $id->JobFlowId ],
                                    VisibleToAllUsers => 0,
   ), "set_visible_to_all_users");
ok($emr->set_termination_protection(JobFlowIds => [ $id->JobFlowId ],
                                    TerminationProtected => 'false',
   ), "set_termination_protection");


#use Data::Dumper; print Dumper($result);
#print STDERR Dumper($result->as_hash);

ok($emr->add_job_flow_steps(JobFlowId => $id->JobFlowId,
                            Steps => [ 
                                { ActionOnFailure => 'CONTINUE',
                                  HadoopJarStep => { Jar => 'MyJar',
                                                     MainClass => 'MainClass',
                                                     Args => [ 'arg1' ] },
                                  Name => 'TestStep',
                                }
                            ]), 'add_job_flow_steps');

                   
ok($emr->terminate_job_flows(JobFlowIds => [ $id->JobFlowId ]), "terminate_job_flow");

#$result = $emr->describe_job_flows(JobFlowIds => [ $id->JobFlowId ]);
$result = $emr->describe_cluster(ClusterId => $id->JobFlowId);

#print STDERR "VTAU: " . $result->JobFlows->[0]->VisibleToAllUsers . "\n";
ok(! $result->Cluster->VisibleToAllUsers, "set_visible_to_all_users value");
ok(! $result->Cluster->TerminationProtected, "set_termination_protection value");



