/*	$NetBSD: if_rgereg.h,v 1.9.2.1 2025/11/24 17:32:48 martin Exp $	*/
/*	$OpenBSD: if_rgereg.h,v 1.16 2025/10/27 03:18:36 kevlo Exp $	*/

/*
 * Copyright (c) 2019, 2020, 2025 Kevin Lo <kevlo@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define RGE_PCI_BAR0		PCI_MAPREG_START
#define RGE_PCI_BAR1		(PCI_MAPREG_START + 4)
#define RGE_PCI_BAR2		(PCI_MAPREG_START + 8)

#define RGE_MAC0		0x0000
#define RGE_MAC4		0x0004
#define RGE_MAR0		0x0008
#define RGE_MAR4		0x000c
#define RGE_TXDESC_ADDR_LO	0x0020
#define RGE_TXDESC_ADDR_HI	0x0024
#define RGE_INT_CFG0		0x0034
#define RGE_CMD			0x0037
#define RGE_IMR			0x0038
#define RGE_ISR			0x003c
#define RGE_TXCFG		0x0040
#define RGE_RXCFG		0x0044
#define RGE_TIMERCNT		0x0048
#define RGE_EECMD		0x0050
#define RGE_CFG0		0x0051
#define RGE_CFG1		0x0052
#define RGE_CFG2		0x0053
#define RGE_CFG3		0x0054
#define RGE_CFG4		0x0055
#define RGE_CFG5		0x0056
#define RGE_TDFNR		0x0057
#define RGE_TIMERINT0		0x0058
#define RGE_TIMERINT1		0x005c
#define RGE_CSIDR		0x0064
#define RGE_CSIAR		0x0068
#define RGE_PHYSTAT		0x006c
#define RGE_PMCH		0x006f
#define RGE_INT_CFG1		0x007a
#define RGE_EPHYAR		0x0080
#define RGE_TIMERINT2		0x008c
#define RGE_TXSTART		0x0090
#define RGE_MACOCP		0x00b0
#define RGE_PHYOCP		0x00b8
#define RGE_DLLPR		0x00d0
#define RGE_TWICMD		0x00d2
#define RGE_MCUCMD		0x00d3
#define RGE_RXMAXSIZE		0x00da
#define RGE_CPLUSCMD		0x00e0
#define RGE_IM			0x00e2
#define RGE_RXDESC_ADDR_LO	0x00e4
#define RGE_RXDESC_ADDR_HI	0x00e8
#define RGE_PPSW		0x00f2
#define RGE_TIMERINT3		0x00f4
#define RGE_RADMFIFO_PROTECT	0x0402
#define RGE_INTMITI(i)		(0x0a00 + (i) * 4)
#define RGE_PHYBASE		0x0a40
#define RGE_EPHYAR_EXT_ADDR	0x0ffe
#define RGE_ADDR0		0x19e0
#define RGE_ADDR1		0x19e4
#define RGE_RSS_CTRL		0x4500
#define RGE_RXQUEUE_CTRL	0x4800
#define RGE_EEE_TXIDLE_TIMER	0x6048

/* Flags for register RGE_INT_CFG0 */
#define RGE_INT_CFG0_EN			0x01
#define RGE_INT_CFG0_TIMEOUT_BYPASS	0x02
#define RGE_INT_CFG0_MITIGATION_BYPASS	0x04
#define RGE_INT_CFG0_RDU_BYPASS_8126	0x10
#define RGE_INT_CFG0_AVOID_MISS_INTR	0x40

/* Flags for register RGE_CMD */
#define RGE_CMD_RXBUF_EMPTY	0x01
#define RGE_CMD_TXENB		0x04
#define RGE_CMD_RXENB		0x08
#define RGE_CMD_RESET		0x10
#define RGE_CMD_STOPREQ		0x80

/* Flags for register RGE_ISR */
#define RGE_ISR_RX_OK		0x00000001
#define RGE_ISR_RX_ERR		0x00000002
#define RGE_ISR_TX_OK		0x00000004
#define RGE_ISR_TX_ERR		0x00000008
#define RGE_ISR_RX_DESC_UNAVAIL	0x00000010
#define RGE_ISR_LINKCHG		0x00000020
#define RGE_ISR_RX_FIFO_OFLOW	0x00000040
#define RGE_ISR_TX_DESC_UNAVAIL	0x00000080
#define RGE_ISR_SWI		0x00000100
#define RGE_ISR_PCS_TIMEOUT	0x00004000
#define RGE_ISR_SYSTEM_ERR	0x00008000

#define RGE_INTRS		\
	(RGE_ISR_RX_OK | RGE_ISR_RX_ERR | RGE_ISR_TX_OK |		\
	RGE_ISR_TX_ERR | RGE_ISR_RX_DESC_UNAVAIL | RGE_ISR_LINKCHG |	\
	RGE_ISR_TX_DESC_UNAVAIL | RGE_ISR_PCS_TIMEOUT | RGE_ISR_SYSTEM_ERR)

#define RGE_INTRS_TIMER		\
	(RGE_ISR_RX_ERR | RGE_ISR_TX_ERR | RGE_ISR_PCS_TIMEOUT |	\
	RGE_ISR_SYSTEM_ERR)

#define	RGE_MAX_NINTRS	1

/* Flags for register RGE_TXCFG */
#define RGE_TXCFG_HWREV		0x7cf00000

/* Flags for register RGE_RXCFG */
#define RGE_RXCFG_ALLPHYS	0x00000001
#define RGE_RXCFG_INDIV		0x00000002
#define RGE_RXCFG_MULTI		0x00000004
#define RGE_RXCFG_BROAD		0x00000008
#define RGE_RXCFG_RUNT		0x00000010
#define RGE_RXCFG_ERRPKT	0x00000020
#define RGE_RXCFG_VLANSTRIP	0x00c00000

/* Flags for register RGE_EECMD */
#define RGE_EECMD_WRITECFG	0xc0

/* Flags for register RGE_CFG1 */
#define RGE_CFG1_PM_EN		0x01
#define RGE_CFG1_SPEED_DOWN	0x10

/* Flags for register RGE_CFG2 */
#define RGE_CFG2_PMSTS_EN	0x20
#define RGE_CFG2_CLKREQ_EN	0x80

/* Flags for register RGE_CFG3 */
#define RGE_CFG3_RDY_TO_L23	0x02
#define RGE_CFG3_WOL_LINK	0x10
#define RGE_CFG3_WOL_MAGIC	0x20

/* Flags for register RGE_CFG5 */
#define RGE_CFG5_PME_STS	0x01
#define RGE_CFG5_WOL_LANWAKE	0x02
#define RGE_CFG5_WOL_UCAST	0x10
#define RGE_CFG5_WOL_MCAST	0x20
#define RGE_CFG5_WOL_BCAST	0x40

/* Flags for register RGE_CSIAR */
#define RGE_CSIAR_BYTE_EN	0x0000000f
#define RGE_CSIAR_BYTE_EN_SHIFT	12
#define RGE_CSIAR_ADDR_MASK	0x00000fff
#define RGE_CSIAR_BUSY		0x80000000

/* Flags for register RGE_PHYSTAT */
#define RGE_PHYSTAT_FDX		0x0001
#define RGE_PHYSTAT_LINK	0x0002
#define RGE_PHYSTAT_10MBPS	0x0004
#define RGE_PHYSTAT_100MBPS	0x0008
#define RGE_PHYSTAT_1000MBPS	0x0010
#define RGE_PHYSTAT_RXFLOW	0x0020
#define RGE_PHYSTAT_TXFLOW	0x0040
#define RGE_PHYSTAT_2500MBPS	0x0400
#define RGE_PHYSTAT_5000MBPS	0x1000
#define RGE_PHYSTAT_10000MBPS	0x4000

/* Flags for register RGE_EPHYAR */
#define RGE_EPHYAR_DATA_MASK	0x0000ffff
#define RGE_EPHYAR_BUSY		0x80000000
#define RGE_EPHYAR_ADDR_MASK	0x0000007f
#define RGE_EPHYAR_ADDR_SHIFT	16

/* Flags for register RGE_TXSTART */
#define RGE_TXSTART_START	0x0001

/* Flags for register RGE_MACOCP */
#define RGE_MACOCP_DATA_MASK	0x0000ffff
#define RGE_MACOCP_BUSY		0x80000000
#define RGE_MACOCP_ADDR_SHIFT	16

/* Flags for register RGE_PHYOCP */
#define RGE_PHYOCP_DATA_MASK	0x0000ffff
#define RGE_PHYOCP_BUSY		0x80000000
#define RGE_PHYOCP_ADDR_SHIFT	16

/* Flags for register RGE_DLLPR. */
#define RGE_DLLPR_PFM_EN	0x40
#define RGE_DLLPR_TX_10M_PS_EN	0x80

/* Flags for register RGE_MCUCMD */
#define RGE_MCUCMD_RXFIFO_EMPTY	0x10
#define RGE_MCUCMD_TXFIFO_EMPTY	0x20
#define RGE_MCUCMD_IS_OOB	0x80

/* Flags for register RGE_CPLUSCMD */
#define RGE_CPLUSCMD_RXCSUM	0x0020

#define RGE_TX_NSEGS		32
#define	RGE_TX_LIST_CNT		1024
#define	RGE_RX_LIST_CNT		1024
#define RGE_ALIGN		256
#define RGE_TX_LIST_SZ		(sizeof(struct rge_tx_desc) * RGE_TX_LIST_CNT)
#define RGE_RX_LIST_SZ		(sizeof(struct rge_rx_desc) * RGE_RX_LIST_CNT)
#define RGE_NEXT_TX_DESC(x)	(((x) + 1) % RGE_TX_LIST_CNT)
#define RGE_NEXT_RX_DESC(x)	(((x) + 1) % RGE_RX_LIST_CNT)
#define RGE_ADDR_LO(y)		((uint64_t) (y) & 0xffffffff)
#define RGE_ADDR_HI(y)		((uint64_t) (y) >> 32)
#define RGE_OWN(x)							\
	(letoh32((x)->hi_qword1.rx_qword4.rge_cmdsts) & RGE_RDCMDSTS_OWN)
#define RGE_RXBYTES(x)							\
	(letoh32((x)->hi_qword1.rx_qword4.rge_cmdsts) & RGE_RDCMDSTS_FRAGLEN)

#define RGE_ADV_2500TFDX	0x0080
#define RGE_ADV_5000TFDX	0x0100
#define RGE_ADV_10000TFDX	0x1000

/* Tx descriptor */
struct rge_tx_desc {
	uint32_t		rge_cmdsts;
	uint32_t		rge_extsts;
	uint64_t		rge_addr;
	uint32_t		reserved[4];
};

#define RGE_TDCMDSTS_COLL	0x000f0000
#define RGE_TDCMDSTS_EXCESSCOLL	0x00100000
#define RGE_TDCMDSTS_TXERR	0x00800000
#define RGE_TDCMDSTS_EOF	0x10000000
#define RGE_TDCMDSTS_SOF	0x20000000
#define RGE_TDCMDSTS_EOR	0x40000000
#define RGE_TDCMDSTS_OWN	0x80000000

#define RGE_TDEXTSTS_VTAG	0x00020000
#define RGE_TDEXTSTS_IPCSUM	0x20000000
#define RGE_TDEXTSTS_TCPCSUM	0x40000000
#define RGE_TDEXTSTS_UDPCSUM	0x80000000

/* Rx descriptor */
struct rge_rx_desc {
	union {
		struct {
			uint32_t	rsvd0;
			uint32_t	rsvd1;
		} rx_qword0;
	} lo_qword0;

	union {
		struct {
			uint32_t	rss;
			uint16_t	length;
			uint16_t	hdr_info;
		} rx_qword1;

		struct {
			uint32_t	rsvd2;
			uint32_t	rsvd3;
		} rx_qword2;
	} lo_qword1;

	union {
		uint64_t		rge_addr;

		struct {
			uint64_t	timestamp;
		} rx_timestamp;

		struct {
			uint32_t	rsvd4;
			uint32_t	rsvd5;
		} rx_qword3;
	} hi_qword0;

	union {
		struct {
			uint32_t	rge_extsts;
			uint32_t	rge_cmdsts;
		} rx_qword4;

		struct {
			uint16_t	rsvd6;
			uint16_t	rsvd7;
			uint32_t	rsvd8;
		} rx_ptp;
	} hi_qword1;
};

#define RGE_RDCMDSTS_RXERRSUM	0x00100000
#define RGE_RDCMDSTS_EOF	0x01000000
#define RGE_RDCMDSTS_SOF	0x02000000
#define RGE_RDCMDSTS_EOR	0x40000000
#define RGE_RDCMDSTS_OWN	0x80000000
#define RGE_RDCMDSTS_FRAGLEN	0x00003fff

#define RGE_RDEXTSTS_VTAG	0x00010000
#define RGE_RDEXTSTS_VLAN_MASK	0x0000ffff
#define RGE_RDEXTSTS_TCPCSUMERR 0x01000000
#define RGE_RDEXTSTS_UDPCSUMERR 0x02000000
#define RGE_RDEXTSTS_IPCSUMERR  0x04000000
#define RGE_RDEXTSTS_TCPPKT     0x10000000
#define RGE_RDEXTSTS_UDPPKT     0x20000000
#define RGE_RDEXTSTS_IPV4	0x40000000
#define RGE_RDEXTSTS_IPV6	0x80000000

struct rge_txq {
	struct mbuf		*txq_mbuf;
	bus_dmamap_t		txq_dmamap;
	int			txq_descidx;
};

struct rge_rxq {
	struct mbuf		*rxq_mbuf;
	bus_dmamap_t		rxq_dmamap;
};

struct rge_tx {
	struct rge_txq		rge_txq[RGE_TX_LIST_CNT];
	int			rge_txq_prodidx;
	int			rge_txq_considx;

	bus_dma_segment_t	rge_tx_listseg;
	int			rge_tx_listnseg;
	bus_dmamap_t		rge_tx_list_map;
	struct rge_tx_desc	*rge_tx_list;
};

struct rge_rx {
	struct rge_rxq		rge_rxq[RGE_RX_LIST_CNT];
	int			rge_rxq_prodidx;
	int			rge_rxq_considx;

	bus_dma_segment_t	rge_rx_listseg;
	int			rge_rx_listnseg;
	bus_dmamap_t		rge_rx_list_map;
	struct rge_rx_desc	*rge_rx_list;

	struct mbuf		*rge_head;
	struct mbuf		**rge_tail;
};

struct rge_queues {
	struct rge_softc	*q_sc;
	void			*q_ihc;
	int			q_index;
	char			q_name[16];
	pci_intr_handle_t	q_ih;
	struct rge_tx		q_tx;
	struct rge_rx		q_rx;
};

/* Ram version */
#define RGE_MAC_R25D_RCODE_VER		0x0027
#define RGE_MAC_R26_1_RCODE_VER		0x0033
#define RGE_MAC_R26_2_RCODE_VER		0x0060
#define RGE_MAC_R27_RCODE_VER		0x0036
#define RGE_MAC_R25_RCODE_VER		0x0b33
#define RGE_MAC_R25B_RCODE_VER		0x0b99

enum rge_mac_type {
	MAC_CFG_UNKNOWN = 1,
	MAC_R25,
	MAC_R25B,
	MAC_R25D,
	MAC_R26_1,
	MAC_R26_2,
	MAC_R27
};

#define RGE_TYPE_R26(sc)						\
	((sc)->rge_type == MAC_R26_1 || (sc)->rge_type == MAC_R26_2)

#define RGE_TIMEOUT		100

#if 0
#define RGE_JUMBO_FRAMELEN	9216
#else
#define RGE_JUMBO_FRAMELEN	(MCLBYTES - ETHER_ALIGN)
#endif
#define RGE_JUMBO_MTU							\
	(RGE_JUMBO_FRAMELEN - ETHER_HDR_LEN - ETHER_CRC_LEN - 		\
	ETHER_VLAN_ENCAP_LEN)

#define RGE_TXCFG_CONFIG	0x03000700
#define RGE_RXCFG_CONFIG	0x41000700
#define RGE_RXCFG_CONFIG_8125B	0x41000c00
#define RGE_RXCFG_CONFIG_8125D	0x41200c00
#define RGE_RXCFG_CONFIG_8126	0x41200d00

struct rge_softc {
	device_t		sc_dev;
	struct ethercom		sc_ec; 		/* Ethernet common data */
	void			*sc_ihs[RGE_MAX_NINTRS]; /* interrupt vectoring */
	bus_space_handle_t	rge_bhandle;	/* bus space handle */
	bus_space_tag_t		rge_btag;	/* bus space tag */
	bus_size_t		rge_bsize;
	bus_dma_tag_t		sc_dmat;
	pci_chipset_tag_t	sc_pc;
	pcitag_t		sc_tag;
	pci_intr_handle_t	*sc_intrs;
	bus_dma_segment_t 	sc_rx_seg;
	bus_dmamap_t		sc_rx_dmamap;
	struct ifmedia		sc_media;	/* media info */
	enum rge_mac_type	rge_type;

	struct rge_queues	*sc_queues;
	unsigned int		sc_nqueues;

	callout_t		sc_timeout;	/* tick timeout */

	uint8_t 		sc_enaddr[ETHER_ADDR_LEN];
	uint64_t		rge_mcodever;
	uint16_t		rge_rcodever;
	uint32_t		rge_flags;
#define RGE_FLAG_MSI		0x00000001

	uint32_t		rge_intrs;
	uint32_t		rge_tx_ack;
	uint32_t		rge_rx_ack;
	int			rge_timerintr;
#define RGE_IMTYPE_NONE		0
#define RGE_IMTYPE_SIM		1
};

/*
 * Register space access macros.
 */
#define RGE_WRITE_4(sc, reg, val)	\
	bus_space_write_4(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_2(sc, reg, val)	\
	bus_space_write_2(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_1(sc, reg, val)	\
	bus_space_write_1(sc->rge_btag, sc->rge_bhandle, reg, val)

#define RGE_READ_4(sc, reg)		\
	bus_space_read_4(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_2(sc, reg)		\
	bus_space_read_2(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_1(sc, reg)		\
	bus_space_read_1(sc->rge_btag, sc->rge_bhandle, reg)

#define RGE_SETBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) | (val))
#define RGE_SETBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) | (val))
#define RGE_SETBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) | (val))

#define RGE_CLRBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) & ~(val))
#define RGE_CLRBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) & ~(val))
#define RGE_CLRBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) & ~(val))

#define RGE_EPHY_SETBIT(sc, reg, val)	\
	rge_write_ephy(sc, reg, rge_read_ephy(sc, reg) | (val))

#define RGE_EPHY_CLRBIT(sc, reg, val)	\
	rge_write_ephy(sc, reg, rge_read_ephy(sc, reg) & ~(val))

#define RGE_PHY_SETBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) | (val))

#define RGE_PHY_CLRBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) & ~(val))

#define RGE_MAC_SETBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) | (val))

#define RGE_MAC_CLRBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) & ~(val))

static const struct {
	uint16_t	reg;
	uint16_t	val;
} rtl8125_mac_bps[] = {
	{ 0xf800, 0xe010 }, { 0xf802, 0xe012 }, { 0xf804, 0xe022 },
	{ 0xf806, 0xe024 }, { 0xf808, 0xe029 }, { 0xf80a, 0xe02b },
	{ 0xf80c, 0xe094 }, { 0xf80e, 0xe09d }, { 0xf810, 0xe09f },
	{ 0xf812, 0xe0aa }, { 0xf814, 0xe0b5 }, { 0xf816, 0xe0c6 },
	{ 0xf818, 0xe0cc }, { 0xf81a, 0xe0d1 }, { 0xf81c, 0xe0d6 },
	{ 0xf81e, 0xe0d8 }, { 0xf820, 0xc602 }, { 0xf822, 0xbe00 },
	{ 0xf824, 0x0000 }, { 0xf826, 0xc60f }, { 0xf828, 0x73c4 },
	{ 0xf82a, 0x49b3 }, { 0xf82c, 0xf106 }, { 0xf82e, 0x73c2 },
	{ 0xf830, 0xc608 }, { 0xf832, 0xb406 }, { 0xf834, 0xc609 },
	{ 0xf836, 0xff80 }, { 0xf838, 0xc605 }, { 0xf83a, 0xb406 },
	{ 0xf83c, 0xc605 }, { 0xf83e, 0xff80 }, { 0xf840, 0x0544 },
	{ 0xf842, 0x0568 }, { 0xf844, 0xe906 }, { 0xf846, 0xcde8 },
	{ 0xf848, 0xc602 }, { 0xf84a, 0xbe00 }, { 0xf84c, 0x0000 },
	{ 0xf84e, 0x48c1 }, { 0xf850, 0x48c2 }, { 0xf852, 0x9c46 },
	{ 0xf854, 0xc402 }, { 0xf856, 0xbc00 }, { 0xf858, 0x0a12 },
	{ 0xf85a, 0xc602 }, { 0xf85c, 0xbe00 }, { 0xf85e, 0x0eba },
	{ 0xf860, 0x1501 }, { 0xf862, 0xf02a }, { 0xf864, 0x1500 },
	{ 0xf866, 0xf15d }, { 0xf868, 0xc661 }, { 0xf86a, 0x75c8 },
	{ 0xf86c, 0x49d5 }, { 0xf86e, 0xf00a }, { 0xf870, 0x49d6 },
	{ 0xf872, 0xf008 }, { 0xf874, 0x49d7 }, { 0xf876, 0xf006 },
	{ 0xf878, 0x49d8 }, { 0xf87a, 0xf004 }, { 0xf87c, 0x75d2 },
	{ 0xf87e, 0x49d9 }, { 0xf880, 0xf150 }, { 0xf882, 0xc553 },
	{ 0xf884, 0x77a0 }, { 0xf886, 0x75c8 }, { 0xf888, 0x4855 },
	{ 0xf88a, 0x4856 }, { 0xf88c, 0x4857 }, { 0xf88e, 0x4858 },
	{ 0xf890, 0x48da }, { 0xf892, 0x48db }, { 0xf894, 0x49fe },
	{ 0xf896, 0xf002 }, { 0xf898, 0x485a }, { 0xf89a, 0x49ff },
	{ 0xf89c, 0xf002 }, { 0xf89e, 0x485b }, { 0xf8a0, 0x9dc8 },
	{ 0xf8a2, 0x75d2 }, { 0xf8a4, 0x4859 }, { 0xf8a6, 0x9dd2 },
	{ 0xf8a8, 0xc643 }, { 0xf8aa, 0x75c0 }, { 0xf8ac, 0x49d4 },
	{ 0xf8ae, 0xf033 }, { 0xf8b0, 0x49d0 }, { 0xf8b2, 0xf137 },
	{ 0xf8b4, 0xe030 }, { 0xf8b6, 0xc63a }, { 0xf8b8, 0x75c8 },
	{ 0xf8ba, 0x49d5 }, { 0xf8bc, 0xf00e }, { 0xf8be, 0x49d6 },
	{ 0xf8c0, 0xf00c }, { 0xf8c2, 0x49d7 }, { 0xf8c4, 0xf00a },
	{ 0xf8c6, 0x49d8 }, { 0xf8c8, 0xf008 }, { 0xf8ca, 0x75d2 },
	{ 0xf8cc, 0x49d9 }, { 0xf8ce, 0xf005 }, { 0xf8d0, 0xc62e },
	{ 0xf8d2, 0x75c0 }, { 0xf8d4, 0x49d7 }, { 0xf8d6, 0xf125 },
	{ 0xf8d8, 0xc528 }, { 0xf8da, 0x77a0 }, { 0xf8dc, 0xc627 },
	{ 0xf8de, 0x75c8 }, { 0xf8e0, 0x4855 }, { 0xf8e2, 0x4856 },
	{ 0xf8e4, 0x4857 }, { 0xf8e6, 0x4858 }, { 0xf8e8, 0x48da },
	{ 0xf8ea, 0x48db }, { 0xf8ec, 0x49fe }, { 0xf8ee, 0xf002 },
	{ 0xf8f0, 0x485a }, { 0xf8f2, 0x49ff }, { 0xf8f4, 0xf002 },
	{ 0xf8f6, 0x485b }, { 0xf8f8, 0x9dc8 }, { 0xf8fa, 0x75d2 },
	{ 0xf8fc, 0x4859 }, { 0xf8fe, 0x9dd2 }, { 0xf900, 0xc616 },
	{ 0xf902, 0x75c0 }, { 0xf904, 0x4857 }, { 0xf906, 0x9dc0 },
	{ 0xf908, 0xc613 }, { 0xf90a, 0x75c0 }, { 0xf90c, 0x49da },
	{ 0xf90e, 0xf003 }, { 0xf910, 0x49d0 }, { 0xf912, 0xf107 },
	{ 0xf914, 0xc60b }, { 0xf916, 0xc50e }, { 0xf918, 0x48d9 },
	{ 0xf91a, 0x9dc0 }, { 0xf91c, 0x4859 }, { 0xf91e, 0x9dc0 },
	{ 0xf920, 0xc608 }, { 0xf922, 0xc702 }, { 0xf924, 0xbf00 },
	{ 0xf926, 0x3ae0 }, { 0xf928, 0xe860 }, { 0xf92a, 0xb400 },
	{ 0xf92c, 0xb5d4 }, { 0xf92e, 0xe908 }, { 0xf930, 0xe86c },
	{ 0xf932, 0x1200 }, { 0xf934, 0xc409 }, { 0xf936, 0x6780 },
	{ 0xf938, 0x48f1 }, { 0xf93a, 0x8f80 }, { 0xf93c, 0xc404 },
	{ 0xf93e, 0xc602 }, { 0xf940, 0xbe00 }, { 0xf942, 0x10aa },
	{ 0xf944, 0xc010 }, { 0xf946, 0xea7c }, { 0xf948, 0xc602 },
	{ 0xf94a, 0xbe00 }, { 0xf94c, 0x0000 }, { 0xf94e, 0x740a },
	{ 0xf950, 0x4846 }, { 0xf952, 0x4847 }, { 0xf954, 0x9c0a },
	{ 0xf956, 0xc607 }, { 0xf958, 0x74c0 }, { 0xf95a, 0x48c6 },
	{ 0xf95c, 0x9cc0 }, { 0xf95e, 0xc602 }, { 0xf960, 0xbe00 },
	{ 0xf962, 0x13fe }, { 0xf964, 0xe054 }, { 0xf966, 0x72ca },
	{ 0xf968, 0x4826 }, { 0xf96a, 0x4827 }, { 0xf96c, 0x9aca },
	{ 0xf96e, 0xc607 }, { 0xf970, 0x72c0 }, { 0xf972, 0x48a6 },
	{ 0xf974, 0x9ac0 }, { 0xf976, 0xc602 }, { 0xf978, 0xbe00 },
	{ 0xf97a, 0x07dc }, { 0xf97c, 0xe054 }, { 0xf97e, 0xc60f },
	{ 0xf980, 0x74c4 }, { 0xf982, 0x49cc }, { 0xf984, 0xf109 },
	{ 0xf986, 0xc60c }, { 0xf988, 0x74ca }, { 0xf98a, 0x48c7 },
	{ 0xf98c, 0x9cca }, { 0xf98e, 0xc609 }, { 0xf990, 0x74c0 },
	{ 0xf992, 0x4846 }, { 0xf994, 0x9cc0 }, { 0xf996, 0xc602 },
	{ 0xf998, 0xbe00 }, { 0xf99a, 0x2480 }, { 0xf99c, 0xe092 },
	{ 0xf99e, 0xe0c0 }, { 0xf9a0, 0xe054 }, { 0xf9a2, 0x7420 },
	{ 0xf9a4, 0x48c0 }, { 0xf9a6, 0x9c20 }, { 0xf9a8, 0x7444 },
	{ 0xf9aa, 0xc602 }, { 0xf9ac, 0xbe00 }, { 0xf9ae, 0x12f8 },
	{ 0xf9b0, 0x1bff }, { 0xf9b2, 0x46eb }, { 0xf9b4, 0x1bff },
	{ 0xf9b6, 0xc102 }, { 0xf9b8, 0xb900 }, { 0xf9ba, 0x0d5a },
	{ 0xf9bc, 0x1bff }, { 0xf9be, 0x46eb }, { 0xf9c0, 0x1bff },
	{ 0xf9c2, 0xc102 }, { 0xf9c4, 0xb900 }, { 0xf9c6, 0x0e2a },
	{ 0xf9c8, 0xc602 }, { 0xf9ca, 0xbe00 }, { 0xf9cc, 0x0000 },
	{ 0xf9ce, 0xc602 }, { 0xf9d0, 0xbe00 }, { 0xf9d2, 0x0000 },
	{ 0xf9d4, 0x0000 }, { 0xf9d6, 0x0000 }, { 0xf9d8, 0x0000 },
	{ 0xf9da, 0x0000 }, { 0xf9dc, 0x0000 }, { 0xf9de, 0x0000 },
	{ 0xf9e0, 0x0000 }, { 0xf9e2, 0x0000 }, { 0xf9e4, 0x0000 },
	{ 0xf9e6, 0x0000 }, { 0xf9e8, 0x0000 }, { 0xf9ea, 0x0000 },
	{ 0xf9ec, 0x0000 }, { 0xf9ee, 0x0000 }, { 0xf9f0, 0x0000 },
	{ 0xf9f2, 0x0000 }, { 0xf9f4, 0x0000 }, { 0xf9f6, 0x0000 },
	{ 0xf9f8, 0x0000 }, { 0xf9fa, 0x0000 }, { 0xf9fc, 0x0000 },
	{ 0xf9fe, 0x0000 }
}, rtl8125b_mac_bps[] = {
	{ 0xf800, 0xe010 }, { 0xf802, 0xe01b }, { 0xf804, 0xe026 },
	{ 0xf806, 0xe037 }, { 0xf808, 0xe03d }, { 0xf80a, 0xe057 },
	{ 0xf80c, 0xe05b }, { 0xf80e, 0xe060 }, { 0xf810, 0xe062 },
	{ 0xf812, 0xe064 }, { 0xf814, 0xe066 }, { 0xf816, 0xe068 },
	{ 0xf818, 0xe06a }, { 0xf81a, 0xe06c }, { 0xf81c, 0xe06e },
	{ 0xf81e, 0xe070 }, { 0xf820, 0x740a }, { 0xf822, 0x4846 },
	{ 0xf824, 0x4847 }, { 0xf826, 0x9c0a }, { 0xf828, 0xc607 },
	{ 0xf82a, 0x74c0 }, { 0xf82c, 0x48c6 }, { 0xf82e, 0x9cc0 },
	{ 0xf830, 0xc602 }, { 0xf832, 0xbe00 }, { 0xf834, 0x13f0 },
	{ 0xf836, 0xe054 }, { 0xf838, 0x72ca }, { 0xf83a, 0x4826 },
	{ 0xf83c, 0x4827 }, { 0xf83e, 0x9aca }, { 0xf840, 0xc607 },
	{ 0xf842, 0x72c0 }, { 0xf844, 0x48a6 }, { 0xf846, 0x9ac0 },
	{ 0xf848, 0xc602 }, { 0xf84a, 0xbe00 }, { 0xf84c, 0x081c },
	{ 0xf84e, 0xe054 }, { 0xf850, 0xc60f }, { 0xf852, 0x74c4 },
	{ 0xf854, 0x49cc }, { 0xf856, 0xf109 }, { 0xf858, 0xc60c },
	{ 0xf85a, 0x74ca }, { 0xf85c, 0x48c7 }, { 0xf85e, 0x9cca },
	{ 0xf860, 0xc609 }, { 0xf862, 0x74c0 }, { 0xf864, 0x4846 },
	{ 0xf866, 0x9cc0 }, { 0xf868, 0xc602 }, { 0xf86a, 0xbe00 },
	{ 0xf86c, 0x2494 }, { 0xf86e, 0xe092 }, { 0xf870, 0xe0c0 },
	{ 0xf872, 0xe054 }, { 0xf874, 0x7420 }, { 0xf876, 0x48c0 },
	{ 0xf878, 0x9c20 }, { 0xf87a, 0x7444 }, { 0xf87c, 0xc602 },
	{ 0xf87e, 0xbe00 }, { 0xf880, 0x12dc }, { 0xf882, 0x733a },
	{ 0xf884, 0x21b5 }, { 0xf886, 0x25bc }, { 0xf888, 0x1304 },
	{ 0xf88a, 0xf111 }, { 0xf88c, 0x1b12 }, { 0xf88e, 0x1d2a },
	{ 0xf890, 0x3168 }, { 0xf892, 0x3ada }, { 0xf894, 0x31ab },
	{ 0xf896, 0x1a00 }, { 0xf898, 0x9ac0 }, { 0xf89a, 0x1300 },
	{ 0xf89c, 0xf1fb }, { 0xf89e, 0x7620 }, { 0xf8a0, 0x236e },
	{ 0xf8a2, 0x276f }, { 0xf8a4, 0x1a3c }, { 0xf8a6, 0x22a1 },
	{ 0xf8a8, 0x41b5 }, { 0xf8aa, 0x9ee2 }, { 0xf8ac, 0x76e4 },
	{ 0xf8ae, 0x486f }, { 0xf8b0, 0x9ee4 }, { 0xf8b2, 0xc602 },
	{ 0xf8b4, 0xbe00 }, { 0xf8b6, 0x4a26 }, { 0xf8b8, 0x733a },
	{ 0xf8ba, 0x49bb }, { 0xf8bc, 0xc602 }, { 0xf8be, 0xbe00 },
	{ 0xf8c0, 0x47a2 }, { 0xf8c2, 0x48c1 }, { 0xf8c4, 0x48c2 },
	{ 0xf8c6, 0x9c46 }, { 0xf8c8, 0xc402 }, { 0xf8ca, 0xbc00 },
	{ 0xf8cc, 0x0a52 }, { 0xf8ce, 0xc602 }, { 0xf8d0, 0xbe00 },
	{ 0xf8d2, 0x0000 }, { 0xf8d4, 0xc602 }, { 0xf8d6, 0xbe00 },
	{ 0xf8d8, 0x0000 }, { 0xf8da, 0xc602 }, { 0xf8dc, 0xbe00 },
	{ 0xf8de, 0x0000 }, { 0xf8e0, 0xc602 }, { 0xf8e2, 0xbe00 },
	{ 0xf8e4, 0x0000 }, { 0xf8e6, 0xc602 }, { 0xf8e8, 0xbe00 },
	{ 0xf8ea, 0x0000 }, { 0xf8ec, 0xc602 }, { 0xf8ee, 0xbe00 },
	{ 0xf8f0, 0x0000 }, { 0xf8f2, 0xc602 }, { 0xf8f4, 0xbe00 },
	{ 0xf8f6, 0x0000 }, { 0xf8f8, 0xc602 }, { 0xf8fa, 0xbe00 },
	{ 0xf8fc, 0x0000 }, { 0xf8fe, 0xc602 }, { 0xf900, 0xbe00 },
	{ 0xf902, 0x0000 }, { 0xfc26, 0x8000 }, { 0xfc28, 0x13e6 },
	{ 0xfc2a, 0x0812 }, { 0xfc2c, 0x248c }, { 0xfc2e, 0x12da },
	{ 0xfc30, 0x4a20 }, { 0xfc32, 0x47a0 }, { 0xfc48, 0x003f }
};

struct rge_mac_bps {
	int		count;
	const uint16_t	*regs;
	const uint16_t	*vals;
};

static const uint16_t rtl8126_1_mac_bps_regs[] = {
	0xf800, 0xf802, 0xf804, 0xf806, 0xf808, 0xf80a, 0xf80c,
	0xf80e, 0xf810, 0xf812, 0xf814, 0xf816, 0xf818, 0xf81a,
	0xf81c, 0xf81e, 0xf820, 0xf822, 0xf824, 0xf826, 0xf828,
	0xf82a, 0xf82c, 0xf82e, 0xf830, 0xf832, 0xf834, 0xf836,
	0xf838, 0xf83a, 0xf83c, 0xf83e, 0xf840, 0xf842, 0xf844,
	0xf846, 0xf848, 0xf84a, 0xf84c, 0xf84e, 0xf850, 0xf852,
	0xf854, 0xf856, 0xf858, 0xf85a, 0xf85c, 0xf85e, 0xf860,
	0xf862, 0xf864, 0xf866, 0xf868, 0xf86a, 0xf86c, 0xf86e,
	0xf870, 0xf872, 0xf874, 0xf876, 0xf878, 0xf87a, 0xf87c,
	0xf87e, 0xf880, 0xf882, 0xf884, 0xf886, 0xf888, 0xf88a,
	0xf88c, 0xf88e, 0xf890, 0xf892, 0xf894, 0xf896, 0xf898,
	0xf89a, 0xf89c, 0xf89e, 0xf8a0, 0xf8a2, 0xf8a4, 0xf8a6,
	0xf8a8, 0xf8aa, 0xf8ac, 0xf8ae, 0xf8b0, 0xf8b2, 0xf8b4,
	0xf8b6, 0xf8b8, 0xf8ba, 0xf8bc, 0xf8be, 0xf8c0, 0xf8c2,
	0xf8c4, 0xf8c6, 0xf8c8, 0xf8ca, 0xf8cc, 0xf8ce, 0xf8d0,
	0xf8d2, 0xf8d4, 0xf8d6, 0xf8d8, 0xf8da, 0xf8dc, 0xf8de,
	0xf8e0, 0xf8e2, 0xf8e4, 0xf8e6, 0xf8e8, 0xf8ea, 0xf8ec,
	0xf8ee, 0xf8f0, 0xf8f2, 0xf8f4, 0xf8f6, 0xf8f8, 0xf8fa,
	0xf8fc, 0xf8fe, 0xf900, 0xf902, 0xf904, 0xf906, 0xf908,
	0xf90a, 0xf90c, 0xf90e, 0xf910, 0xf912, 0xf914, 0xf916,
	0xf918, 0xf91a, 0xf91c, 0xf91e, 0xf920, 0xf922, 0xf924,
	0xf926, 0xf928, 0xf92a, 0xf92c, 0xf92e, 0xf930, 0xf932,
	0xf934, 0xf936, 0xf938, 0xf93a, 0xf93c, 0xf93e, 0xf940,
	0xf942, 0xf944, 0xf946, 0xf948, 0xf94a, 0xf94c, 0xf94e,
	0xf950, 0xf952, 0xf954, 0xf956, 0xf958, 0xf95a, 0xf95c,
	0xf95e, 0xf960, 0xf962, 0xf964, 0xf966, 0xf968, 0xf96a,
	0xf96c, 0xf96e, 0xf970, 0xf972, 0xf974, 0xf976, 0xf978,
	0xf97a, 0xf97c, 0xf97e, 0xf980, 0xf982, 0xf984, 0xf986,
	0xf988, 0xf98a, 0xf98c, 0xf98e, 0xf990, 0xf992, 0xf994,
	0xf996, 0xf998, 0xf99a, 0xf99c, 0xf99e, 0xf9a0, 0xf9a2,
	0xf9a4, 0xf9a6, 0xf9a8, 0xf9aa, 0xf9ac, 0xf9ae, 0xf9b0
};

static const uint16_t rtl8126_1_mac_bps_vals[] = {
	0xe010, 0xe02c, 0xe04e, 0xe0a4, 0xe0a8, 0xe0ab, 0xe0ae,
	0xe0b1, 0xe0b5, 0xe0b7, 0xe0b9, 0xe0bb, 0xe0bd, 0xe0bf,
	0xe0c1, 0xe0c3, 0xc716, 0xc616, 0x9ee0, 0xc616, 0x65c0,
	0x1500, 0xf009, 0xc714, 0x66e0, 0x41b5, 0x8ee0, 0xc611,
	0x75c0, 0x4858, 0x9dc0, 0xc707, 0xc608, 0x9ee0, 0xc608,
	0xc502, 0xbd00, 0x0100, 0xe86c, 0xe000, 0xa000, 0xb404,
	0xb430, 0xc070, 0xe926, 0xc2fe, 0x400a, 0xf11a, 0x63a4,
	0x1a00, 0x49b0, 0xf002, 0x4820, 0x49b1, 0xf002, 0x4821,
	0x49b2, 0xf002, 0x4822, 0x49b3, 0xf002, 0x4823, 0xc411,
	0x6380, 0x48b0, 0x8b80, 0x6320, 0x41da, 0x8b20, 0x6380,
	0x4830, 0x8b80, 0xe003, 0x73a4, 0x9b20, 0xc302, 0xbb00,
	0x4a18, 0xc070, 0xe022, 0xc054, 0x7102, 0x4992, 0xf149,
	0x4893, 0x9902, 0x1b1f, 0xc74e, 0x72e0, 0x2521, 0x48a5,
	0x0b01, 0x1c4f, 0x9c00, 0x2121, 0x1d01, 0x41aa, 0x2521,
	0x9de0, 0x4856, 0x9de0, 0x1ccf, 0xe839, 0x48d6, 0x9de0,
	0x7102, 0x4996, 0xf1fe, 0x4814, 0x9902, 0x1cff, 0x0c01,
	0x1400, 0xf00c, 0x7102, 0x4996, 0xf0fb, 0x7102, 0x4990,
	0xf0fe, 0x1c1f, 0xe826, 0x7102, 0x4992, 0xf004, 0x4813,
	0x9902, 0xe01d, 0x1300, 0xf104, 0x4817, 0x9902, 0xe018,
	0x4894, 0x9902, 0x4995, 0xf00b, 0x121f, 0xf0f3, 0x131e,
	0xf003, 0x4998, 0xf0ef, 0x0201, 0x4818, 0x9902, 0xe7c9,
	0x1200, 0xf0e9, 0x4998, 0xf002, 0x1b01, 0x0a01, 0x4898,
	0x9902, 0xe7c0, 0xc00a, 0xc606, 0xbe00, 0x0c01, 0x1400,
	0xf1fe, 0xff80, 0x2362, 0xd456, 0xd404, 0xe400, 0x4166,
	0x9cf6, 0xc002, 0xb800, 0x14a6, 0x49d1, 0xc602, 0xbe00,
	0x4160, 0x49d1, 0xc602, 0xbe00, 0x41e6, 0x49d1, 0xc602,
	0xbe00, 0x4282, 0xc104, 0xc202, 0xba00, 0x234c, 0xd116,
	0xc602, 0xbe00, 0x0000, 0xc602, 0xbe00, 0x0000, 0xc602,
	0xbe00, 0x0000, 0xc602, 0xbe00, 0x0000, 0xc602, 0xbe00,
	0x0000, 0xc602, 0xbe00, 0x0000, 0xc602, 0xbe00, 0x0000,
	0xc602, 0xbe00, 0x0000, 0x6847, 0x0019, 0x041c, 0x1322
};

static const struct rge_mac_bps rtl8126_1_mac_bps = {
	nitems(rtl8126_1_mac_bps_regs),
	rtl8126_1_mac_bps_regs,
	rtl8126_1_mac_bps_vals
};

static const uint16_t rtl8126_2_mac_bps_regs[] = {
	0xf800, 0xf802, 0xf804, 0xf806, 0xf808, 0xf80a, 0xf80c,
	0xf80e, 0xf810, 0xf812, 0xf814, 0xf816, 0xf818, 0xf81a,
	0xf81c, 0xf81e, 0xf820, 0xf822, 0xf824, 0xf826, 0xf828,
	0xf82a, 0xf82c, 0xf82e, 0xf830, 0xf832, 0xf834, 0xf836,
	0xf838, 0xf83a, 0xf83c, 0xf83e, 0xf840, 0xf842, 0xf844,
	0xf846, 0xf848, 0xf84a, 0xf84c, 0xf84e, 0xf850, 0xf852,
	0xf854, 0xf856, 0xf858, 0xf85a, 0xf85c, 0xf85e, 0xf860,
	0xf862, 0xf864, 0xf866, 0xf868, 0xf86a, 0xf86c, 0xf86e,
	0xf870, 0xf872, 0xf874, 0xf876, 0xf878, 0xf87a, 0xf87c,
	0xf87e, 0xf880, 0xf882, 0xf884, 0xf886, 0xf888, 0xf88a,
	0xf88c, 0xf88e, 0xf890, 0xf892, 0xf894, 0xf896, 0xf898,
	0xf89a, 0xf89c, 0xf89e, 0xf8a0, 0xf8a2, 0xf8a4, 0xf8a6,
	0xf8a8, 0xf8aa, 0xf8ac, 0xf8ae, 0xf8b0, 0xf8b2, 0xf8b4,
	0xf8b6, 0xf8b8, 0xf8ba, 0xf8bc, 0xf8be, 0xf8c0, 0xf8c2,
	0xf8c4, 0xf8c6, 0xf8c8, 0xf8ca, 0xf8cc, 0xf8ce, 0xf8d0,
	0xf8d2, 0xf8d4, 0xf8d6, 0xf8d8, 0xf8da, 0xf8dc, 0xf8de,
	0xf8e0, 0xf8e2, 0xf8e4, 0xf8e6, 0xf8e8, 0xf8ea, 0xf8ec,
	0xf8ee, 0xf8f0, 0xf8f2, 0xf8f4, 0xf8f6, 0xf8f8, 0xf8fa,
	0xf8fc, 0xf8fe, 0xf900, 0xf902, 0xf904, 0xf906, 0xf908,
	0xf90a, 0xf90c, 0xf90e, 0xf910, 0xf912, 0xf914, 0xf916,
	0xf918, 0xf91a, 0xf91c, 0xf91e, 0xf920, 0xf922, 0xf924,
	0xf926, 0xf928, 0xf92a, 0xf92c, 0xf92e, 0xf930, 0xf932,
	0xf934, 0xf936, 0xf938, 0xf93a, 0xf93c, 0xf93e, 0xf940,
	0xf942, 0xf944, 0xf946, 0xf948, 0xf94a, 0xf94c, 0xf94e,
	0xf950, 0xf952, 0xf954, 0xf956, 0xf958, 0xf95a, 0xf95c,
	0xf95e, 0xf960, 0xf962, 0xf964, 0xf966, 0xf968, 0xf96a,
	0xf96c, 0xf96e, 0xf970, 0xf972, 0xf974, 0xf976, 0xf978,
	0xf97a, 0xf97c, 0xf97e, 0xf980, 0xf982, 0xf984, 0xf986,
	0xf988, 0xf98a, 0xf98c, 0xf98e, 0xf990, 0xf992, 0xf994,
	0xf996, 0xf998, 0xf99a, 0xf99c, 0xf99e, 0xf9a0, 0xf9a2,
	0xf9a4, 0xf9a6, 0xf9a8, 0xf9aa, 0xf9ac, 0xf9ae, 0xf9b0,
	0xf9b2, 0xf9b4, 0xf9b6, 0xf9b8, 0xf9ba, 0xf9bc, 0xf9be,
	0xf9c0, 0xf9c2, 0xf9c4, 0xf9c6, 0xf9c8, 0xf9ca, 0xf9cc,
	0xf9ce, 0xf9d0, 0xf9d2, 0xf9d4, 0xf9d6, 0xf9d8, 0xf9da,
	0xf9dc, 0xf9de, 0xf9e0, 0xf9e2, 0xf9e4, 0xf9e6, 0xf9e8,
	0xf9ea, 0xf9ec, 0xf9ee, 0xf9f0, 0xf9f2, 0xf9f4, 0xf9f6,
	0xf9f8, 0xf9fa, 0xf9fc, 0xf9fe, 0xf800, 0xf802, 0xf804,
	0xf806, 0xf808, 0xf80a, 0xf80c, 0xf80e, 0xf810, 0xf812,
	0xf814, 0xf816, 0xf818, 0xf81a, 0xf81c, 0xf81e, 0xf820,
	0xf822, 0xf824, 0xf826, 0xf828, 0xf82a, 0xf82c, 0xf82e,
	0xf830, 0xf832, 0xf834, 0xf836, 0xf838, 0xf83a, 0xf83c,
	0xf83e, 0xf840, 0xf842, 0xf844, 0xf846, 0xf848, 0xf84a,
	0xf84c, 0xf84e, 0xf850, 0xf852, 0xf854, 0xf856, 0xf858,
	0xf85a, 0xf85c, 0xf85e, 0xf860, 0xf862, 0xf864, 0xf866,
	0xf868, 0xf86a, 0xf86c, 0xf86e, 0xf870, 0xf872, 0xf874,
	0xf876, 0xf878, 0xf87a, 0xf87c, 0xf87e, 0xf880, 0xf882,
	0xf884, 0xf886, 0xf888, 0xf88a, 0xf88c, 0xf88e, 0xf890,
	0xf892, 0xf894, 0xf896, 0xf898, 0xf89a, 0xf89c, 0xf89e,
	0xf8a0, 0xf8a2, 0xf8a4
};

static const uint16_t rtl8126_2_mac_bps_vals[] = {
	0xe00a, 0xe026, 0xe048, 0xe04c, 0xe04f, 0xe052, 0xe055,
	0xe0ab, 0xe0f8, 0xe141, 0xc716, 0xc616, 0x9ee0, 0xc616,
	0x65c0, 0x1500, 0xf009, 0xc714, 0x66e0, 0x41b5, 0x8ee0,
	0xc611, 0x75c0, 0x4858, 0x9dc0, 0xc707, 0xc608, 0x9ee0,
	0xc608, 0xc502, 0xbd00, 0x0100, 0xe86c, 0xe000, 0xa000,
	0xb404, 0xb430, 0xc070, 0xe926, 0xc2fe, 0x400a, 0xf11a,
	0x63a4, 0x1a00, 0x49b0, 0xf002, 0x4820, 0x49b1, 0xf002,
	0x4821, 0x49b2, 0xf002, 0x4822, 0x49b3, 0xf002, 0x4823,
	0xc411, 0x6380, 0x48b0, 0x8b80, 0x6320, 0x41da, 0x8b20,
	0x6380, 0x4830, 0x8b80, 0xe003, 0x73a4, 0x9b20, 0xc302,
	0xbb00, 0x55e2, 0xc070, 0xe022, 0x4166, 0x9cf6, 0xc602,
	0xbe00, 0x14a6, 0x49d1, 0xc602, 0xbe00, 0x4178, 0x49d1,
	0xc602, 0xbe00, 0x41fe, 0x49d1, 0xc602, 0xbe00, 0x429a,
	0xc74b, 0x76e2, 0xc54a, 0x402e, 0xf034, 0x76e0, 0x402e,
	0xf006, 0xc503, 0xc403, 0xbc00, 0xc0bc, 0x0aae, 0x76f0,
	0x1601, 0xf023, 0xc741, 0x1e04, 0x9ee0, 0x1e40, 0x9ee4,
	0xc63d, 0x9ee8, 0xc73d, 0x76e0, 0x4863, 0x9ee0, 0xc73a,
	0x76e0, 0x48ea, 0x48eb, 0x9ee0, 0xc736, 0x1e01, 0x9ee2,
	0xc72d, 0x76e0, 0x486f, 0x9ee0, 0xc72d, 0x76e0, 0x48e3,
	0x9ee0, 0xc728, 0x1e0e, 0x9ee0, 0xc71d, 0x1e01, 0x9ee4,
	0xe00d, 0x1e00, 0x9ef0, 0x1e05, 0xc715, 0x9ee0, 0xe00a,
	0x1e00, 0x9ee2, 0xc614, 0x75cc, 0x48d2, 0x9dcc, 0x1e04,
	0xc70b, 0x9ee0, 0xb000, 0xb001, 0xb002, 0xb003, 0xb004,
	0xb005, 0xb006, 0xb007, 0xffc0, 0xe428, 0xd3c0, 0xbeef,
	0x5b4a, 0xdc46, 0xe0cc, 0xe84e, 0xc0a2, 0x0100, 0xc010,
	0xe85a, 0xe812, 0xc0b4, 0xc5f4, 0x74a0, 0xc6f3, 0x4026,
	0xf107, 0x74a2, 0xc6ef, 0x4026, 0xf107, 0xc6ed, 0xbe00,
	0x753a, 0xc602, 0xbe00, 0x5a3a, 0x7520, 0x49de, 0xf102,
	0xe7f9, 0xc6a1, 0x67c6, 0x7520, 0x22d2, 0x26dd, 0x1500,
	0xf002, 0xe7f1, 0x7532, 0x26d5, 0x0530, 0x0d6c, 0xc42d,
	0x308d, 0x7540, 0x4025, 0xf11e, 0x7542, 0x4025, 0xf11b,
	0x7544, 0x4025, 0xf118, 0xc423, 0x7546, 0x4025, 0xf114,
	0x7548, 0x4025, 0xf111, 0x754a, 0x4025, 0xf10e, 0xc5c0,
	0xc4c0, 0x9ca2, 0xc6c0, 0x75cc, 0x4852, 0x9dcc, 0xc6b8,
	0x1d7d, 0x9dc2, 0x1d01, 0x9dc0, 0xe7c9, 0xc40b, 0x7546,
	0x4025, 0xf1fc, 0x7548, 0x4025, 0xf1f9, 0x754a, 0x4025,
	0xf1f6, 0xe7c0, 0xffff, 0xeeee, 0xc2a6, 0x7340, 0xc2a5,
	0x4013, 0xf013, 0xc2ac, 0x7340, 0x4835, 0x9b40, 0xc240,
	0x7358, 0x48b7, 0x48b2, 0x9b58, 0x7346, 0x48b7, 0x48b2,
	0x9b46, 0x7340, 0x48b7, 0x48b2, 0x9b40, 0xe012, 0xc29a,
	0x7340, 0x48b5, 0x9b40, 0xc22e, 0x7358, 0x4837, 0x4832,
	0x9b58, 0x7346, 0x4837, 0x4832, 0x9b46, 0x7340, 0x4837,
	0x4832, 0x9b40, 0xc283, 0x7340, 0x49bf, 0xf010, 0xc21b,
	0x7344, 0x1300, 0xf104, 0x1b00, 0xc217, 0x9b40, 0x1b01,
	0xc213, 0x9b44, 0xc213, 0x734c, 0x48b7, 0x9b4c, 0xe008,
	0xc20c, 0x1b00, 0x9b44, 0xc20b, 0x734c, 0x4837, 0x9b4c,
	0xc204, 0xc302, 0xbb00, 0x2384, 0xe092, 0xd3c0, 0xe428,
	0xdc46, 0xc104, 0xc202, 0xba00, 0x234c, 0xd116, 0x6847,
	0x0119, 0x041c, 0x1134
};

static const struct rge_mac_bps rtl8126_2_mac_bps = {
	nitems(rtl8126_2_mac_bps_regs),
	rtl8126_2_mac_bps_regs,
	rtl8126_2_mac_bps_vals
};

static const uint16_t rtl8127_mac_bps_regs[] = {
	0xf800, 0xf802, 0xf804, 0xf806, 0xf808, 0xf80a, 0xf80c,
	0xf80e, 0xf810, 0xf812, 0xf814, 0xf816, 0xf818, 0xf81a,
	0xf81c, 0xf81e, 0xf820, 0xf822, 0xf824, 0xf826, 0xf828,
	0xf82a, 0xf82c, 0xf82e, 0xf830, 0xf832, 0xf834, 0xf836,
	0xf838, 0xf83a, 0xf83c, 0xf83e, 0xf840, 0xf842, 0xf844,
	0xf846, 0xf848, 0xf84a, 0xf84c, 0xf84e, 0xf850, 0xf852,
	0xf854, 0xf856, 0xf858, 0xf85a, 0xf85c, 0xf85e, 0xf860,
	0xf862, 0xf864, 0xf866, 0xf868, 0xf86a, 0xf86c, 0xf86e,
	0xf870, 0xf872, 0xf874, 0xf876, 0xf878, 0xf87a, 0xf87c,
	0xf87e, 0xf880, 0xf882, 0xf884, 0xf886, 0xf888, 0xf88a,
	0xf88c, 0xf88e, 0xf890, 0xf892, 0xf894, 0xf896, 0xf898,
	0xf89a, 0xf89c, 0xf89e, 0xf8a0, 0xf8a2, 0xf8a4, 0xf8a6,
	0xf8a8, 0xf8aa, 0xf8ac, 0xf8ae, 0xf8b0, 0xf8b2, 0xf8b4,
	0xf8b6, 0xf8b8, 0xf8ba, 0xf8bc, 0xf8be, 0xf8c0, 0xf8c2,
	0xf8c4, 0xf8c6, 0xf8c8, 0xf8ca, 0xf8cc, 0xf8ce, 0xf8d0,
	0xf8d2, 0xf8d4, 0xf8d6, 0xf8d8, 0xf8da, 0xf8dc, 0xf8de,
	0xf8e0, 0xf8e2, 0xf8e4, 0xf8e6, 0xf8e8, 0xf8ea, 0xf8ec,
	0xf8ee, 0xf8f0, 0xf8f2, 0xf8f4, 0xf8f6, 0xf8f8, 0xf8fa,
	0xf8fc, 0xf8fe, 0xf900, 0xf902, 0xf904, 0xf906, 0xf908,
	0xf90a, 0xf90c, 0xf90e, 0xf910, 0xf912, 0xf914, 0xf916,
	0xf918, 0xf91a, 0xf91c, 0xf91e, 0xf920, 0xf922, 0xf924,
	0xf926, 0xf928, 0xf92a, 0xf92c, 0xf92e, 0xf930, 0xf932,
	0xf934, 0xf936, 0xf938, 0xf93a, 0xf93c, 0xf93e, 0xf940,
	0xf942, 0xf944, 0xf946, 0xf948, 0xf94a, 0xf94c, 0xf94e,
	0xf950, 0xf952, 0xf954, 0xf956, 0xf958, 0xf95a, 0xf95c,
	0xf95e, 0xf960, 0xf962, 0xf964, 0xf966, 0xf968, 0xf96a,
	0xf96c, 0xf96e, 0xf970, 0xf972, 0xf974, 0xf976, 0xf978,
	0xf97a, 0xf97c, 0xf97e, 0xf980, 0xf982, 0xf984, 0xf986,
	0xf988, 0xf98a, 0xf98c, 0xf98e, 0xf990, 0xf992, 0xf994,
	0xf996, 0xf998, 0xf99a, 0xf99c, 0xf99e, 0xf9a0, 0xf9a2,
	0xf9a4, 0xf9a6, 0xf9a8, 0xf9aa, 0xf9ac, 0xf9ae, 0xf9b0,
	0xf9b2, 0xf9b4, 0xf9b6, 0xf9b8, 0xf9ba, 0xf9bc, 0xf9be,
	0xf9c0, 0xf9c2, 0xf9c4, 0xf9c6, 0xf9c8, 0xf9ca, 0xf9cc,
	0xf9ce, 0xf9d0, 0xf9d2, 0xf9d4, 0xf9d6, 0xf9d8, 0xf9da,
	0xf9dc, 0xf9de, 0xf9e0, 0xf9e2, 0xf9e4, 0xf9e6, 0xf9e8,
	0xf9ea, 0xf9ec, 0xf9ee, 0xf9f0, 0xf9f2, 0xf9f4, 0xf9f6,
	0xf9f8, 0xf9fa, 0xf9fc, 0xf9fe, 0xf800, 0xf802, 0xf804,
	0xf806, 0xf808, 0xf80a, 0xf80c, 0xf80e, 0xf810, 0xf812,
	0xf814, 0xf816, 0xf818, 0xf81a, 0xf81c, 0xf81e, 0xf820,
	0xf822, 0xf824, 0xf826, 0xf828, 0xf82a, 0xf82c, 0xf82e,
	0xf830, 0xf832, 0xf834, 0xf836, 0xf838, 0xf83a, 0xf83c,
	0xf83e, 0xf840, 0xf842, 0xf844, 0xf846, 0xf848, 0xf84a,
	0xf84c, 0xf84e, 0xf850, 0xf852, 0xf854, 0xf856, 0xf858,
	0xf85a, 0xf85c, 0xf85e, 0xf860, 0xf862, 0xf864, 0xf866,
	0xf868, 0xf86a, 0xf86c, 0xf86e, 0xf870, 0xf872, 0xf874,
	0xf876, 0xf878, 0xf87a, 0xf87c
};

static const uint16_t rtl8127_mac_bps_vals[] = {
	0xe010, 0xe023, 0xe036, 0xe049, 0xe05c, 0xe075, 0xe0b1,
	0xe117, 0xe11b, 0xe11d, 0xe11f, 0xe121, 0xe123, 0xe125,
	0xe127, 0xe129, 0x7020, 0xb405, 0xb404, 0xc50f, 0x74a0,
	0xc50e, 0x4025, 0xf005, 0x4850, 0x4025, 0xf002, 0xe002,
	0x4809, 0xb004, 0xb005, 0xc502, 0xbd00, 0x1522, 0xd006,
	0x0004, 0x7760, 0xb405, 0xb404, 0xc50f, 0x74a0, 0xc50e,
	0x4025, 0xf005, 0x4850, 0x4025, 0xf002, 0xe002, 0x4879,
	0xb004, 0xb005, 0xc002, 0xb800, 0x41e2, 0xd006, 0x0004,
	0x7160, 0xb405, 0xb404, 0xc50f, 0x74a0, 0xc50e, 0x4025,
	0xf005, 0x4850, 0x4025, 0xf002, 0xe002, 0x4819, 0xb004,
	0xb005, 0xc302, 0xbb00, 0x508e, 0xd006, 0x0004, 0x7720,
	0xb405, 0xb404, 0xc50f, 0x74a0, 0xc50e, 0x4025, 0xf005,
	0x4850, 0x4025, 0xf002, 0xe002, 0x4879, 0xb004, 0xb005,
	0xc102, 0xb900, 0x50f8, 0xd006, 0x0004, 0x61a9, 0xb403,
	0xb404, 0xc313, 0x7460, 0xc312, 0x4023, 0xf005, 0x4830,
	0x4023, 0xf002, 0xe003, 0x4997, 0xf003, 0xc00a, 0xe002,
	0xc009, 0xb004, 0xb003, 0xc102, 0xb900, 0x34fc, 0xd006,
	0x0004, 0x02af, 0x041f, 0xb407, 0xb406, 0xb405, 0xb404,
	0xb403, 0xb402, 0xb401, 0xb400, 0x49d2, 0xf116, 0xc62f,
	0x77c0, 0x49f9, 0xf020, 0x49fa, 0xf11e, 0x49f2, 0xf102,
	0xe01b, 0x48f2, 0x9fc0, 0xc625, 0x75c2, 0x4852, 0x9dc2,
	0xc122, 0x7020, 0x4801, 0x4802, 0x9820, 0xe00f, 0xc61a,
	0x77c0, 0x49f2, 0xf10b, 0xc618, 0x77c0, 0x49f2, 0xf007,
	0x48f2, 0x9fc0, 0xc611, 0x75c2, 0x48d2, 0x9dc2, 0xb000,
	0xb001, 0xb002, 0xb003, 0xb004, 0xb005, 0xb006, 0xb007,
	0x9d6c, 0xc502, 0xbd00, 0x0168, 0xe024, 0xc010, 0xd410,
	0xd460, 0xb407, 0xb406, 0xb405, 0xb404, 0xb403, 0xb402,
	0xb401, 0xb400, 0xc152, 0x7020, 0x4981, 0xf043, 0xc050,
	0x7100, 0xb401, 0xc14c, 0x489e, 0x481d, 0x9900, 0xc24a,
	0x7340, 0x49b7, 0xf135, 0xc144, 0x9900, 0xc245, 0x7340,
	0xc447, 0x401c, 0xf109, 0x734c, 0x1301, 0xf12b, 0xc343,
	0x9b40, 0xc33e, 0x9b40, 0xe022, 0xc63a, 0x77c0, 0x48f4,
	0x48f5, 0x48f9, 0x48fa, 0x9fc0, 0xc231, 0x7344, 0x48b4,
	0x9b44, 0xc22e, 0x7340, 0x4830, 0x48b1, 0x4832, 0x483c,
	0x48bd, 0x48be, 0x48bf, 0x9b40, 0xc223, 0xc32a, 0x9b48,
	0xc327, 0x9b46, 0xc324, 0x9b40, 0xc321, 0x9b42, 0xc31e,
	0x9b40, 0xe005, 0xc113, 0x7020, 0x4881, 0x9820, 0xb001,
	0xc010, 0x9900, 0xb000, 0xb001, 0xb002, 0xb003, 0xb004,
	0xb005, 0xb006, 0xb007, 0x2125, 0xc102, 0xb900, 0x1a6c,
	0xd410, 0xc000, 0xe86c, 0xb600, 0xb800, 0xb40a, 0xe024,
	0x5a00, 0x5a81, 0x0073, 0x5a80, 0x0042, 0x0001, 0xc104,
	0xc202, 0xba00, 0x1a2e, 0xc896, 0xc302, 0xbb00, 0x0000,
	0xc002, 0xb800, 0x0000, 0xc002, 0xb800, 0x0000, 0xc502,
	0xbd00, 0x0000, 0xc102, 0xb900, 0x0000, 0xc102, 0xb900,
	0x0000, 0xc602, 0xbe00, 0x0000, 0xc602, 0xbe00, 0x0000,
	0x6961, 0x0019, 0x050c, 0x140c
};

static const struct rge_mac_bps rtl8127_mac_bps = {
	nitems(rtl8127_mac_bps_regs),
	rtl8127_mac_bps_regs,
	rtl8127_mac_bps_vals
};

static const struct {
	uint16_t	reg;
	uint16_t	val;
} mac_r25_ephy[] = {
	{ 0x0004, 0xd000 }, { 0x000a, 0x8653 }, { 0x0023, 0xab66 },
	{ 0x0020, 0x9455 }, { 0x0021, 0x99ff }, { 0x0029, 0xfe04 },
	{ 0x0044, 0xd000 }, { 0x004a, 0x8653 }, { 0x0063, 0xab66 },
	{ 0x0060, 0x9455 }, { 0x0061, 0x99ff }, { 0x0069, 0xfe04 }
}, mac_r25b_ephy[] = {
	{ 0x000b, 0xa908 }, { 0x001e, 0x20eb }, { 0x0022, 0x0023 },
	{ 0x0002, 0x60c2 }, { 0x0029, 0xff00 }, { 0x004b, 0xa908 },
	{ 0x005e, 0x28eb }, { 0x0062, 0x0023 }, { 0x0042, 0x60c2 },
	{ 0x0069, 0xff00 }
}, mac_r27_ephy[] = {
	{ 0x8088, 0x0064 }, { 0x8488, 0x0064 }, { 0x8888, 0x0064 },
	{ 0x8c88, 0x0064 }, { 0x8188, 0x0064 }, { 0x8588, 0x0064 },
	{ 0x8988, 0x0064 }, { 0x8d88, 0x0064 }, { 0x808c, 0x09b0 },
	{ 0x848c, 0x09b0 }, { 0x888c, 0x0f90 }, { 0x8c8c, 0x0f90 },
	{ 0x818c, 0x09b0 }, { 0x858c, 0x09b0 }, { 0x898c, 0x0f90 },
	{ 0x8d8c, 0x0f90 }, { 0x808a, 0x09b8 }, { 0x848a, 0x09b8 },
	{ 0x888a, 0x0f98 }, { 0x8c8a, 0x0f98 }, { 0x818a, 0x09b8 },
	{ 0x858a, 0x09b8 }, { 0x898a, 0x0f98 }, { 0x8d8a, 0x0f98 },
	{ 0x9020, 0x0080 }, { 0x9420, 0x0080 }, { 0x9820, 0x0080 },
	{ 0x9c20, 0x0080 }, { 0x901e, 0x0190 }, { 0x941e, 0x0190 },
	{ 0x981e, 0x0140 }, { 0x9c1e, 0x0140 }, { 0x901c, 0x0190 },
	{ 0x941c, 0x0190 }, { 0x981c, 0x0140 }, { 0x9c1c, 0x0140 }
};

static const struct {
	uint16_t        reg;
	uint16_t        val;
} mac_r26_1_mcu[] = {
	{ 0xa436, 0x8023 }, { 0xa438, 0x4700 }, { 0xa436, 0xb82e },
	{ 0xa438, 0x0001 }, { 0xb820, 0x0090 }, { 0xa436, 0xa016 },
	{ 0xa438, 0x0000 }, { 0xa436, 0xa012 }, { 0xa438, 0x0000 },
	{ 0xa436, 0xa014 }, { 0xa438, 0x1800 }, { 0xa438, 0x8010 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8025 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8033 }, { 0xa438, 0x1800 }, { 0xa438, 0x8037 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x803c }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8044 }, { 0xa438, 0x1800 }, { 0xa438, 0x8054 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8059 }, { 0xa438, 0xd504 },
	{ 0xa438, 0xc9b5 }, { 0xa438, 0xd500 }, { 0xa438, 0xd707 },
	{ 0xa438, 0x4070 }, { 0xa438, 0x1800 }, { 0xa438, 0x107a },
	{ 0xa438, 0xd504 }, { 0xa438, 0xc994 }, { 0xa438, 0xd500 },
	{ 0xa438, 0xd707 }, { 0xa438, 0x60d0 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x252d }, { 0xa438, 0x8023 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x1064 }, { 0xa438, 0x1800 }, { 0xa438, 0x107a },
	{ 0xa438, 0x1800 }, { 0xa438, 0x1052 }, { 0xa438, 0xd504 },
	{ 0xa438, 0xc9d0 }, { 0xa438, 0xd500 }, { 0xa438, 0xd707 },
	{ 0xa438, 0x60d0 }, { 0xa438, 0xd701 }, { 0xa438, 0x252d },
	{ 0xa438, 0x8031 }, { 0xa438, 0x1800 }, { 0xa438, 0x1171 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x1187 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x116a }, { 0xa438, 0xc0ff }, { 0xa438, 0xcaff },
	{ 0xa438, 0x1800 }, { 0xa438, 0x00d6 }, { 0xa438, 0xd504 },
	{ 0xa438, 0xa001 }, { 0xa438, 0xd704 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x128b }, { 0xa438, 0xd707 }, { 0xa438, 0x2005 },
	{ 0xa438, 0x8042 }, { 0xa438, 0xd75e }, { 0xa438, 0x1800 },
	{ 0xa438, 0x137a }, { 0xa438, 0x1800 }, { 0xa438, 0x13ed },
	{ 0xa438, 0x61d0 }, { 0xa438, 0xd701 }, { 0xa438, 0x60a5 },
	{ 0xa438, 0xd504 }, { 0xa438, 0xc9b2 }, { 0xa438, 0xd500 },
	{ 0xa438, 0xf004 }, { 0xa438, 0xd504 }, { 0xa438, 0xc9b1 },
	{ 0xa438, 0xd500 }, { 0xa438, 0xd707 }, { 0xa438, 0x6070 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x10a8 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x10bd }, { 0xa438, 0xd500 }, { 0xa438, 0xc492 },
	{ 0xa438, 0xd501 }, { 0xa438, 0x1800 }, { 0xa438, 0x13c1 },
	{ 0xa438, 0xa980 }, { 0xa438, 0xd500 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x143b }, { 0xa436, 0xa026 }, { 0xa438, 0x143a },
	{ 0xa436, 0xa024 }, { 0xa438, 0x13c0 }, { 0xa436, 0xa022 },
	{ 0xa438, 0x10bc }, { 0xa436, 0xa020 }, { 0xa438, 0x1379 },
	{ 0xa436, 0xa006 }, { 0xa438, 0x128a }, { 0xa436, 0xa004 },
	{ 0xa438, 0x00d5 }, { 0xa436, 0xa002 }, { 0xa438, 0x1182 },
	{ 0xa436, 0xa000 }, { 0xa438, 0x1075 }, { 0xa436, 0xa008 },
	{ 0xa438, 0xff00 }, { 0xa436, 0xa016 }, { 0xa438, 0x0010 },
	{ 0xa436, 0xa012 }, { 0xa438, 0x0000 }, { 0xa436, 0xa014 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8010 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8015 }, { 0xa438, 0x1800 }, { 0xa438, 0x801a },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801e }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8027 }, { 0xa438, 0x1800 }, { 0xa438, 0x8027 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8027 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8027 }, { 0xa438, 0x0c0f }, { 0xa438, 0x0505 },
	{ 0xa438, 0xba01 }, { 0xa438, 0x1800 }, { 0xa438, 0x015e },
	{ 0xa438, 0x0c0f }, { 0xa438, 0x0506 }, { 0xa438, 0xba02 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x017c }, { 0xa438, 0x9910 },
	{ 0xa438, 0x9a03 }, { 0xa438, 0x1800 }, { 0xa438, 0x02d4 },
	{ 0xa438, 0x8580 }, { 0xa438, 0xc090 }, { 0xa438, 0x9a03 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x02c9 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fa3 }, { 0xa438, 0x1800 }, { 0xa438, 0x0067 },
	{ 0xa436, 0xa08e }, { 0xa438, 0xffff }, { 0xa436, 0xa08c },
	{ 0xa438, 0xffff }, { 0xa436, 0xa08a }, { 0xa438, 0xffff },
	{ 0xa436, 0xa088 }, { 0xa438, 0xffff }, { 0xa436, 0xa086 },
	{ 0xa438, 0x018c }, { 0xa436, 0xa084 }, { 0xa438, 0x02d3 },
	{ 0xa436, 0xa082 }, { 0xa438, 0x017a }, { 0xa436, 0xa080 },
	{ 0xa438, 0x015c }, { 0xa436, 0xa090 }, { 0xa438, 0x000f },
	{ 0xa436, 0xa016 }, { 0xa438, 0x0020 }, { 0xa436, 0xa012 },
	{ 0xa438, 0x0000 }, { 0xa436, 0xa014 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8010 }, { 0xa438, 0x1800 }, { 0xa438, 0x8023 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8313 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x831a }, { 0xa438, 0x1800 }, { 0xa438, 0x8489 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x86b9 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x86c1 }, { 0xa438, 0x1800 }, { 0xa438, 0x87ad },
	{ 0xa438, 0x1000 }, { 0xa438, 0x124e }, { 0xa438, 0x9308 },
	{ 0xa438, 0xb201 }, { 0xa438, 0xb301 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x5fe0 }, { 0xa438, 0xd2ff }, { 0xa438, 0xb302 },
	{ 0xa438, 0xd200 }, { 0xa438, 0xb201 }, { 0xa438, 0xb309 },
	{ 0xa438, 0xd701 }, { 0xa438, 0x5fe0 }, { 0xa438, 0xd2ff },
	{ 0xa438, 0xb302 }, { 0xa438, 0xd200 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x0025 }, { 0xa438, 0xd706 }, { 0xa438, 0x6069 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6421 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x43ab }, { 0xa438, 0x800a }, { 0xa438, 0x8190 },
	{ 0xa438, 0x8204 }, { 0xa438, 0xa280 }, { 0xa438, 0x8406 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa108 },
	{ 0xa438, 0x9503 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0f19 }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5fb3 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x8f1f }, { 0xa438, 0x9503 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd70c }, { 0xa438, 0x7f33 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x11bd }, { 0xa438, 0x1800 },
	{ 0xa438, 0x81aa }, { 0xa438, 0x8710 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x33b1 }, { 0xa438, 0x8051 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x60b5 }, { 0xa438, 0xd706 }, { 0xa438, 0x6069 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8056 }, { 0xa438, 0xa00a },
	{ 0xa438, 0xa280 }, { 0xa438, 0xa404 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x80f3 }, { 0xa438, 0xd173 }, { 0xa438, 0xd04d },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xd173 }, { 0xa438, 0xd05d },
	{ 0xa438, 0xd10d }, { 0xa438, 0xd049 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fb4 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x64f5 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5ee7 }, { 0xa438, 0xb920 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd71f }, { 0xa438, 0x7fb4 },
	{ 0xa438, 0x9920 }, { 0xa438, 0xcb3c }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd71f }, { 0xa438, 0x7d94 },
	{ 0xa438, 0x6045 }, { 0xa438, 0xfffa }, { 0xa438, 0xb820 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fa5 }, { 0xa438, 0x9820 }, { 0xa438, 0xcb3d },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x60b5 }, { 0xa438, 0xd71f }, { 0xa438, 0x7bb4 },
	{ 0xa438, 0x61b6 }, { 0xa438, 0xfff8 }, { 0xa438, 0xbb80 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x9b80 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x60e7 }, { 0xa438, 0xcb3f }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8094 }, { 0xa438, 0xcb3e }, { 0xa438, 0x1800 },
	{ 0xa438, 0x810f }, { 0xa438, 0x1800 }, { 0xa438, 0x80f3 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xae04 },
	{ 0xa438, 0x9503 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x8e04 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd706 }, { 0xa438, 0x65fe },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0d04 }, { 0xa438, 0x8dc0 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x11bd }, { 0xa438, 0xd70c },
	{ 0xa438, 0x414b }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0040 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0102 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x03e0 }, { 0xa438, 0xccce }, { 0xa438, 0x1800 },
	{ 0xa438, 0x80b7 }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0040 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0100 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0380 }, { 0xa438, 0xcc9c }, { 0xa438, 0x8710 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1118 }, { 0xa438, 0xa104 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x112a }, { 0xa438, 0x8104 },
	{ 0xa438, 0xa202 }, { 0xa438, 0xa140 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x112a }, { 0xa438, 0x8140 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1121 }, { 0xa438, 0xaa0f }, { 0xa438, 0xa130 },
	{ 0xa438, 0xaa2f }, { 0xa438, 0xa2d5 }, { 0xa438, 0xa405 },
	{ 0xa438, 0xa720 }, { 0xa438, 0xa00a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x80f3 }, { 0xa438, 0xd704 }, { 0xa438, 0x3cf1 },
	{ 0xa438, 0x80d5 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d02 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x80d7 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d01 }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0d40 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x11bd }, { 0xa438, 0x8710 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1118 }, { 0xa438, 0xa108 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x112a }, { 0xa438, 0x8108 },
	{ 0xa438, 0xa203 }, { 0xa438, 0x8a2f }, { 0xa438, 0xa130 },
	{ 0xa438, 0x8204 }, { 0xa438, 0xa140 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x112a }, { 0xa438, 0x8140 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1121 }, { 0xa438, 0xd17a }, { 0xa438, 0xd04b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xa204 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fa7 },
	{ 0xa438, 0xb920 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x7fb4 }, { 0xa438, 0x9920 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x6125 }, { 0xa438, 0x6054 }, { 0xa438, 0xfffb },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fa7 }, { 0xa438, 0x1800 }, { 0xa438, 0x80f7 },
	{ 0xa438, 0xb820 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x7fa5 }, { 0xa438, 0x9820 },
	{ 0xa438, 0x9b01 }, { 0xa438, 0xd402 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x110d }, { 0xa438, 0xd701 }, { 0xa438, 0x33b1 },
	{ 0xa438, 0x811c }, { 0xa438, 0xd701 }, { 0xa438, 0x60b5 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x6069 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x811e }, { 0xa438, 0x1800 }, { 0xa438, 0x8183 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x40ab }, { 0xa438, 0x800a },
	{ 0xa438, 0x8110 }, { 0xa438, 0x8284 }, { 0xa438, 0x8404 },
	{ 0xa438, 0xa710 }, { 0xa438, 0x8120 }, { 0xa438, 0x8241 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1118 }, { 0xa438, 0xa104 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x112a }, { 0xa438, 0x8104 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1121 }, { 0xa438, 0xaa2f },
	{ 0xa438, 0xd70c }, { 0xa438, 0x438b }, { 0xa438, 0xa284 },
	{ 0xa438, 0xd078 }, { 0xa438, 0x800a }, { 0xa438, 0x8110 },
	{ 0xa438, 0xa284 }, { 0xa438, 0x8404 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xa108 }, { 0xa438, 0x9503 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0f19 }, { 0xa438, 0x9503 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd70c }, { 0xa438, 0x5fb3 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x8f1f },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x7f33 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d06 }, { 0xa438, 0x8dc0 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x11bd }, { 0xa438, 0x8110 }, { 0xa438, 0xa284 },
	{ 0xa438, 0xa404 }, { 0xa438, 0xa00a }, { 0xa438, 0xd70c },
	{ 0xa438, 0x40a1 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xad10 }, { 0xa438, 0x9503 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x414b }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0080 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0102 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0340 }, { 0xa438, 0xcc52 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x816b }, { 0xa438, 0x80c0 }, { 0xa438, 0x8103 },
	{ 0xa438, 0x83e0 }, { 0xa438, 0x8cff }, { 0xa438, 0xd193 },
	{ 0xa438, 0xd047 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1193 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5f74 }, { 0xa438, 0xa110 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0x1000 }, { 0xa438, 0x1193 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5f6a }, { 0xa438, 0xa180 },
	{ 0xa438, 0xd1f5 }, { 0xa438, 0xd049 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0x1000 }, { 0xa438, 0x1193 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5f74 }, { 0xa438, 0x8710 },
	{ 0xa438, 0xa00a }, { 0xa438, 0x8190 }, { 0xa438, 0x8204 },
	{ 0xa438, 0xa280 }, { 0xa438, 0xa404 }, { 0xa438, 0xbb80 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xb920 }, { 0xa438, 0x9b80 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fb4 }, { 0xa438, 0x9920 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xcb33 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x6105 }, { 0xa438, 0x5f74 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fa7 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x818e }, { 0xa438, 0xa710 },
	{ 0xa438, 0xb820 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x7f65 }, { 0xa438, 0x9820 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x81f1 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d04 }, { 0xa438, 0x8dc0 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x11bd }, { 0xa438, 0xa00a }, { 0xa438, 0x8280 },
	{ 0xa438, 0xa710 }, { 0xa438, 0xd103 }, { 0xa438, 0xd04c },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d06 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0x8710 }, { 0xa438, 0xa190 }, { 0xa438, 0xa204 },
	{ 0xa438, 0x8280 }, { 0xa438, 0xa404 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fa7 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xa110 }, { 0xa438, 0xa284 },
	{ 0xa438, 0xa404 }, { 0xa438, 0xcb33 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x5f54 }, { 0xa438, 0xb920 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd71f }, { 0xa438, 0x7fb4 },
	{ 0xa438, 0x9920 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x6145 }, { 0xa438, 0x6074 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x81d3 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fa7 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x81cd }, { 0xa438, 0xb820 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fa5 }, { 0xa438, 0xa710 }, { 0xa438, 0x9820 },
	{ 0xa438, 0xbb20 }, { 0xa438, 0x9308 }, { 0xa438, 0xb210 },
	{ 0xa438, 0xb301 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd701 }, { 0xa438, 0x5fa4 }, { 0xa438, 0xb302 },
	{ 0xa438, 0x9210 }, { 0xa438, 0xa00a }, { 0xa438, 0xa190 },
	{ 0xa438, 0xa284 }, { 0xa438, 0xa404 }, { 0xa438, 0xcb34 },
	{ 0xa438, 0xd701 }, { 0xa438, 0x33b1 }, { 0xa438, 0x823f },
	{ 0xa438, 0xd706 }, { 0xa438, 0x60a9 }, { 0xa438, 0xd1f5 },
	{ 0xa438, 0xd049 }, { 0xa438, 0x1800 }, { 0xa438, 0x8201 },
	{ 0xa438, 0xd13c }, { 0xa438, 0xd04a }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0x1000 }, { 0xa438, 0x1193 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5f74 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5f2b }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d03 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0x8190 }, { 0xa438, 0x8204 }, { 0xa438, 0xa280 },
	{ 0xa438, 0xa00a }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x8304 }, { 0xa438, 0x9503 }, { 0xa438, 0xcb35 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x414b }, { 0xa438, 0x8280 },
	{ 0xa438, 0x800a }, { 0xa438, 0xd411 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x110d }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0xa280 }, { 0xa438, 0xa00a }, { 0xa438, 0xd40a },
	{ 0xa438, 0xcb36 }, { 0xa438, 0x1000 }, { 0xa438, 0x110d },
	{ 0xa438, 0xd706 }, { 0xa438, 0x431b }, { 0xa438, 0x800a },
	{ 0xa438, 0x8180 }, { 0xa438, 0x8280 }, { 0xa438, 0x8404 },
	{ 0xa438, 0xa004 }, { 0xa438, 0x1000 }, { 0xa438, 0x112a },
	{ 0xa438, 0x8004 }, { 0xa438, 0xa001 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x112a }, { 0xa438, 0x8001 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x0902 }, { 0xa438, 0xa00a }, { 0xa438, 0xd14a },
	{ 0xa438, 0xd048 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1193 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5f74 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d06 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0xd70c }, { 0xa438, 0x414b }, { 0xa438, 0x0cc0 },
	{ 0xa438, 0x0080 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0101 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x03a0 }, { 0xa438, 0xccb5 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8256 }, { 0xa438, 0x0cc0 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0101 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0320 }, { 0xa438, 0xcc21 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x0c30 },
	{ 0xa438, 0x0120 }, { 0xa438, 0xa304 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x674b }, { 0xa438, 0xd704 },
	{ 0xa438, 0x471a }, { 0xa438, 0xa301 }, { 0xa438, 0x800a },
	{ 0xa438, 0xa110 }, { 0xa438, 0x8180 }, { 0xa438, 0xa204 },
	{ 0xa438, 0x82a0 }, { 0xa438, 0xa404 }, { 0xa438, 0xaa40 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xaa01 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd178 }, { 0xa438, 0xd049 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1193 }, { 0xa438, 0xd700 }, { 0xa438, 0x5f74 },
	{ 0xa438, 0x8301 }, { 0xa438, 0xa00a }, { 0xa438, 0x8110 },
	{ 0xa438, 0xa180 }, { 0xa438, 0xa284 }, { 0xa438, 0x8220 },
	{ 0xa438, 0xa404 }, { 0xa438, 0xd178 }, { 0xa438, 0xd048 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1193 }, { 0xa438, 0xd700 }, { 0xa438, 0x5f74 },
	{ 0xa438, 0xcb3a }, { 0xa438, 0x8301 }, { 0xa438, 0xa00a },
	{ 0xa438, 0xa190 }, { 0xa438, 0xa280 }, { 0xa438, 0x8224 },
	{ 0xa438, 0xa404 }, { 0xa438, 0xd700 }, { 0xa438, 0x6041 },
	{ 0xa438, 0xa402 }, { 0xa438, 0xd178 }, { 0xa438, 0xd049 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1193 }, { 0xa438, 0xd700 }, { 0xa438, 0x5f74 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x82ab }, { 0xa438, 0xa00a },
	{ 0xa438, 0xa190 }, { 0xa438, 0xa2a4 }, { 0xa438, 0xa404 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6041 }, { 0xa438, 0xa402 },
	{ 0xa438, 0xcb37 }, { 0xa438, 0xd706 }, { 0xa438, 0x60a9 },
	{ 0xa438, 0xd13d }, { 0xa438, 0xd04a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x82a7 }, { 0xa438, 0xd13c }, { 0xa438, 0xd04b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1193 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5f6b }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d07 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0xd40d }, { 0xa438, 0x1000 }, { 0xa438, 0x110d },
	{ 0xa438, 0xa208 }, { 0xa438, 0x8204 }, { 0xa438, 0xaa40 },
	{ 0xa438, 0xcb38 }, { 0xa438, 0xd706 }, { 0xa438, 0x6129 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x608b }, { 0xa438, 0xd17a },
	{ 0xa438, 0xd047 }, { 0xa438, 0xf006 }, { 0xa438, 0xd13d },
	{ 0xa438, 0xd04b }, { 0xa438, 0xf003 }, { 0xa438, 0xd196 },
	{ 0xa438, 0xd04b }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1193 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5f74 }, { 0xa438, 0xd704 }, { 0xa438, 0x35ac },
	{ 0xa438, 0x8311 }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0101 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0320 }, { 0xa438, 0xcc21 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d03 }, { 0xa438, 0x8dc0 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x11bd }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0c07 }, { 0xa438, 0x0c07 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xa280 }, { 0xa438, 0x8780 },
	{ 0xa438, 0x0c60 }, { 0xa438, 0x0700 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x409c }, { 0xa438, 0xd110 },
	{ 0xa438, 0xd04d }, { 0xa438, 0xf003 }, { 0xa438, 0xd110 },
	{ 0xa438, 0xd04d }, { 0xa438, 0xcb4a }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fb4 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa240 },
	{ 0xa438, 0xa180 }, { 0xa438, 0xa201 }, { 0xa438, 0xa780 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd114 }, { 0xa438, 0xd04a },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0xcb4b }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fb4 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x0bc3 }, { 0xa438, 0x1800 }, { 0xa438, 0x0bc3 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x110d }, { 0xa438, 0xd419 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x110d }, { 0xa438, 0x1800 },
	{ 0xa438, 0x01ae }, { 0xa438, 0x8110 }, { 0xa438, 0xa180 },
	{ 0xa438, 0x8280 }, { 0xa438, 0xa404 }, { 0xa438, 0xa00a },
	{ 0xa438, 0x8402 }, { 0xa438, 0xcb42 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x3de9 }, { 0xa438, 0x837a }, { 0xa438, 0xd704 },
	{ 0xa438, 0x35ac }, { 0xa438, 0x8380 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fab },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0d06 }, { 0xa438, 0x8dc0 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x11bd }, { 0xa438, 0xd418 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x110d }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d03 }, { 0xa438, 0x8dc0 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x11bd }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xa780 }, { 0xa438, 0xa20e }, { 0xa438, 0x9503 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x409c }, { 0xa438, 0xd114 },
	{ 0xa438, 0xd04d }, { 0xa438, 0xf003 }, { 0xa438, 0xd114 },
	{ 0xa438, 0xd04d }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xa003 }, { 0xa438, 0x9503 }, { 0xa438, 0xcb4c },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x0c60 }, { 0xa438, 0x0720 }, { 0xa438, 0xa220 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xcb4d }, { 0xa438, 0xd704 },
	{ 0xa438, 0x409c }, { 0xa438, 0xd128 }, { 0xa438, 0xd04f },
	{ 0xa438, 0xf003 }, { 0xa438, 0xd128 }, { 0xa438, 0xd04f },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x0c60 }, { 0xa438, 0x0740 }, { 0xa438, 0xa210 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd704 }, { 0xa438, 0x409c },
	{ 0xa438, 0xd114 }, { 0xa438, 0xd04e }, { 0xa438, 0xf003 },
	{ 0xa438, 0xd114 }, { 0xa438, 0xd04e }, { 0xa438, 0xcb4e },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d06 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0x0cc0 }, { 0xa438, 0x0000 }, { 0xa438, 0x0c07 },
	{ 0xa438, 0x0c01 }, { 0xa438, 0xd704 }, { 0xa438, 0x40b5 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa23c },
	{ 0xa438, 0x9503 }, { 0xa438, 0xb920 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd71f }, { 0xa438, 0x7fb4 },
	{ 0xa438, 0x8710 }, { 0xa438, 0x9920 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd71f }, { 0xa438, 0x6105 },
	{ 0xa438, 0x6054 }, { 0xa438, 0xfffb }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fa7 },
	{ 0xa438, 0xffef }, { 0xa438, 0xa710 }, { 0xa438, 0xb820 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fa5 }, { 0xa438, 0x9820 }, { 0xa438, 0xa00a },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa103 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xbb20 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x60dd }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d07 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x0c30 },
	{ 0xa438, 0x0120 }, { 0xa438, 0xa304 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xa190 }, { 0xa438, 0xa2a0 }, { 0xa438, 0xa404 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xa604 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x6041 }, { 0xa438, 0xa402 }, { 0xa438, 0xcb43 },
	{ 0xa438, 0xd17a }, { 0xa438, 0xd048 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0x1000 }, { 0xa438, 0x1193 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5f74 }, { 0xa438, 0x609d },
	{ 0xa438, 0xd417 }, { 0xa438, 0x1000 }, { 0xa438, 0x110d },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1193 }, { 0xa438, 0xd700 }, { 0xa438, 0x5f7a },
	{ 0xa438, 0xd704 }, { 0xa438, 0x5f36 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x6089 }, { 0xa438, 0xd40c }, { 0xa438, 0x1000 },
	{ 0xa438, 0x110d }, { 0xa438, 0xaa40 }, { 0xa438, 0xbb10 },
	{ 0xa438, 0xcb50 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1193 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x5f75 }, { 0xa438, 0x8190 }, { 0xa438, 0x82a0 },
	{ 0xa438, 0x8402 }, { 0xa438, 0xa404 }, { 0xa438, 0x800a },
	{ 0xa438, 0x8718 }, { 0xa438, 0x9b10 }, { 0xa438, 0x9b20 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fb5 }, { 0xa438, 0xcb51 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd71f }, { 0xa438, 0x5f94 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x6089 }, { 0xa438, 0xd141 },
	{ 0xa438, 0xd043 }, { 0xa438, 0xf003 }, { 0xa438, 0xd141 },
	{ 0xa438, 0xd044 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fb4 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x60e5 }, { 0xa438, 0xd704 }, { 0xa438, 0x60be },
	{ 0xa438, 0xd706 }, { 0xa438, 0x29b1 }, { 0xa438, 0x83fb },
	{ 0xa438, 0xf002 }, { 0xa438, 0xa880 }, { 0xa438, 0xa00a },
	{ 0xa438, 0xa190 }, { 0xa438, 0x8220 }, { 0xa438, 0xa280 },
	{ 0xa438, 0xa404 }, { 0xa438, 0xa620 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xc5aa }, { 0xa438, 0x9503 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6061 }, { 0xa438, 0xa402 },
	{ 0xa438, 0xa480 }, { 0xa438, 0xcb52 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fba },
	{ 0xa438, 0xd704 }, { 0xa438, 0x5f76 }, { 0xa438, 0xb920 },
	{ 0xa438, 0xcb53 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x7fb4 }, { 0xa438, 0x9920 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xa190 }, { 0xa438, 0xa280 },
	{ 0xa438, 0x8220 }, { 0xa438, 0xa404 }, { 0xa438, 0xb580 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x40a1 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xa602 }, { 0xa438, 0x9503 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa310 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xcb60 }, { 0xa438, 0xd1c8 },
	{ 0xa438, 0xd045 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fb4 }, { 0xa438, 0xaa10 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x2833 }, { 0xa438, 0x8434 },
	{ 0xa438, 0xf003 }, { 0xa438, 0x1000 }, { 0xa438, 0x1238 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x40a6 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xa140 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x40a3 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xac20 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xa90c }, { 0xa438, 0xaa80 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d07 }, { 0xa438, 0x8dc0 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x11bd }, { 0xa438, 0xa00a }, { 0xa438, 0xa190 },
	{ 0xa438, 0xa280 }, { 0xa438, 0x8220 }, { 0xa438, 0xa404 },
	{ 0xa438, 0xb580 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xc500 }, { 0xa438, 0x9503 }, { 0xa438, 0x83e0 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x40c1 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xa602 }, { 0xa438, 0x9503 },
	{ 0xa438, 0x8e01 }, { 0xa438, 0xd14a }, { 0xa438, 0xd058 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x4063 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x11f2 }, { 0xa438, 0xcb62 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd704 }, { 0xa438, 0x2e70 },
	{ 0xa438, 0x8479 }, { 0xa438, 0xd71f }, { 0xa438, 0x626e },
	{ 0xa438, 0xd704 }, { 0xa438, 0x3868 }, { 0xa438, 0x847d },
	{ 0xa438, 0xd70c }, { 0xa438, 0x2f18 }, { 0xa438, 0x8483 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5db5 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xc5aa }, { 0xa438, 0x9503 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0320 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x0d6f }, { 0xa438, 0x1800 }, { 0xa438, 0x0f15 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0dae }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xc5aa }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0fc9 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xc5aa }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0d84 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e5 }, { 0xa438, 0xd70c }, { 0xa438, 0x5fa4 },
	{ 0xa438, 0xa706 }, { 0xa438, 0xd70c }, { 0xa438, 0x408b },
	{ 0xa438, 0xa701 }, { 0xa438, 0xa502 }, { 0xa438, 0xa880 },
	{ 0xa438, 0x8801 }, { 0xa438, 0x8e01 }, { 0xa438, 0xca50 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x852e }, { 0xa438, 0xca51 },
	{ 0xa438, 0xd70e }, { 0xa438, 0x2210 }, { 0xa438, 0x852c },
	{ 0xa438, 0xd70c }, { 0xa438, 0x4084 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x5efd }, { 0xa438, 0xf007 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e9 }, { 0xa438, 0xd70c }, { 0xa438, 0x5ca2 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x15b2 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x605a }, { 0xa438, 0x9a10 }, { 0xa438, 0x8e40 },
	{ 0xa438, 0x8404 }, { 0xa438, 0x1000 }, { 0xa438, 0x174e },
	{ 0xa438, 0x8e80 }, { 0xa438, 0xca62 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x3084 }, { 0xa438, 0x850e }, { 0xa438, 0xba10 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x8608 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0100 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x4638 }, { 0xa438, 0xd1c4 },
	{ 0xa438, 0xd044 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x170f }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f7c }, { 0xa438, 0x8108 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0907 }, { 0xa438, 0x8940 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1702 }, { 0xa438, 0xa0c4 }, { 0xa438, 0x8610 },
	{ 0xa438, 0x8030 }, { 0xa438, 0x8706 }, { 0xa438, 0x0c07 },
	{ 0xa438, 0x0b06 }, { 0xa438, 0x8410 }, { 0xa438, 0xa980 },
	{ 0xa438, 0xa702 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd045 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x16e5 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x170f }, { 0xa438, 0xd70c }, { 0xa438, 0x5f7c },
	{ 0xa438, 0x0c07 }, { 0xa438, 0x0b06 }, { 0xa438, 0xa030 },
	{ 0xa438, 0xa610 }, { 0xa438, 0xd700 }, { 0xa438, 0x6041 },
	{ 0xa438, 0xa501 }, { 0xa438, 0xa108 }, { 0xa438, 0xd1c4 },
	{ 0xa438, 0xd045 }, { 0xa438, 0xca63 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e5 }, { 0xa438, 0x1000 }, { 0xa438, 0x170f },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f7c }, { 0xa438, 0xd702 },
	{ 0xa438, 0x6078 }, { 0xa438, 0x9920 }, { 0xa438, 0xf003 },
	{ 0xa438, 0xb920 }, { 0xa438, 0xa880 }, { 0xa438, 0x9a10 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x16e5 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x170f }, { 0xa438, 0xd71f }, { 0xa438, 0x5f73 },
	{ 0xa438, 0xf011 }, { 0xa438, 0xd70c }, { 0xa438, 0x409b },
	{ 0xa438, 0x9920 }, { 0xa438, 0x9a10 }, { 0xa438, 0xfff5 },
	{ 0xa438, 0x80fe }, { 0xa438, 0x8610 }, { 0xa438, 0x8501 },
	{ 0xa438, 0x8980 }, { 0xa438, 0x8702 }, { 0xa438, 0xa410 },
	{ 0xa438, 0xa940 }, { 0xa438, 0x81c0 }, { 0xa438, 0xae80 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x84b3 }, { 0xa438, 0x8804 },
	{ 0xa438, 0xa704 }, { 0xa438, 0x8788 }, { 0xa438, 0xff80 },
	{ 0xa438, 0xbb08 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0907 },
	{ 0xa438, 0x8940 }, { 0xa438, 0x1000 }, { 0xa438, 0x1702 },
	{ 0xa438, 0x8701 }, { 0xa438, 0x8502 }, { 0xa438, 0xa0f4 },
	{ 0xa438, 0xa610 }, { 0xa438, 0xd700 }, { 0xa438, 0x6061 },
	{ 0xa438, 0xa002 }, { 0xa438, 0xa501 }, { 0xa438, 0x8706 },
	{ 0xa438, 0x8410 }, { 0xa438, 0xa980 }, { 0xa438, 0xca64 },
	{ 0xa438, 0xd110 }, { 0xa438, 0xd040 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e5 }, { 0xa438, 0x1000 }, { 0xa438, 0x170f },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f7c }, { 0xa438, 0x8804 },
	{ 0xa438, 0xa706 }, { 0xa438, 0x1800 }, { 0xa438, 0x848d },
	{ 0xa438, 0x1800 }, { 0xa438, 0x1384 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x405f }, { 0xa438, 0xf036 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x6234 }, { 0xa438, 0xd70c }, { 0xa438, 0x41c6 },
	{ 0xa438, 0xd70d }, { 0xa438, 0x419d }, { 0xa438, 0xd70d },
	{ 0xa438, 0x417e }, { 0xa438, 0xd704 }, { 0xa438, 0x6127 },
	{ 0xa438, 0x2951 }, { 0xa438, 0x8543 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x4083 }, { 0xa438, 0xd70c }, { 0xa438, 0x2e81 },
	{ 0xa438, 0x8543 }, { 0xa438, 0xf0c5 }, { 0xa438, 0x80fe },
	{ 0xa438, 0x8610 }, { 0xa438, 0x8501 }, { 0xa438, 0x8704 },
	{ 0xa438, 0x0c30 }, { 0xa438, 0x0410 }, { 0xa438, 0xa701 },
	{ 0xa438, 0xac02 }, { 0xa438, 0xa502 }, { 0xa438, 0x8980 },
	{ 0xa438, 0xca60 }, { 0xa438, 0xa004 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x6065 }, { 0xa438, 0x1800 }, { 0xa438, 0x8554 },
	{ 0xa438, 0x8004 }, { 0xa438, 0xa804 }, { 0xa438, 0x0c0f },
	{ 0xa438, 0x0602 }, { 0xa438, 0x0c70 }, { 0xa438, 0x0730 },
	{ 0xa438, 0xa708 }, { 0xa438, 0xd704 }, { 0xa438, 0x609c },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0912 }, { 0xa438, 0xf003 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x090e }, { 0xa438, 0xa940 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1702 }, { 0xa438, 0xa780 },
	{ 0xa438, 0xf0a2 }, { 0xa438, 0xd704 }, { 0xa438, 0x63eb },
	{ 0xa438, 0xd705 }, { 0xa438, 0x43b1 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x339c }, { 0xa438, 0x8607 }, { 0xa438, 0x8788 },
	{ 0xa438, 0x8704 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0907 },
	{ 0xa438, 0x8940 }, { 0xa438, 0x1000 }, { 0xa438, 0x1702 },
	{ 0xa438, 0x8410 }, { 0xa438, 0xa0f4 }, { 0xa438, 0xa610 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6061 }, { 0xa438, 0xa002 },
	{ 0xa438, 0xa501 }, { 0xa438, 0xa706 }, { 0xa438, 0x8804 },
	{ 0xa438, 0xa980 }, { 0xa438, 0xd70c }, { 0xa438, 0x6085 },
	{ 0xa438, 0x8701 }, { 0xa438, 0x8502 }, { 0xa438, 0x8c02 },
	{ 0xa438, 0xa701 }, { 0xa438, 0xa502 }, { 0xa438, 0xf082 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x60c5 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x6053 }, { 0xa438, 0xf07d }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8604 }, { 0xa438, 0xd70d }, { 0xa438, 0x4d1b },
	{ 0xa438, 0xba10 }, { 0xa438, 0xae40 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x03b4 }, { 0xa438, 0x0cfc }, { 0xa438, 0x05b4 },
	{ 0xa438, 0xd1c4 }, { 0xa438, 0xd044 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e5 }, { 0xa438, 0x1000 }, { 0xa438, 0x170f },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f7c }, { 0xa438, 0x8706 },
	{ 0xa438, 0x8280 }, { 0xa438, 0xace0 }, { 0xa438, 0xa680 },
	{ 0xa438, 0xa240 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x170f }, { 0xa438, 0xd702 },
	{ 0xa438, 0x5f79 }, { 0xa438, 0x8240 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x6898 }, { 0xa438, 0xd702 }, { 0xa438, 0x4957 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x85f6 }, { 0xa438, 0xa1c0 },
	{ 0xa438, 0x0c3f }, { 0xa438, 0x0220 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x030c }, { 0xa438, 0x0cfc }, { 0xa438, 0x050c },
	{ 0xa438, 0x8108 }, { 0xa438, 0x8640 }, { 0xa438, 0xa120 },
	{ 0xa438, 0xa640 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0101 },
	{ 0xa438, 0xa110 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd044 },
	{ 0xa438, 0xca84 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x170f }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f7c }, { 0xa438, 0xd702 }, { 0xa438, 0x60fc },
	{ 0xa438, 0x8210 }, { 0xa438, 0x0ce0 }, { 0xa438, 0x0320 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0520 }, { 0xa438, 0xf002 },
	{ 0xa438, 0xa210 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd043 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x16e5 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x170f }, { 0xa438, 0xd70c }, { 0xa438, 0x5f7c },
	{ 0xa438, 0x8233 }, { 0xa438, 0x0cfc }, { 0xa438, 0x036c },
	{ 0xa438, 0x0cfc }, { 0xa438, 0x056c }, { 0xa438, 0xd1c4 },
	{ 0xa438, 0xd044 }, { 0xa438, 0xca85 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e5 }, { 0xa438, 0x1000 }, { 0xa438, 0x170f },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f7c }, { 0xa438, 0xa680 },
	{ 0xa438, 0xa240 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x170f }, { 0xa438, 0xd702 },
	{ 0xa438, 0x5f79 }, { 0xa438, 0x8240 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x0390 }, { 0xa438, 0x0cfc }, { 0xa438, 0x0590 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x6058 }, { 0xa438, 0xf002 },
	{ 0xa438, 0xfec7 }, { 0xa438, 0x81c0 }, { 0xa438, 0x8880 },
	{ 0xa438, 0x8706 }, { 0xa438, 0xca61 }, { 0xa438, 0xd1c4 },
	{ 0xa438, 0xd054 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x170f }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f7d }, { 0xa438, 0xa706 }, { 0xa438, 0xf004 },
	{ 0xa438, 0x8788 }, { 0xa438, 0xa404 }, { 0xa438, 0x8702 },
	{ 0xa438, 0x0800 }, { 0xa438, 0x8443 }, { 0xa438, 0x8303 },
	{ 0xa438, 0x8280 }, { 0xa438, 0x9920 }, { 0xa438, 0x8ce0 },
	{ 0xa438, 0x8004 }, { 0xa438, 0xa1c0 }, { 0xa438, 0xd70e },
	{ 0xa438, 0x404a }, { 0xa438, 0xa280 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x3bd0 }, { 0xa438, 0x8618 }, { 0xa438, 0x0c3f },
	{ 0xa438, 0x0223 }, { 0xa438, 0xf003 }, { 0xa438, 0x0c3f },
	{ 0xa438, 0x0220 }, { 0xa438, 0x0cfc }, { 0xa438, 0x0308 },
	{ 0xa438, 0x0cfc }, { 0xa438, 0x0508 }, { 0xa438, 0x8108 },
	{ 0xa438, 0x8640 }, { 0xa438, 0xa120 }, { 0xa438, 0xa640 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x6077 }, { 0xa438, 0x8103 },
	{ 0xa438, 0xf003 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0101 },
	{ 0xa438, 0xa110 }, { 0xa438, 0xd702 }, { 0xa438, 0x6077 },
	{ 0xa438, 0xa108 }, { 0xa438, 0xf006 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x6077 }, { 0xa438, 0x8108 }, { 0xa438, 0xf002 },
	{ 0xa438, 0xa108 }, { 0xa438, 0xd193 }, { 0xa438, 0xd045 },
	{ 0xa438, 0xca82 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0xd70e }, { 0xa438, 0x606a }, { 0xa438, 0x1000 },
	{ 0xa438, 0x170f }, { 0xa438, 0xd70c }, { 0xa438, 0x5f3c },
	{ 0xa438, 0xd702 }, { 0xa438, 0x60fc }, { 0xa438, 0x8210 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0320 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0520 }, { 0xa438, 0xf002 }, { 0xa438, 0xa210 },
	{ 0xa438, 0xd1c4 }, { 0xa438, 0xd043 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e5 }, { 0xa438, 0xd70e }, { 0xa438, 0x606a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x170f }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f3c }, { 0xa438, 0xd702 }, { 0xa438, 0x3bd0 },
	{ 0xa438, 0x8656 }, { 0xa438, 0x0c3f }, { 0xa438, 0x020c },
	{ 0xa438, 0xf002 }, { 0xa438, 0x823f }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x034c }, { 0xa438, 0x0cfc }, { 0xa438, 0x054c },
	{ 0xa438, 0xd1c4 }, { 0xa438, 0xd044 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x16e5 }, { 0xa438, 0xd70e }, { 0xa438, 0x606a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x170f }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f3c }, { 0xa438, 0x820c }, { 0xa438, 0xa360 },
	{ 0xa438, 0xa560 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd043 },
	{ 0xa438, 0xca83 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0xd70e }, { 0xa438, 0x606a }, { 0xa438, 0x1000 },
	{ 0xa438, 0x170f }, { 0xa438, 0xd70c }, { 0xa438, 0x5f3c },
	{ 0xa438, 0xd70e }, { 0xa438, 0x406a }, { 0xa438, 0x8680 },
	{ 0xa438, 0xf002 }, { 0xa438, 0xa680 }, { 0xa438, 0xa240 },
	{ 0xa438, 0x0c0f }, { 0xa438, 0x0604 }, { 0xa438, 0x0c70 },
	{ 0xa438, 0x0750 }, { 0xa438, 0xa708 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x609c }, { 0xa438, 0x0c1f }, { 0xa438, 0x0914 },
	{ 0xa438, 0xf003 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0910 },
	{ 0xa438, 0xa940 }, { 0xa438, 0x1000 }, { 0xa438, 0x1702 },
	{ 0xa438, 0xa780 }, { 0xa438, 0x1000 }, { 0xa438, 0x16e5 },
	{ 0xa438, 0xd70e }, { 0xa438, 0x606a }, { 0xa438, 0x1000 },
	{ 0xa438, 0x170f }, { 0xa438, 0xd702 }, { 0xa438, 0x399c },
	{ 0xa438, 0x8689 }, { 0xa438, 0x8240 }, { 0xa438, 0x8788 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x63f8 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x643c }, { 0xa438, 0xa402 }, { 0xa438, 0xf012 },
	{ 0xa438, 0x8402 }, { 0xa438, 0xd705 }, { 0xa438, 0x611b },
	{ 0xa438, 0xa401 }, { 0xa438, 0xa302 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x417d }, { 0xa438, 0xa440 }, { 0xa438, 0xa280 },
	{ 0xa438, 0xf008 }, { 0xa438, 0x8401 }, { 0xa438, 0x8302 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x6060 }, { 0xa438, 0xa301 },
	{ 0xa438, 0xf002 }, { 0xa438, 0x8301 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x4080 }, { 0xa438, 0xd70e }, { 0xa438, 0x604a },
	{ 0xa438, 0xff5f }, { 0xa438, 0xd705 }, { 0xa438, 0x3cdd },
	{ 0xa438, 0x86b8 }, { 0xa438, 0xff5b }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x0390 }, { 0xa438, 0x0cfc }, { 0xa438, 0x0590 },
	{ 0xa438, 0x0800 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d00 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0xa504 }, { 0xa438, 0x1800 }, { 0xa438, 0x0fd3 },
	{ 0xa438, 0xd70d }, { 0xa438, 0x407d }, { 0xa438, 0xa710 },
	{ 0xa438, 0xf002 }, { 0xa438, 0xa710 }, { 0xa438, 0x9580 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa304 },
	{ 0xa438, 0x9503 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d07 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x11bd },
	{ 0xa438, 0xcb81 }, { 0xa438, 0xd70c }, { 0xa438, 0x4882 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x407a }, { 0xa438, 0xd70c },
	{ 0xa438, 0x4807 }, { 0xa438, 0xd706 }, { 0xa438, 0x405a },
	{ 0xa438, 0x8910 }, { 0xa438, 0xa210 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x611c }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0080 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0101 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x03a0 }, { 0xa438, 0xccb5 }, { 0xa438, 0x0cc0 },
	{ 0xa438, 0x0080 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0102 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0340 }, { 0xa438, 0xcc52 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x42ba }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0f1c },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5fb3 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x8f1f }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x7f33 }, { 0xa438, 0x8190 }, { 0xa438, 0x8204 },
	{ 0xa438, 0xf016 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0f1b }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5fb3 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x8f1f }, { 0xa438, 0x9503 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd70c }, { 0xa438, 0x7f33 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x6047 }, { 0xa438, 0xf002 },
	{ 0xa438, 0xf00c }, { 0xa438, 0xd403 }, { 0xa438, 0xcb82 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x110d }, { 0xa438, 0xd40a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x110d }, { 0xa438, 0xd70c },
	{ 0xa438, 0x4247 }, { 0xa438, 0x1000 }, { 0xa438, 0x1225 },
	{ 0xa438, 0x8a40 }, { 0xa438, 0x1000 }, { 0xa438, 0x1118 },
	{ 0xa438, 0xa104 }, { 0xa438, 0x1000 }, { 0xa438, 0x112a },
	{ 0xa438, 0x8104 }, { 0xa438, 0x1000 }, { 0xa438, 0x1121 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa704 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xcb88 }, { 0xa438, 0xf012 },
	{ 0xa438, 0xa210 }, { 0xa438, 0xa00a }, { 0xa438, 0xaa40 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1118 }, { 0xa438, 0xa104 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x112a }, { 0xa438, 0x8104 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1121 }, { 0xa438, 0xa190 },
	{ 0xa438, 0xa284 }, { 0xa438, 0xa404 }, { 0xa438, 0x8a10 },
	{ 0xa438, 0x8a80 }, { 0xa438, 0xcb84 }, { 0xa438, 0xd13e },
	{ 0xa438, 0xd05a }, { 0xa438, 0xd13e }, { 0xa438, 0xd06b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x3559 }, { 0xa438, 0x874b }, { 0xa438, 0xfffb },
	{ 0xa438, 0xd700 }, { 0xa438, 0x604b }, { 0xa438, 0xcb8a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x3659 }, { 0xa438, 0x8754 }, { 0xa438, 0xfffb },
	{ 0xa438, 0xd700 }, { 0xa438, 0x606b }, { 0xa438, 0xcb8b },
	{ 0xa438, 0x5eeb }, { 0xa438, 0xd700 }, { 0xa438, 0x6041 },
	{ 0xa438, 0xa402 }, { 0xa438, 0xcb8c }, { 0xa438, 0xd706 },
	{ 0xa438, 0x609a }, { 0xa438, 0xd1f5 }, { 0xa438, 0xd048 },
	{ 0xa438, 0xf003 }, { 0xa438, 0xd160 }, { 0xa438, 0xd04b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xcb8d }, { 0xa438, 0x8710 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x5fd4 }, { 0xa438, 0xb920 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fb4 }, { 0xa438, 0x9920 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd71f }, { 0xa438, 0x6105 },
	{ 0xa438, 0x6054 }, { 0xa438, 0xfffb }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1175 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fab },
	{ 0xa438, 0xfff0 }, { 0xa438, 0xa710 }, { 0xa438, 0xb820 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fa5 }, { 0xa438, 0x9820 }, { 0xa438, 0xd114 },
	{ 0xa438, 0xd040 }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fba }, { 0xa438, 0xd704 },
	{ 0xa438, 0x5f76 }, { 0xa438, 0xd700 }, { 0xa438, 0x5f34 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6081 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x405a }, { 0xa438, 0xa480 }, { 0xa438, 0xcb86 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x609a }, { 0xa438, 0xd1c8 },
	{ 0xa438, 0xd045 }, { 0xa438, 0xf003 }, { 0xa438, 0xd17a },
	{ 0xa438, 0xd04b }, { 0xa438, 0x1000 }, { 0xa438, 0x1175 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fb4 }, { 0xa438, 0x0cc0 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0101 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0320 }, { 0xa438, 0xcc29 },
	{ 0xa438, 0xa208 }, { 0xa438, 0x8204 }, { 0xa438, 0xd114 },
	{ 0xa438, 0xd040 }, { 0xa438, 0xd700 }, { 0xa438, 0x5ff4 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0bc3 }, { 0xa438, 0xa00a },
	{ 0xa438, 0x9308 }, { 0xa438, 0xb210 }, { 0xa438, 0xb301 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1175 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x5fa4 }, { 0xa438, 0xb302 }, { 0xa438, 0x9210 },
	{ 0xa438, 0x800a }, { 0xa438, 0x1800 }, { 0xa438, 0x0573 },
	{ 0xa436, 0xa10e }, { 0xa438, 0x0572 }, { 0xa436, 0xa10c },
	{ 0xa438, 0x0e47 }, { 0xa436, 0xa10a }, { 0xa438, 0x0fd2 },
	{ 0xa436, 0xa108 }, { 0xa438, 0x1503 }, { 0xa436, 0xa106 },
	{ 0xa438, 0x0c0d }, { 0xa436, 0xa104 }, { 0xa438, 0x01ac },
	{ 0xa436, 0xa102 }, { 0xa438, 0x0956 }, { 0xa436, 0xa100 },
	{ 0xa438, 0x001c }, { 0xa436, 0xa110 }, { 0xa438, 0x00ff },
	{ 0xa436, 0xa016 }, { 0xa438, 0x0020 }, { 0xa436, 0xa012 },
	{ 0xa438, 0x1ff8 }, { 0xa436, 0xa014 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x85f0 }, { 0xa438, 0xa2a0 }, { 0xa438, 0x8880 },
	{ 0xa438, 0x0d00 }, { 0xa438, 0xc500 }, { 0xa438, 0x800a },
	{ 0xa438, 0xae01 }, { 0xa436, 0xa164 }, { 0xa438, 0x1013 },
	{ 0xa436, 0xa166 }, { 0xa438, 0x1014 }, { 0xa436, 0xa168 },
	{ 0xa438, 0x0f98 }, { 0xa436, 0xa16a }, { 0xa438, 0x0dca },
	{ 0xa436, 0xa16c }, { 0xa438, 0x109b }, { 0xa436, 0xa16e },
	{ 0xa438, 0x10a2 }, { 0xa436, 0xa170 }, { 0xa438, 0x0f33 },
	{ 0xa436, 0xa172 }, { 0xa438, 0x0f6e }, { 0xa436, 0xa162 },
	{ 0xa438, 0x00ff }, { 0xa436, 0xb87c }, { 0xa438, 0x8a45 },
	{ 0xa436, 0xb87e }, { 0xa438, 0xaf8a }, { 0xa438, 0x5daf },
	{ 0xa438, 0x8a63 }, { 0xa438, 0xaf8a }, { 0xa438, 0x6caf },
	{ 0xa438, 0x8a78 }, { 0xa438, 0xaf8a }, { 0xa438, 0x87af },
	{ 0xa438, 0x8a90 }, { 0xa438, 0xaf8a }, { 0xa438, 0x96af },
	{ 0xa438, 0x8acf }, { 0xa438, 0x028a }, { 0xa438, 0xecaf },
	{ 0xa438, 0x211f }, { 0xa438, 0x0265 }, { 0xa438, 0xcb02 },
	{ 0xa438, 0x8fb4 }, { 0xa438, 0xaf21 }, { 0xa438, 0x6fa1 },
	{ 0xa438, 0x1903 }, { 0xa438, 0x028f }, { 0xa438, 0x3d02 },
	{ 0xa438, 0x2261 }, { 0xa438, 0xaf21 }, { 0xa438, 0x2ead },
	{ 0xa438, 0x2109 }, { 0xa438, 0xe08f }, { 0xa438, 0xffac },
	{ 0xa438, 0x2503 }, { 0xa438, 0xaf4b }, { 0xa438, 0xeeaf },
	{ 0xa438, 0x4beb }, { 0xa438, 0xad35 }, { 0xa438, 0x03af },
	{ 0xa438, 0x421b }, { 0xa438, 0xaf42 }, { 0xa438, 0x5ce1 },
	{ 0xa438, 0x8652 }, { 0xa438, 0xaf49 }, { 0xa438, 0xdcef },
	{ 0xa438, 0x31e1 }, { 0xa438, 0x8ffd }, { 0xa438, 0xac28 },
	{ 0xa438, 0x2ebf }, { 0xa438, 0x6dda }, { 0xa438, 0x0274 },
	{ 0xa438, 0x95ad }, { 0xa438, 0x2825 }, { 0xa438, 0xe28f },
	{ 0xa438, 0xe4ef }, { 0xa438, 0x131b }, { 0xa438, 0x12ac },
	{ 0xa438, 0x2f10 }, { 0xa438, 0xef31 }, { 0xa438, 0x1f44 },
	{ 0xa438, 0xef13 }, { 0xa438, 0xbf6c }, { 0xa438, 0xcf02 },
	{ 0xa438, 0x7476 }, { 0xa438, 0x1a12 }, { 0xa438, 0xae08 },
	{ 0xa438, 0xbf6c }, { 0xa438, 0xcf02 }, { 0xa438, 0x744a },
	{ 0xa438, 0xef13 }, { 0xa438, 0xaf08 }, { 0xa438, 0x66af },
	{ 0xa438, 0x085c }, { 0xa438, 0xe18f }, { 0xa438, 0xe3ad },
	{ 0xa438, 0x2706 }, { 0xa438, 0xe58f }, { 0xa438, 0xe9af },
	{ 0xa438, 0x4091 }, { 0xa438, 0xe08f }, { 0xa438, 0xe1ac },
	{ 0xa438, 0x2002 }, { 0xa438, 0xae03 }, { 0xa438, 0xe18f },
	{ 0xa438, 0xe2e5 }, { 0xa438, 0x8fe9 }, { 0xa438, 0xaf3f },
	{ 0xa438, 0xe5f8 }, { 0xa438, 0xe08f }, { 0xa438, 0xe7a0 },
	{ 0xa438, 0x0005 }, { 0xa438, 0x028b }, { 0xa438, 0x0dae },
	{ 0xa438, 0x13a0 }, { 0xa438, 0x0105 }, { 0xa438, 0x028b },
	{ 0xa438, 0x96ae }, { 0xa438, 0x0ba0 }, { 0xa438, 0x0205 },
	{ 0xa438, 0x028b }, { 0xa438, 0xc2ae }, { 0xa438, 0x0302 },
	{ 0xa438, 0x8c18 }, { 0xa438, 0xfc04 }, { 0xa438, 0xf8fa },
	{ 0xa438, 0xef69 }, { 0xa438, 0xfafb }, { 0xa438, 0xe080 },
	{ 0xa438, 0x15ad }, { 0xa438, 0x2343 }, { 0xa438, 0xe08f },
	{ 0xa438, 0xfdac }, { 0xa438, 0x203d }, { 0xa438, 0xe08f },
	{ 0xa438, 0xe9a0 }, { 0xa438, 0x0002 }, { 0xa438, 0xae35 },
	{ 0xa438, 0xee8f }, { 0xa438, 0xe800 }, { 0xa438, 0x028c },
	{ 0xa438, 0xc8bf }, { 0xa438, 0x8feb }, { 0xa438, 0xd819 },
	{ 0xa438, 0xd9ef }, { 0xa438, 0x64bf }, { 0xa438, 0x8fef },
	{ 0xa438, 0xd819 }, { 0xa438, 0xd9ef }, { 0xa438, 0x7402 },
	{ 0xa438, 0x73a4 }, { 0xa438, 0xad50 }, { 0xa438, 0x18ee },
	{ 0xa438, 0x8fff }, { 0xa438, 0x0102 }, { 0xa438, 0x8e1b },
	{ 0xa438, 0x0273 }, { 0xa438, 0xd7ef }, { 0xa438, 0x47e5 },
	{ 0xa438, 0x85a6 }, { 0xa438, 0xe485 }, { 0xa438, 0xa5ee },
	{ 0xa438, 0x8fe7 }, { 0xa438, 0x01ae }, { 0xa438, 0x33bf },
	{ 0xa438, 0x8f87 }, { 0xa438, 0x0274 }, { 0xa438, 0x4abf },
	{ 0xa438, 0x8f8d }, { 0xa438, 0x0274 }, { 0xa438, 0x4abf },
	{ 0xa438, 0x8f93 }, { 0xa438, 0x0274 }, { 0xa438, 0x4abf },
	{ 0xa438, 0x8f99 }, { 0xa438, 0x0274 }, { 0xa438, 0x4abf },
	{ 0xa438, 0x8f84 }, { 0xa438, 0x0274 }, { 0xa438, 0x53bf },
	{ 0xa438, 0x8f8a }, { 0xa438, 0x0274 }, { 0xa438, 0x53bf },
	{ 0xa438, 0x8f90 }, { 0xa438, 0x0274 }, { 0xa438, 0x53bf },
	{ 0xa438, 0x8f96 }, { 0xa438, 0x0274 }, { 0xa438, 0x5302 },
	{ 0xa438, 0x2261 }, { 0xa438, 0xfffe }, { 0xa438, 0xef96 },
	{ 0xa438, 0xfefc }, { 0xa438, 0x04f8 }, { 0xa438, 0xfafb },
	{ 0xa438, 0xe085 }, { 0xa438, 0xa5e1 }, { 0xa438, 0x85a6 },
	{ 0xa438, 0xef64 }, { 0xa438, 0xd000 }, { 0xa438, 0xe18f },
	{ 0xa438, 0xeaef }, { 0xa438, 0x7402 }, { 0xa438, 0x73f2 },
	{ 0xa438, 0xad50 }, { 0xa438, 0x10e0 }, { 0xa438, 0x8fe8 },
	{ 0xa438, 0xac24 }, { 0xa438, 0x06ee }, { 0xa438, 0x8fe7 },
	{ 0xa438, 0x02ae }, { 0xa438, 0x04ee }, { 0xa438, 0x8fe7 },
	{ 0xa438, 0x03ff }, { 0xa438, 0xfefc }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xf9fa }, { 0xa438, 0xef69 }, { 0xa438, 0xfb02 },
	{ 0xa438, 0x8cc8 }, { 0xa438, 0xbf8f }, { 0xa438, 0xebd8 },
	{ 0xa438, 0x19d9 }, { 0xa438, 0xbf8f }, { 0xa438, 0xf3e2 },
	{ 0xa438, 0x8fe8 }, { 0xa438, 0xef32 }, { 0xa438, 0x4b02 },
	{ 0xa438, 0x1a93 }, { 0xa438, 0xdc19 }, { 0xa438, 0xdd12 },
	{ 0xa438, 0xe68f }, { 0xa438, 0xe8e3 }, { 0xa438, 0x8fe9 },
	{ 0xa438, 0x1b23 }, { 0xa438, 0xad37 }, { 0xa438, 0x07e0 },
	{ 0xa438, 0x8fff }, { 0xa438, 0x4802 }, { 0xa438, 0xae09 },
	{ 0xa438, 0xee8f }, { 0xa438, 0xe810 }, { 0xa438, 0x1f00 },
	{ 0xa438, 0xe48f }, { 0xa438, 0xfee4 }, { 0xa438, 0x8fff },
	{ 0xa438, 0x028e }, { 0xa438, 0x1b02 }, { 0xa438, 0x73d7 },
	{ 0xa438, 0xef47 }, { 0xa438, 0xe585 }, { 0xa438, 0xa6e4 },
	{ 0xa438, 0x85a5 }, { 0xa438, 0xee8f }, { 0xa438, 0xe701 },
	{ 0xa438, 0xffef }, { 0xa438, 0x96fe }, { 0xa438, 0xfdfc },
	{ 0xa438, 0x04f8 }, { 0xa438, 0xf9fa }, { 0xa438, 0xef69 },
	{ 0xa438, 0xfafb }, { 0xa438, 0x028c }, { 0xa438, 0xc8bf },
	{ 0xa438, 0x8feb }, { 0xa438, 0xd819 }, { 0xa438, 0xd9ef },
	{ 0xa438, 0x64bf }, { 0xa438, 0x8fef }, { 0xa438, 0xd819 },
	{ 0xa438, 0xd9ef }, { 0xa438, 0x7402 }, { 0xa438, 0x73a4 },
	{ 0xa438, 0xad50 }, { 0xa438, 0x27bf }, { 0xa438, 0x8fed },
	{ 0xa438, 0xd819 }, { 0xa438, 0xd9ef }, { 0xa438, 0x64bf },
	{ 0xa438, 0x8ff1 }, { 0xa438, 0xd819 }, { 0xa438, 0xd9ef },
	{ 0xa438, 0x7402 }, { 0xa438, 0x73a4 }, { 0xa438, 0xad50 },
	{ 0xa438, 0x11e2 }, { 0xa438, 0x8fe8 }, { 0xa438, 0xe38f },
	{ 0xa438, 0xe9ef }, { 0xa438, 0x0258 }, { 0xa438, 0x0f1b },
	{ 0xa438, 0x03ac }, { 0xa438, 0x2744 }, { 0xa438, 0xae09 },
	{ 0xa438, 0xe08f }, { 0xa438, 0xfee4 }, { 0xa438, 0x8fff },
	{ 0xa438, 0x028e }, { 0xa438, 0x1b02 }, { 0xa438, 0x2261 },
	{ 0xa438, 0xee8f }, { 0xa438, 0xe700 }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x8702 }, { 0xa438, 0x744a }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x8d02 }, { 0xa438, 0x744a }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x9302 }, { 0xa438, 0x744a }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x9902 }, { 0xa438, 0x744a }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x8402 }, { 0xa438, 0x7453 }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x8a02 }, { 0xa438, 0x7453 }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x9002 }, { 0xa438, 0x7453 }, { 0xa438, 0xbf8f },
	{ 0xa438, 0x9602 }, { 0xa438, 0x7453 }, { 0xa438, 0xae1f },
	{ 0xa438, 0x12e6 }, { 0xa438, 0x8fe8 }, { 0xa438, 0xe08f },
	{ 0xa438, 0xffe4 }, { 0xa438, 0x8ffe }, { 0xa438, 0x028d },
	{ 0xa438, 0x3e02 }, { 0xa438, 0x8e1b }, { 0xa438, 0x0273 },
	{ 0xa438, 0xd7ef }, { 0xa438, 0x47e5 }, { 0xa438, 0x85a6 },
	{ 0xa438, 0xe485 }, { 0xa438, 0xa5ee }, { 0xa438, 0x8fe7 },
	{ 0xa438, 0x01ff }, { 0xa438, 0xfeef }, { 0xa438, 0x96fe },
	{ 0xa438, 0xfdfc }, { 0xa438, 0x04f8 }, { 0xa438, 0xf9fa },
	{ 0xa438, 0xef69 }, { 0xa438, 0xfafb }, { 0xa438, 0x1f22 },
	{ 0xa438, 0xee8f }, { 0xa438, 0xeb00 }, { 0xa438, 0xee8f },
	{ 0xa438, 0xec00 }, { 0xa438, 0xee8f }, { 0xa438, 0xed00 },
	{ 0xa438, 0xee8f }, { 0xa438, 0xee00 }, { 0xa438, 0x1f33 },
	{ 0xa438, 0xee8f }, { 0xa438, 0xe500 }, { 0xa438, 0xee8f },
	{ 0xa438, 0xe600 }, { 0xa438, 0xbf53 }, { 0xa438, 0x7d02 },
	{ 0xa438, 0x7662 }, { 0xa438, 0xef64 }, { 0xa438, 0xbf8f },
	{ 0xa438, 0xe5d8 }, { 0xa438, 0x19d9 }, { 0xa438, 0xef74 },
	{ 0xa438, 0x0273 }, { 0xa438, 0xbfef }, { 0xa438, 0x47dd },
	{ 0xa438, 0x89dc }, { 0xa438, 0xd1ff }, { 0xa438, 0xb1fe },
	{ 0xa438, 0x13ad }, { 0xa438, 0x3be0 }, { 0xa438, 0x0d73 },
	{ 0xa438, 0xbf8f }, { 0xa438, 0xedd8 }, { 0xa438, 0x19d9 },
	{ 0xa438, 0xef64 }, { 0xa438, 0xef47 }, { 0xa438, 0x0273 },
	{ 0xa438, 0xa4ad }, { 0xa438, 0x5003 }, { 0xa438, 0xdd89 },
	{ 0xa438, 0xdcef }, { 0xa438, 0x64bf }, { 0xa438, 0x8feb },
	{ 0xa438, 0xd819 }, { 0xa438, 0xd91a }, { 0xa438, 0x46dd },
	{ 0xa438, 0x89dc }, { 0xa438, 0x12ad }, { 0xa438, 0x32b0 },
	{ 0xa438, 0x0d42 }, { 0xa438, 0xdc19 }, { 0xa438, 0xddff },
	{ 0xa438, 0xfeef }, { 0xa438, 0x96fe }, { 0xa438, 0xfdfc },
	{ 0xa438, 0x04f8 }, { 0xa438, 0xf9fa }, { 0xa438, 0xef69 },
	{ 0xa438, 0xfafb }, { 0xa438, 0x1f22 }, { 0xa438, 0xd6ff },
	{ 0xa438, 0xffef }, { 0xa438, 0x03bf }, { 0xa438, 0x8ff3 },
	{ 0xa438, 0xef32 }, { 0xa438, 0x4b02 }, { 0xa438, 0x1a93 },
	{ 0xa438, 0xef30 }, { 0xa438, 0xd819 }, { 0xa438, 0xd9ef },
	{ 0xa438, 0x7402 }, { 0xa438, 0x73a4 }, { 0xa438, 0xac50 },
	{ 0xa438, 0x04ef }, { 0xa438, 0x32ef }, { 0xa438, 0x64e0 },
	{ 0xa438, 0x8fe9 }, { 0xa438, 0x12ef }, { 0xa438, 0x121b },
	{ 0xa438, 0x10ac }, { 0xa438, 0x2fd9 }, { 0xa438, 0xef03 },
	{ 0xa438, 0xbf8f }, { 0xa438, 0xf348 }, { 0xa438, 0x021a },
	{ 0xa438, 0x90ec }, { 0xa438, 0xff19 }, { 0xa438, 0xecff },
	{ 0xa438, 0xd001 }, { 0xa438, 0xae03 }, { 0xa438, 0x0c01 },
	{ 0xa438, 0x83a3 }, { 0xa438, 0x00fa }, { 0xa438, 0xe18f },
	{ 0xa438, 0xff1e }, { 0xa438, 0x10e5 }, { 0xa438, 0x8fff },
	{ 0xa438, 0xfffe }, { 0xa438, 0xef96 }, { 0xa438, 0xfefd },
	{ 0xa438, 0xfc04 }, { 0xa438, 0x725a }, { 0xa438, 0x725d },
	{ 0xa438, 0x7260 }, { 0xa438, 0x7263 }, { 0xa438, 0x71fa },
	{ 0xa438, 0x71fd }, { 0xa438, 0x7200 }, { 0xa438, 0x7203 },
	{ 0xa438, 0x8f4b }, { 0xa438, 0x8f4e }, { 0xa438, 0x8f51 },
	{ 0xa438, 0x8f54 }, { 0xa438, 0x8f57 }, { 0xa438, 0x8f5a },
	{ 0xa438, 0x8f5d }, { 0xa438, 0x8f60 }, { 0xa438, 0x722a },
	{ 0xa438, 0x722d }, { 0xa438, 0x7230 }, { 0xa438, 0x7233 },
	{ 0xa438, 0x721e }, { 0xa438, 0x7221 }, { 0xa438, 0x7224 },
	{ 0xa438, 0x7227 }, { 0xa438, 0x7212 }, { 0xa438, 0x7215 },
	{ 0xa438, 0x7218 }, { 0xa438, 0x721b }, { 0xa438, 0x724e },
	{ 0xa438, 0x7251 }, { 0xa438, 0x7254 }, { 0xa438, 0x7257 },
	{ 0xa438, 0x7242 }, { 0xa438, 0x7245 }, { 0xa438, 0x7248 },
	{ 0xa438, 0x724b }, { 0xa438, 0x7236 }, { 0xa438, 0x7239 },
	{ 0xa438, 0x723c }, { 0xa438, 0x723f }, { 0xa438, 0x8f84 },
	{ 0xa438, 0x8f8a }, { 0xa438, 0x8f90 }, { 0xa438, 0x8f96 },
	{ 0xa438, 0x8f9c }, { 0xa438, 0x8fa2 }, { 0xa438, 0x8fa8 },
	{ 0xa438, 0x8fae }, { 0xa438, 0x8f87 }, { 0xa438, 0x8f8d },
	{ 0xa438, 0x8f93 }, { 0xa438, 0x8f99 }, { 0xa438, 0x8f9f },
	{ 0xa438, 0x8fa5 }, { 0xa438, 0x8fab }, { 0xa438, 0x8fb1 },
	{ 0xa438, 0x8f63 }, { 0xa438, 0x8f66 }, { 0xa438, 0x8f69 },
	{ 0xa438, 0x8f6c }, { 0xa438, 0x8f6f }, { 0xa438, 0x8f72 },
	{ 0xa438, 0x8f75 }, { 0xa438, 0x8f78 }, { 0xa438, 0x8f7b },
	{ 0xa438, 0xf8f9 }, { 0xa438, 0xfaef }, { 0xa438, 0x69fa },
	{ 0xa438, 0xfbe2 }, { 0xa438, 0x8fff }, { 0xa438, 0xad30 },
	{ 0xa438, 0x06d1 }, { 0xa438, 0x00d3 }, { 0xa438, 0x00ae },
	{ 0xa438, 0x04d1 }, { 0xa438, 0x01d3 }, { 0xa438, 0x0fbf },
	{ 0xa438, 0x8d99 }, { 0xa438, 0xd700 }, { 0xa438, 0x0802 },
	{ 0xa438, 0x7677 }, { 0xa438, 0xef13 }, { 0xa438, 0xbf8d },
	{ 0xa438, 0xa1d7 }, { 0xa438, 0x0008 }, { 0xa438, 0x0276 },
	{ 0xa438, 0x77ad }, { 0xa438, 0x3106 }, { 0xa438, 0xd100 },
	{ 0xa438, 0xd300 }, { 0xa438, 0xae04 }, { 0xa438, 0xd101 },
	{ 0xa438, 0xd30f }, { 0xa438, 0xbf8d }, { 0xa438, 0xa9d7 },
	{ 0xa438, 0x0008 }, { 0xa438, 0x0276 }, { 0xa438, 0x77ef },
	{ 0xa438, 0x13bf }, { 0xa438, 0x8db1 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x0802 }, { 0xa438, 0x7677 }, { 0xa438, 0xad32 },
	{ 0xa438, 0x06d1 }, { 0xa438, 0x00d3 }, { 0xa438, 0x00ae },
	{ 0xa438, 0x04d1 }, { 0xa438, 0x01d3 }, { 0xa438, 0x03bf },
	{ 0xa438, 0x8db9 }, { 0xa438, 0xd700 }, { 0xa438, 0x1802 },
	{ 0xa438, 0x7677 }, { 0xa438, 0xef13 }, { 0xa438, 0xbf8d },
	{ 0xa438, 0xd1d7 }, { 0xa438, 0x0018 }, { 0xa438, 0x0276 },
	{ 0xa438, 0x77ad }, { 0xa438, 0x3304 }, { 0xa438, 0xd101 },
	{ 0xa438, 0xae02 }, { 0xa438, 0xd100 }, { 0xa438, 0xd300 },
	{ 0xa438, 0xbf8d }, { 0xa438, 0xe9d7 }, { 0xa438, 0x0010 },
	{ 0xa438, 0x0276 }, { 0xa438, 0x77ef }, { 0xa438, 0x13bf },
	{ 0xa438, 0x8df9 }, { 0xa438, 0xd700 }, { 0xa438, 0x1002 },
	{ 0xa438, 0x7677 }, { 0xa438, 0x1f33 }, { 0xa438, 0xe38f },
	{ 0xa438, 0xfdac }, { 0xa438, 0x3803 }, { 0xa438, 0xaf8f },
	{ 0xa438, 0x35ad }, { 0xa438, 0x3405 }, { 0xa438, 0xe18f },
	{ 0xa438, 0xfbae }, { 0xa438, 0x02d1 }, { 0xa438, 0x00bf },
	{ 0xa438, 0x8e09 }, { 0xa438, 0xd700 }, { 0xa438, 0x1202 },
	{ 0xa438, 0x7677 }, { 0xa438, 0xad35 }, { 0xa438, 0x06d1 },
	{ 0xa438, 0x01d3 }, { 0xa438, 0x04ae }, { 0xa438, 0x04d1 },
	{ 0xa438, 0x00d3 }, { 0xa438, 0x00bf }, { 0xa438, 0x6f8a },
	{ 0xa438, 0x0274 }, { 0xa438, 0x76bf }, { 0xa438, 0x6bd0 },
	{ 0xa438, 0x0274 }, { 0xa438, 0x951a }, { 0xa438, 0x13bf },
	{ 0xa438, 0x6bd0 }, { 0xa438, 0x0274 }, { 0xa438, 0x76bf },
	{ 0xa438, 0x6d2c }, { 0xa438, 0x0274 }, { 0xa438, 0x95ac },
	{ 0xa438, 0x280b }, { 0xa438, 0xbf6d }, { 0xa438, 0x2f02 },
	{ 0xa438, 0x7495 }, { 0xa438, 0xac28 }, { 0xa438, 0x02ae },
	{ 0xa438, 0x0bad }, { 0xa438, 0x3504 }, { 0xa438, 0xd101 },
	{ 0xa438, 0xae0d }, { 0xa438, 0xd10f }, { 0xa438, 0xae09 },
	{ 0xa438, 0xad35 }, { 0xa438, 0x04d1 }, { 0xa438, 0x05ae },
	{ 0xa438, 0x02d1 }, { 0xa438, 0x0fbf }, { 0xa438, 0x8f7e },
	{ 0xa438, 0x0274 }, { 0xa438, 0x76e3 }, { 0xa438, 0x8ffc },
	{ 0xa438, 0xac38 }, { 0xa438, 0x05ad }, { 0xa438, 0x3618 },
	{ 0xa438, 0xae08 }, { 0xa438, 0xbf71 }, { 0xa438, 0x9d02 },
	{ 0xa438, 0x744a }, { 0xa438, 0xae0e }, { 0xa438, 0xd102 },
	{ 0xa438, 0xbf8f }, { 0xa438, 0x8102 }, { 0xa438, 0x7476 },
	{ 0xa438, 0xbf71 }, { 0xa438, 0x9d02 }, { 0xa438, 0x7476 },
	{ 0xa438, 0xfffe }, { 0xa438, 0xef96 }, { 0xa438, 0xfefd },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xf91f }, { 0xa438, 0x33e3 },
	{ 0xa438, 0x8ffd }, { 0xa438, 0xad38 }, { 0xa438, 0x0302 },
	{ 0xa438, 0x8e1b }, { 0xa438, 0xfd04 }, { 0xa438, 0x55b0 },
	{ 0xa438, 0x2055 }, { 0xa438, 0xb0a0 }, { 0xa438, 0x55b1 },
	{ 0xa438, 0x2055 }, { 0xa438, 0xb1a0 }, { 0xa438, 0xfcb0 },
	{ 0xa438, 0x22fc }, { 0xa438, 0xb0a2 }, { 0xa438, 0xfcb1 },
	{ 0xa438, 0x22fc }, { 0xa438, 0xb1a2 }, { 0xa438, 0xfdad },
	{ 0xa438, 0xdaca }, { 0xa438, 0xadda }, { 0xa438, 0x97ad },
	{ 0xa438, 0xda64 }, { 0xa438, 0xadda }, { 0xa438, 0x20ad },
	{ 0xa438, 0xdafd }, { 0xa438, 0xaddc }, { 0xa438, 0xcaad },
	{ 0xa438, 0xdc97 }, { 0xa438, 0xaddc }, { 0xa438, 0x64ad },
	{ 0xa438, 0xdca7 }, { 0xa438, 0xbf1e }, { 0xa438, 0x20bc },
	{ 0xa438, 0x3299 }, { 0xa438, 0xadfe }, { 0xa438, 0x85ad },
	{ 0xa438, 0xfe44 }, { 0xa438, 0xadfe }, { 0xa438, 0x30ad },
	{ 0xa438, 0xfeff }, { 0xa438, 0xae00 }, { 0xa438, 0xebae },
	{ 0xa438, 0x00aa }, { 0xa438, 0xae00 }, { 0xa438, 0x96ae },
	{ 0xa438, 0x00dd }, { 0xa438, 0xad94 }, { 0xa438, 0xccad },
	{ 0xa438, 0x9499 }, { 0xa438, 0xad94 }, { 0xa438, 0x88ad },
	{ 0xa438, 0x94ff }, { 0xa438, 0xad94 }, { 0xa438, 0xeead },
	{ 0xa438, 0x94bb }, { 0xa438, 0xad94 }, { 0xa438, 0xaaad },
	{ 0xa438, 0x94f9 }, { 0xa438, 0xe28f }, { 0xa438, 0xffee },
	{ 0xa438, 0x8fff }, { 0xa438, 0x00e3 }, { 0xa438, 0x8ffd },
	{ 0xa438, 0xee8f }, { 0xa438, 0xfd01 }, { 0xa438, 0xee8f },
	{ 0xa438, 0xfc01 }, { 0xa438, 0x028e }, { 0xa438, 0x1be6 },
	{ 0xa438, 0x8fff }, { 0xa438, 0xe78f }, { 0xa438, 0xfdee },
	{ 0xa438, 0x8ffc }, { 0xa438, 0x00ee }, { 0xa438, 0x8fe7 },
	{ 0xa438, 0x00fd }, { 0xa438, 0x0400 }, { 0xa436, 0xb85e },
	{ 0xa438, 0x211c }, { 0xa436, 0xb860 }, { 0xa438, 0x216c },
	{ 0xa436, 0xb862 }, { 0xa438, 0x212b }, { 0xa436, 0xb864 },
	{ 0xa438, 0x4be8 }, { 0xa436, 0xb886 }, { 0xa438, 0x4209 },
	{ 0xa436, 0xb888 }, { 0xa438, 0x49da }, { 0xa436, 0xb88a },
	{ 0xa438, 0x085a }, { 0xa436, 0xb88c }, { 0xa438, 0x3fdf },
	{ 0xa436, 0xb838 }, { 0xa438, 0x00ff }, { 0xb820, 0x0010 },
	{ 0xa466, 0x0003 }, { 0xa436, 0x8528 }, { 0xa438, 0x0000 },
	{ 0xa436, 0x85f8 }, { 0xa438, 0xaf86 }, { 0xa438, 0x10af },
	{ 0xa438, 0x8622 }, { 0xa438, 0xaf86 }, { 0xa438, 0x4aaf },
	{ 0xa438, 0x8658 }, { 0xa438, 0xaf86 }, { 0xa438, 0x64af },
	{ 0xa438, 0x8685 }, { 0xa438, 0xaf86 }, { 0xa438, 0xc4af },
	{ 0xa438, 0x86cf }, { 0xa438, 0xa104 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x8394 }, { 0xa438, 0xad20 }, { 0xa438, 0x03af },
	{ 0xa438, 0x2b67 }, { 0xa438, 0xaf2a }, { 0xa438, 0xf0af },
	{ 0xa438, 0x2b8d }, { 0xa438, 0xbf6b }, { 0xa438, 0x7202 },
	{ 0xa438, 0x72dc }, { 0xa438, 0xa106 }, { 0xa438, 0x19e1 },
	{ 0xa438, 0x8164 }, { 0xa438, 0xbf6d }, { 0xa438, 0x5b02 },
	{ 0xa438, 0x72bd }, { 0xa438, 0x0d13 }, { 0xa438, 0xbf6d },
	{ 0xa438, 0x5802 }, { 0xa438, 0x72bd }, { 0xa438, 0x0d13 },
	{ 0xa438, 0xbf6d }, { 0xa438, 0x6a02 }, { 0xa438, 0x72bd },
	{ 0xa438, 0x0275 }, { 0xa438, 0x12af }, { 0xa438, 0x380d },
	{ 0xa438, 0x0d55 }, { 0xa438, 0x5d07 }, { 0xa438, 0xffbf },
	{ 0xa438, 0x8b09 }, { 0xa438, 0x0272 }, { 0xa438, 0x91af },
	{ 0xa438, 0x3ee2 }, { 0xa438, 0x023d }, { 0xa438, 0xffbf },
	{ 0xa438, 0x8b09 }, { 0xa438, 0x0272 }, { 0xa438, 0x9aaf },
	{ 0xa438, 0x41a6 }, { 0xa438, 0x0223 }, { 0xa438, 0x24f8 },
	{ 0xa438, 0xfaef }, { 0xa438, 0x69bf }, { 0xa438, 0x6b9c },
	{ 0xa438, 0x0272 }, { 0xa438, 0xdce0 }, { 0xa438, 0x8f7a },
	{ 0xa438, 0x1f01 }, { 0xa438, 0x9e06 }, { 0xa438, 0xe58f },
	{ 0xa438, 0x7a02 }, { 0xa438, 0x7550 }, { 0xa438, 0xef96 },
	{ 0xa438, 0xfefc }, { 0xa438, 0xaf06 }, { 0xa438, 0x8702 },
	{ 0xa438, 0x1cac }, { 0xa438, 0xf8f9 }, { 0xa438, 0xfaef },
	{ 0xa438, 0x69fb }, { 0xa438, 0xd78f }, { 0xa438, 0x97ae },
	{ 0xa438, 0x00bf }, { 0xa438, 0x6d4f }, { 0xa438, 0x0272 },
	{ 0xa438, 0x91d3 }, { 0xa438, 0x00a3 }, { 0xa438, 0x1202 },
	{ 0xa438, 0xae1b }, { 0xa438, 0xbf6d }, { 0xa438, 0x52ef },
	{ 0xa438, 0x1302 }, { 0xa438, 0x72bd }, { 0xa438, 0xef97 },
	{ 0xa438, 0xd9bf }, { 0xa438, 0x6d55 }, { 0xa438, 0x0272 },
	{ 0xa438, 0xbd17 }, { 0xa438, 0x13ae }, { 0xa438, 0xe6bf },
	{ 0xa438, 0x6d4f }, { 0xa438, 0x0272 }, { 0xa438, 0x9aff },
	{ 0xa438, 0xef96 }, { 0xa438, 0xfefd }, { 0xa438, 0xfcaf },
	{ 0xa438, 0x1c05 }, { 0xa438, 0x0000 }, { 0xa438, 0x021b },
	{ 0xa438, 0xf202 }, { 0xa438, 0x8700 }, { 0xa438, 0xaf1b },
	{ 0xa438, 0x73ad }, { 0xa438, 0x2003 }, { 0xa438, 0x0206 },
	{ 0xa438, 0x6ead }, { 0xa438, 0x2108 }, { 0xa438, 0xe280 },
	{ 0xa438, 0x51f7 }, { 0xa438, 0x30e6 }, { 0xa438, 0x8051 },
	{ 0xa438, 0xe180 }, { 0xa438, 0x421e }, { 0xa438, 0x10e5 },
	{ 0xa438, 0x8042 }, { 0xa438, 0xe0ff }, { 0xa438, 0xeee1 },
	{ 0xa438, 0x8043 }, { 0xa438, 0x1e10 }, { 0xa438, 0xe580 },
	{ 0xa438, 0x43e0 }, { 0xa438, 0xffef }, { 0xa438, 0xad20 },
	{ 0xa438, 0x04ee }, { 0xa438, 0x804f }, { 0xa438, 0x1eaf },
	{ 0xa438, 0x0661 }, { 0xa438, 0xf8fa }, { 0xa438, 0xef69 },
	{ 0xa438, 0xe080 }, { 0xa438, 0x4fac }, { 0xa438, 0x2417 },
	{ 0xa438, 0xe080 }, { 0xa438, 0x44ad }, { 0xa438, 0x241a },
	{ 0xa438, 0x0287 }, { 0xa438, 0x2fe0 }, { 0xa438, 0x8044 },
	{ 0xa438, 0xac24 }, { 0xa438, 0x11bf }, { 0xa438, 0x8b0c },
	{ 0xa438, 0x0272 }, { 0xa438, 0x9aae }, { 0xa438, 0x0902 },
	{ 0xa438, 0x88c8 }, { 0xa438, 0x028a }, { 0xa438, 0x9502 },
	{ 0xa438, 0x8a8a }, { 0xa438, 0xef96 }, { 0xa438, 0xfefc },
	{ 0xa438, 0x04f8 }, { 0xa438, 0xe08f }, { 0xa438, 0x96a0 },
	{ 0xa438, 0x0005 }, { 0xa438, 0x0288 }, { 0xa438, 0x6cae },
	{ 0xa438, 0x38a0 }, { 0xa438, 0x0105 }, { 0xa438, 0x0287 },
	{ 0xa438, 0x75ae }, { 0xa438, 0x30a0 }, { 0xa438, 0x0205 },
	{ 0xa438, 0x0287 }, { 0xa438, 0xb3ae }, { 0xa438, 0x28a0 },
	{ 0xa438, 0x0305 }, { 0xa438, 0x0287 }, { 0xa438, 0xc9ae },
	{ 0xa438, 0x20a0 }, { 0xa438, 0x0405 }, { 0xa438, 0x0287 },
	{ 0xa438, 0xd6ae }, { 0xa438, 0x18a0 }, { 0xa438, 0x0505 },
	{ 0xa438, 0x0288 }, { 0xa438, 0x1aae }, { 0xa438, 0x10a0 },
	{ 0xa438, 0x0605 }, { 0xa438, 0x0288 }, { 0xa438, 0x27ae },
	{ 0xa438, 0x08a0 }, { 0xa438, 0x0705 }, { 0xa438, 0x0288 },
	{ 0xa438, 0x48ae }, { 0xa438, 0x00fc }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xfaef }, { 0xa438, 0x69e0 }, { 0xa438, 0x8018 },
	{ 0xa438, 0xad25 }, { 0xa438, 0x2c02 }, { 0xa438, 0x8a67 },
	{ 0xa438, 0xe184 }, { 0xa438, 0x5de5 }, { 0xa438, 0x8f92 },
	{ 0xa438, 0xe58f }, { 0xa438, 0x93e5 }, { 0xa438, 0x8f94 },
	{ 0xa438, 0xe58f }, { 0xa438, 0x9502 }, { 0xa438, 0x88e6 },
	{ 0xa438, 0xe184 }, { 0xa438, 0xf759 }, { 0xa438, 0x0fe5 },
	{ 0xa438, 0x8f7b }, { 0xa438, 0xe58f }, { 0xa438, 0x7ce5 },
	{ 0xa438, 0x8f7d }, { 0xa438, 0xe58f }, { 0xa438, 0x7eee },
	{ 0xa438, 0x8f96 }, { 0xa438, 0x02ae }, { 0xa438, 0x0302 },
	{ 0xa438, 0x8a8a }, { 0xa438, 0xef96 }, { 0xa438, 0xfefc },
	{ 0xa438, 0x04f9 }, { 0xa438, 0x0289 }, { 0xa438, 0x19ac },
	{ 0xa438, 0x3009 }, { 0xa438, 0xee8f }, { 0xa438, 0x9603 },
	{ 0xa438, 0x0288 }, { 0xa438, 0x8eae }, { 0xa438, 0x04ee },
	{ 0xa438, 0x8f96 }, { 0xa438, 0x04fd }, { 0xa438, 0x04fb },
	{ 0xa438, 0x0288 }, { 0xa438, 0x55ad }, { 0xa438, 0x5004 },
	{ 0xa438, 0xee8f }, { 0xa438, 0x9602 }, { 0xa438, 0xff04 },
	{ 0xa438, 0xf902 }, { 0xa438, 0x8943 }, { 0xa438, 0xe28f },
	{ 0xa438, 0x920c }, { 0xa438, 0x245a }, { 0xa438, 0xf0e3 },
	{ 0xa438, 0x84f7 }, { 0xa438, 0x5bf0 }, { 0xa438, 0x1b23 },
	{ 0xa438, 0x9e0f }, { 0xa438, 0x028a }, { 0xa438, 0x52ee },
	{ 0xa438, 0x8f96 }, { 0xa438, 0x0502 }, { 0xa438, 0x888e },
	{ 0xa438, 0x0287 }, { 0xa438, 0xffae }, { 0xa438, 0x04ee },
	{ 0xa438, 0x8f96 }, { 0xa438, 0x06fd }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xf9fa }, { 0xa438, 0xef69 }, { 0xa438, 0xfa1f },
	{ 0xa438, 0x44d2 }, { 0xa438, 0x04bf }, { 0xa438, 0x8f7f },
	{ 0xa438, 0xdc19 }, { 0xa438, 0xdd19 }, { 0xa438, 0x829f },
	{ 0xa438, 0xf9fe }, { 0xa438, 0xef96 }, { 0xa438, 0xfefd },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xfb02 }, { 0xa438, 0x8855 },
	{ 0xa438, 0xad50 }, { 0xa438, 0x04ee }, { 0xa438, 0x8f96 },
	{ 0xa438, 0x04ff }, { 0xa438, 0x04f8 }, { 0xa438, 0xf9fa },
	{ 0xa438, 0xef69 }, { 0xa438, 0x0289 }, { 0xa438, 0x19ac },
	{ 0xa438, 0x3009 }, { 0xa438, 0xee8f }, { 0xa438, 0x9607 },
	{ 0xa438, 0x0288 }, { 0xa438, 0x8eae }, { 0xa438, 0x0702 },
	{ 0xa438, 0x8a8a }, { 0xa438, 0xee8f }, { 0xa438, 0x9601 },
	{ 0xa438, 0xef96 }, { 0xa438, 0xfefd }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xfb02 }, { 0xa438, 0x8855 }, { 0xa438, 0xad50 },
	{ 0xa438, 0x04ee }, { 0xa438, 0x8f96 }, { 0xa438, 0x06ff },
	{ 0xa438, 0x04f8 }, { 0xa438, 0xfae0 }, { 0xa438, 0x8457 },
	{ 0xa438, 0xe184 }, { 0xa438, 0x58ef }, { 0xa438, 0x64e1 },
	{ 0xa438, 0x8f90 }, { 0xa438, 0xd000 }, { 0xa438, 0xef74 },
	{ 0xa438, 0x0271 }, { 0xa438, 0xfffe }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8fa }, { 0xa438, 0xef69 }, { 0xa438, 0xee8f },
	{ 0xa438, 0x9601 }, { 0xa438, 0xee8f }, { 0xa438, 0x9004 },
	{ 0xa438, 0xee8f }, { 0xa438, 0x8f40 }, { 0xa438, 0xbf8b },
	{ 0xa438, 0x0f02 }, { 0xa438, 0x72dc }, { 0xa438, 0xe584 },
	{ 0xa438, 0x5dee }, { 0xa438, 0x8f91 }, { 0xa438, 0x77ef },
	{ 0xa438, 0x96fe }, { 0xa438, 0xfc04 }, { 0xa438, 0xf8fa },
	{ 0xa438, 0xfbef }, { 0xa438, 0x69e1 }, { 0xa438, 0x8f92 },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x0f02 }, { 0xa438, 0x72bd },
	{ 0xa438, 0xe18f }, { 0xa438, 0x93bf }, { 0xa438, 0x8b12 },
	{ 0xa438, 0x0272 }, { 0xa438, 0xbde1 }, { 0xa438, 0x8f94 },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x1502 }, { 0xa438, 0x72bd },
	{ 0xa438, 0xe18f }, { 0xa438, 0x95bf }, { 0xa438, 0x8b18 },
	{ 0xa438, 0x0272 }, { 0xa438, 0xbd02 }, { 0xa438, 0x71e4 },
	{ 0xa438, 0xef47 }, { 0xa438, 0xe484 }, { 0xa438, 0x57e5 },
	{ 0xa438, 0x8458 }, { 0xa438, 0xef96 }, { 0xa438, 0xfffe },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xf8e0 }, { 0xa438, 0x8018 },
	{ 0xa438, 0xad25 }, { 0xa438, 0x15ee }, { 0xa438, 0x8f96 },
	{ 0xa438, 0x00d0 }, { 0xa438, 0x08e4 }, { 0xa438, 0x8f92 },
	{ 0xa438, 0xe48f }, { 0xa438, 0x93e4 }, { 0xa438, 0x8f94 },
	{ 0xa438, 0xe48f }, { 0xa438, 0x9502 }, { 0xa438, 0x888e },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xf9e2 }, { 0xa438, 0x845d },
	{ 0xa438, 0xe38f }, { 0xa438, 0x910d }, { 0xa438, 0x345b },
	{ 0xa438, 0x0f1a }, { 0xa438, 0x32ac }, { 0xa438, 0x3c09 },
	{ 0xa438, 0x0c34 }, { 0xa438, 0x5bf0 }, { 0xa438, 0xe784 },
	{ 0xa438, 0xf7ae }, { 0xa438, 0x04ee }, { 0xa438, 0x84f7 },
	{ 0xa438, 0xf0e3 }, { 0xa438, 0x8f91 }, { 0xa438, 0x5b0f },
	{ 0xa438, 0x1b23 }, { 0xa438, 0xac37 }, { 0xa438, 0x0ae3 },
	{ 0xa438, 0x84f7 }, { 0xa438, 0x1e32 }, { 0xa438, 0xe784 },
	{ 0xa438, 0xf7ae }, { 0xa438, 0x00fd }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xfaef }, { 0xa438, 0x69fa }, { 0xa438, 0xfbd2 },
	{ 0xa438, 0x01d3 }, { 0xa438, 0x04d6 }, { 0xa438, 0x8f92 },
	{ 0xa438, 0xd78f }, { 0xa438, 0x7bef }, { 0xa438, 0x97d9 },
	{ 0xa438, 0xef96 }, { 0xa438, 0xd81b }, { 0xa438, 0x109e },
	{ 0xa438, 0x0480 }, { 0xa438, 0xdcd2 }, { 0xa438, 0x0016 },
	{ 0xa438, 0x1783 }, { 0xa438, 0x9fed }, { 0xa438, 0xfffe },
	{ 0xa438, 0xef96 }, { 0xa438, 0xfefc }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xf9fa }, { 0xa438, 0xfbef }, { 0xa438, 0x79fb },
	{ 0xa438, 0xcffb }, { 0xa438, 0xd200 }, { 0xa438, 0xbe00 },
	{ 0xa438, 0x00ef }, { 0xa438, 0x1229 }, { 0xa438, 0x40d0 },
	{ 0xa438, 0x041c }, { 0xa438, 0x081a }, { 0xa438, 0x10bf },
	{ 0xa438, 0x8b27 }, { 0xa438, 0x0272 }, { 0xa438, 0xbd02 },
	{ 0xa438, 0x89ee }, { 0xa438, 0xbf8f }, { 0xa438, 0x7fef },
	{ 0xa438, 0x1249 }, { 0xa438, 0x021a }, { 0xa438, 0x91d8 },
	{ 0xa438, 0x19d9 }, { 0xa438, 0xef74 }, { 0xa438, 0x0271 },
	{ 0xa438, 0xccef }, { 0xa438, 0x47dd }, { 0xa438, 0x89dc },
	{ 0xa438, 0x18a8 }, { 0xa438, 0x0002 }, { 0xa438, 0xd202 },
	{ 0xa438, 0x8990 }, { 0xa438, 0x12a2 }, { 0xa438, 0x04c8 },
	{ 0xa438, 0xffc7 }, { 0xa438, 0xffef }, { 0xa438, 0x97ff },
	{ 0xa438, 0xfefd }, { 0xa438, 0xfc04 }, { 0xa438, 0xf8f9 },
	{ 0xa438, 0xfafb }, { 0xa438, 0xef79 }, { 0xa438, 0xfbbf },
	{ 0xa438, 0x8f7f }, { 0xa438, 0xef12 }, { 0xa438, 0x4902 },
	{ 0xa438, 0x1a91 }, { 0xa438, 0xd819 }, { 0xa438, 0xd9ef },
	{ 0xa438, 0x64bf }, { 0xa438, 0x8f87 }, { 0xa438, 0xef12 },
	{ 0xa438, 0x4902 }, { 0xa438, 0x1a91 }, { 0xa438, 0xd819 },
	{ 0xa438, 0xd9ef }, { 0xa438, 0x7489 }, { 0xa438, 0x0271 },
	{ 0xa438, 0xb1ad }, { 0xa438, 0x502c }, { 0xa438, 0xef46 },
	{ 0xa438, 0xdc19 }, { 0xa438, 0xdda2 }, { 0xa438, 0x0006 },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x0f02 }, { 0xa438, 0x72dc },
	{ 0xa438, 0xa201 }, { 0xa438, 0x06bf }, { 0xa438, 0x8b12 },
	{ 0xa438, 0x0272 }, { 0xa438, 0xdca2 }, { 0xa438, 0x0206 },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x1502 }, { 0xa438, 0x72dc },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x1802 }, { 0xa438, 0x72dc },
	{ 0xa438, 0xbf8f }, { 0xa438, 0x7b1a }, { 0xa438, 0x92dd },
	{ 0xa438, 0xffef }, { 0xa438, 0x97ff }, { 0xa438, 0xfefd },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xf9f8 }, { 0xa438, 0xfbef },
	{ 0xa438, 0x79fb }, { 0xa438, 0x028a }, { 0xa438, 0xa0bf },
	{ 0xa438, 0x8b1b }, { 0xa438, 0x0272 }, { 0xa438, 0x9a16 },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x1e02 }, { 0xa438, 0x72dc },
	{ 0xa438, 0xac28 }, { 0xa438, 0x02ae }, { 0xa438, 0xf4d6 },
	{ 0xa438, 0x0000 }, { 0xa438, 0xbf8b }, { 0xa438, 0x1b02 },
	{ 0xa438, 0x7291 }, { 0xa438, 0xae03 }, { 0xa438, 0x028a },
	{ 0xa438, 0x8ad2 }, { 0xa438, 0x00d7 }, { 0xa438, 0x0000 },
	{ 0xa438, 0xe18f }, { 0xa438, 0x8f1b }, { 0xa438, 0x12a1 },
	{ 0xa438, 0x0004 }, { 0xa438, 0xef67 }, { 0xa438, 0xae1d },
	{ 0xa438, 0xef12 }, { 0xa438, 0xbf8b }, { 0xa438, 0x2102 },
	{ 0xa438, 0x72bd }, { 0xa438, 0x12bf }, { 0xa438, 0x8b24 },
	{ 0xa438, 0x0272 }, { 0xa438, 0xdcef }, { 0xa438, 0x64ad },
	{ 0xa438, 0x4f04 }, { 0xa438, 0x7eff }, { 0xa438, 0xff16 },
	{ 0xa438, 0x0271 }, { 0xa438, 0xccae }, { 0xa438, 0xd7bf },
	{ 0xa438, 0x8b2d }, { 0xa438, 0x0272 }, { 0xa438, 0x91ff },
	{ 0xa438, 0xef97 }, { 0xa438, 0xfffc }, { 0xa438, 0xfd04 },
	{ 0xa438, 0xf8fa }, { 0xa438, 0xef69 }, { 0xa438, 0xd104 },
	{ 0xa438, 0xbf8f }, { 0xa438, 0x92d8 }, { 0xa438, 0x10dc },
	{ 0xa438, 0x1981 }, { 0xa438, 0x9ff9 }, { 0xa438, 0xef96 },
	{ 0xa438, 0xfefc }, { 0xa438, 0x04f8 }, { 0xa438, 0xfbfa },
	{ 0xa438, 0xef69 }, { 0xa438, 0xbf8f }, { 0xa438, 0x87d0 },
	{ 0xa438, 0x08d1 }, { 0xa438, 0xff02 }, { 0xa438, 0x8a7c },
	{ 0xa438, 0xef96 }, { 0xa438, 0xfeff }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8fa }, { 0xa438, 0xef69 }, { 0xa438, 0xdd19 },
	{ 0xa438, 0x809f }, { 0xa438, 0xfbef }, { 0xa438, 0x96fe },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xf8e0 }, { 0xa438, 0x8044 },
	{ 0xa438, 0xf624 }, { 0xa438, 0xe480 }, { 0xa438, 0x44fc },
	{ 0xa438, 0x04f8 }, { 0xa438, 0xe080 }, { 0xa438, 0x4ff6 },
	{ 0xa438, 0x24e4 }, { 0xa438, 0x804f }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8fa }, { 0xa438, 0xfbef }, { 0xa438, 0x79fb },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x2a02 }, { 0xa438, 0x7291 },
	{ 0xa438, 0xbf8b }, { 0xa438, 0x3302 }, { 0xa438, 0x7291 },
	{ 0xa438, 0xd68b }, { 0xa438, 0x2dd7 }, { 0xa438, 0x8b30 },
	{ 0xa438, 0x0116 }, { 0xa438, 0xad50 }, { 0xa438, 0x0cbf },
	{ 0xa438, 0x8b2a }, { 0xa438, 0x0272 }, { 0xa438, 0x9abf },
	{ 0xa438, 0x8b33 }, { 0xa438, 0x0272 }, { 0xa438, 0x9aff },
	{ 0xa438, 0xef97 }, { 0xa438, 0xfffe }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8f9 }, { 0xa438, 0xfaef }, { 0xa438, 0x49f8 },
	{ 0xa438, 0xccf8 }, { 0xa438, 0xef96 }, { 0xa438, 0x0272 },
	{ 0xa438, 0x9a1f }, { 0xa438, 0x22c7 }, { 0xa438, 0xbd02 },
	{ 0xa438, 0x72dc }, { 0xa438, 0xac28 }, { 0xa438, 0x16ac },
	{ 0xa438, 0x3008 }, { 0xa438, 0x0271 }, { 0xa438, 0xe4ef },
	{ 0xa438, 0x6712 }, { 0xa438, 0xaeee }, { 0xa438, 0xd700 },
	{ 0xa438, 0x0202 }, { 0xa438, 0x71ff }, { 0xa438, 0xac50 },
	{ 0xa438, 0x05ae }, { 0xa438, 0xe3d7 }, { 0xa438, 0x0000 },
	{ 0xa438, 0xfcc4 }, { 0xa438, 0xfcef }, { 0xa438, 0x94fe },
	{ 0xa438, 0xfdfc }, { 0xa438, 0x04cc }, { 0xa438, 0xc010 },
	{ 0xa438, 0x44ac }, { 0xa438, 0x0030 }, { 0xa438, 0xbce0 },
	{ 0xa438, 0x74bc }, { 0xa438, 0xe0b8 }, { 0xa438, 0xbce0 },
	{ 0xa438, 0xfcbc }, { 0xa438, 0xe011 }, { 0xa438, 0xacb4 },
	{ 0xa438, 0xddac }, { 0xa438, 0xb6fa }, { 0xa438, 0xacb4 },
	{ 0xa438, 0xf0ac }, { 0xa438, 0xba92 }, { 0xa438, 0xacb4 },
	{ 0xa438, 0xffac }, { 0xa438, 0x5600 }, { 0xa438, 0xacb4 },
	{ 0xa438, 0xccac }, { 0xa438, 0xb6ff }, { 0xa438, 0xb034 },
	{ 0xa436, 0xb818 }, { 0xa438, 0x2ae4 }, { 0xa436, 0xb81a },
	{ 0xa438, 0x380a }, { 0xa436, 0xb81c }, { 0xa438, 0x3edd },
	{ 0xa436, 0xb81e }, { 0xa438, 0x41a3 }, { 0xa436, 0xb850 },
	{ 0xa438, 0x0684 }, { 0xa436, 0xb852 }, { 0xa438, 0x1c02 },
	{ 0xa436, 0xb878 }, { 0xa438, 0x1b70 }, { 0xa436, 0xb884 },
	{ 0xa438, 0x0633 }, { 0xa436, 0xb832 }, { 0xa438, 0x00ff },
	{ 0xa436, 0xacfc }, { 0xa438, 0x0100 }, { 0xa436, 0xacfe },
	{ 0xa438, 0x8000 }, { 0xa436, 0xad00 }, { 0xa438, 0x27ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3c67 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x000f }, { 0xa436, 0xad00 },
	{ 0xa438, 0x47ff }, { 0xa436, 0xad02 }, { 0xa438, 0x3e67 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x000f },
	{ 0xa436, 0xad00 }, { 0xa438, 0x67ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3067 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x000f }, { 0xa436, 0xad00 }, { 0xa438, 0x87ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3267 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x000f }, { 0xa436, 0xad00 },
	{ 0xa438, 0xa7ff }, { 0xa436, 0xad02 }, { 0xa438, 0x3467 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x000f },
	{ 0xa436, 0xad00 }, { 0xa438, 0xcfff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3667 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x000f }, { 0xa436, 0xad00 }, { 0xa438, 0xefff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3867 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x000f }, { 0xa436, 0xad00 },
	{ 0xa438, 0x0fff }, { 0xa436, 0xad02 }, { 0xa438, 0x3a67 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x000f },
	{ 0xa436, 0xad00 }, { 0xa438, 0x2fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3ce7 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x4fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3ee7 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x6fff }, { 0xa436, 0xad02 }, { 0xa438, 0x30e7 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x8fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x32e7 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xafff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x34e7 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x1008 }, { 0xa436, 0xad06 }, { 0xa438, 0xfff4 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0x36ff },
	{ 0xa436, 0xad04 }, { 0xa438, 0x1048 }, { 0xa436, 0xad06 },
	{ 0xa438, 0xfff5 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x38ff }, { 0xa436, 0xad04 }, { 0xa438, 0x1088 },
	{ 0xa436, 0xad06 }, { 0xa438, 0xfff6 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3aff }, { 0xa436, 0xad04 },
	{ 0xa438, 0x10c8 }, { 0xa436, 0xad06 }, { 0xa438, 0xf417 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad04 }, { 0xa438, 0x1109 }, { 0xa436, 0xad06 },
	{ 0xa438, 0xf434 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x0207 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x1149 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0455 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x2227 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x1189 }, { 0xa436, 0xad06 }, { 0xa438, 0x1476 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x4247 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x11c9 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2517 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x6267 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x1209 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3534 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x1249 }, { 0xa436, 0xad06 }, { 0xa438, 0x0555 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x2027 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x1289 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1576 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x4047 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x12c9 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2517 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x6067 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x1309 }, { 0xa436, 0xad06 }, { 0xa438, 0x3534 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x8087 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x1349 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0555 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xa0a7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x1389 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1576 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xc0c7 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x13c9 }, { 0xa436, 0xad06 }, { 0xa438, 0x2517 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xe0e7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x140b }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3534 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x0107 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0x144b },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0555 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x2127 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x148b }, { 0xa436, 0xad06 }, { 0xa438, 0x1576 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x4147 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x14cb }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2417 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x6167 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0x5109 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3434 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x8287 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x5149 }, { 0xa436, 0xad06 }, { 0xa438, 0x0455 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xa2a7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x5189 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1476 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xc2c7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x51c9 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2417 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xe2e7 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x5009 }, { 0xa436, 0xad06 }, { 0xa438, 0x3434 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x0a0f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x5049 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0455 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x2a2f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x5089 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1476 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x4a4f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x50c9 }, { 0xa436, 0xad06 }, { 0xa438, 0x2517 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x6a6f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x5209 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3534 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x080f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x5249 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0555 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x282f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x5289 }, { 0xa436, 0xad06 }, { 0xa438, 0x1576 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x484f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x52c9 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2517 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x686f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x5309 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3534 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x888f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x5349 }, { 0xa436, 0xad06 }, { 0xa438, 0x0555 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xa8af }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x5389 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1576 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xc8cf }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x53c9 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2517 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xe8ef },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x550b }, { 0xa436, 0xad06 }, { 0xa438, 0x3534 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x090f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x554b }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0555 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x292f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0x558b },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1576 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x494f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x55cb }, { 0xa436, 0xad06 }, { 0xa438, 0x2417 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x696f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x9209 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3434 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x8a8f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x9249 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0455 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xaaaf },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x9289 }, { 0xa436, 0xad06 }, { 0xa438, 0x1476 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xcacf }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x92c9 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2417 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xeaef }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x9009 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3434 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x1217 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x9049 }, { 0xa436, 0xad06 }, { 0xa438, 0x0455 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x3237 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x9089 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1476 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x5257 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x90c9 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2517 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x7277 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x9109 }, { 0xa436, 0xad06 }, { 0xa438, 0x3534 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x1017 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x9149 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0555 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x3037 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x9189 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1576 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x5057 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x91c9 }, { 0xa436, 0xad06 }, { 0xa438, 0x2517 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x7077 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x9309 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3534 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x9097 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x9349 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0555 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xb0b7 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x9389 }, { 0xa436, 0xad06 }, { 0xa438, 0x1576 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xd0d7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x93c9 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2517 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xf0f7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0x960b },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3534 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x1117 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x964b }, { 0xa436, 0xad06 }, { 0xa438, 0x0555 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x3137 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x968b }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1576 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x5157 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0x96cb },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2417 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x7177 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd309 }, { 0xa436, 0xad06 }, { 0xa438, 0x3434 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x9297 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd349 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0455 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xb2b7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0xd389 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1476 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xd2d7 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd3c9 }, { 0xa436, 0xad06 }, { 0xa438, 0x2417 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xf2f7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd009 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3434 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x1a1f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0xd049 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0455 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x3a3f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd089 }, { 0xa436, 0xad06 }, { 0xa438, 0x1476 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x5a5f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd0c9 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2517 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x7a7f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0xd109 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3534 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x181f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd149 }, { 0xa436, 0xad06 }, { 0xa438, 0x0555 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x383f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd189 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1576 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x585f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0xd1c9 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2517 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x787f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd209 }, { 0xa436, 0xad06 }, { 0xa438, 0x3534 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x989f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd249 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0555 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xb8bf }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe0 }, { 0xa436, 0xad04 }, { 0xa438, 0xd289 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1576 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xd8df },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe0 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd2c9 }, { 0xa436, 0xad06 }, { 0xa438, 0x2517 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xf8ff }, { 0xa436, 0xad02 }, { 0xa438, 0xffe0 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd70b }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3534 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x191f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0xd74b },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0555 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x393f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd78b }, { 0xa436, 0xad06 }, { 0xa438, 0x1576 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x595f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd7cb }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2417 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x797f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0x000d },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3434 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x9a9f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x004d }, { 0xa436, 0xad06 }, { 0xa438, 0x0455 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xbabf }, { 0xa436, 0xad02 }, { 0xa438, 0xffe2 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x008d }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1476 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xdadf }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe2 }, { 0xa436, 0xad04 }, { 0xa438, 0x00cd },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2c17 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xfaf8 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe2 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x400d }, { 0xa436, 0xad06 }, { 0xa438, 0x3c34 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x8187 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x404d }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0c55 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xa1a7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0x408d },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1c76 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xc1c7 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x40cd }, { 0xa436, 0xad06 }, { 0xa438, 0x2c97 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xe1e7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x800d }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3cb4 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x898f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0x804d },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0cd5 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xa9af },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x808d }, { 0xa436, 0xad06 }, { 0xa438, 0x1cf6 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xc9cf }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x80cd }, { 0xa436, 0xad06 },
	{ 0xa438, 0x2d17 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xe9ef }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0xc00d },
	{ 0xa436, 0xad06 }, { 0xa438, 0x3d34 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0x9197 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xc04d }, { 0xa436, 0xad06 }, { 0xa438, 0x0d55 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xb1b7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xc08d }, { 0xa436, 0xad06 },
	{ 0xa438, 0x1d76 }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xd1d7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0xc0cd },
	{ 0xa436, 0xad06 }, { 0xa438, 0x2d97 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xf1f7 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x3dbf },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x999f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0ddf }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xb9bf }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe1 }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0x1dff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0002 }, { 0xa436, 0xad00 }, { 0xa438, 0xd9df },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe1 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x2fff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0002 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xf9ff }, { 0xa436, 0xad02 }, { 0xa438, 0xffe1 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xad08 }, { 0xa438, 0x0002 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xd7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe7 }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xf7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe7 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x17ff }, { 0xa436, 0xad02 }, { 0xa438, 0xffe7 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x37ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3d67 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x57ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3f67 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x77ff }, { 0xa436, 0xad02 }, { 0xa438, 0x3167 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x97ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3367 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xb7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3567 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xdfff }, { 0xa436, 0xad02 }, { 0xa438, 0x3767 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3967 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x1fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x3b67 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xad02 }, { 0xa438, 0x3de7 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x5fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3fe7 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x7fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x31e7 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x9fff }, { 0xa436, 0xad02 }, { 0xa438, 0x33e7 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xbfff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x35e7 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x07ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x37e6 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x27ff }, { 0xa436, 0xad02 }, { 0xa438, 0x39e6 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x47ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x3be6 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x67ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x2066 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x07ff }, { 0xa436, 0xad02 }, { 0xa438, 0x2264 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x27ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x2464 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x47ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x2664 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x67ff }, { 0xa436, 0xad02 }, { 0xa438, 0x0064 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x87ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0264 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xa7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x0464 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xc7ff }, { 0xa436, 0xad02 }, { 0xa438, 0x0664 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xe7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0864 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x07ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x0a65 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x27ff }, { 0xa436, 0xad02 }, { 0xa438, 0x0c65 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x47ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0e65 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x67ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x1065 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x87ff }, { 0xa436, 0xad02 }, { 0xa438, 0x1266 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xa7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x1466 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xc7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x1666 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xe7ff }, { 0xa436, 0xad02 }, { 0xa438, 0x2866 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x0fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x2a66 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x2fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x2c66 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x4fff }, { 0xa436, 0xad02 }, { 0xa438, 0x2e66 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x6fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x20e6 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x0fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x22e4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x2fff }, { 0xa436, 0xad02 }, { 0xa438, 0x24e4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x4fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x26e4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x6fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x00e4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x8fff }, { 0xa436, 0xad02 }, { 0xa438, 0x02e4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xafff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x04e4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xcfff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x06e4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xefff }, { 0xa436, 0xad02 }, { 0xa438, 0x08e4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x0fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0ae5 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x2fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x0ce5 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x4fff }, { 0xa436, 0xad02 }, { 0xa438, 0x0ee5 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x6fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x10e5 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x8fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x12e6 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xafff }, { 0xa436, 0xad02 }, { 0xa438, 0x14e6 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xcfff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x16e6 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xefff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x28e6 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x17ff }, { 0xa436, 0xad02 }, { 0xa438, 0x2ae6 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x37ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x2ce6 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x57ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x2ee6 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x77ff }, { 0xa436, 0xad02 }, { 0xa438, 0x2166 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x17ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x2364 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x37ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x2564 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x57ff }, { 0xa436, 0xad02 }, { 0xa438, 0x2764 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x77ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0164 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x97ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x0364 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xb7ff }, { 0xa436, 0xad02 }, { 0xa438, 0x0564 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xd7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0764 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xf7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x0964 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x17ff }, { 0xa436, 0xad02 }, { 0xa438, 0x0b65 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x37ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0d65 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x57ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x0f65 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x77ff }, { 0xa436, 0xad02 }, { 0xa438, 0x1165 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x97ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x1366 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xb7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x1566 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xd7ff }, { 0xa436, 0xad02 }, { 0xa438, 0x1766 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xf7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x2966 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x1fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x2b66 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xad02 }, { 0xa438, 0x2d66 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x5fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x2f66 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x7fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x21e6 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x1fff }, { 0xa436, 0xad02 }, { 0xa438, 0x23e4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x3fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x25e4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x5fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x27e4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x7fff }, { 0xa436, 0xad02 }, { 0xa438, 0x01e4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x9fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x03e4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xbfff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x05e4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xdfff }, { 0xa436, 0xad02 }, { 0xa438, 0x07e4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x09e4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x1fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x0be5 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xad02 }, { 0xa438, 0x0de5 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x5fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x0fe5 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x7fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x11e5 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x9fff }, { 0xa436, 0xad02 }, { 0xa438, 0x13e6 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xbfff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x15e6 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xdfff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x17e6 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0x29e6 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x87ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x2be5 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xa7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x2de5 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xc7ff }, { 0xa436, 0xad02 }, { 0xa438, 0x2fe5 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xe7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x1865 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x8fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x1a65 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xafff }, { 0xa436, 0xad02 }, { 0xa438, 0x1c65 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xcfff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x1e65 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xefff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x18e5 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x97ff }, { 0xa436, 0xad02 }, { 0xa438, 0x1ae5 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xb7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x1ce5 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xd7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x1ee5 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xf7ff }, { 0xa436, 0xad02 }, { 0xa438, 0x1965 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x9fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x1b65 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xbfff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x1d65 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xdfff }, { 0xa436, 0xad02 }, { 0xa438, 0x1f65 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x19e5 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x07ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x1b9c }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x27ff }, { 0xa436, 0xad02 }, { 0xa438, 0x1d9c },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x0fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x1f9c }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x2fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x589c }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x17ff }, { 0xa436, 0xad02 }, { 0xa438, 0x5c9c },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x37ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x599c }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x1fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x5d9c }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xad02 }, { 0xa438, 0x5a9c },
	{ 0xa436, 0xad04 }, { 0xa438, 0x100e }, { 0xa436, 0xad06 },
	{ 0xa438, 0xfff6 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x5eff }, { 0xa436, 0xad04 }, { 0xa438, 0x104e },
	{ 0xa436, 0xad06 }, { 0xa438, 0xfff7 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x5bff }, { 0xa436, 0xad04 },
	{ 0xa438, 0x110e }, { 0xa436, 0xad06 }, { 0xa438, 0xfff6 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0x5fff },
	{ 0xa436, 0xad04 }, { 0xa438, 0x114e }, { 0xa436, 0xad06 },
	{ 0xa438, 0xf817 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad04 }, { 0xa438, 0x120f },
	{ 0xa436, 0xad06 }, { 0xa438, 0xf836 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xc3c7 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x124f }, { 0xa436, 0xad06 }, { 0xa438, 0x0997 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xe3e7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x130f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x19b6 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x0307 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0x134f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x4917 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0x2327 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x510f }, { 0xa436, 0xad06 }, { 0xa438, 0x5936 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x4347 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x514f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0997 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x6367 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0x500f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x19b6 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0x8387 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x504f }, { 0xa436, 0xad06 }, { 0xa438, 0x4817 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xa3a7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x520f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x5836 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xcbcf }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0x524f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0997 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0xebef },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x530f }, { 0xa436, 0xad06 }, { 0xa438, 0x19b6 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x0b0f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x534f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x4917 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x2b2f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0x920f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x5936 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0x4b4f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x924f }, { 0xa436, 0xad06 }, { 0xa438, 0x0997 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x6b6f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x900f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x19b6 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x8b8f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0x904f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x4817 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0xabaf },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x910f }, { 0xa436, 0xad06 }, { 0xa438, 0x5836 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xd3d7 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0x914f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x0997 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xf3f7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0x930f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x19b6 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0x1317 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0x934f }, { 0xa436, 0xad06 }, { 0xa438, 0x4917 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x3337 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd30f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x5936 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x5357 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0xd34f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x0997 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0x7377 },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd00f }, { 0xa436, 0xad06 }, { 0xa438, 0x19b6 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x9397 }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd04f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x4817 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xb3b7 }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0xd10f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x5836 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0xdbdf },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xd14f }, { 0xa436, 0xad06 }, { 0xa438, 0x0997 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xfbff }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xd20f }, { 0xa436, 0xad06 },
	{ 0xa438, 0x19b6 }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x1b1f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0xd24f },
	{ 0xa436, 0xad06 }, { 0xa438, 0x4917 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0x3b3f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x593f },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x5b5f }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x099f }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x7b7f }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffe3 }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0x19bf }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0004 }, { 0xa436, 0xad00 }, { 0xa438, 0x9b9f },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffe3 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x4fff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0004 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xbbbf }, { 0xa436, 0xad02 }, { 0xa438, 0xffe3 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x5fff }, { 0xa436, 0xad08 }, { 0xa438, 0x0004 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x07ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffa4 }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x27ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffa4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x47ff }, { 0xa436, 0xad02 }, { 0xa438, 0xffa4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x67ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x58a4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x0fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x5ca4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x2fff }, { 0xa436, 0xad02 }, { 0xa438, 0x50a4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x4fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x54a4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x6fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x59a4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x17ff }, { 0xa436, 0xad02 }, { 0xa438, 0x5da4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x37ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x51a4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x57ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x55a4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x77ff }, { 0xa436, 0xad02 }, { 0xa438, 0x5aa4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x1fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x5ea4 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0x3fff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x52a4 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x5fff }, { 0xa436, 0xad02 }, { 0xa438, 0x56a4 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0x7fff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x5ba4 }, { 0xa436, 0xad04 }, { 0xa438, 0x2a06 },
	{ 0xa436, 0xad06 }, { 0xa438, 0xfff6 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0x5fff }, { 0xa436, 0xad04 },
	{ 0xa438, 0x2b06 }, { 0xa436, 0xad06 }, { 0xa438, 0xfff7 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0x53ff },
	{ 0xa436, 0xad04 }, { 0xa438, 0x2a06 }, { 0xa436, 0xad06 },
	{ 0xa438, 0xfff4 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0x57ff }, { 0xa436, 0xad04 }, { 0xa438, 0x2b06 },
	{ 0xa436, 0xad06 }, { 0xa438, 0xf615 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0xf63f },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x069f }, { 0xa436, 0xad08 }, { 0xa438, 0x0003 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0x16bf }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0003 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x4fff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0003 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xd0ff },
	{ 0xa436, 0xad04 }, { 0xa438, 0x6a46 }, { 0xa436, 0xad06 },
	{ 0xa438, 0x5ff6 }, { 0xa436, 0xad08 }, { 0xa438, 0x0003 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xd4ff }, { 0xa436, 0xad04 }, { 0xa438, 0x6b46 },
	{ 0xa436, 0xad06 }, { 0xa438, 0xfff7 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xd8ff }, { 0xa436, 0xad04 },
	{ 0xa438, 0x6a46 }, { 0xa436, 0xad06 }, { 0xa438, 0xfff4 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xdcff },
	{ 0xa436, 0xad04 }, { 0xa438, 0x6b46 }, { 0xa436, 0xad06 },
	{ 0xa438, 0xf615 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0xf63f }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x069f },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0003 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x16bf }, { 0xa436, 0xad08 }, { 0xa438, 0x0003 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0x4fff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0003 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xd1ff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xaa86 }, { 0xa436, 0xad06 }, { 0xa438, 0x5ff6 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0003 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xd5ff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xab86 }, { 0xa436, 0xad06 },
	{ 0xa438, 0xfff7 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xd9ff }, { 0xa436, 0xad04 }, { 0xa438, 0xaa86 },
	{ 0xa436, 0xad06 }, { 0xa438, 0xfff4 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xddff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xab86 }, { 0xa436, 0xad06 }, { 0xa438, 0xf615 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0xf63f }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0x069f }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0003 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x16bf },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0003 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x4fff }, { 0xa436, 0xad08 }, { 0xa438, 0x0003 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xd2ff }, { 0xa436, 0xad04 }, { 0xa438, 0xeac6 },
	{ 0xa436, 0xad06 }, { 0xa438, 0x5ff6 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0003 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xd6ff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xebc6 }, { 0xa436, 0xad06 }, { 0xa438, 0xfff7 },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0007 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xdaff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xeac6 }, { 0xa436, 0xad06 },
	{ 0xa438, 0xfff4 }, { 0xa436, 0xad08 }, { 0xa438, 0x0007 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xdeff }, { 0xa436, 0xad04 }, { 0xa438, 0xebc6 },
	{ 0xa436, 0xad06 }, { 0xa438, 0xf615 }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0007 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0xf63f },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0017 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad02 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xffff }, { 0xa436, 0xad06 },
	{ 0xa438, 0x069f }, { 0xa436, 0xad08 }, { 0xa438, 0x0013 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xffff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad04 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad06 }, { 0xa438, 0x16bf }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0013 }, { 0xa436, 0xad00 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xffff }, { 0xa436, 0xad04 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad06 }, { 0xa438, 0x4fff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0013 }, { 0xa436, 0xad00 },
	{ 0xa438, 0xfffa }, { 0xa436, 0xad02 }, { 0xa438, 0xd3ff },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0x5fff }, { 0xa436, 0xad08 }, { 0xa438, 0x0013 },
	{ 0xa436, 0xad00 }, { 0xa438, 0xc7ff }, { 0xa436, 0xad02 },
	{ 0xa438, 0xd7e7 }, { 0xa436, 0xad04 }, { 0xa438, 0xfffe },
	{ 0xa436, 0xad06 }, { 0xa438, 0xffff }, { 0xa436, 0xad08 },
	{ 0xa438, 0x0017 }, { 0xa436, 0xad00 }, { 0xa438, 0xe7ff },
	{ 0xa436, 0xad02 }, { 0xa438, 0xdbe7 }, { 0xa436, 0xad04 },
	{ 0xa438, 0xfffe }, { 0xa436, 0xad06 }, { 0xa438, 0xffff },
	{ 0xa436, 0xad08 }, { 0xa438, 0x0017 }, { 0xa436, 0xad00 },
	{ 0xa438, 0x07ff }, { 0xa436, 0xad02 }, { 0xa438, 0xdfe7 },
	{ 0xa436, 0xad04 }, { 0xa438, 0xfffe }, { 0xa436, 0xad06 },
	{ 0xa438, 0xffff }, { 0xa436, 0xad08 }, { 0xa438, 0x0017 },
	{ 0xa436, 0xacfc }, { 0xa438, 0x0000 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x2000 }, { 0xa436, 0xacce }, { 0xa438, 0x6000 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x2001 }, { 0xa436, 0xacce },
	{ 0xa438, 0x6008 }, { 0xa436, 0xaccc }, { 0xa438, 0x2002 },
	{ 0xa436, 0xacce }, { 0xa438, 0x6010 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x2003 }, { 0xa436, 0xacce }, { 0xa438, 0x6020 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x2004 }, { 0xa436, 0xacce },
	{ 0xa438, 0x6060 }, { 0xa436, 0xaccc }, { 0xa438, 0x2005 },
	{ 0xa436, 0xacce }, { 0xa438, 0x60a0 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x2006 }, { 0xa436, 0xacce }, { 0xa438, 0x60e0 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x2007 }, { 0xa436, 0xacce },
	{ 0xa438, 0x6128 }, { 0xa436, 0xaccc }, { 0xa438, 0x2008 },
	{ 0xa436, 0xacce }, { 0xa438, 0x6178 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x2009 }, { 0xa436, 0xacce }, { 0xa438, 0x61a8 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x200a }, { 0xa436, 0xacce },
	{ 0xa438, 0x61f0 }, { 0xa436, 0xaccc }, { 0xa438, 0x200b },
	{ 0xa436, 0xacce }, { 0xa438, 0x6248 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x200c }, { 0xa436, 0xacce }, { 0xa438, 0x6258 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x200d }, { 0xa436, 0xacce },
	{ 0xa438, 0x6268 }, { 0xa436, 0xaccc }, { 0xa438, 0x200e },
	{ 0xa436, 0xacce }, { 0xa438, 0x6270 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x200f }, { 0xa436, 0xacce }, { 0xa438, 0x6274 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x2010 }, { 0xa436, 0xacce },
	{ 0xa438, 0x627c }, { 0xa436, 0xaccc }, { 0xa438, 0x2011 },
	{ 0xa436, 0xacce }, { 0xa438, 0x6284 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x2012 }, { 0xa436, 0xacce }, { 0xa438, 0x6294 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x2013 }, { 0xa436, 0xacce },
	{ 0xa438, 0x629c }, { 0xa436, 0xaccc }, { 0xa438, 0x2014 },
	{ 0xa436, 0xacce }, { 0xa438, 0x62ac }, { 0xa436, 0xaccc },
	{ 0xa438, 0x2015 }, { 0xa436, 0xacce }, { 0xa438, 0x62bc },
	{ 0xa436, 0xaccc }, { 0xa438, 0x2016 }, { 0xa436, 0xacce },
	{ 0xa438, 0x62c4 }, { 0xa436, 0xaccc }, { 0xa438, 0x2017 },
	{ 0xa436, 0xacce }, { 0xa438, 0x7000 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x2018 }, { 0xa436, 0xacce }, { 0xa438, 0x6000 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x2019 }, { 0xa436, 0xacce },
	{ 0xa438, 0x6000 }, { 0xa436, 0xaccc }, { 0xa438, 0x201a },
	{ 0xa436, 0xacce }, { 0xa438, 0x6000 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x201b }, { 0xa436, 0xacce }, { 0xa438, 0x6000 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x201c }, { 0xa436, 0xacce },
	{ 0xa438, 0x6000 }, { 0xa436, 0xaccc }, { 0xa438, 0x201d },
	{ 0xa436, 0xacce }, { 0xa438, 0x6000 }, { 0xa436, 0xaccc },
	{ 0xa438, 0x201e }, { 0xa436, 0xacce }, { 0xa438, 0x6000 },
	{ 0xa436, 0xaccc }, { 0xa438, 0x201f }, { 0xa436, 0xacce },
	{ 0xa438, 0x6000 }, { 0xa436, 0xacce }, { 0xa438, 0x0000 },
	{ 0xa436, 0x0000 }, { 0xa438, 0x0000 }, { 0xb82e, 0x0000 },
	{ 0xa436, 0x8023 }, { 0xa438, 0x0000 }, { 0xa436, 0x801e },
	{ 0xa438, 0x0027 }, { 0xb820, 0x0000 }, { 0xb892, 0x0000 },
	{ 0xb88e, 0xc15c }, { 0xb890, 0x0303 }, { 0xb890, 0x0506 },
	{ 0xb890, 0x0807 }, { 0xb890, 0x090b }, { 0xb890, 0x0e12 },
	{ 0xb890, 0x1617 }, { 0xb890, 0x1c24 }, { 0xb890, 0x2b37 },
	{ 0xb890, 0x0203 }, { 0xb890, 0x0304 }, { 0xb890, 0x0504 },
	{ 0xb890, 0x0506 }, { 0xb890, 0x0708 }, { 0xb890, 0x090a },
	{ 0xb890, 0x0b0e }, { 0xb890, 0x1013 }, { 0xb890, 0x1519 },
	{ 0xb890, 0x1d22 }, { 0xb890, 0x282e }, { 0xb890, 0x363e },
	{ 0xb890, 0x474b }, { 0xb88e, 0xc196 }, { 0xb890, 0x3f5e },
	{ 0xb890, 0xf834 }, { 0xb890, 0x6c01 }, { 0xb890, 0xa67f },
	{ 0xb890, 0xa06c }, { 0xb890, 0x043b }, { 0xb890, 0x6190 },
	{ 0xb890, 0x88db }, { 0xb890, 0x9ecd }, { 0xb890, 0x4dbc },
	{ 0xb890, 0x6e0e }, { 0xb890, 0x9f2d }, { 0xb890, 0x2c18 },
	{ 0xb890, 0x5e8c }, { 0xb890, 0x5bfe }, { 0xb890, 0x183c },
	{ 0xb890, 0x23c9 }, { 0xb890, 0x3e84 }, { 0xb890, 0x3c20 },
	{ 0xb890, 0xcc56 }, { 0xb890, 0x3480 }, { 0xb890, 0x0040 },
	{ 0xb88e, 0xc00f }, { 0xb890, 0x3502 }, { 0xb890, 0x0203 },
	{ 0xb890, 0x0303 }, { 0xb890, 0x0404 }, { 0xb890, 0x0506 },
	{ 0xb890, 0x0607 }, { 0xb890, 0x080a }, { 0xb890, 0x0b0d },
	{ 0xb890, 0x0e10 }, { 0xb890, 0x1114 }, { 0xb890, 0x171b },
	{ 0xb890, 0x1f22 }, { 0xb890, 0x2832 }, { 0xb890, 0x0101 },
	{ 0xb890, 0x0101 }, { 0xb890, 0x0202 }, { 0xb890, 0x0303 },
	{ 0xb890, 0x0404 }, { 0xb890, 0x0506 }, { 0xb890, 0x0709 },
	{ 0xb890, 0x0a0d }, { 0xb88e, 0xc047 }, { 0xb890, 0x365f },
	{ 0xb890, 0xbe10 }, { 0xb890, 0x84e4 }, { 0xb890, 0x60e9 },
	{ 0xb890, 0xa86a }, { 0xb890, 0xf1e3 }, { 0xb890, 0xf73f },
	{ 0xb890, 0x5c02 }, { 0xb890, 0x9547 }, { 0xb890, 0xc30c },
	{ 0xb890, 0xb064 }, { 0xb890, 0x079a }, { 0xb890, 0x1e23 },
	{ 0xb890, 0x1b5d }, { 0xb890, 0x92e7 }, { 0xb890, 0x4baf },
	{ 0xb890, 0x2386 }, { 0xb890, 0x01b6 }, { 0xb890, 0x6f82 },
	{ 0xb890, 0xdc1c }, { 0xb890, 0x8c92 }, { 0xb88e, 0xc110 },
	{ 0xb890, 0x0c7f }, { 0xb890, 0x1014 }, { 0xb890, 0x231d },
	{ 0xb890, 0x2023 }, { 0xb890, 0x2628 }, { 0xb890, 0x2a2d },
	{ 0xb890, 0x2d2c }, { 0xb890, 0x2c2e }, { 0xb890, 0x320d },
	{ 0xb88e, 0xc186 }, { 0xb890, 0x0306 }, { 0xb890, 0x0804 },
	{ 0xb890, 0x0406 }, { 0xb890, 0x0707 }, { 0xb890, 0x0709 },
	{ 0xb890, 0x0b0f }, { 0xb890, 0x161d }, { 0xb890, 0x202a },
	{ 0xb890, 0x3f5e }, { 0xb88e, 0xc1c1 }, { 0xb890, 0x0040 },
	{ 0xb890, 0x5920 }, { 0xb890, 0x88cd }, { 0xb890, 0x1ca1 },
	{ 0xb890, 0x3d20 }, { 0xb890, 0x3ae4 }, { 0xb890, 0x6a43 },
	{ 0xb890, 0x30af }, { 0xb890, 0xdd16 }, { 0xb88e, 0xc283 },
	{ 0xb890, 0x1611 }, { 0xb890, 0x161c }, { 0xb890, 0x2127 },
	{ 0xb890, 0x2c32 }, { 0xb890, 0x373d }, { 0xb890, 0x4247 },
	{ 0xb890, 0x4d52 }, { 0xb890, 0x585a }, { 0xb890, 0x0004 },
	{ 0xb890, 0x080c }, { 0xb890, 0x1014 }, { 0xb890, 0x181b },
	{ 0xb890, 0x1f23 }, { 0xb890, 0x272b }, { 0xb890, 0x2f33 },
	{ 0xb890, 0x363a }, { 0xb890, 0x3e42 }, { 0xb890, 0x464a },
	{ 0xb890, 0x4d51 }, { 0xb890, 0x5559 }, { 0xb890, 0x5d65 },
	{ 0xb890, 0xe769 }, { 0xb890, 0xeb56 }, { 0xb890, 0xc04b },
	{ 0xb890, 0xd502 }, { 0xb890, 0x2fb1 }, { 0xb890, 0x33b5 },
	{ 0xb890, 0x37f8 }, { 0xb890, 0xbb98 }, { 0xb890, 0x7450 },
	{ 0xb890, 0x4c48 }, { 0xb890, 0x12dc }, { 0xb890, 0xdcdc },
	{ 0xb890, 0x934a }, { 0xb890, 0x3e33 }, { 0xb890, 0xe496 },
	{ 0xb890, 0x724e }, { 0xb890, 0x2b07 }, { 0xb890, 0xe4c0 },
	{ 0xb890, 0x9c79 }, { 0xb890, 0x5512 }, { 0xb88e, 0xc212 },
	{ 0xb890, 0x2020 }, { 0xb890, 0x2020 }, { 0xb890, 0x2020 },
	{ 0xb890, 0x2020 }, { 0xb890, 0x2020 }, { 0xb890, 0x2019 },
	{ 0xb88e, 0xc24d }, { 0xb890, 0x8400 }, { 0xb890, 0x0000 },
	{ 0xb890, 0x0000 }, { 0xb890, 0x0000 }, { 0xb890, 0x0000 },
	{ 0xb890, 0x0000 }, { 0xb88e, 0xc2d3 }, { 0xb890, 0x5524 },
	{ 0xb890, 0x2526 }, { 0xb890, 0x2728 }, { 0xb88e, 0xc2e3 },
	{ 0xb890, 0x3323 }, { 0xb890, 0x2324 }, { 0xb890, 0x2425 }
}, mac_r26_2_mcu[] = {
	{ 0xa436, 0x8023 }, { 0xa438, 0x4701 }, { 0xa436, 0xb82e },
	{ 0xa438, 0x0001 }, { 0xb820, 0x0090 }, { 0xa436, 0xa016 },
	{ 0xa438, 0x0000 }, { 0xa436, 0xa012 }, { 0xa438, 0x0000 },
	{ 0xa436, 0xa014 }, { 0xa438, 0x1800 }, { 0xa438, 0x8010 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x802a }, { 0xa438, 0x1800 }, { 0xa438, 0x8032 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x803a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x803e }, { 0xa438, 0x1800 }, { 0xa438, 0x8044 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x804b }, { 0xa438, 0xd504 },
	{ 0xa438, 0xc9b5 }, { 0xa438, 0xd500 }, { 0xa438, 0xd707 },
	{ 0xa438, 0x4070 }, { 0xa438, 0x1800 }, { 0xa438, 0x1082 },
	{ 0xa438, 0xd504 }, { 0xa438, 0x1800 }, { 0xa438, 0x107a },
	{ 0xa438, 0x61d0 }, { 0xa438, 0xd701 }, { 0xa438, 0x60a5 },
	{ 0xa438, 0xd504 }, { 0xa438, 0xc9b2 }, { 0xa438, 0xd500 },
	{ 0xa438, 0xf004 }, { 0xa438, 0xd504 }, { 0xa438, 0xc9b1 },
	{ 0xa438, 0xd500 }, { 0xa438, 0xd707 }, { 0xa438, 0x6070 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x10b0 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x10c5 }, { 0xa438, 0xd707 }, { 0xa438, 0x2005 },
	{ 0xa438, 0x8030 }, { 0xa438, 0xd75e }, { 0xa438, 0x1800 },
	{ 0xa438, 0x138c }, { 0xa438, 0x1800 }, { 0xa438, 0x13ff },
	{ 0xa438, 0xc504 }, { 0xa438, 0xce20 }, { 0xa438, 0xcf01 },
	{ 0xa438, 0xd70a }, { 0xa438, 0x4005 }, { 0xa438, 0xcf02 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x1b99 }, { 0xa438, 0xa980 },
	{ 0xa438, 0xd500 }, { 0xa438, 0x1800 }, { 0xa438, 0x144d },
	{ 0xa438, 0x907f }, { 0xa438, 0x91a3 }, { 0xa438, 0x9306 },
	{ 0xa438, 0xb118 }, { 0xa438, 0x1800 }, { 0xa438, 0x2147 },
	{ 0xa438, 0x907f }, { 0xa438, 0x9209 }, { 0xa438, 0x91a3 },
	{ 0xa438, 0x9306 }, { 0xa438, 0xb118 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x203c }, { 0xa438, 0xd707 }, { 0xa438, 0x4121 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x40fc }, { 0xa438, 0xd70a },
	{ 0xa438, 0x40b5 }, { 0xa438, 0xd028 }, { 0xa438, 0xd1c1 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8057 }, { 0xa438, 0xd07b },
	{ 0xa438, 0xd1c5 }, { 0xa438, 0xd503 }, { 0xa438, 0xa108 },
	{ 0xa438, 0xd505 }, { 0xa438, 0x8103 }, { 0xa438, 0xd504 },
	{ 0xa438, 0xa002 }, { 0xa438, 0xa302 }, { 0xa438, 0xd707 },
	{ 0xa438, 0x4061 }, { 0xa438, 0xd503 }, { 0xa438, 0x8b01 },
	{ 0xa438, 0xd500 }, { 0xa438, 0xc48a }, { 0xa438, 0xd503 },
	{ 0xa438, 0xcc09 }, { 0xa438, 0xcd58 }, { 0xa438, 0xaf01 },
	{ 0xa438, 0xd500 }, { 0xa438, 0xbe10 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1739 }, { 0xa438, 0xd719 }, { 0xa438, 0x606c },
	{ 0xa438, 0xd704 }, { 0xa438, 0x645c }, { 0xa438, 0xd75e },
	{ 0xa438, 0x604d }, { 0xa438, 0xfff8 }, { 0xa438, 0x9e10 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1739 }, { 0xa438, 0xd719 },
	{ 0xa438, 0x606c }, { 0xa438, 0xd704 }, { 0xa438, 0x631c },
	{ 0xa438, 0xd75e }, { 0xa438, 0x404d }, { 0xa438, 0xfff8 },
	{ 0xa438, 0xd504 }, { 0xa438, 0xaa18 }, { 0xa438, 0xa001 },
	{ 0xa438, 0xa1e0 }, { 0xa438, 0xd500 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1739 }, { 0xa438, 0xd719 }, { 0xa438, 0x7fac },
	{ 0xa438, 0xd504 }, { 0xa438, 0xa001 }, { 0xa438, 0xd500 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1739 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x5f5c }, { 0xa438, 0xd719 }, { 0xa438, 0x3aaf },
	{ 0xa438, 0x8091 }, { 0xa438, 0xf016 }, { 0xa438, 0xd707 },
	{ 0xa438, 0x6121 }, { 0xa438, 0x1000 }, { 0xa438, 0x16d8 },
	{ 0xa438, 0xd503 }, { 0xa438, 0xcd59 }, { 0xa438, 0xaf01 },
	{ 0xa438, 0xd500 }, { 0xa438, 0x1800 }, { 0xa438, 0x0ddc },
	{ 0xa438, 0xd503 }, { 0xa438, 0x8040 }, { 0xa438, 0xd500 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x16d8 }, { 0xa438, 0xd503 },
	{ 0xa438, 0xcd5a }, { 0xa438, 0xaf01 }, { 0xa438, 0xd500 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0dbf }, { 0xa438, 0xd504 },
	{ 0xa438, 0xa008 }, { 0xa438, 0xa204 }, { 0xa438, 0xd500 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1739 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x5fa0 }, { 0xa438, 0xd503 }, { 0xa438, 0xa082 },
	{ 0xa438, 0xd500 }, { 0xa438, 0xd71e }, { 0xa438, 0x4097 },
	{ 0xa438, 0xd078 }, { 0xa438, 0xd1aa }, { 0xa438, 0xf003 },
	{ 0xa438, 0xd078 }, { 0xa438, 0xd1aa }, { 0xa438, 0xd707 },
	{ 0xa438, 0x40c1 }, { 0xa438, 0xd706 }, { 0xa438, 0x409c },
	{ 0xa438, 0xd70a }, { 0xa438, 0x4055 }, { 0xa438, 0xf010 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x6065 }, { 0xa438, 0xcc89 },
	{ 0xa438, 0xf002 }, { 0xa438, 0xcc8b }, { 0xa438, 0x1000 },
	{ 0xa438, 0x0b7b }, { 0xa438, 0xd705 }, { 0xa438, 0x2ad0 },
	{ 0xa438, 0x80ca }, { 0xa438, 0xf003 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x0b81 }, { 0xa438, 0x1000 }, { 0xa438, 0x0b87 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x0c53 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x12d7 }, { 0xa436, 0xa026 }, { 0xa438, 0x125d },
	{ 0xa436, 0xa024 }, { 0xa438, 0x2033 }, { 0xa436, 0xa022 },
	{ 0xa438, 0x213f }, { 0xa436, 0xa020 }, { 0xa438, 0x144c },
	{ 0xa436, 0xa006 }, { 0xa438, 0x1b98 }, { 0xa436, 0xa004 },
	{ 0xa438, 0x138b }, { 0xa436, 0xa002 }, { 0xa438, 0x10c4 },
	{ 0xa436, 0xa000 }, { 0xa438, 0x1079 }, { 0xa436, 0xa008 },
	{ 0xa438, 0xff00 }, { 0xa436, 0xa016 }, { 0xa438, 0x0000 },
	{ 0xa436, 0xa012 }, { 0xa438, 0x0ff8 }, { 0xa436, 0xa014 },
	{ 0xa438, 0xd04d }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa436, 0xa152 },
	{ 0xa438, 0x12dc }, { 0xa436, 0xa154 }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa156 }, { 0xa438, 0x3fff }, { 0xa436, 0xa158 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa15a }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa15c }, { 0xa438, 0x3fff }, { 0xa436, 0xa15e },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa160 }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa150 }, { 0xa438, 0x0001 }, { 0xa436, 0xa016 },
	{ 0xa438, 0x0020 }, { 0xa436, 0xa012 }, { 0xa438, 0x0000 },
	{ 0xa436, 0xa014 }, { 0xa438, 0x1800 }, { 0xa438, 0x8010 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8022 }, { 0xa438, 0x1800 }, { 0xa438, 0x8233 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8332 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x855f }, { 0xa438, 0x1800 }, { 0xa438, 0x8619 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x86af }, { 0xa438, 0xd706 },
	{ 0xa438, 0x60a9 }, { 0xa438, 0xd700 }, { 0xa438, 0x60a1 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0962 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x0962 }, { 0xa438, 0x1800 }, { 0xa438, 0x0982 },
	{ 0xa438, 0x800a }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d00 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x12b5 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0f99 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x6201 }, { 0xa438, 0xd702 }, { 0xa438, 0x40a0 },
	{ 0xa438, 0xd70d }, { 0xa438, 0x419d }, { 0xa438, 0x1800 },
	{ 0xa438, 0x802c }, { 0xa438, 0xd701 }, { 0xa438, 0x611a },
	{ 0xa438, 0x8710 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x8280 }, { 0xa438, 0x8780 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xf002 }, { 0xa438, 0xa710 }, { 0xa438, 0x9580 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa304 },
	{ 0xa438, 0x9503 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d07 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x12b5 },
	{ 0xa438, 0xcb81 }, { 0xa438, 0xd70c }, { 0xa438, 0x48e2 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x407a }, { 0xa438, 0xd70c },
	{ 0xa438, 0x4867 }, { 0xa438, 0xd706 }, { 0xa438, 0x405a },
	{ 0xa438, 0x8910 }, { 0xa438, 0xa210 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x611c }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0080 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0101 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x03a0 }, { 0xa438, 0xccb5 }, { 0xa438, 0x0cc0 },
	{ 0xa438, 0x0080 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0102 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0340 }, { 0xa438, 0xcc52 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x42da }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0f1c },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5fb3 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x8f1f }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd70c },
	{ 0xa438, 0x7f33 }, { 0xa438, 0x8190 }, { 0xa438, 0x8204 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8087 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0f1b },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5fb3 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x8f1f }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd70c },
	{ 0xa438, 0x7f33 }, { 0xa438, 0xd70c }, { 0xa438, 0x6067 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8087 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8092 }, { 0xa438, 0xd403 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1203 }, { 0xa438, 0xcb82 }, { 0xa438, 0xd40a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1203 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x4267 }, { 0xa438, 0x1000 }, { 0xa438, 0x131d },
	{ 0xa438, 0x8a40 }, { 0xa438, 0x1000 }, { 0xa438, 0x120e },
	{ 0xa438, 0xa104 }, { 0xa438, 0x1000 }, { 0xa438, 0x1220 },
	{ 0xa438, 0x8104 }, { 0xa438, 0x1000 }, { 0xa438, 0x1217 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa704 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xcb88 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x81b7 }, { 0xa438, 0xd702 }, { 0xa438, 0x6161 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x40a0 }, { 0xa438, 0xd70d },
	{ 0xa438, 0x40fd }, { 0xa438, 0x1800 }, { 0xa438, 0x80b0 },
	{ 0xa438, 0xd701 }, { 0xa438, 0x607a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x80b0 }, { 0xa438, 0x1800 }, { 0xa438, 0x81a6 },
	{ 0xa438, 0xa210 }, { 0xa438, 0x8a10 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x643e }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d04 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x12b5 },
	{ 0xa438, 0x0cc0 }, { 0xa438, 0x0040 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x0102 }, { 0xa438, 0x0ce0 }, { 0xa438, 0x03e0 },
	{ 0xa438, 0xccce }, { 0xa438, 0xa00a }, { 0xa438, 0xa280 },
	{ 0xa438, 0xd110 }, { 0xa438, 0xd04c }, { 0xa438, 0xcba0 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x8710 }, { 0xa438, 0xaa0f },
	{ 0xa438, 0xa130 }, { 0xa438, 0xaa2f }, { 0xa438, 0xa2d5 },
	{ 0xa438, 0xa405 }, { 0xa438, 0xa720 }, { 0xa438, 0xa00a },
	{ 0xa438, 0xcba1 }, { 0xa438, 0x1800 }, { 0xa438, 0x80fa },
	{ 0xa438, 0xd704 }, { 0xa438, 0x3cf1 }, { 0xa438, 0x80db },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0d02 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x80dd }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d01 },
	{ 0xa438, 0x0cc0 }, { 0xa438, 0x0d40 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x12b5 }, { 0xa438, 0x8710 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x120e }, { 0xa438, 0xa108 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1220 }, { 0xa438, 0x8108 }, { 0xa438, 0xa203 },
	{ 0xa438, 0x8a2f }, { 0xa438, 0xa130 }, { 0xa438, 0x8204 },
	{ 0xa438, 0xa140 }, { 0xa438, 0x1000 }, { 0xa438, 0x1220 },
	{ 0xa438, 0x8140 }, { 0xa438, 0x1000 }, { 0xa438, 0x1217 },
	{ 0xa438, 0xcba2 }, { 0xa438, 0xd17a }, { 0xa438, 0xd04b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xa204 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd700 }, { 0xa438, 0x5fa7 },
	{ 0xa438, 0xb920 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd71f }, { 0xa438, 0x7fb4 }, { 0xa438, 0x9920 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd71f },
	{ 0xa438, 0x6145 }, { 0xa438, 0x6074 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8104 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fa7 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x80fe }, { 0xa438, 0xb820 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd71f }, { 0xa438, 0x7fa5 },
	{ 0xa438, 0x9820 }, { 0xa438, 0x9b01 }, { 0xa438, 0xd402 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1203 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x33b1 }, { 0xa438, 0x8124 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x60b5 }, { 0xa438, 0xd706 }, { 0xa438, 0x6069 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8126 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8196 }, { 0xa438, 0xd70c }, { 0xa438, 0x40ab },
	{ 0xa438, 0x800a }, { 0xa438, 0x8110 }, { 0xa438, 0x8284 },
	{ 0xa438, 0x8404 }, { 0xa438, 0xa710 }, { 0xa438, 0x8120 },
	{ 0xa438, 0x8241 }, { 0xa438, 0x1000 }, { 0xa438, 0x120e },
	{ 0xa438, 0xa104 }, { 0xa438, 0x1000 }, { 0xa438, 0x1220 },
	{ 0xa438, 0x8104 }, { 0xa438, 0x1000 }, { 0xa438, 0x1217 },
	{ 0xa438, 0xaa2f }, { 0xa438, 0xcba3 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x438b }, { 0xa438, 0xa284 }, { 0xa438, 0xd078 },
	{ 0xa438, 0x800a }, { 0xa438, 0x8110 }, { 0xa438, 0xa284 },
	{ 0xa438, 0x8404 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xa108 }, { 0xa438, 0x9503 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0f19 },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5fb3 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x8f1f }, { 0xa438, 0x9503 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd70c },
	{ 0xa438, 0x7f33 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d07 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x12b5 },
	{ 0xa438, 0x8110 }, { 0xa438, 0xa284 }, { 0xa438, 0xa404 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xcba4 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x40a1 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xad10 }, { 0xa438, 0x9503 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x414b }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0080 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0102 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0340 }, { 0xa438, 0xcc52 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8175 }, { 0xa438, 0x80c0 }, { 0xa438, 0x8103 },
	{ 0xa438, 0x83e0 }, { 0xa438, 0x8cff }, { 0xa438, 0x60ba },
	{ 0xa438, 0xd110 }, { 0xa438, 0xd041 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x817c }, { 0xa438, 0xd193 }, { 0xa438, 0xd047 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xa110 }, { 0xa438, 0xcba5 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5faa }, { 0xa438, 0xa180 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x6041 }, { 0xa438, 0xa402 }, { 0xa438, 0xcba6 },
	{ 0xa438, 0x60ba }, { 0xa438, 0xd1f5 }, { 0xa438, 0xd045 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8192 }, { 0xa438, 0xd1f5 },
	{ 0xa438, 0xd049 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fb4 }, { 0xa438, 0x8710 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xa190 }, { 0xa438, 0xa204 },
	{ 0xa438, 0xa280 }, { 0xa438, 0xa404 }, { 0xa438, 0xcba7 },
	{ 0xa438, 0xbb80 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd71f }, { 0xa438, 0x5fb4 }, { 0xa438, 0xb920 },
	{ 0xa438, 0x9b80 }, { 0xa438, 0x1800 }, { 0xa438, 0x81e5 },
	{ 0xa438, 0xa210 }, { 0xa438, 0xa00a }, { 0xa438, 0xaa40 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x120e }, { 0xa438, 0xa104 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1220 }, { 0xa438, 0x8104 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1217 }, { 0xa438, 0xa190 },
	{ 0xa438, 0xa284 }, { 0xa438, 0xa404 }, { 0xa438, 0x8a10 },
	{ 0xa438, 0x8a80 }, { 0xa438, 0xcb84 }, { 0xa438, 0xd13e },
	{ 0xa438, 0xd05a }, { 0xa438, 0xd13e }, { 0xa438, 0xd06b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x3559 }, { 0xa438, 0x81c2 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x81bb }, { 0xa438, 0xd700 }, { 0xa438, 0x604b },
	{ 0xa438, 0xcb8a }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd700 }, { 0xa438, 0x3659 }, { 0xa438, 0x81cc },
	{ 0xa438, 0x1800 }, { 0xa438, 0x81c5 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x606b }, { 0xa438, 0xcb8b }, { 0xa438, 0x5ecb },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6041 }, { 0xa438, 0xa402 },
	{ 0xa438, 0xcb8c }, { 0xa438, 0xd706 }, { 0xa438, 0x60ba },
	{ 0xa438, 0xd179 }, { 0xa438, 0xd049 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x81dc }, { 0xa438, 0xd160 }, { 0xa438, 0xd04b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xcb8d }, { 0xa438, 0x8710 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x5fd4 }, { 0xa438, 0xb920 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd71f },
	{ 0xa438, 0x7fb4 }, { 0xa438, 0x9920 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd71f }, { 0xa438, 0x6145 },
	{ 0xa438, 0x6074 }, { 0xa438, 0x1800 }, { 0xa438, 0x81ea },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fab }, { 0xa438, 0x1800 }, { 0xa438, 0x81e4 },
	{ 0xa438, 0xa710 }, { 0xa438, 0xb820 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd71f }, { 0xa438, 0x7fa5 },
	{ 0xa438, 0x9820 }, { 0xa438, 0xd114 }, { 0xa438, 0xd040 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fba }, { 0xa438, 0xd704 }, { 0xa438, 0x5f76 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5f34 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x6081 }, { 0xa438, 0xd706 }, { 0xa438, 0x405a },
	{ 0xa438, 0xa480 }, { 0xa438, 0xcb86 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x60fa }, { 0xa438, 0xd700 }, { 0xa438, 0x60e1 },
	{ 0xa438, 0xd1c8 }, { 0xa438, 0xd045 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8218 }, { 0xa438, 0xd17a }, { 0xa438, 0xd04b },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0101 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0320 }, { 0xa438, 0xcc29 }, { 0xa438, 0xa208 },
	{ 0xa438, 0x8204 }, { 0xa438, 0xd704 }, { 0xa438, 0x40f5 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa280 },
	{ 0xa438, 0x8780 }, { 0xa438, 0x9503 }, { 0xa438, 0x8e04 },
	{ 0xa438, 0xd114 }, { 0xa438, 0xd040 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5ff4 }, { 0xa438, 0x1800 }, { 0xa438, 0x0c3e },
	{ 0xa438, 0xd706 }, { 0xa438, 0x609d }, { 0xa438, 0xd417 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1203 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0x1000 }, { 0xa438, 0x1289 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5f7a }, { 0xa438, 0xd704 },
	{ 0xa438, 0x5f36 }, { 0xa438, 0xd706 }, { 0xa438, 0x6089 },
	{ 0xa438, 0xd40c }, { 0xa438, 0x1000 }, { 0xa438, 0x1203 },
	{ 0xa438, 0xaa40 }, { 0xa438, 0xbb10 }, { 0xa438, 0xcb50 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa310 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xcb5f }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0x1000 }, { 0xa438, 0x1289 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x5f75 }, { 0xa438, 0x8190 },
	{ 0xa438, 0x82a0 }, { 0xa438, 0x8402 }, { 0xa438, 0xa404 },
	{ 0xa438, 0x800a }, { 0xa438, 0x8718 }, { 0xa438, 0x9b10 },
	{ 0xa438, 0x9b20 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd71f }, { 0xa438, 0x7fb5 }, { 0xa438, 0xcb51 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd71f },
	{ 0xa438, 0x5f94 }, { 0xa438, 0xd706 }, { 0xa438, 0x61a9 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x40a1 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x4079 }, { 0xa438, 0xd706 }, { 0xa438, 0x609d },
	{ 0xa438, 0xd141 }, { 0xa438, 0xd043 }, { 0xa438, 0xf006 },
	{ 0xa438, 0xd101 }, { 0xa438, 0xd040 }, { 0xa438, 0xf003 },
	{ 0xa438, 0xd141 }, { 0xa438, 0xd044 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd700 }, { 0xa438, 0x5fb4 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x60e5 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x60be }, { 0xa438, 0xd706 }, { 0xa438, 0x29b1 },
	{ 0xa438, 0x8280 }, { 0xa438, 0xf002 }, { 0xa438, 0xa880 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xa190 }, { 0xa438, 0x8220 },
	{ 0xa438, 0xa280 }, { 0xa438, 0xa404 }, { 0xa438, 0xa620 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xc5aa },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd700 }, { 0xa438, 0x6061 },
	{ 0xa438, 0xa402 }, { 0xa438, 0xa480 }, { 0xa438, 0xcb52 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fba }, { 0xa438, 0xd704 }, { 0xa438, 0x5f76 },
	{ 0xa438, 0xb920 }, { 0xa438, 0xcb53 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd71f }, { 0xa438, 0x7fb4 },
	{ 0xa438, 0x9920 }, { 0xa438, 0xa00a }, { 0xa438, 0xa190 },
	{ 0xa438, 0xa280 }, { 0xa438, 0x8220 }, { 0xa438, 0xa404 },
	{ 0xa438, 0xb580 }, { 0xa438, 0xd700 }, { 0xa438, 0x40a1 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa602 },
	{ 0xa438, 0x9503 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xa310 }, { 0xa438, 0x9503 }, { 0xa438, 0xcb60 },
	{ 0xa438, 0xd101 }, { 0xa438, 0xd040 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd700 }, { 0xa438, 0x5fb4 },
	{ 0xa438, 0xaa10 }, { 0xa438, 0xd70c }, { 0xa438, 0x2833 },
	{ 0xa438, 0x82b9 }, { 0xa438, 0xf003 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1330 }, { 0xa438, 0xd70c }, { 0xa438, 0x40a6 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa140 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd70c }, { 0xa438, 0x40a3 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xac20 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xa90c }, { 0xa438, 0xaa80 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0d07 }, { 0xa438, 0x8dc0 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x12b5 }, { 0xa438, 0xa00a },
	{ 0xa438, 0xa190 }, { 0xa438, 0xa280 }, { 0xa438, 0x8220 },
	{ 0xa438, 0xa404 }, { 0xa438, 0xb580 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xc500 }, { 0xa438, 0x9503 },
	{ 0xa438, 0x83e0 }, { 0xa438, 0x8e01 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x40a1 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xa602 }, { 0xa438, 0x9503 }, { 0xa438, 0xd14a },
	{ 0xa438, 0xd058 }, { 0xa438, 0x1000 }, { 0xa438, 0x12d7 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x4063 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x12ea }, { 0xa438, 0xcb6f }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd704 }, { 0xa438, 0x2e70 },
	{ 0xa438, 0x8327 }, { 0xa438, 0xd71f }, { 0xa438, 0x676e },
	{ 0xa438, 0xd704 }, { 0xa438, 0x3868 }, { 0xa438, 0x8302 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x61c2 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x2f18 }, { 0xa438, 0x8308 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5d35 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xc5aa }, { 0xa438, 0x9503 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0320 }, { 0xa438, 0x1800 }, { 0xa438, 0x830e },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xc5aa },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1800 }, { 0xa438, 0x832e },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xc5aa },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1800 }, { 0xa438, 0x8330 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x12d7 }, { 0xa438, 0xae02 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x4063 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x12ea }, { 0xa438, 0xcb61 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd704 }, { 0xa438, 0x2e70 },
	{ 0xa438, 0x8327 }, { 0xa438, 0xd704 }, { 0xa438, 0x3868 },
	{ 0xa438, 0x832e }, { 0xa438, 0xd706 }, { 0xa438, 0x61e2 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x612e }, { 0xa438, 0xd70c },
	{ 0xa438, 0x2f18 }, { 0xa438, 0x8330 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x830e }, { 0xa438, 0x8e02 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x0f99 }, { 0xa438, 0xae04 }, { 0xa438, 0x8310 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0e31 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x85ac }, { 0xa438, 0x1800 }, { 0xa438, 0x0e07 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17be }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5fa4 }, { 0xa438, 0xa706 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x404b }, { 0xa438, 0xa880 }, { 0xa438, 0x8801 },
	{ 0xa438, 0x8e01 }, { 0xa438, 0xca50 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x83d5 }, { 0xa438, 0xca51 }, { 0xa438, 0xd70e },
	{ 0xa438, 0x2210 }, { 0xa438, 0x83d3 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x4084 }, { 0xa438, 0xd705 }, { 0xa438, 0x5efd },
	{ 0xa438, 0xf007 }, { 0xa438, 0x1000 }, { 0xa438, 0x17c2 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5ce2 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x1692 }, { 0xa438, 0xd70c }, { 0xa438, 0x605a },
	{ 0xa438, 0x9a10 }, { 0xa438, 0x8e40 }, { 0xa438, 0x8404 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1827 }, { 0xa438, 0x8e80 },
	{ 0xa438, 0xca62 }, { 0xa438, 0xd705 }, { 0xa438, 0x3084 },
	{ 0xa438, 0x83b5 }, { 0xa438, 0xba10 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x1000 }, { 0xa438, 0x84ae },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x0100 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x4638 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd044 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17be }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17e8 }, { 0xa438, 0xd70c }, { 0xa438, 0x5f7c },
	{ 0xa438, 0x8108 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0907 },
	{ 0xa438, 0x8940 }, { 0xa438, 0x1000 }, { 0xa438, 0x17db },
	{ 0xa438, 0xa0c4 }, { 0xa438, 0x8610 }, { 0xa438, 0x8030 },
	{ 0xa438, 0x8706 }, { 0xa438, 0x0c07 }, { 0xa438, 0x0b06 },
	{ 0xa438, 0x8410 }, { 0xa438, 0xa980 }, { 0xa438, 0xa702 },
	{ 0xa438, 0xd1c4 }, { 0xa438, 0xd045 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f7c }, { 0xa438, 0x0c07 },
	{ 0xa438, 0x0b06 }, { 0xa438, 0xa030 }, { 0xa438, 0xa610 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6041 }, { 0xa438, 0xa501 },
	{ 0xa438, 0xa108 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd045 },
	{ 0xa438, 0xca63 }, { 0xa438, 0x1000 }, { 0xa438, 0x17be },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17e8 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f7c }, { 0xa438, 0xd702 }, { 0xa438, 0x6078 },
	{ 0xa438, 0x9920 }, { 0xa438, 0xf003 }, { 0xa438, 0xb920 },
	{ 0xa438, 0xa880 }, { 0xa438, 0x9a10 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd71f }, { 0xa438, 0x5f73 }, { 0xa438, 0xf011 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x409b }, { 0xa438, 0x9920 },
	{ 0xa438, 0x9a10 }, { 0xa438, 0xfff5 }, { 0xa438, 0x80fe },
	{ 0xa438, 0x8610 }, { 0xa438, 0x8501 }, { 0xa438, 0x8980 },
	{ 0xa438, 0x8702 }, { 0xa438, 0xa410 }, { 0xa438, 0xa940 },
	{ 0xa438, 0x81c0 }, { 0xa438, 0xae80 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x835a }, { 0xa438, 0x8804 }, { 0xa438, 0xa704 },
	{ 0xa438, 0x8788 }, { 0xa438, 0xff82 }, { 0xa438, 0xbb08 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0907 }, { 0xa438, 0x8940 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17db }, { 0xa438, 0x8701 },
	{ 0xa438, 0x8502 }, { 0xa438, 0xa0f4 }, { 0xa438, 0xa610 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6061 }, { 0xa438, 0xa002 },
	{ 0xa438, 0xa501 }, { 0xa438, 0x8706 }, { 0xa438, 0x8410 },
	{ 0xa438, 0xa980 }, { 0xa438, 0xca64 }, { 0xa438, 0xd110 },
	{ 0xa438, 0xd040 }, { 0xa438, 0x1000 }, { 0xa438, 0x17be },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17e8 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f7c }, { 0xa438, 0x8804 }, { 0xa438, 0xa706 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8336 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x147c }, { 0xa438, 0xd705 }, { 0xa438, 0x405f },
	{ 0xa438, 0xf037 }, { 0xa438, 0xd701 }, { 0xa438, 0x4259 },
	{ 0xa438, 0xd705 }, { 0xa438, 0x6234 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x41c6 }, { 0xa438, 0xd70d }, { 0xa438, 0x419d },
	{ 0xa438, 0xd70d }, { 0xa438, 0x417e }, { 0xa438, 0xd704 },
	{ 0xa438, 0x6127 }, { 0xa438, 0x2951 }, { 0xa438, 0x83ec },
	{ 0xa438, 0xd70c }, { 0xa438, 0x4083 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x2e81 }, { 0xa438, 0x83ec }, { 0xa438, 0xf0c2 },
	{ 0xa438, 0x80fe }, { 0xa438, 0x8610 }, { 0xa438, 0x8501 },
	{ 0xa438, 0x8704 }, { 0xa438, 0x0c30 }, { 0xa438, 0x0410 },
	{ 0xa438, 0xac02 }, { 0xa438, 0xa502 }, { 0xa438, 0x8980 },
	{ 0xa438, 0xca60 }, { 0xa438, 0xa004 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x6065 }, { 0xa438, 0x1800 }, { 0xa438, 0x83fc },
	{ 0xa438, 0x8004 }, { 0xa438, 0xa804 }, { 0xa438, 0x0c0f },
	{ 0xa438, 0x0602 }, { 0xa438, 0x0c70 }, { 0xa438, 0x0730 },
	{ 0xa438, 0xa708 }, { 0xa438, 0xd704 }, { 0xa438, 0x609c },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0912 }, { 0xa438, 0xf003 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x090e }, { 0xa438, 0xa940 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17db }, { 0xa438, 0xa780 },
	{ 0xa438, 0xf0a0 }, { 0xa438, 0xd704 }, { 0xa438, 0x63ab },
	{ 0xa438, 0xd705 }, { 0xa438, 0x4371 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x339c }, { 0xa438, 0x84ad }, { 0xa438, 0x8788 },
	{ 0xa438, 0x8704 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0907 },
	{ 0xa438, 0x8940 }, { 0xa438, 0x1000 }, { 0xa438, 0x17db },
	{ 0xa438, 0x8410 }, { 0xa438, 0xa0f4 }, { 0xa438, 0xa610 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6061 }, { 0xa438, 0xa002 },
	{ 0xa438, 0xa501 }, { 0xa438, 0xa706 }, { 0xa438, 0x8804 },
	{ 0xa438, 0xa980 }, { 0xa438, 0xd70c }, { 0xa438, 0x6085 },
	{ 0xa438, 0x8701 }, { 0xa438, 0x8502 }, { 0xa438, 0x8c02 },
	{ 0xa438, 0xf082 }, { 0xa438, 0xd70c }, { 0xa438, 0x60c5 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x6053 }, { 0xa438, 0xf07d },
	{ 0xa438, 0x1800 }, { 0xa438, 0x84aa }, { 0xa438, 0xd70d },
	{ 0xa438, 0x4d1b }, { 0xa438, 0xba10 }, { 0xa438, 0xae40 },
	{ 0xa438, 0x0cfc }, { 0xa438, 0x03b4 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x05b4 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd044 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17be }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17e8 }, { 0xa438, 0xd70c }, { 0xa438, 0x5f7c },
	{ 0xa438, 0x8706 }, { 0xa438, 0x8280 }, { 0xa438, 0xace0 },
	{ 0xa438, 0xa680 }, { 0xa438, 0xa240 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x5f79 }, { 0xa438, 0x8240 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x6898 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x4957 }, { 0xa438, 0x1800 }, { 0xa438, 0x849c },
	{ 0xa438, 0xa1c0 }, { 0xa438, 0x0c3f }, { 0xa438, 0x0220 },
	{ 0xa438, 0x0cfc }, { 0xa438, 0x030c }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x050c }, { 0xa438, 0x8108 }, { 0xa438, 0x8640 },
	{ 0xa438, 0xa120 }, { 0xa438, 0xa640 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x0101 }, { 0xa438, 0xa110 }, { 0xa438, 0xd1c4 },
	{ 0xa438, 0xd044 }, { 0xa438, 0xca84 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f7c }, { 0xa438, 0xd702 },
	{ 0xa438, 0x60fc }, { 0xa438, 0x8210 }, { 0xa438, 0x0ce0 },
	{ 0xa438, 0x0320 }, { 0xa438, 0x0ce0 }, { 0xa438, 0x0520 },
	{ 0xa438, 0xf002 }, { 0xa438, 0xa210 }, { 0xa438, 0xd1c4 },
	{ 0xa438, 0xd043 }, { 0xa438, 0x1000 }, { 0xa438, 0x17be },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17e8 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f7c }, { 0xa438, 0x8233 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x036c }, { 0xa438, 0x0cfc }, { 0xa438, 0x056c },
	{ 0xa438, 0xd1c4 }, { 0xa438, 0xd044 }, { 0xa438, 0xca85 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17be }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17e8 }, { 0xa438, 0xd70c }, { 0xa438, 0x5f7c },
	{ 0xa438, 0xa680 }, { 0xa438, 0xa240 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x5f79 }, { 0xa438, 0x8240 },
	{ 0xa438, 0x0cfc }, { 0xa438, 0x0390 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x0590 }, { 0xa438, 0xd702 }, { 0xa438, 0x6058 },
	{ 0xa438, 0xf002 }, { 0xa438, 0xfec8 }, { 0xa438, 0x81c0 },
	{ 0xa438, 0x8880 }, { 0xa438, 0x8706 }, { 0xa438, 0xca61 },
	{ 0xa438, 0xd1c4 }, { 0xa438, 0xd054 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f7d }, { 0xa438, 0xa706 },
	{ 0xa438, 0xf004 }, { 0xa438, 0x8788 }, { 0xa438, 0xa404 },
	{ 0xa438, 0x8702 }, { 0xa438, 0x0800 }, { 0xa438, 0x8443 },
	{ 0xa438, 0x8303 }, { 0xa438, 0x8280 }, { 0xa438, 0x9920 },
	{ 0xa438, 0x8ce0 }, { 0xa438, 0x8004 }, { 0xa438, 0xa1c0 },
	{ 0xa438, 0xd70e }, { 0xa438, 0x404a }, { 0xa438, 0xa280 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x3bd0 }, { 0xa438, 0x84be },
	{ 0xa438, 0x0c3f }, { 0xa438, 0x0223 }, { 0xa438, 0xf003 },
	{ 0xa438, 0x0c3f }, { 0xa438, 0x0220 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x0308 }, { 0xa438, 0x0cfc }, { 0xa438, 0x0508 },
	{ 0xa438, 0x8108 }, { 0xa438, 0x8640 }, { 0xa438, 0xa120 },
	{ 0xa438, 0xa640 }, { 0xa438, 0xd702 }, { 0xa438, 0x6077 },
	{ 0xa438, 0x8103 }, { 0xa438, 0xf003 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x0101 }, { 0xa438, 0xa110 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x6077 }, { 0xa438, 0xa108 }, { 0xa438, 0xf006 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x6077 }, { 0xa438, 0x8108 },
	{ 0xa438, 0xf002 }, { 0xa438, 0xa108 }, { 0xa438, 0xd193 },
	{ 0xa438, 0xd045 }, { 0xa438, 0xca82 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0xd70e }, { 0xa438, 0x606a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17e8 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f3c }, { 0xa438, 0xd702 }, { 0xa438, 0x60fc },
	{ 0xa438, 0x8210 }, { 0xa438, 0x0ce0 }, { 0xa438, 0x0320 },
	{ 0xa438, 0x0ce0 }, { 0xa438, 0x0520 }, { 0xa438, 0xf002 },
	{ 0xa438, 0xa210 }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd043 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17be }, { 0xa438, 0xd70e },
	{ 0xa438, 0x606a }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f3c }, { 0xa438, 0xd702 },
	{ 0xa438, 0x3bd0 }, { 0xa438, 0x84fc }, { 0xa438, 0x0c3f },
	{ 0xa438, 0x020c }, { 0xa438, 0xf002 }, { 0xa438, 0x823f },
	{ 0xa438, 0x0cfc }, { 0xa438, 0x034c }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x054c }, { 0xa438, 0xd1c4 }, { 0xa438, 0xd044 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17be }, { 0xa438, 0xd70e },
	{ 0xa438, 0x606a }, { 0xa438, 0x1000 }, { 0xa438, 0x17e8 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x5f3c }, { 0xa438, 0x820c },
	{ 0xa438, 0xa360 }, { 0xa438, 0xa560 }, { 0xa438, 0xd1c4 },
	{ 0xa438, 0xd043 }, { 0xa438, 0xca83 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0xd70e }, { 0xa438, 0x606a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17e8 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x5f3c }, { 0xa438, 0xd70e }, { 0xa438, 0x406a },
	{ 0xa438, 0x8680 }, { 0xa438, 0xf002 }, { 0xa438, 0xa680 },
	{ 0xa438, 0xa240 }, { 0xa438, 0x0c0f }, { 0xa438, 0x0604 },
	{ 0xa438, 0x0c70 }, { 0xa438, 0x0750 }, { 0xa438, 0xa708 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x609c }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0914 }, { 0xa438, 0xf003 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0910 }, { 0xa438, 0xa940 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17db }, { 0xa438, 0xa780 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x17be }, { 0xa438, 0xd70e }, { 0xa438, 0x606a },
	{ 0xa438, 0x1000 }, { 0xa438, 0x17e8 }, { 0xa438, 0xd702 },
	{ 0xa438, 0x399c }, { 0xa438, 0x852f }, { 0xa438, 0x8240 },
	{ 0xa438, 0x8788 }, { 0xa438, 0xd702 }, { 0xa438, 0x63f8 },
	{ 0xa438, 0xd705 }, { 0xa438, 0x643c }, { 0xa438, 0xa402 },
	{ 0xa438, 0xf012 }, { 0xa438, 0x8402 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x611b }, { 0xa438, 0xa401 }, { 0xa438, 0xa302 },
	{ 0xa438, 0xd702 }, { 0xa438, 0x417d }, { 0xa438, 0xa440 },
	{ 0xa438, 0xa280 }, { 0xa438, 0xf008 }, { 0xa438, 0x8401 },
	{ 0xa438, 0x8302 }, { 0xa438, 0xd70c }, { 0xa438, 0x6060 },
	{ 0xa438, 0xa301 }, { 0xa438, 0xf002 }, { 0xa438, 0x8301 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x4080 }, { 0xa438, 0xd70e },
	{ 0xa438, 0x604a }, { 0xa438, 0xff5f }, { 0xa438, 0xd705 },
	{ 0xa438, 0x3cdd }, { 0xa438, 0x855e }, { 0xa438, 0xff5b },
	{ 0xa438, 0x0cfc }, { 0xa438, 0x0390 }, { 0xa438, 0x0cfc },
	{ 0xa438, 0x0590 }, { 0xa438, 0x0800 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x60f9 }, { 0xa438, 0xd704 }, { 0xa438, 0x6958 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x6902 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x1001 }, { 0xa438, 0xa220 }, { 0xa438, 0xa404 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x4054 }, { 0xa438, 0xa740 },
	{ 0xa438, 0xa504 }, { 0xa438, 0xd704 }, { 0xa438, 0x40b5 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa003 },
	{ 0xa438, 0x9503 }, { 0xa438, 0x8190 }, { 0xa438, 0xcb91 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x10af }, { 0xa438, 0xd704 },
	{ 0xa438, 0x7fb9 }, { 0xa438, 0x8220 }, { 0xa438, 0x8404 },
	{ 0xa438, 0xa280 }, { 0xa438, 0xa110 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x4041 }, { 0xa438, 0xa180 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x130c }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x850f }, { 0xa438, 0x9503 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d08 }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0d80 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x12b5 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x10af }, { 0xa438, 0xd704 }, { 0xa438, 0x615f },
	{ 0xa438, 0xd70c }, { 0xa438, 0x6103 }, { 0xa438, 0x8504 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x40b5 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x8003 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xcb92 }, { 0xa438, 0x1000 }, { 0xa438, 0x10af },
	{ 0xa438, 0xd706 }, { 0xa438, 0x7fa3 }, { 0xa438, 0x8280 },
	{ 0xa438, 0x8190 }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0x0c0f }, { 0xa438, 0x050a }, { 0xa438, 0x9503 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0d00 }, { 0xa438, 0x8dc0 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x12b5 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x1001 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d00 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0x1000 }, { 0xa438, 0x12b5 },
	{ 0xa438, 0x800a }, { 0xa438, 0xd705 }, { 0xa438, 0x40b9 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x6063 }, { 0xa438, 0xa020 },
	{ 0xa438, 0xf003 }, { 0xa438, 0xd705 }, { 0xa438, 0x8020 },
	{ 0xa438, 0xa504 }, { 0xa438, 0xd704 }, { 0xa438, 0x40b5 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0xa003 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd704 }, { 0xa438, 0x4054 },
	{ 0xa438, 0xa740 }, { 0xa438, 0x8190 }, { 0xa438, 0xcb93 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x6063 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x609c }, { 0xa438, 0xd14b }, { 0xa438, 0xd040 },
	{ 0xa438, 0xf003 }, { 0xa438, 0xd120 }, { 0xa438, 0xd040 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x10af }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5fb4 }, { 0xa438, 0xa008 }, { 0xa438, 0xd706 },
	{ 0xa438, 0x4040 }, { 0xa438, 0xa002 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x4079 }, { 0xa438, 0x1000 }, { 0xa438, 0x1313 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x85f0 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd705 }, { 0xa438, 0x40d9 },
	{ 0xa438, 0xd70c }, { 0xa438, 0x6083 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d09 }, { 0xa438, 0xf003 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0d0a }, { 0xa438, 0x0cc0 }, { 0xa438, 0x0d80 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x12b5 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x10af }, { 0xa438, 0x8020 }, { 0xa438, 0xd705 },
	{ 0xa438, 0x4199 }, { 0xa438, 0xd704 }, { 0xa438, 0x615f },
	{ 0xa438, 0xd70c }, { 0xa438, 0x6103 }, { 0xa438, 0x8504 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x40b5 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0x8003 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xcb94 }, { 0xa438, 0x1000 }, { 0xa438, 0x10af },
	{ 0xa438, 0xd706 }, { 0xa438, 0x7fa2 }, { 0xa438, 0x800a },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x85f0 },
	{ 0xa438, 0x9503 }, { 0xa438, 0xd705 }, { 0xa438, 0x40b9 },
	{ 0xa438, 0x0c1f }, { 0xa438, 0x0d00 }, { 0xa438, 0x8dc0 },
	{ 0xa438, 0xf005 }, { 0xa438, 0x0c1f }, { 0xa438, 0x0d07 },
	{ 0xa438, 0x8dc0 }, { 0xa438, 0xa190 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x12b5 }, { 0xa438, 0xd705 }, { 0xa438, 0x39cc },
	{ 0xa438, 0x8617 }, { 0xa438, 0x1800 }, { 0xa438, 0x1001 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x82c7 }, { 0xa438, 0xcb13 },
	{ 0xa438, 0xd706 }, { 0xa438, 0x6089 }, { 0xa438, 0xd1b8 },
	{ 0xa438, 0xd04a }, { 0xa438, 0xf003 }, { 0xa438, 0xd11c },
	{ 0xa438, 0xd04b }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd701 }, { 0xa438, 0x67d5 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x5f74 }, { 0xa438, 0xd70c }, { 0xa438, 0x610c },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd700 },
	{ 0xa438, 0x6846 }, { 0xa438, 0xd706 }, { 0xa438, 0x647b },
	{ 0xa438, 0xfffa }, { 0xa438, 0x1000 }, { 0xa438, 0x1330 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x0c1f },
	{ 0xa438, 0x0f16 }, { 0xa438, 0x9503 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd70c }, { 0xa438, 0x5fb3 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x8f1f },
	{ 0xa438, 0x9503 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd70c }, { 0xa438, 0x7f33 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x12b5 }, { 0xa438, 0x0c07 }, { 0xa438, 0x0c02 },
	{ 0xa438, 0x0cc0 }, { 0xa438, 0x0080 }, { 0xa438, 0xd14a },
	{ 0xa438, 0xd048 }, { 0xa438, 0x1000 }, { 0xa438, 0x126b },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fb4 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8629 }, { 0xa438, 0x800a }, { 0xa438, 0x1000 },
	{ 0xa438, 0x120e }, { 0xa438, 0xa004 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1220 }, { 0xa438, 0x8004 }, { 0xa438, 0xa001 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1220 }, { 0xa438, 0x8001 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1217 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x0902 }, { 0xa438, 0x1800 }, { 0xa438, 0x04ed },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd71f },
	{ 0xa438, 0x5fab }, { 0xa438, 0xba08 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd71f }, { 0xa438, 0x7f8b },
	{ 0xa438, 0x9a08 }, { 0xa438, 0x1800 }, { 0xa438, 0x0581 },
	{ 0xa438, 0x800a }, { 0xa438, 0xd702 }, { 0xa438, 0x6555 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x120e }, { 0xa438, 0xa004 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1220 }, { 0xa438, 0x8004 },
	{ 0xa438, 0xa001 }, { 0xa438, 0x1000 }, { 0xa438, 0x1220 },
	{ 0xa438, 0x8001 }, { 0xa438, 0x1000 }, { 0xa438, 0x1217 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xa780 }, { 0xa438, 0xcb14 },
	{ 0xa438, 0xd1b8 }, { 0xa438, 0xd04a }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd700 }, { 0xa438, 0x5fb4 },
	{ 0xa438, 0x6286 }, { 0xa438, 0xd706 }, { 0xa438, 0x5f5b },
	{ 0xa438, 0x800a }, { 0xa438, 0x1000 }, { 0xa438, 0x120e },
	{ 0xa438, 0xa004 }, { 0xa438, 0x1000 }, { 0xa438, 0x1220 },
	{ 0xa438, 0x8004 }, { 0xa438, 0xa001 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1220 }, { 0xa438, 0x8001 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1217 }, { 0xa438, 0x0c03 }, { 0xa438, 0x0902 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8671 }, { 0xa438, 0xa00a },
	{ 0xa438, 0x9308 }, { 0xa438, 0xb210 }, { 0xa438, 0xb301 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x126b }, { 0xa438, 0xd701 },
	{ 0xa438, 0x5fa4 }, { 0xa438, 0xb302 }, { 0xa438, 0x9210 },
	{ 0xa438, 0xd409 }, { 0xa438, 0x1000 }, { 0xa438, 0x1203 },
	{ 0xa438, 0xd103 }, { 0xa438, 0xd04c }, { 0xa438, 0x1000 },
	{ 0xa438, 0x126b }, { 0xa438, 0xd700 }, { 0xa438, 0x5fb4 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0581 }, { 0xa438, 0xd70c },
	{ 0xa438, 0x60b3 }, { 0xa438, 0x1800 }, { 0xa438, 0x86b3 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x001a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x12cb }, { 0xa436, 0xa10e }, { 0xa438, 0x12cf },
	{ 0xa436, 0xa10c }, { 0xa438, 0x04f8 }, { 0xa436, 0xa10a },
	{ 0xa438, 0x1003 }, { 0xa436, 0xa108 }, { 0xa438, 0x15fb },
	{ 0xa436, 0xa106 }, { 0xa438, 0x0d2b }, { 0xa436, 0xa104 },
	{ 0xa438, 0x0ecb }, { 0xa436, 0xa102 }, { 0xa438, 0x1119 },
	{ 0xa436, 0xa100 }, { 0xa438, 0x0960 }, { 0xa436, 0xa110 },
	{ 0xa438, 0x00ff }, { 0xa436, 0xa016 }, { 0xa438, 0x0020 },
	{ 0xa436, 0xa012 }, { 0xa438, 0x1ff8 }, { 0xa436, 0xa014 },
	{ 0xa438, 0xa704 }, { 0xa438, 0x82c7 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa436, 0xa164 },
	{ 0xa438, 0x119f }, { 0xa436, 0xa166 }, { 0xa438, 0x11a1 },
	{ 0xa436, 0xa168 }, { 0xa438, 0x3fff }, { 0xa436, 0xa16a },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa16c }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa16e }, { 0xa438, 0x3fff }, { 0xa436, 0xa170 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa172 }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa162 }, { 0xa438, 0x0003 }, { 0xa436, 0xb87c },
	{ 0xa438, 0x8a63 }, { 0xa436, 0xb87e }, { 0xa438, 0xaf8a },
	{ 0xa438, 0x7baf }, { 0xa438, 0x8ab6 }, { 0xa438, 0xaf8a },
	{ 0xa438, 0xd6af }, { 0xa438, 0x8ae4 }, { 0xa438, 0xaf8a },
	{ 0xa438, 0xf2af }, { 0xa438, 0x8b01 }, { 0xa438, 0xaf8b },
	{ 0xa438, 0x0aaf }, { 0xa438, 0x8b10 }, { 0xa438, 0xad35 },
	{ 0xa438, 0x27bf }, { 0xa438, 0x7308 }, { 0xa438, 0x027b },
	{ 0xa438, 0x07ac }, { 0xa438, 0x280d }, { 0xa438, 0xbf73 },
	{ 0xa438, 0x0b02 }, { 0xa438, 0x7b07 }, { 0xa438, 0xac28 },
	{ 0xa438, 0x04d0 }, { 0xa438, 0x05ae }, { 0xa438, 0x02d0 },
	{ 0xa438, 0x01d1 }, { 0xa438, 0x01d3 }, { 0xa438, 0x04ee },
	{ 0xa438, 0x8640 }, { 0xa438, 0x00ee }, { 0xa438, 0x8641 },
	{ 0xa438, 0x00af }, { 0xa438, 0x6aa6 }, { 0xa438, 0xd100 },
	{ 0xa438, 0xd300 }, { 0xa438, 0xee86 }, { 0xa438, 0x4001 },
	{ 0xa438, 0xee86 }, { 0xa438, 0x4124 }, { 0xa438, 0xd00f },
	{ 0xa438, 0xaf6a }, { 0xa438, 0xa6bf }, { 0xa438, 0x739e },
	{ 0xa438, 0x027b }, { 0xa438, 0x07ad }, { 0xa438, 0x280b },
	{ 0xa438, 0xe18f }, { 0xa438, 0xfdad }, { 0xa438, 0x2805 },
	{ 0xa438, 0xe08f }, { 0xa438, 0xfeae }, { 0xa438, 0x03e0 },
	{ 0xa438, 0x8fff }, { 0xa438, 0xe489 }, { 0xa438, 0xe7e0 },
	{ 0xa438, 0x89e7 }, { 0xa438, 0xaf67 }, { 0xa438, 0x9fa0 },
	{ 0xa438, 0x9402 }, { 0xa438, 0xae03 }, { 0xa438, 0xa0b5 },
	{ 0xa438, 0x03af }, { 0xa438, 0x0d89 }, { 0xa438, 0xaf0d },
	{ 0xa438, 0xafa0 }, { 0xa438, 0x9402 }, { 0xa438, 0xae03 },
	{ 0xa438, 0xa0b5 }, { 0xa438, 0x03af }, { 0xa438, 0x0c64 },
	{ 0xa438, 0xaf0c }, { 0xa438, 0xcce0 }, { 0xa438, 0x86a5 },
	{ 0xa438, 0xad25 }, { 0xa438, 0x0602 }, { 0xa438, 0x6ba4 },
	{ 0xa438, 0x0265 }, { 0xa438, 0x4faf }, { 0xa438, 0x6e9a },
	{ 0xa438, 0xac24 }, { 0xa438, 0x03af }, { 0xa438, 0x6bb4 },
	{ 0xa438, 0xaf6b }, { 0xa438, 0xb602 }, { 0xa438, 0x7ae8 },
	{ 0xa438, 0xaf6c }, { 0xa438, 0xa100 }, { 0xa436, 0xb85e },
	{ 0xa438, 0x6a7f }, { 0xa436, 0xb860 }, { 0xa438, 0x679c },
	{ 0xa436, 0xb862 }, { 0xa438, 0x0d86 }, { 0xa436, 0xb864 },
	{ 0xa438, 0x0c61 }, { 0xa436, 0xb886 }, { 0xa438, 0x6e7c },
	{ 0xa436, 0xb888 }, { 0xa438, 0x6bae }, { 0xa436, 0xb88a },
	{ 0xa438, 0x6c9b }, { 0xa436, 0xb88c }, { 0xa438, 0xffff },
	{ 0xa436, 0xb838 }, { 0xa438, 0x007f }, { 0xb820, 0x0010 },
	{ 0xa436, 0x8629 }, { 0xa438, 0xaf86 }, { 0xa438, 0x41af },
	{ 0xa438, 0x8644 }, { 0xa438, 0xaf88 }, { 0xa438, 0x0caf },
	{ 0xa438, 0x8813 }, { 0xa438, 0xaf88 }, { 0xa438, 0x4baf },
	{ 0xa438, 0x884b }, { 0xa438, 0xaf88 }, { 0xa438, 0x4baf },
	{ 0xa438, 0x884b }, { 0xa438, 0xaf1d }, { 0xa438, 0x8a02 },
	{ 0xa438, 0x864d }, { 0xa438, 0x0210 }, { 0xa438, 0x64af },
	{ 0xa438, 0x1063 }, { 0xa438, 0xf8fa }, { 0xa438, 0xef69 },
	{ 0xa438, 0xe080 }, { 0xa438, 0x4cac }, { 0xa438, 0x2517 },
	{ 0xa438, 0xe080 }, { 0xa438, 0x40ad }, { 0xa438, 0x251a },
	{ 0xa438, 0x0286 }, { 0xa438, 0x7ce0 }, { 0xa438, 0x8040 },
	{ 0xa438, 0xac25 }, { 0xa438, 0x11bf }, { 0xa438, 0x87f4 },
	{ 0xa438, 0x0277 }, { 0xa438, 0xf6ae }, { 0xa438, 0x0902 },
	{ 0xa438, 0x87b3 }, { 0xa438, 0x0287 }, { 0xa438, 0xe902 },
	{ 0xa438, 0x87de }, { 0xa438, 0xef96 }, { 0xa438, 0xfefc },
	{ 0xa438, 0x04f8 }, { 0xa438, 0xe080 }, { 0xa438, 0x18ad },
	{ 0xa438, 0x2611 }, { 0xa438, 0xe08f }, { 0xa438, 0x9cac },
	{ 0xa438, 0x2005 }, { 0xa438, 0x0286 }, { 0xa438, 0x99ae },
	{ 0xa438, 0x0302 }, { 0xa438, 0x8707 }, { 0xa438, 0x0287 },
	{ 0xa438, 0x5002 }, { 0xa438, 0x87de }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8f9 }, { 0xa438, 0xef79 }, { 0xa438, 0xfbbf },
	{ 0xa438, 0x87f7 }, { 0xa438, 0x0278 }, { 0xa438, 0x385c },
	{ 0xa438, 0x2000 }, { 0xa438, 0x0d4d }, { 0xa438, 0xa101 },
	{ 0xa438, 0x51bf }, { 0xa438, 0x87f7 }, { 0xa438, 0x0278 },
	{ 0xa438, 0x385c }, { 0xa438, 0x07ff }, { 0xa438, 0xe38f },
	{ 0xa438, 0x9d1b }, { 0xa438, 0x319f }, { 0xa438, 0x410d },
	{ 0xa438, 0x48e3 }, { 0xa438, 0x8f9e }, { 0xa438, 0x1b31 },
	{ 0xa438, 0x9f38 }, { 0xa438, 0xbf87 }, { 0xa438, 0xfa02 },
	{ 0xa438, 0x7838 }, { 0xa438, 0x5c07 }, { 0xa438, 0xffe3 },
	{ 0xa438, 0x8f9f }, { 0xa438, 0x1b31 }, { 0xa438, 0x9f28 },
	{ 0xa438, 0x0d48 }, { 0xa438, 0xe38f }, { 0xa438, 0xa01b },
	{ 0xa438, 0x319f }, { 0xa438, 0x1fbf }, { 0xa438, 0x87fd },
	{ 0xa438, 0x0278 }, { 0xa438, 0x385c }, { 0xa438, 0x07ff },
	{ 0xa438, 0xe38f }, { 0xa438, 0xa11b }, { 0xa438, 0x319f },
	{ 0xa438, 0x0f0d }, { 0xa438, 0x48e3 }, { 0xa438, 0x8fa2 },
	{ 0xa438, 0x1b31 }, { 0xa438, 0x9f06 }, { 0xa438, 0xee8f },
	{ 0xa438, 0x9c01 }, { 0xa438, 0xae04 }, { 0xa438, 0xee8f },
	{ 0xa438, 0x9c00 }, { 0xa438, 0xffef }, { 0xa438, 0x97fd },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xf8f9 }, { 0xa438, 0xef79 },
	{ 0xa438, 0xfbbf }, { 0xa438, 0x87f7 }, { 0xa438, 0x0278 },
	{ 0xa438, 0x385c }, { 0xa438, 0x2000 }, { 0xa438, 0x0d4d },
	{ 0xa438, 0xa100 }, { 0xa438, 0x20bf }, { 0xa438, 0x87f7 },
	{ 0xa438, 0x0278 }, { 0xa438, 0x385c }, { 0xa438, 0x0600 },
	{ 0xa438, 0x0d49 }, { 0xa438, 0xe38f }, { 0xa438, 0xa31b },
	{ 0xa438, 0x319f }, { 0xa438, 0x0ebf }, { 0xa438, 0x8800 },
	{ 0xa438, 0x0277 }, { 0xa438, 0xf6bf }, { 0xa438, 0x8806 },
	{ 0xa438, 0x0277 }, { 0xa438, 0xf6ae }, { 0xa438, 0x0cbf },
	{ 0xa438, 0x8800 }, { 0xa438, 0x0277 }, { 0xa438, 0xedbf },
	{ 0xa438, 0x8806 }, { 0xa438, 0x0277 }, { 0xa438, 0xedee },
	{ 0xa438, 0x8f9c }, { 0xa438, 0x00ff }, { 0xa438, 0xef97 },
	{ 0xa438, 0xfdfc }, { 0xa438, 0x04f8 }, { 0xa438, 0xf9ef },
	{ 0xa438, 0x79fb }, { 0xa438, 0xbf87 }, { 0xa438, 0xf702 },
	{ 0xa438, 0x7838 }, { 0xa438, 0x5c20 }, { 0xa438, 0x000d },
	{ 0xa438, 0x4da1 }, { 0xa438, 0x014a }, { 0xa438, 0xbf87 },
	{ 0xa438, 0xf702 }, { 0xa438, 0x7838 }, { 0xa438, 0x5c07 },
	{ 0xa438, 0xffe3 }, { 0xa438, 0x8fa4 }, { 0xa438, 0x1b31 },
	{ 0xa438, 0x9f3a }, { 0xa438, 0x0d48 }, { 0xa438, 0xe38f },
	{ 0xa438, 0xa51b }, { 0xa438, 0x319f }, { 0xa438, 0x31bf },
	{ 0xa438, 0x87fa }, { 0xa438, 0x0278 }, { 0xa438, 0x38e3 },
	{ 0xa438, 0x8fa6 }, { 0xa438, 0x1b31 }, { 0xa438, 0x9f24 },
	{ 0xa438, 0x0d48 }, { 0xa438, 0xe38f }, { 0xa438, 0xa71b },
	{ 0xa438, 0x319f }, { 0xa438, 0x1bbf }, { 0xa438, 0x87fd },
	{ 0xa438, 0x0278 }, { 0xa438, 0x38e3 }, { 0xa438, 0x8fa8 },
	{ 0xa438, 0x1b31 }, { 0xa438, 0x9f0e }, { 0xa438, 0xbf88 },
	{ 0xa438, 0x0302 }, { 0xa438, 0x77f6 }, { 0xa438, 0xbf88 },
	{ 0xa438, 0x0902 }, { 0xa438, 0x77f6 }, { 0xa438, 0xae00 },
	{ 0xa438, 0xffef }, { 0xa438, 0x97fd }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8ef }, { 0xa438, 0x79fb }, { 0xa438, 0xe080 },
	{ 0xa438, 0x18ad }, { 0xa438, 0x261c }, { 0xa438, 0xee8f },
	{ 0xa438, 0x9c00 }, { 0xa438, 0xbf88 }, { 0xa438, 0x0002 },
	{ 0xa438, 0x77ed }, { 0xa438, 0xbf88 }, { 0xa438, 0x0602 },
	{ 0xa438, 0x77ed }, { 0xa438, 0xbf88 }, { 0xa438, 0x0302 },
	{ 0xa438, 0x77ed }, { 0xa438, 0xbf88 }, { 0xa438, 0x0902 },
	{ 0xa438, 0x77ed }, { 0xa438, 0xffef }, { 0xa438, 0x97fc },
	{ 0xa438, 0x04f8 }, { 0xa438, 0xe080 }, { 0xa438, 0x40f6 },
	{ 0xa438, 0x25e4 }, { 0xa438, 0x8040 }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8e0 }, { 0xa438, 0x804c }, { 0xa438, 0xf625 },
	{ 0xa438, 0xe480 }, { 0xa438, 0x4cfc }, { 0xa438, 0x0455 },
	{ 0xa438, 0xa4ba }, { 0xa438, 0xf0a6 }, { 0xa438, 0x4af0 },
	{ 0xa438, 0xa64c }, { 0xa438, 0xf0a6 }, { 0xa438, 0x4e66 },
	{ 0xa438, 0xa4b6 }, { 0xa438, 0x55a4 }, { 0xa438, 0xb600 },
	{ 0xa438, 0xac56 }, { 0xa438, 0x11ac }, { 0xa438, 0x56ee },
	{ 0xa438, 0x804c }, { 0xa438, 0x3aaf }, { 0xa438, 0x0627 },
	{ 0xa438, 0xbf88 }, { 0xa438, 0x4802 }, { 0xa438, 0x77ed },
	{ 0xa438, 0xd203 }, { 0xa438, 0xe083 }, { 0xa438, 0x8a0d },
	{ 0xa438, 0x01f6 }, { 0xa438, 0x271b }, { 0xa438, 0x03aa },
	{ 0xa438, 0x0182 }, { 0xa438, 0xe083 }, { 0xa438, 0x890d },
	{ 0xa438, 0x01f6 }, { 0xa438, 0x271b }, { 0xa438, 0x03aa },
	{ 0xa438, 0x0182 }, { 0xa438, 0xe083 }, { 0xa438, 0x880d },
	{ 0xa438, 0x01f6 }, { 0xa438, 0x271b }, { 0xa438, 0x03aa },
	{ 0xa438, 0x0782 }, { 0xa438, 0xbf88 }, { 0xa438, 0x4802 },
	{ 0xa438, 0x77f6 }, { 0xa438, 0xaf16 }, { 0xa438, 0x1500 },
	{ 0xa438, 0xa86a }, { 0xa436, 0xb818 }, { 0xa438, 0x1d84 },
	{ 0xa436, 0xb81a }, { 0xa438, 0x1060 }, { 0xa436, 0xb81c },
	{ 0xa438, 0x0623 }, { 0xa436, 0xb81e }, { 0xa438, 0x15ef },
	{ 0xa436, 0xb850 }, { 0xa438, 0xffff }, { 0xa436, 0xb852 },
	{ 0xa438, 0xffff }, { 0xa436, 0xb878 }, { 0xa438, 0xffff },
	{ 0xa436, 0xb884 }, { 0xa438, 0xffff }, { 0xa436, 0xb832 },
	{ 0xa438, 0x000f }, { 0xa436, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xb82e, 0x0000 }, { 0xa436, 0x8023 }, { 0xa438, 0x0000 },
	{ 0xb820, 0x0000 }
}, mac_r27_mcu[] = {
	{ 0xa436, 0x8023 }, { 0xa438, 0x6100 }, { 0xa436, 0xb82e },
	{ 0xa438, 0x0001 }, { 0xb820, 0x0090 }, { 0xa436, 0xa016 },
	{ 0xa438, 0x0000 }, { 0xa436, 0xa012 }, { 0xa438, 0x0000 },
	{ 0xa436, 0xa014 }, { 0xa438, 0x1800 }, { 0xa438, 0x8010 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x801a }, { 0xa438, 0x1800 }, { 0xa438, 0x801a },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801a }, { 0xa438, 0x1800 },
	{ 0xa438, 0x801a }, { 0xa438, 0x1800 }, { 0xa438, 0x801a },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801a }, { 0xa438, 0xce00 },
	{ 0xa438, 0x2941 }, { 0xa438, 0x8017 }, { 0xa438, 0x2c59 },
	{ 0xa438, 0x8017 }, { 0xa438, 0x1800 }, { 0xa438, 0x0e11 },
	{ 0xa438, 0x8aff }, { 0xa438, 0x1800 }, { 0xa438, 0x0e11 },
	{ 0xa436, 0xa026 }, { 0xa438, 0xffff }, { 0xa436, 0xa024 },
	{ 0xa438, 0xffff }, { 0xa436, 0xa022 }, { 0xa438, 0xffff },
	{ 0xa436, 0xa020 }, { 0xa438, 0xffff }, { 0xa436, 0xa006 },
	{ 0xa438, 0xffff }, { 0xa436, 0xa004 }, { 0xa438, 0xffff },
	{ 0xa436, 0xa002 }, { 0xa438, 0xffff }, { 0xa436, 0xa000 },
	{ 0xa438, 0x0e10 }, { 0xa436, 0xa008 }, { 0xa438, 0x0100 },
	{ 0xa436, 0xa016 }, { 0xa438, 0x0000 }, { 0xa436, 0xa012 },
	{ 0xa438, 0x0ff8 }, { 0xa436, 0xa014 }, { 0xa438, 0x219a },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa436, 0xa152 }, { 0xa438, 0x21a4 },
	{ 0xa436, 0xa154 }, { 0xa438, 0x3fff }, { 0xa436, 0xa156 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa158 }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa15a }, { 0xa438, 0x3fff }, { 0xa436, 0xa15c },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa15e }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa160 }, { 0xa438, 0x3fff }, { 0xa436, 0xa150 },
	{ 0xa438, 0x0001 }, { 0xa436, 0xa016 }, { 0xa438, 0x0010 },
	{ 0xa436, 0xa012 }, { 0xa438, 0x0000 }, { 0xa436, 0xa014 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8010 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8014 }, { 0xa438, 0x1800 }, { 0xa438, 0x801a },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801e }, { 0xa438, 0x1800 },
	{ 0xa438, 0x8026 }, { 0xa438, 0x1800 }, { 0xa438, 0x802e },
	{ 0xa438, 0x1800 }, { 0xa438, 0x8036 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x803a }, { 0xa438, 0xce01 }, { 0xa438, 0x8208 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0028 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x02c5 }, { 0xa438, 0x1000 }, { 0xa438, 0x0304 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x0119 }, { 0xa438, 0xce01 },
	{ 0xa438, 0x8208 }, { 0xa438, 0x1800 }, { 0xa438, 0x009e },
	{ 0xa438, 0xd501 }, { 0xa438, 0xce01 }, { 0xa438, 0xa50f },
	{ 0xa438, 0x8208 }, { 0xa438, 0xd500 }, { 0xa438, 0xaa0f },
	{ 0xa438, 0x1800 }, { 0xa438, 0x015b }, { 0xa438, 0xd501 },
	{ 0xa438, 0xce01 }, { 0xa438, 0xa50f }, { 0xa438, 0x8208 },
	{ 0xa438, 0xd500 }, { 0xa438, 0xaa0f }, { 0xa438, 0x1800 },
	{ 0xa438, 0x01a9 }, { 0xa438, 0xd501 }, { 0xa438, 0xce01 },
	{ 0xa438, 0xa50f }, { 0xa438, 0x8208 }, { 0xa438, 0xd500 },
	{ 0xa438, 0xaa0f }, { 0xa438, 0x1800 }, { 0xa438, 0x01f4 },
	{ 0xa438, 0x8208 }, { 0xa438, 0xd500 }, { 0xa438, 0x1800 },
	{ 0xa438, 0x02a5 }, { 0xa438, 0xa208 }, { 0xa438, 0xd500 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x02b8 }, { 0xa436, 0xa08e },
	{ 0xa438, 0x02b7 }, { 0xa436, 0xa08c }, { 0xa438, 0x02a4 },
	{ 0xa436, 0xa08a }, { 0xa438, 0x01e7 }, { 0xa436, 0xa088 },
	{ 0xa438, 0x019c }, { 0xa436, 0xa086 }, { 0xa438, 0x014e },
	{ 0xa436, 0xa084 }, { 0xa438, 0x009d }, { 0xa436, 0xa082 },
	{ 0xa438, 0x0117 }, { 0xa436, 0xa080 }, { 0xa438, 0x0027 },
	{ 0xa436, 0xa090 }, { 0xa438, 0x00ff }, { 0xa436, 0xa016 },
	{ 0xa438, 0x0020 }, { 0xa436, 0xa012 }, { 0xa438, 0x0000 },
	{ 0xa436, 0xa014 }, { 0xa438, 0x1800 }, { 0xa438, 0x8010 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x801d }, { 0xa438, 0x1800 },
	{ 0xa438, 0x803b }, { 0xa438, 0x1800 }, { 0xa438, 0x8087 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x808e }, { 0xa438, 0x1800 },
	{ 0xa438, 0x809d }, { 0xa438, 0x1800 }, { 0xa438, 0x80b7 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x80c4 }, { 0xa438, 0xd1bc },
	{ 0xa438, 0xd040 }, { 0xa438, 0x1000 }, { 0xa438, 0x1cd2 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fba }, { 0xa438, 0xd700 },
	{ 0xa438, 0x273d }, { 0xa438, 0x801b }, { 0xa438, 0x1800 },
	{ 0xa438, 0x07d1 }, { 0xa438, 0x1800 }, { 0xa438, 0x080e },
	{ 0xa438, 0xd700 }, { 0xa438, 0x37c9 }, { 0xa438, 0x8032 },
	{ 0xa438, 0x33a9 }, { 0xa438, 0x802a }, { 0xa438, 0xd705 },
	{ 0xa438, 0x4084 }, { 0xa438, 0xd1f4 }, { 0xa438, 0xd048 },
	{ 0xa438, 0xf013 }, { 0xa438, 0xd1b7 }, { 0xa438, 0xd04b },
	{ 0xa438, 0xf010 }, { 0xa438, 0xd705 }, { 0xa438, 0x4084 },
	{ 0xa438, 0xd1f4 }, { 0xa438, 0xd048 }, { 0xa438, 0xf00b },
	{ 0xa438, 0xd1b7 }, { 0xa438, 0xd04b }, { 0xa438, 0xf008 },
	{ 0xa438, 0xd705 }, { 0xa438, 0x4084 }, { 0xa438, 0xd1f4 },
	{ 0xa438, 0xd048 }, { 0xa438, 0xf003 }, { 0xa438, 0xd1b7 },
	{ 0xa438, 0xd04b }, { 0xa438, 0x1800 }, { 0xa438, 0x14cc },
	{ 0xa438, 0xd700 }, { 0xa438, 0x2b59 }, { 0xa438, 0x803f },
	{ 0xa438, 0xf003 }, { 0xa438, 0x1800 }, { 0xa438, 0x118f },
	{ 0xa438, 0x6060 }, { 0xa438, 0x1800 }, { 0xa438, 0x1167 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x60c7 }, { 0xa438, 0xd704 },
	{ 0xa438, 0x609f }, { 0xa438, 0xd705 }, { 0xa438, 0x4043 },
	{ 0xa438, 0xf003 }, { 0xa438, 0x1800 }, { 0xa438, 0x1150 },
	{ 0xa438, 0x0c03 }, { 0xa438, 0x1502 }, { 0xa438, 0x8702 },
	{ 0xa438, 0x8011 }, { 0xa438, 0x9503 }, { 0xa438, 0x800a },
	{ 0xa438, 0x81a0 }, { 0xa438, 0x8302 }, { 0xa438, 0x8480 },
	{ 0xa438, 0x8686 }, { 0xa438, 0xcde0 }, { 0xa438, 0xd1ff },
	{ 0xa438, 0xd049 }, { 0xa438, 0x1000 }, { 0xa438, 0x1cd2 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fba }, { 0xa438, 0xd705 },
	{ 0xa438, 0x417e }, { 0xa438, 0x0c03 }, { 0xa438, 0x1502 },
	{ 0xa438, 0xa011 }, { 0xa438, 0x9503 }, { 0xa438, 0xd1c8 },
	{ 0xa438, 0xd045 }, { 0xa438, 0x1000 }, { 0xa438, 0x1cd2 },
	{ 0xa438, 0xd700 }, { 0xa438, 0x5fba }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xa702 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xa00a }, { 0xa438, 0xa1a0 }, { 0xa438, 0xa480 },
	{ 0xa438, 0xa686 }, { 0xa438, 0xd705 }, { 0xa438, 0x605e },
	{ 0xa438, 0xa302 }, { 0xa438, 0x9503 }, { 0xa438, 0xd700 },
	{ 0xa438, 0x37c9 }, { 0xa438, 0x8083 }, { 0xa438, 0x33a9 },
	{ 0xa438, 0x807f }, { 0xa438, 0xd178 }, { 0xa438, 0xd04b },
	{ 0xa438, 0x1800 }, { 0xa438, 0x115d }, { 0xa438, 0xd1c8 },
	{ 0xa438, 0xd04b }, { 0xa438, 0x1800 }, { 0xa438, 0x115d },
	{ 0xa438, 0xd1e6 }, { 0xa438, 0xd04b }, { 0xa438, 0x1800 },
	{ 0xa438, 0x115d }, { 0xa438, 0xd71f }, { 0xa438, 0x6080 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x1800 }, { 0xa438, 0x1bc0 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x1bc4 }, { 0xa438, 0x4134 },
	{ 0xa438, 0xd115 }, { 0xa438, 0xd04f }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1d0b }, { 0xa438, 0x1000 }, { 0xa438, 0x80ad },
	{ 0xa438, 0x1800 }, { 0xa438, 0x01f2 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1d0b }, { 0xa438, 0x1000 }, { 0xa438, 0x80ad },
	{ 0xa438, 0x1800 }, { 0xa438, 0x01f9 }, { 0xa438, 0x2969 },
	{ 0xa438, 0x80a3 }, { 0xa438, 0xd700 }, { 0xa438, 0x606b },
	{ 0xa438, 0xd701 }, { 0xa438, 0x60b4 }, { 0xa438, 0x1000 },
	{ 0xa438, 0x80ad }, { 0xa438, 0x1800 }, { 0xa438, 0x0551 },
	{ 0xa438, 0xd196 }, { 0xa438, 0xd04d }, { 0xa438, 0x1000 },
	{ 0xa438, 0x80ad }, { 0xa438, 0x1800 }, { 0xa438, 0x054d },
	{ 0xa438, 0xd208 }, { 0xa438, 0x0c09 }, { 0xa438, 0x1301 },
	{ 0xa438, 0x1000 }, { 0xa438, 0x1cd2 }, { 0xa438, 0xd701 },
	{ 0xa438, 0x5fa3 }, { 0xa438, 0xb302 }, { 0xa438, 0xd200 },
	{ 0xa438, 0x0800 }, { 0xa438, 0xd705 }, { 0xa438, 0x6064 },
	{ 0xa438, 0x1800 }, { 0xa438, 0x140a }, { 0xa438, 0x8810 },
	{ 0xa438, 0xd199 }, { 0xa438, 0xd04b }, { 0xa438, 0x1000 },
	{ 0xa438, 0x1cd2 }, { 0xa438, 0xd700 }, { 0xa438, 0x5fba },
	{ 0xa438, 0x1800 }, { 0xa438, 0x140a }, { 0xa438, 0x8910 },
	{ 0xa438, 0xd704 }, { 0xa438, 0x61a8 }, { 0xa438, 0x8480 },
	{ 0xa438, 0x8604 }, { 0xa438, 0x8302 }, { 0xa438, 0x81a0 },
	{ 0xa438, 0xd703 }, { 0xa438, 0x4112 }, { 0xa438, 0x0c03 },
	{ 0xa438, 0x1502 }, { 0xa438, 0xa990 }, { 0xa438, 0x9503 },
	{ 0xa438, 0xb904 }, { 0xa438, 0xf002 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0800 }, { 0xa436, 0xa10e }, { 0xa438, 0x1de8 },
	{ 0xa436, 0xa10c }, { 0xa438, 0x1352 }, { 0xa436, 0xa10a },
	{ 0xa438, 0x0545 }, { 0xa436, 0xa108 }, { 0xa438, 0x01ed },
	{ 0xa436, 0xa106 }, { 0xa438, 0x1bbf }, { 0xa436, 0xa104 },
	{ 0xa438, 0x114b }, { 0xa436, 0xa102 }, { 0xa438, 0x14bf },
	{ 0xa436, 0xa100 }, { 0xa438, 0x07ce }, { 0xa436, 0xa110 },
	{ 0xa438, 0x00ff }, { 0xa436, 0xa016 }, { 0xa438, 0x0020 },
	{ 0xa436, 0xa012 }, { 0xa438, 0x1ff8 }, { 0xa436, 0xa014 },
	{ 0xa438, 0xd1ce }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa438, 0x0000 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x0000 }, { 0xa436, 0xa164 },
	{ 0xa438, 0x07fc }, { 0xa436, 0xa166 }, { 0xa438, 0x143d },
	{ 0xa436, 0xa168 }, { 0xa438, 0x3fff }, { 0xa436, 0xa16a },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa16c }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa16e }, { 0xa438, 0x3fff }, { 0xa436, 0xa170 },
	{ 0xa438, 0x3fff }, { 0xa436, 0xa172 }, { 0xa438, 0x3fff },
	{ 0xa436, 0xa162 }, { 0xa438, 0x0003 }, { 0xa436, 0xb87c },
	{ 0xa438, 0x8994 }, { 0xa436, 0xb87e }, { 0xa438, 0xaf89 },
	{ 0xa438, 0xacaf }, { 0xa438, 0x89e4 }, { 0xa438, 0xaf89 },
	{ 0xa438, 0xecaf }, { 0xa438, 0x8a04 }, { 0xa438, 0xaf8a },
	{ 0xa438, 0x2eaf }, { 0xa438, 0x8a4a }, { 0xa438, 0xaf8d },
	{ 0xa438, 0x31af }, { 0xa438, 0x8dc6 }, { 0xa438, 0x1f55 },
	{ 0xa438, 0xe18f }, { 0xa438, 0xe3a1 }, { 0xa438, 0x0007 },
	{ 0xa438, 0xee86 }, { 0xa438, 0xe900 }, { 0xa438, 0xaf4f },
	{ 0xa438, 0x9ead }, { 0xa438, 0x281b }, { 0xa438, 0xe18f },
	{ 0xa438, 0xfcef }, { 0xa438, 0x71bf }, { 0xa438, 0x74f6 },
	{ 0xa438, 0x027e }, { 0xa438, 0xd2ef }, { 0xa438, 0x641c },
	{ 0xa438, 0x670d }, { 0xa438, 0x67ef }, { 0xa438, 0x461f },
	{ 0xa438, 0x00bf }, { 0xa438, 0x74f6 }, { 0xa438, 0x027e },
	{ 0xa438, 0xdee1 }, { 0xa438, 0x8fe3 }, { 0xa438, 0x0d11 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xe313 }, { 0xa438, 0xaeca },
	{ 0xa438, 0x028d }, { 0xa438, 0xd1d3 }, { 0xa438, 0x01af },
	{ 0xa438, 0x40d1 }, { 0xa438, 0xbf7a }, { 0xa438, 0x6102 },
	{ 0xa438, 0x7d44 }, { 0xa438, 0xa100 }, { 0xa438, 0x09e0 },
	{ 0xa438, 0x8ffa }, { 0xa438, 0xe18f }, { 0xa438, 0xfbaf },
	{ 0xa438, 0x683d }, { 0xa438, 0x027f }, { 0xa438, 0xa9af },
	{ 0xa438, 0x682c }, { 0xa438, 0xbf8e }, { 0xa438, 0x4102 },
	{ 0xa438, 0x7d44 }, { 0xa438, 0xe58f }, { 0xa438, 0xecbf },
	{ 0xa438, 0x74cc }, { 0xa438, 0x027d }, { 0xa438, 0x44e3 },
	{ 0xa438, 0x8fed }, { 0xa438, 0x0d31 }, { 0xa438, 0xf63f },
	{ 0xa438, 0x0d11 }, { 0xa438, 0xf62f }, { 0xa438, 0x1b13 },
	{ 0xa438, 0xad2f }, { 0xa438, 0x06bf }, { 0xa438, 0x8e41 },
	{ 0xa438, 0x027c }, { 0xa438, 0xf9d1 }, { 0xa438, 0x01af },
	{ 0xa438, 0x5974 }, { 0xa438, 0xee88 }, { 0xa438, 0x8600 },
	{ 0xa438, 0xe08f }, { 0xa438, 0xebad }, { 0xa438, 0x200b },
	{ 0xa438, 0xe18f }, { 0xa438, 0xecbf }, { 0xa438, 0x8e41 },
	{ 0xa438, 0x027d }, { 0xa438, 0x25ae }, { 0xa438, 0x04ee },
	{ 0xa438, 0x8feb }, { 0xa438, 0x01af }, { 0xa438, 0x5945 },
	{ 0xa438, 0xad28 }, { 0xa438, 0x2ce0 }, { 0xa438, 0x8fea },
	{ 0xa438, 0xa000 }, { 0xa438, 0x0502 }, { 0xa438, 0x8af0 },
	{ 0xa438, 0xae1e }, { 0xa438, 0xa001 }, { 0xa438, 0x0502 },
	{ 0xa438, 0x8b9f }, { 0xa438, 0xae16 }, { 0xa438, 0xa002 },
	{ 0xa438, 0x0502 }, { 0xa438, 0x8c0f }, { 0xa438, 0xae0e },
	{ 0xa438, 0xa003 }, { 0xa438, 0x0502 }, { 0xa438, 0x8c95 },
	{ 0xa438, 0xae06 }, { 0xa438, 0xa004 }, { 0xa438, 0x0302 },
	{ 0xa438, 0x8d08 }, { 0xa438, 0xaf63 }, { 0xa438, 0x8902 },
	{ 0xa438, 0x8a7f }, { 0xa438, 0xaf63 }, { 0xa438, 0x81f8 },
	{ 0xa438, 0xef49 }, { 0xa438, 0xf8e0 }, { 0xa438, 0x8015 },
	{ 0xa438, 0xad21 }, { 0xa438, 0x19bf }, { 0xa438, 0x7bd8 },
	{ 0xa438, 0x027c }, { 0xa438, 0xf9bf }, { 0xa438, 0x7bf3 },
	{ 0xa438, 0x027d }, { 0xa438, 0x44bf }, { 0xa438, 0x7bf6 },
	{ 0xa438, 0x027c }, { 0xa438, 0xf902 }, { 0xa438, 0x638e },
	{ 0xa438, 0xee8f }, { 0xa438, 0xea00 }, { 0xa438, 0xe080 },
	{ 0xa438, 0x16ad }, { 0xa438, 0x233d }, { 0xa438, 0xbf7b },
	{ 0xa438, 0xf302 }, { 0xa438, 0x7d44 }, { 0xa438, 0xbf7a },
	{ 0xa438, 0x9402 }, { 0xa438, 0x7cf9 }, { 0xa438, 0xbf8e },
	{ 0xa438, 0x4402 }, { 0xa438, 0x7cf9 }, { 0xa438, 0xbf7a },
	{ 0xa438, 0xa602 }, { 0xa438, 0x7cf9 }, { 0xa438, 0xbf7a },
	{ 0xa438, 0xa302 }, { 0xa438, 0x7cf9 }, { 0xa438, 0xbf7a },
	{ 0xa438, 0xa902 }, { 0xa438, 0x7cf9 }, { 0xa438, 0xbf7a },
	{ 0xa438, 0xac02 }, { 0xa438, 0x7cf9 }, { 0xa438, 0xbf8e },
	{ 0xa438, 0x4702 }, { 0xa438, 0x7cf9 }, { 0xa438, 0xbf8e },
	{ 0xa438, 0x4a02 }, { 0xa438, 0x7cf9 }, { 0xa438, 0x0263 },
	{ 0xa438, 0x8eee }, { 0xa438, 0x8fea }, { 0xa438, 0x00bf },
	{ 0xa438, 0x7c02 }, { 0xa438, 0x027c }, { 0xa438, 0xf9fc },
	{ 0xa438, 0xef94 }, { 0xa438, 0xfc04 }, { 0xa438, 0xf8f9 },
	{ 0xa438, 0xfbef }, { 0xa438, 0x79fb }, { 0xa438, 0xe080 },
	{ 0xa438, 0x15ac }, { 0xa438, 0x2103 }, { 0xa438, 0xaf8b },
	{ 0xa438, 0x70ee }, { 0xa438, 0x8888 }, { 0xa438, 0x00ee },
	{ 0xa438, 0x888a }, { 0xa438, 0x00ee }, { 0xa438, 0x888b },
	{ 0xa438, 0x00bf }, { 0xa438, 0x7bd8 }, { 0xa438, 0x027d },
	{ 0xa438, 0x02bf }, { 0xa438, 0x6000 }, { 0xa438, 0xd788 },
	{ 0xa438, 0x881f }, { 0xa438, 0x44d4 }, { 0xa438, 0x000c },
	{ 0xa438, 0x0273 }, { 0xa438, 0x3b02 }, { 0xa438, 0x7fa9 },
	{ 0xa438, 0xac28 }, { 0xa438, 0x05ac }, { 0xa438, 0x290d },
	{ 0xa438, 0xae18 }, { 0xa438, 0xe188 }, { 0xa438, 0x98bf },
	{ 0xa438, 0x7be1 }, { 0xa438, 0x027d }, { 0xa438, 0x25ae },
	{ 0xa438, 0x18e1 }, { 0xa438, 0x8898 }, { 0xa438, 0x0d11 },
	{ 0xa438, 0xbf7b }, { 0xa438, 0xe102 }, { 0xa438, 0x7d25 },
	{ 0xa438, 0xae0b }, { 0xa438, 0xe188 }, { 0xa438, 0x980d },
	{ 0xa438, 0x12bf }, { 0xa438, 0x7be1 }, { 0xa438, 0x027d },
	{ 0xa438, 0x25bf }, { 0xa438, 0x88a0 }, { 0xa438, 0xda19 },
	{ 0xa438, 0xdb19 }, { 0xa438, 0xd819 }, { 0xa438, 0xd91f },
	{ 0xa438, 0x77bf }, { 0xa438, 0x88b1 }, { 0xa438, 0xde19 },
	{ 0xa438, 0xdf19 }, { 0xa438, 0xdc19 }, { 0xa438, 0xdd19 },
	{ 0xa438, 0x17a7 }, { 0xa438, 0x0004 }, { 0xa438, 0xf302 },
	{ 0xa438, 0x63cd }, { 0xa438, 0xee8f }, { 0xa438, 0xea01 },
	{ 0xa438, 0xe080 }, { 0xa438, 0x16ad }, { 0xa438, 0x2319 },
	{ 0xa438, 0xee88 }, { 0xa438, 0x8800 }, { 0xa438, 0xee88 },
	{ 0xa438, 0x8a00 }, { 0xa438, 0xee88 }, { 0xa438, 0x8b00 },
	{ 0xa438, 0xbf8e }, { 0xa438, 0x4402 }, { 0xa438, 0x7d02 },
	{ 0xa438, 0x0263 }, { 0xa438, 0xcdee }, { 0xa438, 0x8fea },
	{ 0xa438, 0x0102 }, { 0xa438, 0x70de }, { 0xa438, 0xbf7c },
	{ 0xa438, 0x0202 }, { 0xa438, 0x7d02 }, { 0xa438, 0xffef },
	{ 0xa438, 0x97ff }, { 0xa438, 0xfdfc }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xf9fa }, { 0xa438, 0xef69 }, { 0xa438, 0xfae0 },
	{ 0xa438, 0x888a }, { 0xa438, 0xe188 }, { 0xa438, 0x8b14 },
	{ 0xa438, 0xe488 }, { 0xa438, 0x8ae5 }, { 0xa438, 0x888b },
	{ 0xa438, 0xbf88 }, { 0xa438, 0x94d8 }, { 0xa438, 0x19d9 },
	{ 0xa438, 0xef64 }, { 0xa438, 0xe088 }, { 0xa438, 0x8ae1 },
	{ 0xa438, 0x888b }, { 0xa438, 0x1b46 }, { 0xa438, 0x9f30 },
	{ 0xa438, 0x1f44 }, { 0xa438, 0xe488 }, { 0xa438, 0x8ae5 },
	{ 0xa438, 0x888b }, { 0xa438, 0xe080 }, { 0xa438, 0x15ad },
	{ 0xa438, 0x211a }, { 0xa438, 0x0260 }, { 0xa438, 0xece0 },
	{ 0xa438, 0x8016 }, { 0xa438, 0xad23 }, { 0xa438, 0x1602 },
	{ 0xa438, 0x7c86 }, { 0xa438, 0xef47 }, { 0xa438, 0xe48f },
	{ 0xa438, 0xe9e5 }, { 0xa438, 0x8fe8 }, { 0xa438, 0xee8f },
	{ 0xa438, 0xea02 }, { 0xa438, 0xae0b }, { 0xa438, 0x028c },
	{ 0xa438, 0x2eae }, { 0xa438, 0x0602 }, { 0xa438, 0x8bfe },
	{ 0xa438, 0x0270 }, { 0xa438, 0xdefe }, { 0xa438, 0xef96 },
	{ 0xa438, 0xfefd }, { 0xa438, 0xfc04 }, { 0xa438, 0xf8e1 },
	{ 0xa438, 0x8888 }, { 0xa438, 0x11e5 }, { 0xa438, 0x8888 },
	{ 0xa438, 0xad2a }, { 0xa438, 0x04ee }, { 0xa438, 0x8888 },
	{ 0xa438, 0x00fc }, { 0xa438, 0x04f8 }, { 0xa438, 0xfafb },
	{ 0xa438, 0xe08f }, { 0xa438, 0xe9e1 }, { 0xa438, 0x8fe8 },
	{ 0xa438, 0xef64 }, { 0xa438, 0x1f00 }, { 0xa438, 0xe18f },
	{ 0xa438, 0xe6ef }, { 0xa438, 0x7402 }, { 0xa438, 0x7ca1 },
	{ 0xa438, 0xad50 }, { 0xa438, 0x0302 }, { 0xa438, 0x8c2e },
	{ 0xa438, 0xfffe }, { 0xa438, 0xfc04 }, { 0xa438, 0xf8fa },
	{ 0xa438, 0xef69 }, { 0xa438, 0xfbbf }, { 0xa438, 0x7bf3 },
	{ 0xa438, 0x027d }, { 0xa438, 0x44ac }, { 0xa438, 0x284c },
	{ 0xa438, 0x0264 }, { 0xa438, 0x1cbf }, { 0xa438, 0x8e47 },
	{ 0xa438, 0x027d }, { 0xa438, 0x02bf }, { 0xa438, 0x8e4a },
	{ 0xa438, 0x027d }, { 0xa438, 0x02d1 }, { 0xa438, 0x43b1 },
	{ 0xa438, 0xfebf }, { 0xa438, 0x7aa6 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x7aa3 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x7aa9 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x7aac }, { 0xa438, 0x027d },
	{ 0xa438, 0x02d1 }, { 0xa438, 0x80e0 }, { 0xa438, 0x8888 },
	{ 0xa438, 0x100e }, { 0xa438, 0x11b0 }, { 0xa438, 0xfcbf },
	{ 0xa438, 0x7a94 }, { 0xa438, 0x027d }, { 0xa438, 0x2502 },
	{ 0xa438, 0x7c86 }, { 0xa438, 0xef47 }, { 0xa438, 0xe48f },
	{ 0xa438, 0xe9e5 }, { 0xa438, 0x8fe8 }, { 0xa438, 0xee8f },
	{ 0xa438, 0xea03 }, { 0xa438, 0xae07 }, { 0xa438, 0xee8f },
	{ 0xa438, 0xea01 }, { 0xa438, 0x0270 }, { 0xa438, 0xdeff },
	{ 0xa438, 0xef96 }, { 0xa438, 0xfefc }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xf9fa }, { 0xa438, 0xfbef }, { 0xa438, 0x79fb },
	{ 0xa438, 0xbf7a }, { 0xa438, 0x9402 }, { 0xa438, 0x7d44 },
	{ 0xa438, 0xef21 }, { 0xa438, 0xbf7a }, { 0xa438, 0xb802 },
	{ 0xa438, 0x7d44 }, { 0xa438, 0x1f21 }, { 0xa438, 0x9e19 },
	{ 0xa438, 0xe08f }, { 0xa438, 0xe9e1 }, { 0xa438, 0x8fe8 },
	{ 0xa438, 0xef64 }, { 0xa438, 0x1f00 }, { 0xa438, 0xe18f },
	{ 0xa438, 0xe4ef }, { 0xa438, 0x7402 }, { 0xa438, 0x7ca1 },
	{ 0xa438, 0xad50 }, { 0xa438, 0x3dee }, { 0xa438, 0x8fe7 },
	{ 0xa438, 0x01bf }, { 0xa438, 0x7a94 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x7aa6 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x7aa3 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x7aa9 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x7aac }, { 0xa438, 0x027d },
	{ 0xa438, 0x02bf }, { 0xa438, 0x8e47 }, { 0xa438, 0x027c },
	{ 0xa438, 0xf9bf }, { 0xa438, 0x8e4a }, { 0xa438, 0x027c },
	{ 0xa438, 0xf902 }, { 0xa438, 0x7c86 }, { 0xa438, 0xef47 },
	{ 0xa438, 0xe48f }, { 0xa438, 0xe9e5 }, { 0xa438, 0x8fe8 },
	{ 0xa438, 0xee8f }, { 0xa438, 0xea04 }, { 0xa438, 0xffef },
	{ 0xa438, 0x97ff }, { 0xa438, 0xfefd }, { 0xa438, 0xfc04 },
	{ 0xa438, 0xf8fa }, { 0xa438, 0xfbe0 }, { 0xa438, 0x8fe9 },
	{ 0xa438, 0xe18f }, { 0xa438, 0xe8ef }, { 0xa438, 0x641f },
	{ 0xa438, 0x00e1 }, { 0xa438, 0x8fe5 }, { 0xa438, 0xef74 },
	{ 0xa438, 0x027c }, { 0xa438, 0xa1ad }, { 0xa438, 0x500d },
	{ 0xa438, 0x0263 }, { 0xa438, 0x8e02 }, { 0xa438, 0x8bfe },
	{ 0xa438, 0xee8f }, { 0xa438, 0xea01 }, { 0xa438, 0x0270 },
	{ 0xa438, 0xdeff }, { 0xa438, 0xfefc }, { 0xa438, 0x04e3 },
	{ 0xa438, 0x8fd8 }, { 0xa438, 0xe787 }, { 0xa438, 0x75e4 },
	{ 0xa438, 0x8fe1 }, { 0xa438, 0xe58f }, { 0xa438, 0xe2bf },
	{ 0xa438, 0x8fd9 }, { 0xa438, 0xef32 }, { 0xa438, 0x0c31 },
	{ 0xa438, 0x1a93 }, { 0xa438, 0xdc19 }, { 0xa438, 0xdd02 },
	{ 0xa438, 0x7fa9 }, { 0xa438, 0xac2a }, { 0xa438, 0x18e0 },
	{ 0xa438, 0x8fe1 }, { 0xa438, 0xe18f }, { 0xa438, 0xe2ef },
	{ 0xa438, 0x74e1 }, { 0xa438, 0x8775 }, { 0xa438, 0x1f00 },
	{ 0xa438, 0xef64 }, { 0xa438, 0xe18f }, { 0xa438, 0xd8e5 },
	{ 0xa438, 0x8775 }, { 0xa438, 0xaf4d }, { 0xa438, 0x72bf },
	{ 0xa438, 0x7b3c }, { 0xa438, 0xef32 }, { 0xa438, 0x4b03 },
	{ 0xa438, 0x1a93 }, { 0xa438, 0x027d }, { 0xa438, 0x44ef },
	{ 0xa438, 0x64e1 }, { 0xa438, 0x8fff }, { 0xa438, 0x1f00 },
	{ 0xa438, 0xef74 }, { 0xa438, 0x1b67 }, { 0xa438, 0xac4f },
	{ 0xa438, 0xcee0 }, { 0xa438, 0x8ffd }, { 0xa438, 0xe18f },
	{ 0xa438, 0xfeef }, { 0xa438, 0x64e0 }, { 0xa438, 0x8fe1 },
	{ 0xa438, 0xe18f }, { 0xa438, 0xe2ef }, { 0xa438, 0x7402 },
	{ 0xa438, 0x7c53 }, { 0xa438, 0xac50 }, { 0xa438, 0x02ae },
	{ 0xa438, 0xb6e1 }, { 0xa438, 0x8775 }, { 0xa438, 0x1f00 },
	{ 0xa438, 0xef64 }, { 0xa438, 0xe18f }, { 0xa438, 0xfcef },
	{ 0xa438, 0x711c }, { 0xa438, 0x670d }, { 0xa438, 0x67ef },
	{ 0xa438, 0x46e5 }, { 0xa438, 0x8775 }, { 0xa438, 0xef32 },
	{ 0xa438, 0xd101 }, { 0xa438, 0xa300 }, { 0xa438, 0x02ae },
	{ 0xa438, 0x050c }, { 0xa438, 0x1183 }, { 0xa438, 0xaef6 },
	{ 0xa438, 0xe08f }, { 0xa438, 0xe31e }, { 0xa438, 0x10e5 },
	{ 0xa438, 0x8fe3 }, { 0xa438, 0xae89 }, { 0xa438, 0xe287 },
	{ 0xa438, 0x75e6 }, { 0xa438, 0x8fd8 }, { 0xa438, 0x1f22 },
	{ 0xa438, 0xaf4d }, { 0xa438, 0x42f8 }, { 0xa438, 0xf9ef },
	{ 0xa438, 0x59fa }, { 0xa438, 0xfbbf }, { 0xa438, 0x8fee },
	{ 0xa438, 0x027f }, { 0xa438, 0xa90d }, { 0xa438, 0x1149 },
	{ 0xa438, 0x041a }, { 0xa438, 0x91d7 }, { 0xa438, 0x8df3 },
	{ 0xa438, 0xd68e }, { 0xa438, 0x2302 }, { 0xa438, 0x72aa },
	{ 0xa438, 0xfffe }, { 0xa438, 0xef95 }, { 0xa438, 0xfdfc },
	{ 0xa438, 0x0400 }, { 0xa438, 0x7591 }, { 0xa438, 0x0275 },
	{ 0xa438, 0x4404 }, { 0xa438, 0x758e }, { 0xa438, 0x2675 },
	{ 0xa438, 0x4100 }, { 0xa438, 0x8e26 }, { 0xa438, 0x028e },
	{ 0xa438, 0x2304 }, { 0xa438, 0x759d }, { 0xa438, 0x2675 },
	{ 0xa438, 0x4700 }, { 0xa438, 0x8e32 }, { 0xa438, 0x028e },
	{ 0xa438, 0x2f04 }, { 0xa438, 0x8e2c }, { 0xa438, 0x268e },
	{ 0xa438, 0x2900 }, { 0xa438, 0x8e3e }, { 0xa438, 0x028e },
	{ 0xa438, 0x3b04 }, { 0xa438, 0x8e38 }, { 0xa438, 0x268e },
	{ 0xa438, 0x35fe }, { 0xa438, 0xad96 }, { 0xa438, 0xdcad },
	{ 0xa438, 0x96ba }, { 0xa438, 0xad96 }, { 0xa438, 0x98ad },
	{ 0xa438, 0x9676 }, { 0xa438, 0xad98 }, { 0xa438, 0x54ad },
	{ 0xa438, 0x9876 }, { 0xa438, 0xae38 }, { 0xa438, 0x54ae },
	{ 0xa438, 0x38fe }, { 0xa438, 0xae3a }, { 0xa438, 0xdcae },
	{ 0xa438, 0x3abb }, { 0xa438, 0xbf14 }, { 0xa438, 0x99bd },
	{ 0xa438, 0xe0cc }, { 0xa438, 0xbdc8 }, { 0xa438, 0xddbd },
	{ 0xa438, 0xc800 }, { 0xa436, 0xb85e }, { 0xa438, 0x4f9a },
	{ 0xa436, 0xb860 }, { 0xa438, 0x40cf }, { 0xa436, 0xb862 },
	{ 0xa438, 0x6829 }, { 0xa436, 0xb864 }, { 0xa438, 0x5972 },
	{ 0xa436, 0xb886 }, { 0xa438, 0x5941 }, { 0xa436, 0xb888 },
	{ 0xa438, 0x636b }, { 0xa436, 0xb88a }, { 0xa438, 0x4d6b },
	{ 0xa436, 0xb88c }, { 0xa438, 0x4d40 }, { 0xa436, 0xb838 },
	{ 0xa438, 0x00ff }, { 0xb820, 0x0010 }, { 0xa436, 0x8608 },
	{ 0xa438, 0xaf86 }, { 0xa438, 0xdaaf }, { 0xa438, 0x894c },
	{ 0xa438, 0xaf8a }, { 0xa438, 0xf8af }, { 0xa438, 0x8bf3 },
	{ 0xa438, 0xaf8b }, { 0xa438, 0xf3af }, { 0xa438, 0x8bf3 },
	{ 0xa438, 0xaf8b }, { 0xa438, 0xf3af }, { 0xa438, 0x8bf3 },
	{ 0xa438, 0x006f }, { 0xa438, 0x4a03 }, { 0xa438, 0x6f47 },
	{ 0xa438, 0x266f }, { 0xa438, 0x5900 }, { 0xa438, 0x6f4d },
	{ 0xa438, 0x016f }, { 0xa438, 0x5004 }, { 0xa438, 0x6f56 },
	{ 0xa438, 0x056f }, { 0xa438, 0x5f06 }, { 0xa438, 0x6f5c },
	{ 0xa438, 0x2774 }, { 0xa438, 0x7800 }, { 0xa438, 0x6f68 },
	{ 0xa438, 0x246f }, { 0xa438, 0x6b20 }, { 0xa438, 0x6f6e },
	{ 0xa438, 0x206f }, { 0xa438, 0x7410 }, { 0xa438, 0x7469 },
	{ 0xa438, 0x1074 }, { 0xa438, 0x6c10 }, { 0xa438, 0x746f },
	{ 0xa438, 0x1074 }, { 0xa438, 0x7225 }, { 0xa438, 0x8bfc },
	{ 0xa438, 0x008c }, { 0xa438, 0x0802 }, { 0xa438, 0x8c02 },
	{ 0xa438, 0x038b }, { 0xa438, 0xff04 }, { 0xa438, 0x6eed },
	{ 0xa438, 0x278c }, { 0xa438, 0x0520 }, { 0xa438, 0x74da },
	{ 0xa438, 0x2074 }, { 0xa438, 0xdd20 }, { 0xa438, 0x74e0 },
	{ 0xa438, 0x0074 }, { 0xa438, 0xe300 }, { 0xa438, 0x6ef3 },
	{ 0xa438, 0x006e }, { 0xa438, 0xf600 }, { 0xa438, 0x6ef9 },
	{ 0xa438, 0x006e }, { 0xa438, 0xfc00 }, { 0xa438, 0x6eff },
	{ 0xa438, 0x006f }, { 0xa438, 0x0200 }, { 0xa438, 0x6f05 },
	{ 0xa438, 0x026f }, { 0xa438, 0x0802 }, { 0xa438, 0x6f0b },
	{ 0xa438, 0x026f }, { 0xa438, 0x0e02 }, { 0xa438, 0x6f11 },
	{ 0xa438, 0x026f }, { 0xa438, 0x1402 }, { 0xa438, 0x6f17 },
	{ 0xa438, 0x226f }, { 0xa438, 0x1a00 }, { 0xa438, 0x723e },
	{ 0xa438, 0x016e }, { 0xa438, 0xed24 }, { 0xa438, 0x6f50 },
	{ 0xa438, 0x0072 }, { 0xa438, 0x4701 }, { 0xa438, 0x724a },
	{ 0xa438, 0x0272 }, { 0xa438, 0x4d23 }, { 0xa438, 0x7250 },
	{ 0xa438, 0x1074 }, { 0xa438, 0x6910 }, { 0xa438, 0x746c },
	{ 0xa438, 0x1074 }, { 0xa438, 0x6f00 }, { 0xa438, 0x7472 },
	{ 0xa438, 0x158c }, { 0xa438, 0x0b15 }, { 0xa438, 0x8c0e },
	{ 0xa438, 0x158c }, { 0xa438, 0x1105 }, { 0xa438, 0x8c14 },
	{ 0xa438, 0x006f }, { 0xa438, 0x4a03 }, { 0xa438, 0x6f47 },
	{ 0xa438, 0x266f }, { 0xa438, 0x5900 }, { 0xa438, 0x731f },
	{ 0xa438, 0x0273 }, { 0xa438, 0x2203 }, { 0xa438, 0x8c08 },
	{ 0xa438, 0xee84 }, { 0xa438, 0x7100 }, { 0xa438, 0x0286 },
	{ 0xa438, 0xece0 }, { 0xa438, 0x8043 }, { 0xa438, 0xf626 },
	{ 0xa438, 0xe480 }, { 0xa438, 0x43af }, { 0xa438, 0x6611 },
	{ 0xa438, 0xf8e0 }, { 0xa438, 0x8012 }, { 0xa438, 0xac26 },
	{ 0xa438, 0x03af }, { 0xa438, 0x86ff }, { 0xa438, 0x0287 },
	{ 0xa438, 0x0102 }, { 0xa438, 0x8906 }, { 0xa438, 0x0289 },
	{ 0xa438, 0x29fc }, { 0xa438, 0x04f8 }, { 0xa438, 0xf9ef },
	{ 0xa438, 0x59f9 }, { 0xa438, 0xfaee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x00d6 }, { 0xa438, 0x008f }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8fe7 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xe8ae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8fe7 }, { 0xa438, 0xe18f }, { 0xa438, 0xe8ee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x01d6 }, { 0xa438, 0x00c0 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x00d6 }, { 0xa438, 0x0090 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8fe9 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xeaae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8fe9 }, { 0xa438, 0xe18f }, { 0xa438, 0xeaee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x01d6 }, { 0xa438, 0x00c1 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x00d6 }, { 0xa438, 0x0091 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8feb },
	{ 0xa438, 0xe58f }, { 0xa438, 0xecae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8feb }, { 0xa438, 0xe18f }, { 0xa438, 0xecee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x01d6 }, { 0xa438, 0x00c2 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x01d6 }, { 0xa438, 0x008f }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8fed },
	{ 0xa438, 0xe58f }, { 0xa438, 0xeeae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8fed }, { 0xa438, 0xe18f }, { 0xa438, 0xeeee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x02d6 }, { 0xa438, 0x00c0 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x01d6 }, { 0xa438, 0x0090 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8fef },
	{ 0xa438, 0xe58f }, { 0xa438, 0xf0ae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8fef }, { 0xa438, 0xe18f }, { 0xa438, 0xf0ee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x02d6 }, { 0xa438, 0x00c1 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x01d6 }, { 0xa438, 0x0091 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8ff1 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xf2ae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8ff1 }, { 0xa438, 0xe18f }, { 0xa438, 0xf2ee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x02d6 }, { 0xa438, 0x00c2 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x02d6 }, { 0xa438, 0x008f }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8ff3 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xf4ae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8ff3 }, { 0xa438, 0xe18f }, { 0xa438, 0xf4ee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x04d6 }, { 0xa438, 0x00c0 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x02d6 }, { 0xa438, 0x0090 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8ff5 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xf6ae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8ff5 }, { 0xa438, 0xe18f }, { 0xa438, 0xf6ee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x04d6 }, { 0xa438, 0x00c1 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x02d6 }, { 0xa438, 0x0091 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8ff7 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xf8ae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8ff7 }, { 0xa438, 0xe18f }, { 0xa438, 0xf8ee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x04d6 }, { 0xa438, 0x00c2 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x03d6 }, { 0xa438, 0x008f }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8ff9 },
	{ 0xa438, 0xe58f }, { 0xa438, 0xfaae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8ff9 }, { 0xa438, 0xe18f }, { 0xa438, 0xfaee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x08d6 }, { 0xa438, 0x00c0 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x03d6 }, { 0xa438, 0x0090 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8ffb },
	{ 0xa438, 0xe58f }, { 0xa438, 0xfcae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8ffb }, { 0xa438, 0xe18f }, { 0xa438, 0xfcee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x08d6 }, { 0xa438, 0x00c1 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71ee }, { 0xa438, 0x8476 },
	{ 0xa438, 0x03d6 }, { 0xa438, 0x0091 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x53ef }, { 0xa438, 0x643e }, { 0xa438, 0x1200 },
	{ 0xa438, 0xac4f }, { 0xa438, 0x08e4 }, { 0xa438, 0x8ffd },
	{ 0xa438, 0xe58f }, { 0xa438, 0xfeae }, { 0xa438, 0x06e0 },
	{ 0xa438, 0x8ffd }, { 0xa438, 0xe18f }, { 0xa438, 0xfeee },
	{ 0xa438, 0x8476 }, { 0xa438, 0x08d6 }, { 0xa438, 0x00c2 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71fe }, { 0xa438, 0xfdef },
	{ 0xa438, 0x95fd }, { 0xa438, 0xfc04 }, { 0xa438, 0xf8f9 },
	{ 0xa438, 0xfad4 }, { 0xa438, 0x0400 }, { 0xa438, 0xd600 },
	{ 0xa438, 0x0dd3 }, { 0xa438, 0x0fe7 }, { 0xa438, 0x8476 },
	{ 0xa438, 0x0266 }, { 0xa438, 0x71d4 }, { 0xa438, 0x1400 },
	{ 0xa438, 0xd600 }, { 0xa438, 0x0dd3 }, { 0xa438, 0x0fe7 },
	{ 0xa438, 0x8476 }, { 0xa438, 0x0266 }, { 0xa438, 0x71fe },
	{ 0xa438, 0xfdfc }, { 0xa438, 0x04f8 }, { 0xa438, 0xf9fa },
	{ 0xa438, 0xd410 }, { 0xa438, 0x00d6 }, { 0xa438, 0x000d },
	{ 0xa438, 0xd30f }, { 0xa438, 0xe784 }, { 0xa438, 0x7602 },
	{ 0xa438, 0x6671 }, { 0xa438, 0xd400 }, { 0xa438, 0x00d6 },
	{ 0xa438, 0x000d }, { 0xa438, 0xd30f }, { 0xa438, 0xe784 },
	{ 0xa438, 0x7602 }, { 0xa438, 0x6671 }, { 0xa438, 0xfefd },
	{ 0xa438, 0xfc04 }, { 0xa438, 0xe080 }, { 0xa438, 0x4fac },
	{ 0xa438, 0x2317 }, { 0xa438, 0xe080 }, { 0xa438, 0x44ad },
	{ 0xa438, 0x231a }, { 0xa438, 0x0289 }, { 0xa438, 0x75e0 },
	{ 0xa438, 0x8044 }, { 0xa438, 0xac23 }, { 0xa438, 0x11bf },
	{ 0xa438, 0x6ecf }, { 0xa438, 0x0276 }, { 0xa438, 0x74ae },
	{ 0xa438, 0x0902 }, { 0xa438, 0x8adb }, { 0xa438, 0x021f },
	{ 0xa438, 0xe702 }, { 0xa438, 0x1fbb }, { 0xa438, 0xaf1f },
	{ 0xa438, 0x95f8 }, { 0xa438, 0xf9ef }, { 0xa438, 0x59f9 },
	{ 0xa438, 0xfafb }, { 0xa438, 0xe080 }, { 0xa438, 0x12ac },
	{ 0xa438, 0x2303 }, { 0xa438, 0xaf8a }, { 0xa438, 0xd0d4 },
	{ 0xa438, 0x0120 }, { 0xa438, 0xd600 }, { 0xa438, 0x10d2 },
	{ 0xa438, 0x0fe6 }, { 0xa438, 0x8476 }, { 0xa438, 0x0266 },
	{ 0xa438, 0x71ee }, { 0xa438, 0x846f }, { 0xa438, 0x00d4 },
	{ 0xa438, 0x000f }, { 0xa438, 0xbf72 }, { 0xa438, 0x9e02 },
	{ 0xa438, 0x7697 }, { 0xa438, 0x0275 }, { 0xa438, 0xbeef },
	{ 0xa438, 0x47e4 }, { 0xa438, 0x8474 }, { 0xa438, 0xe584 },
	{ 0xa438, 0x75bf }, { 0xa438, 0x729b }, { 0xa438, 0x0276 },
	{ 0xa438, 0xb6e5 }, { 0xa438, 0x846f }, { 0xa438, 0xef31 },
	{ 0xa438, 0xbf6e }, { 0xa438, 0x0602 }, { 0xa438, 0x76b6 },
	{ 0xa438, 0xef64 }, { 0xa438, 0xbf6e }, { 0xa438, 0x0902 },
	{ 0xa438, 0x76b6 }, { 0xa438, 0x1e64 }, { 0xa438, 0xbf6e },
	{ 0xa438, 0x0f02 }, { 0xa438, 0x76b6 }, { 0xa438, 0x1e64 },
	{ 0xa438, 0xac40 }, { 0xa438, 0x05a3 }, { 0xa438, 0x0f0c },
	{ 0xa438, 0xae26 }, { 0xa438, 0xa303 }, { 0xa438, 0x02ae },
	{ 0xa438, 0x21a3 }, { 0xa438, 0x0c02 }, { 0xa438, 0xae1c },
	{ 0xa438, 0xe084 }, { 0xa438, 0x74e1 }, { 0xa438, 0x8475 },
	{ 0xa438, 0xef64 }, { 0xa438, 0xd000 }, { 0xa438, 0xd196 },
	{ 0xa438, 0xef74 }, { 0xa438, 0x0275 }, { 0xa438, 0xd9ad },
	{ 0xa438, 0x50b7 }, { 0xa438, 0xe083 }, { 0xa438, 0xecf7 },
	{ 0xa438, 0x23e4 }, { 0xa438, 0x83ec }, { 0xa438, 0xbf72 },
	{ 0xa438, 0x9e02 }, { 0xa438, 0x766b }, { 0xa438, 0x0287 },
	{ 0xa438, 0x0102 }, { 0xa438, 0x8906 }, { 0xa438, 0xee83 },
	{ 0xa438, 0xe800 }, { 0xa438, 0xbf72 }, { 0xa438, 0x6b02 },
	{ 0xa438, 0x766b }, { 0xa438, 0xbf72 }, { 0xa438, 0x6e02 },
	{ 0xa438, 0x766b }, { 0xa438, 0xbf72 }, { 0xa438, 0x7102 },
	{ 0xa438, 0x766b }, { 0xa438, 0xbf72 }, { 0xa438, 0x7402 },
	{ 0xa438, 0x766b }, { 0xa438, 0xbf72 }, { 0xa438, 0x7702 },
	{ 0xa438, 0x766b }, { 0xa438, 0xbf72 }, { 0xa438, 0x7a02 },
	{ 0xa438, 0x766b }, { 0xa438, 0xd400 }, { 0xa438, 0x0fbf },
	{ 0xa438, 0x7295 }, { 0xa438, 0x0276 }, { 0xa438, 0x97d7 },
	{ 0xa438, 0x0400 }, { 0xa438, 0xbf6e }, { 0xa438, 0x0602 },
	{ 0xa438, 0x76b6 }, { 0xa438, 0xef64 }, { 0xa438, 0xbf6e },
	{ 0xa438, 0x0902 }, { 0xa438, 0x76b6 }, { 0xa438, 0x1e64 },
	{ 0xa438, 0xbf6e }, { 0xa438, 0x0f02 }, { 0xa438, 0x76b6 },
	{ 0xa438, 0x1e64 }, { 0xa438, 0xac40 }, { 0xa438, 0x0fbf },
	{ 0xa438, 0x7298 }, { 0xa438, 0x0276 }, { 0xa438, 0xb6e5 },
	{ 0xa438, 0x83e8 }, { 0xa438, 0xa10f }, { 0xa438, 0x28af },
	{ 0xa438, 0x8a95 }, { 0xa438, 0xbf8b }, { 0xa438, 0xf302 },
	{ 0xa438, 0x76b6 }, { 0xa438, 0xac28 }, { 0xa438, 0x02ae },
	{ 0xa438, 0x0bbf }, { 0xa438, 0x8bf9 }, { 0xa438, 0x0276 },
	{ 0xa438, 0xb6e5 }, { 0xa438, 0x83e8 }, { 0xa438, 0xae09 },
	{ 0xa438, 0xbf8b }, { 0xa438, 0xf602 }, { 0xa438, 0x76b6 },
	{ 0xa438, 0xe583 }, { 0xa438, 0xe8a1 }, { 0xa438, 0x0303 },
	{ 0xa438, 0xaf8a }, { 0xa438, 0x95b7 }, { 0xa438, 0xafe2 },
	{ 0xa438, 0x83ec }, { 0xa438, 0xf735 }, { 0xa438, 0xe683 },
	{ 0xa438, 0xecbf }, { 0xa438, 0x7295 }, { 0xa438, 0x0276 },
	{ 0xa438, 0x6bbf }, { 0xa438, 0x726b }, { 0xa438, 0x0276 },
	{ 0xa438, 0x74bf }, { 0xa438, 0x726e }, { 0xa438, 0x0276 },
	{ 0xa438, 0x74bf }, { 0xa438, 0x7271 }, { 0xa438, 0x0276 },
	{ 0xa438, 0x74bf }, { 0xa438, 0x7274 }, { 0xa438, 0x0276 },
	{ 0xa438, 0x74bf }, { 0xa438, 0x7277 }, { 0xa438, 0x0276 },
	{ 0xa438, 0x74bf }, { 0xa438, 0x727a }, { 0xa438, 0x0276 },
	{ 0xa438, 0x7402 }, { 0xa438, 0x8929 }, { 0xa438, 0xd401 },
	{ 0xa438, 0x28d6 }, { 0xa438, 0x0010 }, { 0xa438, 0xd20f },
	{ 0xa438, 0xe684 }, { 0xa438, 0x7602 }, { 0xa438, 0x6671 },
	{ 0xa438, 0x021f }, { 0xa438, 0xbbff }, { 0xa438, 0xfefd },
	{ 0xa438, 0xef95 }, { 0xa438, 0xfdfc }, { 0xa438, 0x04f8 },
	{ 0xa438, 0xf9ef }, { 0xa438, 0x59f9 }, { 0xa438, 0xe080 },
	{ 0xa438, 0x12ad }, { 0xa438, 0x230c }, { 0xa438, 0xbf72 },
	{ 0xa438, 0x9e02 }, { 0xa438, 0x766b }, { 0xa438, 0xbf72 },
	{ 0xa438, 0x9502 }, { 0xa438, 0x766b }, { 0xa438, 0xfdef },
	{ 0xa438, 0x95fd }, { 0xa438, 0xfc04 }, { 0xa438, 0xbf6e },
	{ 0xa438, 0x0602 }, { 0xa438, 0x76b6 }, { 0xa438, 0xef64 },
	{ 0xa438, 0xbf6e }, { 0xa438, 0x0902 }, { 0xa438, 0x76b6 },
	{ 0xa438, 0x1e64 }, { 0xa438, 0xbf6e }, { 0xa438, 0x0f02 },
	{ 0xa438, 0x76b6 }, { 0xa438, 0x1e64 }, { 0xa438, 0xac40 },
	{ 0xa438, 0x0ebf }, { 0xa438, 0x7298 }, { 0xa438, 0x0276 },
	{ 0xa438, 0xb6e5 }, { 0xa438, 0x8478 }, { 0xa438, 0xa10f },
	{ 0xa438, 0x26ae }, { 0xa438, 0x47bf }, { 0xa438, 0x8bf3 },
	{ 0xa438, 0x0276 }, { 0xa438, 0xb6ac }, { 0xa438, 0x2802 },
	{ 0xa438, 0xae0b }, { 0xa438, 0xbf8b }, { 0xa438, 0xf902 },
	{ 0xa438, 0x76b6 }, { 0xa438, 0xe584 }, { 0xa438, 0x78ae },
	{ 0xa438, 0x09bf }, { 0xa438, 0x8bf6 }, { 0xa438, 0x0276 },
	{ 0xa438, 0xb6e5 }, { 0xa438, 0x8478 }, { 0xa438, 0xa103 },
	{ 0xa438, 0x02ae }, { 0xa438, 0x23e0 }, { 0xa438, 0x8474 },
	{ 0xa438, 0xe184 }, { 0xa438, 0x75ef }, { 0xa438, 0x64e0 },
	{ 0xa438, 0x83fc }, { 0xa438, 0xe183 }, { 0xa438, 0xfdef },
	{ 0xa438, 0x7402 }, { 0xa438, 0x75d9 }, { 0xa438, 0xad50 },
	{ 0xa438, 0x0ae0 }, { 0xa438, 0x83ec }, { 0xa438, 0xf721 },
	{ 0xa438, 0xe483 }, { 0xa438, 0xecae }, { 0xa438, 0x03af },
	{ 0xa438, 0x68e4 }, { 0xa438, 0xbf72 }, { 0xa438, 0x9502 },
	{ 0xa438, 0x766b }, { 0xa438, 0xe083 }, { 0xa438, 0xebad },
	{ 0xa438, 0x2170 }, { 0xa438, 0xbf73 }, { 0xa438, 0x7f02 },
	{ 0xa438, 0x766b }, { 0xa438, 0xd700 }, { 0xa438, 0x64bf },
	{ 0xa438, 0x73c4 }, { 0xa438, 0x0276 }, { 0xa438, 0xb6a4 },
	{ 0xa438, 0x0000 }, { 0xa438, 0x02ae }, { 0xa438, 0x0d87 },
	{ 0xa438, 0xa700 }, { 0xa438, 0x00ef }, { 0xa438, 0xe183 },
	{ 0xa438, 0xecf7 }, { 0xa438, 0x2ae5 }, { 0xa438, 0x83ec },
	{ 0xa438, 0xbf73 }, { 0xa438, 0xbe02 }, { 0xa438, 0x766b },
	{ 0xa438, 0xbf73 }, { 0xa438, 0xb802 }, { 0xa438, 0x766b },
	{ 0xa438, 0xbf73 }, { 0xa438, 0xc102 }, { 0xa438, 0x766b },
	{ 0xa438, 0xbf73 }, { 0xa438, 0xbb02 }, { 0xa438, 0x766b },
	{ 0xa438, 0xe084 }, { 0xa438, 0x9ee1 }, { 0xa438, 0x849f },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x7d02 }, { 0xa438, 0x7697 },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x8002 }, { 0xa438, 0x7697 },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x8302 }, { 0xa438, 0x7697 },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x8602 }, { 0xa438, 0x7697 },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x8902 }, { 0xa438, 0x7674 },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x8c02 }, { 0xa438, 0x7674 },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x8f02 }, { 0xa438, 0x7674 },
	{ 0xa438, 0xbf72 }, { 0xa438, 0x9202 }, { 0xa438, 0x7674 },
	{ 0xa438, 0xee84 }, { 0xa438, 0x7700 }, { 0xa438, 0xe080 },
	{ 0xa438, 0x44f6 }, { 0xa438, 0x21e4 }, { 0xa438, 0x8044 },
	{ 0xa438, 0xaf68 }, { 0xa438, 0xe411 }, { 0xa438, 0xd1a4 },
	{ 0xa438, 0x10bc }, { 0xa438, 0x7432 }, { 0xa438, 0xbc74 },
	{ 0xa438, 0xbbbf }, { 0xa438, 0x14cc }, { 0xa438, 0xbfaa },
	{ 0xa438, 0x00bf }, { 0xa438, 0x9055 }, { 0xa438, 0xbf06 },
	{ 0xa438, 0x10bf }, { 0xa438, 0xb876 }, { 0xa438, 0xbe02 },
	{ 0xa438, 0x54be }, { 0xa438, 0x0232 }, { 0xa438, 0xbe02 },
	{ 0xa438, 0x10be }, { 0xa438, 0x0200 }, { 0xa436, 0x8fe7 },
	{ 0xa438, 0x1200 }, { 0xa436, 0x8fe9 }, { 0xa438, 0x1200 },
	{ 0xa436, 0x8feb }, { 0xa438, 0x1200 }, { 0xa436, 0x8fed },
	{ 0xa438, 0x1200 }, { 0xa436, 0x8fef }, { 0xa438, 0x1200 },
	{ 0xa436, 0x8ff1 }, { 0xa438, 0x1200 }, { 0xa436, 0x8ff3 },
	{ 0xa438, 0x1200 }, { 0xa436, 0x8ff5 }, { 0xa438, 0x1200 },
	{ 0xa436, 0x8ff7 }, { 0xa438, 0x1200 }, { 0xa436, 0x8ff9 },
	{ 0xa438, 0x1200 }, { 0xa436, 0x8ffb }, { 0xa438, 0x1200 },
	{ 0xa436, 0x8ffd }, { 0xa438, 0x1200 }, { 0xa436, 0xb818 },
	{ 0xa438, 0x6602 }, { 0xa436, 0xb81a }, { 0xa438, 0x1f75 },
	{ 0xa436, 0xb81c }, { 0xa438, 0x67eb }, { 0xa436, 0xb81e },
	{ 0xa438, 0xffff }, { 0xa436, 0xb850 }, { 0xa438, 0xffff },
	{ 0xa436, 0xb852 }, { 0xa438, 0xffff }, { 0xa436, 0xb878 },
	{ 0xa438, 0xffff }, { 0xa436, 0xb884 }, { 0xa438, 0xffff },
	{ 0xa436, 0xb832 }, { 0xa438, 0x0007 }, { 0xb82e, 0x0000 },
	{ 0xa436, 0x8023 }, { 0xa438, 0x0000 }, { 0xb820, 0x0000 },
	{ 0xb892, 0x0000 }, { 0xb88e, 0xc07c }, { 0xb890, 0x0203 },
	{ 0xb890, 0x0304 }, { 0xb890, 0x0405 }, { 0xb890, 0x0607 },
	{ 0xb890, 0x0809 }, { 0xb890, 0x0b0d }, { 0xb890, 0x0f11 },
	{ 0xb890, 0x1418 }, { 0xb890, 0x1b20 }, { 0xb890, 0x252b },
	{ 0xb890, 0x343e }, { 0xb890, 0x4854 }, { 0xb890, 0x6203 },
	{ 0xb890, 0x0304 }, { 0xb890, 0x0506 }, { 0xb890, 0x080a },
	{ 0xb890, 0x0c0e }, { 0xb890, 0x1216 }, { 0xb890, 0x1b22 },
	{ 0xb890, 0x2a34 }, { 0xb890, 0x404f }, { 0xb890, 0x6171 },
	{ 0xb890, 0x7884 }, { 0xb890, 0x9097 }, { 0xb890, 0x0203 },
	{ 0xb890, 0x0406 }, { 0xb890, 0x080b }, { 0xb890, 0x0e13 },
	{ 0xb890, 0x1820 }, { 0xb890, 0x2a39 }, { 0xb890, 0x4856 },
	{ 0xb890, 0xe060 }, { 0xb890, 0xe050 }, { 0xb890, 0xd080 },
	{ 0xb890, 0x8070 }, { 0xb890, 0x70a0 }, { 0xb890, 0x1000 },
	{ 0xb890, 0x60d0 }, { 0xb890, 0xb010 }, { 0xb890, 0xe0b0 },
	{ 0xb890, 0x80c0 }, { 0xb890, 0xe000 }, { 0xb890, 0x2020 },
	{ 0xb890, 0x1020 }, { 0xb890, 0xe090 }, { 0xb890, 0x80c0 },
	{ 0xb890, 0x3020 }, { 0xb890, 0x00e0 }, { 0xb890, 0x40a0 },
	{ 0xb890, 0xe020 }, { 0xb890, 0x5060 }, { 0xb890, 0xe0d0 },
	{ 0xb890, 0xa000 }, { 0xb890, 0x3030 }, { 0xb890, 0x4070 },
	{ 0xb890, 0xe0e0 }, { 0xb890, 0xd080 }, { 0xb890, 0xa010 },
	{ 0xb890, 0xe040 }, { 0xb890, 0x80b0 }, { 0xb890, 0x50b0 },
	{ 0xb890, 0x2090 }, { 0xb820, 0x0000 }
};

#define MAC_R25_MCU		\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8093 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8097 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x809d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a1 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80aa },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x607b },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0x42da },	\
	{ 0xa438, 0xf01e },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x615b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf024 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf02c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac11 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4779 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac22 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa420 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4559 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf023 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac44 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa440 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4339 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf012 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac88 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa480 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0xc48f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x141b },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x121a },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0898 },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a0e },	\
	{ 0xa438, 0xd064 },	\
	{ 0xa438, 0xd18a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0b7e },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0x8804 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x053b },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0648 },	\
	{ 0xa438, 0xc520 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x252d },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x4006 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0308 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0x0307 },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x1645 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x0647 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x053a },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x0b7c },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x0a0c },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x0896 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0xff00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8015 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x02d7 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ed },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0xc100 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x008f },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x008d },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x00eb },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8018 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8024 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8055 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8072 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80dc },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa70c },	\
	{ 0xa438, 0x9402 },	\
	{ 0xa438, 0x890c },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0xa380 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xcb91 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd139 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xb404 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4085 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07f0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb6 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd158 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x03d4 },	\
	{ 0xa438, 0x94bc },	\
	{ 0xa438, 0x870c },	\
	{ 0xa438, 0x8380 },	\
	{ 0xa438, 0xd10d },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0xd130 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xbb80 },	\
	{ 0xa438, 0xd1c4 },	\
	{ 0xa438, 0xd074 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x604b },	\
	{ 0xa438, 0xa90c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0xcb92 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd116 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd119 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x60a0 },	\
	{ 0xa438, 0x6241 },	\
	{ 0xa438, 0x63e2 },	\
	{ 0xa438, 0x6583 },	\
	{ 0xa438, 0xf054 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d10 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf02f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d50 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf02a },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d20 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf021 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d60 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d30 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf013 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d70 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d40 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d80 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5ff4 },	\
	{ 0xa438, 0xa008 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4046 },	\
	{ 0xa438, 0xa002 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07fb },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x7f6f },	\
	{ 0xa438, 0x7f4e },	\
	{ 0xa438, 0x7f2d },	\
	{ 0xa438, 0x7f0c },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb5 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x3ad4 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd1f5 },	\
	{ 0xa438, 0xd049 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01ec },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x01ea },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x06a9 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x078a },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x03d2 },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x067f },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0665 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00fc },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8530 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x3caf },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x45af },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xee82 },	\
	{ 0xa438, 0xf900 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa438, 0xaf03 },	\
	{ 0xa438, 0xb7f8 },	\
	{ 0xa438, 0xe0a6 },	\
	{ 0xa438, 0x00e1 },	\
	{ 0xa438, 0xa601 },	\
	{ 0xa438, 0xef01 },	\
	{ 0xa438, 0x58f0 },	\
	{ 0xa438, 0xa080 },	\
	{ 0xa438, 0x37a1 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0xae16 },	\
	{ 0xa438, 0xa185 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa438, 0x8702 },	\
	{ 0xa438, 0xae0c },	\
	{ 0xa438, 0xa188 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x07a1 },	\
	{ 0xa438, 0x8902 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0xae1c },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x03b3 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0x0001 }

#define MAC_R25B_MCU		\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x3701 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0001 },	\
	{ 0xb820, 0x0090 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8045 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8067 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x806d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8071 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80b1 },	\
	{ 0xa438, 0xd093 },	\
	{ 0xa438, 0xd1c4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x5fbc },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xc9f1 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0fc9 },	\
	{ 0xa438, 0xbb50 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8c0f },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1519 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x5fae },	\
	{ 0xa438, 0x9b50 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x7fae },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x40a7 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4071 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1557 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x2f70 },	\
	{ 0xa438, 0x803b },	\
	{ 0xa438, 0x2f73 },	\
	{ 0xa438, 0x156a },	\
	{ 0xa438, 0x5e70 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x155d },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xffed },	\
	{ 0xa438, 0xd709 },	\
	{ 0xa438, 0x4054 },	\
	{ 0xa438, 0xa788 },	\
	{ 0xa438, 0xd70b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x172a },	\
	{ 0xa438, 0xc0c1 },	\
	{ 0xa438, 0xc0c0 },	\
	{ 0xa438, 0xd05a },	\
	{ 0xa438, 0xd1ba },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x2529 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0xd0a7 },	\
	{ 0xa438, 0xd1b9 },	\
	{ 0xa438, 0xa208 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x080e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x408b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a65 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a6b },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0915 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0909 },	\
	{ 0xa438, 0x228f },	\
	{ 0xa438, 0x804e },	\
	{ 0xa438, 0x9801 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x5d61 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0x2005 },	\
	{ 0xa438, 0x091a },	\
	{ 0xa438, 0x3bd9 },	\
	{ 0xa438, 0x0919 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0916 },	\
	{ 0xa438, 0xd090 },	\
	{ 0xa438, 0xd1c9 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1064 },	\
	{ 0xa438, 0xd096 },	\
	{ 0xa438, 0xd1a9 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0xa104 },	\
	{ 0xa438, 0x0c07 },	\
	{ 0xa438, 0x0902 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xbc10 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0x8201 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xc484 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0xcc02 },	\
	{ 0xa438, 0xcd0d },	\
	{ 0xa438, 0xaf01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x4371 },	\
	{ 0xa438, 0xbd08 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x5fb3 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0xd0f5 },	\
	{ 0xa438, 0xd1c6 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0e50 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd0f5 },	\
	{ 0xa438, 0xd1c6 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0ea0 },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd07b },	\
	{ 0xa438, 0xd1c5 },	\
	{ 0xa438, 0x8ef0 },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0x9d08 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x7fb3 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x5fad },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14c5 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x3181 },	\
	{ 0xa438, 0x80af },	\
	{ 0xa438, 0x60ad },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x5fba },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0cc7 },	\
	{ 0xa438, 0xa802 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0xa801 },	\
	{ 0xa438, 0xc004 },	\
	{ 0xa438, 0xd710 },	\
	{ 0xa438, 0x4000 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1e79 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0x1e78 },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x0c93 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x1062 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x0915 },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x020a },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x1726 },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x1542 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x0fc7 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0xff00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6090 },	\
	{ 0xa438, 0x60d1 },	\
	{ 0xa438, 0xc95c },	\
	{ 0xa438, 0xf007 },	\
	{ 0xa438, 0x60b1 },	\
	{ 0xa438, 0xc95a },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xc956 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc94e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00cd },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6090 },	\
	{ 0xa438, 0x60d1 },	\
	{ 0xa438, 0xc95c },	\
	{ 0xa438, 0xf007 },	\
	{ 0xa438, 0x60b1 },	\
	{ 0xa438, 0xc95a },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xc956 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc94e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0132 },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x012f },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x00cc },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8020 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8035 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0xd107 },	\
	{ 0xa438, 0xd042 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x09df },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6065 },	\
	{ 0xa438, 0xd125 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd12b },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x077f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c50 },	\
	{ 0xa438, 0xd104 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0aa8 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a2e },	\
	{ 0xa438, 0xcb9b },	\
	{ 0xa438, 0xd110 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b7b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x09df },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x081b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x09df },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x7fb8 },	\
	{ 0xa438, 0xa718 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x074e },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x074d },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0818 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0a2c },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x077e },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x000f },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8625 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0x3daf },	\
	{ 0xa438, 0x8689 },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x69af },	\
	{ 0xa438, 0x8887 },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x9caf },	\
	{ 0xa438, 0x88be },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0xbeaf },	\
	{ 0xa438, 0x88be },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x49d7 },	\
	{ 0xa438, 0x0040 },	\
	{ 0xa438, 0x0277 },	\
	{ 0xa438, 0x7daf },	\
	{ 0xa438, 0x2727 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7205 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7208 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x71f3 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x71f6 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7229 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x722c },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7217 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x721a },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x721d },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7211 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7220 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7214 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x722f },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7223 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7232 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7226 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfae0 },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x3802 },	\
	{ 0xa438, 0xad27 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x8830 },	\
	{ 0xa438, 0x1f66 },	\
	{ 0xa438, 0xef65 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x1f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00da },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x2f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00db },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4202 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1b03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4502 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1a03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4802 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x3f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00da },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x4f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00db },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4b02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1b03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4e02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1a03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5102 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef56 },	\
	{ 0xa438, 0xd020 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5702 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5a02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0xa0ef },	\
	{ 0xa438, 0x0348 },	\
	{ 0xa438, 0x0a28 },	\
	{ 0xa438, 0x05ef },	\
	{ 0xa438, 0x201b },	\
	{ 0xa438, 0x01ad },	\
	{ 0xa438, 0x2735 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x88e1 },	\
	{ 0xa438, 0x8589 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5d02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x8ee1 },	\
	{ 0xa438, 0x858f },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x94e1 },	\
	{ 0xa438, 0x8595 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6302 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x9ae1 },	\
	{ 0xa438, 0x859b },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6602 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x3cbf },	\
	{ 0xa438, 0x883f },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x9cad },	\
	{ 0xa438, 0x2835 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xf8e1 },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5d02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfae1 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfce1 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6302 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfee1 },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6602 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x3ce1 },	\
	{ 0xa438, 0x85a1 },	\
	{ 0xa438, 0x1b21 },	\
	{ 0xa438, 0xad37 },	\
	{ 0xa438, 0x341f },	\
	{ 0xa438, 0x44e0 },	\
	{ 0xa438, 0x858a },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x8bbf },	\
	{ 0xa438, 0x885d },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8590 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x91bf },	\
	{ 0xa438, 0x8860 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8596 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x97bf },	\
	{ 0xa438, 0x8863 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x859c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x9dbf },	\
	{ 0xa438, 0x8866 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7dae },	\
	{ 0xa438, 0x401f },	\
	{ 0xa438, 0x44e0 },	\
	{ 0xa438, 0x858c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x8dbf },	\
	{ 0xa438, 0x885d },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8592 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x93bf },	\
	{ 0xa438, 0x8860 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8598 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x99bf },	\
	{ 0xa438, 0x8863 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x859e },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x9fbf },	\
	{ 0xa438, 0x8866 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7dae },	\
	{ 0xa438, 0x0ce1 },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x3904 },	\
	{ 0xa438, 0xac2f },	\
	{ 0xa438, 0x04ee },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x39d9 },	\
	{ 0xa438, 0x22ac },	\
	{ 0xa438, 0xeaf0 },	\
	{ 0xa438, 0xacf6 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfaf0 },	\
	{ 0xa438, 0xacf8 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfcf0 },	\
	{ 0xa438, 0xad00 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfef0 },	\
	{ 0xa438, 0xacf0 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xf4f0 },	\
	{ 0xa438, 0xacf2 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xb0f0 },	\
	{ 0xa438, 0xacae },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xacf0 },	\
	{ 0xa438, 0xacaa },	\
	{ 0xa438, 0xa100 },	\
	{ 0xa438, 0x0ce1 },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf26 },	\
	{ 0xa438, 0xe9e1 },	\
	{ 0xa438, 0x8ff6 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf26 },	\
	{ 0xa438, 0xf520 },	\
	{ 0xa438, 0xac86 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x3f02 },	\
	{ 0xa438, 0x6e9c },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x3324 },	\
	{ 0xa438, 0xad38 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x32e6 },	\
	{ 0xa438, 0xaf32 },	\
	{ 0xa438, 0xfbee },	\
	{ 0xa438, 0x826a },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x88a6 },	\
	{ 0xa438, 0xaf04 },	\
	{ 0xa438, 0x78f8 },	\
	{ 0xa438, 0xfaef },	\
	{ 0xa438, 0x69e0 },	\
	{ 0xa438, 0x8015 },	\
	{ 0xa438, 0xad20 },	\
	{ 0xa438, 0x06bf },	\
	{ 0xa438, 0x88bb },	\
	{ 0xa438, 0x0275 },	\
	{ 0xa438, 0xb1ef },	\
	{ 0xa438, 0x96fe },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0x00b8 },	\
	{ 0xa438, 0x7a00 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ff6 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x0705 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x19cc },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x28e3 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x1047 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x0a45 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x271e },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0x3846 },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0x26e6 },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0x32e3 },	\
	{ 0xa436, 0xb886 },	\
	{ 0xa438, 0x0474 },	\
	{ 0xa436, 0xb888 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb88a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb88c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb838 },	\
	{ 0xa438, 0x001f },	\
	{ 0xb820, 0x0010 },	\
	{ 0xa436, 0x846e },	\
	{ 0xa438, 0xaf84 },	\
	{ 0xa438, 0x86af },	\
	{ 0xa438, 0x8690 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0xa4af },	\
	{ 0xa438, 0x8934 },	\
	{ 0xa438, 0xaf89 },	\
	{ 0xa438, 0x60af },	\
	{ 0xa438, 0x897e },	\
	{ 0xa438, 0xaf89 },	\
	{ 0xa438, 0xa9af },	\
	{ 0xa438, 0x89a9 },	\
	{ 0xa438, 0xee82 },	\
	{ 0xa438, 0x5f00 },	\
	{ 0xa438, 0x0284 },	\
	{ 0xa438, 0x90af },	\
	{ 0xa438, 0x0441 },	\
	{ 0xa438, 0xf8e0 },	\
	{ 0xa438, 0x8ff3 },	\
	{ 0xa438, 0xa000 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x84a4 },	\
	{ 0xa438, 0xae06 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x0302 },	\
	{ 0xa438, 0x84c8 },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xef59 },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x15ad },	\
	{ 0xa438, 0x2702 },	\
	{ 0xa438, 0xae03 },	\
	{ 0xa438, 0xaf84 },	\
	{ 0xa438, 0xc3bf },	\
	{ 0xa438, 0x53ca },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ad },	\
	{ 0xa438, 0x2807 },	\
	{ 0xa438, 0x0285 },	\
	{ 0xa438, 0x2cee },	\
	{ 0xa438, 0x8ff3 },	\
	{ 0xa438, 0x01ef },	\
	{ 0xa438, 0x95fd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfaef },	\
	{ 0xa438, 0x69bf },	\
	{ 0xa438, 0x53ca },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ac },	\
	{ 0xa438, 0x2822 },	\
	{ 0xa438, 0xd480 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x8684 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x8687 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868a },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868d },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9ee },	\
	{ 0xa438, 0x8ff3 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x8526 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xf4e1 },	\
	{ 0xa438, 0x8ff5 },	\
	{ 0xa438, 0xe28f },	\
	{ 0xa438, 0xf6e3 },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0x1b45 },	\
	{ 0xa438, 0xac27 },	\
	{ 0xa438, 0x0eee },	\
	{ 0xa438, 0x8ff4 },	\
	{ 0xa438, 0x00ee },	\
	{ 0xa438, 0x8ff5 },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x852c },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x26e0 },	\
	{ 0xa438, 0x8ff4 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf52c },	\
	{ 0xa438, 0x0001 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xf4e5 },	\
	{ 0xa438, 0x8ff5 },	\
	{ 0xa438, 0xef96 },	\
	{ 0xa438, 0xfefd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xef59 },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa18b },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x85da },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xf8e5 },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7502 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfae5 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7802 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfce5 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7b02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfee5 },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x6c02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa102 },	\
	{ 0xa438, 0x13ee },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa438, 0x80ee },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0x00ee },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa438, 0x80ee },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x8599 },	\
	{ 0xa438, 0xa101 },	\
	{ 0xa438, 0x0cbf },	\
	{ 0xa438, 0x534c },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x0303 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x77bf },	\
	{ 0xa438, 0x5322 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x8b02 },	\
	{ 0xa438, 0xae03 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0x64e0 },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf9bf },	\
	{ 0xa438, 0x8684 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9e0 },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x8687 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9e0 },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xfdbf },	\
	{ 0xa438, 0x868a },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9e0 },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xffbf },	\
	{ 0xa438, 0x868d },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9af },	\
	{ 0xa438, 0x867f },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa144 },	\
	{ 0xa438, 0x3cbf },	\
	{ 0xa438, 0x547b },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xf9bf },	\
	{ 0xa438, 0x547e },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x5481 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xfdbf },	\
	{ 0xa438, 0x5484 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xffbf },	\
	{ 0xa438, 0x5322 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x4448 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0xa7bf },	\
	{ 0xa438, 0x5322 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x313c },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x7b02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xf8e5 },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x7e02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfae5 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x8102 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfce5 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfee5 },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa131 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x85a7 },	\
	{ 0xa438, 0xd480 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x8684 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x8687 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868a },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868d },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9ef },	\
	{ 0xa438, 0x95fd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf0d1 },	\
	{ 0xa438, 0x2af0 },	\
	{ 0xa438, 0xd12c },	\
	{ 0xa438, 0xf0d1 },	\
	{ 0xa438, 0x44f0 },	\
	{ 0xa438, 0xd146 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xa102 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xa102 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xaf06 },	\
	{ 0xa438, 0xa570 },	\
	{ 0xa438, 0xce42 },	\
	{ 0xa438, 0xee83 },	\
	{ 0xa438, 0xc800 },	\
	{ 0xa438, 0x0286 },	\
	{ 0xa438, 0xba02 },	\
	{ 0xa438, 0x8728 },	\
	{ 0xa438, 0x0287 },	\
	{ 0xa438, 0xbe02 },	\
	{ 0xa438, 0x87f9 },	\
	{ 0xa438, 0x0288 },	\
	{ 0xa438, 0xc3af },	\
	{ 0xa438, 0x4771 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfafb },	\
	{ 0xa438, 0xef69 },	\
	{ 0xa438, 0xfae0 },	\
	{ 0xa438, 0x8015 },	\
	{ 0xa438, 0xad25 },	\
	{ 0xa438, 0x45d2 },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x8714 },	\
	{ 0xa438, 0xac4f },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x0bef },	\
	{ 0xa438, 0x46f6 },	\
	{ 0xa438, 0x273c },	\
	{ 0xa438, 0x0400 },	\
	{ 0xa438, 0xab26 },	\
	{ 0xa438, 0xae30 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xe9e1 },	\
	{ 0xa438, 0x8fea },	\
	{ 0xa438, 0x1b46 },	\
	{ 0xa438, 0xab26 },	\
	{ 0xa438, 0xef32 },	\
	{ 0xa438, 0x0c31 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xe91a },	\
	{ 0xa438, 0x93d8 },	\
	{ 0xa438, 0x19d9 },	\
	{ 0xa438, 0x1b46 },	\
	{ 0xa438, 0xab0a },	\
	{ 0xa438, 0x19d8 },	\
	{ 0xa438, 0x19d9 },	\
	{ 0xa438, 0x1b46 },	\
	{ 0xa438, 0xaa02 },	\
	{ 0xa438, 0xae0c },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x1202 },	\
	{ 0xa438, 0x58b1 },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x1202 },	\
	{ 0xa438, 0x58a8 },	\
	{ 0xa438, 0xfeef },	\
	{ 0xa438, 0x96ff },	\
	{ 0xa438, 0xfefd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8fb },	\
	{ 0xa438, 0xef79 },	\
	{ 0xa438, 0xa200 },	\
	{ 0xa438, 0x08bf },	\
	{ 0xa438, 0x892e },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ef },	\
	{ 0xa438, 0x64ef },	\
	{ 0xa438, 0x97ff },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfafb },	\
	{ 0xa438, 0xef69 },	\
	{ 0xa438, 0xfae0 },	\
	{ 0xa438, 0x8015 },	\
	{ 0xa438, 0xad25 },	\
	{ 0xa438, 0x50d2 },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x878d },	\
	{ 0xa438, 0xac4f },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x0bef },	\
	{ 0xa438, 0x46f6 },	\
	{ 0xa438, 0x273c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0xab31 },	\
	{ 0xa438, 0xae29 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xede1 },	\
	{ 0xa438, 0x8fee },	\
	{ 0xa438, 0x1b46 },	\
	{ 0xa438, 0xab1f },	\
	{ 0xa438, 0xa200 },	\
	{ 0xa438, 0x04ef },	\
	{ 0xa438, 0x32ae },	\
	{ 0xa438, 0x02d3 },	\
	{ 0xa438, 0x010c },	\
	{ 0xa438, 0x31bf },	\
	{ 0xa438, 0x8fed },	\
	{ 0xa438, 0x1a93 },	\
	{ 0xa438, 0xd819 },	\
	{ 0xa438, 0xd91b },	\
	{ 0xa438, 0x46ab },	\
	{ 0xa438, 0x0e19 },	\
	{ 0xa438, 0xd819 },	\
	{ 0xa438, 0xd91b },	\
	{ 0xa438, 0x46aa },	\
	{ 0xa438, 0x0612 },	\
	{ 0xa438, 0xa205 },	\
	{ 0xa438, 0xc0ae },	\
	{ 0xa438, 0x0cbf },	\
	{ 0xa438, 0x5712 },	\
	{ 0xa438, 0x0258 },	\
	{ 0xa438, 0xb1bf },	\
	{ 0xa438, 0x5712 },	\
	{ 0xa438, 0x0258 },	\
	{ 0xa438, 0xa8fe },	\
	{ 0xa438, 0xef96 },	\
	{ 0xa438, 0xfffe },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x04f8 },	\
	{ 0xa438, 0xfbef },	\
	{ 0xa438, 0x79a2 },	\
	{ 0xa438, 0x0005 },	\
	{ 0xa438, 0xbf89 },	\
	{ 0xa438, 0x1fae },	\
	{ 0xa438, 0x1ba2 },	\
	{ 0xa438, 0x0105 },	\
	{ 0xa438, 0xbf89 },	\
	{ 0xa438, 0x22ae },	\
	{ 0xa438, 0x13a2 },	\
	{ 0xa438, 0x0205 },	\
	{ 0xa438, 0xbf89 },	\
	{ 0xa438, 0x25ae },	\
	{ 0xa438, 0x0ba2 },	\
	{ 0xa438, 0x0305 },	\
	{ 0xa438, 0xbf89 },	\
	{ 0xa438, 0x28ae },	\
	{ 0xa438, 0x03bf },	\
	{ 0xa438, 0x892b },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ef },	\
	{ 0xa438, 0x64ef },	\
	{ 0xa438, 0x97ff },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfaef },	\
	{ 0xa438, 0x69fa },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x15ad },	\
	{ 0xa438, 0x2628 },	\
	{ 0xa438, 0xe081 },	\
	{ 0xa438, 0xabe1 },	\
	{ 0xa438, 0x81ac },	\
	{ 0xa438, 0xef64 },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x1802 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0x1b46 },	\
	{ 0xa438, 0xaa0a },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x1b02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0x1b46 },	\
	{ 0xa438, 0xab0c },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x58b1 },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x58a8 },	\
	{ 0xa438, 0xfeef },	\
	{ 0xa438, 0x96fe },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x04f8 },	\
	{ 0xa438, 0xf9ef },	\
	{ 0xa438, 0x59f9 },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x15ad },	\
	{ 0xa438, 0x2622 },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0x3972 },	\
	{ 0xa438, 0x9e10 },	\
	{ 0xa438, 0xe083 },	\
	{ 0xa438, 0xc9ac },	\
	{ 0xa438, 0x2605 },	\
	{ 0xa438, 0x0288 },	\
	{ 0xa438, 0x2cae },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x8870 },	\
	{ 0xa438, 0xae08 },	\
	{ 0xa438, 0xe283 },	\
	{ 0xa438, 0xc9f6 },	\
	{ 0xa438, 0x36e6 },	\
	{ 0xa438, 0x83c9 },	\
	{ 0xa438, 0xfdef },	\
	{ 0xa438, 0x95fd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfafb },	\
	{ 0xa438, 0xef79 },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x5718 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ef },	\
	{ 0xa438, 0x64e2 },	\
	{ 0xa438, 0x8fe5 },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xe61b },	\
	{ 0xa438, 0x659e },	\
	{ 0xa438, 0x10e4 },	\
	{ 0xa438, 0x8fe5 },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xe6e2 },	\
	{ 0xa438, 0x83c9 },	\
	{ 0xa438, 0xf636 },	\
	{ 0xa438, 0xe683 },	\
	{ 0xa438, 0xc9ae },	\
	{ 0xa438, 0x13e2 },	\
	{ 0xa438, 0x83c9 },	\
	{ 0xa438, 0xf736 },	\
	{ 0xa438, 0xe683 },	\
	{ 0xa438, 0xc902 },	\
	{ 0xa438, 0x5820 },	\
	{ 0xa438, 0xef57 },	\
	{ 0xa438, 0xe68f },	\
	{ 0xa438, 0xe7e7 },	\
	{ 0xa438, 0x8fe8 },	\
	{ 0xa438, 0xffef },	\
	{ 0xa438, 0x97ff },	\
	{ 0xa438, 0xfefd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfafb },	\
	{ 0xa438, 0xef79 },	\
	{ 0xa438, 0xfbe2 },	\
	{ 0xa438, 0x8fe7 },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xe8ef },	\
	{ 0xa438, 0x65e2 },	\
	{ 0xa438, 0x81b8 },	\
	{ 0xa438, 0xe381 },	\
	{ 0xa438, 0xb9ef },	\
	{ 0xa438, 0x7502 },	\
	{ 0xa438, 0x583b },	\
	{ 0xa438, 0xac50 },	\
	{ 0xa438, 0x1abf },	\
	{ 0xa438, 0x5718 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ef },	\
	{ 0xa438, 0x64e2 },	\
	{ 0xa438, 0x8fe5 },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xe61b },	\
	{ 0xa438, 0x659e },	\
	{ 0xa438, 0x1ce4 },	\
	{ 0xa438, 0x8fe5 },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xe6ae },	\
	{ 0xa438, 0x0cbf },	\
	{ 0xa438, 0x5715 },	\
	{ 0xa438, 0x0258 },	\
	{ 0xa438, 0xb1bf },	\
	{ 0xa438, 0x5715 },	\
	{ 0xa438, 0x0258 },	\
	{ 0xa438, 0xa8e2 },	\
	{ 0xa438, 0x83c9 },	\
	{ 0xa438, 0xf636 },	\
	{ 0xa438, 0xe683 },	\
	{ 0xa438, 0xc9ff },	\
	{ 0xa438, 0xef97 },	\
	{ 0xa438, 0xfffe },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x04f8 },	\
	{ 0xa438, 0xf9fa },	\
	{ 0xa438, 0xef69 },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x15ad },	\
	{ 0xa438, 0x264b },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0xca02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x42bf },	\
	{ 0xa438, 0x8931 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ef },	\
	{ 0xa438, 0x54bf },	\
	{ 0xa438, 0x576c },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x001b },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x4c02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xac29 },	\
	{ 0xa438, 0x0dac },	\
	{ 0xa438, 0x2805 },	\
	{ 0xa438, 0xa302 },	\
	{ 0xa438, 0x16ae },	\
	{ 0xa438, 0x20a3 },	\
	{ 0xa438, 0x0311 },	\
	{ 0xa438, 0xae1b },	\
	{ 0xa438, 0xa304 },	\
	{ 0xa438, 0x0cae },	\
	{ 0xa438, 0x16a3 },	\
	{ 0xa438, 0x0802 },	\
	{ 0xa438, 0xae11 },	\
	{ 0xa438, 0xa309 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x0cbf },	\
	{ 0xa438, 0x5715 },	\
	{ 0xa438, 0x0258 },	\
	{ 0xa438, 0xb1bf },	\
	{ 0xa438, 0x5715 },	\
	{ 0xa438, 0x0258 },	\
	{ 0xa438, 0xa8ef },	\
	{ 0xa438, 0x96fe },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x04f0 },	\
	{ 0xa438, 0xa300 },	\
	{ 0xa438, 0xf0a3 },	\
	{ 0xa438, 0x02f0 },	\
	{ 0xa438, 0xa304 },	\
	{ 0xa438, 0xf0a3 },	\
	{ 0xa438, 0x06f0 },	\
	{ 0xa438, 0xa308 },	\
	{ 0xa438, 0xf0a2 },	\
	{ 0xa438, 0x8074 },	\
	{ 0xa438, 0xa600 },	\
	{ 0xa438, 0xac4f },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x0bef },	\
	{ 0xa438, 0x46f6 },	\
	{ 0xa438, 0x273c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0xab1b },	\
	{ 0xa438, 0xae16 },	\
	{ 0xa438, 0xe081 },	\
	{ 0xa438, 0xabe1 },	\
	{ 0xa438, 0x81ac },	\
	{ 0xa438, 0x1b46 },	\
	{ 0xa438, 0xab0c },	\
	{ 0xa438, 0xac32 },	\
	{ 0xa438, 0x04ef },	\
	{ 0xa438, 0x32ae },	\
	{ 0xa438, 0x02d3 },	\
	{ 0xa438, 0x04af },	\
	{ 0xa438, 0x486c },	\
	{ 0xa438, 0xaf48 },	\
	{ 0xa438, 0x82af },	\
	{ 0xa438, 0x4888 },	\
	{ 0xa438, 0xe081 },	\
	{ 0xa438, 0x9be1 },	\
	{ 0xa438, 0x819c },	\
	{ 0xa438, 0xe28f },	\
	{ 0xa438, 0xe3ad },	\
	{ 0xa438, 0x3009 },	\
	{ 0xa438, 0x1f55 },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xe20c },	\
	{ 0xa438, 0x581a },	\
	{ 0xa438, 0x45e4 },	\
	{ 0xa438, 0x83a6 },	\
	{ 0xa438, 0xe583 },	\
	{ 0xa438, 0xa7af },	\
	{ 0xa438, 0x2a75 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xe3ad },	\
	{ 0xa438, 0x201c },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xe10c },	\
	{ 0xa438, 0x44ef },	\
	{ 0xa438, 0x64e0 },	\
	{ 0xa438, 0x8232 },	\
	{ 0xa438, 0xe182 },	\
	{ 0xa438, 0x331b },	\
	{ 0xa438, 0x649f },	\
	{ 0xa438, 0x091f },	\
	{ 0xa438, 0x44e1 },	\
	{ 0xa438, 0x8fe2 },	\
	{ 0xa438, 0x0c48 },	\
	{ 0xa438, 0x1b54 },	\
	{ 0xa438, 0xe683 },	\
	{ 0xa438, 0xa6e7 },	\
	{ 0xa438, 0x83a7 },	\
	{ 0xa438, 0xaf2b },	\
	{ 0xa438, 0xd900 },	\
	{ 0xa436, 0xb818 },	\
	{ 0xa438, 0x043d },	\
	{ 0xa436, 0xb81a },	\
	{ 0xa438, 0x06a3 },	\
	{ 0xa436, 0xb81c },	\
	{ 0xa438, 0x476d },	\
	{ 0xa436, 0xb81e },	\
	{ 0xa438, 0x4852 },	\
	{ 0xa436, 0xb850 },	\
	{ 0xa438, 0x2a69 },	\
	{ 0xa436, 0xb852 },	\
	{ 0xa438, 0x2bd3 },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb884 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb832 },	\
	{ 0xa438, 0x003f },	\
	{ 0xb844, 0xffff },	\
	{ 0xa436, 0x8fe9 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x8feb },	\
	{ 0xa438, 0x02fe },	\
	{ 0xa436, 0x8fed },	\
	{ 0xa438, 0x0019 },	\
	{ 0xa436, 0x8fef },	\
	{ 0xa438, 0x0bdb },	\
	{ 0xa436, 0x8ff1 },	\
	{ 0xa438, 0x0ca4 },	\
	{ 0xa436, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x801e },	\
	{ 0xa438, 0x0024 },	\
	{ 0xb820, 0x0000 }

#define MAC_R25D_MCU		\
	{ 0xa436, 0x8023 },	\
	{ 0xa438, 0x3800 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0001 },	\
	{ 0xb820, 0x0090 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8018 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8021 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8029 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8031 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8035 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x819c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x81e9 },	\
	{ 0xa438, 0xd711 },	\
	{ 0xa438, 0x6081 },	\
	{ 0xa438, 0x8904 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x2021 },	\
	{ 0xa438, 0xa904 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x2021 },	\
	{ 0xa438, 0xd75f },	\
	{ 0xa438, 0x4083 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0xa908 },	\
	{ 0xa438, 0x87f0 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x17e0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x13c3 },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x2005 },	\
	{ 0xa438, 0x8027 },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1434 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x14a5 },	\
	{ 0xa438, 0xc504 },	\
	{ 0xa438, 0xce20 },	\
	{ 0xa438, 0xcf01 },	\
	{ 0xa438, 0xd70a },	\
	{ 0xa438, 0x4005 },	\
	{ 0xa438, 0xcf02 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1c50 },	\
	{ 0xa438, 0xa980 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x14f3 },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x67b1 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x65bb },	\
	{ 0xa438, 0x63da },	\
	{ 0xa438, 0x61f9 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c10 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0808 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0470 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0430 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0410 },	\
	{ 0xa438, 0xf02a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c20 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0804 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0470 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0430 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0420 },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c40 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0802 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0470 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0450 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0440 },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c80 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0801 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x04b0 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0490 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0480 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xc48e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5faf },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8e01 },	\
	{ 0xa438, 0x8c0f },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x17e0 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x4074 },	\
	{ 0xa438, 0x6195 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x60f5 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0xf009 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0xf006 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0d03 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x2529 },	\
	{ 0xa438, 0x809c },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x607b },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0xf00f },	\
	{ 0xa438, 0x431a },	\
	{ 0xa438, 0xf021 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x617b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1ad1 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x608e },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x5f34 },	\
	{ 0xa438, 0xf020 },	\
	{ 0xa438, 0xf053 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1ad1 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x608e },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x5f34 },	\
	{ 0xa438, 0xf023 },	\
	{ 0xa438, 0xf067 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1ad1 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x608e },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x5f34 },	\
	{ 0xa438, 0xf026 },	\
	{ 0xa438, 0xf07b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1ad1 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x608e },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x5f34 },	\
	{ 0xa438, 0xf029 },	\
	{ 0xa438, 0xf08f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8173 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x7fb4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8188 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fae },	\
	{ 0xa438, 0xf028 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8173 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x7fb4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8188 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fae },	\
	{ 0xa438, 0xf039 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8173 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x7fb4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8188 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fae },	\
	{ 0xa438, 0xf04a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8173 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x7fb4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8188 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fae },	\
	{ 0xa438, 0xf05b },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac01 },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a2f },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac11 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fb0 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4079 },	\
	{ 0xa438, 0xa80f },	\
	{ 0xa438, 0xf05d },	\
	{ 0xa438, 0x4b98 },	\
	{ 0xa438, 0xa808 },	\
	{ 0xa438, 0xf05a },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac02 },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a2f },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac22 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa420 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fb0 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4079 },	\
	{ 0xa438, 0xa80f },	\
	{ 0xa438, 0xf03f },	\
	{ 0xa438, 0x47d8 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0xf03c },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac04 },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a2f },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac44 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa440 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fb0 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4079 },	\
	{ 0xa438, 0xa80f },	\
	{ 0xa438, 0xf021 },	\
	{ 0xa438, 0x4418 },	\
	{ 0xa438, 0xa802 },	\
	{ 0xa438, 0xf01e },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac08 },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a2f },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac88 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa480 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1a41 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fb0 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4079 },	\
	{ 0xa438, 0xa80f },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x4058 },	\
	{ 0xa438, 0xa801 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x16ed },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x3088 },	\
	{ 0xa438, 0x817a },	\
	{ 0xa438, 0x6193 },	\
	{ 0xa438, 0x6132 },	\
	{ 0xa438, 0x60d1 },	\
	{ 0xa438, 0x3298 },	\
	{ 0xa438, 0x8185 },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0xa808 },	\
	{ 0xa438, 0xf008 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0xf006 },	\
	{ 0xa438, 0xa802 },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xa801 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xa80f },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0800 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x6211 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x619b },	\
	{ 0xa438, 0x611a },	\
	{ 0xa438, 0x6099 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0808 },	\
	{ 0xa438, 0xf009 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0804 },	\
	{ 0xa438, 0xf006 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0802 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0801 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0800 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xc48d },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8d03 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x4045 },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x2529 },	\
	{ 0xa438, 0x81ad },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x607b },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0xf013 },	\
	{ 0xa438, 0x441a },	\
	{ 0xa438, 0xf02d },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x61fb },	\
	{ 0xa438, 0xbb01 },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x6171 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c10 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0410 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0808 },	\
	{ 0xa438, 0xf02a },	\
	{ 0xa438, 0xbb02 },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x6171 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c20 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0420 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0804 },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0xbb04 },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x6171 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c40 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0440 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0802 },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0xbb08 },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x6171 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c80 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0480 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0801 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1616 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x15d8 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x161f },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x14f2 },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x1c4f },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x1433 },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x13c1 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x2020 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0x7f00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x07f8 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x8904 },	\
	{ 0xa438, 0x813c },	\
	{ 0xa438, 0xa13d },	\
	{ 0xa438, 0xcc01 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa152 },	\
	{ 0xa438, 0x1384 },	\
	{ 0xa436, 0xa154 },	\
	{ 0xa438, 0x1fa8 },	\
	{ 0xa436, 0xa156 },	\
	{ 0xa438, 0x218b },	\
	{ 0xa436, 0xa158 },	\
	{ 0xa438, 0x21b8 },	\
	{ 0xa436, 0xa15a },	\
	{ 0xa438, 0x021c },	\
	{ 0xa436, 0xa15c },	\
	{ 0xa438, 0x3fff },	\
	{ 0xa436, 0xa15e },	\
	{ 0xa438, 0x3fff },	\
	{ 0xa436, 0xa160 },	\
	{ 0xa438, 0x3fff },	\
	{ 0xa436, 0xa150 },	\
	{ 0xa438, 0x001f },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8013 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8045 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8049 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x804d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8059 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x805d },	\
	{ 0xa438, 0xc2ff },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0042 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x02e5 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x02b4 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40e3 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f6c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8021 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0073 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0084 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x4061 },	\
	{ 0xa438, 0xba0f },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0x4060 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x802a },	\
	{ 0xa438, 0xba10 },	\
	{ 0xa438, 0x0800 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x60bb },	\
	{ 0xa438, 0x611c },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x1a01 },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0x60fc },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x1a02 },	\
	{ 0xa438, 0xf006 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x1a04 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x1a08 },	\
	{ 0xa438, 0x0800 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0504 },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x02c0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x8021 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0139 },	\
	{ 0xa438, 0x9a1f },	\
	{ 0xa438, 0x8bf0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x02df },	\
	{ 0xa438, 0x9a1f },	\
	{ 0xa438, 0x9910 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x02d7 },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x8d01 },	\
	{ 0xa438, 0x9a1f },	\
	{ 0xa438, 0x9910 },	\
	{ 0xa438, 0x9860 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x85f0 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x015c },	\
	{ 0xa438, 0x8580 },	\
	{ 0xa438, 0x8d02 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x018f },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00dd },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0x00db },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0x018e },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0x015a },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0x02d6 },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0x02de },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0x0137 },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x0071 },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x0041 },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x1ff8 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x001c },	\
	{ 0xa438, 0xce15 },	\
	{ 0xa438, 0xd105 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0x8320 },	\
	{ 0xa438, 0xffd7 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa164 },	\
	{ 0xa438, 0x0260 },	\
	{ 0xa436, 0xa166 },	\
	{ 0xa438, 0x0add },	\
	{ 0xa436, 0xa168 },	\
	{ 0xa438, 0x05cc },	\
	{ 0xa436, 0xa16a },	\
	{ 0xa438, 0x05c5 },	\
	{ 0xa436, 0xa16c },	\
	{ 0xa438, 0x0429 },	\
	{ 0xa436, 0xa16e },	\
	{ 0xa438, 0x07b6 },	\
	{ 0xa436, 0xa170 },	\
	{ 0xa438, 0x0259 },	\
	{ 0xa436, 0xa172 },	\
	{ 0xa438, 0x3fff },	\
	{ 0xa436, 0xa162 },	\
	{ 0xa438, 0x003f },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8023 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x814c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8156 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x815e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8210 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8221 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x822f },	\
	{ 0xa438, 0xa801 },	\
	{ 0xa438, 0x9308 },	\
	{ 0xa438, 0xb201 },	\
	{ 0xa438, 0xb301 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x4000 },	\
	{ 0xa438, 0xd2ff },	\
	{ 0xa438, 0xb302 },	\
	{ 0xa438, 0xd200 },	\
	{ 0xa438, 0xb201 },	\
	{ 0xa438, 0xb309 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x4000 },	\
	{ 0xa438, 0xd2ff },	\
	{ 0xa438, 0xb302 },	\
	{ 0xa438, 0xd200 },	\
	{ 0xa438, 0xa800 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0031 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4543 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x40fe },	\
	{ 0xa438, 0xd1b7 },	\
	{ 0xa438, 0xd049 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fbb },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0x8501 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x0c70 },	\
	{ 0xa438, 0x0b00 },	\
	{ 0xa438, 0x0c07 },	\
	{ 0xa438, 0x0604 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xa510 },	\
	{ 0xa438, 0xce49 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0x8520 },	\
	{ 0xa438, 0xa520 },	\
	{ 0xa438, 0xa501 },	\
	{ 0xa438, 0xd105 },	\
	{ 0xa438, 0xd047 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x6087 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f7b },	\
	{ 0xa438, 0xffe9 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x8501 },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x5e08 },	\
	{ 0xa438, 0x8530 },	\
	{ 0xa438, 0xba20 },	\
	{ 0xa438, 0xf00c },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4098 },	\
	{ 0xa438, 0xd1ef },	\
	{ 0xa438, 0xd047 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd1db },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fbb },	\
	{ 0xa438, 0x8980 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6126 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6060 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6077 },	\
	{ 0xa438, 0x8410 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0xce02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0xcd81 },	\
	{ 0xa438, 0xd412 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1069 },	\
	{ 0xa438, 0xcd82 },	\
	{ 0xa438, 0xd40e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1069 },	\
	{ 0xa438, 0xcd83 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6c26 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6060 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6b77 },	\
	{ 0xa438, 0xa340 },	\
	{ 0xa438, 0x0c06 },	\
	{ 0xa438, 0x0102 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0xa240 },	\
	{ 0xa438, 0xa902 },	\
	{ 0xa438, 0xa204 },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa364 },	\
	{ 0xa438, 0xab02 },	\
	{ 0xa438, 0x8380 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xcd8d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x5fb5 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7fb4 },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x6065 },	\
	{ 0xa438, 0x7c74 },	\
	{ 0xa438, 0xfffb },	\
	{ 0xa438, 0xb820 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7fa5 },	\
	{ 0xa438, 0x9820 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0x8902 },	\
	{ 0xa438, 0xa120 },	\
	{ 0xa438, 0xa380 },	\
	{ 0xa438, 0xce02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xa324 },	\
	{ 0xa438, 0xab02 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0x8118 },	\
	{ 0xa438, 0x863f },	\
	{ 0xa438, 0x87fb },	\
	{ 0xa438, 0xcd8e },	\
	{ 0xa438, 0xd193 },	\
	{ 0xa438, 0xd047 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10a3 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f7b },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10a3 },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x5f78 },	\
	{ 0xa438, 0xa210 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xd047 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd160 },	\
	{ 0xa438, 0xd04b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10a3 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f7b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10a3 },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x5f79 },	\
	{ 0xa438, 0x8120 },	\
	{ 0xa438, 0xbb20 },	\
	{ 0xa438, 0xf04c },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa340 },	\
	{ 0xa438, 0x0c06 },	\
	{ 0xa438, 0x0102 },	\
	{ 0xa438, 0xa240 },	\
	{ 0xa438, 0xa290 },	\
	{ 0xa438, 0xa324 },	\
	{ 0xa438, 0xab02 },	\
	{ 0xa438, 0xd13e },	\
	{ 0xa438, 0xd05a },	\
	{ 0xa438, 0xd13e },	\
	{ 0xa438, 0xd06b },	\
	{ 0xa438, 0xcd84 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x6079 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f5c },	\
	{ 0xa438, 0xcd8a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x6079 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f5d },	\
	{ 0xa438, 0xcd8b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xcd8c },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6050 },	\
	{ 0xa438, 0xab04 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4083 },	\
	{ 0xa438, 0xd160 },	\
	{ 0xa438, 0xd04b },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd193 },	\
	{ 0xa438, 0xd047 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fbb },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x8410 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f94 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7fb4 },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x6105 },	\
	{ 0xa438, 0x6054 },	\
	{ 0xa438, 0xfffb },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x5fb9 },	\
	{ 0xa438, 0xfff0 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0xb820 },	\
	{ 0xa438, 0xcd85 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7fa5 },	\
	{ 0xa438, 0x9820 },	\
	{ 0xa438, 0xbb20 },	\
	{ 0xa438, 0xd105 },	\
	{ 0xa438, 0xd042 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x5fbb },	\
	{ 0xa438, 0x5f85 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f5b },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6090 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4043 },	\
	{ 0xa438, 0xaa20 },	\
	{ 0xa438, 0xcd86 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0xd1c7 },	\
	{ 0xa438, 0xd045 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd17a },	\
	{ 0xa438, 0xd04b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fbb },	\
	{ 0xa438, 0x0c18 },	\
	{ 0xa438, 0x0108 },	\
	{ 0xa438, 0x0c3f },	\
	{ 0xa438, 0x0609 },	\
	{ 0xa438, 0x0cfb },	\
	{ 0xa438, 0x0729 },	\
	{ 0xa438, 0xa308 },	\
	{ 0xa438, 0x8320 },	\
	{ 0xa438, 0xd105 },	\
	{ 0xa438, 0xd042 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fbb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x08f7 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10a3 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x607b },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f2b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a81 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x40bd },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x4065 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1121 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1124 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8f80 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x641d },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x62b2 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x4116 },	\
	{ 0xa438, 0xce15 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8f40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4247 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x3691 },	\
	{ 0xa438, 0x8183 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1108 },	\
	{ 0xa438, 0xcd64 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x3398 },	\
	{ 0xa438, 0x8203 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x620e },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x6096 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x6051 },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x605d },	\
	{ 0xa438, 0xf008 },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x609d },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x405f },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x58fb },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xc7aa },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x6d2e },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x6096 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x6051 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x607d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0cc7 },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x60bd },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x407f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0e42 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x40a4 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8e20 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x40a5 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8e40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x659d },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x62b2 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x4116 },	\
	{ 0xa438, 0xce15 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8f40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4247 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x3691 },	\
	{ 0xa438, 0x81de },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x60e4 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x07a0 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x87f0 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1108 },	\
	{ 0xa438, 0xcd61 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x3398 },	\
	{ 0xa438, 0x8203 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x6096 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x6051 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x607d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0cc7 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x61ce },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x767d },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x563f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0e42 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xae40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0c47 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xaf80 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0b5f },	\
	{ 0xa438, 0x607c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x027a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5fa3 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8e01 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x027d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x40a5 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8e40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x6065 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0cea },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0cf4 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x6fd1 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x6eee },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x4d0f },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x4cc5 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x4c99 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x6c57 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6c11 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8c20 },	\
	{ 0xa438, 0xa608 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x40a7 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa620 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xac40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8290 },	\
	{ 0xa438, 0x8306 },	\
	{ 0xa438, 0x8b02 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0xcd99 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10cc },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x69f1 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x690e },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x5ee6 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x87f0 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xce46 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x40a7 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xcd9a },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6078 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x609a },	\
	{ 0xa438, 0xd109 },	\
	{ 0xa438, 0xd074 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd109 },	\
	{ 0xa438, 0xd075 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10cc },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x65b1 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x64ce },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5efe },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8608 },	\
	{ 0xa438, 0x8c40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x8201 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8290 },	\
	{ 0xa438, 0x8306 },	\
	{ 0xa438, 0x8b02 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xc7aa },	\
	{ 0xa438, 0x8570 },	\
	{ 0xa438, 0x8d08 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xcd9b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0c8b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x61d9 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4193 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xae40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0c47 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0df8 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8339 },	\
	{ 0xa438, 0x0800 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8d08 },	\
	{ 0xa438, 0x8f02 },	\
	{ 0xa438, 0x8c40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x8201 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x40a7 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa620 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8290 },	\
	{ 0xa438, 0x8306 },	\
	{ 0xa438, 0x8b02 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xaa03 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xac20 },	\
	{ 0xa438, 0xa608 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0xcd95 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x7b91 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7aae },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x7ab0 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x7ef3 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x5eb3 },	\
	{ 0xa438, 0x84b0 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa608 },	\
	{ 0xa438, 0xc700 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xce54 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0xa290 },	\
	{ 0xa438, 0xa304 },	\
	{ 0xa438, 0xab02 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6050 },	\
	{ 0xa438, 0xab04 },	\
	{ 0xa438, 0x0c38 },	\
	{ 0xa438, 0x0608 },	\
	{ 0xa438, 0xaa0b },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8d01 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xae40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x40a4 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8e20 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8c20 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6078 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x609a },	\
	{ 0xa438, 0xd109 },	\
	{ 0xa438, 0xd074 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd109 },	\
	{ 0xa438, 0xd075 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x62b2 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x4116 },	\
	{ 0xa438, 0xce54 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x10be },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8f40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4247 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x3691 },	\
	{ 0xa438, 0x8326 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8570 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x109e },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x60f3 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x618e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5b5e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0deb },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xae40 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0c47 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0df8 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8608 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0e2b },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x0d14 },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x0ce8 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x0279 },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x0b19 },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x111f },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0a7b },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0ba3 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x0022 },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x859b },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0xb3af },	\
	{ 0xa438, 0x863b },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0x4caf },	\
	{ 0xa438, 0x8688 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0xceaf },	\
	{ 0xa438, 0x8744 },	\
	{ 0xa438, 0xaf87 },	\
	{ 0xa438, 0x68af },	\
	{ 0xa438, 0x8781 },	\
	{ 0xa438, 0xbf5e },	\
	{ 0xa438, 0x7202 },	\
	{ 0xa438, 0x5f7e },	\
	{ 0xa438, 0xac28 },	\
	{ 0xa438, 0x68e1 },	\
	{ 0xa438, 0x84e6 },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x09bf },	\
	{ 0xa438, 0x5e75 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x7eac },	\
	{ 0xa438, 0x2d59 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xebad },	\
	{ 0xa438, 0x2809 },	\
	{ 0xa438, 0xbf5e },	\
	{ 0xa438, 0x7502 },	\
	{ 0xa438, 0x5f7e },	\
	{ 0xa438, 0xac2e },	\
	{ 0xa438, 0x50e1 },	\
	{ 0xa438, 0x84e6 },	\
	{ 0xa438, 0xac28 },	\
	{ 0xa438, 0x08bf },	\
	{ 0xa438, 0x873e },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x3cae },	\
	{ 0xa438, 0x06bf },	\
	{ 0xa438, 0x873e },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x33bf },	\
	{ 0xa438, 0x8741 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x33ee },	\
	{ 0xa438, 0x8fea },	\
	{ 0xa438, 0x02e1 },	\
	{ 0xa438, 0x84e4 },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x14e1 },	\
	{ 0xa438, 0x8fe8 },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x17e1 },	\
	{ 0xa438, 0x84e5 },	\
	{ 0xa438, 0x11e5 },	\
	{ 0xa438, 0x84e5 },	\
	{ 0xa438, 0xa10c },	\
	{ 0xa438, 0x04ee },	\
	{ 0xa438, 0x84e5 },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x4977 },	\
	{ 0xa438, 0xee84 },	\
	{ 0xa438, 0xdc03 },	\
	{ 0xa438, 0xae1d },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xe811 },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xe8ae },	\
	{ 0xa438, 0x14bf },	\
	{ 0xa438, 0x873e },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x3cbf },	\
	{ 0xa438, 0x8741 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x3cee },	\
	{ 0xa438, 0x8fea },	\
	{ 0xa438, 0x01ee },	\
	{ 0xa438, 0x84e4 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x50c1 },	\
	{ 0xa438, 0x1f00 },	\
	{ 0xa438, 0xbf5a },	\
	{ 0xa438, 0x6102 },	\
	{ 0xa438, 0x5f5f },	\
	{ 0xa438, 0xbf5a },	\
	{ 0xa438, 0x5e02 },	\
	{ 0xa438, 0x5f3c },	\
	{ 0xa438, 0xaf45 },	\
	{ 0xa438, 0x7be0 },	\
	{ 0xa438, 0x8012 },	\
	{ 0xa438, 0xad23 },	\
	{ 0xa438, 0x141f },	\
	{ 0xa438, 0x001f },	\
	{ 0xa438, 0x22d1 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x3fcf },	\
	{ 0xa438, 0x0261 },	\
	{ 0xa438, 0x3412 },	\
	{ 0xa438, 0xa204 },	\
	{ 0xa438, 0xf6ee },	\
	{ 0xa438, 0x8317 },	\
	{ 0xa438, 0x00e0 },	\
	{ 0xa438, 0x8012 },	\
	{ 0xa438, 0xad24 },	\
	{ 0xa438, 0x141f },	\
	{ 0xa438, 0x001f },	\
	{ 0xa438, 0x22d1 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x3fd7 },	\
	{ 0xa438, 0x0261 },	\
	{ 0xa438, 0x3412 },	\
	{ 0xa438, 0xa204 },	\
	{ 0xa438, 0xf6ee },	\
	{ 0xa438, 0x8317 },	\
	{ 0xa438, 0x00ef },	\
	{ 0xa438, 0x96fe },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0xaf42 },	\
	{ 0xa438, 0x9802 },	\
	{ 0xa438, 0x56ec },	\
	{ 0xa438, 0xf70b },	\
	{ 0xa438, 0xac13 },	\
	{ 0xa438, 0x0fbf },	\
	{ 0xa438, 0x5e75 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x7eac },	\
	{ 0xa438, 0x280c },	\
	{ 0xa438, 0xe2ff },	\
	{ 0xa438, 0xcfad },	\
	{ 0xa438, 0x32ee },	\
	{ 0xa438, 0x0257 },	\
	{ 0xa438, 0x05af },	\
	{ 0xa438, 0x00a4 },	\
	{ 0xa438, 0x0286 },	\
	{ 0xa438, 0xaaae },	\
	{ 0xa438, 0xeff8 },	\
	{ 0xa438, 0xf9ef },	\
	{ 0xa438, 0x5902 },	\
	{ 0xa438, 0x1fe1 },	\
	{ 0xa438, 0xbf59 },	\
	{ 0xa438, 0x4d02 },	\
	{ 0xa438, 0x5f3c },	\
	{ 0xa438, 0xac13 },	\
	{ 0xa438, 0x09bf },	\
	{ 0xa438, 0x5e75 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x7ea1 },	\
	{ 0xa438, 0x00f4 },	\
	{ 0xa438, 0xbf59 },	\
	{ 0xa438, 0x4d02 },	\
	{ 0xa438, 0x5f33 },	\
	{ 0xa438, 0xef95 },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x04bf },	\
	{ 0xa438, 0x5e72 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x7eac },	\
	{ 0xa438, 0x284a },	\
	{ 0xa438, 0xe184 },	\
	{ 0xa438, 0xe6ad },	\
	{ 0xa438, 0x2809 },	\
	{ 0xa438, 0xbf5e },	\
	{ 0xa438, 0x7502 },	\
	{ 0xa438, 0x5f7e },	\
	{ 0xa438, 0xac2d },	\
	{ 0xa438, 0x3be1 },	\
	{ 0xa438, 0x8feb },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x09bf },	\
	{ 0xa438, 0x5e75 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x7eac },	\
	{ 0xa438, 0x2e32 },	\
	{ 0xa438, 0xe184 },	\
	{ 0xa438, 0xe6ac },	\
	{ 0xa438, 0x2808 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3e02 },	\
	{ 0xa438, 0x5f3c },	\
	{ 0xa438, 0xae06 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3e02 },	\
	{ 0xa438, 0x5f33 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4102 },	\
	{ 0xa438, 0x5f33 },	\
	{ 0xa438, 0xee8f },	\
	{ 0xa438, 0xea04 },	\
	{ 0xa438, 0xbf5e },	\
	{ 0xa438, 0x4e02 },	\
	{ 0xa438, 0x5f7e },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x1f02 },	\
	{ 0xa438, 0x4b12 },	\
	{ 0xa438, 0xae1a },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3e02 },	\
	{ 0xa438, 0x5f3c },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4102 },	\
	{ 0xa438, 0x5f3c },	\
	{ 0xa438, 0xee8f },	\
	{ 0xa438, 0xea03 },	\
	{ 0xa438, 0xbf5e },	\
	{ 0xa438, 0x2a02 },	\
	{ 0xa438, 0x5f33 },	\
	{ 0xa438, 0xee84 },	\
	{ 0xa438, 0xe701 },	\
	{ 0xa438, 0xaf4a },	\
	{ 0xa438, 0x7444 },	\
	{ 0xa438, 0xac0e },	\
	{ 0xa438, 0x55ac },	\
	{ 0xa438, 0x0ebf },	\
	{ 0xa438, 0x5e75 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x7ead },	\
	{ 0xa438, 0x2d0b },	\
	{ 0xa438, 0xbf5e },	\
	{ 0xa438, 0x36e1 },	\
	{ 0xa438, 0x8fe9 },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x5fae },	\
	{ 0xa438, 0x09bf },	\
	{ 0xa438, 0x5e36 },	\
	{ 0xa438, 0xe184 },	\
	{ 0xa438, 0xe102 },	\
	{ 0xa438, 0x5f5f },	\
	{ 0xa438, 0xee8f },	\
	{ 0xa438, 0xe800 },	\
	{ 0xa438, 0xaf49 },	\
	{ 0xa438, 0xcdbf },	\
	{ 0xa438, 0x595c },	\
	{ 0xa438, 0x025f },	\
	{ 0xa438, 0x7ea1 },	\
	{ 0xa438, 0x0203 },	\
	{ 0xa438, 0xaf87 },	\
	{ 0xa438, 0x79d1 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x877c },	\
	{ 0xa438, 0xe181 },	\
	{ 0xa438, 0x941f },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x3ff7 },	\
	{ 0xa438, 0xac4e },	\
	{ 0xa438, 0x06ac },	\
	{ 0xa438, 0x4003 },	\
	{ 0xa438, 0xaf24 },	\
	{ 0xa438, 0x97af },	\
	{ 0xa438, 0x2467 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x5082 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0x4575 },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0x425f },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0x0096 },	\
	{ 0xa436, 0xb886 },	\
	{ 0xa438, 0x4a44 },	\
	{ 0xa436, 0xb888 },	\
	{ 0xa438, 0x49c4 },	\
	{ 0xa436, 0xb88a },	\
	{ 0xa438, 0x3ff2 },	\
	{ 0xa436, 0xb88c },	\
	{ 0xa438, 0x245c },	\
	{ 0xa436, 0xb838 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xb820, 0x0010 },	\
	{ 0xa436, 0x843d },	\
	{ 0xa438, 0xaf84 },	\
	{ 0xa438, 0xa6af },	\
	{ 0xa438, 0x8540 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0xaeaf },	\
	{ 0xa438, 0x85b5 },	\
	{ 0xa438, 0xaf87 },	\
	{ 0xa438, 0x7daf },	\
	{ 0xa438, 0x8784 },	\
	{ 0xa438, 0xaf87 },	\
	{ 0xa438, 0x87af },	\
	{ 0xa438, 0x87e5 },	\
	{ 0xa438, 0x0066 },	\
	{ 0xa438, 0x0a03 },	\
	{ 0xa438, 0x6607 },	\
	{ 0xa438, 0x2666 },	\
	{ 0xa438, 0x1c00 },	\
	{ 0xa438, 0x660d },	\
	{ 0xa438, 0x0166 },	\
	{ 0xa438, 0x1004 },	\
	{ 0xa438, 0x6616 },	\
	{ 0xa438, 0x0566 },	\
	{ 0xa438, 0x1f06 },	\
	{ 0xa438, 0x6a5d },	\
	{ 0xa438, 0x2766 },	\
	{ 0xa438, 0x1900 },	\
	{ 0xa438, 0x6625 },	\
	{ 0xa438, 0x2466 },	\
	{ 0xa438, 0x2820 },	\
	{ 0xa438, 0x662b },	\
	{ 0xa438, 0x2466 },	\
	{ 0xa438, 0x4600 },	\
	{ 0xa438, 0x664c },	\
	{ 0xa438, 0x0166 },	\
	{ 0xa438, 0x4902 },	\
	{ 0xa438, 0x8861 },	\
	{ 0xa438, 0x0388 },	\
	{ 0xa438, 0x5e05 },	\
	{ 0xa438, 0x886d },	\
	{ 0xa438, 0x0588 },	\
	{ 0xa438, 0x7005 },	\
	{ 0xa438, 0x8873 },	\
	{ 0xa438, 0x0588 },	\
	{ 0xa438, 0x7605 },	\
	{ 0xa438, 0x8879 },	\
	{ 0xa438, 0x0588 },	\
	{ 0xa438, 0x7c05 },	\
	{ 0xa438, 0x887f },	\
	{ 0xa438, 0x0588 },	\
	{ 0xa438, 0x8205 },	\
	{ 0xa438, 0x8885 },	\
	{ 0xa438, 0x0588 },	\
	{ 0xa438, 0x881e },	\
	{ 0xa438, 0x13ad },	\
	{ 0xa438, 0x2841 },	\
	{ 0xa438, 0xbf64 },	\
	{ 0xa438, 0xf102 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x15fc },	\
	{ 0xa438, 0xbf65 },	\
	{ 0xa438, 0xcb02 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0x0d11 },	\
	{ 0xa438, 0xf62f },	\
	{ 0xa438, 0xef31 },	\
	{ 0xa438, 0xd202 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6402 },	\
	{ 0xa438, 0x6b52 },	\
	{ 0xa438, 0xe082 },	\
	{ 0xa438, 0x020d },	\
	{ 0xa438, 0x01f6 },	\
	{ 0xa438, 0x271b },	\
	{ 0xa438, 0x03aa },	\
	{ 0xa438, 0x0182 },	\
	{ 0xa438, 0xe082 },	\
	{ 0xa438, 0x010d },	\
	{ 0xa438, 0x01f6 },	\
	{ 0xa438, 0x271b },	\
	{ 0xa438, 0x03aa },	\
	{ 0xa438, 0x0782 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6402 },	\
	{ 0xa438, 0x6b5b },	\
	{ 0xa438, 0xaf15 },	\
	{ 0xa438, 0xf9bf },	\
	{ 0xa438, 0x65cb },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x9d0d },	\
	{ 0xa438, 0x11f6 },	\
	{ 0xa438, 0x2fef },	\
	{ 0xa438, 0x31e0 },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0xf627 },	\
	{ 0xa438, 0x1b03 },	\
	{ 0xa438, 0xaa20 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf4d0 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x6587 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ee1 },	\
	{ 0xa438, 0x8ff5 },	\
	{ 0xa438, 0xbf65 },	\
	{ 0xa438, 0x8a02 },	\
	{ 0xa438, 0x6b7e },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf6bf },	\
	{ 0xa438, 0x6584 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7eaf },	\
	{ 0xa438, 0x15fc },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf1d0 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x6587 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ee1 },	\
	{ 0xa438, 0x8ff2 },	\
	{ 0xa438, 0xbf65 },	\
	{ 0xa438, 0x8a02 },	\
	{ 0xa438, 0x6b7e },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf3bf },	\
	{ 0xa438, 0x6584 },	\
	{ 0xa438, 0xaf15 },	\
	{ 0xa438, 0xfcd1 },	\
	{ 0xa438, 0x07bf },	\
	{ 0xa438, 0x65ce },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ed1 },	\
	{ 0xa438, 0x0cbf },	\
	{ 0xa438, 0x65d1 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ed1 },	\
	{ 0xa438, 0x03bf },	\
	{ 0xa438, 0x885e },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ed1 },	\
	{ 0xa438, 0x05bf },	\
	{ 0xa438, 0x8867 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ed1 },	\
	{ 0xa438, 0x07bf },	\
	{ 0xa438, 0x886a },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ebf },	\
	{ 0xa438, 0x6a6c },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x5b02 },	\
	{ 0xa438, 0x62b5 },	\
	{ 0xa438, 0xbf6a },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x6b5b },	\
	{ 0xa438, 0xbf64 },	\
	{ 0xa438, 0x4e02 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0xac28 },	\
	{ 0xa438, 0x0bbf },	\
	{ 0xa438, 0x6412 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x9da1 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0xaeec },	\
	{ 0xa438, 0xd104 },	\
	{ 0xa438, 0xbf65 },	\
	{ 0xa438, 0xce02 },	\
	{ 0xa438, 0x6b7e },	\
	{ 0xa438, 0xd104 },	\
	{ 0xa438, 0xbf65 },	\
	{ 0xa438, 0xd102 },	\
	{ 0xa438, 0x6b7e },	\
	{ 0xa438, 0xd102 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6702 },	\
	{ 0xa438, 0x6b7e },	\
	{ 0xa438, 0xd104 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6a02 },	\
	{ 0xa438, 0x6b7e },	\
	{ 0xa438, 0xaf62 },	\
	{ 0xa438, 0x72f6 },	\
	{ 0xa438, 0x0af6 },	\
	{ 0xa438, 0x09af },	\
	{ 0xa438, 0x34e3 },	\
	{ 0xa438, 0x0285 },	\
	{ 0xa438, 0xbe02 },	\
	{ 0xa438, 0x106c },	\
	{ 0xa438, 0xaf10 },	\
	{ 0xa438, 0x6bf8 },	\
	{ 0xa438, 0xfaef },	\
	{ 0xa438, 0x69e0 },	\
	{ 0xa438, 0x804c },	\
	{ 0xa438, 0xac25 },	\
	{ 0xa438, 0x17e0 },	\
	{ 0xa438, 0x8040 },	\
	{ 0xa438, 0xad25 },	\
	{ 0xa438, 0x1a02 },	\
	{ 0xa438, 0x85ed },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x40ac },	\
	{ 0xa438, 0x2511 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6502 },	\
	{ 0xa438, 0x6b5b },	\
	{ 0xa438, 0xae09 },	\
	{ 0xa438, 0x0287 },	\
	{ 0xa438, 0x2402 },	\
	{ 0xa438, 0x875a },	\
	{ 0xa438, 0x0287 },	\
	{ 0xa438, 0x4fef },	\
	{ 0xa438, 0x96fe },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8e0 },	\
	{ 0xa438, 0x8019 },	\
	{ 0xa438, 0xad20 },	\
	{ 0xa438, 0x11e0 },	\
	{ 0xa438, 0x8fe3 },	\
	{ 0xa438, 0xac20 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x860a },	\
	{ 0xa438, 0xae03 },	\
	{ 0xa438, 0x0286 },	\
	{ 0xa438, 0x7802 },	\
	{ 0xa438, 0x86c1 },	\
	{ 0xa438, 0x0287 },	\
	{ 0xa438, 0x4ffc },	\
	{ 0xa438, 0x04f8 },	\
	{ 0xa438, 0xf9ef },	\
	{ 0xa438, 0x79fb },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6802 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0x5c20 },	\
	{ 0xa438, 0x000d },	\
	{ 0xa438, 0x4da1 },	\
	{ 0xa438, 0x0151 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6802 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0x5c07 },	\
	{ 0xa438, 0xffe3 },	\
	{ 0xa438, 0x8fe4 },	\
	{ 0xa438, 0x1b31 },	\
	{ 0xa438, 0x9f41 },	\
	{ 0xa438, 0x0d48 },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xe51b },	\
	{ 0xa438, 0x319f },	\
	{ 0xa438, 0x38bf },	\
	{ 0xa438, 0x876b },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x9d5c },	\
	{ 0xa438, 0x07ff },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xe61b },	\
	{ 0xa438, 0x319f },	\
	{ 0xa438, 0x280d },	\
	{ 0xa438, 0x48e3 },	\
	{ 0xa438, 0x8fe7 },	\
	{ 0xa438, 0x1b31 },	\
	{ 0xa438, 0x9f1f },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6e02 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0x5c07 },	\
	{ 0xa438, 0xffe3 },	\
	{ 0xa438, 0x8fe8 },	\
	{ 0xa438, 0x1b31 },	\
	{ 0xa438, 0x9f0f },	\
	{ 0xa438, 0x0d48 },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xe91b },	\
	{ 0xa438, 0x319f },	\
	{ 0xa438, 0x06ee },	\
	{ 0xa438, 0x8fe3 },	\
	{ 0xa438, 0x01ae },	\
	{ 0xa438, 0x04ee },	\
	{ 0xa438, 0x8fe3 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa438, 0xef97 },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x04f8 },	\
	{ 0xa438, 0xf9ef },	\
	{ 0xa438, 0x79fb },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6802 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0x5c20 },	\
	{ 0xa438, 0x000d },	\
	{ 0xa438, 0x4da1 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6802 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0x5c06 },	\
	{ 0xa438, 0x000d },	\
	{ 0xa438, 0x49e3 },	\
	{ 0xa438, 0x8fea },	\
	{ 0xa438, 0x1b31 },	\
	{ 0xa438, 0x9f0e },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x7102 },	\
	{ 0xa438, 0x6b5b },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x7702 },	\
	{ 0xa438, 0x6b5b },	\
	{ 0xa438, 0xae0c },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x7102 },	\
	{ 0xa438, 0x6b52 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x7702 },	\
	{ 0xa438, 0x6b52 },	\
	{ 0xa438, 0xee8f },	\
	{ 0xa438, 0xe300 },	\
	{ 0xa438, 0xffef },	\
	{ 0xa438, 0x97fd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xef79 },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x8768 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x9d5c },	\
	{ 0xa438, 0x2000 },	\
	{ 0xa438, 0x0d4d },	\
	{ 0xa438, 0xa101 },	\
	{ 0xa438, 0x4abf },	\
	{ 0xa438, 0x8768 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x9d5c },	\
	{ 0xa438, 0x07ff },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xeb1b },	\
	{ 0xa438, 0x319f },	\
	{ 0xa438, 0x3a0d },	\
	{ 0xa438, 0x48e3 },	\
	{ 0xa438, 0x8fec },	\
	{ 0xa438, 0x1b31 },	\
	{ 0xa438, 0x9f31 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6b02 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xed1b },	\
	{ 0xa438, 0x319f },	\
	{ 0xa438, 0x240d },	\
	{ 0xa438, 0x48e3 },	\
	{ 0xa438, 0x8fee },	\
	{ 0xa438, 0x1b31 },	\
	{ 0xa438, 0x9f1b },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x6e02 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0xe38f },	\
	{ 0xa438, 0xef1b },	\
	{ 0xa438, 0x319f },	\
	{ 0xa438, 0x0ebf },	\
	{ 0xa438, 0x8774 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x5bbf },	\
	{ 0xa438, 0x877a },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x5bae },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa438, 0xef97 },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x04f8 },	\
	{ 0xa438, 0xef79 },	\
	{ 0xa438, 0xfbe0 },	\
	{ 0xa438, 0x8019 },	\
	{ 0xa438, 0xad20 },	\
	{ 0xa438, 0x1cee },	\
	{ 0xa438, 0x8fe3 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x8771 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x52bf },	\
	{ 0xa438, 0x8777 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x52bf },	\
	{ 0xa438, 0x8774 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x52bf },	\
	{ 0xa438, 0x877a },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x52ff },	\
	{ 0xa438, 0xef97 },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8e0 },	\
	{ 0xa438, 0x8040 },	\
	{ 0xa438, 0xf625 },	\
	{ 0xa438, 0xe480 },	\
	{ 0xa438, 0x40fc },	\
	{ 0xa438, 0x04f8 },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x4cf6 },	\
	{ 0xa438, 0x25e4 },	\
	{ 0xa438, 0x804c },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0x55a4 },	\
	{ 0xa438, 0xbaf0 },	\
	{ 0xa438, 0xa64a },	\
	{ 0xa438, 0xf0a6 },	\
	{ 0xa438, 0x4cf0 },	\
	{ 0xa438, 0xa64e },	\
	{ 0xa438, 0x66a4 },	\
	{ 0xa438, 0xb655 },	\
	{ 0xa438, 0xa4b6 },	\
	{ 0xa438, 0x00ac },	\
	{ 0xa438, 0x0e66 },	\
	{ 0xa438, 0xac0e },	\
	{ 0xa438, 0xee80 },	\
	{ 0xa438, 0x4c3a },	\
	{ 0xa438, 0xaf07 },	\
	{ 0xa438, 0xd0af },	\
	{ 0xa438, 0x26d0 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0x0ebf },	\
	{ 0xa438, 0x663d },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x52bf },	\
	{ 0xa438, 0x6643 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x52ae },	\
	{ 0xa438, 0x11bf },	\
	{ 0xa438, 0x6643 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x5bd4 },	\
	{ 0xa438, 0x0054 },	\
	{ 0xa438, 0xb4fe },	\
	{ 0xa438, 0xbf66 },	\
	{ 0xa438, 0x3d02 },	\
	{ 0xa438, 0x6b5b },	\
	{ 0xa438, 0xd300 },	\
	{ 0xa438, 0x020d },	\
	{ 0xa438, 0xf6a2 },	\
	{ 0xa438, 0x0405 },	\
	{ 0xa438, 0xe081 },	\
	{ 0xa438, 0x47ae },	\
	{ 0xa438, 0x03e0 },	\
	{ 0xa438, 0x8148 },	\
	{ 0xa438, 0xac23 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x0268 },	\
	{ 0xa438, 0xf01a },	\
	{ 0xa438, 0x10ad },	\
	{ 0xa438, 0x2f04 },	\
	{ 0xa438, 0xd100 },	\
	{ 0xa438, 0xae05 },	\
	{ 0xa438, 0xad2c },	\
	{ 0xa438, 0x02d1 },	\
	{ 0xa438, 0x0f1f },	\
	{ 0xa438, 0x00a2 },	\
	{ 0xa438, 0x0407 },	\
	{ 0xa438, 0x3908 },	\
	{ 0xa438, 0xad2f },	\
	{ 0xa438, 0x02d1 },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x0e1c },	\
	{ 0xa438, 0x2b01 },	\
	{ 0xa438, 0xad3a },	\
	{ 0xa438, 0xc9af },	\
	{ 0xa438, 0x0dee },	\
	{ 0xa438, 0xa000 },	\
	{ 0xa438, 0x2702 },	\
	{ 0xa438, 0x1beb },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xe1ac },	\
	{ 0xa438, 0x2819 },	\
	{ 0xa438, 0xee8f },	\
	{ 0xa438, 0xe101 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xbf65 },	\
	{ 0xa438, 0x9302 },	\
	{ 0xa438, 0x6b9d },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xe21f },	\
	{ 0xa438, 0x44d1 },	\
	{ 0xa438, 0x02bf },	\
	{ 0xa438, 0x6593 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7ee0 },	\
	{ 0xa438, 0x82b1 },	\
	{ 0xa438, 0xae49 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1c4d },	\
	{ 0xa438, 0xae41 },	\
	{ 0xa438, 0xa002 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1c90 },	\
	{ 0xa438, 0xae39 },	\
	{ 0xa438, 0xa003 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1c9d },	\
	{ 0xa438, 0xae31 },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1cbc },	\
	{ 0xa438, 0xae29 },	\
	{ 0xa438, 0xa005 },	\
	{ 0xa438, 0x1e02 },	\
	{ 0xa438, 0x1cc9 },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0xdfac },	\
	{ 0xa438, 0x2013 },	\
	{ 0xa438, 0xac21 },	\
	{ 0xa438, 0x10ac },	\
	{ 0xa438, 0x220d },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xe2bf },	\
	{ 0xa438, 0x6593 },	\
	{ 0xa438, 0x026b },	\
	{ 0xa438, 0x7eee },	\
	{ 0xa438, 0x8fe1 },	\
	{ 0xa438, 0x00ae },	\
	{ 0xa438, 0x08a0 },	\
	{ 0xa438, 0x0605 },	\
	{ 0xa438, 0x021d },	\
	{ 0xa438, 0x07ae },	\
	{ 0xa438, 0x00e0 },	\
	{ 0xa438, 0x82b1 },	\
	{ 0xa438, 0xaf1b },	\
	{ 0xa438, 0xe910 },	\
	{ 0xa438, 0xbf4a },	\
	{ 0xa438, 0x99bf },	\
	{ 0xa438, 0x4a00 },	\
	{ 0xa438, 0xa86a },	\
	{ 0xa438, 0xfdad },	\
	{ 0xa438, 0x5eca },	\
	{ 0xa438, 0xad5e },	\
	{ 0xa438, 0x88bd },	\
	{ 0xa438, 0x2c99 },	\
	{ 0xa438, 0xbd2c },	\
	{ 0xa438, 0x33bd },	\
	{ 0xa438, 0x3222 },	\
	{ 0xa438, 0xbd32 },	\
	{ 0xa438, 0x11bd },	\
	{ 0xa438, 0x3200 },	\
	{ 0xa438, 0xbd32 },	\
	{ 0xa438, 0x77bd },	\
	{ 0xa438, 0x3266 },	\
	{ 0xa438, 0xbd32 },	\
	{ 0xa438, 0x55bd },	\
	{ 0xa438, 0x3244 },	\
	{ 0xa438, 0xbd32 },	\
	{ 0xa436, 0xb818 },	\
	{ 0xa438, 0x15c5 },	\
	{ 0xa436, 0xb81a },	\
	{ 0xa438, 0x6255 },	\
	{ 0xa436, 0xb81c },	\
	{ 0xa438, 0x34e1 },	\
	{ 0xa436, 0xb81e },	\
	{ 0xa438, 0x1068 },	\
	{ 0xa436, 0xb850 },	\
	{ 0xa438, 0x07cc },	\
	{ 0xa436, 0xb852 },	\
	{ 0xa438, 0x26ca },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0x0dbf },	\
	{ 0xa436, 0xb884 },	\
	{ 0xa438, 0x1bb1 },	\
	{ 0xa436, 0xb832 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa436, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xb82e, 0x0000 },	\
	{ 0xa436, 0x8023 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x801e },	\
	{ 0xa438, 0x0031 },	\
	{ 0xb820, 0x0000 },	\
	{ 0xb892, 0x0000 },	\
	{ 0xb88e, 0xc28f },	\
	{ 0xb890, 0x252d },	\
	{ 0xb88e, 0xc290 },	\
	{ 0xb890, 0xc924 },	\
	{ 0xb88e, 0xc291 },	\
	{ 0xb890, 0xc92e },	\
	{ 0xb88e, 0xc292 },	\
	{ 0xb890, 0xf626 },	\
	{ 0xb88e, 0xc293 },	\
	{ 0xb890, 0xf630 },	\
	{ 0xb88e, 0xc294 },	\
	{ 0xb890, 0xa328 },	\
	{ 0xb88e, 0xc295 },	\
	{ 0xb890, 0xa332 },	\
	{ 0xb88e, 0xc296 },	\
	{ 0xb890, 0xd72b },	\
	{ 0xb88e, 0xc297 },	\
	{ 0xb890, 0xd735 },	\
	{ 0xb88e, 0xc298 },	\
	{ 0xb890, 0x8a2e },	\
	{ 0xb88e, 0xc299 },	\
	{ 0xb890, 0x8a38 },	\
	{ 0xb88e, 0xc29a },	\
	{ 0xb890, 0xbe32 },	\
	{ 0xb88e, 0xc29b },	\
	{ 0xb890, 0xbe3c },	\
	{ 0xb88e, 0xc29c },	\
	{ 0xb890, 0x7436 },	\
	{ 0xb88e, 0xc29d },	\
	{ 0xb890, 0x7440 },	\
	{ 0xb88e, 0xc29e },	\
	{ 0xb890, 0xad3b },	\
	{ 0xb88e, 0xc29f },	\
	{ 0xb890, 0xad45 },	\
	{ 0xb88e, 0xc2a0 },	\
	{ 0xb890, 0x6640 },	\
	{ 0xb88e, 0xc2a1 },	\
	{ 0xb890, 0x664a },	\
	{ 0xb88e, 0xc2a2 },	\
	{ 0xb890, 0xa646 },	\
	{ 0xb88e, 0xc2a3 },	\
	{ 0xb890, 0xa650 },	\
	{ 0xb88e, 0xc2a4 },	\
	{ 0xb890, 0x624c },	\
	{ 0xb88e, 0xc2a5 },	\
	{ 0xb890, 0x6256 },	\
	{ 0xb88e, 0xc2a6 },	\
	{ 0xb890, 0xa453 },	\
	{ 0xb88e, 0xc2a7 },	\
	{ 0xb890, 0xa45d },	\
	{ 0xb88e, 0xc2a8 },	\
	{ 0xb890, 0x665a },	\
	{ 0xb88e, 0xc2a9 },	\
	{ 0xb890, 0x6664 },	\
	{ 0xb88e, 0xc2aa },	\
	{ 0xb890, 0xac62 },	\
	{ 0xb88e, 0xc2ab },	\
	{ 0xb890, 0xac6c },	\
	{ 0xb88e, 0xc2ac },	\
	{ 0xb890, 0x746a },	\
	{ 0xb88e, 0xc2ad },	\
	{ 0xb890, 0x7474 },	\
	{ 0xb88e, 0xc2ae },	\
	{ 0xb890, 0xbcfa },	\
	{ 0xb88e, 0xc2af },	\
	{ 0xb890, 0xbcfd },	\
	{ 0xb88e, 0xc2b0 },	\
	{ 0xb890, 0x79ff },	\
	{ 0xb88e, 0xc2b1 },	\
	{ 0xb890, 0x7901 },	\
	{ 0xb88e, 0xc2b2 },	\
	{ 0xb890, 0xf703 },	\
	{ 0xb88e, 0xc2b3 },	\
	{ 0xb890, 0xf706 },	\
	{ 0xb88e, 0xc2b4 },	\
	{ 0xb890, 0x7408 },	\
	{ 0xb88e, 0xc2b5 },	\
	{ 0xb890, 0x740a },	\
	{ 0xb88e, 0xc2b6 },	\
	{ 0xb890, 0xf10c },	\
	{ 0xb88e, 0xc2b7 },	\
	{ 0xb890, 0xf10f },	\
	{ 0xb88e, 0xc2b8 },	\
	{ 0xb890, 0x6f10 },	\
	{ 0xb88e, 0xc2b9 },	\
	{ 0xb890, 0x6f13 },	\
	{ 0xb88e, 0xc2ba },	\
	{ 0xb890, 0xec15 },	\
	{ 0xb88e, 0xc2bb },	\
	{ 0xb890, 0xec18 },	\
	{ 0xb88e, 0xc2bc },	\
	{ 0xb890, 0x6a1a },	\
	{ 0xb88e, 0xc2bd },	\
	{ 0xb890, 0x6a1c },	\
	{ 0xb88e, 0xc2be },	\
	{ 0xb890, 0xe71e },	\
	{ 0xb88e, 0xc2bf },	\
	{ 0xb890, 0xe721 },	\
	{ 0xb88e, 0xc2c0 },	\
	{ 0xb890, 0x6424 },	\
	{ 0xb88e, 0xc2c1 },	\
	{ 0xb890, 0x6425 },	\
	{ 0xb88e, 0xc2c2 },	\
	{ 0xb890, 0xe228 },	\
	{ 0xb88e, 0xc2c3 },	\
	{ 0xb890, 0xe22a },	\
	{ 0xb88e, 0xc2c4 },	\
	{ 0xb890, 0x5f2b },	\
	{ 0xb88e, 0xc2c5 },	\
	{ 0xb890, 0x5f2e },	\
	{ 0xb88e, 0xc2c6 },	\
	{ 0xb890, 0xdc31 },	\
	{ 0xb88e, 0xc2c7 },	\
	{ 0xb890, 0xdc33 },	\
	{ 0xb88e, 0xc2c8 },	\
	{ 0xb890, 0x2035 },	\
	{ 0xb88e, 0xc2c9 },	\
	{ 0xb890, 0x2036 },	\
	{ 0xb88e, 0xc2ca },	\
	{ 0xb890, 0x9f3a },	\
	{ 0xb88e, 0xc2cb },	\
	{ 0xb890, 0x9f3a },	\
	{ 0xb88e, 0xc2cc },	\
	{ 0xb890, 0x4430 },	\
	{ 0xa436, 0xacca }, 	\
	{ 0xa438, 0x0104 }, 	\
	{ 0xa436, 0xaccc }, 	\
	{ 0xa438, 0x8000 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0fff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfd47 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0fff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xe56f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01c0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xed97 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01c8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xf5bf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01d0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb07 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb0f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01d8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa087 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0180 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa00f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0108 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa807 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0100 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa88f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0188 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb027 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0120 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb02f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0128 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb847 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0140 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb84f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0148 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb17 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb1f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa017 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0110 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa01f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0118 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa837 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0130 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa83f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0138 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb097 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0190 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb05f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0158 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb857 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0150 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb89f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0198 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb27 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb2f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x8087 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0180 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x800f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0108 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x8807 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0100 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x888f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0188 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x9027 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0120 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x902f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0128 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x9847 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0140 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x984f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0148 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa0a7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa8af }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa067 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0161 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xa86f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0169 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb37 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb3f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x8017 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0110 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x801f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0118 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x8837 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0130 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x883f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0138 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x9097 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0190 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x905f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0158 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x9857 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0150 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x989f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0198 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb0b7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb8bf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb077 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0171 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xb87f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0179 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb47 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb4f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6087 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0180 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x600f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0108 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6807 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0100 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x688f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0188 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7027 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0120 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x702f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0128 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7847 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0140 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x784f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0148 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x80a7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x88af }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x8067 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0161 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x886f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0169 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb57 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb5f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6017 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0110 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x601f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0118 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6837 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0130 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x683f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0138 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7097 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0190 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x705f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0158 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7857 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0150 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x789f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0198 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x90b7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x98bf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x9077 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0171 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x987f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0179 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb67 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb6f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x4087 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0180 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x400f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0108 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x4807 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0100 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x488f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0188 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x5027 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0120 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x502f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0128 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x5847 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0140 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x584f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0148 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x60a7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x68af }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6067 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0161 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x686f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0169 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb77 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb7f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x4017 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0110 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x401f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0118 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x4837 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0130 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x483f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0138 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x5097 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0190 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x505f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0158 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x5857 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0150 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x589f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0198 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x70b7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x78bf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7077 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0171 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x787f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0179 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb87 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb8f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x40a7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x48af }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x4067 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0161 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x486f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0169 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb97 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb9f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x50b7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x58bf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x5077 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0171 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x587f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0179 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfba7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfbaf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x2067 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0161 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x286f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0169 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfbb7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfbbf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x3077 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0171 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x387f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0179 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfff9 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x17ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfff9 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x17ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0fff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfff8 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0fff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xffff }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb47 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb4f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6087 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0180 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x600f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0108 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6807 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0100 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x688f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0188 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7027 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0120 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x702f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0128 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7847 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0140 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x784f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0148 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x80a7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x88af }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01a8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x8067 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0161 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x886f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0169 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb57 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0xfb5f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x07ff }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6017 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0110 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x601f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0118 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x6837 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0130 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x683f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0138 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7097 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0190 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x705f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0158 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x7857 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0150 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x789f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x0198 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x90b7 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b0 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x98bf }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x01b8 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x9077 }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x1171 }, 	\
	{ 0xa436, 0xacce }, 	\
	{ 0xa438, 0x987f }, 	\
	{ 0xa436, 0xacd0 }, 	\
	{ 0xa438, 0x1179 }, 	\
	{ 0xa436, 0xacca }, 	\
	{ 0xa438, 0x0004 }, 	\
	{ 0xa436, 0xacc6 }, 	\
	{ 0xa438, 0x0008 }, 	\
	{ 0xa436, 0xacc8 }, 	\
	{ 0xa438, 0xc000 }, 	\
	{ 0xa436, 0xacc6 }, 	\
	{ 0xa438, 0x0015 }, 	\
	{ 0xa436, 0xacc8 }, 	\
	{ 0xa438, 0xc043 }, 	\
	{ 0xa436, 0xacc8 }, 	\
	{ 0xa438, 0x0000 }, 	\
	{ 0xb820, 0x0000 }
