#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Various constants
=============================================
|#

#|
=============================================
Unicode characters
=============================================
Originally defined in frontend.lisp, hence the "f-"

f-space is the space character.

f-newline is the newline character. Logiweb uses Unicode 10 (newline) to separate lines of code. This is hardwired into Logiweb and is important to keep that way to ensure interoperability of Logiweb systems on different host operating systems.

When running Logiweb on a system that uses e.g. Unicode 13 (Return) or a CRLF for separating lines of text, one must convert whatever newline sequence is used in the host operating system to Unicode 10 when e.g. reading Logiweb source and converting back when e.g. printing error messages.

Logiweb accepts text to contains Unicode character 10 (newline) and from character 32 and up.

Character code 0-9 and 11-31 are 'blind'. When Logiweb generates text output (e.g. TeX sources), the text is a list of cardinals that can contain arbitrary codes. But all characters with codes from 0-9 and 11-31 are removed before output.

f-tab is the tabulation character. It has code 9, so it is blind and cannot occur in text output from Logiweb. In input it should be converted to code 32 (space). At present, the Logiweb compiler treats a tab as a space when it occurs outside strings and signals a fatal error if a tab occurs in a string.

On a system that uses Unicode 13 (Return) for separating lines of text, the correct way to treat text would be as follows. On input, discard codes 0-8, 10-12, and 14-31. Convert code 13 to code 10 and convert code 9 to code 32. On output, convert code 10 to code 13 (after Logiweb has removed blind characters).
|#

(defc f-space (char-code #\space))

(defc f-newline (char-code #\Newline)) ; Do not change, see above for porting to operating systems that do not use Unicode 10 for separating lines of text.

(defc f-tab (char2card #\tab))

(etst f-space 32)
(etst f-newline 10)
(etst f-tab 9)

#|
=============================================
Predefined concepts
=============================================

Proclaimable computable constructs
----------------------------------
lambda: lambda abstraction.
apply: functional application.
true: the (unique) atom of map theory.
if: if-then-else construct.
quote: quote construct.

Revelation constructs
---------------------
proclaim: construct for connecting syntax with predefined concepts
define: construct for defining any aspect of any symbol.
introduce: formally the same as define, but sends a hint to the compiler.
hide: construct for hiding all subtrees from harvesting
aspect: construct for connecting syntax with predefined aspect

Symbol aspects.
---------------
name: The name of a symbol in the Logiweb source language.
use: Code for typesetting the symbol.
show: Code for typesetting the symbol e.g. in the lhs of a definition.
value: The mathematical value of the symbol.
message: The aspect denoted by a symbol.
macro: Code for macro expanding the symbol.
definition: The kind of definition denoted by a symbol.
priority: For page symbol: The priority table of a page. Otherwise see below.

Priority values.
----------------
pre: Symbols with "pre" as priority aspect denote preassociativity.
post: Same for postassociativity.

Page symbol aspects.
--------------------
unpack: Function for doing user defined unpacking of a vector.
claim: Function for user defined verification of a page.
execute: Thing to be executed when 'executing' a page.
executables: Association list from names to executables.

Hooks.
------
bibliography: The bibliography of a page.
dictionary: An association list of arities of all non-nulary symbols.
body: The body of a page.
cache: The cache of all referenced pages.
vector: The bytes of the page.
diagnose: When non-true, indicates that the page is in error.
codex: The collection of all revelations harvested from a page.
code: Compiled version of value definitions.
expansion: The macro expanded version of the body of the page.

Note that the value on the diagnose hook has been maptagged since logiweb-0.1.10 to allow/simplify the compilation process in cases where the verifier of a page accesses the diagnose of the page itself. Before logiweb-0.1.10, the verifier would simply get T (true) which is wrong for faulty pages.

In addition to the predefined concepts that have names, Logiweb recognises all symbols with id=0 as page symbols and recognizes all symbols with id=1 from pages with empty bibliographies as proclamation symbols. These two 'predefined concepts' are nameless.
|#

(defc card-apply	(string2card "apply"))
(defc card-lambda	(string2card "lambda"))
(defc card-true		(string2card "true"))
(defc card-if		(string2card "if"))
(defc card-quote	(string2card "quote"))

(defc card-proclaim	(string2card "proclaim"))
(defc card-define	(string2card "define"))
(defc card-introduce	(string2card "introduce"))
(defc card-hide		(string2card "hide"))

(defc card-name		(string2card "name"))
(defc card-tex		(string2card "use"))
(defc card-texname	(string2card "show"))
(defc card-value	(string2card "value"))
(defc card-message	(string2card "message"))
(defc card-macro	(string2card "macro"))
(defc card-definition	(string2card "definition"))
(defc card-priority	(string2card "priority"))

(defc card-pre		(string2card "pre"))
(defc card-post		(string2card "post"))

(defc card-unpack	(string2card "unpack"))
(defc card-render	(string2card "render"))
(defc card-claim	(string2card "claim"))
(defc card-execute	(string2card "execute"))
(defc card-executables	(string2card "executables"))

(defc card-bibliography	(string2card "bibliography"))
(defc card-dictionary	(string2card "dictionary"))
(defc card-body		(string2card "body"))
(defc card-cache	(string2card "cluster"))
(defc card-vector	(string2card "vector"))
(defc card-diagnose	(string2card "diagnose"))
(defc card-codex        (string2card "codex"))
(defc card-code         (string2card "code"))
(defc card-expansion    (string2card "expansion"))

#|
=============================================
Return values from spy functions
=============================================
|#

(defc card-spy          (string2card "spy"))
(defc card-trace        (string2card "trace"))
(defc card-print        (string2card "print"))
(defc card-timer        (string2card "timer"))

#|
=============================================
States
=============================================

A state is a trie that maps state ids to values. At the time of writing, a state only has *one* entry. That entry is indexed by id-cache, which equals zero.

At an earlier stage of the development, the state also contained a structure which allowed to translate parse tree adresses back to source code addresses. That structure took up a lot of space and was never used. It turned out that it was easy for a user to locate those errors where the structure could have helped.

The global cache and state are internal to the Logiweb compiler and are inaccessible to user programs. (aget (id-cache) ref) is the cache of a page and contains everything accessible from programs defined on the given page.

The global state is kept in *state* which is declared at the end of the present file.

=============================================
State accessors
=============================================
(codex-get state sref sid aref aid) returns the definition of the given aspect of the given symbol as defined on the home page of the symbol.

(codex-put ...) is update function associated to codex-get

(rack-get state page hook) returns the data attached to the given hook of the rack of the given page.

(rack-put ...) is the update function associated to rack-get.
|#

(defmacro id-cache () 0)

(deff rack-get (state page hook)
 (aget state (id-cache) page page hook))

(deff rack-put (state newval page hook)
 (aput state newval (id-cache) page page hook))

(deff codex-get (state sref sid aref aid)
 (aget state (id-cache) sref sref card-codex sref sid aref aid))

(deff codex-put (state newval sref sid aref aid)
 (aput state newval (id-cache) sref sref card-codex sref sid aref aid))

(deff arity-get (state ref id)
 (aget state (id-cache) ref ref card-dictionary id))

(deff arity-put (state newval ref id)
 (aput state newval (id-cache) ref ref card-dictionary id))

#|
=============================================
State memorization
=============================================
|#

; Set *memorize* to t when once loaded pages should be kept
; (good when the software is stable). Set it to nil
; during development, e.g. when changing the format for
; Logiweb pages. And don't ever change that format.
(defc *memorize* t)

(defvar *state* nil)

