#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
Logiweb
=============================================
Facilities for loading and testing
=============================================
(test x) test that x is non-nil
(ttst x) test that x equals t
(ntst x) test that x equals nil
(etst x y) test that x and y are equalp
(xtst x) test that x throws to tag :exception



(testload filename &key test compile interpret prompt)

Load given file. The filename must be without extension.

If 'test' is true (the default) then execute all test constructs in the file.

By default, filename.lisp is compiled if there is no filename.fas of if filename.fas is older than filename.lisp.

If 'compile' is true: force recompilation.

If 'interpret' is true: suppress recompilation ('compile' takes precedence over 'interpret')

If 'prompt' is true (the default) then stop after each compilation has ended and wait for a newline from the user.



~/.clisprc.lisp could look something like this:

(in-package "COMMON-LISP-USER")
(load "test")
(defvar init-file-name (prompt-file))
<user specific initialisations>
(when init-file-name (testload init-file-name))
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Control variables
=============================================
|#

(defvar *testp* t)
(defvar *compilep* nil)
(defvar *interpretp* nil)
(defvar *promptp* t)
(defvar *stack*)
(defvar *ok* t)
(defvar *abort* nil)
(defvar *lastfile* nil)
(defvar *toplevel*)
(setq *toplevel* t)

#|
=============================================
Test constructs
=============================================
|#

(defun test-indent (x)
 (when x (princ ";") (test-indent (cdr x))))

(defmacro test (x)
`(if *toplevel* (error "Use testload to load file")
  (when (and *ok* *testp*)
   (incf (car *stack*))
   (test-indent *stack*)
   (format t "; ~d ~s~%" (car *stack*) ',x)
   (unless ,x
    (setq *ok* nil)
    (test-indent *stack*)
    (format t "; *** Test failed~%")))))

(defmacro etst (x y)
`(test (equalp ,x ,y)))

(defmacro ttst (x)
`(test (equalp ,x t)))

(defmacro ntst (x)
`(test (equalp ,x nil)))

(defmacro xtst (x)
 `(test (catch :exception ,x nil)))

(defmacro rtst (&rest x)
 `(test (progn ,@x t)))

#|
=============================================
Load constructs
=============================================
|#

(defun testload
 (filename
  &key
  (test t testp)
  (compile nil compilep)
  (interpret nil interpretp)
  (prompt t promptp))
 (when (pathname-type (pathname filename))
  (error "(testload ~a): file name must be given without extension"))
 (when (or *toplevel* testp) (setq *testp* test))
 (when (or *toplevel* compilep) (setq *compilep* compile))
 (when (or *toplevel* interpretp) (setq *interpretp* interpret))
 (when (or *toplevel* promptp) (setq *promptp* prompt))
 (when *toplevel*
  (setq *stack* nil)
  (setq *ok* t)
  (setq *abort* nil)
  (setq *lastfile* filename))
 (when (null filename)
  (format t "Specify file name~%")
  (return-from testload (values)))
 (if (null *toplevel*) (subtestload filename)
  (unwind-protect
   (progn (setq *toplevel* nil) (subtestload filename))
   (setq *toplevel* t)))
 (when *testp*
  (if *ok*
   (format t "Test suite succeeded~%")
   (when (not *abort*) (setq *abort* t) (format t "Test suite failed~%"))))
 (values))

(defun reload (&optional filename)
 (testload (if filename filename *lastfile*)))

(defun recompile (&optional filename)
 (testload filename :compile t))

(defun reinterpret (&optional filename)
 (testload filename :interpret t))

(defun up-to-date (filename)
 (let (
   (source (concatenate 'string filename ".lisp"))
   (target (concatenate 'string filename #+:clisp ".fas" #+:cmu ".x86f")))
  (and
   (probe-file source)
   (probe-file target)
   (< (file-write-date source) (file-write-date target)))))

(defun subtestload (filename)
 (when *ok*
  (push 0 *stack*)
  (when (or *compilep* (and (not *interpretp*) (not (up-to-date filename))))
   (compile-file filename)
   (if (null *promptp*) (fresh-line)
    (unless (equalp (read-line) "")
     (setq *ok* nil)
     (setq *abort* t)
     (format t "Aborted~%")
     (return-from subtestload))))
  (load filename)
  (pop *stack*)
  (when *testp*
   (test-indent *stack*)
   (format t "; end ~a~%" filename))))

(defun prompt-file ()
 #+:clisp (terpri)
 (format t "File to testload~%")
 (format t "empty string = 'source'~%")
 (format t "nil = no file~%")
 (format t "File name: ")
 (let* ((init-file-name (read-line)))
  (cond
   ((equal init-file-name "nil") nil)
   ((equal init-file-name "") "source")
   (t init-file-name))))

#|
=============================================
Temporary cd
=============================================
(in-directory dir &rest forms) cd's to the given directory, executes the
given forms, and cd's back.

indir is like in-directory, just shorter to type, less self-explanatory, and intended for debugging.
|#

(defmacro in-directory (dir &rest forms)
`(let ((pwd (cd)))
  (unwind-protect (progn (cd ,dir) ,@forms) (cd pwd))))

(defmacro indir (dir &rest forms)
`(let ((pwd (cd)))
  (unwind-protect (progn (cd ,dir) ,@forms) (cd pwd))))

