#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Main function
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Print usage
=============================================
|#

(deff print-usage ()
 (format t "
Usage: See 'man 1 lgc'

"))

#|
=============================================
Print version
=============================================
|#

(deff print-version ()
 (format t "~%")
 (format t "Logiweb compiler version ~a~%" *logiweb-version*)
 (format t "Copyright (C) 2004-2010 Klaus Grue~%")
 (format t "Available under GNU GPL~%")
 (format t "For more information see http://logiweb.eu/~%")
 (format t "Logiweb comes with ABSOLUTELY NO WARRANTY~%")
 (format t "~%"))

#|
=============================================
Check file name
=============================================
Some time in the past, Logiweb restricted file names to consist of small
letters, digits, hyphens (-), underscores (_), and plus characters (+).

All these characters are "safe" to use in the sense of RFC 1738 - Uniform
Resource Locators (URL). Furthermore, the characters are safe to use in
file names in all operating systems / system shells I recall.

The restriction has been removed, though, and it is now up to the user to be
sensible. But the present remark is kept just in case the restriction has to
be reintroduced.
|#

#|
=============================================
Main: Run frontend
=============================================
(main-lgs state) translates a Logiweb source file to a Logiweb page, loads it, and returns its reference and the new state as the list (ref state). Returns nil on error.

(main-lgw state) fetches a Logiweb page via www, loads it, and returns its reference and the new state as the list (ref state). Returns nil on error.

(main-frontend state) calls main-lgw if 'lgw=...' is given and main-lgs otherwise.
|#

(deff main-frontend (state)
 (:when (unequal (option "lgw") "") (main-lgw state))
 (:when (unequal (option "post") "") (main-post state))
 (main-lgs state))

(deff main-lgs (state)
 (:let lgs (option "lgs"))
 (:when (equalp lgs "") (format t "Specify 'lgs=' or 'lgw='~%") nil)
 (:let lgs (add-extension lgs ".lgs"))
 (f-preprocess lgs)
 (frontend state))

(deff main-post (state)
 (:let file (string2vector (option "pane")))
 (set-source file :filename "Form input" :silent t)
 (setq *sourceref* :none)
 (frontend state))

(deff set-lgw-source ()
 (:let src (option "src"))
 (:when (equalp src "") (setq *source* nil))
 (:let src (if (equalp src "yes") (remove-extension (option "lgw") ".lgw") src))
 (:let src (add-extension src ".lgs"))
 (:let src (string2card* src))
 (:let source (c-url2vector src))
 (:let source (if (equalp source :does-not-exist) nil source))
 (setq *source* source))

(deff main-lgw (state)
 (:let lgw (add-extension (option "lgw") ".lgw"))
 (:let lgw (string2card* lgw))
 (:let ref (c-url2ref lgw))
 (set-lgw-source)
 (:let state (c-load-ref ref state))
 (:when (main-canned) (raise))
 (list ref state))

#|
=============================================
Main: Print main result
=============================================
|#

(deff print-validity (ref state)
 (:when (not (option "test")) nil)
 (:when (untag (aget state (id-cache) ref ref card-diagnose))
  (format t "Claim failed~%"))
 (:when (c-get-page-aspect ref card-claim (aget state (id-cache) ref))
  (format t "Claim succeeded~%"))
 (format t "No claim found~%"))

#|
=============================================
Main: Expand $lgs, $page etc
=============================================
(main-subst string ref name lgt) expands the given string thus:
$lgs  -> the name of the source file
$page -> the name of the page
$date -> the timestamp of the pge
$16   -> the reference base 16
$32   -> the reference base 32
$64   -> the reference base 64
|#

(deff main-subst (string ref name lgt)
 (:let lgs (pathname-name (option "lgs")))
;(:let page (substitute #\- #\Space name))
 (:let page name)
 (:let date (lgt-2-filename lgt))
 (:let r16 (b-ref-2-base16 ref))
 (:let r32 (b-ref-2-base32 ref))
 (:let r64 (b-ref-2-base64 ref))
 (:let assoc
 `(("lgs" . ,lgs) ("page" . ,page) ("date" . ,date)
   ("16" . ,r16) ("32" . ,r32) ("64" . ,r64)))
 (main-subst0 string assoc))

(deff main-subst0 (string assoc)
 (ct2string (main-subst1 string assoc)))

(deff main-subst1 (string assoc)
 (:let pos (position #\$ string))
 (:when (null pos) string)
 (:let string1 (subseq string 0 pos))
 (:let string (subseq string (+ pos 1)))
 (:let pos (position #\$ string))
 (:when (null pos) (list string1 string))
 (:let string2 (subseq string 0 pos))
 (:let string2 (default string2 (assoc-get string2 assoc)))
 (:let string (subseq string (+ pos 1)))
 (:let string (main-subst1 string assoc))
 (list string1 string2 string))

(etst (main-subst0 "abc" '(("def" . "ghi"))) "abc")
(etst (main-subst0 "abc$def$xyz" '(("def" . "ghi"))) "abcghixyz")
(etst (main-subst0 "abc$def$xyz" '(("DEF" . "ghi"))) "abcghixyz")
(etst (main-subst0 "abc$uvw$xyz" '(("def" . "ghi"))) "abcuvwxyz")
(etst (main-subst0 "abc$def" '(("def" . "ghi"))) "abcdef")

#|
=============================================
Main: Translate string to safe file name
=============================================
|#

(deff main-url-encode (string)
 (:when (equalp string "") "NONE")
 (ct2string (map 'list 'main-url-encode1 string)))

(deff main-url-encode1 (char)
 (:when (alphanumericp char) char)
 (:when (member char '(#\. #\- #\_) :test 'equalp) char)
 (:when (equalp char #\Space) #\+)
 (format nil "%~2,'0x" (char-code char)))

(ttst (equal (main-url-encode "") "NONE"))
(ttst (equal (main-url-encode "aB-c") "aB-c"))
(ttst (equal (main-url-encode "a!b") "a%21b"))

#|
=============================================
Main: Determine directory for rendering
=============================================
Pages are rendered at (option "url")/dir1/dir0 where (dir1 . dir0) is the return value from main-subdir. The destination is based on the dest, destsubmit, destresubmit, and desttemp options. Variables like $lgs$ are expanded.
|#

(deff main-get-dest ()
 (:let dest (option "dest"))
 (:when (unequal dest "") dest)
 (:when (unequal (option "lgw") "") (option "destresubmit"))
 (:when (level '= "submit") (option "destsubmit"))
 (option "desttemp"))

(deff main-subdir (ref name lgt)
 (main-subdir1 ref name lgt (main-get-dest)))

(deff main-subdir1 (ref name lgt dest)
 (:let dest (main-subst dest ref name lgt))
 (:let pos (position #\/ dest))
 (:when (null pos) (complain "Missing slash in destination ~s" dest))
 (:let dir1 (subseq dest 0 pos))
 (:let dir0 (subseq dest (+ pos 1)))
 (:let dir1 (main-url-encode dir1))
 (:let dir0 (main-url-encode dir0))
 (cons dir1 dir0))

(deff main-mirror-dir (ref name lgt)
 (main-subdir1 ref name lgt (option "destmirror")))

#|
=============================================
Main: Determine path of link
=============================================
Pages are rendered at (option "url")/dir1/dir0 where (dir1 . dir0) is the return value from main-subdir. The destination is based on the dest, destsubmit, destresubmit, and desttemp options. Variables like $lgs$ are expanded.
|#

(deff main-get-link ()
 (:let link (option "link"))
 (:when (unequal link "") link)
 (:when (unequal (option "lgw") "") (option "linkresubmit"))
 (:when (level '= "submit") (option "linksubmit"))
 (option "linktemp"))

(deff make-link (target ref name lgt)
 (:let link (main-get-link))
 (:when (equalp link "") nil)
 (:let link (main-subst link ref name lgt))
 (:let pos (position #\/ link :from-end t))
 (:when (null pos) (complain "Missing slash in link ~s" link))
 (:let pos (+ pos 1))
 (:let dir (subseq link 0 pos))
 (:let name (subseq link pos))
 (:when (null (probe-directory dir)) nil)
 (:let name (main-url-encode name))
 (exec-program dir "rm" "-f" name)
 (exec-program dir "ln" "-s" (ct2string target) name)
 (html-dir lgt dir))

#|
=============================================
Main: Run backend
=============================================
|#

(deff relink-latest (dir target)
 (exec-program dir "rm" "-f" "latest")
 (exec-program dir "ln" "-s" (ct2string target) "latest"))

(deff main-wrong-url ()
 (format t "Could not translate url ~s to a directory~%" (option "url"))
 (format t "Known url/directory associations:~%")
 (dolist (x (option "roots")) (format t "URL ~a -> DIR ~a~%" (cdr x) (car x))))

(deff main-submit-notify (ref dir1 dir0)
 (submit-notify ref (list (option "url") dir1 "/" dir0 "/vector/page.lgw")))

(deff print-page-url (page-url)
 (:when (equalp (option "pane") "")
  (format t "Rendering page at ~a~%" page-url))
 (format t "</pre>~%")
 (format t "<p>~%")
 (format t "Rendering page <a href=~s>here</a>" page-url)
 (format t "</p>~%")
 (format t "<pre>~%"))

(deff main-backend (ref state)
 (:when (level '<= "compile") nil)
 (setq *has-warned-about-value* nil)
 (progress "Backend")
 (:let lgt (ref2timestamp ref))
 (:let name (c-symbol2name ref 0 state))
 (:let path2 (url2filename (option "url"))) ; two levels above page root dir
 (:when (null path2) (main-wrong-url))
 (:let (dir1 . dir0) (main-subdir ref name lgt))
 (:let url (ct2string (list (option "url") dir1 "/" dir0 "/")))
 (print-page-url url)
 (setq *hyperbaseurl* (if (level '>= "submit") (url-abs-redirect ref "") url))
 (:let path1 (cat path2 dir1))
 (:let path0 (cat path1 "/" dir0))
 (b-dir path0 lgt name ref state *source* :nomirror)
 (progress "Backend: writing lgwdir")
 (when (level '>= "submit") (main-submit-notify ref dir1 dir0))
 (when (level '>= "submit") (relink-latest path1 dir0))
 (make-link path0 ref name lgt)
 (html-dir lgt path0)
 (html-dir lgt path1)
 (html-dir lgt path2)
 (when (and (unequal (option "post") "") (equalp (option "lgw") ""))
  (html-dir lgt (path2container path2))
  (html-dir lgt (path2container (path2container path2))))
 (progress "Backend: finishing"))

#|
=============================================
Main: Run backend on mirrored pages
=============================================
|#

(deff main-mirror (ref state)
 (:when (not (option "mirror")) nil)
 (progress "Mirror")
 (:let (:ref . ref*) (rack-get state ref card-bibliography))
 (main-mirror* ref* state))

(deff main-mirror* (ref* state)
 (dolist (ref ref*) (main-mirror1 ref state)))

(deff get-source (ref)
 (:when (unequal (option "src") "yes") nil)
 (:let lgw (ct2string (c-ref2url ref)))
 (:let src (remove-extension lgw ".lgw"))
 (:let src (add-extension src ".lgs"))
 (:let src (string2card* src))
 (c-url2vector src))

(deff main-mirror1 (ref state)
 (:let lgt (ref2timestamp ref))
 (:let name (c-symbol2name ref 0 state))
 (:let path2 (url2filename (option "url"))) ; two levels above page root dir
 (:when (null path2) (main-wrong-url))
 (:let (dir1 . dir0) (main-mirror-dir ref name lgt))
 (setq *hyperbaseurl* (url-abs-redirect ref ""))
 (:let path1 (cat path2 dir1))
 (:let path0 (cat path1 "/" dir0))
 (:let lgwpath (cat path0 "/vector/page.lgw"))
 (when (verbose '> 1) (format t "Mirror: Probing ~s~%" lgwpath))
 (:when (probe-file lgwpath)
  (when (verbose '> 1) (format t "Mirror: Already mirrored ~%")))
 (format t "Mirror: Mirroring ~s:~%" name)
 (when (verbose '> 1)
  (format t "Mirror: Ref:~%")
  (format t "~a~%" (ct2string (ref2kana ref))))
 (when (verbose '= 1)
  (format t "Mirror: Ref: ~a ...~%" (ct2string (ref2short-kana 5 ref))))
 (when (verbose '> 0) (format t "Mirror: Path: ~s~%" lgwpath))
 (:let source (get-source ref))
 (b-dir path0 lgt name ref state source :mirror)
 (progress "Mirror: writing lgwdir")
 (main-submit-notify ref dir1 dir0)
 (relink-latest path1 dir0)
 (html-dir lgt path0)
 (html-dir lgt path1)
 (html-dir lgt path2)
 (format t "Mirror: Done mirroring page ~s~%" name))

#|
=============================================
Logiweb garbage collector
=============================================
|#

#|
---------------------------------------------
Parse gcroots option
---------------------------------------------
(gc-complain gcroot) complains about one particular item of the gcroots option.
(parse-gcroot gcroot) parses one item of the gcroots option.
(parse-gcroots gcroots) parses all items of the gcroots option.
|#

(deff gc-complain (gcroot)
 (complain "Invalid element in gcroots option: ~s" gcroot))

(deff parse-gcroot (gcroot)
 (:let gcroot (string-trim " " gcroot))
 (:when (equalp (length gcroot) 0) (gc-complain gcroot))
 (:unless (position (aref gcroot 0) "+-") (gc-complain gcroot))
 (:let sign (aref gcroot 0))
 (:let dir (slash (string-trim " " (subseq gcroot 1))))
 (list sign dir))

(deff parse-gcroots (gcroots)
 (map 'list 'parse-gcroot gcroots))

#|
---------------------------------------------
Translate gcroots to a list of directories
---------------------------------------------
(main-gc-collect-roots pub) returns the list of directories which serves as roots for the garbage collection process. It calls parse-gcroots to process the gcroots option and then performs a number of directory, set-difference, and union operations to construct the list.
|#

(deff main-gc-collect-roots (pub)
 (:let gcroots (parse-gcroots (option "gcroots")))
 (when (verbose '> 0)
  (format t "Patterns:~%")
  (dolist (gcroot gcroots) (format t " ~s~%" gcroot)))
 (in-directory pub (main-gc-collect-roots1 gcroots)))

(deff main-gc-collect-roots1 (gcroots)
 (:when (atom gcroots) nil)
 (:let ((sign path) . gcroots) gcroots)
 (:let dir1 (main-gc-collect-roots1 gcroots))
 (:let dir2 (directory path))
 (:when (equalp sign #\+) (union dir2 dir1 :test 'equal))
 (set-difference dir1 dir2 :test 'equal))

#|
---------------------------------------------
Go up one directory level
---------------------------------------------
(main-path-chop path) removes a trailing slash (if any) and backs up the the previous slash.
|#

(deff main-path-chop (path)
 (:let length (length path))
 (:when (equalp length 0) nil)
 (:let position (position #\/ path :end (- length 1) :from-end t))
 (:when (null position) nil)
 (subseq path 0 (+ position 1)))

(etst (main-path-chop "ab/cd") "ab/")
(etst (main-path-chop "ab/cd/") "ab/")
(etst (main-path-chop "") nil)
(etst (main-path-chop "ab") nil)
(etst (main-path-chop "ab/") nil)

#|
---------------------------------------------
Mark paths for keeping
---------------------------------------------
The *main-gc-hash* hash table allows to put one of the marks :sweep, :sweep1, :keep, og :keep1 on each visited path. The meaning is:

A value of :keep means that the given directory or file has to be kept (i.e. not garbage collected).

A value of :keep1 means that the file system under the given directory contains at least one file or directory to be kept and, hence, effectively, the given directory has to be kept as well.

A value of :sweep1 means that the directory or file has been visited and is in danger of being swept.

A value of :sweep means that the given directory or file has to be swept, i.e. to be garbage collected.

(main-keep0 path pub) puts a :keep mark at the given path and a :keep1 mark at all directories above the given path and up to the given publication directory. The function only puts :keep1 marks on directories which are previously marked as :sweep1 so that at :keep1 cannot overwrite a :keep mark.
|#

(defc *main-gc-hash* (make-hash-table :test 'equal))

(deff main-keep0 (path pub)
 (setf (gethash path *main-gc-hash*) :keep)
 (main-keep1 (main-path-chop path) pub))

(deff main-keep1 (path pub)
 (:when (null path) nil)
 (:when (unequal (gethash path *main-gc-hash*) :sweep1) nil)
 (setf (gethash path *main-gc-hash*) :keep1)
 (:when (equalp path pub) nil)
 (main-keep1 (main-path-chop path) pub))

#|
---------------------------------------------
Scan file system
---------------------------------------------
Scan file system under given path and return association list which maps paths to bibliographies. As a side effect set *main-gc-hash* such that it maps path names to one of :keep, :keep1, or :sweep1.

A value of :keep means that the given directory or file has to be kept (i.e. not garbage collected) because it is too young to be swept. This is controlled by the gcpatience option.

A value of :keep1 means that the file system under the given directory contains at least one file or directory to be kept and, hence, effectively, the given directory has to be kept as well.

A value of :sweep1 means that the directory or file has been processed. Later processes may change :sweep1 to :sweep indicating that the given directory has to be swept (i.e. removed by the garbage collector).

(main-collect-path-bib path nil) scans the file system under the given path.

(main-collect-path-bib* path* nil) scans the file systems under the given paths.

(main-collect-path-bib1 dir result) adds information to the given result if the given dir includes a file named vector/page.lgw.

(main-collect-path-bib2 path time pub) adds a :keep mark on the given path if it is younger than the given time. Directories above the given path up to the given publication directory get a :keep1 mark.
|#

(deff main-collect-path-bib (path time pub result)
 (:when (gethash path *main-gc-hash*) result)
 (setf (gethash path *main-gc-hash*) :sweep1)
 (:let result (main-collect-path-bib1 path result))
 (dolist (file (directory (cat path "*")))
  (main-collect-path-bib2 file time pub))
 (main-collect-path-bib* (directory (cat path "*/")) time pub result))

(deff main-collect-path-bib* (path* time pub result)
 (:when (atom path*) result)
 (:let (path . path*) path*)
 (:let path (namestring path))
 (:let result (main-collect-path-bib path time pub result))
 (main-collect-path-bib* path* time pub result))

(deff main-collect-path-bib1 (dir result)
 (:let path* (directory (cat dir "vector/page.lgw")))
 (:when (null path*) result)
 (:let (path) path*)
 (:let vector (path2bib (namestring path)))
 (:unless (arrayp vector) result)
 (:let (bib . :index) (c-parse-bib vector 0))
 (acons dir bib result))

(deff main-collect-path-bib2 (path time pub)
 (:let path (namestring path))
 (:when (< (file-write-date path) time) nil)
 (main-keep0 path pub))

#|
---------------------------------------------
Translate outcome of main-collect-path-bib
---------------------------------------------
(main-ref-bib-path-array assoc nil) translates the given assoc, which is supposed to be the output from main-collect-path-bib, to an array which maps references to pair (bib . path*) where bib is the Logiweb bibliography of the reference and path* is the list of directories which contain pages with the given reference.
|#

(deff main-ref-bib-path-array (assoc result)
 (:when (atom assoc) result)
 (:let ((path ref . bib) . assoc) assoc)
 (:let (:bib . path*) (array-get ref result))
 (:let path* (cons path path*))
 (:let item (cons bib path*))
 (:let result (array-put ref item result))
 (main-ref-bib-path-array assoc result))

#|
---------------------------------------------
Collect transitively referenced pages
---------------------------------------------
(main-keep array ref* pub result) puts a :keep mark on all pages transitively references pages starting with ref*. The given array is supposed to be the output from main-ref-bib-path-array. Returns an array which maps all references to be kept to t.
|#

(deff main-keep (array ref* pub result)
 (:when (atom ref*) result)
 (:let (ref . ref*) ref*)
 (:let (bib . path*) (array-get ref array))
 (:when (array-get ref result) result)
 (:let result (array-put ref t result))
 (dolist (path path*) (main-keep0 path pub))
 (:let result (main-keep array bib pub result))
 (main-keep array ref* pub result))

#|
---------------------------------------------
Convert paths into relative paths
---------------------------------------------
(main-present-path* (kind path* pub) selects all paths of the given kind and translate them to paths relative to the given publication directory for pretty printing.
|#

(deff main-present-path* (kind path* pub)
 (main-present-path*1 kind path* pub nil))

(deff main-present-path*1 (kind path* pub result)
 (:when (atom path*) (reverse result))
 (:let ((path . value) . path*) path*)
 (:when (unequal kind value) (main-present-path*1 kind path* pub result))
 (:let path (enough-namestring path pub))
 (:let result (cons path result))
 (main-present-path*1 kind path* pub result))

#|
---------------------------------------------
Locate directories to be swept
---------------------------------------------
(main-locate-sweep) puts a :sweep mark on directories that are :sweep1 and whose parent is :keep1. This protects all directories under :keep directories from being sweeped.
|#

(deff main-locate-sweep ()
 (maphash #'main-local-sweep1 *main-gc-hash*))

(deff main-local-sweep1 (path mark)
 (:when (unequal mark :sweep1) nil)
 (:when (unequal (gethash (main-path-chop path) *main-gc-hash*) :keep1) nil)
 (setf (gethash path *main-gc-hash*) :sweep))

#|
---------------------------------------------
Garbage collection step
---------------------------------------------
(main-gc-step item) asks for user confirmation for removing the given directory if it is marked as :sweep.
|#

(deff main-gc-step (item)
 (:let (path . mark) item)
 (:when (unequal mark :sweep) nil)
 (format t "~%Remove ~a~%" path)
 (:when (skipped-make) nil)
 (run-rm "-r" path)
 (html-dir (lgt) (main-path-chop path))
 (format t "~%done~%" path))

#|
---------------------------------------------
Main Logiweb garbage collection function
---------------------------------------------
|#

(deff main-gc ()
 (progress "GC: Locate publication directory")
 (:let pub (url2filename (option "url")))
 (:when (null pub) (main-wrong-url))
 (when (verbose '> 0)
  (format t "Publication directory: ~s~%" pub))
 (progress "GC: Locate root directories")
 (:let dir* (main-gc-collect-roots pub))
 (when (verbose '> 0)
  (format t "Root directories:~%")
  (dolist (dir dir*) (format t " ~s~%" dir)))
 (progress "GC: Locate pages under root directories")
 (:let patience (option "gcpatience"))
 (:let patience (time-mantissa patience 0))
 (:let time (- (get-universal-time) patience))
 (clrhash *main-gc-hash*)
 (:let root-assoc (main-collect-path-bib* dir* time pub nil))
 (:let root* (map 'list 'second root-assoc))
 (when (verbose '> 1)
  (format t "Root references:~%")
  (dolist (root (safe-sort #'> root*)) (format t " ~s~%" root)))
 (progress "GC: Locate pages under publication directory")
 ; Find mappings path->bib, ref->bib, and ref->path
 (clrhash *main-gc-hash*)
 (:let path-bib-assoc (main-collect-path-bib pub time pub nil))
 (:let ref-bib-path-array (main-ref-bib-path-array path-bib-assoc nil))
 (progress "GC: Locate directories to be kept or swept")
 (:let keep* (main-keep ref-bib-path-array root* pub nil))
 (when (verbose '> 1)
  (format t "References to be kept:~%")
  (dolist (keep (array-domain keep*)) (format t " ~s~%" keep)))
 (main-locate-sweep)
 (:let assoc (general-hash2assoc *main-gc-hash*))
 (:let assoc (sort assoc #'string< :key 'car))
 (when (verbose '> 0)
  (format t "Directories to be kept:~%")
  (dolist (keep (main-present-path* :keep assoc pub)) (format t " ~s~%" keep)))
 (:let sweep* (main-present-path* :sweep assoc pub))
 (:when (null sweep*) (format t "Nothing to be swept~%"))
 (format t "Directories to be swept:~%")
 (dolist (sweep sweep*) (format t " ~s~%" sweep))
 (prompt-info1 "garbage collection")
 (:catch () (when (verbose '>= 0) (format t "Goodbye~%~%")))
 (dolist (item assoc) (main-gc-step item))
 (format t "~%Garbage collection complete~%~%"))

#|
=============================================
Inspect/clear memorized state
=============================================
|#

(deff state-ls ()
 (:let cache (aget *state* (id-cache)))
 (if cache
  (state-ls1 cache *state*)
  (format t "The state is empty~%"))
 (values))

(deff state-ls1 (cache state)
 (state-ls2 cache state)
 (format t "Use (state-clear) to clear state~%"))

(deff state-ls2 (cache state)
 (:when (null cache) nil)
 (:let (cache1 . cache2) cache)
 (:when (numberp cache1) (format t "~a~%" (c-symbol2name cache1 0 state)))
 (state-ls2 cache1 state)
 (state-ls2 cache2 state))

(deff state-clear ()
 (setq *state* nil))

#|
=============================================
Print the value of the *diagnose* variable
=============================================
|#

(deff print-diagnose ()
 (:when (null *diagnose*) nil)
 (:when (null (option "diagnose")) nil)
 (format t "Diagnose output:~%")
 (:let depth (option "spydepth"))
 (:let depth (if (< depth 0) nil depth))
 (:let length (option "spylength"))
 (:let length (if (< length 0) nil length))
 (diagnose :depth depth :length length)
 (terpri))

#|
=============================================
Print the value of the *spy* variable
=============================================
|#

(deff print-spy ()
 (:when (null *spy*) nil)
 (:when (null (option "spy")) nil)
 (format t "Spy output:~%")
 (:let depth (option "spydepth"))
 (:let depth (if (= depth -2) 10 (if (= depth -1) nil depth)))
 (:let length (option "spylength"))
 (:let length (if (= length -2) 20 (if (= length -1) nil length)))
 (spy :depth depth :length length)
 (terpri))

#|
=============================================
Main function
=============================================
|#

(defun set-encoding ()
 (setf custom:*terminal-encoding*
  (ext:make-encoding
   :charset "UTF-8"
   :line-terminator :unix
   :input-error-action #\?
   :output-error-action #\?)))

(deff main (&rest args)
;(set-encoding)
 (:catch nil (when (verbose '>= 0) (progress "Goodbye")) (quit))
 (exit-on-error (init-pool args))
 (:catch nil nil)
 (:when (not (option "quit")) (main1))
 (:catch nil (when (verbose '>= 0) (progress "Goodbye")) (quit))
 (exit-on-error (main1))
 (quit))

(deff lgc (&rest args)
 (:catch nil (when (verbose '>= 0) (progress "Goodbye")) (values))
 (init-pool args)
 (main1)
 (values))

(deff lgc-check-parm ()
 (:when (string-prefix "*missing*" (option "url"))
  (format t "Missing url option.~%")
  (format t "The url option is typically set in the user configuration file.~%")
  (raise))
 (:when (level '< "submit") nil)
 (:when (> (option "iterations") 0)
  (format t "level=submit and iterations>0 are incompatible~%")
  (raise)))

(deff main1 ()
 (clear-trace-info)
 (:when (option "help") (print-usage))
 (:when (option "version") (print-version))
 (:when (option "option") (print-option))
 (:let name (option "optionval"))
 (:when (unequal name "") (print-optionval name))
 (:let name (option "optionstr"))
 (:when (unequal name "") (print-optionstr name))
 (:when (option "uninstall") (logiweb-uninstall))
 (:when (option "gc") (main-gc))
 (lgc-check-parm)
 (when (verbose '>= 0) (progress "Frontend"))
 (purge-conventional-wisdom)
 (:let state (when *memorize* *state*))
 (:let (ref state) (main-frontend state))
 (:when (null ref) nil)
 (:when (main-canned) nil)
 (main-backend ref state)
 (main-mirror ref state)
 (dump-machine ref state (option "machine"))
 (dump-machines ref state (option "bin"))
 (when (level '> "compile")
  (c-print-unfit)
  (c-print-unrecognized))
 (print-spy)
 (print-diagnose)
 (print-validity ref state)
 (:unless (option "exec") nil)
 (progress "Execute")
 (format t "~%Status=~s~%" (machine ref state)))










