#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
Logiweb
=============================================
Facilities for clisp make files
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Remember location of maker.lisp
=============================================
The location of maker.lisp is used by make-option which needs to load
all .lisp files in the 'common' directory which contains maker.lisp
|#

(defc *common-dir* (directory-namestring *load-pathname*))
(defc *main-dir* (directory-namestring (unslash *common-dir*)))

#|
=============================================
Main function of clisp make file
=============================================
Clisp make files should end with a line saying (exec-make)
|#

(defmacro exec-make ()
`(progn
  (if (atom *args*)
   (exec-make1 "all")
   (dolist (goal *args*) (exec-make1 goal)))
  (quit)))

(deff exec-make1 (goal)
 (:let symbol (intern (cat "MAKE-" (string-upcase goal))))
 (:when (fboundp symbol) (funcall symbol))
 (format t "Clisp make: Unknown goal~%"))

#|
=============================================
Check whether given file needs update
=============================================
(needs-update file &rest files) is true if at least one of the files
is missing and if the given file is older than at least one among the
given files
|#

(deff needs-update (file &rest files)
 (:when (null (probe-file file)) t)
 (needs-update1 file files))

(deff needs-update1 (file files)
 (:when (stringp files) (needs-update2 file files))
 (:when (pathnamep files) (needs-update2 file files))
 (:when (atom files) nil)
 (or (needs-update1 file (car files)) (needs-update1 file (cdr files))))

(deff needs-update2 (file1 file2)
 (:when (null (probe-file file2)) t)
 (:when (<debug (file-write-date file1) (file-write-date file2)) t)
 nil)

#|
=============================================
Directory listings
=============================================
|#

(deff dir* (&rest dirs)
 (map 'list 'directory dirs))

#|
=============================================
Convert path to relative file name
=============================================
Convert #p"..." path to relative file/directory name
|#

(deff path2relative (path)
 (:let namestring (namestring path))
 (:let length (length namestring))
 (:let position (position #\/ namestring :from-end t :end (- length 1)))
 (:when (integerp position) (subseq namestring (+ position 1)))
 path)

(etst (path2relative #p"/ab/cd") "cd")
(etst (path2relative #p"/ab/cd/") "cd/")

#|
=============================================
Convert directory to list of relative names
=============================================
|#

(deff dir2file* (dir)
 (map 'list 'path2relative (directory (cat dir "*"))))

(deff dir2dir* (dir)
 (map 'list 'path2relative (directory (cat dir "*/"))))

(deff dir2contents (dir)
 (list (dir2dir* dir) (dir2file* dir)))

#|
=============================================
Convert path to containing directory
=============================================
|#

(deff path2container (path)
 (:let namestring (namestring path))
 (:let length (length namestring))
 (:let position (position #\/ namestring :from-end t :end (- length 1)))
 (:when (integerp position) (subseq namestring 0 (+ position 1)))
 path)

(etst (path2container #p"/ab/cd") "/ab/")
(etst (path2container #p"/ab/cd/") "/ab/")

#|
=============================================
Write file
=============================================
|#

(defmacro write-file (stream filename &rest forms)
`(with-open-file
  (,stream ,filename
   :direction :output
   :if-exists :supersede
   :if-does-not-exist :create)
  ,@forms))

#|
=============================================
Make html directory
=============================================
|#

(deff pathname2filename (name)
 (:let name (namestring name))
 (:let end (max 0 (- (length name) 1)))
 (:let position (position #\/ name :from-end t :end end))
 (:when (null position) name)
 (subseq name (+ position 1)))

(deff simple-html-dir (dir)
 (:let html-dir (cat dir "lgwdir.html"))
 (run-rm "-f" html-dir)
 (:let name1 (directory (cat dir "*")))
 (:let name2 (directory (cat dir "*/")))
 (:let name* (append name1 name2))
 (:let name* (map 'list 'pathname2filename name*))
 (:let name* (sort name* #'string<))
 (:let name* (remove-duplicates name* :test #'string=))
 (write-file s html-dir
  (format s "<html>~%")
  (format s "<head>~%")
  (format s "<title>Index of ~a</title>~%" dir)
  (format s "</head>~%")
  (format s "<body>~%")
  (format s "<h2>Index of ~a</h2>~%" dir)
  (format s "<a href=\"..\">Up.</a>~%")
  (format s "<p>~%")
  (dolist (name name*) (format s "<a href=\"~a\">~a</a><br>~%" name name))
  (format s "<address>~%")
  (format s "<a href=\"http://www.diku.dk/~~grue/index.html\">")
  (format s "Klaus Grue</a>, GRD-2006-12-18~%")
  (format s "</address>~%")
  (format s "</body>~%")
  (format s "</html>~%")))

#|
=============================================
Run external programs
=============================================
|#

(deff run-command (command &rest arguments)
 (run-program command :arguments arguments))

(deff run-clisp (&rest arguments)
 (run-program "clisp" :arguments arguments))

(deff run-gcc (&rest arguments)
 (run-program "gcc" :arguments arguments))

(deff run-cp (&rest arguments)
 (run-program "cp" :arguments arguments))

(deff run-ln (&rest arguments)
 (run-program "ln" :arguments arguments))

(deff run-mv (&rest arguments)
 (run-program "mv" :arguments arguments))

(deff run-chmod (&rest arguments)
 (run-program "chmod" :arguments arguments))

(deff run-rm (&rest arguments)
 (run-program "rm" :arguments arguments))

(deff run-rmdir (&rest arguments)
 (run-program "rmdir" :arguments arguments))

(deff run-mkdir (&rest arguments)
 (run-program "mkdir" :arguments arguments))

(deff run-tar (&rest arguments)
 (run-program "tar" :arguments arguments))

(deff run-gzip (&rest arguments)
 (run-program "gzip" :arguments arguments))

(deff run-lgc (&rest arguments)
 (run-program "lgc" :arguments (cons "--" arguments)))

(deff run-lgwping (&rest arguments)
 (run-program "lgwping" :arguments arguments))

(deff file-cat (output &rest input)
 (run-program "cat" :arguments input :output output))

(deff set-owner (file user group)
 (run-command "chown" (cat user ":" group) file))

(deff run-rman (source target)
 (:let arg (list "-l" "Logiweb man page for %s.%s" "-f" "HTML" source))
 (run-program "rman" :output target :arguments arg))

(deff run-rman* (source-dir target-dir)
 (ensure-directories-exist (cat target-dir "a"))
 (in-directory source-dir
  (dolist (source (directory "*/*"))
   (run-rman1 source target-dir))))

(deff run-rman1 (source target-dir)
 (:let name (pathname-name source))
 (:let type (pathname-type source))
 (:let target (cat target-dir name "." type ".html"))
 (:let source (namestring source))
 (run-rman source target))

#|
=============================================
Compile c file
=============================================
|#

(deff val2flag (key val)
 (format nil "-D~a=~s" key val))

(deff compile-c (dst src flag*)
 (:let arg* (list* "-o" dst src flag*))
 (gcc-with-message arg*))

(deff gcc-with-message (arg*)
 (format t "gcc")
 (dolist (arg arg*) (format t " ~a" arg))
 (format t "~%")
 (run-program "gcc" :arguments arg*))

#|
=============================================
Directory cleaning
=============================================
Example: (delete-patterns "*.fas" "*.lib") deletes all .fas and .lib files.
|#

(deff delete-pattern (pattern)
 (dolist (path (directory pattern))
  (run-rm "-r" "-f" path)))
;  (delete-file path)))

(deff delete-patterns (&rest pattern*)
 (dolist (pattern pattern*) (delete-pattern pattern)))

(deff full-dir (pattern)
 (:let path (directory pattern :full t :if-does-not-exist :keep))
 (map 'list 'car path))

(deff delete-link (pattern)
 (dolist (path (full-dir pattern))
  (run-rm path)))

(deff delete-links (&rest pattern*)
 (dolist (pattern pattern*) (delete-link pattern)))

#|
(deff delete-dirlink (pattern)
 (dolist (path (full-dir (slash pattern)))
  (run-rm (unslash (namestring path)))))

(deff delete-dirlinks (&rest pattern*)
 (dolist (pattern pattern*) (delete-dirlink pattern)))
|#

#|
=============================================
Directory manipulation
=============================================
A 'revdir' is the list of directory names from a given directory to the root
directory. As an example, the revdir corresponding to /home/logiweb is ("logiweb" "home").

(cdr revdir) corresponds to 'cd ..'

(file2revdir file) returns the revdir that contains the given file.
(revdir2dir revdir) returns the directory that corresponds to the given revdir.
(revdir2wildfile revdir) returns path of all files of the given revdir.
(revdir2wilddir revdir) returns path of all subdirs of the given revdir.
(revdir-contents revdir) returns pathnames of the contents of the given revdir.

(rmdir-revdir revdir) removes the given revdir (if it is empty)
|#

(deff file2revdir (file)
 (:let (kind . dir) (pathname-directory (pathname (unslash file))))
 (:unless (equalp kind :absolute) :sorry)
 (reverse dir))

(etst (file2revdir "/ab/cd/ef") '("cd" "ab"))
(etst (file2revdir "/ab/cd/ef/") '("cd" "ab"))
(etst (file2revdir "..") :sorry)

(deff revdir2dir (revdir)
 (cons :absolute (reverse revdir)))

(etst (revdir2dir (file2revdir "/ab/cd/ef")) '(:absolute "ab" "cd"))

(deff revdir2path (revdir)
 (make-pathname :directory (revdir2dir revdir)))

(etst (revdir2path (file2revdir "/ab/cd/ef")) #p"/ab/cd/")

(deff revdir2wildfile (revdir)
 (make-pathname :directory (revdir2dir revdir) :name :wild))

(etst (revdir2wildfile (file2revdir "/ab/cd/ef")) #p"/ab/cd/*")

(deff revdir2wilddir (revdir)
 (make-pathname :directory (revdir2dir (cons :wild revdir))))

(etst (revdir2wilddir (file2revdir "/ab/cd/ef")) #p"/ab/cd/*/")

(deff revdir-contents (revdir)
 (append
  (directory (revdir2wildfile revdir))
  (directory (revdir2wilddir revdir))))

(deff rmdir-revdir (revdir)
 (run-rmdir (revdir2path revdir)))

#|
=============================================
Dump executable
=============================================
|#

(deff dump (filename init-function)
 (saveinitmem filename
  :init-function init-function
  :quiet t
  :norc t
  :executable t
  :keep-global-handlers nil))

#|
=============================================
Support for clisp make install
=============================================
|#

(defc *reckless* nil)

(deff prompt-info1 (task)
 (setq *reckless* nil)
 (format t "~%")
 (format t "Stepwise ~a~%" task)
 (format t "~%")
 (format t "At each step, type~%")
 (format t "<return>      to perform the step,~%")
 (format t "'s'           to skip the step,~%")
 (format t "'quit'        to exit,~%")
 (format t "'reckless'    to perform all remaining steps, or~%")
 (format t "anything else to skip the step.~%")
 (format t "~%"))

(deff prompt-info (task user)
 (prompt-info1 task)
 (format t "Do not exit using ctrl-C as that may confuse the terminal~%")
 (:when (null user) nil)
 (:let user1 (read-line (run-program "whoami" :output :stream)))
 (:when (equalp user user1) nil)
 (format t "~%")
 (format t "W A R N I N G ******* W A R N I N G ******* W A R N I N G~%")
 (format t "W A R N I N G ******* W A R N I N G ******* W A R N I N G~%")
 (format t "W A R N I N G ******* W A R N I N G ******* W A R N I N G~%")
 (format t "~%")
 (format t "You are user ~s but several steps in the following~%" user1)
 (format t "require you to be user ~s. It is safe to continue if~%" user)
 (format t "you answer 's' or 'quit' to all questions.~%" user)
 (format t "If you do not want to perform the task as the ~s user,~%" user)
 (format t "edit logiweb.conf or make.lisp to fit your needs.~%")
 (format t "~%")
 (format t "W A R N I N G ******* W A R N I N G ******* W A R N I N G~%")
 (format t "W A R N I N G ******* W A R N I N G ******* W A R N I N G~%")
 (format t "W A R N I N G ******* W A R N I N G ******* W A R N I N G~%"))

(deff skipped-make ()
 (:when *reckless* nil)
 (format t "~%<return>/s/quit/reckless > ")
 (:let line (read-line))
 (:when (equalp line "quit") (format t "~%") (raise))
 (:when (equalp line "reckless") (setq *reckless* t) nil)
 (:when (equalp line "") nil)
 (format t "~%skipped~%")
 t)

(deff install-file (msg from to mode &optional user (group user))
 (format t "~%")
 (format t "   ~a~%" msg)
 (:let to1 (mk-option to))
 (:when (equalp to1 "")
  (format t "   Skipped because ~s option is not set~%" to))
 (format t "   ~a <- ~a mode=~a" to1 from mode)
 (when user (format t " user=~a:~a" user group))
 (format t "~%")
 (:when (skipped-make) nil)
 (ensure-directories-exist to1)
 (install-file1 from to1 mode user group))

(deff install-file* (msg prefix from* to mode &optional user (group user))
 (format t "~%")
 (format t "   ~a~%" msg)
 (:let to1 (mk-option to))
 (:when (equalp to1 "")
  (format t "   Skipped because ~s option is not set~%" to))
 (:let to1 (slash to1))
 (dolist (from from*)
  (format t "   ~a <- ~a mode=~a" (cat to1 from) (cat prefix from) mode)
  (when user (format t " user=~a:~a" user group))
  (format t "~%"))
 (:when (skipped-make) nil)
 (ensure-directories-exist to1)
 (dolist (from from*)
  (install-file1 (cat prefix from) (cat to1 from) mode user group)))

(deff install-file1 (from to mode user group)
 (run-cp from to)
 (run-chmod mode to)
 (when user (set-owner to user group)))

(deff install-dir (msg path mode &optional user (group user))
 (format t "~%")
 (format t "   ~a~%" msg)
 (format t "   mkdir ~a mode=~a" path mode)
 (when user (format t " user=~a:~a" user group))
 (format t "~%")
 (:when (skipped-make) nil)
 (install-dir1 path mode user group))

(deff install-dir1 (path mode user group)
 (ensure-directories-exist path)
 (run-chmod mode path)
 (when user (set-owner path user group)))

(deff clean-install-file-tree (msg from to umask)
 (format t "~%")
 (format t "   ~a~%" msg)
 (:let to1 (mk-option to))
 (:when (equalp to1 "")
  (format t "   Skipped because ~s option is not set~%" to))
 (format t "   ~a <- ~a umask=~3,'0o user=current user" to1 from umask)
 (format t "~%")
 (:when (skipped-make) nil)
 (ensure-directories-exist to1)
 (run-rm "-r" to1)
 (:let umask (umask umask))
 (unwind-protect
  (run-cp "-r" from to1)
  (umask umask)))

(deff install-file-tree (msg from to mode &optional user (group user))
 (format t "~%")
 (format t "   ~a~%" msg)
 (:let to1 (mk-option to))
 (:when (equalp to1 "")
  (format t "   Skipped because ~s option is not set~%" to))
 (format t "   ~a <- ~a mode=~a" to1 from mode)
 (when user (format t " user=~a:~a" user group))
 (format t "~%")
 (:when (skipped-make) nil)
 (ensure-directories-exist to)
 (install-file-tree1 from to1 mode user group))

(deff install-file-tree1 (from to mode user group)
 (install-files from to (dir2file* from) mode user group)
 (install-dir*  from to (dir2dir*  from) mode user group))

(deff install-files (from to file* mode user group)
 (:when (atom file*) nil)
 (:let (file . file*) file*)
 (install-file1 (cat from file) (cat to file) mode user group)
 (install-files from to file* mode user group))

(deff install-dir* (from to dir* mode user group)
 (:when (atom dir*) nil)
 (:let (dir . dir*) dir*)
 (install-dir1 (cat to dir) mode user group)
 (install-file-tree1 (cat from dir) (cat to dir) mode user group)
 (install-dir* from to dir* mode user group))

(deff install-run (msg command &rest args)
 (format t "~%")
 (format t "   ~a~%" msg)
 (format t "   ~a~%" (cons command args))
 (:when (skipped-make) nil)
 (run-program command :arguments args))

#|
=============================================
Support for clisp make uninstall
=============================================
|#

(deff install-rm (msg option)
 (format t "~%")
 (format t "   ~a~%" msg)
 (:let file (mk-option option))
 (:when (equalp file "")
  (format t "   Skipped because ~s option is not set~%" option))
 (format t "   remove ~a~%" file)
 (:when (skipped-make) nil)
 (run-rm file)
 (install-rm-above (file2revdir file)))

(deff install-rm* (msg option file*)
 (format t "~%")
 (format t "   ~a~%" msg)
 (:let dir (mk-option option))
 (:when (equalp dir "")
  (format t "   Skipped because ~s option is not set~%" option))
 (dolist (file file*) (format t "   remove ~a~%" (cat dir file)))
 (:when (skipped-make) nil)
 (dolist (file file*) (run-rm (cat dir file)))
 (:let file (cat dir (head file*)))
 (install-rm-above (file2revdir file)))

(deff install-rm-above (revdir)
 (:when (atom revdir) nil)
 (:when (revdir-contents revdir) nil)
 (format t "~%")
 (format t "   Containing directory has become empty. Remove it?~%")
 (:let path (revdir2path revdir))
 (format t "   remove ~a~%" path)
 (:when (skipped-make) nil)
 (run-rmdir path)
 (install-rm-above (cdr revdir)))

(deff install-rmdir (msg option)
 (format t "~%")
 (format t "   ~a~%" msg)
 (:let dir (slash (mk-option option)))
 (:when (equalp dir "/")
  (format t "   Skipped because ~s option is not set~%" option))
 (format t "   remove directory ~a and all its contents~%" dir)
 (:when (skipped-make) nil)
 (run-rm "-r" dir)
 (install-rm-above (file2revdir dir)))

#|
=============================================
Uninstallation
=============================================
Uninstallation may be requested from make file or from the lgc or logiweb commands.
|#

(deff logiweb-uninstall ()
 (:catch () (quit))
 (prompt-info "uninstall of Logiweb" "root")
 (install-run "Stop logiweb server" "/sbin/service" "logiweb" "stop")
 (install-run "Remove logiweb from boot sequence"
  "/sbin/chkconfig" "--del" "logiweb")
 (install-rm "Uninstall shared object used by server" "varlib")
 (install-rm "Uninstall init script for server" "varinit")
 (install-rmdir "Uninstall log directory" "log")
 (install-rm "Uninstall command for pinging the server" "varlgwping")
 (install-rm "Uninstall CGI user interface of the server" "varrelay")
 (install-rm "Uninstall command used by CGI user interface" "varlgwrelay")
 (install-rm "Uninstall configuration file for http server" "varhttp")
 (install-run "Reinitialize http server" "/sbin/service" "httpd" "reload")
 (install-rm "Uninstall Logiweb compiler" "varlgc")
 (install-rm "Uninstall Logiweb server" "varlogiweb")
#|
 (:let jailcmd '("lgwlatex" "lgwbibtex" "lgwmakeindex" "lgwdvipdfm"))
 (:let jailcmd (list* "lgwmkjail" "lgwrmjail" jailcmd))
 (install-rm* "Uninstall chroot jail programs" "varbin" jailcmd)
 (install-rmdir "Uninstall chroot jail template" "varjail")
|#
 (format t "~%")
 (:let conf (mk-option "varconf"))
 (when (probe-file conf)
  (format t "Remove site configuration file ~a manually~%~%" conf))
 (:let home (slash (mk-option "varhome")))
 (when (probe-directory home)
  (format t "Remove document root ~a manually~%~%" home))
 (:let man (slash (mk-option "varman")))
 (when (probe-directory man)
  (format t "Remove man pages in ~a manually~%~%" man)))

#|
=============================================
Option processing
=============================================
*conf* contains the compiled in default for the conf parameter. *conf* is used by common/option.lisp as a default for locating the main configuration file. *conf* is set in lgc/frontend.lisp when dumping a new lgc command.

The present file (maker.lisp) sets *conf* to point to logiweb.conf in case it does not already point to something else. It may point to something else e.g. when developing and debugging the software from a clisp prompt. *conf* must have a value before common/option.lisp is loaded.

(mk-option-load) tests whether or not common/option.lisp has been loaded. If common/option.lisp has been loaded, mk-option-load does nothing. Otherwise, it
|#

(defvar *conf*)

(deff mk-option-load ()
 (:when (fboundp 'option))
 (in-directory *common-dir* (testload "source" :prompt nil :test nil)))

(deff mk-option (option-name)
 (mk-option-load)
 (funcall 'option option-name))

(defc needed-options '(
 "httphost"
 "roots"
 "varbin"
 "varconf"
 "varhome"
 "varhttp"
 "varinit"
;"varjail"
 "varlgc"
 "varlgwam"
 "varlgwping"
 "varlgwrelay"
 "varlib"
 "varlogiweb"
 "varman"
 "varrelay"
 "varscript"))

(deff check-needed-options ()
 (check-option needed-options))

(deff check-option (option*)
 (:let option (check-option* option*))
 (:when (null option) t)
 (format t "~%")
 (format t "The following options must be set:~%")
 (format t "~a~%" option*)
 (format t "~%")
 (format t "The following option is not set: ~a~%" option)
 (format t "~%")
 (format t "Please set ~a e.g. in ~a~%" option *conf*)
 (format t "~%"))

(deff check-option* (option*)
 (:when (atom option*) nil)
 (:let (option . option*) option*)
 (or (check-option1 option) (check-option* option*)))

(deff proper-option (option)
 (:when (equalp option "") nil)
 (:when (equalp option "/") nil)
 t)

(deff check-option1 (option)
 (:when (proper-option (mk-option option)) nil)
 option)



