/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.hnsw;

import java.io.IOException;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SplittableRandom;
import java.util.concurrent.TimeUnit;
import org.apache.lucene.index.VectorEncoding;
import org.apache.lucene.index.VectorSimilarityFunction;
import org.apache.lucene.util.FixedBitSet;
import org.apache.lucene.util.InfoStream;
import org.apache.lucene.util.hnsw.HnswGraph;
import org.apache.lucene.util.hnsw.HnswGraphSearcher;
import org.apache.lucene.util.hnsw.NeighborArray;
import org.apache.lucene.util.hnsw.NeighborQueue;
import org.apache.lucene.util.hnsw.OnHeapHnswGraph;
import org.apache.lucene.util.hnsw.RandomAccessVectorValues;

public final class HnswGraphBuilder<T> {
    public static final int DEFAULT_MAX_CONN = 16;
    public static final int DEFAULT_BEAM_WIDTH = 100;
    private static final long DEFAULT_RAND_SEED = 42L;
    public static final String HNSW_COMPONENT = "HNSW";
    public static long randSeed = 42L;
    private final int M;
    private final int beamWidth;
    private final double ml;
    private final NeighborArray scratch;
    private final VectorSimilarityFunction similarityFunction;
    private final VectorEncoding vectorEncoding;
    private final RandomAccessVectorValues<T> vectors;
    private final SplittableRandom random;
    private final HnswGraphSearcher<T> graphSearcher;
    final OnHeapHnswGraph hnsw;
    private InfoStream infoStream = InfoStream.getDefault();
    private final RandomAccessVectorValues<T> vectorsCopy;
    private final Set<Integer> initializedNodes;

    public static <T> HnswGraphBuilder<T> create(RandomAccessVectorValues<T> vectors, VectorEncoding vectorEncoding, VectorSimilarityFunction similarityFunction, int M, int beamWidth, long seed) throws IOException {
        return new HnswGraphBuilder<T>(vectors, vectorEncoding, similarityFunction, M, beamWidth, seed);
    }

    public static <T> HnswGraphBuilder<T> create(RandomAccessVectorValues<T> vectors, VectorEncoding vectorEncoding, VectorSimilarityFunction similarityFunction, int M, int beamWidth, long seed, HnswGraph initializerGraph, Map<Integer, Integer> oldToNewOrdinalMap) throws IOException {
        HnswGraphBuilder<T> hnswGraphBuilder = new HnswGraphBuilder<T>(vectors, vectorEncoding, similarityFunction, M, beamWidth, seed);
        hnswGraphBuilder.initializeFromGraph(initializerGraph, oldToNewOrdinalMap);
        return hnswGraphBuilder;
    }

    private HnswGraphBuilder(RandomAccessVectorValues<T> vectors, VectorEncoding vectorEncoding, VectorSimilarityFunction similarityFunction, int M, int beamWidth, long seed) throws IOException {
        this.vectors = vectors;
        this.vectorsCopy = vectors.copy();
        this.vectorEncoding = Objects.requireNonNull(vectorEncoding);
        this.similarityFunction = Objects.requireNonNull(similarityFunction);
        if (M <= 0) {
            throw new IllegalArgumentException("maxConn must be positive");
        }
        if (beamWidth <= 0) {
            throw new IllegalArgumentException("beamWidth must be positive");
        }
        this.M = M;
        this.beamWidth = beamWidth;
        this.ml = M == 1 ? 1.0 : 1.0 / Math.log(1.0 * (double)M);
        this.random = new SplittableRandom(seed);
        this.hnsw = new OnHeapHnswGraph(M);
        this.graphSearcher = new HnswGraphSearcher(vectorEncoding, similarityFunction, new NeighborQueue(beamWidth, true), new FixedBitSet(this.vectors.size()));
        this.scratch = new NeighborArray(Math.max(beamWidth, M + 1), false);
        this.initializedNodes = new HashSet<Integer>();
    }

    public OnHeapHnswGraph build(RandomAccessVectorValues<T> vectorsToAdd) throws IOException {
        if (vectorsToAdd == this.vectors) {
            throw new IllegalArgumentException("Vectors to build must be independent of the source of vectors provided to HnswGraphBuilder()");
        }
        if (this.infoStream.isEnabled(HNSW_COMPONENT)) {
            this.infoStream.message(HNSW_COMPONENT, "build graph from " + vectorsToAdd.size() + " vectors");
        }
        this.addVectors(vectorsToAdd);
        return this.hnsw;
    }

    private void initializeFromGraph(HnswGraph initializerGraph, Map<Integer, Integer> oldToNewOrdinalMap) throws IOException {
        assert (this.hnsw.size() == 0);
        float[] vectorValue = null;
        byte[] binaryValue = null;
        for (int level = 0; level < initializerGraph.numLevels(); ++level) {
            HnswGraph.NodesIterator it = initializerGraph.getNodesOnLevel(level);
            while (it.hasNext()) {
                int oldOrd = it.nextInt();
                int newOrd = oldToNewOrdinalMap.get(oldOrd);
                this.hnsw.addNode(level, newOrd);
                if (level == 0) {
                    this.initializedNodes.add(newOrd);
                }
                switch (this.vectorEncoding) {
                    case FLOAT32: {
                        vectorValue = (float[])this.vectors.vectorValue(newOrd);
                        break;
                    }
                    case BYTE: {
                        binaryValue = (byte[])this.vectors.vectorValue(newOrd);
                    }
                }
                NeighborArray newNeighbors = this.hnsw.getNeighbors(level, newOrd);
                initializerGraph.seek(level, oldOrd);
                int oldNeighbor = initializerGraph.nextNeighbor();
                while (oldNeighbor != Integer.MAX_VALUE) {
                    float score;
                    int newNeighbor = oldToNewOrdinalMap.get(oldNeighbor);
                    switch (this.vectorEncoding) {
                        default: {
                            score = this.similarityFunction.compare(vectorValue, (float[])this.vectorsCopy.vectorValue(newNeighbor));
                            break;
                        }
                        case BYTE: {
                            score = this.similarityFunction.compare(binaryValue, (byte[])this.vectorsCopy.vectorValue(newNeighbor));
                        }
                    }
                    newNeighbors.addOutOfOrder(newNeighbor, score);
                    oldNeighbor = initializerGraph.nextNeighbor();
                }
            }
        }
    }

    private void addVectors(RandomAccessVectorValues<T> vectorsToAdd) throws IOException {
        long start;
        long t2 = start = System.nanoTime();
        for (int node = 0; node < vectorsToAdd.size(); ++node) {
            if (this.initializedNodes.contains(node)) continue;
            this.addGraphNode(node, vectorsToAdd);
            if (node % 10000 != 0 || !this.infoStream.isEnabled(HNSW_COMPONENT)) continue;
            t2 = this.printGraphBuildStatus(node, start, t2);
        }
    }

    public void setInfoStream(InfoStream infoStream) {
        this.infoStream = infoStream;
    }

    public OnHeapHnswGraph getGraph() {
        return this.hnsw;
    }

    public void addGraphNode(int node, T value) throws IOException {
        NeighborQueue candidates;
        int level;
        int nodeLevel = HnswGraphBuilder.getRandomGraphLevel(this.ml, this.random);
        int curMaxLevel = this.hnsw.numLevels() - 1;
        if (this.hnsw.entryNode() == -1) {
            for (int level2 = nodeLevel; level2 >= 0; --level2) {
                this.hnsw.addNode(level2, node);
            }
            return;
        }
        int[] eps = new int[]{this.hnsw.entryNode()};
        for (level = nodeLevel; level > curMaxLevel; --level) {
            this.hnsw.addNode(level, node);
        }
        for (level = curMaxLevel; level > nodeLevel; --level) {
            candidates = this.graphSearcher.searchLevel(value, 1, level, eps, this.vectors, this.hnsw);
            eps = new int[]{candidates.pop()};
        }
        for (level = Math.min(nodeLevel, curMaxLevel); level >= 0; --level) {
            candidates = this.graphSearcher.searchLevel(value, this.beamWidth, level, eps, this.vectors, this.hnsw);
            eps = candidates.nodes();
            this.hnsw.addNode(level, node);
            this.addDiverseNeighbors(level, node, candidates);
        }
    }

    public void addGraphNode(int node, RandomAccessVectorValues<T> values) throws IOException {
        this.addGraphNode(node, values.vectorValue(node));
    }

    private long printGraphBuildStatus(int node, long start, long t2) {
        long now = System.nanoTime();
        this.infoStream.message(HNSW_COMPONENT, String.format(Locale.ROOT, "built %d in %d/%d ms", node, TimeUnit.NANOSECONDS.toMillis(now - t2), TimeUnit.NANOSECONDS.toMillis(now - start)));
        return now;
    }

    private void addDiverseNeighbors(int level, int node, NeighborQueue candidates) throws IOException {
        NeighborArray neighbors = this.hnsw.getNeighbors(level, node);
        assert (neighbors.size() == 0);
        this.popToScratch(candidates);
        int maxConnOnLevel = level == 0 ? this.M * 2 : this.M;
        this.selectAndLinkDiverse(neighbors, this.scratch, maxConnOnLevel);
        int size = neighbors.size();
        for (int i = 0; i < size; ++i) {
            int nbr = neighbors.node[i];
            NeighborArray nbrsOfNbr = this.hnsw.getNeighbors(level, nbr);
            nbrsOfNbr.addOutOfOrder(node, neighbors.score[i]);
            if (nbrsOfNbr.size() <= maxConnOnLevel) continue;
            int indexToRemove = this.findWorstNonDiverse(nbrsOfNbr);
            nbrsOfNbr.removeIndex(indexToRemove);
        }
    }

    private void selectAndLinkDiverse(NeighborArray neighbors, NeighborArray candidates, int maxConnOnLevel) throws IOException {
        for (int i = candidates.size() - 1; neighbors.size() < maxConnOnLevel && i >= 0; --i) {
            int cNode = candidates.node[i];
            float cScore = candidates.score[i];
            assert (cNode < this.hnsw.size());
            if (!this.diversityCheck(cNode, cScore, neighbors)) continue;
            neighbors.addInOrder(cNode, cScore);
        }
    }

    private void popToScratch(NeighborQueue candidates) {
        this.scratch.clear();
        int candidateCount = candidates.size();
        for (int i = 0; i < candidateCount; ++i) {
            float maxSimilarity = candidates.topScore();
            this.scratch.addInOrder(candidates.pop(), maxSimilarity);
        }
    }

    private boolean diversityCheck(int candidate, float score, NeighborArray neighbors) throws IOException {
        return this.isDiverse(candidate, neighbors, score);
    }

    private boolean isDiverse(int candidate, NeighborArray neighbors, float score) throws IOException {
        switch (this.vectorEncoding) {
            case BYTE: {
                return this.isDiverse((byte[])this.vectors.vectorValue(candidate), neighbors, score);
            }
        }
        return this.isDiverse((float[])this.vectors.vectorValue(candidate), neighbors, score);
    }

    private boolean isDiverse(float[] candidate, NeighborArray neighbors, float score) throws IOException {
        for (int i = 0; i < neighbors.size(); ++i) {
            float neighborSimilarity = this.similarityFunction.compare(candidate, (float[])this.vectorsCopy.vectorValue(neighbors.node[i]));
            if (!(neighborSimilarity >= score)) continue;
            return false;
        }
        return true;
    }

    private boolean isDiverse(byte[] candidate, NeighborArray neighbors, float score) throws IOException {
        for (int i = 0; i < neighbors.size(); ++i) {
            float neighborSimilarity = this.similarityFunction.compare(candidate, (byte[])this.vectorsCopy.vectorValue(neighbors.node[i]));
            if (!(neighborSimilarity >= score)) continue;
            return false;
        }
        return true;
    }

    private int findWorstNonDiverse(NeighborArray neighbors) throws IOException {
        int[] uncheckedIndexes = neighbors.sort();
        if (uncheckedIndexes == null) {
            return neighbors.size() - 1;
        }
        int uncheckedCursor = uncheckedIndexes.length - 1;
        for (int i = neighbors.size() - 1; i > 0 && uncheckedCursor >= 0; --i) {
            if (this.isWorstNonDiverse(i, neighbors, uncheckedIndexes, uncheckedCursor)) {
                return i;
            }
            if (i != uncheckedIndexes[uncheckedCursor]) continue;
            --uncheckedCursor;
        }
        return neighbors.size() - 1;
    }

    private boolean isWorstNonDiverse(int candidateIndex, NeighborArray neighbors, int[] uncheckedIndexes, int uncheckedCursor) throws IOException {
        int candidateNode = neighbors.node[candidateIndex];
        switch (this.vectorEncoding) {
            case BYTE: {
                return this.isWorstNonDiverse(candidateIndex, (byte[])this.vectors.vectorValue(candidateNode), neighbors, uncheckedIndexes, uncheckedCursor);
            }
        }
        return this.isWorstNonDiverse(candidateIndex, (float[])this.vectors.vectorValue(candidateNode), neighbors, uncheckedIndexes, uncheckedCursor);
    }

    private boolean isWorstNonDiverse(int candidateIndex, float[] candidateVector, NeighborArray neighbors, int[] uncheckedIndexes, int uncheckedCursor) throws IOException {
        float minAcceptedSimilarity = neighbors.score[candidateIndex];
        if (candidateIndex == uncheckedIndexes[uncheckedCursor]) {
            for (int i = candidateIndex - 1; i >= 0; --i) {
                float neighborSimilarity = this.similarityFunction.compare(candidateVector, (float[])this.vectorsCopy.vectorValue(neighbors.node[i]));
                if (!(neighborSimilarity >= minAcceptedSimilarity)) continue;
                return true;
            }
        } else {
            assert (candidateIndex > uncheckedIndexes[uncheckedCursor]);
            for (int i = uncheckedCursor; i >= 0; --i) {
                float neighborSimilarity = this.similarityFunction.compare(candidateVector, (float[])this.vectorsCopy.vectorValue(neighbors.node[uncheckedIndexes[i]]));
                if (!(neighborSimilarity >= minAcceptedSimilarity)) continue;
                return true;
            }
        }
        return false;
    }

    private boolean isWorstNonDiverse(int candidateIndex, byte[] candidateVector, NeighborArray neighbors, int[] uncheckedIndexes, int uncheckedCursor) throws IOException {
        float minAcceptedSimilarity = neighbors.score[candidateIndex];
        if (candidateIndex == uncheckedIndexes[uncheckedCursor]) {
            for (int i = candidateIndex - 1; i >= 0; --i) {
                float neighborSimilarity = this.similarityFunction.compare(candidateVector, (byte[])this.vectorsCopy.vectorValue(neighbors.node[i]));
                if (!(neighborSimilarity >= minAcceptedSimilarity)) continue;
                return true;
            }
        } else {
            assert (candidateIndex > uncheckedIndexes[uncheckedCursor]);
            for (int i = uncheckedCursor; i >= 0; --i) {
                float neighborSimilarity = this.similarityFunction.compare(candidateVector, (byte[])this.vectorsCopy.vectorValue(neighbors.node[uncheckedIndexes[i]]));
                if (!(neighborSimilarity >= minAcceptedSimilarity)) continue;
                return true;
            }
        }
        return false;
    }

    private static int getRandomGraphLevel(double ml, SplittableRandom random) {
        double randDouble;
        while ((randDouble = random.nextDouble()) == 0.0) {
        }
        return (int)(-Math.log(randDouble) * ml);
    }
}

