/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.bytecomparable;

import com.google.common.base.Preconditions;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import org.apache.cassandra.db.marshal.ValueAccessor;
import org.apache.cassandra.utils.bytecomparable.ByteComparable;
import org.apache.cassandra.utils.memory.MemoryUtil;

public interface ByteSource {
    public static final int END_OF_STREAM = -1;
    public static final ByteSource EMPTY;
    public static final int ESCAPE = 0;
    public static final int ESCAPED_0_CONT = 254;
    public static final int ESCAPED_0_DONE = 255;
    public static final int MIN_SEPARATOR = 16;
    public static final int MAX_SEPARATOR = 239;
    public static final int NEXT_COMPONENT = 64;
    public static final int NEXT_COMPONENT_EMPTY = 63;
    public static final int NEXT_COMPONENT_EMPTY_REVERSED = 65;
    public static final int NEXT_COMPONENT_NULL = 62;
    public static final int MIN_NEXT_COMPONENT = 60;
    public static final int MAX_NEXT_COMPONENT = 68;
    public static final int TERMINATOR = 56;
    public static final int LT_NEXT_COMPONENT = 32;
    public static final int GT_NEXT_COMPONENT = 96;
    public static final int LTLT_NEXT_COMPONENT = 31;
    public static final int GTGT_NEXT_COMPONENT = 97;
    public static final int EXCLUDED = 24;

    public int next();

    public static <V> ByteSource of(ValueAccessor<V> accessor, V data, ByteComparable.Version version) {
        return new AccessorEscaper<V>(accessor, data, version);
    }

    public static ByteSource of(ByteBuffer buf, ByteComparable.Version version) {
        return new BufferEscaper(buf, version);
    }

    public static ByteSource of(byte[] buf, ByteComparable.Version version) {
        return new ArrayEscaper(buf, version);
    }

    public static ByteSource ofMemory(long address, int length, ByteComparable.Version version) {
        return new MemoryEscaper(address, length, version);
    }

    public static ByteSource withTerminator(int terminator, ByteSource ... srcs) {
        if (!(7.$assertionsDisabled || terminator >= 16 && terminator <= 239)) {
            throw new AssertionError();
        }
        if (!7.$assertionsDisabled && terminator >= 60 && terminator <= 68) {
            throw new AssertionError();
        }
        return new Multi(srcs, terminator);
    }

    public static ByteSource withTerminatorLegacy(int terminator, ByteSource ... srcs) {
        return new Multi(srcs, terminator);
    }

    public static ByteSource withTerminatorMaybeLegacy(ByteComparable.Version version, int legacyTerminator, ByteSource ... srcs) {
        return version == ByteComparable.Version.LEGACY ? ByteSource.withTerminatorLegacy(legacyTerminator, srcs) : ByteSource.withTerminator(56, srcs);
    }

    public static ByteSource of(String s2, ByteComparable.Version version) {
        return new ArrayEscaper(s2.getBytes(StandardCharsets.UTF_8), version);
    }

    public static ByteSource of(long value) {
        return new Number(value ^ Long.MIN_VALUE, 8);
    }

    public static ByteSource of(int value) {
        return new Number((long)value ^ 0x80000000L, 4);
    }

    public static <V> ByteSource optionalSignedFixedLengthNumber(ValueAccessor<V> accessor, V data) {
        return !accessor.isEmpty(data) ? ByteSource.signedFixedLengthNumber(accessor, data) : null;
    }

    public static <V> ByteSource signedFixedLengthNumber(ValueAccessor<V> accessor, V data) {
        return new SignedFixedLengthNumber<V>(accessor, data);
    }

    public static <V> ByteSource optionalSignedFixedLengthFloat(ValueAccessor<V> accessor, V data) {
        return !accessor.isEmpty(data) ? ByteSource.signedFixedLengthFloat(accessor, data) : null;
    }

    public static <V> ByteSource signedFixedLengthFloat(ValueAccessor<V> accessor, V data) {
        return new SignedFixedLengthFloat<V>(accessor, data);
    }

    public static ByteSource variableLengthInteger(long value) {
        return new VariableLengthInteger(value);
    }

    public static ByteSource separatorPrefix(ByteSource prevMax, ByteSource currMin) {
        return new Separator(prevMax, currMin, true);
    }

    public static ByteSource separatorGt(ByteSource prevMax, ByteSource currMin) {
        return new Separator(prevMax, currMin, false);
    }

    public static ByteSource oneByte(final int i) {
        if (!(7.$assertionsDisabled || i >= 0 && i <= 255)) {
            throw new AssertionError((Object)"Argument must be a valid unsigned byte.");
        }
        return new ByteSource(){
            boolean consumed = false;

            @Override
            public int next() {
                if (this.consumed) {
                    return -1;
                }
                this.consumed = true;
                return i;
            }
        };
    }

    public static ByteSource cut(final ByteSource src, final int cutoff) {
        return new ByteSource(){
            int pos = 0;

            @Override
            public int next() {
                return this.pos++ < cutoff ? src.next() : -1;
            }
        };
    }

    public static ByteSource cutOrRightPad(final ByteSource src, final int cutoff, final int padding) {
        return new ByteSource(){
            int pos = 0;

            @Override
            public int next() {
                if (this.pos++ >= cutoff) {
                    return -1;
                }
                int next = src.next();
                return next == -1 ? padding : next;
            }
        };
    }

    public static <V> ByteSource optionalFixedLength(ValueAccessor<V> accessor, V data) {
        return !accessor.isEmpty(data) ? ByteSource.fixedLength(accessor, data) : null;
    }

    public static <V> ByteSource fixedLength(final ValueAccessor<V> accessor, final V data) {
        return new ByteSource(){
            int pos = -1;

            @Override
            public int next() {
                return ++this.pos < accessor.size(data) ? accessor.getByte(data, this.pos) & 0xFF : -1;
            }
        };
    }

    public static ByteSource fixedLength(final ByteBuffer b) {
        return new ByteSource(){
            int pos;
            {
                this.pos = b.position() - 1;
            }

            @Override
            public int next() {
                return ++this.pos < b.limit() ? b.get(this.pos) & 0xFF : -1;
            }
        };
    }

    public static ByteSource fixedLength(byte[] b) {
        return ByteSource.fixedLength(b, 0, b.length);
    }

    public static ByteSource fixedLength(final byte[] b, final int offset, final int length) {
        Preconditions.checkArgument(offset >= 0 && offset <= b.length);
        Preconditions.checkArgument(length >= 0 && offset + length <= b.length);
        return new ByteSource(){
            int pos;
            {
                this.pos = offset - 1;
            }

            @Override
            public int next() {
                return ++this.pos < offset + length ? b[this.pos] & 0xFF : -1;
            }
        };
    }

    public static Peekable peekable(ByteSource p) {
        if (p == null) {
            return null;
        }
        return p instanceof Peekable ? (Peekable)p : new Peekable(p);
    }

    static {
        if (7.$assertionsDisabled) {
            // empty if block
        }
        EMPTY = () -> -1;
    }

    public static class Peekable
    implements ByteSource {
        private static final int NONE = Integer.MIN_VALUE;
        private final ByteSource wrapped;
        private int peeked = Integer.MIN_VALUE;

        public Peekable(ByteSource wrapped) {
            this.wrapped = wrapped;
        }

        @Override
        public int next() {
            if (this.peeked != Integer.MIN_VALUE) {
                int val = this.peeked;
                this.peeked = Integer.MIN_VALUE;
                return val;
            }
            return this.wrapped.next();
        }

        public int peek() {
            if (this.peeked == Integer.MIN_VALUE) {
                this.peeked = this.wrapped.next();
            }
            return this.peeked;
        }
    }

    public static class Separator
    implements ByteSource {
        private final ByteSource prev;
        private final ByteSource curr;
        private boolean done = false;
        private final boolean useCurr;

        Separator(ByteSource prevMax, ByteSource currMin, boolean useCurr) {
            this.prev = prevMax;
            this.curr = currMin;
            this.useCurr = useCurr;
        }

        @Override
        public int next() {
            if (this.done) {
                return -1;
            }
            int p = this.prev.next();
            int c = this.curr.next();
            assert (p <= c) : this.prev + " not less than " + this.curr;
            if (p == c) {
                return c;
            }
            this.done = true;
            return this.useCurr ? c : p + 1;
        }
    }

    public static class Multi
    implements ByteSource {
        private final ByteSource[] srcs;
        private int srcnum = -1;
        private final int sequenceTerminator;

        Multi(ByteSource[] srcs, int sequenceTerminator) {
            this.srcs = srcs;
            this.sequenceTerminator = sequenceTerminator;
        }

        @Override
        public int next() {
            if (this.srcnum == this.srcs.length) {
                return -1;
            }
            int b = -1;
            if (this.srcnum >= 0 && this.srcs[this.srcnum] != null) {
                b = this.srcs[this.srcnum].next();
            }
            if (b > -1) {
                return b;
            }
            ++this.srcnum;
            if (this.srcnum == this.srcs.length) {
                return this.sequenceTerminator;
            }
            if (this.srcs[this.srcnum] == null) {
                return 62;
            }
            return 64;
        }
    }

    public static class SignedFixedLengthFloat<V>
    implements ByteSource {
        private final ValueAccessor<V> accessor;
        private final V data;
        private int bufpos;
        private boolean invert;

        public SignedFixedLengthFloat(ValueAccessor<V> accessor, V data) {
            this.accessor = accessor;
            this.data = data;
            this.bufpos = 0;
        }

        @Override
        public int next() {
            if (this.bufpos >= this.accessor.size(this.data)) {
                return -1;
            }
            int v = this.accessor.getByte(this.data, this.bufpos) & 0xFF;
            if (this.bufpos == 0) {
                this.invert = v >= 128;
                v |= 0x80;
            }
            if (this.invert) {
                v ^= 0xFF;
            }
            ++this.bufpos;
            return v;
        }
    }

    public static class Number
    implements ByteSource {
        private final long value;
        private int pos;

        public Number(long value, int length) {
            this.value = value;
            this.pos = length;
        }

        @Override
        public int next() {
            if (this.pos == 0) {
                return -1;
            }
            return (int)(this.value >> --this.pos * 8 & 0xFFL);
        }
    }

    public static class VariableLengthInteger
    implements ByteSource {
        private final long value;
        private int pos;

        public VariableLengthInteger(long value) {
            long negativeMask = value >> 63;
            int bits = 64 - Long.numberOfLeadingZeros((value ^= negativeMask) | 1L);
            int bytes = bits / 7 + 1;
            if (bytes >= 9) {
                value |= Long.MIN_VALUE;
                this.pos = negativeMask < 0L ? 256 : -1;
            } else {
                long mask = -256 >> bytes & 0xFF;
                this.pos = bytes * 8;
                value |= mask << this.pos - 8;
            }
            this.value = value ^= negativeMask;
        }

        @Override
        public int next() {
            if (this.pos <= 0 || this.pos > 64) {
                if (this.pos == 0) {
                    return -1;
                }
                int result = this.pos & 0xFF;
                this.pos = 64;
                return result;
            }
            this.pos -= 8;
            return (int)(this.value >>> this.pos) & 0xFF;
        }
    }

    public static class VariableLengthUnsignedInteger
    implements ByteSource {
        private final long value;
        private int pos = -1;

        public VariableLengthUnsignedInteger(long value) {
            this.value = value;
        }

        @Override
        public int next() {
            if (this.pos == -1) {
                int bitsMinusOne = 63 - Long.numberOfLeadingZeros(this.value | 1L);
                int bytesMinusOne = bitsMinusOne / 7;
                int mask = -256 >> bytesMinusOne;
                this.pos = bytesMinusOne * 8;
                return (int)(this.value >>> this.pos | (long)mask) & 0xFF;
            }
            this.pos -= 8;
            if (this.pos < 0) {
                return -1;
            }
            return (int)(this.value >>> this.pos) & 0xFF;
        }
    }

    public static class SignedFixedLengthNumber<V>
    implements ByteSource {
        private final ValueAccessor<V> accessor;
        private final V data;
        private int bufpos;

        public SignedFixedLengthNumber(ValueAccessor<V> accessor, V data) {
            this.accessor = accessor;
            this.data = data;
            this.bufpos = 0;
        }

        @Override
        public int next() {
            if (this.bufpos >= this.accessor.size(this.data)) {
                return -1;
            }
            int v = this.accessor.getByte(this.data, this.bufpos) & 0xFF;
            if (this.bufpos == 0) {
                v ^= 0x80;
            }
            ++this.bufpos;
            return v;
        }
    }

    public static class MemoryEscaper
    extends AbstractEscaper {
        private final long address;
        private final int length;

        MemoryEscaper(long address, int length, ByteComparable.Version version) {
            super(0, version);
            this.address = address;
            this.length = length;
        }

        @Override
        protected byte get(int index) {
            return MemoryUtil.getByte(this.address + (long)index);
        }

        @Override
        protected int limit() {
            return this.length;
        }
    }

    public static class ArrayEscaper
    extends AbstractEscaper {
        private final byte[] buf;

        private ArrayEscaper(byte[] buf, ByteComparable.Version version) {
            super(0, version);
            this.buf = buf;
        }

        @Override
        protected byte get(int index) {
            return this.buf[index];
        }

        @Override
        protected int limit() {
            return this.buf.length;
        }
    }

    public static class BufferEscaper
    extends AbstractEscaper {
        private final ByteBuffer buf;

        private BufferEscaper(ByteBuffer buf, ByteComparable.Version version) {
            super(buf.position(), version);
            this.buf = buf;
        }

        @Override
        protected int limit() {
            return this.buf.limit();
        }

        @Override
        protected byte get(int index) {
            return this.buf.get(index);
        }
    }

    public static class AccessorEscaper<V>
    extends AbstractEscaper {
        private final V data;
        private final ValueAccessor<V> accessor;

        private AccessorEscaper(ValueAccessor<V> accessor, V data, ByteComparable.Version version) {
            super(0, version);
            this.accessor = accessor;
            this.data = data;
        }

        @Override
        protected int limit() {
            return this.accessor.size(this.data);
        }

        @Override
        protected byte get(int index) {
            return this.accessor.getByte(this.data, index);
        }
    }

    public static abstract class AbstractEscaper
    implements ByteSource {
        private final ByteComparable.Version version;
        private int bufpos;
        private boolean escaped;

        AbstractEscaper(int position, ByteComparable.Version version) {
            this.bufpos = position;
            this.version = version;
        }

        @Override
        public final int next() {
            if (this.bufpos >= this.limit()) {
                if (this.bufpos > this.limit()) {
                    return -1;
                }
                ++this.bufpos;
                if (this.escaped) {
                    this.escaped = false;
                    if (this.version == ByteComparable.Version.LEGACY) {
                        --this.bufpos;
                    }
                    return 254;
                }
                return 0;
            }
            int index = this.bufpos++;
            int b = this.get(index) & 0xFF;
            if (!this.escaped) {
                if (b == 0) {
                    this.escaped = true;
                }
                return b;
            }
            if (b == 0) {
                return 254;
            }
            --this.bufpos;
            this.escaped = false;
            return 255;
        }

        protected abstract byte get(int var1);

        protected abstract int limit();
    }
}

