/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.config;

import com.google.common.primitives.Ints;
import java.util.Arrays;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public abstract class DurationSpec {
    private static final Pattern UNITS_PATTERN = Pattern.compile("^(\\d+)(d|h|s|ms|us|\u00b5s|ns|m)$");
    private final long quantity;
    private final TimeUnit unit;

    private DurationSpec(long quantity, TimeUnit unit, TimeUnit minUnit, long max) {
        this.quantity = quantity;
        this.unit = unit;
        DurationSpec.validateMinUnit(unit, minUnit, quantity + " " + unit);
        DurationSpec.validateQuantity(quantity, unit, minUnit, max);
    }

    private DurationSpec(double quantity, TimeUnit unit, TimeUnit minUnit, long max) {
        this(Math.round(quantity), unit, minUnit, max);
    }

    private DurationSpec(String value, TimeUnit minUnit) {
        Matcher matcher = UNITS_PATTERN.matcher(value);
        if (!matcher.find()) {
            throw new IllegalArgumentException("Invalid duration: " + value + " Accepted units:" + DurationSpec.acceptedUnits(minUnit) + " where case matters and only non-negative values.");
        }
        this.quantity = Long.parseLong(matcher.group(1));
        this.unit = DurationSpec.fromSymbol(matcher.group(2));
    }

    private DurationSpec(String value, TimeUnit minUnit, long max) {
        this(value, minUnit);
        DurationSpec.validateMinUnit(this.unit, minUnit, value);
        DurationSpec.validateQuantity(value, this.quantity(), this.unit(), minUnit, max);
    }

    private static void validateMinUnit(TimeUnit unit, TimeUnit minUnit, String value) {
        if (unit.compareTo(minUnit) < 0) {
            throw new IllegalArgumentException(String.format("Invalid duration: %s Accepted units:%s", value, DurationSpec.acceptedUnits(minUnit)));
        }
    }

    private static String acceptedUnits(TimeUnit minUnit) {
        TimeUnit[] units = TimeUnit.values();
        return Arrays.toString((Object[])Arrays.copyOfRange(units, minUnit.ordinal(), units.length));
    }

    private static void validateQuantity(String value, long quantity, TimeUnit sourceUnit, TimeUnit minUnit, long max) {
        if (minUnit.convert(quantity, sourceUnit) >= max) {
            throw new IllegalArgumentException("Invalid duration: " + value + ". It shouldn't be more than " + (max - 1L) + " in " + minUnit.name().toLowerCase());
        }
    }

    private static void validateQuantity(long quantity, TimeUnit sourceUnit, TimeUnit minUnit, long max) {
        if (quantity < 0L) {
            throw new IllegalArgumentException("Invalid duration: value must be non-negative");
        }
        if (minUnit.convert(quantity, sourceUnit) >= max) {
            throw new IllegalArgumentException(String.format("Invalid duration: %d %s. It shouldn't be more than %d in %s", quantity, sourceUnit.name().toLowerCase(), max - 1L, minUnit.name().toLowerCase()));
        }
    }

    public long quantity() {
        return this.quantity;
    }

    public TimeUnit unit() {
        return this.unit;
    }

    static TimeUnit fromSymbol(String symbol) {
        switch (symbol.toLowerCase()) {
            case "d": {
                return TimeUnit.DAYS;
            }
            case "h": {
                return TimeUnit.HOURS;
            }
            case "m": {
                return TimeUnit.MINUTES;
            }
            case "s": {
                return TimeUnit.SECONDS;
            }
            case "ms": {
                return TimeUnit.MILLISECONDS;
            }
            case "us": 
            case "\u00b5s": {
                return TimeUnit.MICROSECONDS;
            }
            case "ns": {
                return TimeUnit.NANOSECONDS;
            }
        }
        throw new IllegalArgumentException(String.format("Unsupported time unit: %s. Supported units are: %s", symbol, Arrays.stream(TimeUnit.values()).map(DurationSpec::symbol).collect(Collectors.joining(", "))));
    }

    public long to(TimeUnit targetUnit) {
        return targetUnit.convert(this.quantity, this.unit);
    }

    public int hashCode() {
        return Objects.hash(this.unit.toMillis(this.quantity));
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof DurationSpec)) {
            return false;
        }
        DurationSpec other = (DurationSpec)obj;
        if (this.unit == other.unit) {
            return this.quantity == other.quantity;
        }
        return this.unit.convert(other.quantity, other.unit) == this.quantity && other.unit.convert(this.quantity, this.unit) == other.quantity;
    }

    public String toString() {
        return this.quantity + DurationSpec.symbol(this.unit);
    }

    static String symbol(TimeUnit unit) {
        switch (unit) {
            case DAYS: {
                return "d";
            }
            case HOURS: {
                return "h";
            }
            case MINUTES: {
                return "m";
            }
            case SECONDS: {
                return "s";
            }
            case MILLISECONDS: {
                return "ms";
            }
            case MICROSECONDS: {
                return "us";
            }
            case NANOSECONDS: {
                return "ns";
            }
        }
        throw new AssertionError();
    }

    public static final class IntMillisecondsBound
    extends DurationSpec {
        public IntMillisecondsBound(String value) {
            super(value, TimeUnit.MILLISECONDS, Integer.MAX_VALUE);
        }

        public IntMillisecondsBound(long quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.MILLISECONDS, (long)Integer.MAX_VALUE);
        }

        public IntMillisecondsBound(long milliseconds) {
            this(milliseconds, TimeUnit.MILLISECONDS);
        }

        public IntMillisecondsBound(double quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.MILLISECONDS, Integer.MAX_VALUE);
        }

        public int toMilliseconds() {
            return Ints.saturatedCast(this.unit().toMillis(this.quantity()));
        }
    }

    public static final class IntSecondsBound
    extends DurationSpec {
        private static final Pattern VALUES_PATTERN = Pattern.compile("\\d+");

        public IntSecondsBound(String value) {
            super(value, TimeUnit.SECONDS, Integer.MAX_VALUE);
        }

        public IntSecondsBound(long quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.SECONDS, (long)Integer.MAX_VALUE);
        }

        public IntSecondsBound(long seconds) {
            this(seconds, TimeUnit.SECONDS);
        }

        public static IntSecondsBound inSecondsString(String value) {
            Matcher matcher = VALUES_PATTERN.matcher(value);
            if (matcher.matches()) {
                long seconds = Integer.parseInt(value);
                return new IntSecondsBound(seconds, TimeUnit.SECONDS);
            }
            return new IntSecondsBound(value);
        }

        public int toNanoseconds() {
            return Ints.saturatedCast(this.unit().toNanos(this.quantity()));
        }

        public int toMilliseconds() {
            return Ints.saturatedCast(this.unit().toMillis(this.quantity()));
        }

        public int toSeconds() {
            return Ints.saturatedCast(this.unit().toSeconds(this.quantity()));
        }
    }

    public static final class IntMinutesBound
    extends DurationSpec {
        public IntMinutesBound(String value) {
            super(value, TimeUnit.MINUTES, Integer.MAX_VALUE);
        }

        public IntMinutesBound(long quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.MINUTES, (long)Integer.MAX_VALUE);
        }

        public IntMinutesBound(long minutes) {
            this(minutes, TimeUnit.MINUTES);
        }

        public int toMilliseconds() {
            return Ints.saturatedCast(this.unit().toMillis(this.quantity()));
        }

        public int toSeconds() {
            return Ints.saturatedCast(this.unit().toSeconds(this.quantity()));
        }

        public int toMinutes() {
            return Ints.saturatedCast(this.unit().toMinutes(this.quantity()));
        }
    }

    public static final class LongSecondsBound
    extends DurationSpec {
        public LongSecondsBound(String value) {
            super(value, TimeUnit.SECONDS, Long.MAX_VALUE);
        }

        public LongSecondsBound(long quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.SECONDS, Long.MAX_VALUE);
        }

        public LongSecondsBound(long seconds) {
            this(seconds, TimeUnit.SECONDS);
        }

        public long toMilliseconds() {
            return this.unit().toMillis(this.quantity());
        }

        public long toSeconds() {
            return this.unit().toSeconds(this.quantity());
        }
    }

    public static final class LongMillisecondsBound
    extends DurationSpec {
        public LongMillisecondsBound(String value) {
            super(value, TimeUnit.MILLISECONDS, Long.MAX_VALUE);
        }

        public LongMillisecondsBound(long quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.MILLISECONDS, Long.MAX_VALUE);
        }

        public LongMillisecondsBound(long milliseconds) {
            this(milliseconds, TimeUnit.MILLISECONDS);
        }

        public long toMilliseconds() {
            return this.unit().toMillis(this.quantity());
        }
    }

    public static final class LongMicrosecondsBound
    extends DurationSpec {
        public LongMicrosecondsBound(String value) {
            super(value, TimeUnit.MICROSECONDS, Long.MAX_VALUE);
        }

        public LongMicrosecondsBound(long quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.MICROSECONDS, Long.MAX_VALUE);
        }

        public LongMicrosecondsBound(long microseconds) {
            this(microseconds, TimeUnit.MICROSECONDS);
        }

        public long toMicroseconds() {
            return this.unit().toMicros(this.quantity());
        }

        public long toSeconds() {
            return this.unit().toSeconds(this.quantity());
        }
    }

    public static final class LongNanosecondsBound
    extends DurationSpec {
        public LongNanosecondsBound(String value) {
            super(value, TimeUnit.NANOSECONDS, Long.MAX_VALUE);
        }

        public LongNanosecondsBound(long quantity, TimeUnit unit) {
            super(quantity, unit, TimeUnit.NANOSECONDS, Long.MAX_VALUE);
        }

        public LongNanosecondsBound(long nanoseconds) {
            this(nanoseconds, TimeUnit.NANOSECONDS);
        }

        public long toNanoseconds() {
            return this.unit().toNanos(this.quantity());
        }
    }
}

