/*
 * WP database handler : low level mapped file I/O
 *
 * F1OAT 9080117
 */

#include "wpdefs.h"
 
/* Database memory mapping vars */

static int map_size;		/* Size of current memory mapping */
static void *db_map_addr = NULL;/* Pointer to database mmap() file */
static int db_size;		/* Number of records in database */
static int hash_table;		/* Handle returned by hash_open */
static char wp_file[256];	/* Default file */

wp_header *db_header;		/* Database file header */
wp_t *db_records;		/* Database file records */

static void db_create(void)
{
	char *stmp;
	char *ptr;
	FILE *fptr;
	wp_header wph;
	wp_t wp;
	int i;
	
	syslog(LOG_INFO, "Reset database");
		
	/* Create the directories */
	stmp = strdup(wp_file);

	ptr = strchr(stmp+1, '/');
	while (ptr)
	{
		*ptr = '\0';
		mkdir(stmp, 0777);
		*ptr++ = '/';
		ptr = strchr(ptr, '/');
	}

	free (stmp);

	fptr = fopen(wp_file, "w");
	if (fptr == NULL)
		return;

	strcpy(wph.signature, FILE_SIGNATURE);
	wph.nb_record = 0;
	
	if (fwrite(&wph, sizeof(wph), 1, fptr) == 0)
	{
		fclose(fptr);
		unlink(wp_file);
		return;
	}

	/* Allocate some empty records */
	
	memset(&wp, 0, sizeof(wp));
	for (i=0; i<NEW_RECORD_CHUNK; i++) {
		fwrite(&wp, sizeof(wp), 1, fptr);
	}
	
	fclose(fptr);
}

static int db_map(void)
{
	int fd, rc;
	struct stat st;	
	wp_header wph;
	
	/* Get database size */
	
	if (stat(wp_file, &st)) {
		perror(wp_file);
		return -1;
	}
	
	/* Open database and map to memory */
	
	fd = open(wp_file, O_RDWR);
	if (fd < 0) {
		perror(wp_file);
		return -1;
	}

	/* Check header */
	
	rc = read(fd, &wph, sizeof(wph));
	if (rc != sizeof(wph)) {
		syslog(LOG_ERR, "File's header too short");
		close(fd);
		return -2;
	}
	 
	if (strcmp(wph.signature, FILE_SIGNATURE) != 0) {
		syslog(LOG_ERR, "Wrong signature");
		close(fd);
		return -2;		
	}
	
	/* Convert size to number of records (free or not) */
	
	db_size = (st.st_size - sizeof(wp_header))/sizeof(wp_t);
	map_size = sizeof(wp_header) + db_size * sizeof(wp_t);	
	
	/* Check size */
	
	if (wph.nb_record > db_size) {
		syslog(LOG_ERR, "Nb records in db header too large");
		close(fd);;
		return -2;
	}
			
	db_map_addr = mmap(0, map_size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
	if (db_map_addr == (void *)-1) {
		db_map_addr = NULL;
		perror("mmap wp file");
		close(fd);
		return -1;
	}
	
	db_header = (wp_header *)db_map_addr;
	db_records = (wp_t *)(db_header+1);
	close(fd);
		
	syslog(LOG_INFO, "Database size %d / %d used", db_size, db_header->nb_record);
	
	return 0;
}

static void db_unmap(void)
{
	if (db_map_addr) {
		munmap(db_map_addr, map_size);
		db_map_addr = NULL;
		db_header = NULL;
		db_records = NULL;
	}
}

static int db_increase_size(void)
{	
	int fd, i, rc;
	wp_t wp;
	
	db_unmap();
	memset(&wp, 0, sizeof(wp));
	
	fd = open(wp_file, O_RDWR);
	if (fd < 0) {
		perror(wp_file);
		return -1;
	}

	lseek(fd, 0, SEEK_END);
	for (i=0; i<NEW_RECORD_CHUNK; i++) {
		rc = write(fd, &wp, sizeof(wp));
		if (rc != sizeof(wp)) {
			close(fd);
			perror("cannot increase db size");
			return -1;
		}
	}
	close(fd);
			
	return db_map();
}

int db_open(char *file)
{
	int rc;
	int i;
		
	strcpy(wp_file, file);
		
	/* Create database file if necessary */
	
	if (access(wp_file, R_OK|W_OK)) db_create();
	
	/* Map database in memory */
	
	rc = db_map();
	if (rc == -2) {
		db_create();
		rc = db_map();
	}
	if (rc) return rc;
	
	/* Open hash table */
	
	hash_table = hash_open(HASH_BITS, sizeof(wp_t), 
		               offsetof (wp_t, address.srose_call),
		               sizeof (ax25_address),
		               (void **)&db_records);
	if (hash_table < 0) {
		syslog(LOG_ERR, "Cannot create hash table");
		db_unmap();
		return -1;
	}
		               
	/* Index records in hash table */
	
	for (i=0; i<db_header->nb_record; i++) {
		rc = hash_put(hash_table, i);
		if (rc < 0) {
			syslog(LOG_ERR, "Cannot index record #%d", i);
			db_unmap();
			hash_close(hash_table);
			return -1;
		}
	}
	  
	return 0;
}

void db_close(void)
{
	db_unmap();
}

static int db_find(ax25_address *call)
{
	int index;
	
	index = hash_search(hash_table, call);
	return index;
}

int db_get(ax25_address *call, wp_t *wp)
{
	int index;
	
	index = db_find(call);
	if (index < 0) return -1;
	*wp = db_records[index];
	return 0;
}

int db_set(wp_t *wp, int force)
{
	int index;
	int rc;
	
	index = db_find(&wp->address.srose_call);
	if (index < 0) {
		index = (db_header->nb_record)++;
		if (db_header->nb_record > db_size) {
			rc = db_increase_size();
			if (rc < 0) {
				return -1;
			}
		}
		db_records[index] = *wp;
		rc = hash_put(hash_table, index);
	}
	else {
		if (db_records[index].date >= wp->date) return -1;	/* I have newer record */
		if (memcmp(&db_records[index], wp, sizeof(*wp)) == 0) return -1;
		if (!force) {	/* If force==0 and only date different, do nothing */
			wp_t mywp;
			
			mywp = *wp;
			mywp.date = db_records[index].date;
			if (memcmp(&db_records[index], &mywp, sizeof(mywp)) == 0) return -1;
		}
		db_records[index] = *wp;
	}
	
	return index;	
}

static int date_comp(const void *a, const void *b)
{
	unsigned short ia = *(unsigned short *)a, ib = *(unsigned short *)b;
	
	if (db_records[ia].date < db_records[ib].date) return 1;
	if (db_records[ia].date > db_records[ib].date) return -1;
	return 0;
}

/*
 * If vector->date_base == 0, compute best date_base and interval parameters
 *
 * If dirty >= 0, set dirty bits for client context[dirty] according to vector mismatch
 * Dirty flags are set only for the first vector element
 */
 
void db_compute_vector(int dirty, vector_t *vector)
{
	unsigned short *sorted_list;
	int i, vec_index;
	unsigned char crc_buf[4];
	unsigned long treshold;
	unsigned short crc;
	int dirty_first;
	unsigned short mycrc[WP_VECTOR_SIZE];
	int dirty_cnt[WP_VECTOR_SIZE];
	int record_cnt[WP_VECTOR_SIZE];
	char atime[40];
	int stop_dirty = 0;
	
	memset(dirty_cnt, 0, sizeof(dirty_cnt));
	memset(record_cnt, 0, sizeof(record_cnt));
		
	sorted_list = calloc(db_header->nb_record, sizeof(*sorted_list));
	if (!sorted_list) return;
	
	for (i=0; i<db_header->nb_record; i++) {
		sorted_list[i] = i;
	}
	
	/* Sort in date descending order */
	
	qsort(sorted_list, db_header->nb_record, sizeof(*sorted_list), date_comp);
	
	/* Compute best parameters ? */
	
	if (vector->date_base == 0) {
		vector->date_base = db_records[sorted_list[0]].date;
		vector->interval = 1+(vector->date_base - db_records[sorted_list[db_header->nb_record-1]].date) / (1<<WP_VECTOR_SIZE);
	}
	
	strcpy(atime, ctime(&vector->date_base));
	atime[strlen(atime)-1] = 0;
	
	syslog(LOG_INFO, "Vector Base:%s Interval:%.2f min", 
		atime, vector->interval/60.0);
		
	crc16_cumul(NULL, 0);
	*(unsigned short *)crc_buf = htons(vector->seed);
	crc = crc16_cumul(crc_buf, 2);
	
	for (i=0; i<WP_VECTOR_SIZE; i++) mycrc[i] = crc;
	
	vec_index = 0;
	treshold = vector->interval;
	dirty_first = 0;
	
	for (i=0; i<=db_header->nb_record; i++) {
		int last_done = (i == db_header->nb_record);
		int out_timeslot;
		
		/* F6FBB : Debordement de sorted_list si i == db_header->nb_record */
		if (!last_done) {
			out_timeslot = ((vector->date_base > db_records[sorted_list[i]].date) &&
				    (vector->date_base - db_records[sorted_list[i]].date >= treshold));
		}
		
		if (last_done || out_timeslot) {
		     	if (dirty >= 0 && vector->crc[vec_index] != crc) {
				dirty_cnt[vec_index] = record_cnt[vec_index];
				if (!stop_dirty) {
					int k;
					for (k=dirty_first; k<i; k++) {
						set_dirty_context(sorted_list[k], dirty);
					}
					set_vector_when_nodirty(dirty);
					stop_dirty = 1;
				}
			}
			dirty_first = i;
			mycrc[vec_index] = crc;			
			vec_index++;
			if (vec_index == (WP_VECTOR_SIZE-1)) treshold = 0xFFFFFFFF;
			else treshold = treshold *2;
			crc16_cumul(NULL, 0);
			*(unsigned short *)crc_buf = htons(vector->seed);
			crc16_cumul(crc_buf, 2);		
		}
		
		if (!last_done) {
			*(unsigned long *)crc_buf = htonl(db_records[sorted_list[i]].date);
			crc = crc16_cumul(crc_buf, 4);			
			crc = crc16_cumul((unsigned char *)&db_records[sorted_list[i]].address, sizeof(struct sockaddr_rose));
			record_cnt[vec_index] += 1;
		}
	}
			
	for (i=0; i<WP_VECTOR_SIZE; i++) vector->crc[i] = mycrc[i];
	
	free(sorted_list);
	
	{
		char str[256];
		char *p = str;
		p += sprintf(p, "Vector Calculation (dirty/total):");
		for (i=0; i<WP_VECTOR_SIZE; i++) {
			p += sprintf(p, " %d/%d", dirty_cnt[i], record_cnt[i]);
		}
		syslog(LOG_INFO, str);
	}
	
	return;
}
