/*
 *   Copyright 1992, 1993, 1994 John Melton (G0ORX/N6LYT)
 *              All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
	xpg.c

	Pacsat Upload for Linux and X-Windows

	This program has been run using the 1.0 version of the
	Linux kernel with the patches from Alan Cox to provide AX.25
	encapsulation in the SLIP protocol(verion 0.12).

	The TNC must be setup for KISS protocol.

	John Melton
	G0ORX, N6LYT

	4 Charlwoods Close
	Copthorne
	West Sussex
	RH10 3QZ
	England

	INTERNET:	g0orx@amsat.org
			n6lyt@amsat.org
			john@images.demon.co.uk
			J.D.Melton@slh0613.icl.wins.co.uk

	History:

	0.1	initial version.			G0ORX
	0.2	added output ready notify code.		G0ORX
	0.3	fixed error conditions - back to IDLE.	G0ORX
	0.4	fixed restart upload of partial file.	G0ORX
	0.5	converted to Xaw.			G4KLX
		converted program to use a child process to upload file.
	0.6	added double click for upload.		G4KLX
*/

#define VERSION_STRING "(version 0.6 by g0orx/n6lyt/g4klx)"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/socket.h>
#include <signal.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Viewport.h>

#include "xawutils.h"
#include "header.h"
#include "ftl0.h"

Display *dpy;

XtAppContext app_context;

typedef struct
{
	XFontStruct *bold_font, *button_font, *menu_font, *label_font, *list_font;
}
Resources;

Resources  resources;

Widget toplevel, compwindow, quitbutton, updatebutton, messagebutton,
	statuslabel, listlabel, viewport, uploadlist;

static XtResource resource_list[] =
{
	{"boldFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, bold_font), XtRString, XtDefaultFont},
	{"buttonFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, button_font), XtRString, XtDefaultFont},
	{"menuFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, menu_font), XtRString, XtDefaultFont},
	{"labelFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, label_font), XtRString, XtDefaultFont},
	{"listFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffsetOf(Resources, list_font), XtRString, XtDefaultFont}
};

static Arg shell_args[] =
{
	{XtNtitle,		(XtArgVal)NULL}
};

static Arg form_args[] =
{
	{XtNdefaultDistance,	(XtArgVal)0}
};

static Arg label_args[] =
{
	{XtNfromVert,		(XtArgVal)NULL},
	{XtNfromHoriz,		(XtArgVal)NULL},
	{XtNlabel,		(XtArgVal)NULL},
	{XtNwidth,		(XtArgVal)0},
	{XtNfont,		(XtArgVal)NULL},
	{XtNborderWidth,	(XtArgVal)0},
	{XtNresize,		False},
	{XtNjustify,		XtJustifyLeft},
	{XtNvertDistance,	(XtArgVal)6},
	{XtNhorizDistance,	(XtArgVal)8},
	{XtNtop,		XtChainTop},
	{XtNbottom,		XtChainTop},
	{XtNleft,		XtChainLeft},
	{XtNright,		XtChainLeft}
};

static Arg button_args[] =
{
	{XtNcallback,		(XtArgVal)NULL},
	{XtNlabel,		(XtArgVal)NULL},
	{XtNfont,		(XtArgVal)NULL},
	{XtNfromHoriz,		(XtArgVal)NULL},
	{XtNfromVert,		(XtArgVal)NULL},
	{XtNvertDistance,	(XtArgVal)6},
	{XtNhorizDistance,	(XtArgVal)8},
	{XtNresize,		False},
	{XtNtop,		XtChainTop},
	{XtNbottom,		XtChainTop},
	{XtNleft,		XtChainLeft},
	{XtNright,		XtChainLeft}
};

static Arg viewport_args[] =
{
	{XtNfromVert,		(XtArgVal)NULL},
	{XtNwidth,		(XtArgVal)600},
	{XtNheight,		(XtArgVal)150},
	{XtNforceBars,		True},
	{XtNallowVert,		True},
	{XtNresize,		True},
	{XtNvertDistance,	(XtArgVal)0},
	{XtNhorizDistance,	(XtArgVal)0},
	{XtNtop,		XtChainTop},
	{XtNbottom,		XtChainBottom},
	{XtNleft,		XtChainLeft},
	{XtNright,		XtChainRight}
};

static Arg list_args[] =
{
	{XtNcallback,		(XtArgVal)NULL},
	{XtNfont,		(XtArgVal)NULL}
};

#define	MAX_DIR_LIST_SIZE	2000

String dList[MAX_DIR_LIST_SIZE + 1];
int    nList = 0;

int lastIndex = 0;

char satellite[16];
char myCall[16];

XtInputId infd;
int fd[2];
int uploadpid = 0;
int maxFrame;
int T1;

extern void upload(char *);
	
void QuitCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	if (uploadpid != 0)
	{
		kill(uploadpid, SIGINT);
	}

	XtDestroyApplicationContext(app_context);
	
	exit(0);
}

void LoadFile(char * fileName)
{
	FILE *hFile;
	int nBytes, j;
	char *pBuffer, *p;
	char szTemp[128];
	int headerSize;
	HEADER *pHeader;
	int fileNumber;

	sscanf(fileName, "%x", &fileNumber);

	/* open the file */
	if ((hFile = fopen(fileName, "r")) == NULL)
	{
		perror(fileName);
		return;
	}

	/* extracting the header */
	pBuffer = XtMalloc(1024);

	nBytes = fread(pBuffer, 1, 1024, hFile);

	fclose(hFile);

	if ((pHeader = ExtractHeader(pBuffer, nBytes, &headerSize)) == NULL)
	{
		printf("invalid header entry for file: %s\n", fileName);
	}
	else
	{
		/* truncate the source and destination */
		for (j = 0; isalnum(pHeader->source[j]); j++);
      		pHeader->source[j] = '\0';

		for (j = 0; isalnum(pHeader->destination[j]); j++);
      		pHeader->destination[j] = '\0';

		if (strlen(pHeader->title) == 0)
			sprintf(szTemp, "%12s %8lx %-8s %6ld %-45s",
				fileName, pHeader->fileId, pHeader->destination,
				pHeader->fileSize, pHeader->fileName);
		else
			sprintf(szTemp, "%12s %8lx %-8s %6ld %-45s",
				fileName, pHeader->fileId, pHeader->destination,
				pHeader->fileSize, pHeader->title);

		for (p = pHeader->source; *p != '\0'; p++)
			if (islower(*p)) *p = toupper(*p);

		for (p = pHeader->destination; *p != '\0'; p++)
			if (islower(*p)) *p = toupper(*p);

		dList[nList] = XtNewString(szTemp);
		nList++;

		XtFree((char *)pHeader);
	}

	XtFree(pBuffer);
}

void Update(void)
{
	DIR		*pDir;
	struct dirent	*pDirent;
	int i;
	Arg args[2];

	for (i = 0; i < nList; i++)
		if (dList[i] != NULL)
			XtFree(dList[i]);

	nList     = 0;
	lastIndex = -1;

	/* walk through the directory of files */
	pDir = opendir(".");

	while ((pDirent = readdir(pDir)) != NULL)
	{
		/* see if it is a .upl file */
		if (strstr(&pDirent->d_name[i], ".upl") != NULL)
			LoadFile(pDirent->d_name);
	}

	closedir(pDir);

	dList[nList] = NULL;

	XawListChange(uploadlist, dList, nList, 0, False);

	if (nList > 0) XawListHighlight(uploadlist, 0);

	XtSetArg(args[0], XtNheight, 0);
	XtSetArg(args[1], XtNwidth,  0);
	XtSetValues(uploadlist, args, TWO);
}

void UpdateCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	Update();
}

void ReceivedStatus(XtPointer closure, int *s, XtInputId *Id)
{
	int bytes;
	char msg[200];
	
	bytes = read(fd[0], msg, 200);

	XtVaSetValues(statuslabel, XtNlabel, msg, NULL);
}

void SelectCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *selected;
	unsigned long FileId;
	char fileName[30];

	if (uploadpid != 0)
	{
		MessageBox("Upload already in progress");
		return;
	}

	selected = (XawListReturnStruct *)call_data;
	
	if (selected->list_index != lastIndex)
	{
		lastIndex = selected->list_index;
		return;
	}

	lastIndex = -1;
		
	sscanf(selected->string, "%lx", &FileId);
	sprintf(fileName, "%lx.upl", FileId);

	pipe(fd);

	if ((uploadpid = fork()) == 0)
	{
		/* the child process */
		upload(fileName);
	}
	else if (uploadpid > 0)
	{
		/* success */
		infd = XtAppAddInput(app_context, fd[0], (XtPointer)XtInputReadMask,
				     ReceivedStatus, NULL);
	}
	else
	{
		MessageBox("Cannot fork upload");
	}
}

void signal_child(int s)
{
	int pstatus;

	signal(SIGCHLD, signal_child);

	if (wait(&pstatus) == uploadpid)
	{
		XtRemoveInput(infd);
		uploadpid = 0;
	}

	Update();
}

void MessageCb(Widget w, XtPointer client_data, XtPointer call_data)
{
	int pid;

	if ((pid = fork()) == 0)
	{
		/* the child process */
		execlp("message", "message", NULL);
	}
	else if (pid > 0)
	{
		/* success */
	}
	else
	{
		MessageBox("Cannot fork message");
	}
}

int main(int argc, char **argv)
{
	static XtCallbackRec callback[2];
	char title[80], *s;

	signal(SIGCHLD, signal_child);

	if ((s = getenv("SATELLITE")) == NULL)
	{
		printf("SATELLITE environment variable not set.\n");
		return(1);
	}
	
	strcpy(satellite, s);

	if ((s = getenv("MYCALL")) == NULL)
	{
		printf("MYCALL environment variable not set.\n");
		return(1);
	}

	strcpy(myCall, s);
	
	if ((s= getenv("MAXFRAMEDATA")) == NULL)
		maxFrame = 200;
	else
		maxFrame = atoi(s);

	if (maxFrame <= 0 || maxFrame >= 256) maxFrame = 200;

	if ((s = getenv("T1")) == NULL)
		T1 = 20;
	else
		T1 = atoi(s);

	if (T1 <= 0) T1 = 20;

	sprintf(title, "xpg:%s %s", satellite, VERSION_STRING);

	strcat(satellite, "-12");

	toplevel = XtAppInitialize(&app_context, "Xpb", NULL, 0, &argc, argv,
				NULL, shell_args, XtNumber(shell_args));

	XtVaSetValues(toplevel, XtNtitle, title, NULL);

	dpy  = XtDisplay(toplevel);

	XtGetApplicationResources(toplevel, &resources,
				resource_list, XtNumber(resource_list),
				NULL, ZERO);

	compwindow = XtCreateManagedWidget("appForm", formWidgetClass,
				toplevel, form_args, XtNumber(form_args));

	callback[0].callback = QuitCb;
	callback[0].closure  = toplevel;
	button_args[0].value = (XtArgVal)callback;
	button_args[1].value = (XtArgVal)"Quit";
	button_args[2].value = (XtArgVal)resources.button_font;
	quitbutton = XtCreateManagedWidget("quitButton", commandWidgetClass,
				compwindow, button_args, XtNumber(button_args));

	callback[0].callback = UpdateCb;
	callback[0].closure  = toplevel;
	button_args[0].value = (XtArgVal)callback;
	button_args[1].value = (XtArgVal)"Update";
	button_args[3].value = (XtArgVal)quitbutton;
	updatebutton = XtCreateManagedWidget("updateButton", commandWidgetClass,
				compwindow, button_args, XtNumber(button_args));

	callback[0].callback = MessageCb;
	callback[0].closure  = toplevel;
	button_args[0].value = (XtArgVal)callback;
	button_args[1].value = (XtArgVal)"Message";
	button_args[3].value = (XtArgVal)updatebutton;
	messagebutton = XtCreateManagedWidget("messageButton", commandWidgetClass,
				compwindow, button_args, XtNumber(button_args));

	label_args[0].value = (XtArgVal)NULL;
	label_args[1].value = (XtArgVal)messagebutton;
	label_args[2].value = (XtArgVal)"Status: Idle";
	label_args[3].value = (XtArgVal)350;
	label_args[4].value = (XtArgVal)resources.button_font;
	statuslabel = XtCreateManagedWidget("statusLabel", labelWidgetClass,
				compwindow, label_args, XtNumber(label_args));

	label_args[0].value = (XtArgVal)quitbutton;
	label_args[1].value = (XtArgVal)NULL;
	label_args[2].value = (XtArgVal)" File               Id To         Size Title";
	label_args[3].value = (XtArgVal)500;
	label_args[4].value = (XtArgVal)resources.list_font;
	listlabel = XtCreateManagedWidget("listLabel", labelWidgetClass,
				compwindow, label_args, XtNumber(label_args));

	viewport_args[0].value = (XtArgVal)listlabel;
	viewport = XtCreateManagedWidget("uploadViewport", viewportWidgetClass,
				compwindow, viewport_args, XtNumber(viewport_args));
		
	callback[0].callback = SelectCb;
	callback[0].closure  = toplevel;
	list_args[0].value = (XtArgVal)callback;
	list_args[1].value = (XtArgVal)resources.list_font;
	uploadlist = XtCreateManagedWidget("uploadList", listWidgetClass,
				viewport, list_args, XtNumber(list_args));

	createMessagePopup(resources.bold_font, resources.button_font);

	Update();

	XtRealizeWidget(toplevel);

	XtAppMainLoop(app_context);

	return(0);
}
