#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <linux/ax25.h>
#include <linux/icmp.h>
#include <linux/ip.h>
#include <asm/checksum.h>
#include <netinet/protocols.h>

#include "node.h"
#include "io.h"
#include "axutils.h"
#include "axconfig.h"
#include "nrconfig.h"
#include "procutils.h"

/*
 * Initiate a AX.25, NET/ROM or TCP connection to the host specified
 * by `address'.
 */
static int connect_to(char *address[], int family)
{
	int fd;
	fd_set read_fdset;
	fd_set write_fdset;
  	int addrlen;
  	struct full_sockaddr_ax25 addr;
  	struct sockaddr_in *inaddr = NULL;
	char call[10], path[20], *cp, *eol;
	int ret, retlen = sizeof(int);
	int window, paclen;
	struct hostent *hp;
	struct servent *sp;
	struct proc_nr_nodes *np;

	strcpy(call, User.call);
	/*
	 * Fill in protocol specific stuff.
	 */
	switch (family) {
	case AF_NETROM:
		if (check_perms(PERM_NETROM, 0L) == -1) {
			node_perror(NULL, EACCES);
			return -1;
		}
		if ((fd = socket(AF_NETROM, SOCK_SEQPACKET, 0)) < 0) {
			node_perror("socket", errno);
			return -1;
		}
		/* Why on earth is this different from ax.25 ????? */
		sprintf(path, "%s %s", nr_config_get_addr(NULL), call);
		convert_call(path, &addr);
		addrlen = sizeof(struct full_sockaddr_ax25);
		if (bind(fd, (struct sockaddr *)&addr, addrlen) == -1) {
			node_perror("bind", errno);
			close(fd);
			return -1;
		}
		if ((np = find_node(address[0], NULL)) == NULL) {
			tprintf("%s} No such node\n", NrId);
			return -1;
		}
		strcpy(User.dl_name, print_node(np->alias, np->call));
		if (convert_call(np->call, &addr) == -1) {
			close(fd);
			return -1;
		}
		addrlen = sizeof(struct sockaddr_ax25);
		paclen = nr_config_get_paclen(NULL);
		eol = NETROM_EOL;
		break;
	case AF_AX25:
		/*
		 * Use users call with inverted ssid in outgoing connects.
		 */
		if ((cp = strchr(User.call, '-')) != NULL) {
		        *cp = 0;
			sprintf(call, "%s-%d", User.call, 15 - atoi(cp + 1));
			*cp = '-';
		} else {
		        sprintf(call, "%s-15", User.call);
		}
		if (check_perms(PERM_AX25, 0L) == -1 ||
		    (is_hidden(address[0]) && check_perms(PERM_HIDDEN, 0L) == -1)) {
			node_perror(NULL, EACCES);
			return -1;
		}
                if ((window = ax25_config_get_window(address[0])) == 0) {
                        tprintf("%s} Invalid port\n", NrId);
                        return -1;
		}
		if ((fd = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			node_perror("socket", errno);
			return -1;
		}
		if (setsockopt(fd, SOL_AX25, AX25_WINDOW, &window, sizeof(window)) == -1) {
			node_perror("AX25_WINDOW", errno);
			close(fd);
			return -1;
		}
		sprintf(path, "%s %s", call, ax25_config_get_addr(address[0]));
		convert_call(path, &addr);
		addrlen = sizeof(struct full_sockaddr_ax25);
		if (bind(fd, (struct sockaddr *)&addr, addrlen) == -1) {
			node_perror("bind", errno);
			close(fd);
			return -1;
		}
		if (convert_call_arglist(&address[1], &addr) == -1) {
			close(fd);
			return -1;
		}
		strcpy(User.dl_name, strupr(address[1]));
		strcpy(User.dl_port, strupr(address[0]));
		addrlen = sizeof(struct full_sockaddr_ax25);
                paclen = ax25_config_get_paclen(address[0]);
		eol = AX25_EOL;
		break;
	case AF_INET:
		if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
			node_perror("socket", errno);
			return -1;
		}
		if ((hp = gethostbyname(address[0])) == NULL) {
			tprintf("%s} Unknown host %s\n", NrId, address[0]);
			close(fd);
			return -1;
		}
		inaddr = (struct sockaddr_in *)&addr;
		inaddr->sin_family = AF_INET;
		inaddr->sin_addr.s_addr = ((struct in_addr *)(hp->h_addr))->s_addr;
		sp = NULL;
		if (address[1] == NULL)
			sp = getservbyname("telnet", "tcp");
		if (sp == NULL)
			sp = getservbyname(address[1], "tcp");
		if (sp == NULL)
			sp = getservbyport(htons(atoi(address[1])), "tcp");
		if (sp != NULL) {
			inaddr->sin_port = sp->s_port;
		} else if (atoi(address[1]) != 0) {
			inaddr->sin_port = htons(atoi(address[1]));
		} else {
			tprintf("%s} Unknown service %s\n", NrId, address[1]);
			close(fd);
			return -1;
		}
		strcpy(User.dl_name, inet_ntoa(inaddr->sin_addr));
		if (sp != NULL)
			strcpy(User.dl_port, sp->s_name);
		else
			sprintf(User.dl_port, "%d", ntohs(inaddr->sin_port));
		addrlen = sizeof(struct sockaddr_in);
		paclen = 1024;
		eol = INET_EOL;
		if (check_perms(PERM_TELNET, inaddr->sin_addr.s_addr) == -1) {
			node_perror(NULL, EACCES);
			close(fd);
			return -1;
		}
		break;
	default:
		tprintf("%s} Unsupported address family\n", NrId);
		return -1;
	}
	tprintf("%s} Trying %s... Type <RETURN> to abort\n", NrId, User.dl_name);
	usflush(User.fd);
	/*
	 * Ok. Now set up a non-blocking connect...
	 */
	if (fcntl(fd, F_SETFL, O_NONBLOCK) == -1) {
		node_perror("fnctl - fd", errno);
		close(fd);
		return -1;
	}
	if (fcntl(User.fd, F_SETFL, O_NONBLOCK) == -1) {
		node_perror("fnctl - stdin", errno);
		close(fd);
		return -1;
	}
	if (connect(fd, (struct sockaddr *)&addr, addrlen) == -1 && errno != EINPROGRESS) {
		node_perror("connect", errno);
		close(fd);
		return -1;
	}
	User.dl_type = family;
	User.state = STATE_TRYING;
	update_user();
	/*
	 * ... and wait for it to finish (or user to abort).
	 */
	while (1) {
		FD_ZERO(&read_fdset);
		FD_ZERO(&write_fdset);
		FD_SET(fd, &write_fdset);
		FD_SET(User.fd, &read_fdset);
		if (select(fd + 1, &read_fdset, &write_fdset, 0, 0) == -1) {
			node_perror("select", errno);
			break;
		}
		if (FD_ISSET(fd, &write_fdset)) {
			/* See if we got connected or if this was an error */
			getsockopt(fd, SOL_SOCKET, SO_ERROR, &ret, &retlen);
			if (ret != 0) {
				node_perror(NULL, -ret);
				close(fd);
				return -1;
			}
			break;
		}
		if (FD_ISSET(User.fd, &read_fdset)) {
			if (readline(User.fd) != NULL) {
				tprintf("%s} Aborted\n", NrId);
				close(fd);
				return -1;
			} else if (errno != EAGAIN) {
				close(fd);
				return -1;
			}
		}
	}
	tprintf("%s} Connected to %s\n", NrId, User.dl_name);
	usflush(User.fd);
	if (init_io(fd, paclen, eol) == -1) {
		node_perror("Initializing I/O failed", errno);
		close(fd);
		return -1;
	}
	/* If EOL-conventions are compatible switch to binary mode */
	if (family == User.ul_type ||
	    (family == AF_AX25 && User.ul_type == AF_NETROM) ||
	    (family == AF_NETROM && User.ul_type == AF_AX25)) {
		set_eolmode(fd, EOLMODE_BINARY);
		set_eolmode(User.fd, EOLMODE_BINARY);
	}
	if (family == AF_INET)
		set_telnetmode(fd, 1);
	User.state = STATE_CONNECTED;
	update_user();
  	return fd;
}

int do_connect(int argc, char **argv)
{
	int fd, c, family, stay;
	fd_set fdset;

	stay = ReConnectTo;
	if (!strcasecmp(argv[argc - 1], "s")) {
		stay = 1;
		argv[--argc] = NULL;
	} else if (!strcasecmp(argv[argc - 1], "d")) {
		stay = 0;
		argv[--argc] = NULL;
	}
	if (argc < 2) {
		if (*argv[0] == 't')
			tprintf("%s} Usage: telnet <host> [<port>] [d|s]\n", NrId);
		else
			tprintf("%s} Usage: connect [<port>] <call> [via <call1> ...] [d|s]\n", NrId);
		return 0;
	}
	family = (*argv[0] == 't') ? AF_INET : (argc == 2) ? AF_NETROM : AF_AX25;
	if ((fd = connect_to(++argv, family)) == -1) {
		set_eolmode(User.fd, EOLMODE_TEXT);
		if (fcntl(User.fd, F_SETFL, 0) == -1)
			node_perror("fnctl - stdin", errno);
		return 0;
	}
	while (1) {
		FD_ZERO(&fdset);
		FD_SET(fd, &fdset);
		FD_SET(User.fd, &fdset);
		if (select(fd + 1, &fdset, 0, 0, 0) == -1) {
			node_perror("select", errno);
			break;
		}
		if (FD_ISSET(fd, &fdset)) {
			alarm(ConnTimeout);
			while((c = usgetc(fd)) != -1)
				usputc(c, User.fd);
			if (errno != EAGAIN) {
				if (errno && errno != ENOTCONN)
					node_perror(NULL, errno);
				break;
			}
		}
		if (FD_ISSET(User.fd, &fdset)) {
			alarm(ConnTimeout);
			while((c = usgetc(User.fd)) != -1)
				usputc(c, fd);
			if (errno != EAGAIN) {
				stay = 0;
				break;
			}
		}
		usflush(fd);
		usflush(User.fd);
	}
	end_io(fd);
	close(fd);
	if (stay) {
		set_eolmode(User.fd, EOLMODE_TEXT);
		if (fcntl(User.fd, F_SETFL, 0) == -1)
			node_perror("fnctl - stdin", errno);
		tprintf("%s} Reconnected to %s\n", NrId, NrId);
	} else
		do_bye(0, NULL);
	return 0;
}

int do_finger(int argc, char **argv)
{
	int fd, c;
	char *name, *addr[3], *defaulthost, *cp;

	if (tolower(*argv[0]) == 'c') {
		if (argc < 2) {
			tprintf("%s} Usage: callbook <call>[@<server>]\n", NrId);
			return 0;
		}
		defaulthost = CallServer;
		addr[1] = "1235";
	} else {
		defaulthost = "localhost";
		addr[1] = "finger";
	}
	if (argc < 2) {
		name = "";
		addr[0] = defaulthost;
	} else if ((cp = strchr(argv[1], '@')) != NULL) {
		*cp = 0;
		name = argv[1];
		addr[0] = ++cp;
	} else {
		name = argv[1];
		addr[0] = defaulthost;
	}
	addr[2] = NULL;
	if ((fd = connect_to(addr, AF_INET)) != -1) {
		if (fcntl(fd, F_SETFL, 0) == -1)
			node_perror("fnctl - fd", errno);
		init_io(fd, 1024, INET_EOL);
		usprintf(fd, "%s\n", name);
		usflush(fd);
		while((c = usgetc(fd)) != -1)
			usputc(c, User.fd);
		end_io(fd);
		close(fd);
		tprintf("%s} Reconnected to %s\n", NrId, NrId);
	}
	set_eolmode(User.fd, EOLMODE_TEXT);
	if (fcntl(User.fd, F_SETFL, 0) == -1)
		node_perror("fnctl - stdin", errno);
	return 0;
}

/*
 * Returns difference of tv1 and tv2 in milliseconds.
 */
static long calc_rtt(struct timeval tv1, struct timeval tv2)
{
	struct timeval tv;

	tv.tv_usec = tv1.tv_usec - tv2.tv_usec;
	tv.tv_sec = tv1.tv_sec - tv2.tv_sec;
	if (tv.tv_usec < 0) {
		tv.tv_sec -= 1L;
		tv.tv_usec += 1000000L;
	}
	return ((tv.tv_sec * 1000L) + (tv.tv_usec / 1000L));
}

/*
 * Checksum routine for Internet Protocol family headers (C Version)
 */
static unsigned short in_cksum(unsigned char *addr, int len)
{
        register int nleft = len;
        register unsigned char *w = addr;
        register unsigned int sum = 0;
        unsigned short answer = 0;

        /*
         * Our algorithm is simple, using a 32 bit accumulator (sum), we add
         * sequential 16 bit words to it, and at the end, fold back all the
         * carry bits from the top 16 bits into the lower 16 bits.
         */
        while (nleft > 1)  {
                sum += (*(w + 1) << 8) + *(w);
		w     += 2;
                nleft -= 2;
	}

        /* mop up an odd byte, if necessary */
        if (nleft == 1) {
                sum += *w;
	}

        /* add back carry outs from top 16 bits to low 16 bits */
        sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
        sum += (sum >> 16);                     /* add carry */
        answer = ~sum;                          /* truncate to 16 bits */
        return answer;
}

int do_ping(int argc, char **argv)
{
	static int sequence = 0;
	unsigned char buf[256];
	struct hostent *hp;
	struct sockaddr_in to, from;
	struct protoent *prot;
	struct icmphdr *icp;
	struct timeval tv1, tv2;
	struct iphdr *ip;
	fd_set fdset;
	int fd, i, id, len = sizeof(struct icmphdr);
	int salen = sizeof(struct sockaddr);

	if (argc < 2) {
		tprintf("%s} Usage: ping <host> [<size>]\n", NrId);
		return 0;
	}
	if (argc > 2) {
		len = atoi(argv[2]) + sizeof(struct icmphdr);
		if (len > 256) {
			tprintf("%s} Maximum length is %d\n", NrId, 256 - sizeof(struct icmphdr));
			return 0;
		}
	}
	if ((hp = gethostbyname(argv[1])) == NULL) {
		tprintf("%s} Unknown host %s\n", NrId, argv[1]);
		return 0;
	}
	memset(&to, 0, sizeof(to));
	to.sin_family = AF_INET;
	to.sin_addr.s_addr = ((struct in_addr *)(hp->h_addr))->s_addr;
	if ((prot = getprotobyname("icmp")) == NULL) {
		tprintf("%s} Unknown protocol icmp\n", NrId);
                return 0;
	}
	if ((fd = socket(AF_INET, SOCK_RAW, prot->p_proto)) == -1) {
		node_perror("socket", errno);
		return 0;
	}
	tprintf("%s} Pinging %s... Type <RETURN> to abort\n", NrId, inet_ntoa(to.sin_addr));
	usflush(User.fd);
	strcpy(User.dl_name, inet_ntoa(to.sin_addr));
	User.dl_type = AF_INET;
	User.state = STATE_PINGING;
	update_user();
	/*
	 * Fill the data portion (if any) with some garbage.
	 */
	for (i = sizeof(struct icmphdr); i < len; i++)
		buf[i] = (i - sizeof(struct icmphdr)) & 0xff;
	/*
	 * Fill in the icmp header.
	 */
	id = getpid() & 0xffff;
	icp = (struct icmphdr *)buf;
	icp->type = ICMP_ECHO;
	icp->code = 0;
	icp->checksum = 0;
	icp->un.echo.id = id;
	icp->un.echo.sequence = sequence++;
	/*
	 * Calculate checksum.
	 */
	icp->checksum = in_cksum(buf, len);
	/*
	 * Take the time and send the packet.
	 */
	gettimeofday(&tv1, NULL);
	if (sendto(fd, buf, len, 0, (struct sockaddr *)&to, salen) != len) {
		node_perror("sendto", errno);
		close(fd);
		return 0;
	}
	/*
	 * Now wait for it to come back (or user to abort).
	 */
	if (fcntl(User.fd, F_SETFL, O_NONBLOCK) == -1) {
		node_perror("fnctl - stdin", errno);
		close(fd);
		return 0;
	}
	while (1) {
                FD_ZERO(&fdset);
                FD_SET(fd, &fdset);
                FD_SET(User.fd, &fdset);
                if (select(fd + 1, &fdset, 0, 0, 0) == -1) {
                        node_perror("select", errno);
                        break;
		}
                if (FD_ISSET(fd, &fdset)) {
			if ((len = recvfrom(fd, buf, 256, 0, (struct sockaddr *)&from, &salen)) == -1) {
				node_perror("recvfrom", errno);
				break;
			}
			gettimeofday(&tv2, NULL);
			ip = (struct iphdr *)buf;
			/* Is it long enough? */
			if (len >= (ip->ihl << 2) + sizeof(struct icmphdr)) {
				len -= ip->ihl << 2;
				icp = (struct icmphdr *)(buf + (ip->ihl << 2));
				/* Is it ours? */
				if (icp->type == ICMP_ECHOREPLY && icp->un.echo.id == id && icp->un.echo.sequence == sequence - 1) {
					tprintf("%s} %s rtt: %ldms\n", NrId, inet_ntoa(from.sin_addr), calc_rtt(tv2, tv1));
					break;
				}
			}
		}
                if (FD_ISSET(User.fd, &fdset)) {
			if (readline(User.fd) != NULL) {
				tprintf("%s} Aborted\n", NrId);
				break;
			} else if (errno != EAGAIN) {
				break;
			}
		}
	}
	if (fcntl(User.fd, F_SETFL, 0) == -1)
		node_perror("fnctl - stdin", errno);
	close(fd);
	return 0;
}

