/**********************************************************************
 *  MBIBMIO.C - 12/29/86 - System dependent  I/O functions.
 *  (split out from MBIBM.C 12/29/86)
 *****************************************************************
 *   RATS - The Radio Amateurs Telecommunications Society
 *			presents
 *        [PRMBS] - Packet Radio MailBox System
 *        created and written by Brian B. Riley, KA2BQE
 *        with Dave Trulli, NN2Z
 *
 *  All code contained herein is copyrighted by Brian B. Riley,
 *  STORMYLEA Ltd.,  Jan 1987, except where specific credit is
 *  given in the procedure and or module headers for materials
 *  gained from other sources
 *
 *  This code is freely given into the public domain for any and
 *  all uses by such persons as so desire, with the proviso that
 *  this code and subsequent code based on large fragments of it
 *  must be distributed freely with the same proviso and copies of
 *  this copyright notice.
 *
 *  The only rights to compensation anyone shall have for providing
 *  this code or major recognizeable fragments thereof is for 
 *  reasonable reimbursement for expense of delivery, to include 
 *  telephone charges, media/package costs and/or postage.
 *
 *					Brian B. Riley, ka2bqe
 *					Indian Mills, New Jersey
 *
 *---------------------------------------------------------------------
 * Uses interrupt serial I/O routines in MBBIOS.COM by AA4RE
 * Uses brkoff and brkon to disable/enable ctrl-brk by VE3GYQ
 * Uses COMM int 14 routines from NN2Z modified by KA2BQE
 **********************************************************************
 * inchar()	- Return character from current input.
 * instat()	- Return non-zero if a character waits at current input.
 * ioinit()	- Initialize the serial I/O system.
 * iooff()	- restore break, turn off COMn interrupts
 * outchar()	- Put the character out the proper port,
 * port_stat()  - get port status via int14
 * usr_gone()   - if available checks for a hardware (this case DCD)
 *	          indication of the user having gone bye-bye.
 *
 **********************************************************************
 *		INT 14h - command 3 READ STATUS
 *
 *  BIT		Port Status - hi reg (AH)	Modem status - lo reg (AL)
 *  ----------------------------------------------------------------------
 *  bit 7	timed out			rlsd (DCD) carrier detect
 *  bit 6	xmt shift reg empty		ring indicator (RI)
 *  bit 5	xmt hold reg empty		data set ready (DSR)
 *  bit 4	break detected			clear to send (CTS)
 *  bit 3	framing error			change in rlsd
 *  bit 2	parity error			trailing edge RI
 *  bit 1	overrun error			change in DSR
 *  bit 0	data ready (input)		change in CTS
 **********************************************************************/
#include "mb.h"
#include <dos.h>

char comin();
int  havedcd(), comstat(), comtxrdy();
void comout();


#define CS7	2
#define CS8	3
#define NPARITY	0
#define OPARITY	(1<<3)
#define EPARITY	(0x3<<3)
#define STOP1	0
#define STOP2	(1<<2)

#define B300	(2<<5)
#define B600	(3<<5)
#define B1200	(4<<5)
#define B2400	(5<<5)
#define B4800	(6<<5)
#define B9600	(7<<5)
#define B19200		0
#define B38400	(1<<1)

#define	SETUPDRV	0x0000
#define	OUTDRVR		0x0100
#define	INPDRVR		0x0200
#define	INITDRVR	0x0400

/* MBBIOS Functions */

#define MFLOWON		0x0500
#define MFLOWOFF	0x0600
#define MSNDBRK		0x0700
#define SETOPT		0x0900
#define XMTBUF		0x0001
#define HRDSHK		0x0004

/* G8BPQ MBBIOS extensions 
   ('officially' defined in MB.H - for gloabl purposes)

#define BPQVER		0x1f00
#define BPQCALL		0x1f01
#define BPQUNACK	0x1f02
#define BPQRETNODE	0x1f10

*/

/* FOSSIL function defintions	*/

#define FFLOWOFF	0x0601
#define FFLOWON		0x0600


#define BELL	'\007'

int alow_bell = TRUE;

/*******************************************************************
 * io_bios() - FOSSIL calls to X00 driver or MBBIOS calls
 *******************************************************************/
io_bios(port_id,biosfunc)
int port_id, biosfunc;
{
  union REGS regs;

	regs.x.dx = port_id;
	regs.x.ax = biosfunc;
	int86(0x14,&regs,&regs);
	return(regs.x.ax);

}

/*
	eatChars() - clean out active port (port->)
*/
eatChars()
{
	while(instat())
		inchar();
}


/*
	flow_off() - flow control off, raise DTR and CTS
*/
void
flow_off(portid)
int portid;
{
	io_bios(portid, MFLOWOFF);
}


/*
	flow_on() - flow control on, lower DTR and CTS
*/
void
flow_on(portid)
int portid;
{
	io_bios(portid, MFLOWON);
}

/************************************************************************
 * getchar() -  Return character from current input.
 ************************************************************************/
char pgetchar()
{
	char ch;

	if (port->id == 'L')
		ch = bdos(7,0,0);     /* same as getch */
	else
		ch = comin(port->idn);

	return(ch);
}



/************************************************************************
 * inchar() -  Return character from current input w/CONSOLE Echo
 ************************************************************************/

char inchar()
{
	char ch, pgetchar();

	ch = pgetchar();

	if (port->id != 'L') {
		if ((ch != BELL) || alow_bell)
			bdos(2, ch, 0);
		if (ch == '\r')
			bdos(2,'\n',0);
	}

	return (ch);
}


/*******************************************************************
 * instat() - Return non-zero if a character waits at current input.
 *******************************************************************/

instat()
{

	if (port->id == 'L')
		return( kbstat());
	else
		return(comstat(port->idn));
}

/***********************************************************************
 *  charout() - Put the character out the proper port,
 *  	        Note special handling of end-of-line character.
 ***********************************************************************/
void
charout(ch)
char ch;
{

	if (port->id != 'L') { 
	
		while (!comtxrdy(port->idn)) giveaway(1);
	
		if (ch == '\n') {
			comout(port->idn, '\r');
			if (port->type & P_TNC)	/* only cr to a tnc in converse mode */
				return;
			while (!comtxrdy(port->idn)) giveaway(1);
		}
		comout(port->idn, ch);
	}

}




/***********************************************************************
 *  outchar() - Put the character out the proper port,
 *  	        echo to console if port is not the console.
 *  	        Note special handling of end-of-line character.
 ***********************************************************************/
void
outchar(ch)
char ch;
{

			/* echo send all output to console */
	if (ch == '\n')
			bdos(2,'\r',0);

	if (ch != BELL || alow_bell)
		bdos(2, ch, 0);
	
	charout(ch);

}

#ifdef BIGTIME
/************************************************************************
 *	port_xst() - test if port has been initialized by X00.SYS or MBBIOS,
 *  if it has, it is set up at 1 stop, no parity, 8 data bits, and baudrate
 *  *** WARNING *** WARNING *** in the FOSSIL driver this call will clear
 *  transmit and receive buffers, use it wisely!
 ************************************************************************/
port_xst(pp)
PORTS	*pp;
{
		if (io_bios(pp->idn,INITDRVR) == 0xAA55) {
			printf(" - COM%d port, MBBIOS Driver initialized",pp->idn+1);
			/* init MBBIOS3.2 for XMIT buffering and hardware handshake	*/
			io_bios(pp->idn,SETOPT+XMTBUF+HRDSHK); 
		} else {
			printf(" - COM%d port, NO IO Driver found\n", pp->idn+1);
			return (FALSE);
		}

		return (TRUE);
}
#endif


/*
	sendbreak() - send break out port
*/
void
sendbreak(portid)
int portid;
{
	io_bios(portid,MSNDBRK);
}


setbaud(idn,rate)
int idn;
unsigned rate;
{
	unsigned baudrate;
	
	switch(rate) {
	case 300:
		baudrate = B300;
		break;
	case 600:
		baudrate = B600;
		break;
	case 1200:
		baudrate = B1200;
		break;
	case 2400:
		baudrate = B2400;
		break;
	case 4800:
		baudrate = B4800;
		break;
	case 9600:
		baudrate = B9600;
		break;
	case 19200:
		baudrate = B19200;
		break;
	case 38400:
		baudrate = B38400;
		break;
	default:
		return(FALSE);
	}
	io_bios(idn,SETUPDRV+NPARITY+CS8+STOP1+baudrate);
	return (TRUE);
}

havedcd(pp)
PORTS *pp;
{
	union REGS regs;
	if (pp->type & P_BPQ) {
		regs.x.dx = pp->idn;	
		regs.x.ax = BPQCALL;
		int86(0x14,&regs,&regs);
		return(regs.x.si != 0);
	} else {
		return(chekdcd(pp->idn));
	}	
}



/*******************************************************************
 * usr_gone() - if available checks for a hardware (this case DCD)
 *	indication of the user having gone bye-bye. hasDCD is set
 *      in CONFIG on port by port basis and is for TNC2 or TNC1 with
 *      DCD wired to provide CONNECT status to pin 8 of RS232C cable
 *      |br| 12/31/87
 *******************************************************************/
usr_gone()
{
    if (!(port->type & P_CONSOLE))
		if (port->mode & (REMOTE|SYSOP|FORWARD|INPROCESS))
			if (!havedcd(port))	
				return (TRUE);
    return(FALSE); /* always false unless capacity exists and configged */
}
