
//////////////////////////////////////////////////////////////
//
// FBB driver for KAM-Pactor mode
//
// File : fbb_kam.dll
//
// (C) F6FBB 2000
//
//////////////////////////////////////////////////////////////
//
//  Version history
//
//	    - 06/01/2001 - EB5AGF
//		Added version resource and code to read it at runtime
//
//////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <winver.h>
#include "../fbb_drv.h"

#define uchar unsigned char

#define KAM_VHF			0
#define KAM_HF			1
#define KAM_PORTS		2
#define KAM_CHANS		26

#define	FEND	0xc0
#define	FESC	0xdb
#define	TFEND	0xdc
#define	TFESC	0xdd

#define DRV_HOST 99

#define	FBBDLLNAME	"kam.dll"

//------------

int KAM_MAJOR = 0;
int KAM_MINOR = 0;

typedef struct DATAOUT_t {
	char	*pData;
	int		nType;
	int		nLPort;
	int		nChan;
	int		nLen;
	struct	DATAOUT_t *pNext;
} DATAOUT;

typedef struct DATAIN_t {
	char	*pData;
	int		nType;
	int		nLPort;
	int		nChan;
	int		nLen;
	DRVUI	*pUi;
	struct	DATAIN_t *pNext;
} DATAIN;

typedef struct FBBDRV_t
{
	HWND	hWnd;		// FBB Window
	int		nChan[2];	// Nb of channels
	int		nPort[2];	// FBB Port
	int		nTriesP;	// Port of last TRIES command
	int		nTriesC;	// Chan of last TRIES command
	int		nStatW;		// Waiting STATUS command answer
	int		nCom;
	int		nNbCmd;
	BOOL	bUp;
//	char	szMyCall[10];	// BBS callsign
	HANDLE	hDev;
	HANDLE	hThread;
	DWORD	dwThreadId;
	DATAOUT	*pTncHead;	// Head of data_out list
	DATAOUT	*pTncTail;	// Tail of data_out list
	DATAIN	*pHostHead;	// Head of data_in list
	DATAIN	*pHostTail;	// Tail of data_in list
	DRVSTATS pStats[KAM_PORTS][KAM_CHANS+1];
	CRITICAL_SECTION	hCritical;
	struct	FBBDRV_t *pNext;
} FBBDRV;


//////////////////////////////////////////////////////////////
// Local prototypes
//////////////////////////////////////////////////////////////

static DWORD WINAPI KamLoop( LPSTR lpData );
static DATAIN *GetHost(FBBDRV *pFbbDrv);
static BOOL ToTnc(FBBDRV *pFbbDrv, int nType, int nLPort, int nChan, char *pData, int nLen, DRVUI *pUi);
static BOOL KamInit(FBBDRV *pFbbDrv, int nPort, char *pPtr);
static void ToHost(FBBDRV *pFbbDrv, int nType, int nPort, int nChan, char *pData, int nLen, DRVUI *pUi);
static FBBDRV *SearchPort(int nPort, int *nLPort);
static FBBDRV *SearchCom(int nCom);
BOOL SetupConnection(HANDLE hDev, int nBaudrate);
static BOOL StartHostMode(FBBDRV *pFbbDrv, int nPort, char *szCall);
static BOOL EndHostMode(FBBDRV *pFbbDrv);
static void HostInfo(FBBDRV *pFbbDrv, char *sFormat, ...);
static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite);
static int ReadTncChar(FBBDRV *pFbbDrv);
static BOOL InitPort(FBBDRV *pFbbDrv, char *szCall);
static int TncQueueSize(FBBDRV *pFbbDrv);
static void KamSndStats(FBBDRV *pFbbDrv, int nLPort, int nChan);
static BOOL GetDrvVersion(int *nVMaj, int *nVMin);

//////////////////////////////////////////////////////////////
// Global variables
//////////////////////////////////////////////////////////////

FBBDRV	*pDrvHead = NULL;	// Head of port structures

//////////////////////////////////////////////////////////////
//
// Exported functions (5).
// They are :
// BOOL OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo);
// BOOL CloseFbbDriver(int nPort);
// BOOL ReadFbbDriver(int nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi);
// BOOL WriteFbbDriver(int nPort, int *nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi);
// BOOL StatFbbDriver(int nPort, int nChan, int nCmd, void *pPtr, int nLen);
//
//////////////////////////////////////////////////////////////


BOOL APIENTRY DllMain( HANDLE hModule, 
					  DWORD  ul_reason_for_call, 
					  LPVOID lpReserved
					  )
{
    switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
	case DLL_THREAD_ATTACH:
		GetDrvVersion(&KAM_MAJOR, &KAM_MINOR);
	case DLL_THREAD_DETACH:
		break;
	case DLL_PROCESS_DETACH:
		// Clean all linked ports
		while (pDrvHead)
		{
			CloseFbbDriver(pDrvHead->nPort[KAM_VHF]);
			CloseFbbDriver(pDrvHead->nPort[KAM_HF]);
		}
		break;
    }
    return TRUE;
}


BOOL WINAPI OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo)
{
	int		nNb;
	char	szStr[80];
	char	szPort[15];
	BOOL	fRetVal ;
	FBBDRV	*pFbbDrv;
	HANDLE	hDev;
	HANDLE	hFbbThread;
	DWORD	dwThreadID;
	DRVINFO *pInfo;
	COMMTIMEOUTS  CommTimeOuts ;
	
	
	pInfo = (DRVINFO *)pDrvInfo;
	if (pInfo->nMultCh != 1 && pInfo->nMultCh != 0)
		return FALSE;
	
	pFbbDrv = SearchCom(pInfo->nCom);
	
	if (pFbbDrv == NULL)
	{
		wsprintf( szPort, "COM%d", pInfo->nCom ) ;
		
		// open COMM device
		if ((hDev =	CreateFile(
			szPort, 
			GENERIC_READ | GENERIC_WRITE,
			0,                    // exclusive access
			NULL,                 // no security attrs
			OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL, // |
			//FILE_FLAG_OVERLAPPED, // overlapped I/O
			NULL )) == (HANDLE) -1 )
		{
			return FALSE;
		}
		
		// setup device buffers
		SetupComm(hDev, 4096, 4096) ;
		
		// purge any information in the buffer
		PurgeComm(hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
		
		// Set time out
		CommTimeOuts.ReadIntervalTimeout = 0 ;
		CommTimeOuts.ReadTotalTimeoutMultiplier = 0 ;
		CommTimeOuts.ReadTotalTimeoutConstant = 200 ;
		
		// CBR_9600 is approximately 1byte/ms. For our purposes, allow
		// double the expected time per character for a fudge factor.
		CommTimeOuts.WriteTotalTimeoutMultiplier = 2*CBR_9600/pInfo->nBaudrate;
		CommTimeOuts.WriteTotalTimeoutConstant = 0 ;
		
		SetCommTimeouts(hDev, &CommTimeOuts ) ;
		
		if (!SetupConnection(hDev, pInfo->nBaudrate))
			fRetVal = FALSE;
		
		EscapeCommFunction(hDev, SETDTR );
		EscapeCommFunction(hDev, SETRTS );
		
		pFbbDrv = (FBBDRV *)LocalAlloc(LPTR, sizeof(FBBDRV));
		if (pFbbDrv == NULL)
			return FALSE;
		
		pFbbDrv->hDev    = hDev;
		pFbbDrv->bUp     = TRUE;
		pFbbDrv->hWnd    = hWnd;
		pFbbDrv->nCom    = pInfo->nCom;
		pFbbDrv->nPort[pInfo->nMultCh] = nPort;
		pFbbDrv->nChan[pInfo->nMultCh] = pInfo->nNbChan;
		
		//	strcpy(pFbbDrv->szMyCall, pInfo->szMyCall);
		
		// Insert the structure in the list
		pFbbDrv->pNext = pDrvHead;
		pDrvHead = pFbbDrv;
		
		if (hWnd)
			WM_NOTIFY_MSG = RegisterWindowMessage(FBBDRV_NOTIFY);
		
		InitializeCriticalSection(&pFbbDrv->hCritical);
		
		if (StartHostMode(pFbbDrv, pInfo->nMultCh, pInfo->szMyCall))
		{
			// Create a thread to process the KAM protocol.
			hFbbThread = CreateThread( 
				(LPSECURITY_ATTRIBUTES) NULL,
				0,
				(LPTHREAD_START_ROUTINE) KamLoop,
				(LPVOID) pFbbDrv,
				0, 
				&dwThreadID );
		}
		else
			fRetVal = FALSE;
		
		
		if ((hFbbThread == NULL) || (fRetVal == FALSE))
		{
			EscapeCommFunction(hDev, CLRDTR );
			EscapeCommFunction(hDev, CLRRTS );
			CloseHandle(hDev);
			DeleteCriticalSection(&pFbbDrv->hCritical);
			
			// Remove structure from list
			pDrvHead = pDrvHead->pNext;
			
			// Free the structure
			LocalFree(pFbbDrv);
			
			return FALSE;
		}
		
		pFbbDrv->dwThreadId	= dwThreadID;
		pFbbDrv->hThread	= hFbbThread;

		nNb = wsprintf(szStr, "USERS %d", pFbbDrv->nChan[pInfo->nMultCh]);
		ToTnc(pFbbDrv, DRV_COMMAND, 0, 0, szStr, nNb, NULL);
	}
	else
	{
		if (pFbbDrv->nPort[pInfo->nMultCh])
			return FALSE;

		pFbbDrv->nPort[pInfo->nMultCh] = nPort;
		pFbbDrv->nChan[pInfo->nMultCh] = pInfo->nNbChan;

		if (pInfo->nMultCh == KAM_HF)
			nNb = wsprintf(szStr, "USERS %d/", pFbbDrv->nChan[KAM_HF]);
		else
			nNb = wsprintf(szStr, "USERS /%d", pFbbDrv->nChan[KAM_VHF]);
		ToTnc(pFbbDrv, DRV_COMMAND, 0, 0, szStr, nNb, NULL);
	}
	
	return TRUE;
} // end of OpenFbbDriver()

BOOL WINAPI CloseFbbDriver(int nPort)
{
	int nLPort;
	FBBDRV *pFbbDrv = SearchPort(nPort, &nLPort);
	if (pFbbDrv == NULL)
		return FALSE;

	pFbbDrv->nPort[nLPort] = 0;

	// Is the other port active ?
	nLPort = !nLPort;
	if (pFbbDrv->nPort[nLPort])
		return TRUE;

	// No more active ports... Close the driver.

	// TNC to command mode
	EndHostMode(pFbbDrv);
	
	// block until thread has been halted
	pFbbDrv->bUp = FALSE;
	WaitForSingleObject(pFbbDrv->hThread, INFINITE);
	
	DeleteCriticalSection(&pFbbDrv->hCritical);
	
	// Remove structure from list
	if (pDrvHead == pFbbDrv)
		pDrvHead = pDrvHead->pNext;
	else
	{
		FBBDRV *pTmp = pDrvHead;
		while (pTmp->pNext)
		{
			if (pTmp->pNext == pFbbDrv)
			{
				pTmp->pNext = pTmp->pNext->pNext;
				break;
			}
			pTmp = pTmp->pNext;
		}
	}
	
	LocalFree(pFbbDrv);
	
	return ( TRUE ) ;
	
} // end of CloseFbbDriver()

BOOL WINAPI ReadFbbDriver(int *nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	int nLPort;
	FBBDRV *pFbbDrv = SearchPort(*nPort, &nLPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	pPtr = GetHost(pFbbDrv);
	if (pPtr == NULL)
		return FALSE;
	
	if (pPtr->nLen)
	{
		memcpy(pszBuf, pPtr->pData, pPtr->nLen);
		LocalFree(pPtr->pData);
	}
	*nLen = pPtr->nLen;
	*nChan = pPtr->nChan;
	*nPort = pFbbDrv->nPort[pPtr->nLPort];
	*nType = pPtr->nType;
	
	if (pPtr->pUi)
	{
		*pUi = *pPtr->pUi;
		LocalFree(pPtr->pUi);
	}
	LocalFree(pPtr);
	
	return TRUE;
} // end of ReadFbbDriver()

BOOL WINAPI WriteFbbDriver(int nPort, int nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi)
{
	char szStr[80];
	int nUiLen;
	int nLPort;
	FBBDRV *pFbbDrv = SearchPort(nPort, &nLPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	switch (nType)
	{
	case DRV_UNPROTO:
		if (*pUi->szVia)
			nUiLen = wsprintf(szStr, "U %s VIA %s", pUi->szTo, pUi->szVia);
		else
			nUiLen = wsprintf(szStr, "U %s", pUi->szTo);
		ToTnc(pFbbDrv, DRV_COMMAND, nLPort, 0, szStr, nUiLen, pUi);
		return ToTnc(pFbbDrv, DRV_UNPROTO, nLPort, 0, pszBuf, nLen, pUi);
	default:
		return ToTnc(pFbbDrv, nType, nLPort, nChan, pszBuf, nLen, pUi);
	}

	return FALSE;
} // end of WriteFbbDriver()

BOOL WINAPI StatFbbDriver(int nPort, int nCmd, int nChan, void *pPtr, int nLen)
{
	int nLPort;
	FBBDRV *pFbbDrv = SearchPort(nPort, &nLPort);
	if (pFbbDrv == NULL)
		return 0;
	
	switch (nCmd)
	{
	case DRV_INIT:
		return KamInit(pFbbDrv, nLPort, (char *)pPtr);
	case DRV_SNDCMD:
	case DRV_ECHOCMD:
		return (ToTnc(pFbbDrv, DRV_COMMAND, nLPort, nChan, (char *)pPtr, strlen((char *)pPtr), NULL));
	case DRV_PORTCMD:
		return (ToTnc(pFbbDrv, DRV_COMMAND, nLPort, nChan, (char *)pPtr, strlen((char *)pPtr), NULL));
	case DRV_PACLEN:
		*((int *) pPtr) = 250;
		return TRUE;
/*	case DRV_SYSRDY:
		return (ToTnc(pFbbDrv, DRV_COMMAND, nChan, "PACTOR", 6, NULL)); */
	case DRV_VERSION:
		wsprintf((char *)pPtr, 
			"v%d.%02d FBB driver for KAM-Pactor (F6FBB-%s)", 
			KAM_MAJOR, KAM_MINOR, __DATE__);
		return TRUE;
	}
	
	return FALSE;
} // end of StatFbbDriver()


//////////////////////////////////////////////////////////////
//
// Local functions
//
//////////////////////////////////////////////////////////////

static BOOL StartHostMode(FBBDRV *pFbbDrv, int nPort, char *szCall)
{
	//	int nLen;
	int nChar;
	int nNb;
	BOOL bOk;
	char *pszRet = "\r";
	char *pszCmd = "\300Q\300";
	char *pszReset = "MAXUSERS 16/16\rRESET\r";
	char *pszHost = "INTFACE HOST\r";
	
	HostInfo(pFbbDrv, 0, "StartHostMode");

	// Check if TNC is alive
	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, pszRet, strlen(pszRet));
		Sleep(100);
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(10);
			bOk = TRUE;
		}
		
		if (bOk)
			break;

		// TNC does not answer. Try going out from HOST mode
		WriteTncBlock(pFbbDrv, pszCmd, strlen(pszCmd));

		Sleep(1000);
	}
	
	if (nNb == 4)
		return FALSE;
	
	// Go to hostmode
	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, pszHost, strlen(pszHost));
		Sleep(100);
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(10);
		}
		
		WriteTncBlock(pFbbDrv, pszReset, strlen(pszReset));
		Sleep(2000);

		// Be sure the COM is empty ...
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(10);
			if (nChar == FEND)
			{
				bOk = TRUE;
				break;
			}
		}
	
		if (bOk)
			break;
	}
	
	if (nNb == 4)
		return FALSE;
	
	HostInfo(pFbbDrv, 0, "StartHostMode OK");

	// Init PTC HostMode counter
//	pFbbDrv->bInit = TRUE;

	return InitPort(pFbbDrv, szCall);
}

static BOOL EndHostMode(FBBDRV *pFbbDrv)
{
	int nI;

	// Back to cmd: mode
	ToTnc(pFbbDrv, DRV_HOST, 0, 0, "Q", 1, NULL);

	// Wait until all TNC queue is processed
	// Possible deadlock if resync mode !
	// Wait only 10 seconds
	for (nI = 0 ; nI < 100 ; nI++)
	{
		if (TncQueueSize(pFbbDrv) == 0)
			break;
		
		// Wait some time...
		Sleep(100);
	}

	return TRUE;
}

BOOL SetupConnection(HANDLE hDev, int nBaudrate)
{
	BOOL       fRetVal ;
	DCB        dcb ;
	
	dcb.DCBlength = sizeof( DCB ) ;
	
	GetCommState(hDev, &dcb ) ;
	
	dcb.BaudRate = nBaudrate ;
	dcb.ByteSize = 8 ;
	dcb.Parity   = NOPARITY ;
	dcb.StopBits = ONESTOPBIT ;
	
	
	// setup hardware flow control
	
	dcb.fOutxDsrFlow = FALSE ;
	dcb.fDtrControl = DTR_CONTROL_DISABLE ;
	
	dcb.fOutxCtsFlow = FALSE ;
	dcb.fRtsControl = RTS_CONTROL_DISABLE ;
	
	// setup software flow control
	
	dcb.fInX = dcb.fOutX = FALSE ;
	
	// other various settings
	
	dcb.fBinary = TRUE ;
	
	fRetVal = SetCommState(hDev, &dcb ) ;
	
	return ( fRetVal ) ;
	
} // end of SetupConnection()

static int ReadTncChar(FBBDRV *pFbbDrv)
{
	BOOL       fReadStat ;
	COMSTAT    ComStat ;
	DWORD      dwErrorFlags;
	DWORD      dwLength;
	BYTE		nChar;

	// Read one character
	// I don't know how to read more than one character
	// with a timeout, to get any number of bytes
	ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
	fReadStat = ReadFile(pFbbDrv->hDev, &nChar, 1, &dwLength, NULL) ;
	if (!fReadStat)
	{
		ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, "<CE-%u>", dwErrorFlags ) ;
		}
		return -1;
	}
	
	if (dwLength == 0)
	{
		return -1;
	}
	
	return ( nChar ) ;	
} // end of ReadTncChar()

static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite)
{
	
	BOOL        fWriteStat ;
	DWORD       dwBytesWritten ;
	DWORD       dwErrorFlags;
	//	DWORD   	dwError;
	DWORD       dwBytesSent=0;
	COMSTAT     ComStat;
	
	fWriteStat = WriteFile(pFbbDrv->hDev, lpByte, dwBytesToWrite,
		&dwBytesWritten, NULL) ;
	if (!fWriteStat)
	{
		// some other error occurred
		ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, "<CE-%u>", dwErrorFlags ) ;
		}
		return ( FALSE );
	}
	return ( TRUE ) ;
	
} // end of WriteTncBlock()

static FBBDRV *SearchPort(int nPort, int *nLPort)
{
	int nI;
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		for (nI = 0 ; nI < KAM_PORTS ; nI++)
		{
			if (pTmp->nPort[nI] == nPort)
			{
				*nLPort = nI;
				return pTmp;
			}
		}
		pTmp = pTmp->pNext;
	}
	
	return NULL;
}

static FBBDRV *SearchCom(int nCom)
{
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		if (pTmp->nCom == nCom)
			return pTmp;
		pTmp = pTmp->pNext;
	}
	
	return NULL;
}

static BOOL KamInit(FBBDRV *pFbbDrv, int nLPort, char *pPtr)
{
	return ToTnc(pFbbDrv, DRV_COMMAND, nLPort, 0, pPtr, strlen(pPtr), NULL);
}

static BOOL ToTnc(FBBDRV *pFbbDrv, int nType, int nLPort, int nChan, char *pData, int nLen, DRVUI *pUi)
{
	if (nLen == -1)
		nLen = strlen(pData);

	if (nLen <= 0 || nLen > 256)
		return FALSE;
	
	if (nType == DRV_COMMAND && *pData == '#')
		return FALSE;

	DATAOUT *pPtr;
	
	pPtr           = (DATAOUT *)LocalAlloc( LMEM_FIXED, sizeof(DATAOUT));
	pPtr->pData    = (char *)LocalAlloc( LMEM_FIXED, nLen);
	pPtr->nType    = nType;
	pPtr->nLPort   = nLPort;
	pPtr->nChan    = nChan;
	pPtr->nLen     = nLen;
	pPtr->pNext    = NULL;

	memcpy(pPtr->pData, pData, nLen);

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	// Add the structure at end of list
	if (pFbbDrv->pTncHead == NULL)
		pFbbDrv->pTncHead = pPtr;
	else
		pFbbDrv->pTncTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pTncTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	/*
	if (nType == DRV_DATA)
	{
		++pFbbDrv->pStats[nLPort][nChan].nNbFrames;
		pFbbDrv->pStats[nLPort][nChan].nAux = 1;
		KamSndStats(pFbbDrv, nLPort, nChan);
	}
	*/

	return TRUE;
}

static void ToHost(FBBDRV *pFbbDrv, int nType, int nLPort, int nChan, char *pData, int nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	
	if (pData == NULL)
		nLen = 0;
	else if (nLen == -1)
		nLen = strlen(pData);

	if (nLen < 0 || nLen > 256)
		return;

	pPtr           = (DATAIN *)LocalAlloc( LMEM_FIXED, sizeof(DATAIN));
	pPtr->nType    = nType;
	pPtr->nChan    = nChan;
	pPtr->nLPort   = nLPort;
	pPtr->nLen     = nLen;
	pPtr->pUi      = pUi;
	pPtr->pNext    = NULL;
	
	if (nLen > 0)
	{
		pPtr->pData    = (char *)LocalAlloc( LMEM_FIXED, nLen);
		memcpy(pPtr->pData, pData, nLen);
	}
	else
		pPtr->pData = NULL;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	if (pFbbDrv->pHostHead == NULL)
		pFbbDrv->pHostHead = pPtr;
	else
		pFbbDrv->pHostTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pHostTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	// Send Notification message
	if (pFbbDrv->hWnd)
		PostMessage(pFbbDrv->hWnd, WM_NOTIFY_MSG, nType, MAKELONG(pFbbDrv->nPort[nLPort], nChan));
}

static DATAOUT *GetTnc(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	if (pPtr)
		pFbbDrv->pTncHead = pFbbDrv->pTncHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return pPtr;
}

static DATAIN *GetHost(FBBDRV *pFbbDrv)
{
	DATAIN *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pHostHead;
	if (pPtr)
		pFbbDrv->pHostHead = pFbbDrv->pHostHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return pPtr;
}

static int TncQueueSize(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	int nNb = 0;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	while (pPtr)
	{
		pPtr = pPtr->pNext;
		++nNb;
	}
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return nNb;
}

static void HostInfo(FBBDRV *pFbbDrv, char *sFormat, ...)
{
	char	szStr[512];
	va_list pArgPtr;
	int nCnt;
	
	va_start(pArgPtr, sFormat);
	nCnt = wvsprintf(szStr, sFormat, pArgPtr);
	va_end(pArgPtr);
	
	if (nCnt > 0)
		ToHost(pFbbDrv, DRV_INFO, 0, 0, szStr, nCnt, NULL);
}

static BOOL InitPort(FBBDRV *pFbbDrv, char *szCall)
{
	int nNb;
	char szStr[80];
	char *szInit[] = {
		"MX ON",
		"MC ON",
		"MCOM ON",
		"M ON",
		"PBBS 0",
		"PI ON",
		"PI ON/ON",
		"STATS ON",
		"MAXFRAME 2",
		NULL
	};
	
	for (nNb = 0 ; szInit[nNb] ; nNb++)
		ToTnc(pFbbDrv, DRV_COMMAND, 0, 0, szInit[nNb], strlen(szInit[nNb]), NULL);

	nNb = wsprintf(szStr, "MYCALL %s", szCall);
	ToTnc(pFbbDrv, DRV_COMMAND, 0, 0, szStr, nNb, NULL);

	return TRUE;
}

void KamAskStatus(FBBDRV *pFbbDrv)
{
	int nLPort, nChan;
	static clock_t nTimePrec = 0L;
	
	clock_t nTime = clock();
	
	if (nTimePrec > nTime)
		return;
	
	nTimePrec = nTime + (CLOCKS_PER_SEC / 4);
	
	if (pFbbDrv->nStatW)
	{
		--pFbbDrv->nStatW;
		return;
	}
	
	ToTnc(pFbbDrv, DRV_COMMAND, 0, 0, "S", -1, NULL);
	
	pFbbDrv->nStatW = 10;
	
	for (nLPort = 0 ; nLPort < KAM_PORTS ; nLPort++)
	{
		for (nChan = 1 ; nChan <= KAM_CHANS ; nChan++)
		{
			if (++pFbbDrv->nTriesC > KAM_CHANS)
			{
				pFbbDrv->nTriesC = 1;
				if (++pFbbDrv->nTriesP == KAM_PORTS)
					pFbbDrv->nTriesP = 0;
			}
			if (pFbbDrv->pStats[pFbbDrv->nTriesP][pFbbDrv->nTriesC].nState != DRV_DISCONNECTED)
			{
				ToTnc(pFbbDrv, DRV_COMMAND, pFbbDrv->nTriesP, pFbbDrv->nTriesC, "TRI", -1, NULL);
				return;
			}
		}
	}
}

static void SendTncRequest(FBBDRV *pFbbDrv)
{
	int nI;
	int nLen;
	int nSize = 0;
	char *pData;
	DATAOUT *pPtr;
	char szSend[1024];
	static int test = 0;
	
	pPtr = GetTnc(pFbbDrv);

	if (pPtr == NULL)
	{
		// Nothing to send
		KamAskStatus(pFbbDrv);
		return;
	}

	// Header
	szSend[nSize++] = (char)FEND;

	switch(pPtr->nType)
	{
	case DRV_HOST:
		break;
	case DRV_COMMAND:
		szSend[nSize++] = 'C';	// Command
		szSend[nSize++] = pPtr->nLPort + '1';	// Port;
		szSend[nSize++] = (pPtr->nChan == 0) ? '0' : pPtr->nChan + '@';	// Stream;
		++pFbbDrv->nNbCmd;
		break;
	case DRV_DATA:
		szSend[nSize++] = 'D';	// Data
		szSend[nSize++] = pPtr->nLPort + '1';	// Port;
		szSend[nSize++] = (pPtr->nChan == 0) ? '0' : pPtr->nChan + '@';	// Stream;
		break;
	case DRV_UNPROTO:
		szSend[nSize++] = 'D';	// Data
		szSend[nSize++] = pPtr->nLPort + '1';	// Port;
		szSend[nSize++] = '0';	// Stream;
		break;
	}

	nLen = pPtr->nLen;
	pData = pPtr->pData;

	// Add data
	for (nI = 0 ; nI < nLen ; nI++)
	{
		switch (pData[nI])
		{
		case FEND:
			szSend[nSize++] = (char)FESC;
			szSend[nSize++] = (char)TFEND;
			break;
		case FESC:
			szSend[nSize++] = (char)FESC;
			szSend[nSize++] = (char)TFESC;
			break;
		default:
			szSend[nSize++] = pData[nI];
			break;
		}
	}

	// End of block
	szSend[nSize++] = (char)FEND;

	LocalFree(pPtr->pData);
	LocalFree(pPtr);

	WriteTncBlock(pFbbDrv, (char *)szSend, nSize);
}

static void KamSndStats(FBBDRV *pFbbDrv, int nLPort, int nChan)
{
	DRVSTATS DrvStats;

	if (pFbbDrv->pStats[nLPort][nChan].nAux)
	{
		pFbbDrv->pStats[nLPort][nChan].nAux = 0;
		DrvStats = pFbbDrv->pStats[nLPort][nChan];
		DrvStats.nBuffers  = 500;
		DrvStats.nFrameNb  = 0;

		ToHost(pFbbDrv, DRV_STATS, nLPort, nChan, (char *)&DrvStats, sizeof(DRVSTATS), NULL);
	}
}

static void KamGetStatus(FBBDRV *pFbbDrv, char *szLine)
{
	int nI;
	int nStat;
	int nAck;
	int nStream;
	int nLPort;

#define NB_KAM_STAT	13

#define DRV_DISCONNECTED	0
#define DRV_CONNECTED		1
#define DRV_DISCPENDING		2
#define DRV_CONNPENDING		3
#define DRV_CONNWAIT		4
#define DRV_CONNCHECK		5
	static struct
	{
		int nState;
		int nLg;
		char *szTxt;
	}
	kam_stat[NB_KAM_STAT] =

	{
		{
			DRV_DISCONNECTED, 9, "DISCONNEC"
		}
		,
		{
			DRV_CONNPENDING, 9, "CONNECT i"
		}
		,
		{
			DRV_CONNECTED, 9, "FRMR in p"
		}
		,
		{
			DRV_DISCPENDING, 9, "DISC in p"
		}
		,
		{
			DRV_CONNECTED, 9, "CONNECTED"
		}
		,
		{
			DRV_CONNECTED, 1, "x"
		}
		,
		{
			DRV_CONNECTED, 9, "Waiting a"
		}
		,
		{
			DRV_CONNECTED, 9, "Device bu"
		}
		,
		{
			DRV_CONNECTED, 9, "Remote de"
		}
		,
		{
			DRV_CONNECTED, 9, "Both devi"
		}
		,
		{
			DRV_CONNECTED, 17, "Waiting ACK and d"
		}
		,
		{
			DRV_CONNECTED, 17, "Waiting ACK and r"
		}
		,
		{
			DRV_CONNECTED, 17, "Waiting ACK and b"
		}
		,
	};


	char *pScan = szLine;
	if (pScan[1] != '/')
		return;

	nStream = pScan[0] - '@';
	nLPort = (pScan[2] == 'V') ? 0 : 1;

	pScan += 4;
	if (*pScan == '#')
	{
		while ((*pScan) && (*pScan != '('))
			++pScan;
		++pScan;
		nAck = 0;
		while (isdigit (*pScan))
		{
			nAck *= 10;
			nAck += (*pScan - '0');
			++pScan;
		}
		pScan += 2;
	}
	else
	{
		nAck = 0;
	}

	nStat = 20;
	for (nI = 0; nI < NB_KAM_STAT; nI++)
	{
		if (strncmp (kam_stat[nI].szTxt, pScan, kam_stat[nI].nLg) == 0)
		{
			nStat = kam_stat[nI].nState;
			break;
		}
	}

	if (nStat == DRV_DISCONNECTED)
	{
		nAck = 0;
	}

	if (pFbbDrv->pStats[nLPort][nStream].nState != nStat)
	{
		pFbbDrv->pStats[nLPort][nStream].nState = nStat;
		pFbbDrv->pStats[nLPort][nStream].nAux = 1;
	}

	if (pFbbDrv->pStats[nLPort][nStream].nNbFrames != nAck)
	{
		pFbbDrv->pStats[nLPort][nStream].nNbFrames = nAck;
		pFbbDrv->pStats[nLPort][nStream].nAux = 1;
	}

	KamSndStats(pFbbDrv, nLPort, nStream);
}

static void	KamStatus(FBBDRV *pFbbDrv, char *pData)
{
//	int nStream;
//	int nLPort;
	char *pLine;

/*
	for (nLPort = 0 ; nLPort < KAM_PORTS ; nLPort++)
	{
		for (nStream = 1 ; nStream <= KAM_CHANS ; nStream++)
		{
			pFbbDrv->pStats[nLPort][nStream].nState = DRV_DISCONNECTED;
			pFbbDrv->pStats[nLPort][nStream].nNbFrames = 0;
		}
	}
*/
	pLine = strtok(pData, "\r");
	while (pLine)
	{
		KamGetStatus(pFbbDrv, pLine);
		pLine = strtok(NULL, "\r");
	}

	pFbbDrv->nStatW = 0;
}

static void	KamMonitor(FBBDRV *pFbbDrv, int nPort, char *pData)
{
	char *pHead;
	char *pTxt;
	DRVUI ui;
	int nHeadLen;

	// Header
	pHead = strtok(pData, "\r");
	nHeadLen = strlen(pHead);

	// Data
	pTxt = strtok(NULL, "\0");


	// Extract fields for the header line

	char *ptr;
	char *sptr;

	memset (&ui, 0, sizeof (DRVUI));

	ui.nPort = pFbbDrv->nPort[nPort];
	ui.nPid = 0xf0;

	ptr = strtok (pHead, ">"); /* exped */

	if (ptr == NULL)
		return;
	strncpy (ui.szFrom, ptr, 11); ui.szFrom[11] = '\0';

	ptr = strtok (NULL, ":,");	/* desti */

	if (ptr == NULL)
		return;
	strncpy (ui.szTo, ptr, 11); ui.szTo[11] = '\0';

	ptr = strtok (NULL, ":,");	/* digis */

	if (ptr == NULL)
		return;

	if (*ptr != ' ')
	{
		for (;;)
		{
			strncat (ui.szVia, ptr, 12);
			strcat (ui.szVia, " ");

			ptr = strtok (NULL, ":,");	/* digis */

			if (ptr == NULL)
				return;

			if ((*ptr == '\0') || (*ptr == ' '))
				break;

		}
	}

	++ptr;
	sptr = ptr;

	while ((*sptr) && (*sptr != '>'))
		++sptr;
	*sptr = '\0';

	/* controle */
	*ui.szCtl = '\0';

	if (ptr[0] == '<')
	{
		int pos = 0;
		int version = 1;
		int reponse = 0;

		++ptr;
		if (ptr[0] == '<')
		{
			version = 2;
			/* AX25 Version 2 */
			++ptr;
		}

		sptr = ptr;
		if (*sptr == 'F')
		{
			strcpy (ptr, "FRMR");
			ui.nPid = 0;
			pos = 4;
		}
		else if (*sptr == 'U')
		{
			pos = 2;
			if (sptr[1] == 'A')
				reponse = 1;
		}
		else if (*sptr == 'C')
		{
			strcpy (ptr, "SABM");
			ui.nPid = 0;
			pos = 4;
		}
		else if (*sptr == 'D')
		{
			strcpy (ptr, "DISC");
			ui.nPid = 0;
			pos = 4;
		}
		else if (*sptr == 'I')
		{
			pos = 3;
		}
		else
		{
			if (*sptr == 'r')
			{
				strupr (sptr);
				reponse = 1;
			}
			if (sptr[1] == 'R')
				pos = 3;
			else
				pos = 4;
		}

		if (version == 1)
		{
			if (reponse)
				sptr[pos] = '\0';
			else
				sptr[pos] = '!';
		}
		else
		{
			if (reponse)
				sptr[pos] = '-';
			else
				sptr[pos] = '+';
		}
		sptr[pos + 1] = '\0';
		strncpy (ui.szCtl, ptr, 4); ui.szCtl[4] = '\0';
	}

	ui.bUi = (strncmp (ui.szCtl, "UI", 2) == 0);

	DRVUI *pUi = (DRVUI *)LocalAlloc(LMEM_FIXED, sizeof(DRVUI));
	if (pUi == NULL)
		return;

	*pUi = ui;
	ToHost(pFbbDrv, DRV_UNPROTO, nPort, 0, pTxt, -1, pUi);
}

static void ReadTncAnswer(FBBDRV *pFbbDrv)
{
	int nChar;
	int nLen;
	int nPort;
	int nStream;
	int nStatus;
	BOOL bFesc;
	BOOL bIn;
	char szBuf[4096];
	char szCall[30];
	char *pData;
	
/*
	nChar = ReadTncChar(pFbbDrv);
	if (nChar == -1)
		return;
	
	if (nChar != FEND)
		return;
*/
	
	// Fills a buffer
	bIn = FALSE;
	nLen = 0;
	bFesc = FALSE;
	
	while (pFbbDrv->bUp)
	{
		nChar = ReadTncChar(pFbbDrv);
		if (nChar == -1)
		{
			//			Resync(pFbbDrv);
			return;
		}
		else if (bIn)
		{
			if (nChar == FEND)
			{
				if (nLen > 0)
					break;
			}
			else if (nChar == FESC)
			{
				bFesc = TRUE;
			}
			else if (bFesc)
			{
				bFesc = FALSE;
				if (nChar == TFESC)
					szBuf[nLen++] = (char)FESC;
				else if (nChar == TFEND)
					szBuf[nLen++] = (char)FEND;
				else
					return;	// Block error ... Abort
			}
			else
			{
				szBuf[nLen++] = nChar;
			}
		}
		else if (nChar == FEND)
		{
			bIn = TRUE;
		}
	}
	
	if (!pFbbDrv->bUp)
		return;
	
	// Decode the block
	nStatus = szBuf[0];
	nPort   = szBuf[1] - '1';
	nStream = (szBuf[2] == '0') ? 0 : szBuf[2] - '@';
	pData   = szBuf+3;
	nLen   -= 3;
	
	pData[nLen] = '\0';
	
	switch (nStatus)
	{
	case '?':	// Status
		HostInfo(pFbbDrv, "S:%d%c %02x %02x", nStream, pData[0], pData[1] & 0xff, pData[2] & 0xff);
		break;
	case 'C':	// Command answer
		--pFbbDrv->nNbCmd;
		if (strncmp(pData, "FREE BYTES", 10) == 0)
			KamStatus(pFbbDrv, pData);
		else if (strncmp(pData, "TRIES", 5) == 0)
		{
			int nTries = atoi(pData+7);
			nPort = pFbbDrv->nTriesP;
			nStream = pFbbDrv->nTriesC;
			if (pFbbDrv->pStats[nPort][nStream].nNbRetry != nTries)
			{
				pFbbDrv->pStats[nPort][nStream].nNbRetry = nTries;
				pFbbDrv->pStats[nPort][nStream].nAux = 1;
			}
		}
		break;
	case 'D':	// Data
		ToHost(pFbbDrv, DRV_DATA, nPort, nStream, pData, nLen, NULL);
		break;
	case 'M':	// Monitor
		KamMonitor(pFbbDrv, nPort, pData);
		break;
	case 'R':	// Connect request
		break;
	case 'S':	// Link state
		if (strncmp(pData, "*** ", 4) != 0)
			break;
		switch (pData[4])
		{
		case 'r':
			// Retry count exceeded
			wsprintf(szBuf, "(%d) LINK FAILURE with KAM", nStream);
			ToHost(pFbbDrv, DRV_COMMAND, nPort, nStream, szBuf, -1, NULL);
			break;
		case 'C':
			// Connected
			sscanf(pData, "%*s %*s %*s %s", szCall);
			wsprintf(szBuf, "(%d) CONNECTED to %s", nStream, szCall);
			ToHost(pFbbDrv, DRV_COMMAND, nPort, nStream, szBuf, -1, NULL);
			break;
		case 'D':
			// Disconnected
			wsprintf(szBuf, "(%d) DISCONNECTED fm KAM", nStream);
			ToHost(pFbbDrv, DRV_COMMAND, nPort, nStream, szBuf, -1, NULL);
			break;
		}
		break;
	case 'T':	// Trace
		break;
	}
}

// KAM Thread

DWORD WINAPI KamLoop(LPSTR lpData)
{
	FBBDRV	*pFbbDrv = (FBBDRV*) lpData ;

	while ( pFbbDrv->bUp )
	{
		// Send the request
		SendTncRequest(pFbbDrv);
		
		// Read the answer
		ReadTncAnswer(pFbbDrv);
	}
	
	pFbbDrv->dwThreadId = 0 ;
	pFbbDrv->hThread    = NULL ;
	
	return( TRUE ) ;
		
} // end of KamLoop()


BOOL GetDrvVersion(int *nVMaj, int *nVMin)
{
	DWORD dwVersionInfoSize;
	DWORD dwTmpHandle;
	LPVOID lpVersionInfo; 
	BOOL bRet = false;


	dwVersionInfoSize = GetFileVersionInfoSize(FBBDLLNAME, &dwTmpHandle);

	if(dwVersionInfoSize)
	{
		lpVersionInfo = LocalAlloc(LPTR, dwVersionInfoSize);
		if(lpVersionInfo)
		{
			if(GetFileVersionInfo(FBBDLLNAME, dwTmpHandle, dwVersionInfoSize, lpVersionInfo))
			{
				LPVOID lpBuffer = LocalAlloc(LPTR, dwVersionInfoSize);
				UINT dwBytes;

				if( VerQueryValue(lpVersionInfo, 
						TEXT("\\StringFileInfo\\000004B0\\FileVersion"), 
						&lpBuffer, 
						&dwBytes)) 
				{
						sscanf((TCHAR *)lpBuffer, "%d,%d", nVMaj, nVMin);
						bRet = true;
				}

				LocalFree(lpBuffer);
			}

			LocalFree(lpVersionInfo);
 		}
 	}

	return bRet;
}
