
/******************************************************
 *                                                    *
 * FPAC project.            FPAC PAD                  *
 *                                                    *
 * Parts of code from different sources of ax25-utils *
 *                                                    *
 * F6FBB 05-1997                                      *
 *                                                    *
 ******************************************************/

/******************************************************
 * 12/05/97 1.00 F6FBB First draft !
 *
 ******************************************************/
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include <ctype.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <net/if_arp.h>
#include <net/if.h>

#include <linux/ax25.h>
#include <linux/rose.h>

#include "axutils.h"
#include "axconfig.h"
#include "config.h"

#include "fpad.h"

#define FPACCONF "/etc/ax25/fpac.conf"

static int l3_conf(int verbose);
static void create_rsports(void);

int read_conf(int verbose)
{
	if (cfg_open(&cfg) != 0)
		return(0);

	if (l3_conf(verbose) == 0)
		return(0);

	return(1);
}

/* Check if the last modified of the conf file has changed */
int conf_changed(void)
{
	struct stat st;

	if (stat(FPACCONF, &st) == 0)
	{
		if (cfg.date != st.st_mtime)
		{
			read_conf(1);
			return(0);
		}
	}
	return(0);
}

static int add_route(int s, route_t *r)
{
	int i;
	char *dev;
	char *ptr;
	char nodeaddr[11];
	char nodes[80];
	char call[80];
	node_t *n = NULL;
	struct rose_route_struct rs_node;

	nodeaddr[10] = '\0';
	memset(&nodeaddr, '0', 10);
	strncpy(nodeaddr, r->addr, strlen(r->addr));
		
	rs_node.mask = strlen(r->addr);

	if (convert_rose_address(nodeaddr, rs_node.address.rose_addr) != 0) 
	{
		fprintf(stderr, "rsparms: nodes: invalid address %s\n", nodeaddr);
		return(0);
	}

	/* Validates all nodes for this route */

	strcpy(nodes, r->nodes);
	ptr = strtok(nodes, " ");

	while (ptr)
	{
		/* Search node in the node list */
		n = cfg.node;
		while (n)
		{
			if (strcasecmp(n->name, ptr) == 0)
				break;
			n = n->next;
		}

		if (n == NULL)
		{
			fprintf(stderr, "Node %s not found\n", ptr);
			return(0);
		}
		printf("Route %s -> %s (%s)\n", nodeaddr, n->call, n->port);
		
		if ((dev = ax25_config_get_dev((n->port[0]) ? n->port : NULL)) == NULL) 
		{
			fprintf(stderr, "invalid port name {%s}\n", n->port);
			return(0);
		}

		strcpy(rs_node.device, dev);

		strcpy(call, n->call);
		ptr = strtok(call, " ");

		if (ptr)
		{
			if (convert_call_entry(ptr, rs_node.neighbour.ax25_call) != 0) 
			{
				fprintf(stderr, "rsparms: nodes: invalid callsign %s\n", ptr);
				return(0);
			}
		}

		for (i = 0; (ptr = strtok(NULL, " ")) && i < AX25_MAX_DIGIS; i++) 
		{
			if (convert_call_entry(ptr, rs_node.digipeaters[i].ax25_call) != 0) 
			{
				fprintf(stderr, "rsparms: nodes: invalid callsign %s\n", ptr);
				return(0);
			}
		}

		rs_node.ndigis = i;

		if (ioctl(s, SIOCADDRT, &rs_node) == -1)
		{
			perror("rsparms: SIOCADDRT");
			return(0);
		}

		ptr = strtok(NULL, " ");
	}

	return(1);
}

static int l3_conf(int verbose)
{
	int s;
	ax25_address rose_call;
	route_t route;
	addrp_t *a = NULL;
	route_t *r = NULL;
	node_t *n = NULL;
	cover_t *c = NULL;

	/* Configure the Node Callsign */
	
	if (convert_call_entry(cfg.callsign, rose_call.ax25_call) == -1) 
	{
		fprintf(stderr, "rsparms: invalid callsign %s\n", cfg.callsign);
		return(0);
	}

	if ((s = socket(AF_ROSE, SOCK_SEQPACKET, 0)) < 0) 
	{
		perror("l3_conf: socket");
		return(0);
	}
		
	if (ioctl(s, SIOCRSCLRRT, NULL) == -1)
	{
		perror("l3_conf: SIOCRSCLRRT");
		/* close(s);
		return(0); */
	}

	if (ioctl(s, SIOCRSL2CALL, &rose_call) == -1) 
	{
		perror("l3_conf: SIOCRSL2CALL");
		close(s);
		return(0);
	}

#ifndef LINUX_2_1
	printf("Configuring loopback :\n");

	/* Add the RSLOOP to the beginning of the list of nodes */
	n = calloc(1, sizeof(node_t));
	if (n == NULL)
		return(0);
	n->next = cfg.node;
	cfg.node = n;
	
	strcpy(n->name, "LoopBack");
	strcpy(n->call, "RSLOOP");
	strcpy(n->dnic, cfg.dnic);
	strcpy(n->addr, cfg.address);
	strcpy(n->port, "");
	
	strcpy(route.nodes, "LoopBack");
	strcpy(route.addr, cfg.fulladdr);

	/* Add a route for the local address to loopback */
	if (add_route(s, &route) == 0)
	{
		fprintf(stderr, "Error in route %s %s\n", route.addr, route.nodes);
	}

	/* Add a route for each coverage address to loopback */
	for (c = cfg.cover ; c ; c = c->next)
	{
		memcpy(route.addr+4, c->addr, 6);
		if (add_route(s, &route) == 0)
		{
			fprintf(stderr, "Error in route %s %s\n", route.addr, route.nodes);
		}
	}

	/* Add a route for each port address to loopback */
	for (a = cfg.addrp ; a ; a = a->next)
	{
		memcpy(route.addr+4, a->addr, 6);
		if (add_route(s, &route) == 0)
		{
			fprintf(stderr, "Error in route %s %s\n", route.addr, route.nodes);
		}
	}
#endif

	printf("Configuring routes :\n");
	
	/* Add routes to adjacents */
	r = cfg.route;
	while (r)
	{
		if (add_route(s, r) == 0)
		{
			fprintf(stderr, "Error in route %s %s\n", r->addr, r->nodes);
			close(s);
			return(0);
		}
		r = r->next;
	}

	printf("\n");

	close(s);

	create_rsports();
	
	return(1);
}

static int getfreedev(char *dev, char *address)
{
	struct ifreq ifr;
	int fd;
	int i;
	
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("rsattach: socket");
		return FALSE;
	}

	for (i = 0; i < 10; i++) 
	{
		sprintf(dev, "rose%d", i);
		strcpy(ifr.ifr_name, dev);
	
		if (ioctl(fd, SIOCGIFHWADDR, &ifr) != 0) 
		{
			perror("rsattach: SIOCGIFHWADDR");
			close(fd);
			return(-1);
		}

		if (strcmp(address, rose2asc((rose_address *)ifr.ifr_hwaddr.sa_data)) == 0)
		{
			if (ioctl(fd, SIOCGIFFLAGS, &ifr) < 0) 
			{
				perror("rsattach: SIOCGIFFLAGS");
				close(fd);
				return(-1);
			}

			close(fd);

			if (ifr.ifr_flags & IFF_UP) 
			{
				/* Already ok ... */
				return(0);
			}
			else
			{
				/* must be UP */
				return(1);
			}
		}
	}

	/* No device for this address... Look for a free one */
	for (i = 0; i < 6; i++) 
	{
		sprintf(dev, "rose%d", i);
		strcpy(ifr.ifr_name, dev);
	
		if (ioctl(fd, SIOCGIFFLAGS, &ifr) < 0) 
		{
			perror("rsattach: SIOCGIFFLAGS");
			close(fd);
			return(-1);
		}


		if (!(ifr.ifr_flags & IFF_UP)) 
		{
			close(fd);
			return(1);
		}
	}

	close(fd);

	return -1;
}

static int startiface(char *dev, char *address /*, struct hostent *hp */)
{
	struct ifreq ifr;
	char addr[5];
	int fd;
	
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("rsattach: socket");
		return FALSE;
	}

	strcpy(ifr.ifr_name, dev);
	
	/*
	if (hp != NULL) {
		ifr.ifr_addr.sa_family = AF_INET;
		
		ifr.ifr_addr.sa_data[0] = 0;
		ifr.ifr_addr.sa_data[1] = 0;
		ifr.ifr_addr.sa_data[2] = hp->h_addr_list[0][0];
		ifr.ifr_addr.sa_data[3] = hp->h_addr_list[0][1];
		ifr.ifr_addr.sa_data[4] = hp->h_addr_list[0][2];
		ifr.ifr_addr.sa_data[5] = hp->h_addr_list[0][3];
		ifr.ifr_addr.sa_data[6] = 0;

		if (ioctl(fd, SIOCSIFADDR, &ifr) < 0) {
			perror("rsattach: SIOCSIFADDR");
			return FALSE;
		}
	}
	*/

	if (convert_rose_address(address, addr) == -1)
		return FALSE;

	ifr.ifr_hwaddr.sa_family = ARPHRD_ROSE;
	memcpy(ifr.ifr_hwaddr.sa_data, addr, 5);
	
	if (ioctl(fd, SIOCSIFHWADDR, &ifr) != 0) {
		perror("rsattach: SIOCSIFHWADDR");
		return FALSE;
	}

	ifr.ifr_mtu = ROSE_MTU;

	if (ioctl(fd, SIOCSIFMTU, &ifr) < 0) {
		perror("rsattach: SIOCSIFMTU");
		return FALSE;
	}

	if (ioctl(fd, SIOCGIFFLAGS, &ifr) < 0) {
		perror("rsattach: SIOCGIFFLAGS");
		return FALSE;
	}

	ifr.ifr_flags &= IFF_NOARP;
	ifr.ifr_flags |= IFF_UP;
	ifr.ifr_flags |= IFF_RUNNING;

	if (ioctl(fd, SIOCSIFFLAGS, &ifr) < 0) {
		perror("rsattach: SIOCSIFFLAGS");
		return FALSE;
	}
	
	close(fd);
	
	return TRUE;
}

/* Creates a new rsports file */
static void create_rsports(void)
{	
	FILE *fptr;
	time_t sec = time(NULL);
	
	fptr = fopen(CONF_RSPORTS_FILE, "w");
	if (fptr == NULL)
	{
		fprintf(stderr, "Cannot create rsports file\n");
		return;
	}
	fprintf(fptr,
		"# %s\n"
		"#\n"
		"# Automatic generation by fpad - %s"
		"#\n"
		"# The format of this file is:\n"
		"#\n"
		"# name address     description\n"
		"#\n",
		CONF_RSPORTS_FILE, ctime(&sec));
	fclose(fptr);
}

static void add_rsports(char *name, char *address)
{
	FILE *fptr;
	
	fptr = fopen(CONF_RSPORTS_FILE, "a");
	if (fptr == NULL)
	{
		fprintf(stderr, "Cannot find rsports file\n");
		return;
	}
	fprintf(fptr, "%s  %s  Rose port %s\n", name, address, address+4);
	fclose(fptr);
}

char *l3_attach(char *rsaddr, int verbose)
{
	int result;
	static char dev[64];

	result = getfreedev(dev, rsaddr);

	if (result == -1) 
	{
		fprintf(stderr, "Cannot find free Rose device\n");
		return(NULL);
	}
	
	add_rsports(dev, rsaddr);

	if (result == 0)
	{
		printf("Rose address %s already bound to device %s\n", rsaddr, dev);
		return(dev);
	}

	if (!startiface(dev, rsaddr))
		return(NULL);		

	if (verbose)
		printf("Rose address %s bound to device %s\n", rsaddr, dev);

	return(dev);
}

/* Returns the user port of a callsign or the default port */
char *user_port(ax25_address *addr)
{
	luser_t *l = cfg.luser;
	ax25_address buf;

	while (l)
	{
		if (!convert_call_entry(l->call, (char *)&buf) && !ax25cmp(addr, &buf))
		{
			return(l->port);
		}
		l = l->next;
	}
	return(cfg.def_port);
}
