package PGProcess;

use strict;
use Dumpvalue;
my @EXPORT = qw(

);

my $VERSION = '0.3';

sub new {
	my ($self,$superUser) = @_;
	# If customizing ps columns, the USER should be first,
	# the PID should be second, and COMMAND/CMD last

	#
	# BSD-style ps arguments mean:
	#
	#	x show processes with no controlling terminal
	#	w 132 column display
	#	w another 'w' means display as wide as needed, no limit
	#	o specify list of columns
	#
	#	This option would be nice, but Linux treats it differently
	#	r sort by cpu usage
	#
	# On Linux, args with no dash are BSD args, else SysV
	#
	# set this to customize your ps command
	
	$self = {
		_sortBy => "pid",
		_desc => "",
		_t =>(),
		_out =>(),
		_globalDebug => 0,
		_user => "",
		_pgPort => 5432,
		_mySuperUser => $superUser,
		_superUser => "",
		_ps => "ps",
#		_psArgs => "xwwouser,pid,start,%mem,vsz,inblk,oublk,state,%cpu,time,command",
		_psArgs => "xwwouser,pid,start,%mem,state,%cpu,time,command",
		_psUserArg => "-U",
		_psPidArg => "-p",
		_psUserEnd => 0,
		_psPidParam => 0,
		_psUserParam => 0,
		_psCmdCol => 0,
		_psHeading => ""
	};
	
	bless $self,'PGProcess';
	
	$self->init;
	
	return $self;
}

sub updateProcess {
	my $self = shift;

	my @out = ();
	
	my @output = `$self->{_ps} $self->{_psArgs} $self->{_psUserArg} $self->{_superUser}`;
	@output = grep(/$self->{_superUser}.*([0-9]+(\.[0-9]+){3}|\[local\]|localhost)/,@output);
	
	
	#quick and dirty remove of lines
#	@output = grep(/[0-9]+(\.[0-9]+){3}|\[local\]|localhost/, @output);
	foreach my $p ( @output ){
		
		#$p = substr($p,$self->{_psUserEnd}-length($p));
		$p =~ s/\ +/ /g;
		$p =~ s/(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\ /$1_/g;
		$p =~ s/^$self->{_superUser}\ +|^\ +|\n|\([^\\)]*\)|\ +$//g;

		my @lineSplit = split(/\ /,$p,11);
		if($lineSplit[8] =~ /^[0-9]+(\.[0-9]+){3}$|^\[local\]$|^localhost$/){
			($lineSplit[8],$lineSplit[9]) = ($lineSplit[9],$lineSplit[8]);
		}
		$lineSplit[1] =~ s/_/\ /;
		 push @out,{"pid",$lineSplit[0],
			"start",$lineSplit[1],
			"mem",$lineSplit[2],
			"status",$lineSplit[3],
			"cpu",$lineSplit[4],
			"time",$lineSplit[5],
			"user",$lineSplit[7],
			"database",$lineSplit[8],
			"connection",$lineSplit[9],
			"query",$lineSplit[10]};
	}
	
	
	if($self->{_desc} eq ""){
		if($self->{_sortBy} eq "pid"){
			@out = sort {$a->{$self->{_sortBy}} <=> $b->{$self->{_sortBy}} } @out;
		}else{
			@out = sort {$a->{$self->{_sortBy}} cmp $b->{$self->{_sortBy}} } @out;
		}
	}else{
		if($self->{_sortBy} eq "pid"){
			@out = sort {$b->{$self->{_sortBy}} <=> $a->{$self->{_sortBy}} } @out;
		}else{
			@out = sort {$b->{$self->{_sortBy}} cmp $a->{$self->{_sortBy}} } @out;
		}
	}

	$self->{_out} = \@out;
	return $self->{_out};
}

sub init{
	my $self = shift;
	$self->setUser;
	$self->setSuperUser;
	$self->setPs;
}

sub setUser{
	my $self = shift;

	if(!(defined $ENV{USER})){
		die("Can not determine your user name.")
	}else{
		$self->{_user} = $ENV{USER};
	}
}

sub setSuperUser{
	my $self = shift;
	
	if($ENV{'PGPORT'}){	$self->{_pgPort} = $ENV{'PGPORT'};}# defaultne nastaven na 5432 v new

	my @tmp = ();
	# get pg username, either from command line or postmaster process owner
	# Zkopirovano s puvodniho Tcl/Tk monitoru
	if($self->{_mySuperUser} and (@tmp = getpwnam($self->{_mySuperUser}) )){
		$self->{_superUser} = $tmp[0];
	#try PGDATA directory ownership
	}elsif($ENV{'PGDATA'} and (@tmp = stat($ENV{'PGDATA'})) and (@tmp = getpwuid($tmp[4])) ){
		$self->{_superUser} = $tmp[0];
	#try user name for postmaster from lock file
	}elsif(@tmp = stat("/tmp/.s.PGSQL.$self->{_pgPort}.lock") and (@tmp = getpwuid($tmp[4])) ){
		$self->{_superUser} = $tmp[0];
	# try user name for postmaster from socket
	}elsif(@tmp = stat("/var/run/postgresql/.s.PGSQL.$self->{_pgPort}") and (@tmp = getpwuid($tmp[4])) ){
		$self->{_superUser} = $tmp[0];
	}elsif(@tmp = stat("/tmp/.s.PGSQL.$self->{_pgPort}") and (@tmp = getpwuid($tmp[4])) ){
		$self->{_superUser} = $tmp[0];
	}else{
	die("Can't find the username of the PostgreSQL server.\nEither start the postmaster, define PGDATA or PGPORT, or\nsupply the username on the command line.");
	}
}

sub setPs{
	my $self = shift;
	
	my $fail = 1;
	
	if(($fail = $self->tryPs) == 1){
		print STDERR "Solaris custom ps args failed\nTrying BSD-style -u on Solaris";
		#	u display user information
		#	x show processes with no controlling terminal
		#	w 132 column display
		#	w another 'w' means display as wide as needed, no limit
		$self->{_psArgs} = "uxww";
		# Try Solaris first because this is the one that displays arg changes
		$self->{_ps} = "/usr/ucb/ps";
	}
	
	if($fail  and (($fail = $self->tryPs) == 1)){
		print STDERR "BSD-style Solaris custom ps args failed\nTrying non-Solaris";
		# Try ordinary ps
		$self->{_ps} = "ps";
	}
	
	if($fail  and (($fail = $self->tryPs) == 1)){
		print STDERR "BSD-style -u ps args failed\nTrying SysV-style";
		#
		# try SysV-style ps flags:
		#
		#	f display full listing, needs dash
		#	e display all processes
		$self->{_psArgs} = "-ef";

		#	u show only certain user's processes
		$self->{_psUserArg} = "-u"
	}
	
	if($fail  and (($fail = $self->tryPs) == 1)){
		die("Can't run 'ps'\n");
	}
	return $self;
}
sub tryPs{
	my $self = shift;
	
	# This proc either validates the ps_args, ps_user_arg,
	# ps_pid_arg values, or throws an error.  If successful, derived
	# information is stored into ps_pid_param and other globals.

	# get USER column parameter number
	my @tmp = `$self->{_ps} $self->{_psArgs} $self->{_psPidArg} 1 2>/dev/null`;
	my $head = $tmp[0];

	my $pos = -1;

	if((($pos = index($head,"USER",0)) > -1) or (($pos = index($head,"UID",0)) > -1) ){
		$self->{_psUserParam} = $pos;
	}else{
		print STDERR "Can't find USER/UID column heading";
		return 1;
	}
	
	# check other columns before we test for postmaster and
	# and process arg columns
	if(index($head,"PID",0) == -1){
		print STDERR "Can't find PID column heading";
		return 1;
	}
	
	if((index($head,"COMMAND",0) == -1) and (index($head,"CMD",0) == -1)){
		print STDERR "Can't find COMMAND/CMD column heading";
		return 1;
	}

	# get end of user column so it can be clipped off
	if($self->{_psUserParam} == 0){
		$self->{_psUserEnd} = length($self->{_superUser}) + 1;
	}else{
		$self->{_psUserEnd} = 1;
	}

	# get PID column parameter number
	@tmp = `$self->{_ps} $self->{_psArgs} $self->{_psPidArg} 1 2>/dev/null`;
	my $headnoUser = substr($tmp[0],$self->{_psUserEnd}-length($tmp[0])); 
	if(($pos = index($head,"USER",0)) > -1){
		$self->{_psPidParam} = $pos;
	}else{
		print STDERR "Can't find PID column heading";
		return 1;
	}

	# get a new heading without the user column
	@tmp = `$self->{_ps} $self->{_psArgs} $self->{_psPidArg} 1 2>/dev/null`;
	$self->{_psHeading} = substr($tmp[0],$self->{_psUserEnd}-length($tmp[0])); 
	
	# find the column of the COMMAND/CMD
	if(($pos = index($self->{_psHeading},"COMMAND",0)) > -1){
		$self->{_psCmdCol} = $pos;
	}elsif(($pos = index($self->{_psHeading},"CMD",0)) > -1){
		$self->{_psCmdCol} = $pos;
	}else{
		print STDERR "Can't find COMMAND/CMD column heading";
		return 1;
	}

	# adjust heading to be the way we want it
	my $tmp = sprintf("%5.5s %-8.8s %-4.4s%2.2s %-4.4s %8.8s",split(" ",$self->{_psHeading}),6);
	$self->{_psHeading} = sprintf("%s %-10.10s %-10.10s %-14s %-s\n",$tmp,"USER", "DATABASE", "CONNECTION", "QUERY");
	return 0;
}

sub getProcess{
	my $self = shift;
	return $self->{_out};
}

sub sendSignal{
	my ($self,$pid,$signal) = @_;
	if (!($pid =~ /^-?\d/) or ($pid < 0) or ($signal < 0)) {return "Not valid PID or signal";}
	
	
	my $out = kill $signal,$pid;

	if($out != 1){
		return "Can't kill process";
	}

	return 0;
	
}

sub fetchQuery{
	my ($self,$pid) = @_;

	if (!($pid =~ /^-?\d/) or ($pid < 0)) {return "Not valid PID";}

	if($self->{_globalDebug} == 0){
		my $command = "gdb -q  -x /dev/stdin postgres $pid 2>&1 <<EOF\nset print elements 0\nprint (char *)debug_query_string\nquit\nEOF";
		#$command = "gdb -q  -x /dev/stdin postgres $pid 2>&1 <<EOF\nset print elements 0\nprint (char *)debug_query_string\nquit\nEOF";
	#	print $command;
		$_ = join(""=>`$command`);

		if(m/No symbol/gsi){$self->{_globalDebug} = 1;}
	}

	if($self->{_globalDebug} == 1){
		my $command = "gdb -q  -x /dev/stdin postgres $pid 2>&1 << EOF\nset print elements 0\nprint pg_exec_query_string::query_string\nquit\nEOF";
		#$command = "gdb -q  -x /dev/stdin postgres $pid 2>&1 << EOF\nset print elements 0\nprint pg_exec_query_string::query_string\nquit\nEOF";
	#	print $command;
		$_ = join(""=>`$command`);
	}

	if(m/\ permit/gsi){	return "No permission. Try running as root or as a PostgreSQL owner.";	}


	m/\$1\ \=\ [a-z0-9\ ]+\"(.*)\"$/igs;

	if(defined $1 ){$_ = $1;}else{return "No query being executed.";	}

	$_ =~ s/(\\)(\"|\')/$2/g;
	$_ =~ s/^\s*(.*?)\s*$/$1/;
	return $_;
}

sub printLines{
	my ($self,$head,$mode) = @_;

	if(!defined $self->{_out}){
		$self->updateProcess;
	}
	
	my $output = "";

	if($head){
		$output .= $self->{_psHeading};
	}
	foreach (@{$self->{_out}}){
		$output .= sprintf("%5.5s %-8.8s %-4.4s%2.2s %-4.4s%9.9s %-10.10s %-10.10s %-14s %-s\n",$_->{pid},$_->{start},$_->{mem},$_->{status},$_->{cpu},$_->{'time'},$_->{user},$_->{database},$_->{connection},$_->{query});
		
		if((defined $mode) and ($mode == 1) and (!($_->{query} =~ /idle/i)) ){
			$output .= $self->fetchQuery($_->{pid})."\n";
		}
	}
 return $output;
}

sub printHtml{
	my ($self,$head,$mode,$add) = @_;

	if(!defined $self->{_out}){
		$self->updateProcess;
	}

	my $output = "";

	$output .= "<table ".((defined $add)?$add:"").">\n";

	if($head){
		my $th = $self->{_psHeading};
		$th =~ s/^\s*(.*?)\s*$/$1/;#Trim
		$th =~ s/(\ )+/<\/td><td>/g;
		
		$output .= "<tr>\n<td>".$th."</tr>\n";
	}
	
	foreach(@{$self->{_out}}){
		$output .= "<tr>\n<td>".sprintf("%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s\n",$_->{pid},$_->{start},$_->{mem},$_->{status},$_->{cpu},$_->{'time'},$_->{user},$_->{database},$_->{connection},$_->{query})."</td>\n</tr>\n";

		if((defined $mode) and ($mode == 1) and (!($_->{query} =~ /idle/i)) ){
			$output .= "<tr>\n<td colspan='".scalar(keys(%$_))."'>". $self->fetchQuery($_->{pid})."</td>\n</tr>\n";
		}

	}
	$output .= "</table>\n";

	return $output;
}

return 1;

=head1 Name

PGProcess  Module for view load and base managing of postgreSQL server

=head1 Synopsis

    use PGProcess;

    $pgprocess = PGProcess->new();
    $pgprocess->updateProcess;
    $pgprocessHash = getProcess;

=head1 Description

Based on Proc::ProcessTable it's allow you to view postmaster processes and with root's permissions view actual queries, kill process

=head1 Methods

=over 5

=item new

Create	new instance of PGProcess

=item init

Initialize PGProcess; looking fir ps and its format of arguments

=item updateProcess

Fetch actual information about running postmaster processes

=item sendSignal

Try to send singnal to process. Takes PID as argument, you need to be root or PostgreSQL user 

=item fetchQuery

Try to fetch actual query, uses gdb. Takes PID as argument, need to be root

=item setUser

Looking for user's name

=item setSuperUser

Looking for username under which PostgreSQL is running

=item setPs

Testing few ps's formats

=item tryPs

It's called from setPs and runs some tests 

=item getProcess

Returns currently loaded array of processes

=item printLines

Simply returns proceses as string

=item printHtml

Returns the same output as printLines, but formatted in HTML table

=back

=head1 Example

   #Quick use
   use PGProcess;

   $pgProcess = new PGProcess;
	 print $pgProcess->printLines;

=head1 Author

   Vlastimil Seliga, xseliga@fi.muni.cz

=cut
