/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN mullo

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/mullo.h>

static int
test_simde_mm512_mullo_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[32];
    const int16_t b[32];
    const int16_t r[32];
  } test_vec[] = {
    { { -INT16_C( 31159), -INT16_C( 12097),  INT16_C( 29918),  INT16_C(  1377),  INT16_C( 32398),  INT16_C(  6544), -INT16_C( 30801),  INT16_C( 30357),
         INT16_C( 14887),  INT16_C(  9940),  INT16_C(  3636), -INT16_C( 23516),  INT16_C( 28807), -INT16_C(  9340), -INT16_C( 22852),  INT16_C(  1558),
        -INT16_C( 10708),  INT16_C(  2774),  INT16_C( 14154), -INT16_C( 10224), -INT16_C( 24394),  INT16_C( 26098), -INT16_C( 30937),  INT16_C( 20444),
        -INT16_C( 20287), -INT16_C(  2699), -INT16_C( 26177),  INT16_C( 18073),  INT16_C(  7434), -INT16_C( 14815),  INT16_C( 14275), -INT16_C(  3892) },
      { -INT16_C( 23795),  INT16_C( 22778),  INT16_C(  2778), -INT16_C( 28624),  INT16_C(  8875), -INT16_C( 11530), -INT16_C( 11607),  INT16_C( 27169),
        -INT16_C( 26750),  INT16_C( 16736), -INT16_C(  1744),  INT16_C( 14983), -INT16_C( 22505), -INT16_C(  9727), -INT16_C( 12832), -INT16_C(  4662),
        -INT16_C( 14992),  INT16_C( 19269),  INT16_C( 30415),  INT16_C( 31451), -INT16_C( 11880),  INT16_C( 16973),  INT16_C( 28323),  INT16_C(  9900),
         INT16_C(  3077),  INT16_C( 13927), -INT16_C(  4346),  INT16_C(  7536),  INT16_C( 29079),  INT16_C( 30711), -INT16_C( 15809), -INT16_C( 20635) },
      {  INT16_C( 19637), -INT16_C( 32122),  INT16_C( 12556), -INT16_C( 28112),  INT16_C( 25818), -INT16_C( 20384),  INT16_C(  8327), -INT16_C(  1227),
        -INT16_C( 30514),  INT16_C( 25472),  INT16_C( 15808), -INT16_C( 18692), -INT16_C( 19423),  INT16_C( 17284),  INT16_C( 28800),  INT16_C( 11100),
        -INT16_C( 28864), -INT16_C( 25170), -INT16_C( 12074),  INT16_C( 30128),  INT16_C(   528),  INT16_C(  3530), -INT16_C( 12331),  INT16_C( 20432),
         INT16_C( 32709),  INT16_C( 28691), -INT16_C(  5254),  INT16_C( 14320), -INT16_C( 29978), -INT16_C( 32553),  INT16_C( 32509),  INT16_C( 29820) } },
    { { -INT16_C( 21881),  INT16_C( 22266), -INT16_C( 10720), -INT16_C( 17967),  INT16_C(  7847),  INT16_C( 19451), -INT16_C( 22644), -INT16_C( 28047),
        -INT16_C( 10060), -INT16_C( 17720),  INT16_C( 14535),  INT16_C( 24535), -INT16_C( 12630), -INT16_C(  5674),  INT16_C( 15248),  INT16_C(  6040),
        -INT16_C( 27674),  INT16_C(  1646),  INT16_C( 16233),  INT16_C(  4287), -INT16_C( 17827), -INT16_C(  5797), -INT16_C( 13214),  INT16_C(  5755),
         INT16_C( 17317),  INT16_C( 27856), -INT16_C( 22660),  INT16_C(  9931), -INT16_C( 23947),  INT16_C(  1551), -INT16_C( 22563), -INT16_C( 15587) },
      { -INT16_C( 29894), -INT16_C( 23606), -INT16_C( 30262),  INT16_C( 10164),  INT16_C(  3908), -INT16_C( 23023), -INT16_C( 29476), -INT16_C( 32324),
        -INT16_C( 29488),  INT16_C( 19693), -INT16_C( 18125), -INT16_C( 22414), -INT16_C( 32421),  INT16_C( 14510), -INT16_C( 13272),  INT16_C( 25596),
        -INT16_C( 14761),  INT16_C(  8710), -INT16_C( 17841), -INT16_C( 27831),  INT16_C( 23242), -INT16_C( 22983), -INT16_C(  2585), -INT16_C( 18649),
         INT16_C(  5249), -INT16_C( 19453),  INT16_C( 30157),  INT16_C( 10333),  INT16_C(  3062),  INT16_C(  7777),  INT16_C( 24023),  INT16_C( 12161) },
      { -INT16_C(  4202), -INT16_C( 12476),  INT16_C(  5440),  INT16_C( 32244), -INT16_C(  4772), -INT16_C( 12885), -INT16_C( 29616),  INT16_C( 31740),
        -INT16_C( 32192),  INT16_C( 19240),  INT16_C(  7845), -INT16_C( 14914),  INT16_C(  8302), -INT16_C( 16524),  INT16_C(  3712),  INT16_C(   416),
         INT16_C( 10026), -INT16_C( 15724), -INT16_C(  9369),  INT16_C( 29559), -INT16_C( 16542), -INT16_C(  2237),  INT16_C( 13934),  INT16_C( 22973),
        -INT16_C(  1499), -INT16_C( 31120), -INT16_C( 13748), -INT16_C( 12353),  INT16_C(  9070),  INT16_C(  3503),  INT16_C( 17307), -INT16_C( 23395) } },
    { { -INT16_C( 30685),  INT16_C( 29265), -INT16_C( 26046),  INT16_C(  3078),  INT16_C( 16373), -INT16_C(  9038), -INT16_C(  9931), -INT16_C( 18797),
        -INT16_C( 26898), -INT16_C( 17557), -INT16_C( 14325),  INT16_C(   484),  INT16_C( 17875), -INT16_C( 21729), -INT16_C( 24158), -INT16_C( 14886),
         INT16_C( 11049),  INT16_C( 27447),  INT16_C( 15813), -INT16_C( 17800),  INT16_C( 10877), -INT16_C( 19818),  INT16_C( 10500), -INT16_C(  3480),
        -INT16_C( 11329), -INT16_C( 13651), -INT16_C( 28261),  INT16_C( 28619), -INT16_C(  5162),  INT16_C( 30746), -INT16_C(  2932), -INT16_C( 19139) },
      {  INT16_C( 29983), -INT16_C(  7136), -INT16_C( 26446),  INT16_C( 12191),  INT16_C( 13763), -INT16_C( 14367),  INT16_C( 19039),  INT16_C(  7865),
         INT16_C( 26141), -INT16_C( 17943), -INT16_C( 19208), -INT16_C( 12760),  INT16_C( 17055),  INT16_C( 11079), -INT16_C( 31690),  INT16_C( 21984),
         INT16_C(   505), -INT16_C( 21447), -INT16_C( 10087),  INT16_C( 23771), -INT16_C( 17138),  INT16_C( 27939), -INT16_C(  9209),  INT16_C(  9355),
         INT16_C( 29763),  INT16_C( 15325),  INT16_C(  1321), -INT16_C( 14327),  INT16_C( 20551),  INT16_C( 32244), -INT16_C( 11051), -INT16_C( 12590) },
      {  INT16_C( 31549),  INT16_C( 28192),  INT16_C( 29156), -INT16_C( 28230),  INT16_C( 28831),  INT16_C( 22130), -INT16_C(  4949),  INT16_C( 10811),
        -INT16_C(  4874), -INT16_C(  6301), -INT16_C( 31064), -INT16_C( 15456), -INT16_C( 15347), -INT16_C( 21863), -INT16_C( 24532), -INT16_C( 32576),
         INT16_C(  9185), -INT16_C( 11457),  INT16_C(  8893), -INT16_C( 23384), -INT16_C( 25642),  INT16_C( 18562), -INT16_C( 28900),  INT16_C( 15992),
        -INT16_C(  2307), -INT16_C( 10663),  INT16_C( 22739), -INT16_C( 31197),  INT16_C( 18522),  INT16_C( 10952),  INT16_C( 26748), -INT16_C( 15862) } },
    { {  INT16_C(  3285),  INT16_C( 28538),  INT16_C( 22244), -INT16_C(  3381), -INT16_C(  4333),  INT16_C(  6751), -INT16_C(  5173),  INT16_C(  3646),
         INT16_C(  7263), -INT16_C( 30647),  INT16_C( 21281),  INT16_C( 26961),  INT16_C( 17827),  INT16_C( 30950), -INT16_C( 18151), -INT16_C(  4281),
        -INT16_C( 15931), -INT16_C( 22178),  INT16_C( 10519),  INT16_C( 10908), -INT16_C(  1256), -INT16_C(  7100), -INT16_C( 31770),  INT16_C( 18162),
         INT16_C( 15519), -INT16_C( 16178),  INT16_C(  8079),  INT16_C( 12841),  INT16_C(  4196),  INT16_C( 32427), -INT16_C(  3383), -INT16_C( 29075) },
      { -INT16_C( 13389), -INT16_C( 13513), -INT16_C( 11276),  INT16_C(  3573),  INT16_C( 15055), -INT16_C( 18959), -INT16_C(  7235),  INT16_C( 23803),
        -INT16_C( 13793), -INT16_C( 20964),  INT16_C( 18153),  INT16_C( 20193), -INT16_C( 29610),  INT16_C(  8140),  INT16_C( 14718),  INT16_C( 12717),
        -INT16_C(  7164), -INT16_C(  1796), -INT16_C(  3400), -INT16_C( 30971), -INT16_C(  2516), -INT16_C(  5828),  INT16_C( 14554), -INT16_C(  1723),
         INT16_C( 24834), -INT16_C(  5208), -INT16_C( 30297), -INT16_C(   711),  INT16_C(  1301), -INT16_C( 27876), -INT16_C( 14018),  INT16_C( 17092) },
      { -INT16_C(  8209), -INT16_C( 20170), -INT16_C( 17072), -INT16_C( 21689), -INT16_C( 24995), -INT16_C(   401),  INT16_C(  5599),  INT16_C( 16074),
         INT16_C( 25985), -INT16_C( 31236), -INT16_C( 20727),  INT16_C( 15921), -INT16_C( 30526),  INT16_C( 12616), -INT16_C( 21682),  INT16_C( 18939),
         INT16_C( 31508), -INT16_C( 14200),  INT16_C( 18056),  INT16_C(  6412),  INT16_C( 14368),  INT16_C( 25584), -INT16_C( 24100), -INT16_C( 32454),
        -INT16_C( 18370), -INT16_C( 24272),  INT16_C(  7497), -INT16_C( 20447),  INT16_C( 19508),  INT16_C(  2996), -INT16_C( 25170),  INT16_C(  9588) } },
    { { -INT16_C( 15954),  INT16_C( 26171),  INT16_C( 16563), -INT16_C(  8211),  INT16_C( 10551),  INT16_C(  4552),  INT16_C(  3425),  INT16_C( 25354),
        -INT16_C( 19858),  INT16_C(  5711), -INT16_C( 30661),  INT16_C( 20499),  INT16_C( 12430), -INT16_C( 13085), -INT16_C( 22279), -INT16_C( 22769),
         INT16_C( 19049),  INT16_C(  7181), -INT16_C(  1398), -INT16_C( 15877), -INT16_C( 15580), -INT16_C( 31278), -INT16_C(  8752),  INT16_C( 16105),
         INT16_C( 14479), -INT16_C( 13484),  INT16_C( 26816),  INT16_C( 19995), -INT16_C(   104), -INT16_C( 28389),  INT16_C( 10919),  INT16_C(  4153) },
      {  INT16_C( 18036), -INT16_C(   468),  INT16_C( 10049),  INT16_C( 26048), -INT16_C( 27926), -INT16_C( 17686), -INT16_C( 11409), -INT16_C(     8),
         INT16_C( 19723), -INT16_C( 13110), -INT16_C(  6731),  INT16_C( 19738),  INT16_C( 13796), -INT16_C( 29730),  INT16_C(  5983), -INT16_C( 11365),
        -INT16_C( 14498), -INT16_C( 24622), -INT16_C( 27922), -INT16_C( 10236), -INT16_C(  4572), -INT16_C( 27502), -INT16_C( 29758), -INT16_C( 12909),
         INT16_C( 24024), -INT16_C( 29287), -INT16_C( 19390),  INT16_C( 10202), -INT16_C( 18199),  INT16_C( 18866),  INT16_C( 20176),  INT16_C( 11804) },
      {  INT16_C( 22232),  INT16_C(  7204), -INT16_C( 19853),  INT16_C( 29376),  INT16_C(  2630), -INT16_C( 28464), -INT16_C( 16369), -INT16_C(  6224),
        -INT16_C( 16198), -INT16_C( 29098),  INT16_C(  6327), -INT16_C( 10002), -INT16_C( 23432), -INT16_C(  4646),  INT16_C(  4967), -INT16_C( 31979),
        -INT16_C(  3698),  INT16_C(  5546), -INT16_C( 24500), -INT16_C( 12308), -INT16_C(  5872), -INT16_C( 17980),  INT16_C(  1952), -INT16_C( 19253),
        -INT16_C( 21592), -INT16_C( 14028),  INT16_C(   384), -INT16_C( 24578), -INT16_C(  7848), -INT16_C( 26682), -INT16_C( 30288),  INT16_C(  1084) } },
    { { -INT16_C(  4587),  INT16_C(  1229), -INT16_C( 11904), -INT16_C( 23076),  INT16_C( 28607), -INT16_C( 32455), -INT16_C( 13062), -INT16_C( 11697),
        -INT16_C(  6103),  INT16_C( 27487),  INT16_C( 14748), -INT16_C( 31086),  INT16_C( 17905), -INT16_C( 15921), -INT16_C(  5229), -INT16_C( 22289),
        -INT16_C( 17190),  INT16_C( 23212), -INT16_C( 30323),  INT16_C( 19967),  INT16_C( 14584), -INT16_C(  3378),  INT16_C(  7428),  INT16_C( 11716),
         INT16_C(  8966), -INT16_C( 23911),  INT16_C( 11100),  INT16_C( 19752), -INT16_C(  2192),  INT16_C(   783), -INT16_C(   285), -INT16_C( 16980) },
      {  INT16_C( 22715),  INT16_C( 18455),  INT16_C(  6113), -INT16_C(  9835),  INT16_C( 25679),  INT16_C( 21707), -INT16_C( 28799), -INT16_C( 30847),
         INT16_C(  6834),  INT16_C(  3626),  INT16_C( 21062), -INT16_C( 18852),  INT16_C( 27466),  INT16_C( 11706),  INT16_C( 26217),  INT16_C(  9450),
         INT16_C(   446), -INT16_C( 24467),  INT16_C(   536),  INT16_C( 26745),  INT16_C( 17766), -INT16_C(  5956),  INT16_C( 15828), -INT16_C( 30865),
        -INT16_C( 26280), -INT16_C( 24939), -INT16_C(  3604),  INT16_C( 13908),  INT16_C(  3676), -INT16_C( 14749),  INT16_C( 19828),  INT16_C( 13290) },
      {  INT16_C(  8535),  INT16_C(  5739), -INT16_C( 24192),  INT16_C(  1292),  INT16_C(  6129),  INT16_C( 11315), -INT16_C(  4102), -INT16_C( 23857),
        -INT16_C( 27006), -INT16_C( 12394), -INT16_C( 18264),  INT16_C( 10360), -INT16_C(  3414),  INT16_C( 13158),  INT16_C( 12619),  INT16_C(  1654),
         INT16_C(   972),  INT16_C(  6972), -INT16_C(   200),  INT16_C( 30087), -INT16_C( 30000), -INT16_C(   184), -INT16_C(  1200),  INT16_C( 13308),
        -INT16_C( 24560),  INT16_C(  4365), -INT16_C( 27440), -INT16_C( 16096),  INT16_C(  3136), -INT16_C( 14131), -INT16_C( 14884), -INT16_C( 23752) } },
    { {  INT16_C( 22350),  INT16_C( 26579),  INT16_C( 19546), -INT16_C( 16177), -INT16_C( 29807),  INT16_C( 26280),  INT16_C(  6344),  INT16_C(  8429),
        -INT16_C( 32079), -INT16_C( 25154),  INT16_C(  4980), -INT16_C( 12077),  INT16_C( 13857), -INT16_C( 26986), -INT16_C( 32381), -INT16_C( 11575),
        -INT16_C( 25384),  INT16_C( 12857),  INT16_C(  2280),  INT16_C( 31475), -INT16_C( 25709),  INT16_C( 23520), -INT16_C( 12877),  INT16_C( 25980),
         INT16_C( 14927), -INT16_C( 15614), -INT16_C( 10675),  INT16_C( 28564),  INT16_C( 10764), -INT16_C( 28667), -INT16_C( 12629), -INT16_C( 31646) },
      { -INT16_C( 25750),  INT16_C( 21174), -INT16_C( 22109),  INT16_C( 14028), -INT16_C( 21435), -INT16_C(  1903),  INT16_C(  3449), -INT16_C( 13987),
         INT16_C( 24648), -INT16_C( 27252),  INT16_C(  8246),  INT16_C( 16900),  INT16_C(  2379), -INT16_C(  2350),  INT16_C( 13527),  INT16_C( 16762),
         INT16_C( 12751),  INT16_C( 29332),  INT16_C( 24794),  INT16_C(  8104),  INT16_C( 14861), -INT16_C( 31208),  INT16_C( 30023), -INT16_C( 28849),
        -INT16_C(  9003),  INT16_C(  2853),  INT16_C( 10748),  INT16_C( 18254),  INT16_C(  8243),  INT16_C(  2622), -INT16_C( 18347),  INT16_C(  9292) },
      {  INT16_C( 24652),  INT16_C( 26114),  INT16_C(  1870),  INT16_C( 20212),  INT16_C(  2581), -INT16_C(  6872), -INT16_C(  8568),  INT16_C(  2841),
         INT16_C(  8648), -INT16_C(  9752), -INT16_C( 25992), -INT16_C( 22196),  INT16_C(  1195), -INT16_C( 21748),  INT16_C( 24837),  INT16_C( 31946),
         INT16_C( 10920),  INT16_C( 27380), -INT16_C( 27248),  INT16_C(  7288),  INT16_C( 13431), -INT16_C(  8960), -INT16_C(  9307), -INT16_C( 27324),
         INT16_C( 26555),  INT16_C( 17738),  INT16_C( 18636),  INT16_C(  2840), -INT16_C(  8092),  INT16_C(  4918), -INT16_C( 31033),  INT16_C(  5400) } },
    { { -INT16_C(  7959), -INT16_C( 15209),  INT16_C( 16192),  INT16_C( 19939), -INT16_C(  1159), -INT16_C( 15916),  INT16_C(  9073),  INT16_C( 18000),
         INT16_C( 30207), -INT16_C(   942), -INT16_C( 24417), -INT16_C( 11709), -INT16_C( 32320),  INT16_C(  5596),  INT16_C( 10298),  INT16_C(  9018),
        -INT16_C( 12024),  INT16_C( 18919), -INT16_C( 13552), -INT16_C( 30058),  INT16_C( 27334),  INT16_C( 14155), -INT16_C( 25714), -INT16_C( 29314),
        -INT16_C( 12271), -INT16_C( 20343), -INT16_C( 12944),  INT16_C( 12418),  INT16_C( 24142), -INT16_C( 30650), -INT16_C( 32633), -INT16_C( 28756) },
      { -INT16_C( 27823),  INT16_C( 25048),  INT16_C( 28510),  INT16_C(  9707),  INT16_C( 14041),  INT16_C( 26460), -INT16_C(  9518), -INT16_C(  7179),
         INT16_C( 32426),  INT16_C(  6803),  INT16_C(  5451), -INT16_C( 26037), -INT16_C( 28301), -INT16_C(  1502), -INT16_C( 12783),  INT16_C( 25226),
         INT16_C( 25186), -INT16_C( 16189), -INT16_C( 20527), -INT16_C( 21531),  INT16_C( 17125), -INT16_C( 18670),  INT16_C(  1820), -INT16_C( 14438),
         INT16_C( 11654), -INT16_C( 11807),  INT16_C( 11330), -INT16_C( 18837), -INT16_C( 28995), -INT16_C( 12624),  INT16_C( 14940), -INT16_C( 16848) },
      { -INT16_C(  2887),  INT16_C(  5736), -INT16_C(  1664),  INT16_C( 20065), -INT16_C( 20591), -INT16_C(  3024),  INT16_C( 19634),  INT16_C( 14992),
        -INT16_C(  8874),  INT16_C( 14102),  INT16_C(  6549), -INT16_C(  6239),  INT16_C(  2368), -INT16_C( 16584),  INT16_C( 22490),  INT16_C( 12612),
         INT16_C(  5392), -INT16_C( 29963), -INT16_C( 18416),  INT16_C( 10798), -INT16_C( 28898), -INT16_C( 32698), -INT16_C(  6776),  INT16_C(  4044),
        -INT16_C(  6682),  INT16_C(   361),  INT16_C( 14048), -INT16_C( 19882), -INT16_C(  7274),  INT16_C(  1056), -INT16_C( 14716), -INT16_C( 26560) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_mullo_epi16(a, b);
    simde_test_x86_assert_equal_i16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mullo_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(   355975199), -INT32_C(  1700022260),  INT32_C(  1271212220),  INT32_C(  1338643536),  INT32_C(   295713745),  INT32_C(   236686063),  INT32_C(  1041828026),  INT32_C(  2021440918),
         INT32_C(   140657916), -INT32_C(  1197273604), -INT32_C(  2063308747),  INT32_C(  1305857660), -INT32_C(   111250166), -INT32_C(  1190692353),  INT32_C(   418914690), -INT32_C(   628067106) },
      { -INT32_C(  1243417927),  INT32_C(   896435711),  INT32_C(  1773826797),  INT32_C(  1253478208),  INT32_C(    54727684),  INT32_C(   280841102),  INT32_C(  1244181868), -INT32_C(   500909783),
        -INT32_C(  1416100181),  INT32_C(  2044724876), -INT32_C(  1193043336),  INT32_C(   771987754),  INT32_C(  1009927854), -INT32_C(    45289583), -INT32_C(   850955100), -INT32_C(   642749651) },
      {  INT32_C(  1891294105), -INT32_C(   657289228), -INT32_C(  1505595892),  INT32_C(  1690158080), -INT32_C(   438287548),  INT32_C(  1735267730),  INT32_C(   113979512), -INT32_C(  2045525242),
         INT32_C(   820116564),  INT32_C(   603950544),  INT32_C(  2143271640),  INT32_C(  1089469528), -INT32_C(  2044180276),  INT32_C(    47524463), -INT32_C(   751673528),  INT32_C(  1390236934) } },
    { { -INT32_C(    24885134), -INT32_C(   965254066),  INT32_C(   679369470), -INT32_C(  1571323404),  INT32_C(  1507756488),  INT32_C(   475474552), -INT32_C(   857104738),  INT32_C(  2091227402),
         INT32_C(   796535265), -INT32_C(  1946816115),  INT32_C(  1085568076), -INT32_C(  1109259275),  INT32_C(   202883220), -INT32_C(  2010616086),  INT32_C(   374608140), -INT32_C(  1936524885) },
      { -INT32_C(  1346630622),  INT32_C(  1245426174),  INT32_C(   462090021), -INT32_C(  1898418950),  INT32_C(   379252524),  INT32_C(  1788789341),  INT32_C(  2122380243),  INT32_C(   252384236),
         INT32_C(   499042079), -INT32_C(  1637353096), -INT32_C(   491130392), -INT32_C(  1989111459), -INT32_C(   560002431), -INT32_C(  1605878068),  INT32_C(   488556849), -INT32_C(    80991780) },
      {  INT32_C(   581891876),  INT32_C(   550073188), -INT32_C(  1474148170),  INT32_C(  1915213896),  INT32_C(  1194419808),  INT32_C(  1072258456), -INT32_C(  1189235142),  INT32_C(  1551029048),
         INT32_C(  1213659455),  INT32_C(   150363928), -INT32_C(  1477381920),  INT32_C(  1544822017), -INT32_C(   110505324), -INT32_C(  1125839240), -INT32_C(  1562267828), -INT32_C(   357633548) } },
    { {  INT32_C(  1763306480), -INT32_C(   855146268), -INT32_C(   810565518), -INT32_C(   765910959), -INT32_C(   139331542),  INT32_C(  1738012982), -INT32_C(  1635404350), -INT32_C(   811945505),
        -INT32_C(  2126990436), -INT32_C(  1521598669),  INT32_C(  1366687231),  INT32_C(  1210306077), -INT32_C(    46148410), -INT32_C(  1872439602),  INT32_C(  1781524875),  INT32_C(   926599579) },
      { -INT32_C(  1346866564), -INT32_C(  1319827790),  INT32_C(   553830916),  INT32_C(  1583949464), -INT32_C(   933517062),  INT32_C(   173654142),  INT32_C(  1148487849), -INT32_C(   847466927),
        -INT32_C(   746834911),  INT32_C(  1065668923),  INT32_C(   861964187), -INT32_C(  1483552083), -INT32_C(   244257422),  INT32_C(  1476118957), -INT32_C(  1566871727),  INT32_C(  1064245022) },
      {  INT32_C(   404695104),  INT32_C(   737212040),  INT32_C(   528283080),  INT32_C(   519058968), -INT32_C(  1248466684), -INT32_C(  1216665452), -INT32_C(  1833169646), -INT32_C(  1881417329),
        -INT32_C(  1067788516),  INT32_C(   502084545), -INT32_C(  1744682395),  INT32_C(  2019503001),  INT32_C(  1527254572),  INT32_C(  1590093622), -INT32_C(  1131075077),  INT32_C(  1345686826) } },
    { { -INT32_C(  2028803252),  INT32_C(  1489409725), -INT32_C(   896784867),  INT32_C(  1668423408), -INT32_C(  1185619445), -INT32_C(    66039893), -INT32_C(   593581122),  INT32_C(   253431235),
        -INT32_C(  1248449032), -INT32_C(   519152444),  INT32_C(  1940691586), -INT32_C(  1009377608), -INT32_C(  1417926144),  INT32_C(   933727353), -INT32_C(    82557640),  INT32_C(  1242181458) },
      {  INT32_C(   553689181),  INT32_C(  2114064124),  INT32_C(  1626451624), -INT32_C(   870070324),  INT32_C(  1786224881),  INT32_C(  1688346156), -INT32_C(  1252018589),  INT32_C(  1107323365),
         INT32_C(    90374153), -INT32_C(  1232837106), -INT32_C(   518621932),  INT32_C(   783104317),  INT32_C(   110699993), -INT32_C(  1486210237), -INT32_C(   698561807),  INT32_C(  1025072179) },
      { -INT32_C(  1326515556),  INT32_C(  1932834828), -INT32_C(   282216184),  INT32_C(  1875720000),  INT32_C(   659336283),  INT32_C(  1918483300),  INT32_C(   821526138), -INT32_C(   505918865),
         INT32_C(    19897784),  INT32_C(   885992120),  INT32_C(  1967488040),  INT32_C(   691292632),  INT32_C(  1750178304),  INT32_C(  1196664491),  INT32_C(   215177656), -INT32_C(  1844779690) } },
    { {  INT32_C(  1782742108), -INT32_C(   148846878),  INT32_C(  2044212796),  INT32_C(  1235715440), -INT32_C(   296795990),  INT32_C(  1821751931), -INT32_C(  1220284028), -INT32_C(  1426826162),
        -INT32_C(  1156237352),  INT32_C(   967980541), -INT32_C(   592278932), -INT32_C(  1171957233),  INT32_C(   380138906), -INT32_C(  1283310289),  INT32_C(  2087372078), -INT32_C(    98083039) },
      { -INT32_C(  1816839018), -INT32_C(   573741199),  INT32_C(    12156913),  INT32_C(  1958404057), -INT32_C(  2088082860), -INT32_C(   835318625), -INT32_C(   213212974), -INT32_C(  1779600897),
         INT32_C(   522822317), -INT32_C(    84085239),  INT32_C(  1341896309), -INT32_C(   373049963), -INT32_C(  1217639144),  INT32_C(   730178137),  INT32_C(  1109381186), -INT32_C(   271119295) },
      { -INT32_C(  1725534744), -INT32_C(  1569278014),  INT32_C(   684642940),  INT32_C(   261925872),  INT32_C(  1650871240), -INT32_C(   978413979), -INT32_C(   388587960),  INT32_C(     6053810),
        -INT32_C(   196776712),  INT32_C(  2117141477), -INT32_C(   672789668),  INT32_C(    14091707),  INT32_C(  2116180080), -INT32_C(  2123758761), -INT32_C(  1873922596),  INT32_C(   271299425) } },
    { { -INT32_C(  1207041873),  INT32_C(  1823673078),  INT32_C(  1438363328),  INT32_C(  2067693155),  INT32_C(   607365835), -INT32_C(  1890535348), -INT32_C(   892244088),  INT32_C(   716810363),
        -INT32_C(  1612462167), -INT32_C(  1844734255), -INT32_C(  1477982652),  INT32_C(   253961796),  INT32_C(   489969360), -INT32_C(  1750301682),  INT32_C(  1851882995), -INT32_C(   828827099) },
      { -INT32_C(  1301381919),  INT32_C(  1447328018), -INT32_C(  2063782848),  INT32_C(   580132946), -INT32_C(  2059417482),  INT32_C(  1058859852), -INT32_C(  1901232792),  INT32_C(  2019313303),
        -INT32_C(   735393086), -INT32_C(  2077593788), -INT32_C(   318232421), -INT32_C(  1089495992), -INT32_C(  1337700508), -INT32_C(  1544593350),  INT32_C(  1949408733), -INT32_C(  1494446621) },
      { -INT32_C(  2049614385), -INT32_C(   769848500),  INT32_C(  1689563136), -INT32_C(  2129863754), -INT32_C(  1942728302), -INT32_C(  2123072880),  INT32_C(  2004006720),  INT32_C(   734528141),
        -INT32_C(  1679644654),  INT32_C(  1405267588),  INT32_C(   107981612), -INT32_C(  1490013408), -INT32_C(  1636034240),  INT32_C(  1101867820),  INT32_C(  1273676231), -INT32_C(  1656451121) } },
    { { -INT32_C(  1669720488),  INT32_C(   539010437),  INT32_C(   353183949),  INT32_C(   701767109), -INT32_C(  1495656340),  INT32_C(  1430899064),  INT32_C(  1254718054),  INT32_C(  1626387720),
         INT32_C(  1375496908), -INT32_C(   596501489),  INT32_C(   166887236),  INT32_C(   137610908),  INT32_C(  1471090143),  INT32_C(  1034811606),  INT32_C(  2072475251), -INT32_C(   119834836) },
      { -INT32_C(   246818847),  INT32_C(   936229875), -INT32_C(   683557061),  INT32_C(  1709208710),  INT32_C(  1471975297), -INT32_C(    90936953),  INT32_C(   209001440),  INT32_C(  1946439826),
         INT32_C(   442846503),  INT32_C(  1146237449),  INT32_C(  2015073266), -INT32_C(  1998718201),  INT32_C(   316643722), -INT32_C(   485723133),  INT32_C(   586121871), -INT32_C(    90770478) },
      {  INT32_C(  1193904984),  INT32_C(  1986413631),  INT32_C(   847665727), -INT32_C(   941303522), -INT32_C(  1965383060), -INT32_C(   420776376),  INT32_C(  1758583616), -INT32_C(   168182128),
         INT32_C(  1141874964),  INT32_C(    38529671), -INT32_C(   226731448),  INT32_C(  1372676676), -INT32_C(  1494695626), -INT32_C(   103619966), -INT32_C(  1506897859),  INT32_C(  1106150936) } },
    { {  INT32_C(  1226111808),  INT32_C(   529360429),  INT32_C(     9939449),  INT32_C(   797471908),  INT32_C(   289499150),  INT32_C(  1811172828), -INT32_C(  1567759409),  INT32_C(   379331542),
         INT32_C(  1281404958),  INT32_C(   275508503),  INT32_C(   990970774), -INT32_C(  2056611465), -INT32_C(   577328383), -INT32_C(   934704392),  INT32_C(  1131075181),  INT32_C(   408553210) },
      { -INT32_C(   849036618),  INT32_C(  1054789799),  INT32_C(  1249505235), -INT32_C(  1966021752), -INT32_C(  2040043890),  INT32_C(  1582215409), -INT32_C(  2120042361),  INT32_C(  1973025982),
         INT32_C(  1564671670), -INT32_C(  1583669042), -INT32_C(  1729424369), -INT32_C(  2061321225),  INT32_C(   319588642), -INT32_C(  1049535942), -INT32_C(   784198893), -INT32_C(   985212145) },
      { -INT32_C(  1737023616), -INT32_C(   112707749),  INT32_C(  1700892475),  INT32_C(  1831559968), -INT32_C(   748729404),  INT32_C(   395086876),  INT32_C(   747498025), -INT32_C(  1274068780),
        -INT32_C(  1992885932), -INT32_C(  1201100670),  INT32_C(   620293578), -INT32_C(   859701551),  INT32_C(   647118626), -INT32_C(  1963152336),  INT32_C(   871498007), -INT32_C(  1147517274) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mullo_epi32(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_mullo_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(   466803588), -INT32_C(  1704804295), -INT32_C(   294739476), -INT32_C(    39638017), -INT32_C(   523648680), -INT32_C(  1903034581), -INT32_C(  1147760279), -INT32_C(   982530232),
         INT32_C(   631872747),  INT32_C(  1220480092), -INT32_C(  1405735507), -INT32_C(  1314268583), -INT32_C(   946769253), -INT32_C(   615177614), -INT32_C(    57218124), -INT32_C(  1648294478) },
      UINT16_C(27554),
      { -INT32_C(  2106065214),  INT32_C(  2108630087),  INT32_C(  2052524241),  INT32_C(  1273885626),  INT32_C(   258891706), -INT32_C(   788913606), -INT32_C(   522802146),  INT32_C(   206272585),
        -INT32_C(  1114717578), -INT32_C(  1220854298),  INT32_C(     3314246),  INT32_C(  1061887877), -INT32_C(  1202800258), -INT32_C(    91731749),  INT32_C(  1054498548),  INT32_C(  1296704982) },
      { -INT32_C(   854927129),  INT32_C(  1535460629),  INT32_C(  1566291928),  INT32_C(  1419618262),  INT32_C(  1007479648),  INT32_C(   674665524), -INT32_C(   916057869),  INT32_C(   471248949),
        -INT32_C(  1628888696),  INT32_C(  1056534118), -INT32_C(    73640667),  INT32_C(  1548696060),  INT32_C(  1486379812), -INT32_C(   478032144),  INT32_C(   330098654),  INT32_C(   540001176) },
      { -INT32_C(   466803588), -INT32_C(  1710744365), -INT32_C(   294739476), -INT32_C(    39638017), -INT32_C(   523648680),  INT32_C(  1456765896), -INT32_C(  1147760279), -INT32_C(  1063774435),
        -INT32_C(   330398544),  INT32_C(  1592360356), -INT32_C(  1405735507), -INT32_C(  1912429588), -INT32_C(   946769253),  INT32_C(  1908377424),  INT32_C(  1962705816), -INT32_C(  1648294478) } },
    { {  INT32_C(  1253972452), -INT32_C(  1383483257),  INT32_C(   162014477), -INT32_C(  2107246498),  INT32_C(  1138490963), -INT32_C(  1423549236), -INT32_C(   608251069),  INT32_C(  2063396502),
        -INT32_C(  1899709945),  INT32_C(  2134592882), -INT32_C(   796334990),  INT32_C(   794029788), -INT32_C(  1200410900), -INT32_C(   849110646),  INT32_C(    44638828),  INT32_C(   394044688) },
      UINT16_C(16735),
      { -INT32_C(   510733659), -INT32_C(   624623279),  INT32_C(   617128401),  INT32_C(  1129493712), -INT32_C(   790766484),  INT32_C(  1408387498), -INT32_C(   923270580), -INT32_C(  1106684135),
         INT32_C(  2074056745),  INT32_C(  1783981209), -INT32_C(   695263995),  INT32_C(  1025106385),  INT32_C(  1745745598), -INT32_C(  1933836480),  INT32_C(   441758465),  INT32_C(   886660618) },
      { -INT32_C(  1884325642), -INT32_C(   486996771), -INT32_C(   222787551),  INT32_C(   674222698),  INT32_C(   143670728),  INT32_C(  1049905980),  INT32_C(   140044798),  INT32_C(  1027354951),
        -INT32_C(  2016613462),  INT32_C(   275367407), -INT32_C(  1207819698), -INT32_C(  1126157580), -INT32_C(  1413123985), -INT32_C(  1175889477), -INT32_C(  1983823294),  INT32_C(   499580531) },
      { -INT32_C(   607731058), -INT32_C(  1176475923),  INT32_C(   873194481),  INT32_C(  1791065632),  INT32_C(  1738141792), -INT32_C(  1423549236),  INT32_C(  1246327656),  INT32_C(  2063396502),
         INT32_C(  2018815546),  INT32_C(  2134592882), -INT32_C(   796334990),  INT32_C(   794029788), -INT32_C(  1200410900), -INT32_C(   849110646),  INT32_C(   482175042),  INT32_C(   394044688) } },
    { { -INT32_C(   643525911), -INT32_C(  1494675880),  INT32_C(   610200624), -INT32_C(  1914683874),  INT32_C(  1765320110),  INT32_C(  1092756223), -INT32_C(   674503836), -INT32_C(   873098783),
         INT32_C(  2091161892), -INT32_C(   685601369), -INT32_C(  1745125255),  INT32_C(  1814355134), -INT32_C(  2133500543), -INT32_C(   473761921), -INT32_C(  1128624678),  INT32_C(  1116188446) },
      UINT16_C(11081),
      { -INT32_C(   524750658), -INT32_C(  1017105720), -INT32_C(   291561783),  INT32_C(  1598759306),  INT32_C(  1666632353),  INT32_C(  1743794605),  INT32_C(  1947668461),  INT32_C(   278880337),
         INT32_C(   418404176),  INT32_C(  1406947721), -INT32_C(    96371857),  INT32_C(  1046056092),  INT32_C(    60928086), -INT32_C(   831876383), -INT32_C(   230522463),  INT32_C(   839049697) },
      { -INT32_C(  1035275464), -INT32_C(  1290459580),  INT32_C(  1051547298),  INT32_C(   947652578), -INT32_C(  1740955977),  INT32_C(  1365681584),  INT32_C(   138651687), -INT32_C(  1053145463),
         INT32_C(  2105771321),  INT32_C(  1295030443), -INT32_C(   796074258), -INT32_C(  1677195035), -INT32_C(   701218010),  INT32_C(   271031017), -INT32_C(   887592126), -INT32_C(   359902287) },
      {  INT32_C(  1206190992), -INT32_C(  1494675880),  INT32_C(   610200624), -INT32_C(  1659477548),  INT32_C(  1765320110),  INT32_C(  1092756223),  INT32_C(  1694364955), -INT32_C(   873098783),
         INT32_C(   684720336),  INT32_C(  1396952707), -INT32_C(  1745125255),  INT32_C(   123709324), -INT32_C(  2133500543),  INT32_C(  1805781193), -INT32_C(  1128624678),  INT32_C(  1116188446) } },
    { { -INT32_C(  2090397480), -INT32_C(  1747871832),  INT32_C(  1516723573), -INT32_C(  1930006427), -INT32_C(  1671288141),  INT32_C(   111971012), -INT32_C(  1496201739), -INT32_C(   258974184),
         INT32_C(   376698734),  INT32_C(    78464142),  INT32_C(   123606433),  INT32_C(   949179781),  INT32_C(  1154872703),  INT32_C(  1951039871),  INT32_C(  1578769478), -INT32_C(   397497734) },
      UINT16_C(49825),
      { -INT32_C(  1408880386), -INT32_C(  1832802252),  INT32_C(  1122453167),  INT32_C(  1396139902), -INT32_C(   170543189),  INT32_C(  1175526187), -INT32_C(   923759750),  INT32_C(  1921684083),
        -INT32_C(   165768766), -INT32_C(   393683143), -INT32_C(  1557499867), -INT32_C(  2097716777),  INT32_C(  1148701720),  INT32_C(  1636469223), -INT32_C(  2010482156), -INT32_C(   822430708) },
      {  INT32_C(  2126780485),  INT32_C(   476466679), -INT32_C(  1799384899),  INT32_C(   169260786),  INT32_C(  1783532930),  INT32_C(   734779414),  INT32_C(  1605629267), -INT32_C(   282153558),
        -INT32_C(  1133579579), -INT32_C(    52898753),  INT32_C(  1469093733), -INT32_C(   798906802),  INT32_C(  1245360180), -INT32_C(   613087608), -INT32_C(  1522849541), -INT32_C(  1701484075) },
      {  INT32_C(  2025522294), -INT32_C(  1747871832),  INT32_C(  1516723573), -INT32_C(  1930006427), -INT32_C(  1671288141),  INT32_C(   947756466), -INT32_C(  1496201739),  INT32_C(  1655109470),
         INT32_C(   376698734),  INT32_C(   111289095),  INT32_C(   123606433),  INT32_C(   949179781),  INT32_C(  1154872703),  INT32_C(  1951039871),  INT32_C(   726978972),  INT32_C(   225229308) } },
    { { -INT32_C(  1688861861),  INT32_C(  1016540887),  INT32_C(   345188550),  INT32_C(    48559566),  INT32_C(   760029093),  INT32_C(   537510437), -INT32_C(  1060748053),  INT32_C(   140204973),
         INT32_C(   899920222), -INT32_C(  1502463008),  INT32_C(   834274659), -INT32_C(  1623941382), -INT32_C(   489848387),  INT32_C(   772003395), -INT32_C(   940586726), -INT32_C(  2100344284) },
      UINT16_C(29691),
      {  INT32_C(   699325367),  INT32_C(   992940417),  INT32_C(  1994008898), -INT32_C(  1762158647),  INT32_C(  2104245114),  INT32_C(  1481016937),  INT32_C(   480406093), -INT32_C(  1550868756),
        -INT32_C(    70435463), -INT32_C(  1858667442), -INT32_C(   301527003),  INT32_C(   579141544), -INT32_C(  1549799366), -INT32_C(  1006836362),  INT32_C(  1004576335), -INT32_C(  1226936516) },
      { -INT32_C(    72242259), -INT32_C(   846403673), -INT32_C(  1598319368), -INT32_C(   842907501), -INT32_C(  1485807312), -INT32_C(  1788187578),  INT32_C(  1204832779),  INT32_C(  1744678586),
         INT32_C(    23244378), -INT32_C(  1899040874),  INT32_C(   372148867), -INT32_C(    85724982), -INT32_C(  1751035055), -INT32_C(   886305600),  INT32_C(   269679702),  INT32_C(    74911914) },
      {  INT32_C(  1316272043), -INT32_C(   200551897),  INT32_C(   345188550), -INT32_C(  1854904213),  INT32_C(   573858016), -INT32_C(  1468592970), -INT32_C(   414919857),  INT32_C(  2006572920),
         INT32_C(   121756298), -INT32_C(  1232558156),  INT32_C(   834274659), -INT32_C(  1623941382), -INT32_C(   352858022), -INT32_C(   625631104),  INT32_C(  1255731850), -INT32_C(  2100344284) } },
    { {  INT32_C(  1409735358),  INT32_C(  1289934025),  INT32_C(   677515358),  INT32_C(  1361265920),  INT32_C(  1491649688),  INT32_C(   656610512), -INT32_C(  1154009584),  INT32_C(    79671110),
         INT32_C(  1381614985), -INT32_C(   123847782),  INT32_C(  1277231180), -INT32_C(   576830395), -INT32_C(   650738168), -INT32_C(  1426040421), -INT32_C(   714721393),  INT32_C(  1876567782) },
      UINT16_C(13291),
      {  INT32_C(  1601078721), -INT32_C(  1621116290), -INT32_C(  1511807993), -INT32_C(  1205081214), -INT32_C(  1005467964), -INT32_C(   654532238),  INT32_C(  1358881398),  INT32_C(   327412306),
        -INT32_C(   311168401), -INT32_C(  1282616660), -INT32_C(   111644809),  INT32_C(   531727451), -INT32_C(  1125858742), -INT32_C(   627711901),  INT32_C(   338334658), -INT32_C(   316100995) },
      {  INT32_C(  1289395104), -INT32_C(   469735571),  INT32_C(   836589782),  INT32_C(   659656412), -INT32_C(  1260178095), -INT32_C(   678529003), -INT32_C(  1980974836),  INT32_C(   125178983),
         INT32_C(   475222447), -INT32_C(  1912515656), -INT32_C(  2000626004), -INT32_C(  1095822995),  INT32_C(  1534235462),  INT32_C(   389153035),  INT32_C(   564207290), -INT32_C(   517466318) },
      { -INT32_C(  1033276512), -INT32_C(  1748297306),  INT32_C(   677515358), -INT32_C(   192774216),  INT32_C(  1491649688), -INT32_C(  2086490534), -INT32_C(   396237944),  INT32_C(  1747824382),
        -INT32_C(  1074472735), -INT32_C(   265212000),  INT32_C(  1277231180), -INT32_C(   576830395), -INT32_C(   720265668), -INT32_C(  1297676479), -INT32_C(   714721393),  INT32_C(  1876567782) } },
    { {  INT32_C(   553548648),  INT32_C(  2075131855),  INT32_C(  1241739229), -INT32_C(   972508288),  INT32_C(  1361148742),  INT32_C(   912872316), -INT32_C(  1537799566), -INT32_C(  1970897119),
        -INT32_C(   911571718),  INT32_C(  1615092099),  INT32_C(  1219184840),  INT32_C(  1091482619), -INT32_C(  1433260242), -INT32_C(   169804925),  INT32_C(   664352517), -INT32_C(  1313792074) },
      UINT16_C(23459),
      { -INT32_C(  1078647174),  INT32_C(   839351687), -INT32_C(   723189050), -INT32_C(   721152957), -INT32_C(  1613658178),  INT32_C(   366401148), -INT32_C(  1372221955), -INT32_C(  1207248834),
        -INT32_C(  2038972417),  INT32_C(    45645372), -INT32_C(  1009279616),  INT32_C(  1889131441), -INT32_C(   569415070),  INT32_C(  1056171328), -INT32_C(  1326700430),  INT32_C(     6944257) },
      { -INT32_C(   226041675), -INT32_C(   537641377), -INT32_C(  1918645285),  INT32_C(   150813862), -INT32_C(   421131098),  INT32_C(  1680136945),  INT32_C(    51646722), -INT32_C(  1123844857),
        -INT32_C(  1112569506), -INT32_C(  1550015545),  INT32_C(   355483503),  INT32_C(   555625851),  INT32_C(   705168441), -INT32_C(   510776098),  INT32_C(  1172611901),  INT32_C(  2114119712) },
      { -INT32_C(  2070219710), -INT32_C(  1629862119),  INT32_C(  1241739229), -INT32_C(   972508288),  INT32_C(  1361148742),  INT32_C(   458588860), -INT32_C(  1537799566), -INT32_C(  1566822478),
        -INT32_C(   133630302), -INT32_C(  1235594076),  INT32_C(  1219184840), -INT32_C(  1253528821),  INT32_C(   371734482), -INT32_C(   169804925),  INT32_C(  1166948650), -INT32_C(  1313792074) } },
    { {  INT32_C(   960213361), -INT32_C(  1008936876), -INT32_C(  1814492137),  INT32_C(  1924462393), -INT32_C(   627262213),  INT32_C(   649800681), -INT32_C(   294936626), -INT32_C(   110269049),
         INT32_C(  1932699678), -INT32_C(  1741287808),  INT32_C(  1395330842),  INT32_C(    46522118), -INT32_C(  2049154660),  INT32_C(  1521194892), -INT32_C(  1102506186), -INT32_C(  1548241276) },
      UINT16_C(59742),
      {  INT32_C(  1291312918), -INT32_C(  1571024521),  INT32_C(   696345188),  INT32_C(  1082793316),  INT32_C(  1322719138), -INT32_C(  1167782287), -INT32_C(  2089752116), -INT32_C(  1569927284),
         INT32_C(   636445614), -INT32_C(   658027660),  INT32_C(   302074029),  INT32_C(   139627366),  INT32_C(   341191330),  INT32_C(    80657208),  INT32_C(   830947237), -INT32_C(  1126894834) },
      { -INT32_C(   891174058), -INT32_C(  1146968050),  INT32_C(  1456317424), -INT32_C(   782294994), -INT32_C(  2098875062),  INT32_C(   377926513),  INT32_C(     4656626),  INT32_C(  1455168256),
        -INT32_C(   333406754),  INT32_C(   917029445), -INT32_C(  1819511451), -INT32_C(   547034219),  INT32_C(   308365729), -INT32_C(   282531843), -INT32_C(   185569292),  INT32_C(  1766501515) },
      {  INT32_C(   960213361),  INT32_C(  1187843202), -INT32_C(  1392121408),  INT32_C(  1754908664),  INT32_C(  1171021524),  INT32_C(   649800681), -INT32_C(   735027496), -INT32_C(   110269049),
        -INT32_C(  1820711196), -INT32_C(  1741287808),  INT32_C(  1395330842), -INT32_C(   338049954), -INT32_C(  2049154660), -INT32_C(   738292136), -INT32_C(   936299452), -INT32_C(  1694932838) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_mullo_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_mullo_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(24017),
      { -INT32_C(   266698699),  INT32_C(   176354454),  INT32_C(   403676777),  INT32_C(  2072272096),  INT32_C(  1766446988),  INT32_C(  2109727987),  INT32_C(   675115709), -INT32_C(  1366946183),
         INT32_C(   614375566), -INT32_C(    30531180), -INT32_C(  1625932353),  INT32_C(   639277722),  INT32_C(  1703896177),  INT32_C(   115494472),  INT32_C(   976101569), -INT32_C(  1108822994) },
      { -INT32_C(   371095723),  INT32_C(  1743196328), -INT32_C(   418972083),  INT32_C(   168632472), -INT32_C(   848323196), -INT32_C(  1395437077), -INT32_C(  1595539087),  INT32_C(   190697398),
        -INT32_C(    17547690), -INT32_C(  1671046066),  INT32_C(  1921215450),  INT32_C(   276599179),  INT32_C(   433974062), -INT32_C(  1362710467),  INT32_C(  1733209265), -INT32_C(   781014149) },
      { -INT32_C(  1644329831),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   620232752),  INT32_C(           0),  INT32_C(   912286317), -INT32_C(   302571258),
        -INT32_C(  1551608908),  INT32_C(           0),  INT32_C(  1803315622),  INT32_C(  1151019934), -INT32_C(  1618688178),  INT32_C(           0), -INT32_C(   642494095),  INT32_C(           0) } },
    { UINT16_C(26602),
      {  INT32_C(   876820687),  INT32_C(  1486822868),  INT32_C(   216607631),  INT32_C(   418846523), -INT32_C(   154651600),  INT32_C(   832797155),  INT32_C(  1407000289), -INT32_C(  1078278160),
        -INT32_C(   722141697), -INT32_C(  1439919334), -INT32_C(   105507394),  INT32_C(  1544662316), -INT32_C(   984360478),  INT32_C(   888600147),  INT32_C(    76010260), -INT32_C(  1698413926) },
      {  INT32_C(  1517205567),  INT32_C(   168073803), -INT32_C(   620512593),  INT32_C(  1228346727),  INT32_C(  1091472110), -INT32_C(  1787492992),  INT32_C(  1939471832),  INT32_C(  2114805055),
         INT32_C(  1641577237), -INT32_C(   999564267), -INT32_C(    23105898),  INT32_C(  1917310595), -INT32_C(   508340639),  INT32_C(   863381851),  INT32_C(  1705381926), -INT32_C(  2082229395) },
      {  INT32_C(           0),  INT32_C(  1843348764),  INT32_C(           0),  INT32_C(   553594813),  INT32_C(           0), -INT32_C(   690134912), -INT32_C(   779130152),  INT32_C(  1355375632),
         INT32_C(   474634987), -INT32_C(  1477416670), -INT32_C(  1828036268),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1954470529), -INT32_C(   330132232),  INT32_C(           0) } },
    { UINT16_C(48174),
      {  INT32_C(  1335378916), -INT32_C(  1497551097),  INT32_C(  1954365741),  INT32_C(  1724571315),  INT32_C(   915350975), -INT32_C(    12143271), -INT32_C(    21777638),  INT32_C(   448454966),
         INT32_C(   745099813), -INT32_C(  1361893503),  INT32_C(   455299176), -INT32_C(   310252242),  INT32_C(  1814237459), -INT32_C(   513054266),  INT32_C(  1407131165),  INT32_C(   628005120) },
      {  INT32_C(  1834080235),  INT32_C(  1713054974), -INT32_C(  1568588172),  INT32_C(  1066402604), -INT32_C(   626281708),  INT32_C(   985339421),  INT32_C(   898472501),  INT32_C(   526056243),
        -INT32_C(   779310125),  INT32_C(  1144563664),  INT32_C(   317176294), -INT32_C(   799967300),  INT32_C(  1202453546),  INT32_C(  1199662610),  INT32_C(   880545537), -INT32_C(   581708278) },
      {  INT32_C(           0), -INT32_C(   905009934),  INT32_C(  1882680932),  INT32_C(  1563058116),  INT32_C(           0), -INT32_C(  1600442091),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0), -INT32_C(   995269264),  INT32_C(  1010339784), -INT32_C(   280985314), -INT32_C(   596093972),  INT32_C(           0), -INT32_C(  2059929088) } },
    { UINT16_C(57218),
      { -INT32_C(   410561873),  INT32_C(  2107665814), -INT32_C(   789291649), -INT32_C(   657711315), -INT32_C(   398467482), -INT32_C(  1560854490), -INT32_C(   931593868), -INT32_C(  1901593633),
        -INT32_C(   478859699),  INT32_C(   442570139), -INT32_C(  1595255438), -INT32_C(   612845964), -INT32_C(  1144801387),  INT32_C(  1818082039), -INT32_C(  1707813189),  INT32_C(   505994193) },
      { -INT32_C(  1526555382),  INT32_C(   633365427),  INT32_C(   767929016), -INT32_C(   167231903), -INT32_C(   307115019),  INT32_C(  1079578245), -INT32_C(  1227125275),  INT32_C(  1926562664),
         INT32_C(  1410914209), -INT32_C(   226895814), -INT32_C(   484491390),  INT32_C(  1926834045),  INT32_C(  2002750194),  INT32_C(  2126035097), -INT32_C(  1372286139),  INT32_C(   924846486) },
      {  INT32_C(           0), -INT32_C(  1072848414),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   893587560),
         INT32_C(   298633581), -INT32_C(  1602441954), -INT32_C(   791383580), -INT32_C(   569326940),  INT32_C(  1665457370),  INT32_C(           0), -INT32_C(  1618525849),  INT32_C(  2093688182) } },
    { UINT16_C(14560),
      {  INT32_C(    84941451),  INT32_C(   742691597),  INT32_C(  1347731830), -INT32_C(  1814411725),  INT32_C(  1984656318),  INT32_C(   638161393),  INT32_C(  1596956479),  INT32_C(  1654132951),
         INT32_C(   929540138), -INT32_C(  1302025413), -INT32_C(  1610434452), -INT32_C(  1137451778),  INT32_C(  1093828176),  INT32_C(  1315388175), -INT32_C(  1297180709), -INT32_C(   787200345) },
      {  INT32_C(   705264878), -INT32_C(  1797493465),  INT32_C(   590667301),  INT32_C(   182478778), -INT32_C(   179563803),  INT32_C(   692302670), -INT32_C(   237244086),  INT32_C(   650374967),
        -INT32_C(  1823421333),  INT32_C(  1579625529), -INT32_C(   998155510), -INT32_C(  1479581246), -INT32_C(  1029956748),  INT32_C(   434888910),  INT32_C(   151701201),  INT32_C(   556780981) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   319355502), -INT32_C(   493660362), -INT32_C(   467052751),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1452558724),  INT32_C(  1995680832),  INT32_C(   411280914),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(32665),
      { -INT32_C(   609496396), -INT32_C(  1305037520),  INT32_C(  1209268345),  INT32_C(  1013155743),  INT32_C(   874263113),  INT32_C(  1442508107),  INT32_C(   623096054), -INT32_C(  2052801327),
        -INT32_C(  1084207217),  INT32_C(  2121373188),  INT32_C(   767985038),  INT32_C(  1449732620), -INT32_C(  1484094116),  INT32_C(  1778156915),  INT32_C(   110042933),  INT32_C(  1804284892) },
      { -INT32_C(  2027230333),  INT32_C(   252025985),  INT32_C(   758959137),  INT32_C(  1384359670), -INT32_C(  1627845077), -INT32_C(   938936941), -INT32_C(   238119147),  INT32_C(  1297897930),
        -INT32_C(   975862204),  INT32_C(  1138022946), -INT32_C(  1670311770), -INT32_C(   487656266), -INT32_C(  1786714366), -INT32_C(   228751139), -INT32_C(   354211041), -INT32_C(   935903356) },
      { -INT32_C(  1595929060),  INT32_C(           0),  INT32_C(           0), -INT32_C(   437097270),  INT32_C(  1115516995),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1908254186),
         INT32_C(  2083847164), -INT32_C(   869968760), -INT32_C(  1832814060),  INT32_C(  1509905544), -INT32_C(  2023027016), -INT32_C(   693416889), -INT32_C(   208556437),  INT32_C(           0) } },
    { UINT16_C( 3269),
      {  INT32_C(  1642588301), -INT32_C(  1670214357),  INT32_C(   412166186), -INT32_C(  1962896630), -INT32_C(  2045518551), -INT32_C(  1296944177),  INT32_C(  1425094173), -INT32_C(  1939753217),
        -INT32_C(   890353506), -INT32_C(    10067755), -INT32_C(  1827080312), -INT32_C(  1273096053), -INT32_C(  1002819083), -INT32_C(  2122912668),  INT32_C(   584476451), -INT32_C(  1129433315) },
      {  INT32_C(  1418108031), -INT32_C(  2058031876),  INT32_C(  1863871716),  INT32_C(  2032350852), -INT32_C(   868393625),  INT32_C(  1833808714),  INT32_C(   932127514), -INT32_C(   638370470),
        -INT32_C(   701597222),  INT32_C(  1247511142),  INT32_C(  1924756462),  INT32_C(   300670121), -INT32_C(  2082658247), -INT32_C(   168809765), -INT32_C(  1473478834), -INT32_C(  1753079619) },
      { -INT32_C(   510637581),  INT32_C(           0),  INT32_C(   405372264),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   270038514),  INT32_C(  2114066598),
         INT32_C(           0),  INT32_C(           0),  INT32_C(  1773156464), -INT32_C(  2020694077),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(45209),
      { -INT32_C(   919404691),  INT32_C(    37494857),  INT32_C(  2111760018), -INT32_C(   727377673), -INT32_C(  1929477989), -INT32_C(  1593095051),  INT32_C(  2009188597), -INT32_C(   853058721),
        -INT32_C(  1533650598),  INT32_C(   195482233), -INT32_C(  1350007368),  INT32_C(   948120989),  INT32_C(   583303853), -INT32_C(  1010577202), -INT32_C(   130382440),  INT32_C(   986014176) },
      {  INT32_C(   870210490), -INT32_C(   432110291), -INT32_C(  1483356662), -INT32_C(  1595991565),  INT32_C(  1774363803),  INT32_C(   204310132), -INT32_C(   352033013), -INT32_C(  2094675511),
         INT32_C(  1387725860), -INT32_C(  1824983671), -INT32_C(  1355100740), -INT32_C(  2125456922),  INT32_C(   837489341), -INT32_C(  1556211560),  INT32_C(  1217282687),  INT32_C(   818721804) },
      { -INT32_C(  2023795662),  INT32_C(           0),  INT32_C(           0),  INT32_C(   391106933),  INT32_C(  1894853081),  INT32_C(           0),  INT32_C(           0), -INT32_C(   866167657),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1911315527),  INT32_C(  1404783184),  INT32_C(           0), -INT32_C(  2003691904) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_mullo_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mullo_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mullo_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_mullo_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_mullo_epi32)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
