/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** Clownfish::CFC::Model::Hierarchy - A class hierarchy.
 *
 * A Clownfish::CFC::Model::Hierarchy consists of all the classes defined in
 * files within a source directory and its subdirectories.
 *
 * There may be more than one tree within the Hierarchy, since all "inert"
 * classes are root nodes, and since Clownfish does not officially define any
 * core classes itself from which all instantiable classes must descend.
 */

#ifndef H_CFCHIERARCHY
#define H_CFCHIERARCHY

#ifdef __cplusplus
extern "C" {
#endif

typedef struct CFCHierarchy CFCHierarchy;
struct CFCClass;
struct CFCFile;

/**
 * @param dest The directory where the autogenerated files will be written.
 */
CFCHierarchy*
CFCHierarchy_new(const char *dest);

CFCHierarchy*
CFCHierarchy_init(CFCHierarchy *self, const char *dest);

void
CFCHierarchy_destroy(CFCHierarchy *self);

/** Add path C<source_dir> to the list of source directories.
 */
void
CFCHierarchy_add_source_dir(CFCHierarchy *self, const char *source_dir);

/** Add path C<include_dir> to the list of include directories.
 */
void
CFCHierarchy_add_include_dir(CFCHierarchy *self, const char *include_dir);

/** Add parcel to the list of required parcel.
 */
void
CFCHierarchy_add_prereq(CFCHierarchy *self, const char *parcel);

/** Parse every Clownfish header file which can be found under any of the
 * source and include directories, building up the object hierarchy.
 */
void
CFCHierarchy_build(CFCHierarchy *self);

/** Read host-specific data for included parcels.
 */
void
CFCHierarchy_read_host_data_json(CFCHierarchy *self, const char *host_lang);

/** Visit all File objects in the hierarchy.  If a parent node is modified, mark
 * all of its children as modified.
 *
 * If the supplied argument is true, mark all Files as modified.
 */
int
CFCHierarchy_propagate_modified(CFCHierarchy *self, int modified);

/** Write a JSON files with statistics about the class hierarchy. At the
 * moment, this file is empty. It is only used for reliable dependency
 * handling in the Makefiles of the C build. Thus, the log file should only
 * be updated after all other files were written successfully.
 */
void
CFCHierarchy_write_log(CFCHierarchy *self);

/** Return all Classes as an array with the property that every parent class
 * will precede all of its children.
 */
struct CFCClass**
CFCHierarchy_ordered_classes(CFCHierarchy *self);

struct CFCFile**
CFCHierarchy_files(CFCHierarchy *self);

const char**
CFCHierarchy_get_source_dirs(CFCHierarchy *self);

const char**
CFCHierarchy_get_include_dirs(CFCHierarchy *self);

const char*
CFCHierarchy_get_dest(CFCHierarchy *self);

const char*
CFCHierarchy_get_include_dest(CFCHierarchy *self);

const char*
CFCHierarchy_get_source_dest(CFCHierarchy *self);

#ifdef __cplusplus
}
#endif

#endif /* H_CFCHIERARCHY */

