package AI::ConfusionMatrix;
$AI::ConfusionMatrix::VERSION = '0.001';
use Carp;
use Exporter 'import';
our @EXPORT= qw (makeConfusionMatrix);
use Tie::File;

# ABSTRACT: Make a confusion matrix

sub makeConfusionMatrix {
    my ($matrix, $filename) = @_;
    carp ('First argument must be a hash reference') if ref($matrix) ne 'HASH';
    tie my @array, 'Tie::File', $filename or carp "$!";
    my $n = 1;
    my @columns;
    my @expected = sort keys %{$matrix};
    my %stats;
    for my $expected (@expected) {
        $array[$n] = $expected;
        ++$n;
        $stats{$expected}{'fn'} = 0;
        $stats{$expected}{'tp'} = 0;
        for my $predicted (keys %{$matrix->{$expected}}) {
            $stats{$expected}{'total'} += $matrix->{$expected}->{$predicted};
            $stats{$expected}{'tp'} += $matrix->{$expected}->{$predicted} if $expected == $predicted;
            if ($expected != $predicted) {
                $stats{$expected}{'fn'} += $matrix->{$expected}->{$predicted};
                $stats{$predicted}{'fp'} += $matrix->{$expected}->{$predicted};
            }
            push @columns, $predicted unless _findIndex($predicted, \@columns);
        }
        $stats{$expected}{'acc'} = sprintf("%.2f\%", ($stats{$expected}{'tp'} * 100) / $stats{$expected}{'total'});
    }
    @columns = sort @columns;
    map {$array[0] .= ',' . $_} join ',', (@columns, 'TOTAL', 'TP', 'FP', 'FN', 'ACC');
    $n = 1;
    for my $expected (@expected) {
        $stats{$expected}{'fp'} //= 0;
        my $lastIndex = 0;
        for my $predicted (sort keys %{$matrix->{$expected}}) {
            $index = _findIndex($predicted, \@columns);
            $array[$n] .= ',' x ($index - $lastIndex) . %{$matrix}{$expected}->{$predicted};
            $lastIndex = $index;
        }
        $array[$n] .= ',' x (scalar(@columns) - $lastIndex + 1);
        $array[$n] .= join ',', ($stats{$expected}{'total'},
                                 $stats{$expected}{'tp'},
                                 $stats{$expected}{'fp'},
                                 $stats{$expected}{'fn'},
                                 $stats{$expected}{'acc'}
                             );
        ++$n;
    }
}

sub _findIndex {
    my ($string, $array) = @_;
    for (0 .. scalar(@{$array})) {
        return $_ + 1 if ($string eq @{$array}[$_]);
    }
}

=head1 NAME

AI::ConfusionMatrix - make a confusion matrix

=head1 SYNOPSIS

    my %matrix;

    Loop over your tests

    ---

    $matrix{$expected}{$predicted} += 1;

    ---

    makeConfusionMatrix(\%matrix, 'output.csv');


=head1 DESCRIPTION

This module prints a L<confusion matrix|https://en.wikipedia.org/wiki/Confusion_matrix>.

=head3 function

=head4 C<makeConfusionMatrix($hash_ref, $filename)>

This function makes a confusion matrix from C<$hash_ref> and writes it to C<$filename>.

Example:

    makeConfusionMatrix(\%matrix, 'output.csv');

The hash reference must look like this :

    $VAR1 = {


              'value_expected1' => {
                          'value_predicted1' => value
                        },
              'value_expected2' => {
                          'value_predicted1' => value,
                          'value_predicted2' => value
                        },
              'value_expected3' => {
                          'value_predicted3' => value
                        }

            };

The output will be in CSV. Here is an example:


    ,1997,1998,2001,2003,2005,2008,2012,2015,TOTAL,TP,FP,FN,ACC
    1997,1,,,,,,,1,2,1,0,1,50.00%
    1998,,1,,,,,,,1,1,0,0,100.00%
    2001,,,1,,,,,,1,1,0,0,100.00%
    2003,,,,5,,,,2,7,5,0,2,71.43%
    2005,,,,,5,,,4,9,5,0,4,55.56%
    2008,,,,,,3,,,3,3,0,0,100.00%
    2012,,,,,,,5,,5,5,0,0,100.00%
    2015,,,,,,,,2,2,2,7,0,100.00%

=over

=item TP:

True Positive

=item FP:

False Positive

=item FN:

False Negative

=item ACC:

Accuracy

=back

=head1 AUTHOR

Vincent Lequertier <sky@riseup.net>

=head1 LICENSE

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

1;

# vim: set ts=4 sw=4 tw=0 fdm=marker :

